/*******************************************************************************
 * The information contained in this file is confidential and proprietary to
 * QLogic Corporation.  No part of this file may be reproduced or
 * distributed, in any form or by any means for any purpose, without the
 * express written permission of QLogic Corporation.
 *
 * (c) COPYRIGHT 2015 QLogic Corporation, ALL RIGHTS RESERVED.
 *******************************************************************************/
/* **********************************************************
  * Copyright 2015 VMware, Inc.  All rights reserved. -- VMware Confidential
  * **********************************************************/

 /*
  * qfle3_utils.c --
  *
  *      Utility function implementation of native bxe driver.
  */

#include "qfle3.h"

/*
 * Heap Allocation
 */
/*
 *-----------------------------------------------------------------------------
 *
 * qfle3_heap_alloc --
 *
 *      Allocate heap memory and zero it out.
 *
 * Results:
 *      Valid pointer on success, NULL if otherwise.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------
 */
inline void *
qfle3_heap_alloc(vmk_uint32 size)       // IN: allocation size
{
   void *p;
   p = vmk_HeapAlloc(qfle3_mod_info.heapID, size);
   if (p != NULL) {
      vmk_Memset(p, 0, size);
   }
   return p;
}

/*
 *-----------------------------------------------------------------------------
 *
 * qfle3_heap_alloc_aligned --
 *
 *      Allocate heap memory with specified alignment and zero it out.
 *
 * Results:
 *      Valid pointer on success, NULL if otherwise.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------
 */
inline void *
qfle3_heap_alloc_aligned(vmk_uint32 size,       // IN: allocation size
                         vmk_uint32 align)      // IN: alignment
{
   void *p;
   p = vmk_HeapAlign(qfle3_mod_info.heapID, size, align);
   if (p != NULL) {
      vmk_Memset(p, 0, size);
   }
   return p;
}

/*
 *-----------------------------------------------------------------------------
 *
 * qfle3_heap_free --
 *
 *      Free previously allocated heap memory.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------
 */
inline void
qfle3_heap_free(void *p)        // IN: pointer to heap memory
{
if(p) {
   vmk_HeapFree(qfle3_mod_info.heapID, p);
}
}

/*
 * Mempool Operations
 */
/*
 *-----------------------------------------------------------------------------
 *
 * qfle3_mempool_alloc --
 *
 *      Allocate physically contiguous memory from qfle3 mempool.
 *      The actually size of memory allocation is rounded up to
 *      VMK_PAGE_SIZE.
 *
 * Results:
 *      VA to allocated memory on success, NULL if otherwise.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------
 */
inline void *
qfle3_mempool_alloc(vmk_uint32 size)    // IN: allocation size
{
   VMK_ReturnStatus status;
   vmk_VA va;
   vmk_MemPoolAllocRequest request;
   vmk_MemPoolAllocProps props;
   vmk_MapRequest mapReq;
   vmk_MpnRange mpnRange;

   size = VMK_UTIL_ROUNDUP(size, VMK_PAGE_SIZE);

   props.physContiguity = VMK_MEM_PHYS_CONTIGUOUS;
   props.physRange = VMK_PHYS_ADDR_ANY;
   props.creationTimeoutMS = VMK_TIMEOUT_UNLIMITED_MS;
   request.numPages = size >> VMK_PAGE_SHIFT;
   request.numElements = 1;
   request.mpnRanges = &mpnRange;
   status = vmk_MemPoolAlloc(qfle3_mod_info.memPoolID, &props, &request);
   if (status != VMK_OK) {
      QFLE3_ERROR("Failed to allocate %d pages from mempool (%x)",
                  request.numPages, status);
      goto fail_alloc;
   }

   mapReq.mapType = VMK_MAPTYPE_DEFAULT;
   mapReq.mapAttrs = VMK_MAPATTRS_READWRITE;
   mapReq.numElements = 1;
   mapReq.mpnRanges = &mpnRange;
   status = vmk_Map(vmk_ModuleCurrentID, &mapReq, &va);
   if (status != VMK_OK) {
      QFLE3_ERROR("Failed to map %d pages (%x)", mpnRange.numPages, status);
      goto fail_map;
   }
   vmk_Memset((void *) va, 0, size);
   return (void *) va;

  fail_map:
   vmk_MemPoolFree(&request);
  fail_alloc:
   return NULL;
}

/*
 *-----------------------------------------------------------------------------
 *
 * qfle3_mempool_free --
 *
 *      Allocated memory ranges are unmapped and freed from nbxe mempool.
 *      The actually size of memory range is rounded up to VMK_PAGE_SIZE.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------
 */
inline void
qfle3_mempool_free(void *va,    // IN: VA to free
                   vmk_uint32 size)     // IN: allocated size
{
   vmk_MA ma;
   vmk_MpnRange mpnRange;
   vmk_MemPoolAllocRequest request;

   size = VMK_UTIL_ROUNDUP(size, VMK_PAGE_SIZE);

   vmk_VA2MA((vmk_VA) va, size, &ma);
   vmk_Unmap((vmk_VA) va);
   mpnRange.startMPN = vmk_MA2MPN(ma);
   mpnRange.numPages = size >> VMK_PAGE_SHIFT;
   request.numPages = mpnRange.numPages;
   request.numElements = 1;
   request.mpnRanges = &mpnRange;
   vmk_MemPoolFree(&request);
}

/*
 * DMA Operations
 */
/*
 *-----------------------------------------------------------------------------
 *
 * qfle3_dma_map_ma --
 *
 *      Map specified MA range to an IO address range.
 *
 * Results:
 *      VMK_OK on success with IO address returned in ioa.
 *      Error code if otherwise.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------
 */
VMK_ReturnStatus
qfle3_dma_map_ma(qfle3_adapter * adapter,       // IN: adapter
                 vmk_MA ma,     // IN: machine address
                 vmk_uint32 size,       // IN: size
                 vmk_DMADirection direction,    // IN: DMA mapping direction
                 vmk_DMAEngine engine, vmk_IOA * ioa)   // OUT: starting IO address
{
   VMK_ReturnStatus status;
   vmk_SgElem in, out;
   vmk_DMAMapErrorInfo err;

   in.addr = ma;
   in.length = size;
   status = vmk_DMAMapElem(engine, direction, &in, VMK_TRUE, &out, &err);

   if (status != VMK_OK) {
      if (status == VMK_DMA_MAPPING_FAILED) {
         QFLE3_ERR("Failed to map range [0x%lx, 0x%lx]: %s",
                   in.addr, in.addr + in.length - 1,
                   vmk_DMAMapErrorReasonToString(err.reason));
      } else {
         QFLE3_ERR("Failed to map range [0x%lx, 0x%lx] (%x)",
                   in.addr, in.addr + in.length - 1, status);
      }
   } else {
      VMK_ASSERT(in.length == out.length);
      *ioa = out.ioAddr;
   }
   return status;
}

/*
 *-----------------------------------------------------------------------------
 *
 * qfle3_dma_map_va --
 *
 *      Map specified VA range to an IO address range. The VA range MUST
 *      correspond to an established physically contiguous memory range.
 *
 *      WARNING: VA allocated from nbxe driver heap shouldn't
 *               use this function, since it is not physically
 *               contiguous, and will break the assumption of
 *               vmk_VA2MA.
 *
 * Results:
 *      VMK_OK on success with IO address returned in ioa.
 *      Error code if otherwise.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------
 */
VMK_ReturnStatus
qfle3_dma_map_va(qfle3_adapter * adapter,       // IN: adapter
                 void *va,      // IN: virtual address
                 vmk_uint32 size,       // IN: size
                 vmk_DMADirection direction,    // IN: DMA mapping direction
                 vmk_DMAEngine engine, vmk_IOA * ioa)   // OUT: starting IO address
{
   vmk_MA ma;
   vmk_VA2MA((vmk_VA) va, size, &ma);
   return qfle3_dma_map_ma(adapter, ma, size, direction, engine, ioa);
}

/*
 *-----------------------------------------------------------------------------
 *
 * qfle3_dma_unmap --
 *
 *      Unmap specified MA range.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------
 */
void
qfle3_dma_unmap(qfle3_adapter * adapter,        // IN: adapter
                vmk_IOA ioAddr, // IN: IO address
                vmk_uint32 size,        // IN: size
                vmk_DMADirection direction,     // IN: DMA mapping direction
                vmk_DMAEngine engine)   // IN: coherent or not
{
   vmk_SgElem elem;
   VMK_ReturnStatus status;

   elem.ioAddr = ioAddr;
   elem.length = size;
   status = vmk_DMAUnmapElem(engine, direction, &elem);
   VMK_ASSERT(status == VMK_OK);
}

/*
 *-----------------------------------------------------------------------------
 *
 * qfle3_AllocCoherentDMAMapping --
 *
 *      Allocate kernel memory from mempool and map with coherent DMA mapping.
 *
 * Results:
 *      Return valid virtual address for allocated memory, and its associated
 *      IO address. NULL if otherwise.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------
 */
void *
qfle3_alloc_dma_mapping(qfle3_adapter * adapter,        // IN:  adapter
                        vmk_DMAEngine engine, vmk_uint32 size,  // IN:  size
                        vmk_IOA * ioAddr)       // OUT: IO address
{
   VMK_ReturnStatus status;
   void *va;

   *ioAddr = 0;
   va = qfle3_mempool_alloc(size);
   if (va == NULL) {
      goto fail_mempool_alloc;
   }
   status = qfle3_dma_map_va(adapter, va, size,
                             VMK_DMA_DIRECTION_BIDIRECTIONAL, engine, ioAddr);

   if (status != VMK_OK) {
      goto fail_dma_map;
   }

   return va;

  fail_dma_map:
   qfle3_mempool_free(va, size);
  fail_mempool_alloc:
   return NULL;
}

/*
 *-----------------------------------------------------------------------------
 *
 * qfle3_FreeCoherentDMAMapping --
 *
 *      Unmap DMA mapping and free allocated kernel memory from mempool.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------
 */
void
qfle3_free_dma_mapping(qfle3_adapter * adapter, // IN: adapter
                       vmk_DMAEngine engine, void *va,  // IN: virtual address
                       vmk_IOA ioAddr,  // IN: IO address
                       vmk_uint32 size) // IN: size
{
   qfle3_dma_unmap(adapter, ioAddr, size, VMK_DMA_DIRECTION_BIDIRECTIONAL, engine);
   qfle3_mempool_free(va, size);
}

/*
 * Locking Operations
 */
/*
 *-----------------------------------------------------------------------------
 *
 * qfle3_CreateSpinlock --
 *
 *      Create spin lock with specified name and lock rank.
 *
 * Results:
 *      VMK_OK on success, and lock created. Error code if otherwise.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------
 */
inline VMK_ReturnStatus
qfle3_create_spinlock(const char *lckName,      // IN:  lock name
                      const char *devName,      // IN:  device name
                      vmk_LockRank rank,        // IN:  lock rank
                      vmk_Lock * lock)  // OUT: created lock
{
   vmk_SpinlockCreateProps lockProps;
   VMK_ReturnStatus status;
   lockProps.moduleID = vmk_ModuleCurrentID;
   lockProps.heapID = qfle3_mod_info.heapID;
   if (devName) {
      vmk_NameFormat(&lockProps.name, "qfle3_%s_%s", devName, lckName);
   } else {
      vmk_NameFormat(&lockProps.name, "qfle3_%s", lckName);
   }
   lockProps.type = VMK_SPINLOCK;
   lockProps.domain = qfle3_mod_info.lockDomain;
   lockProps.rank = rank;
   status = vmk_SpinlockCreate(&lockProps, lock);
   if (status != VMK_OK) {
      QFLE3_ERROR("Failed to create %s spinlock (%x)",
                  QFLE3_NAME_TO_STRING(lockProps.name), status);
      *lock = NULL;	
   }
   return status;
}

inline void
qfle3_init_spinlock(vmk_Lock * lock,
                    struct qfle3_adapter *adapter,
                    const char *lckName, vmk_LockRank rank)
{
   //const char *devName;
   struct qfle3_lock *q_lock = NULL;
   VMK_ReturnStatus status = VMK_OK;

   status = qfle3_create_spinlock(lckName,
                                  QFLE3_NAME_TO_STRING((adapter)->pdev_name), rank,
                                  lock);
   if (status != VMK_OK) {
      return;
   }
   q_lock = qfle3_heap_alloc(sizeof(*q_lock));
   if (q_lock == NULL) {
      QFLE3_ERROR("Failed to allocate memory for spinlock link");
      vmk_SpinlockDestroy(*lock);
      return;
   }

   q_lock->spin_lock = *lock;
   q_lock->type = QFLE3_LOCK_TYPE_SPINLOCK;

   vmk_SpinlockLock(qfle3_mod_info.drv_lock);
   vmk_ListInsert(&q_lock->link, vmk_ListAtFront(&qfle3_mod_info.ecore_lock_list));
   vmk_SpinlockUnlock(qfle3_mod_info.drv_lock);
}

/*
 *-----------------------------------------------------------------------------
 *
 * qfle3_DestroySpinlock --
 *
 *      Destroy the spin lock.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------
 */
inline void
qfle3_destroy_spinlock(vmk_Lock lock)   // IN: lock to destroy
{
   if (lock) {
      vmk_SpinlockDestroy(lock);
   }
//TODO: remove lock from global list if required.
}

void
qfle3_init_mutex(vmk_Semaphore * lock)
{
   struct qfle3_lock *q_lock = NULL;
   VMK_ReturnStatus status = VMK_OK;

#if (VMKAPI_REVISION >= VMK_API_2_4_0_0)
   status = vmk_SemaCreate(lock, qfle3_mod_info.heapID, "ecore", 1);
#else
   status = vmk_SemaCreate(lock, vmk_ModuleCurrentID, "ecore", 1);
#endif
   if (status != VMK_OK) {
      vmk_Warning(qfle3_mod_info.logID, "Failed to create Mutex");
      return;
   }

   q_lock = qfle3_heap_alloc(sizeof(*q_lock));
   if (q_lock == NULL) {
      vmk_Warning(qfle3_mod_info.logID, "Failed to allocate memory for semaphore link");
      vmk_SemaDestroy(lock);
      return;
   }

   q_lock->sema = *lock;
   q_lock->type = QFLE3_LOCK_TYPE_SEMAPHORE;

   vmk_SpinlockLock(qfle3_mod_info.drv_lock);
   vmk_ListInsert(&q_lock->link, vmk_ListAtFront(&qfle3_mod_info.ecore_lock_list));
   vmk_SpinlockUnlock(qfle3_mod_info.drv_lock);
}

inline VMK_ReturnStatus
qfle3_pkt_alloc_and_map(qfle3_adapter * adapter,        // IN:  adapter
                        vmk_ByteCountSmall len, // IN:  pkt length
                        vmk_DMAEngine engine, vmk_PktHandle ** pkt,     // OUT: allocated pkt
                        vmk_IOA * dmaAddr)      // OUT: starting IO address
{
   VMK_ReturnStatus status;
   vmk_PktHandle *newPkt;
   const vmk_SgElem *sge;
   vmk_IOA ioa;

   status = vmk_PktAllocForDMAEngine(len, engine, &newPkt);
   if (VMK_UNLIKELY(status != VMK_OK)) {
      //QFLE3_ERR("failed to allocate pkt buffer status 0x%x", status);
      //adapter->drv_stats.rxBufAllocFailure++;
      goto fail_alloc;
   }

   /*
    * The pkt should be a flat one! 
    */
   sge = vmk_PktSgElemGet(newPkt, 0);
   VMK_ASSERT(sge);
   status = qfle3_dma_map_ma(adapter, sge->addr, sge->length,
                             VMK_DMA_DIRECTION_TO_MEMORY, engine, &ioa);
   if (VMK_UNLIKELY(status != VMK_OK)) {
   	  QFLE3_ERR("failed map len %d", sge->length);
      //rq->stats.rxDMAMapFailure++;
      goto fail_map;
   }
   
   *pkt = newPkt;
   *dmaAddr = ioa;
   return VMK_OK;

  fail_map:
   vmk_PktRelease(newPkt);
  fail_alloc:
   return status;
}

/*
 *-----------------------------------------------------------------------------
 *
 * qfle3_PageAllocAndMap --
 *
 *      Allocate and map a page for refilling non-SOP buffer in RX buffer ring.
 *
 * Results:
 *      A new page is allocated, and DMA mapped.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------
 */
inline VMK_ReturnStatus
qfle3_page_alloc_and_map(qfle3_adapter * adapter,       // IN:  adapter
                         vmk_DMAEngine engine, vmk_MPN * page,  // OUT: allocated page
                         vmk_IOA * dmaAddr)     // OUT: starting IO address
{
   VMK_ReturnStatus status;
   vmk_MPN mpn;
   vmk_IOA ioa;

   status = vmk_PktAllocPage(VMK_PHYS_ADDR_ANY, &mpn);
   if (VMK_UNLIKELY(status != VMK_OK)) {
      //rq->stats.rxBufAllocFailure++;
      goto fail_alloc;
   }

   status = qfle3_dma_map_ma(adapter,
                             vmk_MPN2MA(mpn),
                             VMK_PAGE_SIZE, VMK_DMA_DIRECTION_TO_MEMORY, engine, &ioa);
   if (VMK_UNLIKELY(status != VMK_OK)) {
      //rq->stats.rxDMAMapFailure++;
      goto fail_map;
   }

   *page = mpn;
   *dmaAddr = ioa;
   return VMK_OK;

  fail_map:
   vmk_PktFreePage(VMK_PHYS_ADDR_ANY, mpn);
  fail_alloc:
   return status;
}

/*
 *-----------------------------------------------------------------------------
 *
 * qfle3_PageUnmapAndFree --
 *
 *      Unmap and free a page previously set up for refilling non-SOP buffer.
 *
 * Results:
 *      The specified page is unmapped and freed.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------
 */
inline void
qfle3_page_unmap_and_free(qfle3_adapter * adapter,      // IN:  adapter
                          vmk_MPN page, // IN:  allocated page
                          vmk_IOA dmaAddr)      // IN:  starting IO address
{
   qfle3_dma_unmap(adapter, dmaAddr, VMK_PAGE_SIZE,
                   VMK_DMA_DIRECTION_TO_MEMORY, adapter->dmaEngine);
   vmk_PktFreePage(VMK_PHYS_ADDR_ANY, page);
}

VMK_ReturnStatus
qfle3_create_helper(qfle3_adapter * adapter,
                    vmk_Helper * helper, const char *helper_name)
{
   vmk_HelperProps props;
   VMK_ReturnStatus status;

   vmk_NameFormat(&props.name, "qfle3_%s_%s", helper_name,
                  QFLE3_NAME_TO_STRING(adapter->pdev_name));
   props.heap = qfle3_mod_info.heapID;
   props.preallocRequests = VMK_FALSE;
   props.blockingSubmit = VMK_FALSE;
   props.maxRequests = 8;
   props.mutables.minWorlds = 0;
   props.mutables.maxWorlds = 1;
   props.mutables.maxIdleTime = 0;
   props.mutables.maxRequestBlockTime = 0;
   props.tagCompare = NULL;
   props.constructor = NULL;
   props.constructorArg = (vmk_AddrCookie) NULL;

   status = vmk_HelperCreate(&props, helper);
   if (status != VMK_OK) {
      QFLE3_ERR("Failed to create helper world queue (%x)", status);
   }
   return status;
}

VMK_ReturnStatus
qfle3_cancel_helper_requests(qfle3_adapter * adapter, vmk_Helper helper)
{
    vmk_uint32 num_cancelled;
    vmk_AddrCookie data;
    VMK_ReturnStatus status = VMK_OK;

    if (!helper) {
	return VMK_OK;
    }

    data.ptr = adapter;
	while (vmk_HelperCurrentRequestCount(helper)) {
		/* Avoid busy wait */
		vmk_WorldSleep(5000);
		status = vmk_HelperCancelRequest(helper, data, &num_cancelled);
		if (status != VMK_OK) {
			QFLE3_DBG(QFLE3_DBG_SP,
					"Unable to cancel the task from the queue.\n");
			continue;
		}
	}

    return status;
}

VMK_ReturnStatus
qfle3_destroy_helper(qfle3_adapter * adapter, vmk_Helper helper)
{
    VMK_ReturnStatus status = VMK_OK;
    qfle3_cancel_helper_requests(adapter, helper);
    status = vmk_HelperDestroy(helper);
    return status;
}


VMK_ReturnStatus
qfle3_schedule_helper(qfle3_adapter * adapter, vmk_Helper helper,
                      vmk_HelperRequestFunc req_func, vmk_uint32 timeout)
{
   vmk_HelperRequestProps props;
   VMK_ReturnStatus status;
   vmk_AddrCookie data;

   props.requestMayBlock = VMK_FALSE;
   props.tag.ptr = adapter;
   props.cancelFunc = NULL;
   props.worldToBill = VMK_INVALID_WORLD_ID;
   data.ptr = adapter;
   if (timeout) {
	   status = vmk_HelperSubmitDelayedRequest(helper,
			   req_func, data, timeout, &props);
   } else {
	   status = vmk_HelperSubmitRequest(helper,
			   req_func, data, &props);
   }

   if (status != VMK_OK) {
      QFLE3_ERR("Failed to submit reset request to " "helper world queue (%x)", status);
   }

   return status;
}

void qfle3_disable_fwdmp(qfle3_adapter *adapter)
{
   qfle3_adapter *adapter_tmp = NULL;
   vmk_ListLinks *link = NULL;

   VMK_LIST_FORALL(&qfle3_mod_info.adapterList, link){
      adapter_tmp = VMK_LIST_ENTRY(link, struct qfle3_adapter, adapterLink);
      if (adapter_tmp->pdev_addr.bus == adapter->pdev_addr.bus) {
         adapter_tmp->fwdmp_enable = 0;
         //  QFLE3_INFO("Firmware dump for adapter %u has been disabled.\n", adapter_tmp);
      }
   }
}
