/*******************************************************************************
 * See COPYRIGHT.txt & LICENSE.txt for copyright and licensing details.
 *******************************************************************************/

#ifndef __QFLE3I_H__
#define __QFLE3I_H__

#include <vmkapi.h>
#include "qfle3i_vmk.h"
#include "qfle3i_global.h"
#include "qcnic_if.h"
#include "57xx_iscsi_constants.h"
#include "57xx_iscsi_hsi.h"
#include "iscsi_proto_vmk.h"

#ifndef __LITTLE_ENDIAN
#define __LITTLE_ENDIAN
#endif

#define QFLE3I_DRV_MODULE_VERSION	"1.0.23.0"
#define DRV_MODULE_VERSION QFLE3I_DRV_MODULE_VERSION
#define QFLE3I_DRIVER_NAME  "qfle3i"
#define BNX2I_DRIVER_NAME   "bnx2i"
#define QFLE3I_SCSI_HBA_NAME  "qfle3i_scsi"
#define QFLE3I_INTERFACE_VERSION	0

#define QFLE3I_HEAP_INITIAL_SIZE ( 16 * 1024 * 1024 )
#define QFLE3I_HEAP_MAXIMUM_SIZE ( 1024 * 1024 * 1024 )

#define MAX_ADDR_LEN	32		/* Largest hardware address length */
#define INITIATOR_MAX_LEN	256
#define TARGET_MAX_LEN		256

#define MAX_SCSISCAN_COUNTER 10

#define QFLE3I_MAX_HBA		32

// Fix me
#define QFLE3I_MAX_IO_SECTORS					512
#define QFLE3I_MAX_TARGET                       128
#define QFLE3I_MAX_LUNS                         512
#define QFLE3I_MAX_CMD_LEN                      16
#define ISCSI_MAX_ADAPTERS			8
#define ISCSI_MAX_CONNS_PER_HBA			128
#define ISCSI_MAX_SESS_PER_HBA			ISCSI_MAX_CONNS_PER_HBA
#define ISCSI_MAX_CMDS_PER_SESS			128
#define ISCSI_MAX_BDS_PER_CMD			255
/* should be equal to size of BDs but for safer side, lets
	keep it as half of BDs.
*/
#define ISCSI_MAX_SG_PER_CMD			ISCSI_MAX_BDS_PER_CMD

#define MAX_KEY_VAL_STRING_LEN 4096

/*
	* ESX-6.5:
	* For IPV6, to be backward compatible, vmware asked us to use
	* old definition of AF_INET6 = 0xa.
	* This will replace VMK_SOCKET_AF_INET6.
*/
#define AF_INET6       0xa /* IPV6 family */


/* capabilities & params */
#define QFLE3I_TRANSPORT_CAPS                                   \
                   VMK_ISCSI_CAP_RECOVERY_L0,                     \
                   VMK_ISCSI_CAP_MULTI_R2T,                       \
                   VMK_ISCSI_CAP_HDRDGST,                         \
                   VMK_ISCSI_CAP_DATADGST,                        \
                   VMK_ISCSI_CAP_SESSION_PERSISTENT,              \
                   VMK_ISCSI_CAP_KERNEL_POLL,                       \
                   VMK_ISCSI_CAP_CONN_CLEANUP

#define QFLE3I_TRANSPORT_CONN_PARAMS                              \
                   VMK_ISCSI_CONN_PARAM_MAX_RECV_DLENGTH,         \
                   VMK_ISCSI_CONN_PARAM_MAX_XMIT_DLENGTH,         \
                   VMK_ISCSI_CONN_PARAM_HDRDGST_EN,               \
                   VMK_ISCSI_CONN_PARAM_DATADGST_EN,              \
                   VMK_ISCSI_CONN_PARAM_EXP_STATSN,               \
                   VMK_ISCSI_CONN_PARAM_PERSISTENT_PORT,          \
                   VMK_ISCSI_CONN_PARAM_PERSISTENT_ADDRESS,       \
                   VMK_ISCSI_CONN_PARAM_IFMARKER_EN,              \
                   VMK_ISCSI_CONN_PARAM_OFMARKER_EN,               \
                   VMK_ISCSI_CONN_PARAM_CONN_PORT,                \
                   VMK_ISCSI_CONN_PARAM_CONN_ADDRESS              \

#define QFLE3I_TRANSPORT_SESS_PARAMS                            \
                   VMK_ISCSI_SESS_PARAM_MAX_R2T,                  \
                   VMK_ISCSI_SESS_PARAM_INITIAL_R2T_EN,           \
                   VMK_ISCSI_SESS_PARAM_IMM_DATA_EN,              \
                   VMK_ISCSI_SESS_PARAM_FIRST_BURST,              \
                   VMK_ISCSI_SESS_PARAM_MAX_BURST,                \
                   VMK_ISCSI_SESS_PARAM_PDU_INORDER_EN,           \
                   VMK_ISCSI_SESS_PARAM_DATASEQ_INORDER_EN,       \
                   VMK_ISCSI_SESS_PARAM_ERL,                      \
                   VMK_ISCSI_SESS_PARAM_TARGET_NAME,              \
                   VMK_ISCSI_SESS_PARAM_TPGT,                     \
                   VMK_ISCSI_SESS_PARAM_ISID

#define QFLE3I_TRANSPORT_HOST_PARAMS                                      \
                   VMK_ISCSI_HOST_PARAM_HWADDRESS,                \
                   VMK_ISCSI_HOST_PARAM_IPADDRESS,                \
                   VMK_ISCSI_HOST_PARAM_INITIATOR_NAME,           \
                   VMK_ISCSI_HOST_PARAM_NETDEV_NAME


enum QFLE3I_DEBUG_LEVEL {
	QFLE3I_DBG_ALERT,
	QFLE3I_DBG_WARN,
	QFLE3I_DBG_INFO,
	QFLE3I_DBG_DEBUG,
};

#define DRV_WARN_MODLOAD_FAIL(msg, status)                                      \
           vmk_WarningMessage(msg ": %s",                                       \
                           vmk_StatusToString(status));                         \
           vmk_WarningMessage("Could not load " QFLE3I_DRIVER_NAME " module")

#define DRV_WARN(msg, status)                                          \
   vmk_Warning(qfle3i_vmkLog,                                              \
               "failed: system error=%s; driver error: "                        \
               msg,                                                             \
               vmk_StatusToString(status));

#if defined(VMX86_DEBUG)
#define DRV_DEBUG(level, msg, args...)                                          \
   if (qfle3i_vmkLog != NULL) {                                                 \
      vmk_LogDebug(qfle3i_vmkLog, level, msg, ##args);                     \
   } else {                                                                     \
      vmk_LogDebugMessage(msg, ##args);                                         \
   }
#else /* !defined(VMX86_DEBUG) */
#define DRV_DEBUG(level, msg, args...)
#endif /* defined(VMX86_DEBUG) */

/* VK: Copy from bnx2i.h, need to relook */
#define DRV_MODULE_NAME		"qfle3i"
//#define DRV_MODULE_VERSION	"2.713.10"
#define DRV_MODULE_RELDATE	"Aug. 15, 2016"

#ifndef __FREE_IRQ_FIX__
#define __FREE_IRQ_FIX__	1
#endif

#define BNX2_ISCSI_DRIVER_NAME			"qfle3i"

#define vmk_AlertMessage_dbg(hba, fmt, __args...) 	do {			 \
	vmk_LogLevel(VMK_LOG_URGENCY_ALERT, qfle3i_vmkLog,               \
			QFLE3I_DBG_ALERT, "qfle3i:%s: %s:%d " fmt,				 \
			(hba != NULL ? vmk_NameToString(&hba->vmhba_name) : "?"),\
			__func__, __LINE__, ##__args);                           \
	} while(0)

#define vmk_WarningMessage_dbg(hba, fmt, __args...) 	do {         \
	vmk_LogLevel(VMK_LOG_URGENCY_WARNING, qfle3i_vmkLog,             \
			QFLE3I_DBG_WARN, "qfle3i:%s:%s:%d " fmt,                 \
			(hba != NULL ? vmk_NameToString(&hba->vmhba_name) : "?"),\
			__func__, __LINE__, ##__args);                           \
	} while(0)


#define vmk_LogMessage_dbg(hba, fmt, __args...) 	do {             \
	vmk_LogLevel(VMK_LOG_URGENCY_NORMAL, qfle3i_vmkLog,              \
			QFLE3I_DBG_INFO, "qfle3i:%s: %s:%d " fmt,                \
			(hba != NULL ? vmk_NameToString(&hba->vmhba_name) : "?"),\
			__func__, __LINE__, ##__args);                           \
	} while(0)

#define vmk_DebugMessage_dbg(hba, fmt, __args...) 	do {			 \
	vmk_LogLevel(VMK_LOG_URGENCY_DEBUG, qfle3i_vmkLog,               \
			QFLE3I_DBG_DEBUG, "qfle3i:%s: %s:%d " fmt,				 \
			(hba != NULL ? vmk_NameToString(&hba->vmhba_name) : "?"),\
			__func__, __LINE__, ##__args);                           \
	} while(0)

/*
	VK: check this later, see why its not working.
	vmk_LogDebug("qfle3i:%s: %s:%d " fmt,				\
			(hba != NULL ? vmk_NameToString(&hba->vmhba_name) : "?"), 		\
			__func__, __LINE__, ##__args);		\
	} while(0)
*/

#define PRINT_EMERG(hba, fmt, __args...) \
	vmk_AlertMessage_dbg(hba, fmt, ##__args)
#define PRINT_ALERT(hba, fmt, __args...) \
	vmk_AlertMessage_dbg(hba, fmt, ##__args)
#define PRINT_CRIT(hba, fmt, __args...) \
	vmk_AlertMessage_dbg(hba, fmt, ##__args)
#define PRINT_ERR(hba, fmt, __args...) \
	vmk_WarningMessage_dbg(hba, fmt, ##__args)
#define PRINT_WARNING(hba, fmt, __args...) \
	vmk_WarningMessage_dbg(hba, fmt, ##__args)
#define PRINT_NOTICE(hba, fmt, __args...) \
	vmk_LogMessage_dbg(hba, fmt, ##__args)
#define PRINT_INFO(hba, fmt, __args...) \
	vmk_LogMessage_dbg(hba, fmt, ##__args)
#define PRINT_DEBUG(hba, fmt, __args...) \
	vmk_DebugMessage_dbg(hba, fmt, ##__args)
#define PRINT(hba, fmt, __args...)	 \
	vmk_DebugMessage_dbg(hba, fmt, ##__args)

#define MIN(x,y)            ((x)<(y)?(x):(y))

extern vmk_uint32 qfle3i_debug_level;
/*
 * 0x1 will enable DEFAULT_DBG_LEVEL bitmap
 */
#define DBG_DEFAULT     0x00000001
#define DBG_INIT        0x00000002
#define DBG_CONN_SETUP  0x00000004
#define DBG_TMF         0x00000008
#define DBG_ISCSI_NOP   0x00000010
#define DBG_CNIC_IF     0x00000020
#define DBG_ITT_CLEANUP 0x00000040
#define DBG_CONN_EVENT  0x00000080
#define DBG_SESS_RECO   0x00000100
#define DBG_INTERNAL    0x00000200
#define DBG_IO          0x00000400
#define DBG_APP         0x00000800

#define DEF_DBG_LEVEL  (DBG_INIT | DBG_CONN_SETUP | DBG_TMF |          \
						DBG_CNIC_IF | DBG_ITT_CLEANUP |                \
						DBG_CONN_EVENT | DBG_SESS_RECO | DBG_INTERNAL)

#define QFLE3I_DBG(level, hba, fmt, __args...)		do {		\
		if (VMK_UNLIKELY(level & qfle3i_debug_level))	\
			vmk_LogMessage_dbg(hba, fmt, ##__args);		\
	} while(0)

#ifndef PCI_DEVICE_ID_NX2_5706
#define PCI_DEVICE_ID_NX2_5706			0x164a
#endif

#ifndef PCI_DEVICE_ID_NX2_5706S
#define PCI_DEVICE_ID_NX2_5706S			0x16aa
#endif

#ifndef PCI_DEVICE_ID_NX2_5708
#define PCI_DEVICE_ID_NX2_5708			0x164c
#endif

#ifndef PCI_DEVICE_ID_NX2_5708S
#define PCI_DEVICE_ID_NX2_5708S			0x16ac
#endif

#ifndef PCI_DEVICE_ID_NX2_5709
#define PCI_DEVICE_ID_NX2_5709			0x1639
#endif

#ifndef PCI_DEVICE_ID_NX2_5709S
#define PCI_DEVICE_ID_NX2_5709S			0x163a
#endif

#ifndef PCI_DEVICE_ID_NX2_5716
#define PCI_DEVICE_ID_NX2_5716			0x163b
#endif

#ifndef PCI_DEVICE_ID_NX2_5716S
#define PCI_DEVICE_ID_NX2_5716S			0x163c
#endif


#ifndef PCI_DEVICE_ID_NX2_57710
#define PCI_DEVICE_ID_NX2_57710			0x164e
#endif

#ifndef PCI_DEVICE_ID_NX2_57711
#define PCI_DEVICE_ID_NX2_57711			0x164f
#endif

#ifndef PCI_DEVICE_ID_NX2_57712
#define PCI_DEVICE_ID_NX2_57712			0x1662
#endif

#ifndef PCI_DEVICE_ID_NX2_57800
#define PCI_DEVICE_ID_NX2_57800			0x168a
#endif

#ifndef PCI_DEVICE_ID_NX2_57810
#define PCI_DEVICE_ID_NX2_57810			0x168e
#endif

#ifndef PCI_DEVICE_ID_NX2_57811
#define PCI_DEVICE_ID_NX2_57811			0x163d
#endif

#ifndef PCI_DEVICE_ID_NX2_57811S
#define PCI_DEVICE_ID_NX2_57811S		0x163e
#endif

#ifndef PCI_DEVICE_ID_NX2_57840
#define PCI_DEVICE_ID_NX2_57840			0x16a1
#endif

#ifndef PCI_DEVICE_ID_NX2_57840S
#define PCI_DEVICE_ID_NX2_57840S		0x16a4
#endif

#ifndef PCI_DEVICE_ID_NX2_57840_20G
#define PCI_DEVICE_ID_NX2_57840_20G			0x16a2
#endif

#define QFLE3I_REGISTER_HBA_SUPPORTED		0
#define QFLE3I_REGISTER_HBA_FORCED		1

#define MAX_PAGES_PER_CTRL_STRUCT_POOL		16
#define QFLE3I_RESERVED_SLOW_PATH_CMD_SLOTS	4

#define QFLE3I_5771X_DBELL_PAGE_SIZE		128

/* 5706/08 hardware has limit on maximum buffer size per BD it can handle */
#define MAX_BD_LENGTH				65535
#define BD_SPLIT_SIZE				32768

/* min, max & default values for SQ/RQ/CQ size, configurable via' modparam */
#define QFLE3I_SQ_WQES_MIN 			16
#define QFLE3I_570X_SQ_WQES_MAX			128
#define QFLE3I_5770X_SQ_WQES_MAX			512

#define QFLE3I_570X_SQ_WQES_DEFAULT 		32
#define QFLE3I_5709_SQ_WQES_DEFAULT 		64
#define QFLE3I_5770X_SQ_WQES_DEFAULT 		128
#define QFLE3I_5770X_SQ_WQES_DEFAULT_X86		64

#define QFLE3I_CQ_WQES_MIN 			16
#define QFLE3I_CQ_WQES_MAX 			256
#define QFLE3I_CQ_WQES_DEFAULT 		24

#define QFLE3I_RQ_WQES_MIN 			16
#define QFLE3I_RQ_WQES_MAX 			32
#define QFLE3I_RQ_WQES_DEFAULT 			16

/* CCELLs per conn */
#define QFLE3I_CCELLS_MIN			16
#define QFLE3I_CCELLS_MAX			96
#define QFLE3I_CCELLS_DEFAULT			64

#define ISCSI_CONN_LOGIN_BUF_SIZE		16384
#define ITT_INVALID_SIGNATURE			0xFFFF

#define ISCSI_CMD_CLEANUP_TIMEOUT		100

#define QFLE3I_CONN_CTX_BUF_SIZE			16384

#define QFLE3I_SQ_WQE_SIZE			64
#define QFLE3I_RQ_WQE_SIZE			256
#define QFLE3I_CQE_SIZE				64

#define QFLE3I_TCP_WINDOW_MIN			(16 * 1024)
#define QFLE3I_TCP_WINDOW_MAX			(1 * 1024 * 1024)
#define QFLE3I_TCP_WINDOW_DEFAULT		(64 * 1024)

#define MB_KERNEL_CTX_SHIFT			8
#define MB_KERNEL_CTX_SIZE			(1 << MB_KERNEL_CTX_SHIFT)

#define CTX_SHIFT				7
#define GET_CID_NUM(cid_addr)			((cid_addr) >> CTX_SHIFT)

#define CTX_OFFSET 				0x10000
#define MAX_CID_CNT				0x4000

#define BNX2_TXP_SCRATCH			0x00060000
#define BNX2_TPAT_SCRATCH			0x000a0000
#define BNX2_RXP_SCRATCH			0x000e0000
#define BNX2_COM_SCRATCH			0x00120000
#define BNX2_CP_SCRATCH				0x001a0000

#define BNX2_PCICFG_REG_WINDOW_ADDRESS		0x00000078
#define BNX2_PCICFG_REG_WINDOW_ADDRESS_VAL	(0xfffffL<<2)
#define BNX2_PCICFG_REG_WINDOW			0x00000080

/* 5709 context registers */
#define BNX2_MQ_CONFIG2				0x00003d00
#define BNX2_MQ_CONFIG2_CONT_SZ			(0x7L<<4)
#define BNX2_MQ_CONFIG2_FIRST_L4L5		(0x1fL<<8)

/* 57710's BAR2 is mapped to doorbell registers */
#define BNX2X_DB_SHIFT				3
#define BNX2X_DOORBELL_PCI_BAR			2
#define BNX2X_MAX_CQS				8

#ifndef DMA_64BIT_MASK
#define DMA_64BIT_MASK 				((vmk_uint64) 0xffffffffffffffffULL)
#define DMA_32BIT_MASK 				((vmk_uint64) 0x00000000ffffffffULL)
#endif

#ifndef DMA_40BIT_MASK
#define DMA_40BIT_MASK 				((vmk_uint64) 0x000000ffffffffffULL)
#endif

#define CNIC_ARM_CQE			1
#define CNIC_DISARM_CQE			0

#define QFLE3I_TBL_TYPE_NONE		0
#define QFLE3I_TBL_TYPE_PG		1
#define QFLE3I_TBL_TYPE_BD		2
#define REG_RD(__hba, offset)				\
		RD_REG_DWORD(__hba->regview + offset)
#define REG_WR(__hba, offset, val)			\
		WRT_REG_DWORD(val, __hba->regview + offset)

#define GET_STATS_64(__hba, dst, field)				\
	do {							\
		dst->field##_lo = __hba->stats.field##_lo;	\
		dst->field##_hi = __hba->stats.field##_hi;	\
	} while (0)

#define ADD_STATS_64(__hba, field, len)				\
	do {							\
		if (vmk_SpinlockTryLock((__hba)->stat_lock) == VMK_OK) {	\
			if ((__hba)->stats.field##_lo + len <	\
			    (__hba)->stats.field##_lo)		\
				(__hba)->stats.field##_hi++;	\
			(__hba)->stats.field##_lo += len;	\
			vmk_SpinlockUnlock((__hba)->stat_lock);	\
		}						\
	} while (0)

enum ql_iscsi_pcie_link_width {
    QL_ISCSI_PCIE_LNK_WIDTH_RESRV    = 0x00,
    QL_ISCSI_PCIE_LNK_X1     = 0x01,
    QL_ISCSI_PCIE_LNK_X2     = 0x02,
    QL_ISCSI_PCIE_LNK_X4     = 0x04,
    QL_ISCSI_PCIE_LNK_X8     = 0x08,
    QL_ISCSI_PCIE_LNK_X12        = 0x0C,
    QL_ISCSI_PCIE_LNK_X16        = 0x10,
    QL_ISCSI_PCIE_LNK_X32        = 0x20,
    QL_ISCSI_PCIE_LNK_WIDTH_UNKNOWN  = 0xFF,
};

enum ql_iscsi_pci_bus_speed {
    QL_ISCSI_PCI_SPEED_33MHz         = 0x00,
    QL_ISCSI_PCI_SPEED_66MHz         = 0x01,
    QL_ISCSI_PCI_SPEED_66MHz_PCIX        = 0x02,
    QL_ISCSI_PCI_SPEED_100MHz_PCIX       = 0x03,
    QL_ISCSI_PCI_SPEED_133MHz_PCIX       = 0x04,
    QL_ISCSI_PCI_SPEED_66MHz_PCIX_ECC    = 0x05,
    QL_ISCSI_PCI_SPEED_100MHz_PCIX_ECC   = 0x06,
    QL_ISCSI_PCI_SPEED_133MHz_PCIX_ECC   = 0x07,
    QL_ISCSI_PCI_SPEED_66MHz_PCIX_266    = 0x09,
    QL_ISCSI_PCI_SPEED_100MHz_PCIX_266   = 0x0a,
    QL_ISCSI_PCI_SPEED_133MHz_PCIX_266   = 0x0b,
    QL_ISCSI_AGP_UNKNOWN         = 0x0c,
    QL_ISCSI_AGP_1X              = 0x0d,
    QL_ISCSI_AGP_2X              = 0x0e,
    QL_ISCSI_AGP_4X              = 0x0f,
    QL_ISCSI_AGP_8X              = 0x10,
    QL_ISCSI_PCI_SPEED_66MHz_PCIX_533    = 0x11,
    QL_ISCSI_PCI_SPEED_100MHz_PCIX_533   = 0x12,
    QL_ISCSI_PCI_SPEED_133MHz_PCIX_533   = 0x13,
    QL_ISCSI_PCIE_SPEED_2_5GT        = 0x14,
    QL_ISCSI_PCIE_SPEED_5_0GT        = 0x15,
    QL_ISCSI_PCIE_SPEED_8_0GT        = 0x16,
    QL_ISCSI_PCI_SPEED_UNKNOWN       = 0xff,
};

#define QL_ISCSI_PCICFG_LINK_CONTROL     0xbc
#define QL_ISCSI_PCICFG_LINK_WIDTH       0x1f00000
#define QL_ISCSI_PCICFG_LINK_WIDTH_SHIFT     20
#define QL_ISCSI_PCICFG_LINK_SPEED       0xf0000
#define QL_ISCSI_PCICFG_LINK_SPEED_SHIFT     16

struct qfle3i_dma {
	vmk_ListLinks link;
	int size;
	char *mem;
	vmk_IOA mapping;
	int pgtbl_type;
	int pgtbl_size;
	char *pgtbl;
	vmk_IOA pgtbl_map;
};

/*
 * struct bd_resc_page - tracks DMA'able memory allocated for BD tables
 *
 * @link:               list head to link elements
 * @max_ptrs:           maximun pointers that can be stored in this page
 * @num_valid:          number of pointer valid in this page
 * @page:               base addess for page pointer array
 *
 * structure to track DMA'able memory allocated for command BD tables
 */
struct bd_resc_page {
	vmk_ListLinks link;
	vmk_uint32 max_ptrs;
	vmk_uint32 num_valid;
	void *page[1];
};

/*
 * struct io_bdt - I/O buffer destricptor table
 *
 * @link:               list head to link elements
 * @bd_tbl:             BD table's virtual address
 * @bd_tbl_dma:         BD table's dma address
 * @cmdp:               command structure this BD is allocated
 * @max_bd_cnt:         max BD entries in this table
 * @bd_valid:           num valid BD entries
 *
 * IO BD table
 */
typedef struct io_bdt {
	vmk_ListLinks link;
	struct iscsi_bd *bd_tbl;
	vmk_IOA bd_tbl_dma;
	struct qfle3i_cmd *cmdp;
	vmk_uint16 max_bd_cnt;
	vmk_uint16 bd_valid;
}VMK_ATTRIBUTE_L1_ALIGNED io_bdt;

/*
 * struct generic_pdu_resc - login pdu resource structure
 *
 * @pdu_hdr:            buffer to copy iscsi header prepared by 'iscsid'
 * @cmd:                iSCSI command pointer
 * @login_itt:          iSCSI ITT to be used with login exchanges
 * @req_buf:            driver buffer used to stage payload associated with
 *                      the login request
 * @req_dma_addr:       dma address for iscsi login request payload buffer
 * @req_buf_size:       actual login request payload length
 * @req_wr_ptr:         pointer into login request buffer when next data is
 *                      to be written
 * @resp_hdr:           iscsi header where iscsi login response header is to
 *                      be recreated
 * @resp_buf:           buffer to stage login response payload
 * @resp_dma_addr:      login response payload buffer dma address
 * @resp_buf_size:      login response paylod length
 * @resp_wr_ptr:        pointer into login response buffer when next data is
 *                      to be written
 * @req_bd_tbl:         BD table to indicate login request payload buffer details
 * @req_bd_dma:         login request BD table dma address
 * @resp_bd_tbl:        BD table to indicate login response payload buffer details
 * @resp_bd_dma:        login request BD table dma address
 *
 * following structure defines buffer info for generic pdus such as iSCSI Login,
 *	Logout and NOP
 */
struct generic_pdu_resc {
	struct iscsi_hdr pdu_hdr;
	vmk_uint32 login_itt;
	struct qfle3i_dma login_req;
#define req_buf		login_req.mem
	vmk_uint32 req_buf_size;
	char *req_wr_ptr;
	struct iscsi_hdr resp_hdr;
	struct qfle3i_dma login_resp;
#define resp_buf	login_resp.mem
	vmk_uint32 resp_buf_size;
	char *resp_wr_ptr;
	struct iscsi_hdr nopout_hdr;
	struct iscsi_hdr nopin_hdr;
	struct iscsi_hdr async_hdr;
};


/*
 * qfle3i_cmd - iscsi command structure
 *
 * @link:               list head to link elements
 * @iscsi_opcode:       iscsi command opcode, NOPIN, LOGIN, SCSICMD, etc'
 * @cmd_state:          command state tracking flag
 * @scsi_status_rcvd:   flag determines whether SCSI response is received
 *                      for this task or not
 * @scsi_cmd:           SCSI-ML task pointer corresponding to this iscsi cmd
 * @tmf_ref_itt:        reference ITT of the command being aborted
 * @tmf_ref_cmd:        pointer of the command being aborted by this command
 * @tmf_ref_sc:         SCSI-ML's task pointer of aborted command
 * @sg:                 SG list
 * @bd_tbl:             buffer descriptor (BD) table
 * @bd_tbl_dma:         buffer descriptor (BD) table's dma address
 */
typedef struct qfle3i_cmd {
	vmk_ListLinks link;
	vmk_uint8 iscsi_opcode;
	vmk_uint8 rsvd1;
	vmk_uint16 itt;
	vmk_atomic64 cmd_state;
		#define ISCSI_CMD_STATE_FREED			0x000
		#define ISCSI_CMD_STATE_INITIATED		0x001
		#define ISCSI_CMD_STATE_ABORT_REQ		0x002
		#define ISCSI_CMD_STATE_ABORT_PEND		0x004
		#define ISCSI_CMD_STATE_ABORT_COMPL		0x008
		#define ISCSI_CMD_STATE_CLEANUP_START		0x010
		#define ISCSI_CMD_STATE_CLEANUP_PEND		0x020
		#define ISCSI_CMD_STATE_CLEANUP_CMPL		0x040
		#define ISCSI_CMD_STATE_FAILED			0x100
		#define ISCSI_CMD_STATE_TMF_TIMEOUT		0x200
		#define ISCSI_CMD_STATE_CMPL_RCVD		0x400
		#define ISCSI_CMD_STATE_COMPLETED		0x800
	int scsi_status_rcvd;

	struct qfle3i_conn *conn;
	vmk_ScsiCommand *scsi_cmd;
	struct scatterlist *sg;
	struct io_bdt *bd_tbl;
	vmk_IOA bd_tbl_dma;
	vmk_uint32 reserved0;

	struct iscsi_cmd_request req;
	struct qfle3i_iscsilun *ilun;
	/* TMF RELATED */
	vmk_uint8 tmf_func;
	vmk_uint8 tmf_response;
	int tmf_lun;
	vmk_uint32 tmf_ref_itt;
	struct qfle3i_cmd *tmf_ref_cmd;
	vmk_ScsiCommand *tmf_ref_sc;
	int failed_reason;
	/* useful for nop-in processing */
	vmk_uint32 ttt;
	int app_cmd;
} qfle3i_cmd;

/*
 * TCP port manager
 */
struct tcp_port_mngt {
	int num_required;
	vmk_uint32 port_tbl_size;
	vmk_uint32 num_free_ports;
	vmk_uint32 prod_idx;
	vmk_uint32 cons_idx;
	vmk_uint32 max_idx;
	vmk_uint16 *free_q;
};

struct qfle3i_scsi_task {
	vmk_ListLinks link;
	vmk_ScsiCommand *scsi_cmd;
	struct qfle3i_iscsilun *ilun;
	int app_cmd;
};

/*
 * struct qfle3i_conn - iscsi connection structure
 *
 * @link:                  list head to link elements
 * @sess:                  iscsi session pointer
 * @cls_conn:              pointer to iscsi cls conn
 * @state:                 flag to trace command state
 * @stop_state:            stop state request by open-iscsi
 * @stage:                 iscsi login state
 * @in_shutdown:           flags to indicate connection is in shutdown mode
 * @lead_conn:             lead iscsi connection of session
 * @conn_cid:              iscsi cid per rfc
 * @exp_statsn:            iscsi expected statsn
 * @header_digest_en:      header digest parameter
 * @data_digest_en:        data digest parameter
 * @max_data_seg_len_xmit: iscsi initiator's mrdsl
 * @max_data_seg_len_recv: iscsi target's mrdsl
 * @ifmarker_enable:       ifmarker parameter
 * @ofmarker_enable:       ofmarker parameter
 * @persist_port:          iscsi target side TCP port number
 * @persist_address:       iscsi target's IP address
 * @iscsi_conn_cid:        iscsi conn id
 * @fw_cid:                firmware iscsi context id
 * @lock:                  lock to synchronize access
 * @ep:                    endpoint structure pointer
 * @gen_pdu:               login/nopout/logout pdu resources
 * @nopout_num_scsi_cmds:  scsi cmds issue counter to detect idle link
 * @total_data_octets_sent:conn stats - data bytes sent on this conn
 * @total_data_octets_rcvd:conn stats - data bytes received on this conn
 * @num_login_req_pdus:    conn stats - num login pdus sent
 * @num_login_resp_pdus:   conn stats - num login pdus received
 * @num_scsi_cmd_pdus:     conn stats - num scsicmd pdus sent
 * @num_scsi_resp_pdus:    conn stats - num scsicmd pdus received
 * @num_nopout_pdus:       conn stats - num nopout pdus sent
 * @num_nopin_pdus         conn stats - num nopout pdus received:
 * @num_reject_pdus:       conn stats - num reject pdus received
 * @num_async_pdus:        conn stats - num async pdus received
 * @num_dataout_pdus:      conn stats - num dout pdus sent
 * @num_r2t_pdus:          conn stats - num r2t pdus received
 * @num_datain_pdus:       conn stats - num din pdus received
 * @num_snack_pdus:        conn stats - num snack pdus received
 * @num_text_req_pdus:     conn stats - num text pdus sent
 * @num_text_resp_pdus:    conn stats - num text pdus received
 * @num_tmf_req_pdus:      conn stats - num tmf pdus sent
 * @num_tmf_resp_pdus:     conn stats - num tmf pdus received
 * @num_logout_req_pdus:   conn stats - num logout pdus sent
 * @num_logout_resp_pdus:  conn stats - num logout pdus received
 *
 * iSCSI connection structure
 */
typedef struct qfle3i_conn {
	vmk_ListLinks link;
	vmk_ListLinks scsi_scan_list;
	struct qfle3i_sess *sess;
	vmk_IscsiTransConnection *trans_conn;

	vmk_uint32 state;
		#define CONN_STATE_IDLE				0x00
		#define CONN_STATE_XPORT_READY			0x01
		#define CONN_STATE_IN_LOGIN			0x02
		#define CONN_STATE_FFP_STATE			0x04
		#define CONN_STATE_IN_LOGOUT			0x08
		#define CONN_STATE_IN_CLEANUP			0x10
		#define CONN_STATE_XPORT_FREEZE			0x20
		#define CONN_STATE_STOPPED			0x80
	vmk_atomic64 stop_state;
	vmk_uint32 stage;
	vmk_uint32 in_shutdown;

	vmk_uint32 lead_conn;
	vmk_uint32 conn_cid;

	vmk_Timer poll_timer;
	void (*ring_doorbell)(struct qfle3i_conn *);
	/*
	 * Following are iSCSI sequencing & operational parameters
	 */
	vmk_uint32 exp_statsn;
		#define STATSN_UPDATE_SIGNATURE		0xFABCAFE
	vmk_uint32 header_digest_en;
	vmk_uint32 data_digest_en;
	vmk_uint32 max_data_seg_len_xmit;	/* Target */
	vmk_uint32 max_data_seg_len_recv;	/* Initiator */
	int ifmarker_enable;
	int ofmarker_enable;
	int persist_port;
	char *persist_address;

	vmk_Bool reject_recvd;

	vmk_uint32 iscsi_conn_cid;
		#define QFLE3I_CID_RESERVED	0x5AFF
	vmk_uint32 fw_cid;

	/*
	 * Queue Pair (QP) related structure elements.
	 */
	struct qfle3i_endpoint *ep;

	vmk_atomic64 worker_running;
	vmk_atomic64 worker_enabled;
	vmk_atomic64 worker_enabled_cnt;
	vmk_atomic64 worker_disabled_cnt;
	ql_vmk_tasklet_t conn_tasklet;
	char isid[13];
/* DEBUG ONLY */
	vmk_uint32 tasklet_freeze;
	int tasklet_state;
	int tasklet_tmf_exit;
	int tasklet_timeslice_exit;
	int tasklet_reschedule;
	int tasklet_entry;
	int cqe_process_state;
	unsigned long cqe_process_jiffies;
	int tasklet_loop;
	unsigned long que_jiff;
	unsigned long cqe_jiff;
	unsigned long task_jiff;
	vmk_atomic64 lastSched;

	/*
	 * Buffer for login negotiation process
	 */
	struct generic_pdu_resc gen_pdu;

	vmk_uint32 nopout_num_scsi_cmds;
	/*
	 * Connection Statistics
	 */
	vmk_uint64 total_data_octets_sent;
	vmk_uint64 total_data_octets_rcvd;
	vmk_uint32 num_login_req_pdus;
	vmk_uint32 num_login_resp_pdus;
	vmk_uint32 num_scsi_cmd_pdus;
	vmk_uint32 num_scsi_resp_pdus;
	vmk_uint32 num_nopout_pdus;
	vmk_uint32 num_nopin_pdus;
	vmk_uint32 num_reject_pdus;
	vmk_uint32 num_async_pdus;
	vmk_uint32 num_dataout_pdus;
	vmk_uint32 num_r2t_pdus;
	vmk_uint32 num_datain_pdus;
	vmk_uint32 num_snack_pdus;
	vmk_uint32 num_text_req_pdus;
	vmk_uint32 num_text_resp_pdus;
	vmk_uint32 num_tmf_req_pdus;
	vmk_uint32 num_tmf_resp_pdus;
	vmk_uint32 num_logout_req_pdus;
	vmk_uint32 num_logout_resp_pdus;
}VMK_ATTRIBUTE_L1_ALIGNED qfle3i_conn;

typedef struct qfle3i_iscsilun {
	vmk_ListLinks link;

	struct qfle3i_sess *sess;
	unsigned long io_cnt;   /* total xfer count since boot */
	int q_depth;
	int channel_id;
	int target_id;
	int lun_id;

	/* In Native, IOs are not blocked when TMF command is
	   active, Driver has to take care of it by its own.
	*/
	vmk_Bool tmf_in_progress;
}VMK_ATTRIBUTE_L1_ALIGNED qfle3i_iscsilun;
/*
 * struct qfle3i_sess - iscsi session structure
 *
 * @link:                  list head to link elements
 * @hba:                   adapter structure pointer
 * @shost:                 scsi host pointer
 * @state:                 flag to track session state
 * @recovery_state:        recovery state identifier
 * @old_recovery_state:    old recovery state identifier
 * @tmf_active:            TMF is active on this session
 * @lock:                  session lock to synchronize access
 * @abort_timer:           TMF timer
 * @er_wait:               wait queue for recovery process
 * @cmd_pages:             table to track pages allocated for cmd struct
 * @pend_cmds:             pend command list
 * @num_pend_cmds:         number of pend command
 * @free_cmds:             free command list
 * @num_free_cmds:         num free commands
 * @allocated_cmds:        total number of allocated commands
 * @sq_size:               SQ size
 * @itt_q:                 ITT queue
 * @bd_resc_page:          table to track BD resource page memory
 * @bd_tbl_list:           BD table list
 * @bd_tbl_active:         active BD table list
 * @active_cmds:           active command list
 * @num_active_cmds:       num active commands
 * @cmdsn:                 iscsi command sequence number
 * @exp_cmdsn:             iscsi expected command sequence number
 * @max_cmdsn:             iscsi max command sequence number
 * @initial_r2t:           intial R2T is enabled/disable
 * @max_r2t:               maximun outstanding T2T 
 * @imm_data:              indicates if immediate data is enabled
 * @first_burst_len:       negotiated first burst length
 * @max_burst_len:         negotiated max burst length
 * @time2wait:             time 2 wait value
 * @time2retain:           time 2 retain value
 * @pdu_inorder:           indicated if PDU order needs to be maintained
 * @dataseq_inorder:       indicated if data sequence order needs to be
 *                         maintained
 * @erl:                   supported error recovery level
 * @tgt_prtl_grp:          target portal group tag
 * @target_name:           target name
 * @isid:                  isid for this session
 * @tsih:                  target returned TSIH
 * @lead_conn:             points to lead connection pointer
 * @conn_list:             list of connection belonging to this session
 * @num_active_conn:       num active connections
 * @max_conns:             maximun connection per session
 * @violation_notified:    bit mask used to track iscsi error/warning messages
 *                         already printed out
 * iSCSI Session Structure
 */
typedef struct qfle3i_sess {
	vmk_ListLinks link;
	struct qfle3i_hba *hba;
	vmk_IscsiTransSession *trans_sess;
	vmk_IscsiTransTransport  *trans;
	unsigned long timestamp;
	unsigned long worker_time_slice;
	vmk_uint32 state;
		#define QFLE3I_SESS_INITIAL		0x01
		#define QFLE3I_SESS_IN_FFP		0x02
		#define QFLE3I_SESS_IN_RECOVERY		0x04
		#define QFLE3I_SESS_IN_SHUTDOWN		0x08
		#define QFLE3I_SESS_IN_LOGOUT		0x40
		/* Do not notify device offline to vmkernel to until
		 * iscsi transport calls destroy_session()
		 */
		#define QFLE3I_SESS_DESTROYED		0x80
		/* if session encounters an error before transitioning
		 * to FFP, target_destroy() will be called before
		 * session_destroy() and this requires another flag
		 * to identify this to make adjustments as to how
		 * resources will be freed
		 */
		#define QFLE3I_SESS_TARGET_DESTROYED	0x100
		#define is_sess_active(_sess)	\
			(((_sess)->state & QFLE3I_SESS_IN_FFP))
	unsigned long recovery_state;
		#define ISCSI_SESS_RECOVERY_START	0x01
		#define ISCSI_SESS_RECOVERY_OPEN_ISCSI	0x02
		#define ISCSI_SESS_RECOVERY_COMPLETE 	0x04
		#define ISCSI_SESS_RECOVERY_FAILED	0x08
		#define ISCSI_SESS_RECOVERY_MASK	0xFFFF
	unsigned long old_recovery_state;
	vmk_atomic64 tmf_active;
	vmk_atomic64 do_recovery_inprogess;
	vmk_atomic64 device_offline;

#ifndef _USE_ITT_QUE
	struct qfle3i_cmd **itt_cmd;
		#define get_cmnd(sess, itt)	sess->itt_cmd[itt]
#endif

	vmk_Lock lock;	/* protects session structure */
	vmk_Semaphore tmf_mutex;
	vmk_Lock device_lock;	/* serialize device unblock/offline */

	/* Command abort timer */
	vmk_Timer abort_timer;
	/* event wait queue used during error recovery */
	struct ql_vmk_cmpl er_wait;

	/*
	 * Per session command (task) structure management
	 */
	void *cmd_pages[MAX_PAGES_PER_CTRL_STRUCT_POOL];
	vmk_ListLinks free_cmds;
	int num_free_cmds;
	int allocated_cmds;
	int total_cmds_allocated;
	int total_cmds_freed;

	int sq_size;
#ifdef _USE_ITT_QUEUE
	struct itt_queue itt_q;
		#define MAX_BD_RESOURCE_PAGES		8
#endif

	vmk_ListLinks bd_resc_page;
	void *bdt_dma_info;
	vmk_ListLinks bdt_dma_resc;
	vmk_ListLinks bd_tbl_list;
	vmk_ListLinks bd_tbl_active;

	/*
	 * command queue management
	 */
	vmk_atomic64 login_noop_pending;
	vmk_atomic64 tmf_pending;
	vmk_atomic64 logout_pending;
	vmk_atomic64 nop_resp_pending;
	struct qfle3i_cmd *login_nopout_cmd;
	struct qfle3i_cmd *scsi_tmf_cmd;
	struct qfle3i_cmd *nopout_resp_cmd;

	void *task_list_mem;
	vmk_ListLinks scsi_task_list;
	vmk_ListLinks pend_cmd_list;
	vmk_uint32 pend_cmd_count;
	vmk_ListLinks active_cmd_list;
	vmk_uint32 active_cmd_count;
	int cmd_cleanup_req;
	int cmd_cleanup_cmpl;

	/* Debug counter */
	vmk_uint32 total_cmds_sent;
	vmk_uint32 total_cmds_queued;
	vmk_uint32 total_cmds_completed;
	vmk_uint32 total_cmds_failed;
	vmk_uint32 total_cmds_completed_by_chip;
	vmk_uint32 cmd_win_closed;
	vmk_uint32 host_busy_cmd_win;
	vmk_uint32 alloc_scsi_task_failed;

	vmk_uint32 max_cmds;
	vmk_uint32 q_depth;
	vmk_uint32 target_id;
	vmk_uint32 channel_id;

	/* list to store iscsilun structure */
	vmk_ListLinks iscsilun_list;
	vmk_Lock iscsilun_lock;
	/*
	 * iSCSI session related sequencing parameters.
	 */
	unsigned int cmdsn;
	unsigned int exp_cmdsn;
	unsigned int max_cmdsn;

	/*
	 * Following pointers are linked to corresponding entry in
	 * operational parameter table associated with this session.
	 * These are to be filled when session becomes operational (FFP).
	 */
	int initial_r2t;
	int max_r2t;
	int imm_data;
	vmk_uint32 first_burst_len;
	vmk_uint32 max_burst_len;
	int time2wait;
	int time2retain;
	int pdu_inorder;
	int dataseq_inorder;
	int erl;
	int tgt_prtl_grp;
	char *target_name;

	unsigned char isid[13];
	unsigned short tsih;

	struct qfle3i_conn *lead_conn;
	vmk_ListLinks conn_list;
	vmk_uint32 num_active_conn;
	vmk_uint32 max_conns;

	/* Driver private statistics */
	vmk_uint64 violation_notified;

	unsigned long last_nooput_requested;
	unsigned long last_nooput_posted;
	unsigned long last_noopin_indicated;
	unsigned long last_noopin_processed;
	vmk_uint32 last_nooput_sn;
	vmk_uint32 noopout_resp_count;
	vmk_uint32 unsol_noopout_count;
	int noopout_requested_count;
	int noopout_posted_count;
	int noopin_indicated_count;
	int noopin_processed_count;
	int tgt_noopin_count;
	int scsi_scan_counter;
	vmk_uint32 max_iscsi_tasks;

	/* session block/unbock state has to be maintained
	   by driver. this flag will help to return IO commands
	   with WOULD_BLOCK status in queuecommand function.
	*/
	vmk_Bool if_unblocked;
}VMK_ATTRIBUTE_L1_ALIGNED qfle3i_sess;

/*
 * struct iscsi_cid_queue - Per adapter iscsi cid queue
 *
 * @cid_que_base:           queue base memory
 * @cid_que:                queue memory pointer
 * @cid_q_prod_idx:         produce index
 * @cid_q_cons_idx:         consumer index
 * @cid_q_max_idx:          max index. used to detect wrap around condition
 * @cid_free_cnt:           queue size
 * @conn_cid_tbl:           iscsi cid to conn structure mapping table
 *
 * Per adapter iSCSI CID Queue
 */
struct iscsi_cid_queue {
	void *cid_que_base;
	vmk_uint32 *cid_que;
	vmk_uint32 cid_q_prod_idx;
	vmk_uint32 cid_q_cons_idx;
	vmk_uint32 cid_q_max_idx;
	vmk_uint32 cid_free_cnt;
	struct qfle3i_conn **conn_cid_tbl;
};

struct iscsi_login_stats_info {
	vmk_uint32 successful_logins;			/* Total login successes */
	vmk_uint32 login_failures;			/* Total login failures */
	vmk_uint32 login_negotiation_failures;		/* Text negotiation failed */
	vmk_uint32 login_authentication_failures;	/* login Authentication failed */
	vmk_uint32 login_redirect_responses;		/* Target redirects to another portal */
	vmk_uint32 connection_timeouts;		/* TCP connection timeouts */
	vmk_uint32 session_failures;			/* Errors resulting in sess recovery */
	vmk_uint32 digest_errors;			/* Errors resulting in digest errors */
};

/*******************************************************************************
 * 	QP [ SQ / RQ / CQ ] info.
 ******************************************************************************/

/*
 * SQ/RQ/CQ generic structure definition
 */
struct 	sqe {
	vmk_uint8 sqe_byte[QFLE3I_SQ_WQE_SIZE];
};

struct 	rqe {
	vmk_uint8 rqe_byte[QFLE3I_RQ_WQE_SIZE];
};

struct 	cqe {
	vmk_uint8 cqe_byte[QFLE3I_CQE_SIZE];
};


enum {
#if defined(__LITTLE_ENDIAN)
	CNIC_EVENT_COAL_INDEX	= 0x0,
	CNIC_SEND_DOORBELL	= 0x4,
	CNIC_EVENT_CQ_ARM	= 0x7,
	CNIC_RECV_DOORBELL	= 0x8
#elif defined(__BIG_ENDIAN)
	CNIC_EVENT_COAL_INDEX	= 0x2,
	CNIC_SEND_DOORBELL	= 0x6,
	CNIC_EVENT_CQ_ARM	= 0x4,
	CNIC_RECV_DOORBELL	= 0xa
#endif
};



/*
 * CQ DB
 */
struct bnx2x_iscsi_cq_pend_cmpl {
	/* CQ producer, updated by Ustorm */
        vmk_uint16 ustrom_prod;
	/* CQ pending completion counter */
        vmk_uint16 pend_cntr;
};


struct qfle3i_5771x_cq_db {
        struct bnx2x_iscsi_cq_pend_cmpl qp_pend_cmpl[BNX2X_MAX_CQS];
	/* CQ pending completion ITT array */
        vmk_uint16 itt[BNX2X_MAX_CQS];
	/* Cstorm CQ sequence to notify array, updated by driver */;
        vmk_uint16 sqn[BNX2X_MAX_CQS];
        vmk_uint32 reserved[4] /* 16 byte allignment */;
};


struct qfle3i_5771x_sq_rq_db {
	vmk_uint16 prod_idx;
	vmk_uint8 reserved0[62]; /* Pad structure size to 64 bytes */
};


struct qfle3i_5771x_dbell_hdr {
        vmk_uint8 header;
	/* 1 for rx doorbell, 0 for tx doorbell */
#define B577XX_DOORBELL_HDR_RX				(0x1<<0)
#define B577XX_DOORBELL_HDR_RX_SHIFT			0
	/* 0 for normal doorbell, 1 for advertise wnd doorbell */
#define B577XX_DOORBELL_HDR_DB_TYPE			(0x1<<1)
#define B577XX_DOORBELL_HDR_DB_TYPE_SHIFT		1
	/* rdma tx only: DPM transaction size specifier (64/128/256/512B) */
#define B577XX_DOORBELL_HDR_DPM_SIZE			(0x3<<2)
#define B577XX_DOORBELL_HDR_DPM_SIZE_SHIFT		2
	/* connection type */
#define B577XX_DOORBELL_HDR_CONN_TYPE			(0xF<<4)
#define B577XX_DOORBELL_HDR_CONN_TYPE_SHIFT		4
};

struct qfle3i_5771x_dbell {
	struct qfle3i_5771x_dbell_hdr dbell;
	vmk_uint8 pad[3];

};


/*
 * struct qp_info - QP (share queue region) atrributes structure
 *
 * @ctx_base:           ioremapped pci register base to access doorbell register
 *                      pertaining to this offloaded connection
 * @sq_virt:            virtual address of send queue (SQ) region
 * @sq_phys:            DMA address of SQ memory region
 * @sq_mem_size:        SQ size
 * @sq_prod_qe:         SQ producer entry pointer
 * @sq_cons_qe:         SQ consumer entry pointer
 * @sq_first_qe:        virtaul address of first entry in SQ
 * @sq_last_qe:         virtaul address of last entry in SQ
 * @sq_prod_idx:        SQ producer index
 * @sq_cons_idx:        SQ consumer index
 * @sqe_left:           number sq entry left
 * @sq_pgtbl_virt:      page table describing buffer consituting SQ region
 * @sq_pgtbl_phys:      dma address of 'sq_pgtbl_virt'
 * @sq_pgtbl_size:      SQ page table size
 * @cq_virt:            virtual address of completion queue (CQ) region
 * @cq_phys:            DMA address of RQ memory region
 * @cq_mem_size:        CQ size
 * @cq_prod_qe:         CQ producer entry pointer
 * @cq_cons_qe:         CQ consumer entry pointer
 * @cq_first_qe:        virtaul address of first entry in CQ
 * @cq_last_qe:         virtaul address of last entry in CQ
 * @cq_prod_idx:        CQ producer index
 * @cq_cons_idx:        CQ consumer index
 * @cqe_left:           number cq entry left
 * @cqe_size:           size of each CQ entry
 * @cqe_exp_seq_sn:     next expected CQE sequence number
 * @cq_pgtbl_virt:      page table describing buffer consituting CQ region  
 * @cq_pgtbl_phys:      dma address of 'cq_pgtbl_virt'  
 * @cq_pgtbl_size:    	CQ page table size    
 * @rq_virt:            virtual address of receive queue (RQ) region
 * @rq_phys:            DMA address of RQ memory region
 * @rq_mem_size:        RQ size
 * @rq_prod_qe:         RQ producer entry pointer
 * @rq_cons_qe:         RQ consumer entry pointer
 * @rq_first_qe:        virtaul address of first entry in RQ
 * @rq_last_qe:         virtaul address of last entry in RQ
 * @rq_prod_idx:        RQ producer index
 * @rq_cons_idx:        RQ consumer index
 * @rqe_left:           number rq entry left
 * @rq_pgtbl_virt:      page table describing buffer consituting RQ region
 * @rq_pgtbl_phys:      dma address of 'rq_pgtbl_virt'
 * @rq_pgtbl_size:      RQ page table size
 *
 * queue pair (QP) is a per connection shared data structure which is used
 *	to send work requests (SQ), receive completion notifications (CQ)
 *	and receive asynchoronous / scsi sense info (RQ). 'qp_info' structure
 *	below holds queue memory, consumer/producer indexes and page table
 *	information
 */
struct qp_info {
#define DPM_TRIGER_TYPE			0x40

#define QFLE3I_570x_QUE_DB_SIZE		0
#define QFLE3I_5771x_QUE_DB_SIZE		16
	struct qfle3i_dma sq_dma;
#define sq_virt		sq_dma.mem
	struct sqe *sq_prod_qe;
	struct sqe *sq_first_qe;
	struct sqe *sq_last_qe;
	vmk_uint16 sq_prod_idx;

	struct qfle3i_dma cq_dma;
#define cq_virt		cq_dma.mem
	struct cqe *cq_cons_qe;
	struct cqe *cq_first_qe;
	struct cqe *cq_last_qe;
	vmk_uint16 cq_cons_idx;
	vmk_uint32 cqe_left;
	vmk_uint32 cqe_size;
	vmk_uint32 cqe_exp_seq_sn;

	struct qfle3i_dma rq_dma;
#define rq_virt		rq_dma.mem

	struct rqe *rq_prod_qe;
	struct rqe *rq_cons_qe;
	struct rqe *rq_first_qe;
	struct rqe *rq_last_qe;
	vmk_uint16 rq_prod_idx;
	vmk_uint16 rq_cons_idx;
	vmk_uint32 rqe_left;

	vmk_uint64  ctx_base;
};


/*
 * CID handles
 */
struct ep_handles {
	vmk_uint32 fw_cid;
	vmk_uint32 drv_iscsi_cid;
	vmk_uint16 pg_cid;
	vmk_uint16 rsvd;
};

/*
 * struct qfle3i_endpoint - representation of tcp connection in NX2 world
 *
 * @link:               list head to link elements
 * @hba:                adapter to which this connection belongs
 * @conn:               iscsi connection this EP is linked to
 * @sess:               iscsi session this EP is linked to
 * @cm_sk:              cnic sock struct
 * @hba_age:            age to detect if 'iscsid' issues ep_disconnect()
 *                      after HBA reset is completed by qfle3i/cnic/bnx2
 *                      modules
 * @state:              tracks offload connection state machine
 * @tcp_port:           Local TCP port number used in this connection
 * @qp:                 QP information
 * @ids:                contains chip allocated *context id* & driver assigned
 *                      *iscsi cid*
 * @ofld_timer:         offload timer to detect timeout
 * @ofld_wait:          wait queue
 *
 * Endpoint Structure - equivalent of tcp socket structure
 */
struct qfle3i_endpoint {
	vmk_ListLinks link;
	struct qfle3i_hba *hba;
	struct qfle3i_conn *conn;
	struct qfle3i_sess *sess;
	struct cnic_sock *cm_sk;
	vmk_uint32 hba_age;
	vmk_uint32 state;
		#define EP_STATE_IDLE			0x00000000
		#define EP_STATE_PG_OFLD_START		0x00000001
		#define EP_STATE_PG_OFLD_COMPL		0x00000002
		#define EP_STATE_OFLD_START		0x00000004
		#define EP_STATE_OFLD_COMPL		0x00000008
		#define EP_STATE_CONNECT_START		0x00000010
		#define EP_STATE_CONNECT_COMPL		0x00000020
		#define EP_STATE_ULP_UPDATE_START	0x00000040
		#define EP_STATE_ULP_UPDATE_COMPL	0x00000080
		#define EP_STATE_DISCONN_START		0x00000100
		#define EP_STATE_DISCONN_COMPL		0x00000200
		#define EP_STATE_CLEANUP_START		0x00000400
		#define EP_STATE_CLEANUP_CMPL		0x00000800
		#define EP_STATE_TCP_FIN_RCVD		0x00001000
		#define EP_STATE_TCP_RST_RCVD		0x00002000
		#define EP_STATE_ULP_UPDATE_TIMEOUT	0x00004000
		#define EP_STATE_PG_OFLD_FAILED		0x01000000
		#define EP_STATE_ULP_UPDATE_FAILED	0x02000000
		#define EP_STATE_CLEANUP_FAILED		0x04000000
		#define EP_STATE_OFLD_FAILED		0x08000000
		#define EP_STATE_CONNECT_FAILED		0x10000000
		#define EP_STATE_DISCONN_TIMEDOUT	0x20000000
		#define EP_STATE_OFLD_FAILED_CID_BUSY	0x80000000

	unsigned long timestamp;
	int teardown_mode;
#define QFLE3I_ABORTIVE_SHUTDOWN		0
#define QFLE3I_GRACEFUL_SHUTDOWN		1
	vmk_uint16 tcp_port;

	vmk_atomic64 fp_kcqe_events;
	struct qp_info qp;
	struct ep_handles ids;
		#define ep_iscsi_cid	ids.drv_iscsi_cid
		#define ep_cid		ids.fw_cid
		#define ep_pg_cid	ids.pg_cid
	vmk_Timer ofld_timer;
	struct ql_vmk_cmpl ofld_wait;
	vmk_uint32 in_progress;
};

#if 0

static inline struct Scsi_Host *qfle3i_conn_get_shost(struct qfle3i_conn *conn)
{
	struct Scsi_Host *shost;

	shost = conn->sess->hba->shost;
	return shost;
}

static inline struct Scsi_Host *qfle3i_sess_get_shost(struct qfle3i_sess *sess)
{
	struct Scsi_Host *shost;

	shost = sess->hba->shost;
	return shost;
}

#endif

enum qfle3i_lock_rank_t{
        QFLE3I_LOW_LOCK_RANK = 1,
        QFLE3I_HIGH_LOCK_RANK,
};


extern struct qfle3i_driver_info_t qfle3i_driver_info;
extern vmk_LogComponent qfle3i_vmkLog;

struct qfle3i_driver_info_t {
	vmk_ModuleID module_id;
	vmk_Name driver_name;
	vmk_HeapID heap_id_dma;
	vmk_HeapID heap_id;
	vmk_Driver qfle3i_driver;
	vmk_LockDomainID lock_domain;
	vmk_Lock drv_lock;
	vmk_MgmtHandle kv_mgmt_handle;
	vmk_DumpFileHandle dump_handler;
	vmk_TimerQueue timer_queue;
	ql_vmk_singlethread_workqueue_t *delayed_wq;
};

struct qla_dmamem {
	vmk_SgOpsHandle SgOps_handle;
	vmk_SgOpsHandle SgOps_handle_out;
	vmk_SgArray *MachSg;
	vmk_uint32 dma_size;
	void *virtaddr;
};

typedef struct qfle3i_hba {
	vmk_ListLinks link;

	vmk_uint32 host_num;
	vmk_uint32 q_depth;
	vmk_Device ldev;
	vmk_Device iSCSIDevice;
	//vmk_DMAEngine scsiDmaEngine;
	//qfle3_adapter *net_adapter;
	vmk_ScsiAdapter *scsiAdapter;

	/* Transport */
	vmk_IscsiTransTransport *iscsiTransport;
	vmk_Bool isRegisteredWithTransport;
	vmk_MgmtHandle kvMgmtHandle;
	vmk_Bool mgmtKeyAdded;

	/* PCI device */
	vmk_PCIDevice pcidev;
	vmk_PCIDeviceID pcidev_id;
	vmk_PCIDeviceAddr sbdf;
	vmk_Name vmnic_name;
	vmk_uint16 pci_did;

	vmk_SgOpsHandle	SgOps_handle;
	vmk_DMAEngine	dmaEngine;

	struct cnic_dev *cnic;
	struct net_device *netdev;
 	void *regview;
	void *doorbells;
	vmk_uint32 age;
	vmk_BitVector *cnic_dev_type;
		#define QFLE3I_NX2_DEV_5706		0x0
		#define QFLE3I_NX2_DEV_5708		0x1
		#define QFLE3I_NX2_DEV_5709		0x2
		#define QFLE3I_NX2_DEV_57710		0x3
		#define CNIC_DEV_TYPE_MASK		0xFFFF
	vmk_uint32 mail_queue_access;
		#define QFLE3I_MQ_KERNEL_MODE		0x0
		#define QFLE3I_MQ_KERNEL_BYPASS_MODE	0x1
		#define QFLE3I_MQ_BIN_MODE		0x2
	vmk_BitVector  *reg_with_cnic;
		#define QFLE3I_CNIC_REGISTERED		1

	vmk_BitVector *adapter_state;
		#define ADAPTER_STATE_UP		0
		#define ADAPTER_STATE_GOING_DOWN	1
		#define ADAPTER_STATE_LINK_DOWN		2
		#define ADAPTER_STATE_FW_RECOVERY	4
		#define ADAPTER_STATE_INIT_FAILED	31
		#define QFLE3I_DEV_STATE_MASK		0xFFFF
	vmk_uint32 mtu_supported;
		#define QFLE3I_MAX_MTU_SUPPORTED		9000

	vmk_MemPool qfle3i_pool;
		#define BRCM_ISCSI_XPORT_NAME_PREFIX		"qfle3i"
		#define BRCM_ISCSI_XPORT_NAME_SIZE_MAX		128

	vmk_uint32 target_id;
	vmk_uint32 channel_id;
	//struct device vm_pcidev;
	vmk_uint32 max_sqes;
	vmk_uint32 max_rqes;
	vmk_uint32 max_cqes;
	vmk_uint32 num_ccell;

	vmk_Timer hba_poll_timer;
	/* different page table setup requirments for 5771x and 570x */
	void (*setup_pgtbl)(struct qfle3i_hba *, struct qfle3i_dma *, vmk_int32);

	vmk_ListLinks active_sess;
	vmk_int32 num_active_sess;
	vmk_int32 ofld_conns_active;

	vmk_int32 max_active_conns;
	struct iscsi_cid_queue cid_que;
	vmk_Lock cid_que_lock;
	vmk_Lock scsi_scan_list_lck;
	vmk_ListLinks scsi_scan_list;

	vmk_ListLinks ep_ofld_list;
	vmk_ListLinks ep_destroy_list;

	/*
	 * BD table to be used with MP (Middle Path requests.
	 */
	struct qfle3i_dma mp_dma_buf;

	vmk_Lock lock;	/* protects hba structure access */
	vmk_Semaphore net_dev_lock;/* sync net device access */

	/* Error handling */
	vmk_Timer hba_timer;
	struct ql_vmk_cmpl eh_wait;
	ql_vmk_singlethread_workitem_t err_rec_task;
	ql_vmk_singlethread_workitem_t link_update;
	struct qfle3i_sess **sess_recov_list;
	vmk_int32 sess_recov_prod_idx;
	vmk_int32 sess_recov_cons_idx;
	vmk_int32 sess_recov_max_idx;
	vmk_uint8	mac_addr[VMK_ETH_ADDR_LENGTH];

	vmk_int32 conn_teardown_tmo;
	vmk_int32 conn_ctx_destroy_tmo;
	vmk_int32 hba_shutdown_tmo;
	vmk_uint32 ctx_ccell_tasks;
	/*
	 * Following are a bunch of statistics useful during development
	 * and later stage for score boarding.
	 */
	vmk_uint32 num_wqe_sent;
	vmk_uint32 num_cqe_rcvd;
	vmk_uint32 num_intr_claimed;
	vmk_uint32 link_changed_count;
	vmk_uint32 ipaddr_changed_count;
	vmk_uint32 num_sess_opened;
	vmk_uint32 num_conn_opened;
	vmk_uint32 stop_event_ifc_abort_poll;
	vmk_uint32 stop_event_ifc_abort_bind;
	vmk_uint32 stop_event_ifc_abort_login;
	vmk_uint32 stop_event_ep_conn_failed;
	vmk_uint32 stop_event_repeat;
	vmk_uint32 task_cleanup_failed;
	vmk_uint32 tcp_error_kcqes;
	vmk_uint32 iscsi_error_kcqes;
	vmk_Name vmhba_name;

	vmk_Lock stat_lock;
	struct iscsi_stats_info stats;
	struct iscsi_login_stats_info login_stats;

	vmk_ListLinks ep_stale_list;

	/* conn disconnect timeout handling */
	struct ql_vmk_cmpl ep_tmo_wait;
	vmk_ListLinks ep_tmo_list;
	ql_vmk_singlethread_workitem_t ep_poll_task;
	vmk_atomic64 ep_tmo_poll_enabled;
	vmk_uint32    ep_tmo_active_cnt;
	vmk_uint32    ep_tmo_cmpl_cnt;
	vmk_uint32    max_scsi_task_queued;
	vmk_uint32 instance;

	/* iscsi_host_param*/
	char *name_string;
	char *vmkdev_name;

	ql_vmk_tasklet_t scan_bh;

	vmk_WorldEventID scsi_pass_comp;
	struct qla_dmamem ioctl_mem_phys;
	void *ioctl_mem;
	vmk_Lock ioctl_cmpl_lock;
	vmk_uint8 SCSIPT_InProgress;
	vmk_uint16 cmpl_done;
	vmk_uint16 ioctl_cmpl_sem;
	vmk_uint16 is_nic_up;
} VMK_ATTRIBUTE_L1_ALIGNED qfle3i_hba;


/* qfle3i_main */
VMK_ReturnStatus qfle3i_dma_map(
			vmk_DMAEngine dma_engine,
			vmk_DMADirection direction,
			vmk_MA ma,
			vmk_ByteCountSmall length,
			vmk_IOA *ioa);
VMK_ReturnStatus qfle3i_dma_unmap(
				vmk_DMAEngine dma_engine,
				vmk_DMADirection direction,
				vmk_IOA ioa,
				vmk_ByteCountSmall length);
VMK_ReturnStatus qfle3i_spin_lock_init(vmk_Lock *lock);

VMK_ReturnStatus qfle3i_setup_scsiadapter(qfle3i_hba *adapter);

VMK_ReturnStatus qfle3i_register_adapter(qfle3i_hba *adapter);
VMK_ReturnStatus qfle3i_unregister_adapter(qfle3i_hba *adapter);

VMK_ReturnStatus qfle3i_register_transport(qfle3i_hba *adapter);
VMK_ReturnStatus qfle3i_unregister_transport(qfle3i_hba *adapter);

extern unsigned int cmd_cmpl_per_work;
extern int qfle3i_reg_device;

void qfle3i_identify_device(struct qfle3i_hba *hba, struct cnic_dev *dev);
struct qfle3i_hba *qfle3i_alloc_hba(struct cnic_dev *cnic, vmk_Device device);
void qfle3i_check_nx2_dev_busy(void);
void qfle3i_get_link_state(struct qfle3i_hba *hba);
void qfle3i_start(void *handle);
void qfle3i_link_update(void *handle, unsigned long);
void qfle3i_stop(void *handle);
void qfle3i_fw_recovery_event(void *handle, int event);

void qfle3i_add_hba_to_adapter_list(struct qfle3i_hba *hba);
void qfle3i_remove_hba_from_adapter_list(struct qfle3i_hba *hba);

struct qfle3i_conn *qfle3i_get_conn_from_id(struct qfle3i_hba *hba,
					  vmk_uint16 iscsi_cid);

int qfle3i_alloc_ep_pool(void);
void qfle3i_release_ep_pool(void);
struct qfle3i_endpoint *qfle3i_ep_ofld_list_next(struct qfle3i_hba *hba);
struct qfle3i_endpoint *qfle3i_ep_destroy_list_next(struct qfle3i_hba *hba);

struct qfle3i_cmd *qfle3i_alloc_cmd(struct qfle3i_sess *sess);
void qfle3i_free_cmd(struct qfle3i_sess *sess, struct qfle3i_cmd *cmd);
int qfle3i_tcp_conn_active(struct qfle3i_conn *conn);


struct qfle3i_hba *qfle3i_find_hba_for_cnic(struct cnic_dev *cnic);
void qfle3i_free_hba(struct qfle3i_hba *hba);
int qfle3i_process_new_cqes(struct qfle3i_conn *conn, int soft_irq, int num_cqes);
void qfle3i_process_scsi_resp(struct qfle3i_cmd *cmd,
			     struct iscsi_cmd_response *resp_cqe);
int qfle3i_process_nopin(struct qfle3i_conn *conn,
	struct qfle3i_cmd *cmnd, char *data_buf, int data_len);


void qfle3i_update_cmd_sequence(struct qfle3i_sess *sess, vmk_uint32 expsn, vmk_uint32 maxsn);

void qfle3i_get_rq_buf(struct qfle3i_conn *conn, char *ptr, int len);
void qfle3i_put_rq_buf(struct qfle3i_conn *conn, int count);
int qfle3i_indicate_login_resp(struct qfle3i_conn *conn);
int qfle3i_indicate_logout_resp(struct qfle3i_conn *conn);
int qfle3i_indicate_async_mesg(struct qfle3i_conn *conn);
void qfle3i_start_iscsi_hba_shutdown(struct qfle3i_hba *hba);
void qfle3i_iscsi_hba_cleanup(struct qfle3i_hba *hba);
void qfle3i_iscsi_unmap_sg_list(struct qfle3i_hba *hba, struct qfle3i_cmd *cmd);
int qfle3i_do_iscsi_sess_recovery(struct qfle3i_sess *sess, int err_code, int signal);

#if 0
void qfle3i_iscsi_handle_ip_event(struct qfle3i_hba *hba);
void qfle3i_cleanup_tcp_port_mngr(void);
int qfle3i_init_tcp_port_mngr(void);
void qfle3i_init_ctx_dump_mem(struct qfle3i_hba *hba);
void qfle3i_free_ctx_dump_mem(struct qfle3i_hba *hba);
#endif
void qfle3i_return_failed_command(struct qfle3i_sess *sess,
				 vmk_ScsiCommand *cmd, int resid, int err_code);
void qfle3i_fail_cmd(struct qfle3i_sess *sess, struct qfle3i_cmd *cmd);
int qfle3i_complete_cmd(struct qfle3i_sess *sess, struct qfle3i_cmd *cmd);
int qfle3i_alloc_dma(struct qfle3i_hba *hba, struct qfle3i_dma *dma,
		    int size, int pgtbl_type, int pgtbl_off);
void qfle3i_free_dma(struct qfle3i_hba *hba, struct qfle3i_dma *dma);
int qfle3i_setup_mp_bdt(struct qfle3i_hba *hba);
void qfle3i_free_mp_bdt(struct qfle3i_hba *hba);

extern int qfle3i_send_fw_iscsi_init_msg(struct qfle3i_hba *hba);

extern int qfle3i_send_iscsi_login(struct qfle3i_conn *conn,
				  struct qfle3i_cmd *cmnd);
extern int qfle3i_send_iscsi_nopout(struct qfle3i_conn *conn,
				   struct qfle3i_cmd *cmnd,
				   char *datap, int data_len);
extern int qfle3i_send_iscsi_logout(struct qfle3i_conn *conn,
				   struct qfle3i_cmd *cmnd);
#if 0
extern int qfle3i_send_iscsi_text(struct qfle3i_conn *conn,
				 struct qfle3i_cmd *cmnd);
#endif
extern int qfle3i_send_iscsi_tmf(struct qfle3i_conn *conn,
				struct qfle3i_cmd *cmnd);
extern int qfle3i_send_iscsi_scsicmd(struct qfle3i_conn *conn,
				    struct qfle3i_cmd *cmnd);
extern void qfle3i_send_cmd_cleanup_req(struct qfle3i_hba *hba,
				       struct qfle3i_cmd *cmd);

extern int qfle3i_update_iscsi_conn(struct qfle3i_conn *conn);
extern int qfle3i_send_conn_ofld_req(struct qfle3i_hba *hba,
				     struct qfle3i_endpoint *ep);

extern int qfle3i_send_conn_destroy(struct qfle3i_hba *hba,
				    struct qfle3i_endpoint *ep);
extern int qfle3i_alloc_qp_resc(struct qfle3i_hba *hba,
			       struct qfle3i_endpoint *ep);
extern void qfle3i_free_qp_resc(struct qfle3i_hba *hba,
			       struct qfle3i_endpoint *ep);

extern void qfle3i_ep_ofld_timer(unsigned long data);

struct qfle3i_endpoint *qfle3i_find_ep_in_ofld_list(struct qfle3i_hba *hba,
						  vmk_uint32 iscsi_cid);
struct qfle3i_endpoint *qfle3i_find_ep_in_destroy_list(struct qfle3i_hba *hba,
						     vmk_uint32 iscsi_cid);

int qfle3i_map_ep_dbell_regs(struct qfle3i_endpoint *ep);
void qfle3i_ring_sq_dbell_bnx2(struct qfle3i_conn *conn);
void qfle3i_ring_sq_dbell_bnx2x(struct qfle3i_conn *conn);

void qfle3i_arm_cq_event_coalescing(struct qfle3i_endpoint *ep, vmk_uint8 action);

#if 0
int qfle3i_register_xport(struct qfle3i_hba *hba);
int qfle3i_deregister_xport(struct qfle3i_hba *hba);
int qfle3i_free_iscsi_scsi_template(struct qfle3i_hba *hba);
#endif
void qfle3i_update_conn_activity_counter(struct qfle3i_conn *conn);
#if 0
/* Debug related function prototypes */
extern void qfle3i_print_pend_cmd_queue(struct qfle3i_conn *conn);
extern void qfle3i_print_active_cmd_queue(struct qfle3i_conn *conn);
extern void qfle3i_print_xmit_pdu_queue(struct qfle3i_conn *conn);
extern void qfle3i_print_recv_state(struct qfle3i_conn *conn);
extern void qfle3i_print_cqe(struct qfle3i_conn *conn);
extern void qfle3i_print_sqe(struct qfle3i_conn *conn);

#endif
extern int qfle3i_get_stats(void *handle);

extern struct qfle3i_hba* hba_list[QFLE3I_MAX_HBA];

#define MAX_HBA	2048

#if 0
void qfle3i_register_device(struct qfle3i_hba *hba, int force);
void qfle3i_reg_dev_all(void);
void qfle3i_unreg_dev_all(void);
struct qfle3i_hba *get_adapter_list_head(void);
int qfle3i_ioctl_init(void);
int qfle3i_ioctl_exit(void);
void qfle3i_ioctl_cleanup(void);
struct qfle3i_hba *qfle3i_map_netdev_to_hba(struct net_device *netdev);
struct qfle3i_hba *qfle3i_get_hba_from_template(
	struct scsi_transport_template *scsit);
int qfle3i_proc_info(struct Scsi_Host *shost, char *buffer, char **start,
		    off_t offset, int length, int inout);
extern int qfle3i_ioctl(int, void *);
extern unsigned long qfle3i_host_instance[32];
extern vmk_uint32 adapter_count;
#endif
void qfle3i_dump_queues( struct qfle3i_hba *hba,
		struct qfle3i_endpoint *ep);
void qfle3i_print_cqe(struct qfle3i_endpoint *ep);
void qfle3i_print_sqe(struct qfle3i_endpoint *ep);
void qfle3i_free_scsi_task(struct qfle3i_sess *sess,
				 struct qfle3i_scsi_task *scsi_task);
struct qfle3i_scsi_task *qfle3i_alloc_scsi_task(struct qfle3i_sess *);
struct qfle3i_scsi_task *qfle3i_scsi_cmd_in_pend_list(struct qfle3i_sess *,
						    vmk_ScsiCommandId, vmk_uint32, int);
struct qfle3i_cmd *qfle3i_scsi_cmd_in_active_list(struct qfle3i_sess *sess,
						    vmk_ScsiCommandId, vmk_uint32, int);
void qfle3i_cleanup_task_context(struct qfle3i_sess *sess,
					struct qfle3i_cmd *cmd, int reason);
int qfle3i_flush_pend_queue(struct qfle3i_sess *sess,
				   qfle3i_iscsilun *ilun, int reason);
int qfle3i_flush_cmd_queue(struct qfle3i_sess *sess,
				  qfle3i_iscsilun *ilun,
				  int reason, int clear_ctx);

struct qfle3i_sess *qfle3i_get_session(struct qfle3i_hba *, int channel_id,
									int target_id);
struct qfle3i_iscsilun *qfle3i_get_iscsilun(struct qfle3i_sess *, int lun_id);
struct qfle3i_iscsilun *qfle3i_allocate_lun(struct qfle3i_sess *,
				int channel_id, int target_id, int lun_id);
void qfle3i_free_iscsilun(struct qfle3i_iscsilun *iscsilun);
VMK_ReturnStatus qfle3i_queue_scsi_scan(struct qfle3i_conn *conn);
void qfle3i_scsi_scan(unsigned long data);
int qfle3i_adapter_ready(struct qfle3i_hba *hba);
void qfle3i_iscsi_sess_release(struct qfle3i_sess *sess);
VMK_ReturnStatus qfle3iKeyValueInitPerHba(qfle3i_hba *);
VMK_ReturnStatus qfle3i_unreg_iscsi_adapter(qfle3i_hba *hba);
#endif /* __QFLE3I_H__ */
