/*
    Platypus - program for creating Mac OS X application wrappers around scripts
    Copyright (C) 2003-2010 Sveinbjorn Thordarson <sveinbjornt@simnet.is>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#import "IconController.h"
#import "IconFamily.h"
#import "STUtil.h"
#import "UKKQueue.h"
#import "CommonDefs.h"

@implementation IconController

/*****************************************
 - init function
*****************************************/
- (id)init
{
	if (self = [super init]) 
	{
		usesCustomIcnsFile = FALSE;
	}
    return self;
}

-(void)awakeFromNib
{
	// we list ourself as an observer of changes to file system, in case of custom icon file moving
	[[[NSWorkspace sharedWorkspace] notificationCenter]
		addObserver: self selector: @selector(customIcnsFileDidMove) name: UKFileWatcherRenameNotification object: NULL];
	[[[NSWorkspace sharedWorkspace] notificationCenter]
		addObserver: self selector: @selector(customIcnsFileDidMove) name: UKFileWatcherDeleteNotification object: NULL];
}

- (IBAction)copyIcon:(id)sender
{
	[[NSPasteboard generalPasteboard] declareTypes: [NSArray arrayWithObject: NSTIFFPboardType] owner: self];
	[[NSPasteboard generalPasteboard] setData: [[iconImageWell image] TIFFRepresentation] forType: NSTIFFPboardType];
}

- (IBAction)pasteIcon:(id)sender
{
	NSImage *pastedIcon = [[[NSImage alloc] initWithPasteboard: [NSPasteboard generalPasteboard]] autorelease];
	if (pastedIcon != NULL)
	{
		[iconImageWell setImage: pastedIcon];
		[self iconDidChange: self];
		usesCustomIcnsFile = 0;
	}
}

-(void)customIcnsFileDidMove
{
	if (usesCustomIcnsFile && ![[NSFileManager defaultManager] fileExistsAtPath: icnsFilePath])
		[iconNameTextField setTextColor: [NSColor redColor]];
	else
		[iconNameTextField setTextColor: [NSColor blackColor]];
}

/*****************************************
 - Responds when user drags custom icon on image well
*****************************************/

- (IBAction) iconDidChange:(id)sender
{
	[iconNameTextField setStringValue:@"Custom Icon"];
	[iconNameTextField setTextColor: [NSColor blackColor]];
	[iconImageWell performClick: self];//prevents the background from graying out
}

- (IBAction)nextIcon:(id)sender
{
	if ([iconToggleButton intValue] + 1 > [iconToggleButton maxValue])
		[iconToggleButton setIntValue: [iconToggleButton minValue]];
	else
		[iconToggleButton setIntValue: [iconToggleButton intValue] + 1];
	
	[self setAppIconForType: [iconToggleButton intValue]];
}

- (IBAction)previousIcon:(id)sender
{
	if ([iconToggleButton intValue] - 1 < [iconToggleButton minValue])
		[iconToggleButton setIntValue: [iconToggleButton maxValue]];
	else
		[iconToggleButton setIntValue: [iconToggleButton intValue] - 1];
	
	[self setAppIconForType: [iconToggleButton intValue]];
}

/*****************************************
 - Set the icon according to the default icon number index
*****************************************/

- (void)setAppIconForType: (int)type
{
	NSDictionary *iconInfo = [self getIconInfoForType: type];
	[iconImageWell setImage: [iconInfo objectForKey: @"Image"]];
	[iconNameTextField setStringValue: [iconInfo objectForKey: @"Name"]];
	icnsFilePath = [[iconInfo objectForKey: @"Path"] retain];
	[iconNameTextField setTextColor: [NSColor blackColor]];
	usesCustomIcnsFile = NO;
}

// get information about the default icons
- (NSDictionary *)getIconInfoForType: (int)type
{
	NSImage		*iconImage = NULL;
	NSString	*iconName = @"";
	NSString	*iconPath = @"";
	
	switch(type)
	{
		case 0:
			iconImage = [NSImage imageNamed:@"PlatypusDefault"];
			iconName = @"Platypus Default";
			iconPath = [[NSBundle mainBundle] pathForResource: @"PlatypusDefault" ofType:@"icns"];
			break;
		case 1:
			iconImage = [NSImage imageNamed:@"PlatypusInstaller"];
			iconName = @"Platypus Installer";
			iconPath = [[NSBundle mainBundle] pathForResource: @"PlatypusInstaller" ofType:@"icns"];
			break;
		case 2:
			iconImage = [NSImage imageNamed:@"PlatypusPlate"];
			iconName = @"Platypus Plate";
			iconPath = [[NSBundle mainBundle] pathForResource: @"PlatypusPlate" ofType:@"icns"];
			break;	
		case 3:
			iconImage = [NSImage imageNamed: @"PlatypusMenu"];
			iconName = @"Platypus Menu";
			iconPath = [[NSBundle mainBundle] pathForResource: @"PlatypusMenu" ofType:@"icns"];
			break;
		case 4:
			iconImage = [NSImage imageNamed: @"PlatypusCube"];
			iconName = @"Platypus Cube";
			iconPath = [[NSBundle mainBundle] pathForResource: @"PlatypusCube" ofType:@"icns"];
			break;
		case 5:
			iconImage = [NSImage imageNamed: @"NSDefaultApplicationIcon"];
			iconName = @"Generic Application Icon";
			iconPath = @"";
			break;
	}
	
	return [NSDictionary dictionaryWithObjectsAndKeys: iconImage, @"Image", iconName, @"Name", iconPath, @"Path", NULL];
}

- (void)setDefaultIcon
{
	[self setAppIconForType: 0];
}

- (IBAction)switchIcons:(id)sender
{
	[self setAppIconForType: [sender intValue]];
}

- (IBAction)selectIcnsFile:(id)sender
{
	NSOpenPanel *oPanel = [NSOpenPanel openPanel];
	[oPanel setPrompt:@"Select"];
    [oPanel setAllowsMultipleSelection:NO];
	[oPanel setCanChooseDirectories: NO];
	
	[window setTitle: [NSString stringWithFormat: @"%@ - Select an icns file", PROGRAM_NAME]];

	//run open panel
    [oPanel beginSheetForDirectory:nil file:nil types: [NSArray arrayWithObject: @"icns"] modalForWindow: window modalDelegate: self didEndSelector: @selector(selectIcnsFileDidEnd:returnCode:contextInfo:) contextInfo: nil];
}

- (void)selectIcnsFileDidEnd:(NSOpenPanel *)oPanel returnCode:(int)returnCode contextInfo:(void *)contextInfo
{
	if (returnCode == NSOKButton)
	{
		NSImage *customIcon = [[[NSImage alloc] initWithContentsOfFile: [oPanel filename]] autorelease];
		if (customIcon != NULL)
		{
			[self setIcnsPath: [oPanel filename]];
			[iconNameTextField setStringValue: [[oPanel filename] lastPathComponent]];
			
			[[UKKQueue sharedFileWatcher] addPathToQueue:  [oPanel filename]];
		}
		else
			[STUtil alert:@"Corrupt Image File" subText: @"The image file you selected appears to be damaged or corrupt."];
	}
	[window setTitle: PROGRAM_NAME];
}

- (void)setIcnsPath: (NSString *)icnsPath
{
		NSImage *customIcon = [[[NSImage alloc] initWithContentsOfFile: icnsPath] autorelease];
		if (customIcon != NULL)
		{
			icnsFilePath = [icnsPath retain];
			[iconImageWell setImage: customIcon];
			[self iconDidChange: self];
			usesCustomIcnsFile = 1;
			[[UKKQueue sharedFileWatcher] addPathToQueue:  icnsPath];
		}
		else
		{
			IconFamily *iconFam = [[IconFamily alloc] initWithSystemIcon: kQuestionMarkIcon];
			[iconImageWell setImage: [iconFam imageWithAllReps]];
			[iconFam release];
			[iconNameTextField setTextColor: [NSColor redColor]];
			[STUtil alert:@"Corrupted or Missing Image File" subText: @"The icns file you selected appears to be either missing or corrupted."];
		}
}

/*****************************************
 - Write an NSImage as icon to a path
*****************************************/

- (void)writeIconToPath: (NSString *)path
{
	if ([iconImageWell image] == NULL)
		[STUtil alert:@"Icon Error" subText: @"No icon could be found for your application.  Please set an icon to fix this."];

	IconFamily *iconFam = [[IconFamily alloc] initWithThumbnailsOfImage: [iconImageWell image]];
	[iconFam writeToFile: path];
	[iconFam release];
}

/*****************************************
 - Put up dialog prompting file to take icon from
*****************************************/
- (IBAction) importIcon:(id)sender
{
    NSOpenPanel *oPanel = [NSOpenPanel openPanel];
	[oPanel setPrompt:@"Select"];
    [oPanel setAllowsMultipleSelection:NO];
	[oPanel setCanChooseDirectories: YES];
	
	[window setTitle: [NSString stringWithFormat: @"%@ - Import a file's icon", PROGRAM_NAME]];

	//run open panel
    [oPanel beginSheetForDirectory:nil file:nil types: nil modalForWindow: window modalDelegate: self didEndSelector: @selector(importIconDidEnd:returnCode:contextInfo:) contextInfo: nil];	
}

- (void)importIconDidEnd:(NSOpenPanel *)oPanel returnCode:(int)returnCode contextInfo:(void *)contextInfo
{
	if (returnCode == NSOKButton)
	{
		IconFamily  *icon = [IconFamily iconFamilyWithIconOfFile: [oPanel filename]];
		if (icon != NULL)
		{
			[iconImageWell setImage: [icon imageWithAllReps]];
			[self iconDidChange: self];
			usesCustomIcnsFile = 0;
		}
		else
			[STUtil alert:@"Error getting icon" subText: @"The icon of the file you selected appears to be damaged or corrupt."];
	}
	[window setTitle: PROGRAM_NAME];
}

/*****************************************
 - Put up dialog prompting for image file to use as icon
*****************************************/
- (IBAction) selectIcon:(id)sender
{
    NSOpenPanel *oPanel = [NSOpenPanel openPanel];
	[oPanel setPrompt:@"Select"];
    [oPanel setAllowsMultipleSelection:NO];
	[oPanel setCanChooseDirectories: NO];
	
	[window setTitle: [NSString stringWithFormat: @"%@ - Select an image file", PROGRAM_NAME]];

	//run open panel
    [oPanel beginSheetForDirectory:nil file:nil types: [STUtil imageFileSuffixes] modalForWindow: window modalDelegate: self didEndSelector: @selector(selectIconDidEnd:returnCode:contextInfo:) contextInfo: nil];
		
}

- (void)selectIconDidEnd:(NSOpenPanel *)oPanel returnCode:(int)returnCode contextInfo:(void *)contextInfo
{
	if (returnCode == NSOKButton)
	{
		NSImage *customIcon = [[NSImage alloc] initWithContentsOfFile: [oPanel filename]];
		if (customIcon != NULL)
		{
			[iconImageWell setImage: customIcon];
			[customIcon autorelease];
			[self iconDidChange: self];
			usesCustomIcnsFile = 0;
		}
		else
			[STUtil alert:@"Corrupt Image File" subText: @"The image file you selected appears to be damaged or corrupt."];
	}
	[window setTitle: PROGRAM_NAME];
}

- (BOOL) hasCustomIcon
{
	if ([[iconNameTextField stringValue] isEqualToString: @"Custom Icon"])
		return YES;
	
	return NO;
}

- (void) setImage: (NSImage *)img
{
	[iconImageWell setImage: img];
	[iconNameTextField setStringValue: @"Custom Icon"];
}

- (NSData *)getImage
{
	return [[iconImageWell image] TIFFRepresentation];
}

- (BOOL) usesCustomIcnsFile
{
	return (BOOL)usesCustomIcnsFile;
}

- (NSString *) getIcnsFilePath
{
	return icnsFilePath;
}

- (BOOL)validateMenuItem:(NSMenuItem*)anItem 
{
	if ([[anItem title] isEqualToString:@"Paste Icon"])
	{
		NSArray		 *pbTypes = [NSArray arrayWithObjects: NSTIFFPboardType, NSPDFPboardType,NSPICTPboardType,NSPostScriptPboardType, NULL];
		NSString	 *type = [[NSPasteboard generalPasteboard] availableTypeFromArray: pbTypes];
		
		if (type == NULL)
			return NO;
	}
	return YES;
}

@end
