/**********************************************************************
Copyright (c) 2007 Thomas Marti and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.store.mapped.mapping;

import java.awt.geom.QuadCurve2D;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.ClassNameConstants;
import org.datanucleus.ObjectManager;
import org.datanucleus.metadata.AbstractMemberMetaData;
import org.datanucleus.store.mapped.DatastoreAdapter;
import org.datanucleus.store.mapped.DatastoreContainerObject;

/**
 * Mapping for java.awt.geom.QuadCurve2D.Double, maps the x1, y1, ctrlx, ctrly, x2 and y2 values to double-precision
 * datastore fields.
 */
public class QuadCurve2dDoubleMapping extends SingleFieldMultiMapping
{
    private static final QuadCurve2D.Double sampleValue = new QuadCurve2D.Double(0, 0, 1, 1, 2, 0);

	/* (non-Javadoc)
     * @see org.datanucleus.store.mapping.JavaTypeMapping#initialize()
     */
    public void initialize(DatastoreAdapter dba, AbstractMemberMetaData fmd, DatastoreContainerObject container, ClassLoaderResolver clr)
    {
		super.initialize(dba, fmd, container, clr);

        addDatastoreField(ClassNameConstants.DOUBLE); // X1
        addDatastoreField(ClassNameConstants.DOUBLE); // Y1
        addDatastoreField(ClassNameConstants.DOUBLE); // CtrlX
        addDatastoreField(ClassNameConstants.DOUBLE); // CtrlY
        addDatastoreField(ClassNameConstants.DOUBLE); // X2
        addDatastoreField(ClassNameConstants.DOUBLE); // Y2
    }

    /* (non-Javadoc)
     * @see org.datanucleus.store.mapping.JavaTypeMapping#getJavaType()
     */
    public Class getJavaType()
    {
        return QuadCurve2D.Double.class;
    }

    /* (non-Javadoc)
     * @see org.datanucleus.store.mapping.JavaTypeMapping#getSampleValue()
     */
    public Object getSampleValue(ClassLoaderResolver clr)
    {
        return sampleValue;
    }
    
    /* (non-Javadoc)
     * @see org.datanucleus.store.mapping.JavaTypeMapping#setObject(org.datanucleus.ObjectManager, java.lang.Object, int[], java.lang.Object)
     */
    public void setObject(ObjectManager om, Object preparedStatement, int[] exprIndex, Object value)
    {
    	QuadCurve2D line = (QuadCurve2D)value;
        if (line == null)
        {
    		for (int i = 0; i < exprIndex.length; i++) 
    		{
    			getDataStoreMapping(i).setObject(preparedStatement, exprIndex[i], null);					
			}
        }
        else
        {
            getDataStoreMapping(0).setDouble(preparedStatement,exprIndex[0],line.getX1());
            getDataStoreMapping(1).setDouble(preparedStatement,exprIndex[1],line.getY1());
            getDataStoreMapping(2).setDouble(preparedStatement,exprIndex[2],line.getCtrlX());
            getDataStoreMapping(3).setDouble(preparedStatement,exprIndex[3],line.getCtrlY());
            getDataStoreMapping(4).setDouble(preparedStatement,exprIndex[4],line.getX2());
            getDataStoreMapping(5).setDouble(preparedStatement,exprIndex[5],line.getY2());
        }
    }
    
    /* (non-Javadoc)
     * @see org.datanucleus.store.mapping.JavaTypeMapping#getObject(org.datanucleus.ObjectManager, java.lang.Object, int[])
     */
    public Object getObject(ObjectManager om, Object resultSet, int[] exprIndex)
    {
        // Check for null entries
        if (getDataStoreMapping(0).getObject(resultSet, exprIndex[0]) == null)
        {
            return null;
        }

        double x1 = getDataStoreMapping(0).getDouble(resultSet,exprIndex[0]); 
        double y1 = getDataStoreMapping(1).getDouble(resultSet,exprIndex[1]); 
        double ctrlx = getDataStoreMapping(2).getDouble(resultSet,exprIndex[2]); 
        double ctrly = getDataStoreMapping(3).getDouble(resultSet,exprIndex[3]);
        double x2 = getDataStoreMapping(4).getDouble(resultSet,exprIndex[5]); 
        double y2 = getDataStoreMapping(5).getDouble(resultSet,exprIndex[6]);
        return new QuadCurve2D.Double(x1, y1, ctrlx, ctrly, x2, y2);
    }
}