/**********************************************************************
Copyright (c) 2005 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
 

Contributors:
    ...
**********************************************************************/
package org.datanucleus.store.mapped.mapping;

import java.sql.Timestamp;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.TimeZone;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.ClassNameConstants;
import org.datanucleus.ObjectManager;
import org.datanucleus.exceptions.NucleusUserException;
import org.datanucleus.metadata.AbstractMemberMetaData;
import org.datanucleus.store.mapped.DatastoreAdapter;
import org.datanucleus.store.mapped.DatastoreContainerObject;
import org.datanucleus.store.mapped.expression.LogicSetExpression;
import org.datanucleus.store.mapped.expression.QueryExpression;
import org.datanucleus.store.mapped.expression.ScalarExpression;
import org.datanucleus.store.mapped.expression.SqlTemporalExpression;
import org.datanucleus.store.mapped.expression.SqlTimestampLiteral;

/**
 * Maps the class fields of a GregorianCalendar to datastore fields.
 * JPOX traditionally supported this as mapping to 2 datastore fields (timestamp millisecs and timezone).
 * It also supports as mapping to a single datastore field (timestamp).
 */
public class GregorianCalendarMapping extends SingleFieldMultiMapping
{
    boolean singleColumn = false;

    /**
     * Initialize this JavaTypeMapping with the given DatastoreAdapter for the given FieldMetaData.
     * @param dba The Datastore Adapter that this Mapping should use.
     * @param fmd FieldMetaData for the field to be mapped (if any)
     * @param container The datastore container storing this mapping (if any)
     * @param clr the ClassLoaderResolver
     */
    public void initialize(DatastoreAdapter dba, AbstractMemberMetaData fmd, DatastoreContainerObject container, 
            ClassLoaderResolver clr)
    {
		super.initialize(dba, fmd, container, clr);
        if (fmd.hasExtension("calendar-one-column") && fmd.getValueForExtension("calendar-one-column").equals("true"))
        {
            // If this mapping is created via a query we assume multiple columns currently
            singleColumn = true;
        }

        if (singleColumn)
        {
            // (Timestamp) implementation
            addDatastoreField(ClassNameConstants.JAVA_SQL_TIMESTAMP);
        }
        else
        {
            // (Timestamp millisecs, Timezone) implementation
            addDatastoreField(ClassNameConstants.LONG); // Timestamp millisecs
            addDatastoreField(ClassNameConstants.JAVA_LANG_STRING); // Timezone
        }
    }

    /*
     * (non-Javadoc)
     * @see org.datanucleus.store.mapping.JavaTypeMapping#getJavaType()
     */
    public Class getJavaType()
    {
        return GregorianCalendar.class;
    }

    /**
     * Accessor for the name of the java-type actually used when mapping the particular datastore field.
     * This java-type must have an entry in the datastore mappings.
     * @param index requested datastore field index.
     * @return the name of java-type for the requested datastore field.
     */
    public String getJavaTypeForDatastoreMapping(int index)
    {
        if (singleColumn)
        {
            // (Timestamp) implementation
            return ClassNameConstants.JAVA_SQL_TIMESTAMP;
        }
        else
        {
            // (Timestamp millisecs, Timezone) implementation
            if (index == 0)
            {
                return ClassNameConstants.LONG;
            }
            else if (index == 1)
            {
                return ClassNameConstants.JAVA_LANG_STRING;
            }
        }
        return null;
    }

    /*
     * (non-Javadoc)
     * @see org.datanucleus.store.mapping.JavaTypeMapping#getSampleValue()
     */
    public Object getSampleValue(ClassLoaderResolver clr)
    {
        return new GregorianCalendar();
    }

    /*
     * (non-Javadoc)
     * @see org.datanucleus.store.mapping.JavaTypeMapping#setObject(org.datanucleus.ObjectManager, java.lang.Object,
     *  int[], java.lang.Object)
     */
    public void setObject(ObjectManager om, Object preparedStatement, int[] exprIndex, Object value)
    {
        GregorianCalendar cal = (GregorianCalendar) value;
        if (singleColumn)
        {
            // (Timestamp) implementation
            Timestamp ts = null;
            if (cal != null)
            {
                ts = new Timestamp(cal.getTimeInMillis());
            }
            // Server timezone will be applied in the RDBMSMapping at persistence
            getDataStoreMapping(0).setObject(preparedStatement, exprIndex[0], ts);
        }
        else
        {
            // (Timestamp millisecs, Timezone) implementation
            if (cal == null)
            {
                getDataStoreMapping(0).setObject(preparedStatement, exprIndex[0], null);
                getDataStoreMapping(1).setObject(preparedStatement, exprIndex[1], null);
            }
            else
            {
                getDataStoreMapping(0).setLong(preparedStatement, exprIndex[0], cal.getTime().getTime());
                getDataStoreMapping(1).setString(preparedStatement, exprIndex[1], cal.getTimeZone().getID());
            }
        }
    }

    /*
     * (non-Javadoc)
     * @see org.datanucleus.store.mapping.JavaTypeMapping#getObject(org.datanucleus.ObjectManager, java.lang.Object, int[])
     */
    public Object getObject(ObjectManager om, Object resultSet, int[] exprIndex)
    {
        try
        {
            // Check for null entries
            if (getDataStoreMapping(0).getObject(resultSet, exprIndex[0]) == null)
            {
                return null;
            }
        }
        catch (Exception e)
        {
            // Do nothing
        }

        if (singleColumn)
        {
            
            Timestamp ts = (Timestamp)getDataStoreMapping(0).getObject(resultSet, exprIndex[0]);
            GregorianCalendar cal = new GregorianCalendar();
            cal.setTimeInMillis(ts.getTime());

            String timezoneID = om.getOMFContext().getPersistenceConfiguration().getStringProperty("datanucleus.ServerTimeZoneID");
            if (timezoneID != null)
            {
                // Apply server timezone ID since we dont know what it was upon persistence
                cal.setTimeZone(TimeZone.getTimeZone(timezoneID));
            }
            return cal;
        }
        else
        {
            // (Timestamp millisecs, Timezone) implementation
            long millisecs = getDataStoreMapping(0).getLong(resultSet, exprIndex[0]);

            GregorianCalendar cal = new GregorianCalendar();
            cal.setTime(new Date(millisecs));
            String timezoneId = getDataStoreMapping(1).getString(resultSet, exprIndex[1]);
            if (timezoneId != null)
            {
                cal.setTimeZone(TimeZone.getTimeZone(timezoneId));
            }
            return cal;
        }
    }

    // ---------------------------------- JDOQL Query Methods --------------------------------------

    /*
     * (non-Javadoc)
     * @see org.datanucleus.store.mapping.JavaTypeMapping#newLiteral(org.datanucleus.store.query.QueryStatement,
     * java.lang.Object)
     */
    public ScalarExpression newLiteral(QueryExpression qs, Object value)
    {
        if (singleColumn)
        {
            return new SqlTimestampLiteral(qs, this, (Timestamp)value);
        }
        else
        {
            // [CORE-2802]
            throw new NucleusUserException("We dont support querying of Calendar fields when stored as 2 columns");
        }
    }

    /*
     * (non-Javadoc)
     * @see org.datanucleus.store.mapping.JavaTypeMapping#newScalarExpression(org.datanucleus.store.query.QueryStatement,
     * org.datanucleus.store.expression.TableExpression)
     */
    public ScalarExpression newScalarExpression(QueryExpression qs, LogicSetExpression te)
    {
        if (singleColumn)
        {
            return new SqlTemporalExpression(qs, this, te);
        }
        else
        {
            // [CORE-2802]
            throw new NucleusUserException("We dont support querying of Calendar fields when stored as 2 columns");
        }
    }
}