/**********************************************************************
Copyright (c) 26-Oct-2004 Andy Jefferson and others.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

 

Contributors:
    ...
**********************************************************************/
package org.datanucleus.util;

import java.net.MalformedURLException;
import java.util.AbstractCollection;
import java.util.AbstractList;
import java.util.Collection;

import javax.jdo.PersistenceManager;
import javax.jdo.spi.PersistenceCapable;
import javax.jdo.spi.StateManager;

import junit.framework.TestCase;

import org.apache.log4j.Category;
import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.JDOClassLoaderResolver;
import org.datanucleus.util.ClassUtils;
import org.datanucleus.util.StringUtils;

/**
 * Tests for class utilities.
 */
public class ClassUtilsTest extends TestCase
{
    protected static final Category LOG = Category.getInstance("DataNucleus.TEST");
    
    /**
     * Start of the test, so log it and initialise.
     * @param name Name of the test case (not used)
     */
    public ClassUtilsTest(String name)
    {
        super(name);
    }

    /**
     * Test of whether a class is an inner class
     */
    public void testIsInnerClass()
    {
        assertTrue("Should have identified inner class, but failed", ClassUtils.isInnerClass("java.text.DateFormat$Field"));
    }
    
    /**
     * Test of whether a class has a default constructor
     */
    public void testHasDefaultConstructor()
    {
        assertTrue("Class java.lang.String should have been identified as having a default constructor, but failed",
            ClassUtils.hasDefaultConstructor(java.lang.String.class));
        assertTrue("Class java.sql.DriverPropertyInfo should have been identified as not having a default constructor, but passed",
            !ClassUtils.hasDefaultConstructor(java.sql.DriverPropertyInfo.class));
    }
    
    /**
     * Test for the superclasses of a class.
     */
    public void testGetSuperclasses()
    {
        Collection superclasses=ClassUtils.getSuperclasses(java.util.ArrayList.class);
        assertTrue("java.util.ArrayList should have had 3 superclasses, but had " + superclasses.size(),
            superclasses.size() == 3);
        assertTrue("java.util.ArrayList should have had a superclass of AbstractList, but didn't !",
            superclasses.contains(AbstractList.class));
        assertTrue("java.util.ArrayList should have had a superclass of AbstractCollection, but didn't !",
            superclasses.contains(AbstractCollection.class));
        assertTrue("java.util.ArrayList should have had a superclass of Object, but didn't !",
            superclasses.contains(Object.class));
    }
    
    /**
     * Test for the creation of a fully specified class name.
     */
    public void testCreateFullClassName()
    {
        assertTrue("Full classname is incorrect",
            ClassUtils.createFullClassName("org.datanucleus.samples","MyClass").equals("org.datanucleus.samples.MyClass"));
        assertTrue("Full classname is incorrect",
            ClassUtils.createFullClassName("     ","MyClass").equals("MyClass"));
        assertTrue("Full classname is incorrect",
            ClassUtils.createFullClassName("org","MyClass").equals("org.MyClass"));
    }
    
    /**
     * Test of the retrieval of the classname for a filename, given the root directory name.
     * @throws MalformedURLException 
     */
    public void testGetClassnameForFilename() throws MalformedURLException
    {
        String externalForm = getClass().getResource("/org/datanucleus/util/ClassUtilsTest.class").toExternalForm();
        externalForm = StringUtils.replaceAll(externalForm,"file:/","");
        String path = externalForm.substring(0,externalForm.indexOf("/org/datanucleus/util/ClassUtilsTest.class"));
        externalForm = StringUtils.replaceAll(externalForm,"/",System.getProperty("file.separator"));
        externalForm = StringUtils.replaceAll(externalForm,"\\",System.getProperty("file.separator"));
        assertTrue("Classname for filename is incorrect",
            ClassUtils.getClassnameForFilename(externalForm, path).equals("org.datanucleus.util.ClassUtilsTest"));
    }
    
    /**
     * Test for whether classes are descendents of each other.
     */
    public void testClassesAreDescendents()
    {
        ClassLoaderResolver clr = new JDOClassLoaderResolver(null);
        
        assertTrue("java.util.Collection and java.util.ArrayList should have been direct descendents but weren't",
            ClassUtils.classesAreDescendents(clr, "java.util.Collection", "java.util.ArrayList"));
        assertTrue("java.util.ArrayList and java.util.Collection should have been direct descendents but weren't",
            ClassUtils.classesAreDescendents(clr, "java.util.ArrayList", "java.util.Collection"));
        assertTrue("java.util.ArrayList and java.lang.String shouldn't have been direct descendents but were",
            !ClassUtils.classesAreDescendents(clr, "java.util.ArrayList", "java.lang.String"));
    }

    /**
     * Test for java bean getter name generator.
     */
    public void testJavaBeanGetterName()
    {
        assertEquals("Incorrect Java Bean getter name", "getParam", ClassUtils.getJavaBeanGetterName("param", false));
        assertEquals("Incorrect Java Bean getter name", "getABC", ClassUtils.getJavaBeanGetterName("ABC", false));
        assertEquals("Incorrect Java Bean getter name", "getA", ClassUtils.getJavaBeanGetterName("a", false));
    }

    /**
     * Test for java bean setter name generator.
     */
    public void testJavaBeanSetterName()
    {
        assertEquals("Incorrect Java Bean setter name", "setParam", ClassUtils.getJavaBeanSetterName("param"));
        assertEquals("Incorrect Java Bean setter name", "setABC", ClassUtils.getJavaBeanSetterName("ABC"));
        assertEquals("Incorrect Java Bean setter name", "setA", ClassUtils.getJavaBeanSetterName("a"));
    }

    /**
     * Test for field name for java bean getter.
     */
    public void testFieldNameForJavaBeanGetter()
    {
        assertEquals("Incorrect field name for Java Bean getter", "param", ClassUtils.getFieldNameForJavaBeanGetter("getParam"));
        assertEquals("Incorrect field name for Java Bean getter", "ABC", ClassUtils.getFieldNameForJavaBeanGetter("getABC"));
        assertEquals("Incorrect field name for Java Bean getter", "a", ClassUtils.getFieldNameForJavaBeanGetter("getA"));
    }

    /**
     * Test for field name for java bean setter.
     */
    public void testFieldNameForJavaBeanSetter()
    {
        assertEquals("Incorrect field name for Java Bean setter", "param", ClassUtils.getFieldNameForJavaBeanSetter("setParam"));
        assertEquals("Incorrect field name for Java Bean setter", "ABC", ClassUtils.getFieldNameForJavaBeanSetter("setABC"));
        assertEquals("Incorrect field name for Java Bean setter", "a", ClassUtils.getFieldNameForJavaBeanSetter("setA"));
    }

    public class MyPCClass implements PersistenceCapable
    {

        public PersistenceManager jdoGetPersistenceManager()
        {
            return null;
        }

        public void jdoReplaceStateManager(StateManager arg0) throws SecurityException
        {
        }

        public void jdoProvideField(int arg0)
        {
        }

        public void jdoProvideFields(int[] arg0)
        {
        }

        public void jdoReplaceField(int arg0)
        {
        }

        public void jdoReplaceFields(int[] arg0)
        {
        }

        public void jdoReplaceFlags()
        {
        }

        public void jdoCopyFields(Object arg0, int[] arg1)
        {
        }

        public void jdoMakeDirty(String arg0)
        {
        }

        public Object jdoGetObjectId()
        {
            return null;
        }

        public Object jdoGetTransactionalObjectId()
        {
            return null;
        }

        public Object jdoGetVersion()
        {
            return null;
        }

        public boolean jdoIsDirty()
        {
            return false;
        }

        public boolean jdoIsTransactional()
        {
            return false;
        }

        public boolean jdoIsPersistent()
        {
            return false;
        }

        public boolean jdoIsNew()
        {
            return false;
        }

        public boolean jdoIsDeleted()
        {
            return false;
        }

        public boolean jdoIsDetached()
        {
            return false;
        }

        public PersistenceCapable jdoNewInstance(StateManager arg0)
        {
            return null;
        }

        public PersistenceCapable jdoNewInstance(StateManager arg0, Object arg1)
        {
            return null;
        }

        public Object jdoNewObjectIdInstance()
        {
            return null;
        }

        public Object jdoNewObjectIdInstance(Object arg0)
        {
            return null;
        }

        public void jdoCopyKeyFieldsToObjectId(Object arg0)
        {
        }

        public void jdoCopyKeyFieldsToObjectId(ObjectIdFieldSupplier arg0, Object arg1)
        {
        }

        public void jdoCopyKeyFieldsFromObjectId(ObjectIdFieldConsumer arg0, Object arg1)
        {
        }
    }

    public class MyChildPCClass extends MyPCClass
    {
    }
}