/**********************************************************************
Copyright (c) 2007 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.enhancer.asm.method;

import org.datanucleus.enhancer.ClassEnhancer;
import org.datanucleus.enhancer.asm.ASMClassEnhancer;
import org.datanucleus.enhancer.asm.ASMClassMethod;
import org.objectweb.asm.MethodVisitor;
import org.objectweb.asm.Opcodes;

/**
 * Method to generate a static initialisation block for the class using ASM.
 * <pre>
 * static
 * {
 *     jdoFieldNames = __jdoFieldNamesInit();
 *     jdoFieldTypes = __jdoFieldTypesInit();
 *     jdoFieldFlags = __jdoFieldFlagsInit();
 *     jdoInheritedFieldCount = __jdoGetInheritedFieldCount();
 *     jdoPersistenceCapableSuperclass = __jdoPersistenceCapableSuperclassInit();
 *     JDOImplHelper.registerClass(___jdo$loadClass("mydomain.MyClass"),
 *         jdoFieldNames, jdoFieldTypes, jdoFieldFlags,
 *         jdoPersistenceCapableSuperclass, new MyClass());
 * }
 * </pre>
 * @version $Revision: 1.4 $
 */
public class InitClass extends ASMClassMethod
{
    public static InitClass getInstance(ASMClassEnhancer enhancer)
    {
        return new InitClass(enhancer, "<clinit>",
            Opcodes.ACC_STATIC,
            null, null, null);
    }

    /**
     * Constructor.
     * @param enhancer ClassEnhancer
     * @param name Name of method
     * @param access Access type
     * @param returnType Return type
     * @param argTypes Argument types
     * @param argNames Argument names
     */
    public InitClass(ClassEnhancer enhancer, String name, int access, 
        Object returnType, Object[] argTypes, String[] argNames)
    {
        super(enhancer, name, access, returnType, argTypes, argNames);
    }

    /**
     * Method to add the contents of the class method.
     */
    public void execute()
    {
        visitor.visitCode();

        addInitialiseInstructions(visitor);

        visitor.visitInsn(Opcodes.RETURN);
        visitor.visitMaxs(7, 0);

        visitor.visitEnd();
    }

    /**
     * Convenience method to add the initialise instructions to the supplied MethodVisitor.
     * Available as a separate method so that the initialise instructions can be added to an existing
     * static class initialise block (where the class already has one).
     * @param mv MethodVisitor to use
     */
    public void addInitialiseInstructions(MethodVisitor mv)
    {
        mv.visitMethodInsn(Opcodes.INVOKESTATIC, getClassEnhancer().getASMClassName(),
            ClassEnhancer.MN_FieldNamesInitMethod, "()[Ljava/lang/String;");
        mv.visitFieldInsn(Opcodes.PUTSTATIC, getClassEnhancer().getASMClassName(),
            ClassEnhancer.FN_FieldNames, "[Ljava/lang/String;");

        mv.visitMethodInsn(Opcodes.INVOKESTATIC, getClassEnhancer().getASMClassName(),
            ClassEnhancer.MN_FieldTypesInitMethod, "()[Ljava/lang/Class;");
        mv.visitFieldInsn(Opcodes.PUTSTATIC, getClassEnhancer().getASMClassName(),
            ClassEnhancer.FN_FieldTypes, "[Ljava/lang/Class;");

        mv.visitMethodInsn(Opcodes.INVOKESTATIC, getClassEnhancer().getASMClassName(),
            ClassEnhancer.MN_FieldFlagsInitMethod, "()[B");
        mv.visitFieldInsn(Opcodes.PUTSTATIC, getClassEnhancer().getASMClassName(),
            ClassEnhancer.FN_FieldFlags, "[B");

        mv.visitMethodInsn(Opcodes.INVOKESTATIC, getClassEnhancer().getASMClassName(),
            ClassEnhancer.MN_JdoGetInheritedFieldCount, "()I");
        mv.visitFieldInsn(Opcodes.PUTSTATIC, getClassEnhancer().getASMClassName(),
            ClassEnhancer.FN_JdoInheritedFieldCount, "I");

        mv.visitMethodInsn(Opcodes.INVOKESTATIC, getClassEnhancer().getASMClassName(),
            ClassEnhancer.MN_JdoPersistenceCapableSuperclassInit, "()Ljava/lang/Class;");
        mv.visitFieldInsn(Opcodes.PUTSTATIC, getClassEnhancer().getASMClassName(),
            ClassEnhancer.FN_PersistenceCapableSuperclass, "Ljava/lang/Class;");

        mv.visitLdcInsn(getClassEnhancer().className);
        mv.visitMethodInsn(Opcodes.INVOKESTATIC, getClassEnhancer().getASMClassName(),
            ClassEnhancer.MN_jdoLoadClass, "(Ljava/lang/String;)Ljava/lang/Class;");
        mv.visitFieldInsn(Opcodes.GETSTATIC, getClassEnhancer().getASMClassName(),
            ClassEnhancer.FN_FieldNames, "[Ljava/lang/String;");
        mv.visitFieldInsn(Opcodes.GETSTATIC, getClassEnhancer().getASMClassName(),
            ClassEnhancer.FN_FieldTypes, "[Ljava/lang/Class;");
        mv.visitFieldInsn(Opcodes.GETSTATIC, getClassEnhancer().getASMClassName(),
            ClassEnhancer.FN_FieldFlags, "[B");
        mv.visitFieldInsn(Opcodes.GETSTATIC, getClassEnhancer().getASMClassName(),
            ClassEnhancer.FN_PersistenceCapableSuperclass, "Ljava/lang/Class;");
        if (enhancer.getClassMetaData().isAbstractPersistenceCapable())
        {
            mv.visitInsn(Opcodes.ACONST_NULL);
        }
        else
        {
            mv.visitTypeInsn(Opcodes.NEW, getClassEnhancer().getASMClassName());
            mv.visitInsn(Opcodes.DUP);
            mv.visitMethodInsn(Opcodes.INVOKESPECIAL, getClassEnhancer().getASMClassName(),
                "<init>", "()V");
        }
        mv.visitMethodInsn(Opcodes.INVOKESTATIC, "javax/jdo/spi/JDOImplHelper", "registerClass",
            "(Ljava/lang/Class;[Ljava/lang/String;[Ljava/lang/Class;[BLjava/lang/Class;Ljavax/jdo/spi/PersistenceCapable;)V");
    }
}