/**********************************************************************
Copyright (c) 2007 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.enhancer.asm.method;

import org.datanucleus.enhancer.ClassEnhancer;
import org.datanucleus.enhancer.asm.ASMClassMethod;
import org.objectweb.asm.Label;
import org.objectweb.asm.Opcodes;

/**
 * Method to generate the method "jdoIsDetached" using ASM.
 * <pre>
 * public boolean jdoIsDetached()
 * {
 *     if (jdoStateManager == null && jdoDetachedState != null) 
 *     {
 *         return true;
 *     }
 *     return false;
 * }
 * </pre>
 * and if not detachable will get
 * <pre>
 * public boolean jdoIsDetached()
 * {
 *     return false;
 * }
 * </pre>
 * @version $Revision: 1.5 $
 */
public class JdoIsDetached extends ASMClassMethod
{
    public static JdoIsDetached getInstance(ClassEnhancer enhancer)
    {
        return new JdoIsDetached(enhancer, ClassEnhancer.MN_JdoIsDetached, Opcodes.ACC_PUBLIC,
            boolean.class, null, null);
    }

    /**
     * Constructor.
     * @param enhancer ClassEnhancer
     * @param name Name of method
     * @param access Access type
     * @param returnType Return type
     * @param argTypes Argument types
     * @param argNames Argument names
     */
    public JdoIsDetached(ClassEnhancer enhancer, String name, int access, 
        Object returnType, Object[] argTypes, String[] argNames)
    {
        super(enhancer, name, access, returnType, argTypes, argNames);
    }

    /**
     * Method to add the contents of the class method.
     */
    public void execute()
    {
        visitor.visitCode();
        Label startLabel = new Label();
        visitor.visitLabel(startLabel);

        if (getClassEnhancer().getClassMetaData().isDetachable())
        {
            visitor.visitVarInsn(Opcodes.ALOAD, 0);
            visitor.visitFieldInsn(Opcodes.GETFIELD, getClassEnhancer().getASMClassName(), 
                "jdoStateManager", "Ljavax/jdo/spi/StateManager;");
            Label l1 = new Label();
            visitor.visitJumpInsn(Opcodes.IFNONNULL, l1);
            visitor.visitVarInsn(Opcodes.ALOAD, 0);
            visitor.visitFieldInsn(Opcodes.GETFIELD, getClassEnhancer().getASMClassName(), 
                "jdoDetachedState", "[Ljava/lang/Object;");
            visitor.visitJumpInsn(Opcodes.IFNULL, l1);
            visitor.visitInsn(Opcodes.ICONST_1);
            visitor.visitInsn(Opcodes.IRETURN);
            visitor.visitLabel(l1);
        }

        visitor.visitInsn(Opcodes.ICONST_0);
        visitor.visitInsn(Opcodes.IRETURN);
        Label endLabel = new Label();
        visitor.visitLabel(endLabel);
        visitor.visitLocalVariable("this", getClassEnhancer().getClassDescriptor(), null, startLabel, endLabel, 0);
        visitor.visitMaxs(1, 1);
        visitor.visitEnd();
    }
}