// Copyright 2009 Google Inc. All Rights Reserved.
package com.google.appengine.api.datastore;

import static com.google.common.base.Preconditions.checkNotNull;

import com.google.appengine.api.datastore.MultiQueryComponent.Order;
import com.google.appengine.api.datastore.Query.FilterPredicate;
import com.google.common.collect.Iterators;
import com.google.common.collect.Lists;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

/**
 * A class that generates multiple lists of query filters to execute to satisfy the {@link
 * MultiQueryComponent}s given to it.
 *
 * <p>Each component contains multiple {@link List}s of {@link FilterPredicate}s and an order in
 * which MultiQuery should produce queries that contain these filters.
 *
 * <p>If the order of a {@link MultiQueryComponent} is set to {@link Order#SERIAL} the given filters
 * are applied to sequentially generated sets of queries. This denotes that the result of the
 * sequentially generated queries can be concatenated to produce a valid result set. However this
 * class makes no guarantees to this effect and this assurance must come from the code that
 * constructs this class.
 *
 * <p>If the order of a {@link MultiQueryComponent} is set to {@link Order#PARALLEL} the given
 * filters are used to generate multiple queries (one for each list of filters) that are returned
 * together. This is used to denote that results must be merged in memory to create a valid result
 * set. However this class makes no guarantees to this effect and this assurance must come from the
 * code that constructs this class.
 *
 * <p>{@link MultiQueryBuilder#iterator()} will provide an iterator that generates the filters for
 * these queries in the given order. The iterator generates each list of filters as they are needed.
 * In this way, if the user request is satisfied the first few sets of queries, the remaining
 * queries need never be created. This is important because the total number of queries generated by
 * this class is mult_i(|component_i.filters|).
 *
 * <p>This class preserves the order in which {@link MultiQueryComponent} are given to it when
 * applying the filters. However filters are generated most optimally when {@link Order#PARALLEL}
 * are the last to be applied. Thus to provide a convenient way to push components with {@link
 * Order#PARALLEL} to the back, {@link MultiQueryComponent} are sortable.
 *
 * <p>{@link MultiQueryComponent}s with different {@link Order}s can be added to the same {@link
 * MultiQueryBuilder} in any configuration.
 *
 */
class MultiQueryBuilder implements Iterable<List<List<FilterPredicate>>> {
  final List<FilterPredicate> baseFilters;
  final List<MultiQueryComponent> components;
  final int parallelQuerySize;

  MultiQueryBuilder(
      List<FilterPredicate> baseFilters,
      List<MultiQueryComponent> components,
      int parallelQuerySize) {
    this.baseFilters = checkNotNull(baseFilters);
    this.components = components;
    this.parallelQuerySize = parallelQuerySize;
  }

  public MultiQueryBuilder(
      List<FilterPredicate> baseFilters,
      List<QuerySplitComponent> splitComponents,
      boolean hasSort) {
    this.baseFilters = checkNotNull(baseFilters);

    if (splitComponents.isEmpty()) {
      components = Collections.emptyList();
      parallelQuerySize = 1;
    } else {
      components = Lists.newArrayListWithCapacity(splitComponents.size());

      Collections.sort(splitComponents);

      MultiQueryComponent.Order applyToRemaining =
          hasSort ? null : MultiQueryComponent.Order.SERIAL;

      int currentSortIndex = 0;
      int totalParallelQueries = 1;
      for (QuerySplitComponent component : splitComponents) {
        if ((applyToRemaining == null) && (component.getSortIndex() != currentSortIndex)) {
          if (component.getSortIndex() == currentSortIndex + 1) {
            ++currentSortIndex;
          } else {
            applyToRemaining = MultiQueryComponent.Order.PARALLEL;
          }
        }

        components.add(
            new MultiQueryComponent(
                applyToRemaining != null ? applyToRemaining : MultiQueryComponent.Order.SERIAL,
                component.getFilters()));

        if (applyToRemaining == MultiQueryComponent.Order.PARALLEL) {
          totalParallelQueries *= component.getFilters().size();
        }
      }
      this.parallelQuerySize = totalParallelQueries;
    }
  }

  @Override
  public Iterator<List<List<FilterPredicate>>> iterator() {
    if (components.isEmpty()) {
      return Iterators.singletonIterator(Collections.singletonList(baseFilters));
    }
    return new MultiQueryIterator(baseFilters, components);
  }

  public List<FilterPredicate> getBaseFilters() {
    return baseFilters;
  }

  public boolean isSingleton() {
    return components.isEmpty();
  }

  /** @return the parallelQuerySize */
  public int getParallelQuerySize() {
    return parallelQuerySize;
  }

  @Override
  public String toString() {
    return "MultiQueryBuilder [baseFilters=" + baseFilters + ", components=" + components + "]";
  }
}
