/**********************************************************************
Copyright (c) 2002 Mike Martin (TJDO) and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
2003 Andy Jefferson - coding standards
    ...
**********************************************************************/
package org.datanucleus.store.mapped.expression;

import java.util.ArrayList;
import java.util.Iterator;

import org.datanucleus.store.mapped.DatastoreClass;
import org.datanucleus.store.mapped.DatastoreContainerObject;
import org.datanucleus.store.mapped.DatastoreField;
import org.datanucleus.store.mapped.DatastoreIdentifier;
import org.datanucleus.store.mapped.FetchStatement;

/**
 * A SQL table expression that joins superclass tables by constructing a parenthesized sub-SELECT statement.
 * Used by Cloudscape (<10), DB2 (<8), Oracle (<9).
 * @see QueryExpression
 */
public class TableExprAsSubquery extends LogicSetExpression
{
    protected final ArrayList columns = new ArrayList();

    protected boolean multipleTablesReferenced = false;

    /**
     * Constructor for a table expressed as a subquery.
     * @param qs the QueryExpression
     * @param mainTable the main table in the query
     * @param alias The alias/identifier of the table in the query
     */
    public TableExprAsSubquery(QueryExpression qs, DatastoreContainerObject mainTable, DatastoreIdentifier alias)
    {
        super(qs, mainTable, alias);
    }

    public String referenceColumn(DatastoreField col)
    {
        assertNotFrozen();

        DatastoreContainerObject table = col.getDatastoreContainerObject();
        if (!table.equals(mainTable))
        {
            if (!(mainTable instanceof DatastoreClass) || !(table instanceof DatastoreClass))
            {
                throw new TableMismatchException(col, mainTable);
            }

            /*
             * Since both tables are DatastoreClass we assume that the column
             * is a superclass field, meaning 'table' is a supertable of
             * 'mainTable'.  We set the flag indicating that this expression
             * must become a subquery that joins the necessary tables.
             */
            multipleTablesReferenced = true;
        }

        if (!columns.contains(col))
        {
            columns.add(col);
        }

        if (mainAlias.toString().length() > 0)
        {
            return col.applySelectFunction(mainAlias + "." + col.getIdentifier());
        }
        else
        {
            return col.applySelectFunction(col.getIdentifier().toString());
        }
    }

    public String toString()
    {
        if (sqlText == null)
        {
            StringBuffer sb = new StringBuffer();
            DatastoreIdentifier mainTableName = mainTable.getIdentifier();

            if (!multipleTablesReferenced)
            {
                sb.append(mainTable.toString());
                if (!mainAlias.equals(mainTableName))
                {
                    sb.append(' ').append(mainAlias);
                }
            }
            else
            {
                FetchStatement subQuery = mainTable.getStoreManager().getFetchStatement(mainTable);
                Iterator i = columns.iterator();
                while (i.hasNext())
                {
                    subQuery.select((DatastoreField)i.next());
                }

                // Assumed to not want any locking in this statement
                sb.append('(').append(subQuery.toString(false)).append(") ").append(mainAlias);
            }

            sqlText = sb.toString();
        }

        return sqlText;
    }
}