// (c) Microsoft Corporation 2005-2007.

#light

namespace Microsoft.FSharp.Collections

open Microsoft.FSharp.Core
open Microsoft.FSharp.Collections
open System
#if CLI_AT_MOST_1_1
open Microsoft.FSharp.Compatibility
#else
open System.Collections.Generic
#endif

/// <summary>
///   Immutable maps using structural comparison
/// </summary>
///
/// <performance> Maps based on structural comparison are  
/// efficient. They are not a suitable choice if keys are recursive data structures 
/// or require non-structural comparison semantics.</performance>
[<CompilationRepresentation(CompilationRepresentationFlags.ModuleSuffix)>]
module Map = 

    /// For use when not opening module, e.g. Map.t.  Used mainly in OCaml-compatible code.
    type ('key,'a) t = Map<'key,'a> 

    /// Return a new map with the binding added to the given map.
    val add: 'key -> 'a -> Map<'key,'a> -> Map<'key,'a>

    /// Return a new map made from the given bindings
    val of_list: ('key * 'a) list -> Map<'key,'a>

    /// Return a new map made from the given bindings
    val of_array: ('key * 'a) array -> Map<'key,'a>

    /// Return a new map made from the given bindings
    [<System.Obsolete("Consider using of_seq instead")>]
    val of_IEnumerable: #seq<'key * 'a> -> Map<'key,'a>

    /// Return a new map made from the given bindings
    val of_seq: #seq<'key * 'a> -> Map<'key,'a>

    /// View the collection as an enumerable sequence. This collection
    /// type is also directly compatible with 'seq&lt;KeyValuePair&lt;_,_&gt; &gt;'.
    ///
    /// Note this function returns a sequence of tuples, whereas the collection
    /// itself is compatible with the logically equivalent sequence of KeyValuePairs.
    /// Using sequences of tuples tends to be more convenient in F#, however the
    /// collection itself must enumerate KeyValuePairs to conform to the .NET
    /// design guidelines and the IDictionary interface.
    val to_seq: Map<'key,'a> -> seq<'key * 'a> 

    /// Returns a list of all key-value pairs in the mappinng
    val to_list: Map<'key,'a> -> list<'key * 'a> 

    /// Returns an array of all key-value pairs in the mappinng
    val to_array: Map<'key,'a> -> ('key * 'a) array

    /// The empty map
    [<GeneralizableValueAttribute>]
    val empty<'key,'a> : Map<'key,'a>

    /// Is the map empty?
    val is_empty: Map<'key,'a> -> bool

    /// Lookup an element in the map, raising [[Not_found]]/[[KeyNotFoundException]] if no binding
    /// exists in the map.
    val find: 'key -> Map<'key,'a> -> 'a

    /// Search the map looking for the first element where the given function returns a [[Some]] value
    val first: ('key -> 'a -> 'b option) -> Map<'key,'a> -> 'b option

    /// Fold over the bindings in the map 
    val fold: ('key -> 'a -> 'c -> 'c) -> Map<'key,'a> -> 'c -> 'c

    /// Apply the given function to each binding in the dictionary
    val iter: ('key -> 'a -> unit) -> Map<'key,'a> -> unit

    /// Build a new collection whose elements are the results of applying the given function
    /// to each of the elements of the collection.
    val map: ('a -> 'b) -> Map<'key, 'a> -> ('key,'b) Map

    /// Return true if the given predicate returns true for one of the
    /// bindings in the map.
    val exists: ('key -> 'a -> bool) -> Map<'key, 'a> -> bool

    /// Build a new map containing only the bindings for which the given predicate returns 'true'
    val filter: ('key -> 'a -> bool) -> Map<'key, 'a> -> Map<'key, 'a>

    /// Return true if the given predicate returns true for all of the
    /// bindings in the map.
    val for_all: ('key -> 'a -> bool) -> Map<'key, 'a> -> bool

    /// Build a new collection whose elements are the results of applying the given function
    /// to each of the elements of the collection. The index passed to the
    /// function indicates the index of element being transformed.
    val mapi: ('key -> 'a -> 'b) -> Map<'key,'a> -> Map<'key,'b>

    /// Test is an element is in the domain of the map
    val mem: 'key -> Map<'key,'a> -> bool

    /// Build two new maps, one containing the bindings for which the given predicate returns 'true',
    /// and the other the remaining bindings.
    val partition: ('key -> 'a -> bool) -> Map<'key, 'a> -> Map<'key, 'a> * Map<'key, 'a>

    /// Remove an element from the domain of the map.  No exception is raised if the element is not present.
    val remove: 'key -> Map<'key,'a> -> Map<'key,'a>

    /// Lookup an element in the map, returning a [[Some]] value if the element is in the domain 
    /// of the map and [[None]] if not.
    val tryfind: 'key -> Map<'key,'a> -> 'a option

    /// Evaluates the function on each mapping in the collection. Returns the key for the first mapping
    /// where the function returns 'trye'. Raise 'Not_found' if no such element exists.
    val find_index: ('key -> 'a -> bool) -> Map<'key,'a> -> 'key

    /// Return the key of the first mapping in the collection that satisfies the given predicate. 
    /// Return 'None' if no such element exists.
    val tryfind_index: ('key -> 'a -> bool) -> Map<'key,'a> -> 'key option


    [<Obsolete("Consider using Collections.Map<_,_> or just Map<_,_>")>]
    type Map<'key,'a> = Microsoft.FSharp.Collections.Map<'key,'a>

