(* (c) Microsoft Corporation. All rights reserved *)

//----------------------------------------------------------------------------
// Write Abstract IL structures at runtime using Reflection.Emit
//----------------------------------------------------------------------------

#light

module Microsoft.Research.AbstractIL.RuntimeWriter    
  
open Microsoft.Research.AbstractIL
open Microsoft.Research.AbstractIL.Internal
open Microsoft.Research.AbstractIL.Diagnostics 
open Microsoft.Research.AbstractIL.Extensions
open Microsoft.Research.AbstractIL.Extensions.ILX
open Microsoft.Research.AbstractIL.Extensions.ILX.Types
open Microsoft.Research.AbstractIL.AsciiWriter 
open Microsoft.Research.AbstractIL.IL
open Microsoft.Research.AbstractIL.Internal.Nums

open Microsoft.FSharp.Compatibility
open Microsoft.FSharp.Collections.List
open Microsoft.FSharp.Text.Printf

open System
open System.Reflection
open System.Reflection.Emit
open System.Runtime.InteropServices

let codeLabelOrder = Pervasives.compare : code_label -> code_label -> int
let stringOrder = Pervasives.compare : string -> string -> int

let verbose = false
let replicate n x =   let rec go m = if m >= n then [] else x::go(m+1) in go 0

//----------------------------------------------------------------------------
// misc
//----------------------------------------------------------------------------

let flagsL     xs = Enum.combine xs  // less verbose
let flagsIf  b x  = if b then x else flagsL []
let flagsIfL b xs = if b then flagsL xs else flagsL []

let isNonNull (x :> Object) = match (x :> Object) with null -> false | _ -> true
let nonNull msg x = if isNonNull x then x else failwith ("null: " ^ msg) 
let forceM' x m str = match Zmap.tryfind x m with Some y -> y | None -> failwithf "forceM: %s: x = %s" str (any_to_string x)

let equalTypes (s:Type) (t:Type) = s.Equals(t)
let equalTypeLists ss tt =  (length ss = length tt) && for_all2 equalTypes ss tt

let getGenericArgumentsOfType (typT :> Type) = 
#if CLI_AT_LEAST_2_0
    if typT .IsGenericType   then typT .GetGenericArguments() else [| |]
#else
    [| |] 
#endif
let getGenericArgumentsOfMethod (methI :> MethodInfo) = 
#if CLI_AT_LEAST_2_0
    if methI.IsGenericMethod then methI.GetGenericArguments() else [| |] 
#else
    [| |] 
#endif

let getTypeConstructor (ty:>Type) = 
#if CLI_AT_LEAST_2_0
    if ty.IsGenericType then ty.GetGenericTypeDefinition() else (ty :> Type) 
#else
    (ty :> Type)
#endif

let isGenericType (ty:>Type) = 
#if CLI_AT_LEAST_2_0
    ty.IsGenericType
#else
    false
#endif


let makeGenericType (ty:>Type) tyargs = 
#if CLI_AT_LEAST_2_0
    ty.MakeGenericType(tyargs)
#else
    failwith "makeGenericType: emitting generics is not supported by this compiler"
#endif

let makeGenericMethod (meth:>MethodInfo) tyargs = 
#if CLI_AT_LEAST_2_0
    meth.MakeGenericMethod(tyargs)
#else
    meth
#endif

let getConstructor parentTI (consInfo :> ConstructorInfo) = 
#if CLI_AT_LEAST_2_0
    TypeBuilder.GetConstructor(parentTI,consInfo) 
#else
    (consInfo :> ConstructorInfo)
#endif

let getField (parentTI :> Type) (fieldInfo :> FieldInfo) : FieldInfo = 
#if CLI_AT_LEAST_2_0
    TypeBuilder.GetField(parentTI,fieldInfo) 
#else
    (fieldInfo :> FieldInfo)
#endif

let getMethod (parentTI :> Type) (methInfo :> MethodInfo)  : MethodInfo = 
#if CLI_AT_LEAST_2_0
    TypeBuilder.GetMethod(parentTI,methInfo) 
#else
    (methInfo :> MethodInfo)
#endif

let makeArrayType (ty:>Type) rank = 
#if CLI_AT_LEAST_2_0
    if rank = 1 then ty.MakeArrayType() else ty.MakeArrayType(rank)
#else
    let nm2 = ty.FullName + "[" + String.concat "" (replicate (rank-1) ",") + "]" + ", " + ty.Assembly.FullName
    Type.GetType(nm2)
#endif

let makeByRefType (ty:>Type) = 
#if CLI_AT_LEAST_2_0
    ty.MakeByRefType()
#else
    Type.GetType(ty.FullName + "&, " + ty.Assembly.FullName)
#endif

let makePointerType (ty:>Type) = 
#if CLI_AT_LEAST_2_0
    ty.MakePointerType()
#else
    Type.GetType(ty.FullName + "*, " + ty.Assembly.FullName)
#endif

//----------------------------------------------------------------------------
// convAssemblyRef
//----------------------------------------------------------------------------

let convAssemblyRef aref = 
    // -assemRefHash: bytes option;
    // -assemRefRetargetable: bool;  (* The assembly can be retargeted (at runtime) to be from a different publisher. *)
    let asmName = new System.Reflection.AssemblyName()
    asmName.Name    <- aref.assemRefName;
    (* XXX - to set HashAlgorithm - bytes -> enum needed - gap *)   
    (match aref.assemRefPublicKeyInfo with 
     | None -> ()
     | Some (PublicKey      bytes) -> asmName.SetPublicKey(bytes)
     | Some (PublicKeyToken bytes) -> asmName.SetPublicKeyToken(bytes));
    let setVersion (major,minor,build,rev) = 
       asmName.Version <- System.Version (major |> u16_to_int,
                                          minor |> u16_to_int,
                                          build |> u16_to_int,
                                          rev   |> u16_to_int)
    Option.iter setVersion aref.assemRefVersion;
    //  asmName.ProcessorArchitecture <- System.Reflection.ProcessorArchitecture.MSIL;
    //Option.iter (fun name -> asmName.CultureInfo <- System.Globalization.CultureInfo.CreateSpecificCulture(name)) aref.assemRefLocale;
    asmName.CultureInfo <- System.Globalization.CultureInfo.InvariantCulture;
    asmName


/// Convert an Abstract IL type reference to Reflection.Emit System.Type value
/// REVIEW: This ought to be an adequate substitute for this whole function, but it needs 
/// to be thoroughly tested.
///    Type.GetType(qualified_name_of_tref tref) 
/// []              ,name -> name
/// [ns]            ,name -> ns+name
/// [ns;typeA;typeB],name -> ns+typeA+typeB+name
let getTRefType tref = 
    if verbose then dprintf1 "- getTRefType: %s\n" tref.trefName;

    let prefix = 
        match tref.trefNested with
        | []             -> ""
        | enclosingNames -> String.concat "+" enclosingNames ^ "+"
    let qualifiedName = prefix ^ tref.trefName (* e.g. Name.Space.Class+NestedClass, assembly *)
    if verbose then dprintf1 "- - qualifiedName = %s\n" qualifiedName;    
    match tref.trefScope with
    | ScopeRef_assembly asmref ->
        let asmName    = convAssemblyRef asmref
        let currentDom = System.AppDomain.CurrentDomain
        let assembly   = currentDom.Load(asmName)
        if verbose then dprintf1 "- - assembly = %s\n" (assembly.ToString());
        let nm = prefix ^ tref.trefName 
        let typT       = assembly.GetType(nm)
        typT |> nonNull (Printf.sprintf "GetTRefType (assembly %O, ty = %s)" assembly nm)
    | ScopeRef_module _ 
    | ScopeRef_local _ ->
        if verbose then dprintf0 "- - module/local\n";
        let typT = Type.GetType(qualifiedName,true) // just lookup for now - XXX - no trans
        if verbose then dprintf1 "- getTRefType: Got %d\n" (if isNonNull typT then 1 else 0);
        typT |> nonNull (Printf.sprintf "GetTRefType (local %s)" qualifiedName)

/// The global environment
type cenv = 
    { ilg: mscorlib_refs }

/// The (local) emitter env (state). Some of them are scoped fields which should be passed as 
/// and 'environment' parameter. Some of these fields (the ones marked Acc) are effectively accumulators
/// and are probably better placed as hash tables in the global environment.
type emEnv =
    { emTypMapAcc   : (type_ref,Type * TypeBuilder * IL.type_def) Zmap.map;
      emConsMapAcc  : (method_ref,ConstructorBuilder) Zmap.map;    
      emMethMapAcc  : (method_ref,MethodBuilder) Zmap.map;
      emFieldMapAcc : (field_ref,FieldBuilder) Zmap.map;
      emPropMapAcc  : (property_ref,PropertyBuilder) Zmap.map;
      emLocalsAcc   : LocalBuilder array;
      emLabelsAcc   : (IL.code_label,Label) Zmap.map;
      emTyvarsEnv   : Type array list; // stack
      emEntryPtsAcc : (TypeBuilder * string) list }
  
let type_ref_order      = Pervasives.compare : type_ref      -> type_ref      -> int
let method_ref_order    = Pervasives.compare : method_ref    -> method_ref    -> int
let field_ref_order     = Pervasives.compare : field_ref     -> field_ref     -> int
let property_ref_order  = Pervasives.compare : property_ref  -> property_ref  -> int

let emEnv0 = 
    { emTypMapAcc   = Zmap.empty type_ref_order;
      emConsMapAcc  = Zmap.empty method_ref_order;
      emMethMapAcc  = Zmap.empty method_ref_order;
      emFieldMapAcc = Zmap.empty field_ref_order;
      emPropMapAcc = Zmap.empty property_ref_order;
      emLocalsAcc   = [| |];
      emLabelsAcc   = Zmap.empty codeLabelOrder;
      emTyvarsEnv   = [];
      emEntryPtsAcc = []; }

let envBindTypeRef emEnv tref (typT,typB,typeDef)= 
    if verbose then dprintf1 "- envBindTypeRef: %s\n" tref.trefName;
    match typT with 
    | null -> failwithf "binding null type in envBindTypeRef: %s\n" tref.trefName;
    | _ -> {emEnv with emTypMapAcc = Zmap.add tref (typT,typB,typeDef) emEnv.emTypMapAcc}

let envGetTypT emEnv tref = 
    if verbose then dprintf1 "- envGetTypT: %s\n" tref.trefName;
    match Zmap.tryfind tref emEnv.emTypMapAcc with
    | Some (typT,typB,typeDef) -> typT |> nonNull "envGetTypT: null type table?"
    | None                     -> getTRefType tref

let envBindConsRef emEnv mref consB = 
    if verbose then dprintf1 "- envBindConsRef: %s\n" mref.mrefName;
    {emEnv with emConsMapAcc = Zmap.add mref consB emEnv.emConsMapAcc}

let envGetConsB emEnv mref = 
    if verbose then dprintf1 "- envGetConsB: %s\n" mref.mrefName;
    forceM' mref emEnv.emConsMapAcc "envGetConsB: failed"

let envBindMethodRef emEnv mref methB = 
    if verbose then dprintf1 "- envBindMethodRef: %s\n" mref.mrefName;
    {emEnv with emMethMapAcc = Zmap.add mref methB emEnv.emMethMapAcc}

let envGetMethB emEnv mref = 
    if verbose then dprintf1 "- envGetMethB: %s\n" mref.mrefName;
    forceM' mref emEnv.emMethMapAcc "envGetMethB: failed"

let envBindFieldRef emEnv fref fieldB = 
    if verbose then dprintf1 "- envBindFieldRef: %s\n" fref.frefName;
    {emEnv with emFieldMapAcc = Zmap.add fref fieldB emEnv.emFieldMapAcc}

let envGetFieldB emEnv fref =
    forceM' fref emEnv.emFieldMapAcc "- envGetMethB: failed"
      
let envBindPropRef emEnv (pref:PropertyRef) propB = 
    if verbose then dprintf1 "- envBindPropRef: %s\n" pref.Name;
    {emEnv with emPropMapAcc = Zmap.add pref propB emEnv.emPropMapAcc}

let envGetPropB emEnv pref =
    forceM' pref emEnv.emPropMapAcc "- envGetPropB: failed"
      
let envGetTypB emEnv tref = 
    if verbose then dprintf1 "- envGetTypB: %s\n" tref.trefName; 
    forceM' tref emEnv.emTypMapAcc "envGetTypB: failed"
    |> (fun (typT,typB,typeDef) -> typB)
                 
let envGetTypeDef emEnv tref = 
    if verbose then dprintf1 "- envGetTypeDef: %s\n" tref.trefName; 
    forceM' tref emEnv.emTypMapAcc "envGetTypeDef: failed"
    |> (fun (typT,typB,typeDef) -> typeDef)
                 
let envSetLocals emEnv locs = assert (emEnv.emLocalsAcc.Length = 0); // check "locals" is not yet set (scopes once only)
                              {emEnv with emLocalsAcc = locs}
let envGetLocal  emEnv i    = emEnv.emLocalsAcc.(i) // implicit bounds checking

let envSetLabel emEnv name lab =
    assert (not (Zmap.mem name emEnv.emLabelsAcc));
    {emEnv with emLabelsAcc = Zmap.add name lab emEnv.emLabelsAcc}
    
let envGetLabel emEnv name = 
    //if verbose then dprintf1 "- envGetLabel: %s\n" name;
    Zmap.find name emEnv.emLabelsAcc

let envPushTyvars emEnv typs =  {emEnv with emTyvarsEnv = typs :: emEnv.emTyvarsEnv}
let envPopTyvars  emEnv      =  {emEnv with emTyvarsEnv = List.tl emEnv.emTyvarsEnv}
let envGetTyvar   emEnv u16  =  
    let i = int u16 
    let fail() = 
        Printf.eprintf "want tyvar #%d, but only had %d tyvars" i (Array.length (Array.concat emEnv.emTyvarsEnv));
        assert false
    let rec find i (tvss : Type array list) = 
       match tvss  with
       | []     -> fail()
       | tvs:: t -> if i<0 then fail()
                    elif i>= tvs.Length  then find (i - tvs.Length) t
                    else tvs.[i]
    find i emEnv.emTyvarsEnv

let isEmittedTypeRef emEnv tref = Zmap.mem tref emEnv.emTypMapAcc

let envAddEntryPt  emEnv mref = {emEnv with emEntryPtsAcc = mref::emEnv.emEntryPtsAcc}
let envPopEntryPts emEnv      = {emEnv with emEntryPtsAcc = []},emEnv.emEntryPtsAcc

//----------------------------------------------------------------------------
// convCallConv
//----------------------------------------------------------------------------

let convCallConv (Callconv (hasThis,basic)) =
    if verbose then dprintf0 "- convCallconv\n";  
    let ccA = match hasThis with CC_static            -> CallingConventions.Standard
                               | CC_instance_explicit -> CallingConventions.ExplicitThis
                               | CC_instance          -> CallingConventions.HasThis
    let ccB = match basic with   CC_default  -> flagsL[]
                               | CC_cdecl    -> flagsL[]
                               | CC_stdcall  -> flagsL[]
                               | CC_thiscall -> flagsL[] // XXX: check all these
                               | CC_fastcall -> flagsL[]
                               | CC_vararg   -> CallingConventions.VarArgs
    flagsL [ccA;ccB]


//----------------------------------------------------------------------------
// convType
//----------------------------------------------------------------------------

let rec convTypeSpec emEnv tspec =
    if verbose then dprintf1 "- convTypeSpec: %s\n" (tname_of_tspec tspec);
    let typT   = envGetTypT emEnv tspec.tspecTypeRef
    let tyargs = map (convType emEnv) tspec.tspecInst
    match tyargs,isGenericType typT with
    | tyargs,true  -> makeGenericType typT (Array.of_list tyargs) |> nonNull "convTypeSpec: generic" 
    | []    ,false -> typT                                        |> nonNull "convTypeSpec: non generic" 
    | h::_  ,false -> failwithf "- convTypeSpec: non-generic type '%O' has type instance of length %d and head %O?" typT (length tyargs) h
      
and convType emEnv typ =
    if verbose then dprintf2 "- convType: %a\n" AsciiWriter.output_typ typ;
    match typ with
    | Type_void               -> Type.GetType("System.Void",true)
    | Type_array (shape,base) -> 
        let baseT = convType emEnv base |> nonNull "convType: array base"
        let nDims = rank_of_array_shape shape
        // MakeArrayType()  returns "base[]"
        // MakeArrayType(1) returns "base[*]"
        // MakeArrayType(2) returns "base[,]"
        // MakeArrayType(3) returns "base[,,]"
        // All non-equal.
        if nDims=1
        then makeArrayType baseT 1
        else makeArrayType baseT (rank_of_array_shape shape)
    | Type_value tspec        -> convTypeSpec emEnv tspec        |> nonNull "convType: value"
    | Type_boxed tspec        -> convTypeSpec emEnv tspec        |> nonNull "convType: boxed"
    | Type_ptr base           -> let baseT = convType emEnv base |> nonNull "convType: ptr base"
                                 makePointerType baseT           |> nonNull "convType: ptr" 
    | Type_byref base         -> let baseT = convType emEnv base |> nonNull "convType: byref base"
                                 makeByRefType baseT             |> nonNull "convType: byref" 
    | Type_tyvar tv           -> envGetTyvar emEnv tv            |> nonNull "convType: tyvar" 
    | Type_other(e) when is_ilx_ext_typ e -> 
        match dest_ilx_ext_typ e with 
        | EType_erasable_array(shape,ty) -> convType emEnv (Type_array(shape,ty))
  // XXX: REVIEW: complete the following cases.                                                        
    | Type_fptr callsig -> failwith "convType: fptr"
    | Type_modified _   -> failwith "convType: modified"
    | Type_other extty  -> failwith "convType: other"

//----------------------------------------------------------------------------
// buildGenParams
//----------------------------------------------------------------------------

#if CLI_AT_MOST_1_1
#else
let buildGenParamsPass1 emEnv defineGenericParameters (gps : genparam list) = 
    if verbose then dprintf0 "buildGenParamsPass1\n"; 
    match gps with 
    | [] -> () 
    | gps ->
        let gpsNames = map (fun gp -> gp.gpName) gps
        defineGenericParameters (Array.of_list gpsNames)  |> ignore


let buildGenParamsPass1b emEnv (genArgs : Type array) (gps : genparam list) = 
    if verbose then dprintf0 "buildGenParamsPass1b\n"; 
    let genpBs =  genArgs |>  Array.map (fun x -> (x :?> GenericTypeParameterBuilder)) 
    gps |> List.iteri (fun i (gp:genparam) ->
        let gpB = genpBs.(i)
        // the Constraints are either the parent (base) type or interfaces.
        let constraintTs = map (convType emEnv) gp.Constraints
        let interfaceTs,baseTs = List.partition (fun (typ:System.Type) -> typ.IsInterface) constraintTs
        // set base type constraint
        (match baseTs with
            []      -> () // Q: should a baseType be set? It is in some samples. Should this be a failure case?
          | [baseT] -> gpB.SetBaseTypeConstraint(baseT)
          | _       -> failwith "buildGenParam: multiple base types"
        );
        // set interface contraints (interfaces that instances of gp must meet)
        gpB.SetInterfaceConstraints(Array.of_list interfaceTs);

        let flags = GenericParameterAttributes.None 
        let flags =
           match gp.gpVariance with
           | NonVariant    -> flags
           | CoVariant     -> flags ||| GenericParameterAttributes.Covariant
           | ContraVariant -> flags ||| GenericParameterAttributes.Contravariant
       
        let flags = if gp.gpReferenceTypeConstraint        then flags ||| GenericParameterAttributes.ReferenceTypeConstraint        else flags 
        let flags = if gp.gpNotNullableValueTypeConstraint then flags ||| GenericParameterAttributes.NotNullableValueTypeConstraint else flags
        let flags = if gp.gpDefaultConstructorConstraint   then flags ||| GenericParameterAttributes.DefaultConstructorConstraint   else flags
        
        gpB.SetGenericParameterAttributes(flags)
    )
#endif

//----------------------------------------------------------------------------
// convFieldInit
//----------------------------------------------------------------------------

let convFieldInit x = 
    match x with 
    | FieldInit_bytes bytes    -> (bytes  :> Object)  
    | FieldInit_bool bool      -> (bool   :> Object)
    | FieldInit_char u16       -> ((Nums.u16_to_unichar u16)    :> Object)
    | FieldInit_int8 i8        -> (i8     :> Object)
    | FieldInit_int16 i16      -> (i16    :> Object)
    | FieldInit_int32 i32      -> (i32    :> Object)
    | FieldInit_int64 i64      -> (i64    :> Object)
    | FieldInit_uint8 u8       -> (u8     :> Object)
    | FieldInit_uint16 u16     -> (u16    :> Object)
    | FieldInit_uint32 u32     -> (u32    :> Object)
    | FieldInit_uint64 u64     -> (u64    :> Object)
    | FieldInit_float32 ieee32 -> (ieee32 :> Object)
    | FieldInit_float64 ieee64 -> (ieee64 :> Object)
    | FieldInit_ref            -> (null :> Object)

//----------------------------------------------------------------------------
// Some types require hard work...
//----------------------------------------------------------------------------

#if CLI_AT_MOST_1_1
#else
// This is gross. TypeBuilderInstantiation should really be a public type, since we
// have to use alternative means for various Method/Field/Constructor lookups.  However since 
// it isn't we resort to this technique...
let TypeBuilderInstantiationT = Type.GetType("System.Reflection.Emit.TypeBuilderInstantiation" )
#endif

let typeIsNotQueryable (typ : Type) =
    (typ :? TypeBuilder) 
#if CLI_AT_LEAST_2_0
    or ((typ.GetType()).Equals(TypeBuilderInstantiationT))
#endif

//----------------------------------------------------------------------------
// convFieldSpec
//----------------------------------------------------------------------------

let queryableTypeGetField emEnv (parentT:Type) fref  =
    let tyargTs  = getGenericArgumentsOfType parentT
    parentT.GetField(fref.frefName,
                     flagsL [ BindingFlags.Public;
                              BindingFlags.NonPublic;
                              BindingFlags.Instance;
                              BindingFlags.Static ])  
        |> nonNull (sprintf "queryableTypeGetField: %O::%s" parentT fref.frefName)
    
let nonQueryableTypeGetField (parentTI:Type) (fieldInfo :> FieldInfo) : FieldInfo = 
    if isGenericType parentTI then getField parentTI fieldInfo else (fieldInfo :> FieldInfo)


let convFieldSpec emEnv fspec =
    let fref = fspec.fspecFieldRef
    let tref = fref.frefParent 
    let parentTI = convType emEnv fspec.fspecEnclosingType
    if Zmap.mem tref emEnv.emTypMapAcc then
        let fieldB = envGetFieldB emEnv fref
        nonQueryableTypeGetField parentTI fieldB
        // Prior type.
    elif typeIsNotQueryable parentTI then 
        let parentT = getTypeConstructor parentTI
        let fieldInfo = queryableTypeGetField emEnv parentT  fref 
        nonQueryableTypeGetField parentTI fieldInfo
    else 
        queryableTypeGetField emEnv parentTI fspec.fspecFieldRef

//----------------------------------------------------------------------------
// convMethodRef
//----------------------------------------------------------------------------

let queryableTypeGetMethodBySearch emEnv parentT mref =
    assert(not (typeIsNotQueryable(parentT)));
    let cconv = (if is_static_callconv (callconv_of_mref mref) then BindingFlags.Static else BindingFlags.Instance)
    let methInfos = parentT.GetMethods(flagsL [ cconv; 
                                                BindingFlags.Public; 
                                                BindingFlags.NonPublic]) |> CompatArray.to_list
      (* First, filter on name, if unique, then binding "done" *)
    let tyargTs = getGenericArgumentsOfType parentT      
    let methInfos = methInfos |> List.filter (fun methInfo -> methInfo.Name = mref.mrefName)
    match methInfos with 
    | [methInfo] -> 
        if verbose then dprintf2 "Got '%O::%s' by singluar name match\n" parentT mref.mrefName;
        methInfo
    | _ ->
      (* Second, type match. Note type erased (non-generic) F# code would not type match but they have unique names *)
        let select (methInfo:MethodInfo) =
            let tyargTIs = tyargTs
            (* mref implied Types *)
            let mtyargTIs = getGenericArgumentsOfMethod methInfo 
            if Array.length mtyargTIs  <> mref.mrefArity then false (* method generic arity mismatch *) else
            let argTs,resT = 
                let emEnv = envPushTyvars emEnv (Array.append tyargTs mtyargTIs)
                let argTs = map (convType emEnv) mref.mrefArgs
                let resT  = convType emEnv mref.mrefReturn
                argTs,resT 
          
          (* methInfo implied Types *)
            let haveArgTs = 
              let parameters = CompatArray.to_list (methInfo.GetParameters())
              parameters |> map (fun param -> param.ParameterType) 
         
            let haveResT  = methInfo.ReturnType
          (* check for match *)
            if length argTs <> length haveArgTs then false (* method argument length mismatch *) else
            let res = equalTypeLists (resT::argTs) (haveResT::haveArgTs) 
            res
       
        match List.tryfind select methInfos with
        | None          -> failwith "convMethodRef: could not bind to method"
        | Some methInfo -> methInfo (* return MethodInfo for (generic) type's (generic) method *)
                           |> nonNull "convMethodRef"
          
let queryableTypeGetMethod emEnv parentT mref =
    assert(not (typeIsNotQueryable(parentT)));
    if mref.mrefArity = 0 then 
        let tyargTs = getGenericArgumentsOfType parentT      
        let argTs,resT = 
            let emEnv = envPushTyvars emEnv tyargTs
            let argTs = CompatArray.of_list (map (convType emEnv) mref.mrefArgs)
            let resT  = convType emEnv mref.mrefReturn
            argTs,resT 
        let stat = is_static_callconv (callconv_of_mref mref)
        if verbose then dprintf4 "Using GetMethod to get '%O::%s', static = %b, parentT = %s\n" parentT mref.mrefName stat parentT.AssemblyQualifiedName;
        argTs |> CompatArray.iteri (fun i ty -> if verbose then dprintf2 "arg %d = %O\n" i ty);
        let cconv = (if stat then BindingFlags.Static else BindingFlags.Instance)
        let methInfo = 
            try 
              parentT.GetMethod(mref.mrefName,
                                flagsL [ cconv; 
                                         BindingFlags.Public; 
                                         BindingFlags.NonPublic],
                                null,
                                argTs,
                                (null:ParameterModifier[])) 
            // This can fail if there is an ambiguity w.r.t. return type 
            with _ -> null
        if (isNonNull methInfo && equalTypes resT methInfo.ReturnType) then 
            (if verbose then dprintf3 "Got method '%O' using GetMethod, resT = %O, haveResT = %O\n" methInfo resT methInfo.ReturnType;
             methInfo)
        else
            (if verbose then dprintf1 "**** Failed lookup or Incorrect return type for '%O' using GetMethod\n" methInfo;
             queryableTypeGetMethodBySearch emEnv parentT mref)
    else 
        if verbose then dprintf2 "Using queryableTypeGetMethodBySearch to get '%O::%s'\n" parentT mref.mrefName;
        queryableTypeGetMethodBySearch emEnv parentT mref

let nonQueryableTypeGetMethod (parentTI:Type) (methInfo :> MethodInfo) : MethodInfo = 
    if (isGenericType parentTI &&
        not (equalTypes parentTI (getTypeConstructor parentTI))) 
    then getMethod parentTI methInfo 
    else (methInfo :> MethodInfo)

let convMethodRef emEnv (parentTI:Type) mref =
    if verbose then dprintf2 "- convMethodRef %s %s\n" (tname_of_tref mref.mrefParent) mref.mrefName;
    let parent = mref.mrefParent
    if isEmittedTypeRef emEnv parent then
        // Emitted type, can get fully generic MethodBuilder from env.
        let methB = envGetMethB emEnv mref
        if verbose then dprintf0 "- convMethodRef, isEmitted = true\n";
        nonQueryableTypeGetMethod parentTI methB
        |> nonNull "convMethodRef (emitted)"
    else
        // Prior type.
        if typeIsNotQueryable parentTI then 
            let parentT = getTypeConstructor parentTI
            let methInfo = queryableTypeGetMethod emEnv parentT mref 
            nonQueryableTypeGetMethod parentTI methInfo
        else 
            queryableTypeGetMethod emEnv parentTI mref 

//----------------------------------------------------------------------------
// convMethodSpec
//----------------------------------------------------------------------------
      
let convMethodSpec emEnv (mspec:MethodSpec) =
    if verbose then dprintf2 "- convMethodSpec %s with inst=%d\n" (name_of_mspec mspec) (length mspec.GenericArguments);
    let typT     = convType emEnv mspec.EnclosingType       (* (instanced) parent Type *)
    let methInfo = convMethodRef emEnv typT mspec.MethodRef (* (generic)   method of (generic) parent *)
    let methInfo =
      if mspec.GenericArguments = [] then methInfo (* non generic *) 
      else 
        let minstTs  = Array.of_list (map (convType emEnv) mspec.GenericArguments)
        makeGenericMethod methInfo minstTs (* instance method *)
    methInfo |> nonNull "convMethodSpec"

//----------------------------------------------------------------------------
// - QueryableTypeGetConstructors: get a constructor on a non-TypeBuilder type
//----------------------------------------------------------------------------

let queryableTypeGetConstructor emEnv (parentT:Type) mref  =
    let tyargTs  = getGenericArgumentsOfType parentT
    let reqArgTs  = 
        let emEnv = envPushTyvars emEnv tyargTs
        CompatArray.of_list (map (convType emEnv) mref.mrefArgs)
    parentT.GetConstructor(flagsL [ BindingFlags.Public;
                                    BindingFlags.NonPublic;
                                    BindingFlags.Instance ],
                            null, 
                            reqArgTs,
                            null)  

let nonQueryableTypeGetConstructor (parentTI:Type) (consInfo :> ConstructorInfo) : ConstructorInfo = 
    if isGenericType parentTI then getConstructor parentTI consInfo else (consInfo :> ConstructorInfo)

//----------------------------------------------------------------------------
// convConstructorSpec (like convMethodSpec) 
//----------------------------------------------------------------------------

let convConstructorSpec emEnv (mspec:MethodSpec) =
    if verbose then dprintf3 "- convConstructorSpec %s %s with inst=%d\n" (mspec.MethodRef.mrefParent.trefName) (name_of_mspec mspec) (length mspec.GenericArguments);
    let mref   = mspec.MethodRef
    let parentTI = convType emEnv mspec.EnclosingType
    if verbose then dprintf1 "- convConstructorSpec: prior type, parentTI = %O\n" parentTI;
    if isEmittedTypeRef emEnv mref.mrefParent then
        let consB = envGetConsB emEnv mref
        nonQueryableTypeGetConstructor parentTI consB |> nonNull "convConstructorSpec: (emitted)"
    else
        if typeIsNotQueryable parentTI then 
            let parentT  = getTypeConstructor parentTI       
            let ctorG = queryableTypeGetConstructor emEnv parentT mref 
            nonQueryableTypeGetConstructor parentTI ctorG
        else
            queryableTypeGetConstructor emEnv parentTI mref 

//----------------------------------------------------------------------------
// emitLabelMark, defineLabel
//----------------------------------------------------------------------------

let emitLabelMark emEnv (ilG:ILGenerator) (label:code_label) =
    //if verbose then dprintf1 "- emitLabelMark %s\n" (label:string);
    let lab = envGetLabel emEnv label
    ilG.MarkLabel(lab)

let defineLabel (ilG:ILGenerator) emEnv (label:code_label) =
    //if verbose then dprintf1 "- defineLabel %s\n" (label:string);  
    let lab = ilG.DefineLabel()
    envSetLabel emEnv label lab

//----------------------------------------------------------------------------
// emitCustomAttrs
//----------------------------------------------------------------------------

let convCustomAttr emEnv cattr =
    let methInfo = 
       match convConstructorSpec emEnv cattr.customMethod with 
       | null -> failwithf "convCustomAttr: %A" cattr.customMethod
       | res -> res
    let data = cattr.customData 
    (methInfo,data)

let emitCustomAttr  emEnv add cattr  = add (convCustomAttr emEnv cattr)
let emitCustomAttrs emEnv add cattrs = List.iter (emitCustomAttr emEnv add) (dest_custom_attrs cattrs)

//----------------------------------------------------------------------------
// emitInstr cenv - I_arith
//----------------------------------------------------------------------------
 
let rec emitInstrI_arith emEnv (ilG:ILGenerator) x = 
    match x with
    | AI_add                      -> ilG.Emit(OpCodes.Add) 
    | AI_add_ovf                  -> ilG.Emit(OpCodes.Add_Ovf) 
    | AI_add_ovf_un               -> ilG.Emit(OpCodes.Add_Ovf_Un)
    | AI_and                      -> ilG.Emit(OpCodes.And)
    | AI_div                      -> ilG.Emit(OpCodes.Div)
    | AI_div_un                   -> ilG.Emit(OpCodes.Div_Un)
    | AI_ceq                      -> ilG.Emit(OpCodes.Ceq)
    | AI_cgt                      -> ilG.Emit(OpCodes.Cgt)
    | AI_cgt_un                   -> ilG.Emit(OpCodes.Cgt_Un)
    | AI_clt                      -> ilG.Emit(OpCodes.Clt)
    | AI_clt_un                   -> ilG.Emit(OpCodes.Clt_Un)
    (* conversion *)
    | AI_conv dt                  -> (match dt with
                                        | DT_I   -> ilG.Emit(OpCodes.Conv_I)
                                        | DT_I1  -> ilG.Emit(OpCodes.Conv_I1)
                                        | DT_I2  -> ilG.Emit(OpCodes.Conv_I2)
                                        | DT_I4  -> ilG.Emit(OpCodes.Conv_I4)
                                        | DT_I8  -> ilG.Emit(OpCodes.Conv_I8)
                                        | DT_U   -> ilG.Emit(OpCodes.Conv_U)      
                                        | DT_U1  -> ilG.Emit(OpCodes.Conv_U1)      
                                        | DT_U2  -> ilG.Emit(OpCodes.Conv_U2)      
                                        | DT_U4  -> ilG.Emit(OpCodes.Conv_U4)      
                                        | DT_U8  -> ilG.Emit(OpCodes.Conv_U8)
                                        | DT_R   -> ilG.Emit(OpCodes.Conv_R_Un)
                                        | DT_R4  -> ilG.Emit(OpCodes.Conv_R4)
                                        | DT_R8  -> ilG.Emit(OpCodes.Conv_R8)
                                        | DT_REF -> failwith "AI_conv DT_REF?" // XXX - check
                     )
    (* conversion - ovf checks *)
    | AI_conv_ovf dt              -> (match dt with
                                        | DT_I   -> ilG.Emit(OpCodes.Conv_Ovf_I)
                                        | DT_I1  -> ilG.Emit(OpCodes.Conv_Ovf_I1)
                                        | DT_I2  -> ilG.Emit(OpCodes.Conv_Ovf_I2)
                                        | DT_I4  -> ilG.Emit(OpCodes.Conv_Ovf_I4)
                                        | DT_I8  -> ilG.Emit(OpCodes.Conv_Ovf_I8)
                                        | DT_U   -> ilG.Emit(OpCodes.Conv_Ovf_U)      
                                        | DT_U1  -> ilG.Emit(OpCodes.Conv_Ovf_U1)      
                                        | DT_U2  -> ilG.Emit(OpCodes.Conv_Ovf_U2)      
                                        | DT_U4  -> ilG.Emit(OpCodes.Conv_Ovf_U4)
                                        | DT_U8  -> ilG.Emit(OpCodes.Conv_Ovf_U8)
                                        | DT_R   -> failwith "AI_conv_ovf DT_R?" // XXX - check       
                                        | DT_R4  -> failwith "AI_conv_ovf DT_R4?" // XXX - check       
                                        | DT_R8  -> failwith "AI_conv_ovf DT_R8?" // XXX - check       
                                        | DT_REF -> failwith "AI_conv_ovf DT_REF?" // XXX - check
                     )
    (* conversion - ovf checks and unsigned *)
    | AI_conv_ovf_un dt           -> (match dt with
                                        | DT_I   -> ilG.Emit(OpCodes.Conv_Ovf_I_Un)
                                        | DT_I1  -> ilG.Emit(OpCodes.Conv_Ovf_I1_Un)
                                        | DT_I2  -> ilG.Emit(OpCodes.Conv_Ovf_I2_Un)
                                        | DT_I4  -> ilG.Emit(OpCodes.Conv_Ovf_I4_Un)
                                        | DT_I8  -> ilG.Emit(OpCodes.Conv_Ovf_I8_Un)
                                        | DT_U   -> ilG.Emit(OpCodes.Conv_Ovf_U_Un)            
                                        | DT_U1  -> ilG.Emit(OpCodes.Conv_Ovf_U1_Un)      
                                        | DT_U2  -> ilG.Emit(OpCodes.Conv_Ovf_U2_Un)      
                                        | DT_U4  -> ilG.Emit(OpCodes.Conv_Ovf_U4_Un)      
                                        | DT_U8  -> ilG.Emit(OpCodes.Conv_Ovf_U8_Un)
                                        | DT_R   -> failwith "AI_conv_ovf_un DT_R?" // XXX - check       
                                        | DT_R4  -> failwith "AI_conv_ovf_un DT_R4?" // XXX - check       
                                        | DT_R8  -> failwith "AI_conv_ovf_un DT_R8?" // XXX - check       
                                        | DT_REF -> failwith "AI_conv_ovf_un DT_REF?" // XXX - check
                     )
    | AI_mul                      -> ilG.Emit(OpCodes.Mul)
    | AI_mul_ovf                  -> ilG.Emit(OpCodes.Mul_Ovf)
    | AI_mul_ovf_un               -> ilG.Emit(OpCodes.Mul_Ovf_Un)
    | AI_rem                      -> ilG.Emit(OpCodes.Rem)
    | AI_rem_un                   -> ilG.Emit(OpCodes.Rem_Un)
    | AI_shl                      -> ilG.Emit(OpCodes.Shl)
    | AI_shr                      -> ilG.Emit(OpCodes.Shr)
    | AI_shr_un                   -> ilG.Emit(OpCodes.Shr_Un)
    | AI_sub                      -> ilG.Emit(OpCodes.Sub)
    | AI_sub_ovf                  -> ilG.Emit(OpCodes.Sub_Ovf)
    | AI_sub_ovf_un               -> ilG.Emit(OpCodes.Sub_Ovf_Un)
    | AI_xor                      -> ilG.Emit(OpCodes.Xor)
    | AI_or                       -> ilG.Emit(OpCodes.Or)
    | AI_neg                      -> ilG.Emit(OpCodes.Neg)
    | AI_not                      -> ilG.Emit(OpCodes.Not)
    | AI_ldnull                   -> ilG.Emit(OpCodes.Ldnull)
    | AI_dup                      -> ilG.Emit(OpCodes.Dup)
    | AI_pop                      -> ilG.Emit(OpCodes.Pop)
    | AI_ckfinite                 -> ilG.Emit(OpCodes.Ckfinite)
    | AI_nop                      -> ilG.Emit(OpCodes.Nop)
    | AI_ldc (DT_I4,NUM_I4 i32)   -> ilG.Emit(OpCodes.Ldc_I4,i32)
    | AI_ldc (DT_I8,NUM_I8 i64)   -> ilG.Emit(OpCodes.Ldc_I8,i64)
    | AI_ldc (DT_R4,NUM_R4 r32)   -> ilG.Emit(OpCodes.Ldc_R4,ieee32_to_float32 r32)
    | AI_ldc (DT_R8,NUM_R8 r64)   -> ilG.Emit(OpCodes.Ldc_R8,ieee64_to_float r64)
    | AI_ldc (_    ,_         )   -> failwith "emitInstrI_arith (AI_ldc (typ,const)) iltyped"


///Emit comparison instructions
let emitInstrCompare emEnv (ilG:ILGenerator) comp targ = 
    match comp with
    | BI_beq     -> ilG.Emit(OpCodes.Beq    ,envGetLabel emEnv targ)
    | BI_bge     -> ilG.Emit(OpCodes.Bge    ,envGetLabel emEnv targ)
    | BI_bge_un  -> ilG.Emit(OpCodes.Bge_Un ,envGetLabel emEnv targ)
    | BI_bgt     -> ilG.Emit(OpCodes.Bgt    ,envGetLabel emEnv targ)
    | BI_bgt_un  -> ilG.Emit(OpCodes.Bgt_Un ,envGetLabel emEnv targ)
    | BI_ble     -> ilG.Emit(OpCodes.Ble    ,envGetLabel emEnv targ)
    | BI_ble_un  -> ilG.Emit(OpCodes.Ble_Un ,envGetLabel emEnv targ)
    | BI_blt     -> ilG.Emit(OpCodes.Blt    ,envGetLabel emEnv targ)
    | BI_blt_un  -> ilG.Emit(OpCodes.Blt_Un ,envGetLabel emEnv targ)
    | BI_bne_un  -> ilG.Emit(OpCodes.Bne_Un ,envGetLabel emEnv targ)
    | BI_brfalse -> ilG.Emit(OpCodes.Brfalse,envGetLabel emEnv targ)
    | BI_brtrue  -> ilG.Emit(OpCodes.Brtrue ,envGetLabel emEnv targ)


/// Emit the volatile. prefix
let emitInstrVolatile (ilG:ILGenerator) = function
    | Volatile    -> ilG.Emit(OpCodes.Volatile)
    | Nonvolatile -> ()

/// Emit the align. prefix
let emitInstrAlign (ilG:ILGenerator) = function      
    | Aligned     -> ()
    | Unaligned_1 -> ilG.Emit(OpCodes.Unaligned,1L) // note: doc says use "long" overload!
    | Unaligned_2 -> ilG.Emit(OpCodes.Unaligned,2L)
    | Unaligned_4 -> ilG.Emit(OpCodes.Unaligned,3L)

/// Emit the tail. prefix if necessary
let emitInstrTail (ilG:ILGenerator) tail emitTheCall = 
    match tail with
    | Tailcall   -> ilG.Emit(OpCodes.Tailcall); emitTheCall(); ilG.Emit(OpCodes.Ret)
    | Normalcall -> emitTheCall()

let emitInstrNewobj emEnv (ilG:ILGenerator) mspec varargs =
    match varargs with
    | None         -> ilG.Emit(OpCodes.Newobj,convConstructorSpec emEnv mspec)
    | Some vartyps -> failwith "emit: pending new varargs" // XXX - gap

let emitInstrCall emEnv (ilG:ILGenerator) opCall tail (mspec:MethodSpec) varargs =
    if verbose then dprintf0 "emitInstrCall\n";
    emitInstrTail ilG tail (fun () ->
        if mspec.MethodRef.mrefName = ".ctor" || mspec.MethodRef.mrefName = ".cctor" then
            match varargs with
            | None         -> ilG.Emit     (opCall,convConstructorSpec emEnv mspec)
            | Some vartyps -> failwith "emitInstrCall: .ctor and varargs"
        else
            match varargs with
            | None         -> ilG.Emit     (opCall,convMethodSpec emEnv mspec)
            | Some vartyps -> ilG.EmitCall (opCall,convMethodSpec emEnv mspec,CompatArray.of_list (map (convType emEnv) vartyps))
    )


//----------------------------------------------------------------------------
// emitInstr cenv
//----------------------------------------------------------------------------

let rec emitInstr cenv (modB : ModuleBuilder) emEnv (ilG:ILGenerator) instr = 
    match instr with 
    | I_arith ainstr              -> emitInstrI_arith emEnv ilG ainstr
    | I_ldarg  u16                -> ilG.Emit(OpCodes.Ldarg ,UInt16.to_int16 u16)
    | I_ldarga u16                -> ilG.Emit(OpCodes.Ldarga,UInt16.to_int16 u16)
    | I_ldind (align,vol,dt)      -> emitInstrAlign ilG align;
                                     emitInstrVolatile ilG vol;
                                     (match dt with
                                      | DT_I   -> ilG.Emit(OpCodes.Ldind_I)
                                      | DT_I1  -> ilG.Emit(OpCodes.Ldind_I1)
                                      | DT_I2  -> ilG.Emit(OpCodes.Ldind_I2)
                                      | DT_I4  -> ilG.Emit(OpCodes.Ldind_I4)
                                      | DT_I8  -> ilG.Emit(OpCodes.Ldind_I8)
                                      | DT_R   -> failwith "emitInstr cenv: ldind R"
                                      | DT_R4  -> ilG.Emit(OpCodes.Ldind_R4)
                                      | DT_R8  -> ilG.Emit(OpCodes.Ldind_R8)
                                      | DT_U   -> failwith "emitInstr cenv: ldind U"
                                      | DT_U1  -> ilG.Emit(OpCodes.Ldind_U1)
                                      | DT_U2  -> ilG.Emit(OpCodes.Ldind_U2)
                                      | DT_U4  -> ilG.Emit(OpCodes.Ldind_U4)
                                      | DT_U8  -> failwith "emitInstr cenv: ldind U8"
                                      | DT_REF -> ilG.Emit(OpCodes.Ldind_Ref))
    | I_ldloc  u16                -> ilG.Emit(OpCodes.Ldloc ,UInt16.to_int16 u16)
    | I_ldloca u16                -> ilG.Emit(OpCodes.Ldloca,UInt16.to_int16 u16)
    | I_starg  u16                -> ilG.Emit(OpCodes.Starg ,UInt16.to_int16 u16)
    | I_stind (align,vol,dt)      -> emitInstrAlign ilG align;
                                     emitInstrVolatile ilG vol;
                                     (match dt with
                                      | DT_I   -> ilG.Emit(OpCodes.Stind_I)
                                      | DT_I1  -> ilG.Emit(OpCodes.Stind_I1)
                                      | DT_I2  -> ilG.Emit(OpCodes.Stind_I2)
                                      | DT_I4  -> ilG.Emit(OpCodes.Stind_I4)
                                      | DT_I8  -> ilG.Emit(OpCodes.Stind_I8)
                                      | DT_R   -> failwith "emitInstr cenv: stind R"
                                      | DT_R4  -> ilG.Emit(OpCodes.Stind_R4)
                                      | DT_R8  -> ilG.Emit(OpCodes.Stind_R8)
                                      | DT_U   -> ilG.Emit(OpCodes.Stind_I)    // NOTE: unsigned -> int conversion
                                      | DT_U1  -> ilG.Emit(OpCodes.Stind_I1)   // NOTE: follows code ilwrite.ml
                                      | DT_U2  -> ilG.Emit(OpCodes.Stind_I2)   // NOTE: is it ok?
                                      | DT_U4  -> ilG.Emit(OpCodes.Stind_I4)   // NOTE: it is generated by bytearray tests
                                      | DT_U8  -> ilG.Emit(OpCodes.Stind_I8)   // NOTE: unsigned -> int conversion
                                      | DT_REF -> ilG.Emit(OpCodes.Stind_Ref)) 
    | I_stloc  u16                -> ilG.Emit(OpCodes.Stloc,UInt16.to_int16 u16)
    | I_br  label                 -> ilG.Emit(OpCodes.Br,envGetLabel emEnv label)
    | I_jmp mspec                 -> let methInfo = convMethodSpec emEnv mspec
                                     ilG.Emit(OpCodes.Jmp,methInfo)
    | I_brcmp (comp,targ,fall)    -> emitInstrCompare emEnv ilG comp targ;
                                     ilG.Emit(OpCodes.Br,envGetLabel emEnv fall)  // XXX - very likely to be the next instruction...
    | I_switch (labels,next)      -> ilG.Emit(OpCodes.Switch,CompatArray.of_list (map (envGetLabel emEnv) labels));
                                     ilG.Emit(OpCodes.Br,envGetLabel emEnv next)  // XXX - very likely to be the next instruction...
    | I_ret                       -> ilG.Emit(OpCodes.Ret)
    | I_call           (tail,mspec,varargs)   -> emitInstrCall emEnv ilG OpCodes.Call     tail mspec varargs
    | I_callvirt       (tail,mspec,varargs)   -> emitInstrCall emEnv ilG OpCodes.Callvirt tail mspec varargs
#if CLI_AT_LEAST_2_0
    | I_callconstraint (tail,typ,mspec,varargs) -> ilG.Emit(OpCodes.Constrained,convType emEnv typ); 
                                                   emitInstrCall emEnv ilG OpCodes.Callvirt tail mspec varargs   
#endif
    | I_calli (tail,callsig,None)             -> emitInstrTail ilG tail (fun () ->
                                                   ilG.EmitCalli(OpCodes.Calli,
                                                                 convCallConv callsig.callsigCallconv,
                                                                 convType emEnv callsig.callsigReturn,
                                                                 CompatArray.of_list (map (convType emEnv) callsig.callsigArgs),
                                                                 CompatArray.of_list []))
    | I_calli (tail,callsig,Some vartyps)     -> emitInstrTail ilG tail (fun () ->
                                                   ilG.EmitCalli(OpCodes.Calli,
                                                                 convCallConv callsig.callsigCallconv,
                                                                 convType emEnv callsig.callsigReturn,
                                                                 CompatArray.of_list (map (convType emEnv) callsig.callsigArgs),
                                                                 CompatArray.of_list (map (convType emEnv) vartyps)))
    | I_ldftn mspec                           -> ilG.Emit(OpCodes.Ldftn,convMethodSpec emEnv mspec)
    | I_newobj (mspec,varargs)                -> emitInstrNewobj emEnv ilG mspec varargs
    | I_throw                        -> ilG.Emit(OpCodes.Throw)
    | I_endfinally                   -> ilG.Emit(OpCodes.Endfinally) (* capitalization! *)
    | I_endfilter                    -> () (* ilG.Emit(OpCodes.Endfilter) *)
    | I_leave label                  -> ilG.Emit(OpCodes.Leave,envGetLabel emEnv label)
    | I_ldsfld (vol,fspec)           ->                           emitInstrVolatile ilG vol; ilG.Emit(OpCodes.Ldsfld ,convFieldSpec emEnv fspec)
    | I_ldfld (align,vol,fspec)      -> emitInstrAlign ilG align; emitInstrVolatile ilG vol; ilG.Emit(OpCodes.Ldfld  ,convFieldSpec emEnv fspec)
    | I_ldsflda fspec                ->                                                      ilG.Emit(OpCodes.Ldsflda,convFieldSpec emEnv fspec)
    | I_ldflda fspec                 ->                                                      ilG.Emit(OpCodes.Ldflda ,convFieldSpec emEnv fspec)
    | I_stsfld (vol,fspec)           ->                           emitInstrVolatile ilG vol; ilG.Emit(OpCodes.Stsfld ,convFieldSpec emEnv fspec)
    | I_stfld (align,vol,fspec)      -> emitInstrAlign ilG align; emitInstrVolatile ilG vol; ilG.Emit(OpCodes.Stfld  ,convFieldSpec emEnv fspec)
    | I_ldstr     bytes              -> ilG.Emit(OpCodes.Ldstr    ,Bytes.unicode_bytes_as_string bytes)
    | I_isinst    typ                -> ilG.Emit(OpCodes.Isinst   ,convType emEnv typ)
    | I_castclass typ                -> ilG.Emit(OpCodes.Castclass,convType emEnv typ)
    | I_ldtoken (Token_type typ)     -> ilG.Emit(OpCodes.Ldtoken  ,convType emEnv typ)
    | I_ldtoken (Token_method mspec) -> ilG.Emit(OpCodes.Ldtoken  ,convMethodSpec emEnv mspec)
    | I_ldtoken (Token_field fspec)  -> ilG.Emit(OpCodes.Ldtoken  ,convFieldSpec  emEnv fspec)
    | I_ldvirtftn mspec              -> ilG.Emit(OpCodes.Ldvirtftn,convMethodSpec emEnv mspec)
    (* Value type instructions *)
    | I_cpobj     typ             -> ilG.Emit(OpCodes.Cpobj    ,convType emEnv typ)
    | I_initobj   typ             -> ilG.Emit(OpCodes.Initobj  ,convType emEnv typ)
    | I_ldobj (align,vol,typ)     -> emitInstrAlign ilG align; emitInstrVolatile ilG vol; ilG.Emit(OpCodes.Ldobj ,convType emEnv typ)
    | I_stobj (align,vol,typ)     -> emitInstrAlign ilG align; emitInstrVolatile ilG vol; ilG.Emit(OpCodes.Stobj ,convType emEnv typ)
    | I_box       typ             -> ilG.Emit(OpCodes.Box      ,convType emEnv typ)
    | I_unbox     typ             -> ilG.Emit(OpCodes.Unbox    ,convType emEnv typ)
#if CLI_AT_LEAST_2_0
    | I_unbox_any typ             -> ilG.Emit(OpCodes.Unbox_Any,convType emEnv typ)
#endif
    | I_sizeof    typ             -> ilG.Emit(OpCodes.Sizeof   ,convType emEnv typ)
    // Generalized array instructions. 
    // In AbsIL these instructions include 
    // both the single-dimensional variants (with array_shape == sdshape) 
    // and calls to the "special" multi-dimensional "methods" such as 
    //   newobj void string[,]::.ctor(int32, int32) 
    //   call string string[,]::Get(int32, int32) 
    //   call string& string[,]::Address(int32, int32) 
    //   call void string[,]::Set(int32, int32,string) 
    // The IL reader transforms calls of this form to the corresponding 
    // generalized instruction with the corresponding array_shape 
    // argument. This is done to simplify the IL and make it more uniform. 
    // The IL writer then reverses this when emitting the binary. 
    | I_ldelem dt                 -> (match dt with
                                      | DT_I   -> ilG.Emit(OpCodes.Ldelem_I)
                                      | DT_I1  -> ilG.Emit(OpCodes.Ldelem_I1)
                                      | DT_I2  -> ilG.Emit(OpCodes.Ldelem_I2)
                                      | DT_I4  -> ilG.Emit(OpCodes.Ldelem_I4)
                                      | DT_I8  -> ilG.Emit(OpCodes.Ldelem_I8)
                                      | DT_R   -> failwith "emitInstr cenv: ldelem R"
                                      | DT_R4  -> ilG.Emit(OpCodes.Ldelem_R4)
                                      | DT_R8  -> ilG.Emit(OpCodes.Ldelem_R8)
                                      | DT_U   -> failwith "emitInstr cenv: ldelem U"
                                      | DT_U1  -> ilG.Emit(OpCodes.Ldelem_U1)
                                      | DT_U2  -> ilG.Emit(OpCodes.Ldelem_U2)
                                      | DT_U4  -> ilG.Emit(OpCodes.Ldelem_U4)
                                      | DT_U8  -> failwith "emitInstr cenv: ldelem U8"
                                      | DT_REF -> ilG.Emit(OpCodes.Ldelem_Ref))
    | I_stelem dt                 -> (match dt with
                                      | DT_I   -> ilG.Emit(OpCodes.Stelem_I)
                                      | DT_I1  -> ilG.Emit(OpCodes.Stelem_I1)
                                      | DT_I2  -> ilG.Emit(OpCodes.Stelem_I2)
                                      | DT_I4  -> ilG.Emit(OpCodes.Stelem_I4)
                                      | DT_I8  -> ilG.Emit(OpCodes.Stelem_I8)
                                      | DT_R   -> failwith "emitInstr cenv: stelem R"
                                      | DT_R4  -> ilG.Emit(OpCodes.Stelem_R4)
                                      | DT_R8  -> ilG.Emit(OpCodes.Stelem_R8)
                                      | DT_U   -> failwith "emitInstr cenv: stelem U"
                                      | DT_U1  -> failwith "emitInstr cenv: stelem U1"
                                      | DT_U2  -> failwith "emitInstr cenv: stelem U2"
                                      | DT_U4  -> failwith "emitInstr cenv: stelem U4"
                                      | DT_U8  -> failwith "emitInstr cenv: stelem U8"
                                      | DT_REF -> ilG.Emit(OpCodes.Stelem_Ref))
    | I_ldelema (ro,shape,typ)     -> 
#if CLI_AT_LEAST_2_0
        if (ro = ReadonlyAddress) then ilG.Emit(OpCodes.Readonly);
#else
#endif
        if (shape = sdshape) 
        then ilG.Emit(OpCodes.Ldelema,convType emEnv typ)
        else 
            let aty = convType emEnv (Type_array(shape,typ)) 
            let ety = aty.GetElementType()
            let rty = makeByRefType ety 
            let meth = modB.GetArrayMethod(aty,"Address",System.Reflection.CallingConventions.HasThis,rty,CompatArray.create (rank_of_array_shape shape) (type int)) 
            ilG.Emit(OpCodes.Call,meth)
    | I_ldelem_any (shape,typ)     -> 
#if CLI_AT_LEAST_2_0
        if (shape = sdshape)      then ilG.Emit(OpCodes.Ldelem,convType emEnv typ)
        else 
#else
#endif
            let aty = convType emEnv (Type_array(shape,typ)) 
            let ety = aty.GetElementType()
            let meth = modB.GetArrayMethod(aty,"Get",System.Reflection.CallingConventions.HasThis,ety,CompatArray.create (rank_of_array_shape shape) (type int)) 
            ilG.Emit(OpCodes.Call,meth)
    | I_stelem_any (shape,typ)     -> 
#if CLI_AT_LEAST_2_0
        if (shape = sdshape)      then ilG.Emit(OpCodes.Stelem,convType emEnv typ)
        else 
#else
#endif
            let aty = convType emEnv (Type_array(shape,typ)) 
            let ety = aty.GetElementType()
            let meth = modB.GetArrayMethod(aty,"Set",System.Reflection.CallingConventions.HasThis,(null:Type),CompatArray.append (CompatArray.create (rank_of_array_shape shape) (type int)) (CompatArray.of_list [ ety ])) 
            ilG.Emit(OpCodes.Call,meth)
    | I_newarr (shape,typ)         -> 
        if (shape = sdshape)
        then ilG.Emit(OpCodes.Newarr,convType emEnv typ)
        else 
            let aty = convType emEnv (Type_array(shape,typ)) 
            let ety = aty.GetElementType()
            let meth = modB.GetArrayMethod(aty,".ctor",System.Reflection.CallingConventions.HasThis,(null:Type),CompatArray.create (rank_of_array_shape shape) (type int)) 
            ilG.Emit(OpCodes.Newobj,meth)
    | I_ldlen                      -> ilG.Emit(OpCodes.Ldlen)
    | I_mkrefany   typ             -> ilG.Emit(OpCodes.Mkrefany,convType emEnv typ)
    | I_refanytype                 -> ilG.Emit(OpCodes.Refanytype)
    | I_refanyval typ              -> ilG.Emit(OpCodes.Refanyval,convType emEnv typ)
    | I_rethrow                    -> ilG.Emit(OpCodes.Rethrow)
    | I_break                      -> ilG.Emit(OpCodes.Break)
    | I_seqpoint src               -> 
#if CLI_AT_LEAST_2_0
        if not (src.sourceDocument.sourceFile.EndsWith("stdin")) then
            let symDoc = modB.DefineDocument(src.sourceDocument.sourceFile, Guid.Empty, Guid.Empty, Guid.Empty)
            ilG.MarkSequencePoint(symDoc, src.sourceLine, src.sourceColumn, src.sourceEndLine, src.sourceEndColumn)
#else
        ()
#endif        
    | I_arglist                    -> ilG.Emit(OpCodes.Arglist)
    | I_localloc                   -> ilG.Emit(OpCodes.Localloc)
    | I_cpblk (align,vol)          -> emitInstrAlign ilG align;
                                      emitInstrVolatile ilG vol;
                                      ilG.Emit(OpCodes.Cpblk)
    | I_initblk (align,vol)        -> emitInstrAlign ilG align;
                                      emitInstrVolatile ilG vol;
                                      ilG.Emit(OpCodes.Initblk)
    | I_other e when is_ilx_ext_instr e -> 
          match (dest_ilx_ext_instr e) with 
          | EI_stelem_any_erasable (shape,ty) -> 
              emitInstr cenv modB emEnv ilG (I_stelem_any(shape,ty))
          | EI_ldelem_any_erasable (shape,ty) -> 
              emitInstr cenv modB emEnv ilG (I_ldelem_any(shape,ty))
          | EI_newarr_erasable (shape, ty) -> 
              emitInstr cenv modB emEnv ilG (I_newarr(shape,ty))
          | EI_ldlen_multi (n,m) -> 
              emitInstr cenv modB emEnv ilG (mk_ldc_i32 m);
              emitInstr cenv modB emEnv ilG (mk_normal_call(mk_nongeneric_mspec_in_typ(cenv.ilg.typ_Array, instance_callconv, "GetLength", [cenv.ilg.typ_int32], cenv.ilg.typ_int32)))
          | i -> Printf.failwithf "the ILX instruction %s cannot be emitted" (i.ToString())
     |  i -> Printf.failwithf "the IL instruction %s cannot be emitted" (i.ToString())

//----------------------------------------------------------------------------
// emitCode 
//----------------------------------------------------------------------------

let emitBasicBlock cenv  modB emEnv (ilG:ILGenerator) bblock =
    if verbose then dprintf0 "emitBasicBlock cenv\n";    
    emitLabelMark emEnv ilG bblock.bblockLabel;
    Array.iter (emitInstr cenv modB emEnv ilG) bblock.bblockInstrs;
    ()

let emitCode cenv modB emEnv (ilG:ILGenerator) code =
    if verbose then dprintf0 "emitCode cenv\n";  
    // pre define labels pending determining their actual marks
    let labels = labels_of_code code
    //List.iter (fun lab -> if verbose then dprintf1 "Label %s \n" lab) labels;
    let emEnv  = fold_left (defineLabel ilG) emEnv labels
    let rec emitter = function
        | BasicBlock bblock                  -> emitBasicBlock cenv modB emEnv ilG bblock
        | GroupBlock (localDebugInfos,codes) -> if verbose then dprintf0 "emitGroupBlock\n";
                                                List.iter emitter codes
        | RestrictBlock (labels,code)        -> if verbose then dprintf0 "emitRestrictBlock\n";
                                                emitter code (* restrictions ignorable: code_labels unique *)
        | TryBlock (code,seh)                -> 
            if verbose then dprintf0 "emitTryBlock: start\n";
            let endExBlockL = ilG.BeginExceptionBlock()
            emitter code;
            //ilG.MarkLabel endExBlockL;
            emitHandler seh;
            ilG.EndExceptionBlock();
            if verbose then dprintf0 "emitTryBlock: done\n"
    and emitHandler seh =
        if verbose then dprintf0 "emitHandler\n";    
        match seh with      
        | FaultBlock code         -> ilG.BeginFaultBlock();   emitter code
        | FinallyBlock code       -> ilG.BeginFinallyBlock(); emitter code
        | FilterCatchBlock fcodes -> 
            let emitFilter (filter,code) =
                match filter with
                | TypeFilter typ  -> 
                    ilG.BeginCatchBlock (convType emEnv typ); 
                    emitter code
                | CodeFilter test -> 
                    ilG.BeginExceptFilterBlock(); 
                    emitter test; 
                    ilG.BeginCatchBlock null; 
                    emitter code
            fcodes |> List.iter emitFilter 
    emitter code

//----------------------------------------------------------------------------
// emitILMethodBody 
//----------------------------------------------------------------------------

let emitLocal emEnv (ilG : ILGenerator) local =
#if CLI_AT_MOST_1_1
    ilG.DeclareLocal(convType emEnv local.localType)
#else
    ilG.DeclareLocal(convType emEnv local.localType,local.localPinned)
#endif

let emitILMethodBody cenv modB emEnv (ilG:ILGenerator) ilmbody =
    // XXX - REVIEW:
    //      ilNoInlining: bool;
    //      ilSource: source option }
    if verbose then dprintf0 "emitILMethodBody cenv: start\n";  
    // emit locals and record emEnv
    let localBs = map (emitLocal emEnv ilG) ilmbody.ilLocals
    let emEnv = envSetLocals emEnv (Array.of_list localBs)
    emitCode cenv modB emEnv ilG ilmbody.ilCode;
    if verbose then dprintf0 "emitILMethodBody cenv: end\n"

//----------------------------------------------------------------------------
// emitMethodBody 
//----------------------------------------------------------------------------

let emitMethodBody cenv modB emEnv ilG name mbody =
    match dest_mbody mbody with
    | MethodBody_il ilmbody       -> emitILMethodBody cenv modB emEnv (ilG()) ilmbody
    | MethodBody_pinvoke  pinvoke -> () (* Printf.printf "EMIT: pinvoke method %s\n" name *) (* XXX - check *)
    | MethodBody_abstract         -> () (* Printf.printf "EMIT: abstract method %s\n" name *) (* XXX - check *)
    | MethodBody_native           -> failwith "emitMethodBody cenv: native"               (* XXX - gap *)

//----------------------------------------------------------------------------
// emitParameter
//----------------------------------------------------------------------------

let emitParameter emEnv (defineParameter : int * ParameterAttributes * string -> ParameterBuilder) i param =
    //  -paramType: typ;
    //  -paramDefault: field_init option;  
    //  -paramMarshal: native_type option; (* Marshalling map for parameters. COM Interop only. *)
    if verbose then dprintf1 "emitParameter %s\n" (match param.paramName with Some n -> n | None -> "anon");
    let attrs = flagsL [flagsIf param.paramIn       ParameterAttributes.In;
                        flagsIf param.paramOut      ParameterAttributes.Out;
                        flagsIf param.paramOptional ParameterAttributes.Optional]
    let name = match param.paramName with
               | Some name -> name
               | None      -> "X"^string_of_int(i+1)
   
    let parB = defineParameter(i,attrs,name)
    emitCustomAttrs emEnv (fun (x,y) -> parB.SetCustomAttribute(x,y)) param.paramCustomAttrs

//----------------------------------------------------------------------------
// convMethodAttributes
//----------------------------------------------------------------------------

let convMethodAttributes mdef =    
    if verbose then dprintf1 "- convMethodAttributes %s\n" mdef.mdName;
    let attrKind = 
        match mdef.mdKind with 
        | MethodKind_static        -> MethodAttributes.Static
        | MethodKind_cctor         -> flagsL[MethodAttributes.Static] // XXX - check
        | MethodKind_ctor          -> flagsL[]                        // XXX - check
        | MethodKind_nonvirtual    -> flagsL[]
        | MethodKind_virtual vinfo -> flagsL[MethodAttributes.Virtual;
                                             flagsIfL vinfo.virtNewslot   [MethodAttributes.NewSlot];
                                             flagsIfL vinfo.virtFinal     [MethodAttributes.Final];
#if CLI_AT_LEAST_2_0
                                             flagsIfL vinfo.virtStrict    [MethodAttributes.CheckAccessOnOverride];
#endif
                                             flagsIfL vinfo.virtAbstract  [MethodAttributes.Abstract]
                                             (*flagsIfL vinfo.virtOverrides [MethodAttributes.Virtual]*) ]
   
    let attrAccess = 
        match mdef.mdAccess with
        | MemAccess_assembly -> MethodAttributes.Assembly
        | MemAccess_compilercontrolled -> failwith "Method access compiler controled."
        | MemAccess_famandassem        -> MethodAttributes.FamANDAssem
        | MemAccess_famorassem         -> MethodAttributes.FamORAssem
        | MemAccess_family             -> MethodAttributes.Family
        | MemAccess_private            -> MethodAttributes.Private
        | MemAccess_public             -> MethodAttributes.Public
   
    let attrOthers = flagsL [flagsIf mdef.mdHasSecurity MethodAttributes.HasSecurity;
                             flagsIf mdef.mdSpecialName MethodAttributes.SpecialName;
                             flagsIf mdef.mdHideBySig   MethodAttributes.HideBySig;  
                             flagsIf mdef.mdReqSecObj   MethodAttributes.RequireSecObject]
   
    let attrs = flagsL [attrKind;attrAccess;attrOthers]
    attrs

let convMethodImplFlags mdef =    
    if verbose then dprintf1 "- convMethodImplFlags %s\n" mdef.mdName;
    flagsL 
      (List.concat
         [ [(match  mdef.mdCodeKind with 
             | MethodCodeKind_native -> MethodImplAttributes.Native
             | MethodCodeKind_runtime -> MethodImplAttributes.Runtime
             | MethodCodeKind_il  -> MethodImplAttributes.IL)];
           (if mdef.mdInternalCall then [MethodImplAttributes.InternalCall] else []); 
           [(if mdef.mdManaged then MethodImplAttributes.Managed else MethodImplAttributes.Unmanaged)];
           (if mdef.mdForwardRef then [MethodImplAttributes.ForwardRef] else []);
           (if mdef.mdPreserveSig then [MethodImplAttributes.PreserveSig] else []);
           (if mdef.mdSynchronized then [MethodImplAttributes.Synchronized] else []);
  (*       (if mdef.mdMustRun then [MethodImplAttributes.MustRun] else []);*)
           (if (match dest_mbody mdef.mdBody with MethodBody_il b -> b.ilNoInlining | _ -> false) then [MethodImplAttributes.NoInlining] else [])])

//----------------------------------------------------------------------------
// buildMethodPass2
//----------------------------------------------------------------------------
  
let rec buildMethodPass2 tref (typB:TypeBuilder) emEnv (mdef : IL.method_def) =
   // remaining REVIEW:
   // mdCodeKind: method_code_kind;   
   // mdInternalCall: bool;
   // mdManaged: bool;
   // mdForwardRef: bool;
   // mdSecurityDecls: security_decls;
   // mdUnmanagedExport: bool; (* -- The method is exported to unmanaged code using COM interop. *)
   // mdSynchronized: bool;
   // mdPreserveSig: bool;
   // mdMustRun: bool; (* Whidbey feature: SafeHandle finalizer must be run *)
   // mdExport: (i32 * string option) option; 
   // mdVtableEntry: (i32 * i32) option;
    if verbose then dprintf1 "buildMethodPass2 %s\n" mdef.mdName;
    let attrs = convMethodAttributes mdef
    let implflags = convMethodImplFlags mdef
    let cconv = convCallConv mdef.mdCallconv
    let mref = mk_mref_to_mdef (tref,mdef)
    let emEnv = if mdef.mdEntrypoint then envAddEntryPt emEnv (typB,mdef.mdName) else emEnv
    match dest_mbody mdef.mdBody with
    | MethodBody_pinvoke  p -> 
        let argtys = CompatArray.of_list (map (typ_of_param >> convType emEnv) mdef.mdParams) 
        let rty = convType emEnv (typ_of_return mdef.mdReturn) 
        
        let pcc = 
            match p.pinvokeCallconv with 
            | PInvokeCallConvCdecl -> CallingConvention.Cdecl
            | PInvokeCallConvStdcall -> CallingConvention.StdCall
            | PInvokeCallConvThiscall -> CallingConvention.ThisCall
            | PInvokeCallConvFastcall -> CallingConvention.FastCall
            | PInvokeCallConvNone 
            | PInvokeCallConvWinapi -> CallingConvention.Winapi 
        
        let pcs = 
            match p.pinvokeEncoding with 
            | PInvokeEncodingNone -> CharSet.None
            | PInvokeEncodingAnsi -> CharSet.Ansi
            | PInvokeEncodingUnicode -> CharSet.Unicode
            | PInvokeEncodingAuto -> CharSet.Auto 
      
(* p.pinvokeThrowOnUnmappableChar *)
(* p.pinvokeBestFit *)
(* p.pinvokeNoMangle *)

        let methB = typB.DefinePInvokeMethod(mdef.mdName, 
                                             p.pinvokeWhere.modulRefName, 
                                             p.pinvokeName, 
                                             attrs, 
                                             cconv, 
                                             rty, 
#if CLI_AT_LEAST_2_0
                                             null, null, 
#endif
                                             argtys, 
#if CLI_AT_LEAST_2_0
                                             null, null, 
#endif
                                             pcc, 
                                             pcs) 
        methB.SetImplementationFlags(implflags);
        envBindMethodRef emEnv mref methB

    | _ -> 
      match mdef.mdName with
      | ".cctor" 
      | ".ctor" ->
          let consB = typB.DefineConstructor(attrs,
                                             cconv,
                                             CompatArray.of_list (map (typ_of_param >> convType emEnv) mdef.mdParams))
          consB.SetImplementationFlags(implflags);
          envBindConsRef emEnv mref consB
      | name    ->
          // if generics the return/argument types may involve the generic parameters
          let methB = typB.DefineMethod(mdef.mdName,
                                        attrs,
                                        cconv
#if CLI_AT_LEAST_2_0
#else
                                        ,convType emEnv (typ_of_return mdef.mdReturn)
                                        ,CompatArray.of_list (map (typ_of_param >> convType emEnv) mdef.mdParams)
#endif
                                        ) 
        
          // Method generic type parameters         
#if CLI_AT_LEAST_2_0
          buildGenParamsPass1 emEnv (fun x -> methB.DefineGenericParameters(x)) mdef.GenericParams;
#endif
          let genArgs = getGenericArgumentsOfMethod methB 
#if CLI_AT_LEAST_2_0
          begin 
              let emEnv = envPushTyvars emEnv (Array.append (getGenericArgumentsOfType typB) genArgs)
              buildGenParamsPass1b emEnv genArgs mdef.GenericParams;
              // set parameter and return types (may depend on generic args)
              methB.SetParameters(CompatArray.of_list (map (typ_of_param >> convType emEnv) mdef.mdParams));
              methB.SetReturnType(convType emEnv (typ_of_return mdef.mdReturn));
          end;
#endif
          methB.SetImplementationFlags(implflags);
          envBindMethodRef emEnv mref methB


//----------------------------------------------------------------------------
// buildMethodPass3 cenv
//----------------------------------------------------------------------------
    
let rec buildMethodPass3 cenv tref modB (typB:TypeBuilder) emEnv (mdef : IL.method_def) =
    if verbose then dprintf1 "buildMethodPass3 cenv %s\n" mdef.mdName;
    let mref  = mk_mref_to_mdef (tref,mdef)
    match dest_mbody mdef.mdBody with
    | MethodBody_pinvoke  p -> ()
    | _ -> 
         match mdef.mdName with
         | ".cctor" | ".ctor" ->
              let consB = envGetConsB emEnv mref
              // Constructors can not have generic parameters
              assert (mdef.GenericParams=[]);
              // Value parameters       
              let defineParameter (i,attr,name) = consB.DefineParameter(i+1,attr,name)
              mdef.mdParams |> List.iteri (emitParameter emEnv defineParameter);
              // Body
              emitMethodBody cenv modB emEnv (fun () -> consB.GetILGenerator()) mdef.mdName mdef.mdBody;
              emitCustomAttrs emEnv (fun (x,y) -> consB.SetCustomAttribute(x,y)) mdef.mdCustomAttrs;
              ()
         | name ->
              let methB = envGetMethB emEnv mref
              let emEnv = envPushTyvars emEnv (Array.append
                                                 (getGenericArgumentsOfType typB)
                                                 (getGenericArgumentsOfMethod methB))

              begin match (dest_custom_attrs mdef.mdReturn.returnCustomAttrs) with
              | [] -> ()
              | _ ->
                  let retB = methB.DefineParameter(0,System.Reflection.ParameterAttributes.Retval,null) 
                  emitCustomAttrs emEnv (fun (x,y) -> retB.SetCustomAttribute(x,y)) mdef.mdReturn.returnCustomAttrs
              end;

              // Value parameters
              let defineParameter (i,attr,name) = methB.DefineParameter(i+1,attr,name) 
              mdef.mdParams |> List.iteri (emitParameter emEnv defineParameter);
              // Body
              emitMethodBody cenv modB emEnv (fun () -> methB.GetILGenerator()) mdef.mdName mdef.mdBody;
              let emEnv = envPopTyvars emEnv // case fold later...
              emitCustomAttrs emEnv (fun (x,y) -> methB.SetCustomAttribute(x,y)) mdef.mdCustomAttrs;
              ()
      
//----------------------------------------------------------------------------
// buildFieldPass2
//----------------------------------------------------------------------------
  
let buildFieldPass2 tref (typB:TypeBuilder) emEnv (fdef : IL.field_def) =
    (*{ -fdData:  bytes option;
       -fdOffset:  i32 option; (* The explicit offset bytes when explicit layout is used. *)
       -fdMarshal: native_type option;  *)
    if verbose then dprintf1 "buildFieldPass2 %s\n" fdef.fdName;
    let attrsAccess = match fdef.fdAccess with
                      | MemAccess_assembly           -> FieldAttributes.Assembly
                      | MemAccess_compilercontrolled -> failwith "Field access compiler controled."
                      | MemAccess_famandassem        -> FieldAttributes.FamANDAssem
                      | MemAccess_famorassem         -> FieldAttributes.FamORAssem
                      | MemAccess_family             -> FieldAttributes.Family
                      | MemAccess_private            -> FieldAttributes.Private
                      | MemAccess_public             -> FieldAttributes.Public
    let attrsOther = flagsL [flagsIf fdef.fdStatic        FieldAttributes.Static;
                             flagsIf fdef.fdSpecialName   FieldAttributes.SpecialName;
                             flagsIf fdef.fdLiteral       FieldAttributes.Literal;
                             flagsIf fdef.fdInitOnly      FieldAttributes.InitOnly;
                             flagsIf fdef.fdNotSerialized FieldAttributes.NotSerialized]
    let attrs = flagsL [attrsAccess;attrsOther]
    let fieldT = convType emEnv fdef.fdType
    let fieldB = 
        match fdef.fdData with 
        | Some d -> typB.DefineInitializedData(fdef.fdName, d, attrs)
        | None -> typB.DefineField(fdef.fdName,
                                   fieldT,
                                   attrs)
     
    // set default value
    Option.iter (fun initial -> fieldB.SetConstant(convFieldInit initial)) fdef.fdInit;
    // assert unsupported:
    assert (fdef.fdOffset=None);
    assert (fdef.fdMarshal=None);
    // custom attributes: done on pass 3 as they may reference attribute constructors generated on
    // pass 2.
    let fref = mk_fref_in_tref (tref,fdef.fdName,fdef.fdType)    
    envBindFieldRef emEnv fref fieldB

let buildFieldPass3 tref (typB:TypeBuilder) emEnv (fdef : IL.field_def) =
    if verbose then dprintf1 "buildFieldPass3 %s\n" fdef.fdName;
    let fref = mk_fref_in_tref (tref,fdef.fdName,fdef.fdType)    
    let fieldB = envGetFieldB emEnv fref
    emitCustomAttrs emEnv (fun (x,y) -> fieldB.SetCustomAttribute(x,y)) fdef.fdCustomAttrs

//----------------------------------------------------------------------------
// buildPropertyPass2,3
//----------------------------------------------------------------------------
  
let buildPropertyPass2 tref (typB:TypeBuilder) emEnv (prop : IL.property_def) =
    (*{ -propCallconv: hasthis;
        -propArgs: typ list; } *)
    if verbose then dprintf1 "buildPropertyPass2 %s\n" prop.propName;
    let attrs = flagsL [flagsIf prop.propRTSpecialName PropertyAttributes.RTSpecialName;
                flagsIf prop.propSpecialName   PropertyAttributes.SpecialName]
    let propB = typB.DefineProperty(prop.propName,
                                    attrs,
                                    convType emEnv prop.propType,
                                    CompatArray.of_list (map (convType emEnv) prop.propArgs)) // XXX - wrong, if these are tyvars, likely... XXX
   
    // install get/set methods
    let installOp descr setOp opMRef =
        if verbose then dprintf1 "buildPropertyPass2: installing %s" descr;      
        setOp(envGetMethB emEnv opMRef)
   
    prop.propSet |> Option.iter (installOp "set" (fun methB -> propB.SetSetMethod(methB)));
    prop.propGet |> Option.iter (installOp "get" (fun methB -> propB.SetGetMethod(methB)));
    // set default value
    prop.propInit |> Option.iter (fun initial -> propB.SetConstant(convFieldInit initial));
    // custom attributes
    let pref = mk_pref (tref,prop.propName)    
    envBindPropRef emEnv pref propB
    // XXX - propArgs, propCallconv ???

let buildPropertyPass3 tref (typB:TypeBuilder) emEnv (prop : IL.property_def) = 
  let pref = mk_pref (tref,prop.propName)    
  let propB = envGetPropB emEnv pref
  emitCustomAttrs emEnv (fun (x,y) -> propB.SetCustomAttribute(x,y)) prop.propCustomAttrs

//----------------------------------------------------------------------------
// buildMethodImplsPass3
//----------------------------------------------------------------------------
  
let buildMethodImplsPass3 tref (typB:TypeBuilder) emEnv (mimpl : IL.method_impl) =
    if verbose then dprintf1 "buildMethodImplsPass3 %s\n" mimpl.mimplOverrideBy.MethodRef.mrefName;    
    let bodyMethInfo = convMethodRef emEnv (typB :> Type) mimpl.mimplOverrideBy.MethodRef // doc: must be MethodBuilder
    let OverridesSpec (mref,dtyp) = mimpl.mimplOverrides
    let declMethTI = convType emEnv dtyp 
    let declMethInfo = convMethodRef  emEnv declMethTI mref
    typB.DefineMethodOverride(bodyMethInfo,declMethInfo);
    emEnv

//----------------------------------------------------------------------------
// typeAttributesOf*
//----------------------------------------------------------------------------

let typeAttrbutesOfTypeDefKind x = 
    match x with 
    // required for a TypeBuilder
    | TypeDef_class           -> TypeAttributes.Class
    | TypeDef_valuetype       -> TypeAttributes.Class
    | TypeDef_interface       -> TypeAttributes.Interface
    | TypeDef_enum            -> TypeAttributes.Class
    | TypeDef_delegate        -> TypeAttributes.Class
    | TypeDef_other xtdk      -> failwith "typeAttributes of other external"

let typeAttrbutesOfTypeAccess x =
    match x with 
    | TypeAccess_public       -> TypeAttributes.Public
    | TypeAccess_private      -> TypeAttributes.NotPublic
    | TypeAccess_nested macc  -> match macc with
                                 | MemAccess_assembly           -> TypeAttributes.NestedAssembly
                                 | MemAccess_compilercontrolled -> failwith "Nested compiler controled."
                                 | MemAccess_famandassem        -> TypeAttributes.NestedFamANDAssem
                                 | MemAccess_famorassem         -> TypeAttributes.NestedFamORAssem
                                 | MemAccess_family             -> TypeAttributes.NestedFamily
                                 | MemAccess_private            -> TypeAttributes.NestedPrivate
                                 | MemAccess_public             -> TypeAttributes.NestedPublic
                        
let typeAttributesOfTypeEncoding x = 
    match x with 
    | TypeEncoding_ansi     -> TypeAttributes.AnsiClass    
    | TypeEncoding_autochar -> TypeAttributes.AutoClass
    | TypeEncoding_unicode  -> TypeAttributes.UnicodeClass


let typeAttributesOfTypeLayout cenv emEnv x = 
    let attr p = 
      if p.typeSize =None && p.typePack = None then None
      else 
        Some(convCustomAttr emEnv 
               (IL.mk_custom_attribute cenv.ilg
                  (mk_tref (cenv.ilg.mscorlib_scoref,"System.Runtime.InteropServices.StructLayoutAttribute"), 
                   [mk_nongeneric_value_typ (mk_tref (cenv.ilg.mscorlib_scoref,"System.Runtime.InteropServices.LayoutKind")) ],
                   [ CustomElem_int32 0x02 ],
                   (p.typePack |> Option.to_list |> List.map (fun x -> ("Pack", cenv.ilg.typ_int32, false, CustomElem_int32 (u16_to_int x))))  @
                   (p.typeSize |> Option.to_list |> List.map (fun x -> ("Size", cenv.ilg.typ_int32, false, CustomElem_int32 x)))))) in
    match x with 
    | TypeLayout_auto         -> TypeAttributes.AutoLayout,None
    | TypeLayout_explicit p   -> TypeAttributes.ExplicitLayout,(attr p)
    | TypeLayout_sequential p -> TypeAttributes.SequentialLayout, (attr p)


//----------------------------------------------------------------------------
// buildTypeDefPass1 cenv
//----------------------------------------------------------------------------
    
let rec buildTypeDefPass1 cenv emEnv (modB:ModuleBuilder) rootTypeBuilder nesting (tdef : IL.type_def) =
    // -tdComInterop: bool; (* Class or interface generated for COM interop *) 
    // -tdSecurityDecls: security_decls;
    // -tdInitSemantics: type_init;
    // -tdEvents: events;
    if verbose then dprintf1 "buildTypeDefPass1 cenv %s\n" tdef.Name;
    // TypeAttributes
    let attrsKind   = typeAttrbutesOfTypeDefKind tdef.tdKind 
    let attrsAccess = typeAttrbutesOfTypeAccess  tdef.tdAccess
    let attrsLayout,cattrsLayout = typeAttributesOfTypeLayout cenv emEnv tdef.tdLayout
    let attrsEnc    = typeAttributesOfTypeEncoding tdef.tdEncoding
    let attrsOther  = flagsL [flagsIf tdef.tdAbstract     TypeAttributes.Abstract;
                              flagsIf tdef.tdSealed       TypeAttributes.Sealed;
                              flagsIf tdef.tdSerializable TypeAttributes.Serializable;
                              flagsIf tdef.tdSpecialName  TypeAttributes.SpecialName;
                              flagsIf tdef.tdHasSecurity  TypeAttributes.HasSecurity; ]
     
    let attrsType = flagsL [attrsKind;attrsAccess;attrsLayout;attrsEnc;attrsOther]

    // TypeBuilder from TypeAttributes.
    if verbose then dprintf1 "buildTypeDefPass1 cenv: build name = %s\n" tdef.Name;
    let typB : TypeBuilder = rootTypeBuilder  (tdef.Name,attrsType)
    let typB = typB |> nonNull "buildTypeDefPass1 cenv: typB is null!"
    cattrsLayout |> Option.iter (fun (x,y) -> typB.SetCustomAttribute(x,y));

#if CLI_AT_LEAST_2_0
    buildGenParamsPass1 emEnv (fun x -> typB.DefineGenericParameters(x)) tdef.GenericParams; 
#endif
    // bind tref -> (typT,typB)
    let tref = tref_for_nested_tdef ScopeRef_local (nesting,tdef)    
    let typT =
        // Q: would it be ok to use typB :> Type ? 
        let nameInModule = qualified_name_of_tref tref
        if verbose then dprintf1 "buildTypeDefPass1 cenv: nameInModule= %s\n" nameInModule;
        modB.GetType(nameInModule,false,false)
   
    if verbose then dprintf1 "buildTypeDefPass1 cenv: null? %d\n" (if typT=null then 0 else 1);
    let emEnv = envBindTypeRef emEnv tref (typT,typB,tdef)
    // recurse on nested types
    let nesting = nesting @ [tdef]     
    let buildNestedType emEnv tdef = buildTypeTypeDef cenv emEnv modB typB nesting tdef
    let emEnv = fold_left buildNestedType emEnv (dest_tdefs  tdef.NestedTypes)
    emEnv

and buildTypeTypeDef cenv emEnv modB (typB : TypeBuilder) nesting tdef =
    if verbose then dprintf0 "buildTypeTypeDef cenv\n";
    let rootTypB  (name,attrs)        = typB.DefineNestedType(name,attrs)       
    buildTypeDefPass1 cenv emEnv modB rootTypB nesting tdef

//----------------------------------------------------------------------------
// buildTypeDefPass1b
//----------------------------------------------------------------------------
    
let rec buildTypeDefPass1b nesting emEnv (tdef : IL.type_def) = 
    if verbose then dprintf1 "buildTypeDefPass1b %s\n" tdef.Name; 
    let tref = tref_for_nested_tdef ScopeRef_local (nesting,tdef)
    let typB  = envGetTypB emEnv tref
    let genArgs = (getGenericArgumentsOfType typB) 
    begin 
        let emEnv = envPushTyvars emEnv genArgs
        // Parent may reference types being defined, so has to come after it's Pass1 creation 
        tdef.Extends |> Option.iter (fun typ -> typB.SetParent(convType emEnv typ));
        // build constraints on genparams.  Constraints may reference types being defined, 
        // so have to come after all types are created
#if CLI_AT_LEAST_2_0
        buildGenParamsPass1b emEnv genArgs tdef.GenericParams; 
#endif
    end;
    let nesting = nesting @ [tdef]     
    iter (buildTypeDefPass1b nesting emEnv) (dest_tdefs tdef.NestedTypes)

//----------------------------------------------------------------------------
// buildTypeDefPass2
//----------------------------------------------------------------------------

let rec buildTypeDefPass2 nesting emEnv (tdef : IL.type_def) = 
    if verbose then dprintf1 "buildTypeDefPass2 %s\n" tdef.Name; 
    let tref = tref_for_nested_tdef ScopeRef_local (nesting,tdef)
    let typB  = envGetTypB emEnv tref
    let emEnv = envPushTyvars emEnv (getGenericArgumentsOfType typB)
    // add interface impls
    tdef.Implements |> map (convType emEnv) |> iter (fun implT -> typB.AddInterfaceImplementation(implT));
    // add methods, properties
    let emEnv = fold_left (buildMethodPass2      tref typB) emEnv (dest_mdefs tdef.Methods) 
    let emEnv = fold_left (buildFieldPass2       tref typB) emEnv (dest_fdefs tdef.Fields)  
    let emEnv = fold_left (buildPropertyPass2    tref typB) emEnv (dest_pdefs tdef.Properties) 
    let emEnv = envPopTyvars emEnv
    // nested types
    let nesting = nesting @ [tdef]
    let emEnv = fold_left (buildTypeDefPass2 nesting) emEnv (dest_tdefs tdef.NestedTypes)
    emEnv

//----------------------------------------------------------------------------
// buildTypeDefPass3 cenv
//----------------------------------------------------------------------------
    
let rec buildTypeDefPass3 cenv nesting modB emEnv (tdef : IL.type_def) =
    if verbose then dprintf1 "buildTypeDefPass3 cenv %s\n" tdef.Name; 
    let tref = tref_for_nested_tdef ScopeRef_local (nesting,tdef)
    let typB = envGetTypB emEnv tref
    let emEnv = envPushTyvars emEnv (getGenericArgumentsOfType typB)
    // add method bodies, properties
    tdef.Methods |> dest_mdefs |> List.iter (buildMethodPass3 cenv tref modB typB emEnv);
    tdef.Properties |> dest_pdefs |> List.iter (buildPropertyPass3 tref typB emEnv);
    tdef.Fields  |> dest_fdefs |> List.iter (buildFieldPass3 tref typB emEnv);
    let emEnv = fold_left (buildMethodImplsPass3 tref typB) emEnv (dest_mimpls     tdef.tdMethodImpls)
    tdef.tdCustomAttrs |> emitCustomAttrs emEnv (fun (x,y) -> typB.SetCustomAttribute(x,y)) ;
    // custom attributes
    let emEnv = envPopTyvars emEnv
    // nested types
    let nesting = nesting @ [tdef]
    let emEnv = fold_left (buildTypeDefPass3 cenv nesting modB) emEnv (dest_tdefs tdef.NestedTypes)
    emEnv

//----------------------------------------------------------------------------
// buildTypeDefPass4 - Create the Types
// MSDN says: If this type is a nested type, the CreateType method must 
// be called on the enclosing type before it is called on the nested type.
// If the current type derives from an incomplete type or implements 
// incomplete interfaces, call the CreateType method on the parent 
// type and the interface types before calling it on the current type.
// If the enclosing type contains a field that is a value type 
// defined as a nested type (for example, a field that is an 
// enumeration defined as a nested type), calling the CreateType method 
// on the enclosing type will generate a AppDomain.TypeResolve event. 
// This is because the loader cannot determine the size of the enclosing 
// type until the nested type has been completed. The caller should define 
// a handler for the TypeResolve event to complete the definition of the 
// nested type by calling CreateType on the TypeBuilder object that represents 
// the nested type. The code example for this topic shows how to define such 
// an event handler.
//----------------------------------------------------------------------------

let enclosing_trefs_of_tref tref = 
   match enclosing_tnames_of_tref tref with 
   | [] -> []
   | h :: t -> List.scan_left (fun tr nm -> mk_tref_in_tref (tr,nm)) (mk_tref(scoref_of_tref tref, h)) t

let rec trefs_of_typ valueTypesOnly typ acc = 
    match typ with
    | Type_void | Type_tyvar _                              -> acc
    | Type_ptr base | Type_byref base -> acc
    | Type_array (_,base) -> if valueTypesOnly then acc else trefs_of_typ valueTypesOnly base acc
    | Type_value tspec -> tref_of_tspec tspec :: List.fold_right (trefs_of_typ valueTypesOnly) (inst_of_tspec tspec) acc
    | Type_boxed tspec -> if valueTypesOnly then acc else tref_of_tspec tspec :: List.fold_right (trefs_of_typ valueTypesOnly) (inst_of_tspec tspec) acc
    | Type_fptr callsig -> failwith "trefs_of_typ: fptr"
    | Type_modified _   -> failwith "trefs_of_typ: modified"
    | Type_other extty  -> failwith "trefs_of_typ: other"

let verbose2 = false
    
let createTypeRef (visited : Collections.HashMultiMap<_,_>, created : Collections.HashMultiMap<_,_>) emEnv tref = 
    let rec traverseTypeDef priority tref (tdef:TypeDef) =
        // WORKAROUND (ProductStudio FSharp 1.0 bug 615): the constraints on generic method parameters 
        // are resolved overly eagerly by reflection emit's CreateType. The priority drops down to 1 here
        // because we absolutely have to create these types before attempting to create the enclosing type.
        if priority >= 1 then 
            if verbose2 then dprintf1 "buildTypeDefPass4: Doing method constraints of %s\n" tdef.Name; 
            tdef.Methods |> dest_mdefs |> List.iter   (fun md -> md.GenericParams |> List.iter (fun gp -> gp.Constraints |> List.iter (traverseType false 1)));
        // We have to define all struct types in all methods before a class is defined. This only has any effect when there is a struct type
        // being defined simultaneously with this type.
        if priority >= 1 then 
            if verbose2 then dprintf2 "buildTypeDefPass4: Doing value types in method signautres of %s, #mdefs = %d\n" tdef.Name (List.length (tdef.Methods |> dest_mdefs)); 
            tdef.Methods |> dest_mdefs |> List.iter   (fun md -> md.Parameters |> List.iter (fun p -> p.Type |> (traverseType true 1))
                                                                 md.Return.Type |> traverseType true 1);
        // We absolutely need the parent type...
        if priority >= 1 then 
            if verbose2 then dprintf1 "buildTypeDefPass4: Creating Super Class Chain of %s\n" tdef.Name; 
            tdef.Extends    |> Option.iter (traverseType false priority);
        
        // We absolutely need the interface types...
        if priority >= 1 then 
            if verbose2 then dprintf1 "buildTypeDefPass4: Creating Interface Chain of %s\n" tdef.Name; 
            tdef.Implements |> List.iter (traverseType false priority);
        
        if priority >= 1 then 
            if verbose2 then dprintf1 "buildTypeDefPass4: Do value types in fields of %s\n" tdef.Name; 
            tdef.Fields.Details |> List.iter (fun fd -> traverseType true 1 fd.Type);
        
        (* There seem to be some types we can create without creating the enclosing types *)
        (* Hence only attempt to create the enclosing types when the priority is >= 2 *)
        if priority >= 2 then 
            if verbose2 then dprintf1 "buildTypeDefPass4: Creating Enclosing Types of %s\n" tdef.Name; 
            tref |> enclosing_trefs_of_tref |> List.iter (traverseTypeRef priority);

        if verbose2 then dprintf1 "buildTypeDefPass4: Done with dependencies of %s\n" tdef.Name
    and traverseType valueTypesOnly priority typ = 
        if verbose2 then dprintf1 "- traverseType %A\n" typ;
        trefs_of_typ valueTypesOnly typ []
        |> List.filter (isEmittedTypeRef emEnv)
        |> List.iter (traverseTypeRef priority)

    and traverseTypeRef priority  tref = 
        let typB = envGetTypB emEnv tref
        if verbose2 then dprintf1 "- considering reference to type %s\n" typB.FullName;
        if not (visited.Contains(tref)) or visited.[tref] > priority then 
            visited.[tref] <- priority;
            let tdef = envGetTypeDef emEnv tref
            if verbose2 then dprintf1 "- traversing type %s\n" typB.FullName;
            traverseTypeDef priority tref tdef;
            if not (created.Contains(tref)) then 
                created.[tref] <- true;
                if verbose2 then dprintf1 "- creating type %s\n" typB.FullName;
                typB.CreateType()  |> ignore

    traverseTypeRef 2 tref 

let rec buildTypeDefPass4 visited nesting emEnv (tdef : IL.type_def) =
    if verbose2 then dprintf1 "buildTypeDefPass4 %s\n" tdef.Name; 
    let tref = tref_for_nested_tdef ScopeRef_local (nesting,tdef)
    let typB = envGetTypB emEnv tref
    createTypeRef visited emEnv tref;
    // nested types
    let nesting = nesting @ [tdef]
    iter (buildTypeDefPass4 visited nesting emEnv) (dest_tdefs tdef.NestedTypes)

//----------------------------------------------------------------------------
// buildModuleType
//----------------------------------------------------------------------------
     
let buildModuleTypePass1 cenv (modB:ModuleBuilder) emEnv (tdef:IL.type_def) =
    if verbose then dprintf1 "buildModuleTypePass1 cenv, tdef.Name = %s\n" tdef.Name;
    let rootTypB  (name,attrs)        = modB.DefineType(name,attrs)       
    buildTypeDefPass1 cenv emEnv modB rootTypB [] tdef

let buildModuleTypePass1b          emEnv tdef = buildTypeDefPass1b [] emEnv tdef
let buildModuleTypePass2           emEnv tdef = buildTypeDefPass2 [] emEnv tdef
let buildModuleTypePass3 cenv modB emEnv tdef = buildTypeDefPass3 cenv [] modB emEnv tdef
let buildModuleTypePass4 visited   emEnv tdef = buildTypeDefPass4 visited [] emEnv tdef

//----------------------------------------------------------------------------
// buildModuleFragment - only the types the fragment get written
//----------------------------------------------------------------------------
    
let buildModuleFragment cenv emEnv (asmB : AssemblyBuilder) (modB : ModuleBuilder) (m:modul) =
    let tdefs = dest_tdefs m.modulTypeDefs in

    let emEnv = fold_left (buildModuleTypePass1 cenv modB) emEnv tdefs
    CompatArray.iter (fun (tyT:Type) -> if verbose then dprintf1 "fqn = %s\n" tyT.FullName) (modB.GetTypes());
    iter (buildModuleTypePass1b emEnv) tdefs
    let emEnv = fold_left buildModuleTypePass2   emEnv  tdefs
    let emEnv = fold_left (buildModuleTypePass3 cenv modB)   emEnv  tdefs
    let visited = Collections.HashMultiMap<_,_>.Create(HashIdentity.Structural,10) 
    let created = Collections.HashMultiMap<_,_>.Create(HashIdentity.Structural,10) 
    //let created = Collections.HashSet<_>.Create(10) 
    iter (buildModuleTypePass4  (visited,created) emEnv) tdefs
    emitCustomAttrs emEnv (fun (x,y) -> modB.SetCustomAttribute(x,y)) m.modulCustomAttrs;    
    m.modulResources |> dest_resources |> List.iter (fun r -> 
        let attribs = (match r.resourceAccess with Resource_public -> ResourceAttributes.Public | Resource_private -> ResourceAttributes.Private) 
        match r.resourceWhere with 
        | Resource_local bf -> 
#if CLI_AT_LEAST_2_0
            modB.DefineManifestResource(r.resourceName, new IO.MemoryStream(bf()), attribs)
#else
            () // Ildiag.dprintf0 "Ignoring manifest resource on .NET 1.x\n"
#endif
        | Resource_file (mr,n) -> asmB.AddResourceFile(r.resourceName, mr.modulRefName, attribs)
        | Resource_assembly _ -> failwith "references to resources other assemblies may not be emitted using System.Reflection");
    emEnv

//----------------------------------------------------------------------------
// test hook
//----------------------------------------------------------------------------

let mkDynamicAssemblyAndModule assemblyName debugInfo =
    let filename = assemblyName ^ ".dll"
    let currentDom  = System.AppDomain.CurrentDomain
    let asmDir  = "."
    let asmName = new AssemblyName()
    asmName.Name <- assemblyName;
    let asmB = currentDom.DefineDynamicAssembly(asmName,AssemblyBuilderAccess.RunAndSave,asmDir) 
    let modB = asmB.DefineDynamicModule(assemblyName,filename,debugInfo) 
    asmB,modB

let emitModuleFragment ilg emEnv (asmB : AssemblyBuilder) (modB : ModuleBuilder) (modul : IL.modul) =
    let cenv = { ilg = ilg }  
    let emEnv = buildModuleFragment cenv emEnv asmB modB modul
    begin match modul.modulManifest with 
    | None -> ()
    | Some mani ->  
       // REVIEW: remainder of manifest
       emitCustomAttrs emEnv (fun (x,y) -> asmB.SetCustomAttribute(x,y)) mani.manifestCustomAttrs;    
    end;
    // invoke entry point methods
    let execEntryPtFun ((typB : TypeBuilder),methodName) () =
      try 
        ignore (typB.InvokeMember(methodName,
                                  flagsL [BindingFlags.InvokeMethod;
                                          BindingFlags.Public;
                                          BindingFlags.Static],
                                  null,
                                  null,
                                  CompatArray.of_list [ ]));
        None
      with 
         | :? System.Reflection.TargetInvocationException as e ->
             Some(e.InnerException)
   
    let emEnv,entryPts = envPopEntryPts emEnv
    let execs = map execEntryPtFun entryPts
    emEnv,execs

//----------------------------------------------------------------------------
// REVIEW:
//  [ ] events todo
//  [ ] check cust attributes get through, e.g STA.
//  [ ] XXX notes (highlight gaps)
//----------------------------------------------------------------------------
