/*
 * Decompiled with CFR 0.152.
 */
package org.gradle.cache.internal;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.SortedSetMultimap;
import com.google.common.collect.TreeMultimap;
import java.io.File;
import java.io.IOException;
import java.util.SortedSet;
import java.util.concurrent.TimeUnit;
import javax.annotation.Nonnull;
import org.apache.commons.io.FileUtils;
import org.gradle.api.logging.Logger;
import org.gradle.api.logging.Logging;
import org.gradle.api.specs.Spec;
import org.gradle.cache.CleanupProgressMonitor;
import org.gradle.cache.internal.DirectoryCleanupAction;
import org.gradle.cache.internal.VersionSpecificCacheDirectory;
import org.gradle.cache.internal.VersionSpecificCacheDirectoryScanner;
import org.gradle.internal.time.Time;
import org.gradle.internal.time.Timer;
import org.gradle.util.GFileUtils;
import org.gradle.util.GradleVersion;

public class VersionSpecificCacheCleanupAction
implements DirectoryCleanupAction {
    @VisibleForTesting
    static final String MARKER_FILE_PATH = "fileHashes/fileHashes.lock";
    private static final Logger LOGGER = Logging.getLogger(VersionSpecificCacheCleanupAction.class);
    private static final long CLEANUP_INTERVAL_IN_HOURS = 24L;
    private final VersionSpecificCacheDirectoryScanner versionSpecificCacheDirectoryScanner;
    private final long maxUnusedDaysForReleases;
    private final long maxUnusedDaysForSnapshots;

    public VersionSpecificCacheCleanupAction(File cacheBaseDir, long maxUnusedDaysForReleasesAndSnapshots) {
        this(cacheBaseDir, maxUnusedDaysForReleasesAndSnapshots, maxUnusedDaysForReleasesAndSnapshots);
    }

    public VersionSpecificCacheCleanupAction(File cacheBaseDir, long maxUnusedDaysForReleases, long maxUnusedDaysForSnapshots) {
        Preconditions.checkArgument((maxUnusedDaysForReleases >= maxUnusedDaysForSnapshots ? 1 : 0) != 0, (String)"maxUnusedDaysForReleases (%s) must be greater than or equal to maxUnusedDaysForSnapshots (%s)", (long)maxUnusedDaysForReleases, (long)maxUnusedDaysForSnapshots);
        this.versionSpecificCacheDirectoryScanner = new VersionSpecificCacheDirectoryScanner(cacheBaseDir);
        this.maxUnusedDaysForReleases = maxUnusedDaysForReleases;
        this.maxUnusedDaysForSnapshots = maxUnusedDaysForSnapshots;
    }

    @Nonnull
    public String getDisplayName() {
        return "Deleting unused version-specific caches in " + this.versionSpecificCacheDirectoryScanner.getBaseDir();
    }

    @Override
    public boolean execute(@Nonnull CleanupProgressMonitor progressMonitor) {
        if (this.requiresCleanup()) {
            Timer timer = Time.startTimer();
            this.performCleanup(progressMonitor);
            LOGGER.debug("Processed version-specific caches at {} for cleanup in {}", (Object)this.versionSpecificCacheDirectoryScanner.getBaseDir(), (Object)timer.getElapsed());
            return true;
        }
        return false;
    }

    private boolean requiresCleanup() {
        File gcFile = this.getGcFile();
        if (!gcFile.exists()) {
            return gcFile.getParentFile().exists();
        }
        long duration = System.currentTimeMillis() - gcFile.lastModified();
        long timeInHours = TimeUnit.MILLISECONDS.toHours(duration);
        return timeInHours >= 24L;
    }

    private void markCleanedUp() {
        GFileUtils.touch((File)this.getGcFile());
    }

    private File getGcFile() {
        File currentVersionCacheDir = this.versionSpecificCacheDirectoryScanner.getDirectory(GradleVersion.current());
        return new File(currentVersionCacheDir, "gc.properties");
    }

    private void performCleanup(CleanupProgressMonitor progressMonitor) {
        MinimumTimestampProvider minimumTimestampProvider = new MinimumTimestampProvider();
        SortedSetMultimap<GradleVersion, VersionSpecificCacheDirectory> cacheDirsByBaseVersion = this.scanForVersionSpecificCacheDirs();
        for (GradleVersion baseVersion : cacheDirsByBaseVersion.keySet()) {
            this.performCleanup(cacheDirsByBaseVersion.get((Object)baseVersion), minimumTimestampProvider, progressMonitor);
        }
        this.markCleanedUp();
    }

    private SortedSetMultimap<GradleVersion, VersionSpecificCacheDirectory> scanForVersionSpecificCacheDirs() {
        TreeMultimap cacheDirsByBaseVersion = TreeMultimap.create();
        for (VersionSpecificCacheDirectory cacheDir : this.versionSpecificCacheDirectoryScanner.getExistingDirectories()) {
            cacheDirsByBaseVersion.put((Object)cacheDir.getVersion().getBaseVersion(), (Object)cacheDir);
        }
        return cacheDirsByBaseVersion;
    }

    private void performCleanup(SortedSet<VersionSpecificCacheDirectory> cacheDirsWithSameBaseVersion, MinimumTimestampProvider minimumTimestampProvider, CleanupProgressMonitor progressMonitor) {
        CleanupCondition cleanupCondition = new CleanupCondition(cacheDirsWithSameBaseVersion, minimumTimestampProvider);
        for (VersionSpecificCacheDirectory cacheDir : cacheDirsWithSameBaseVersion) {
            if (cleanupCondition.isSatisfiedBy(cacheDir)) {
                progressMonitor.incrementDeleted();
                try {
                    this.deleteCacheDir(cacheDir.getDir());
                }
                catch (Exception e) {
                    LOGGER.error("Failed to process/clean up version-specific cache directory: {}", (Object)cacheDir.getDir(), (Object)e);
                }
                continue;
            }
            progressMonitor.incrementSkipped();
        }
    }

    private void deleteCacheDir(File cacheDir) throws IOException {
        LOGGER.debug("Deleting version-specific cache directory at {}", (Object)cacheDir);
        FileUtils.deleteDirectory((File)cacheDir);
    }

    private class MinimumTimestampProvider {
        private final long minimumReleaseTimestamp;
        private final long minimumSnapshotTimestamp;

        MinimumTimestampProvider() {
            long startTime = System.currentTimeMillis();
            this.minimumReleaseTimestamp = this.compute(startTime, VersionSpecificCacheCleanupAction.this.maxUnusedDaysForReleases);
            this.minimumSnapshotTimestamp = this.compute(startTime, VersionSpecificCacheCleanupAction.this.maxUnusedDaysForSnapshots);
        }

        private long compute(long startTime, long maxUnusedDays) {
            return Math.max(0L, startTime - TimeUnit.DAYS.toMillis(maxUnusedDays));
        }

        long forReleases() {
            return this.minimumReleaseTimestamp;
        }

        long forSnapshots() {
            return this.minimumSnapshotTimestamp;
        }
    }

    private static class CleanupCondition
    implements Spec<VersionSpecificCacheDirectory> {
        private final SortedSet<VersionSpecificCacheDirectory> cacheDirsWithSameBaseVersion;
        private final MinimumTimestampProvider minimumTimestampProvider;

        CleanupCondition(SortedSet<VersionSpecificCacheDirectory> cacheDirsWithSameBaseVersion, MinimumTimestampProvider minimumTimestampProvider) {
            this.cacheDirsWithSameBaseVersion = cacheDirsWithSameBaseVersion;
            this.minimumTimestampProvider = minimumTimestampProvider;
        }

        public boolean isSatisfiedBy(VersionSpecificCacheDirectory cacheDir) {
            if (cacheDir.getVersion().compareTo(GradleVersion.current()) >= 0) {
                return false;
            }
            File markerFile = new File(cacheDir.getDir(), VersionSpecificCacheCleanupAction.MARKER_FILE_PATH);
            return markerFile.exists() && this.markerFileHasNotBeenTouchedRecently(cacheDir, markerFile);
        }

        private boolean markerFileHasNotBeenTouchedRecently(VersionSpecificCacheDirectory cacheDir, File markerFile) {
            if (markerFile.lastModified() < this.minimumTimestampProvider.forReleases()) {
                return true;
            }
            if (cacheDir.getVersion().isSnapshot() && markerFile.lastModified() < this.minimumTimestampProvider.forSnapshots()) {
                return this.cacheDirsWithSameBaseVersion.tailSet(cacheDir).size() > 1;
            }
            return false;
        }
    }
}

