/*
 * Copyright 2019 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.gradle.instantexecution.serialization.codecs

import org.gradle.instantexecution.serialization.Codec
import org.gradle.instantexecution.serialization.ReadContext
import org.gradle.instantexecution.serialization.WriteContext


object BrokenValueCodec : Codec<BrokenValue> {

    override suspend fun WriteContext.encode(value: BrokenValue) {
        maybeEncode(value.failure.cause)
        writeNullableString(value.failure.message)
    }

    private
    fun WriteContext.maybeEncode(failure: Throwable?) {
        if (failure == null) {
            writeBoolean(false)
        } else {
            writeBoolean(true)
            maybeEncode(failure.cause)
            writeNullableString(failure.message)
        }
    }

    override suspend fun ReadContext.decode(): BrokenValue {
        val cause = maybeDecode()
        val message = readNullableString()
        val exception = RuntimeException(message, cause)
        return BrokenValue(exception)
    }

    private
    fun ReadContext.maybeDecode(): Throwable? {
        if (readBoolean()) {
            val cause = maybeDecode()
            val message = readNullableString()
            return RuntimeException(message, cause)
        } else {
            return null
        }
    }
}
