/*
 * Copyright 2017 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.gradle.composite.internal;

import com.google.common.collect.ImmutableList;
import org.gradle.api.artifacts.component.BuildIdentifier;
import org.gradle.api.internal.project.ProjectStateRegistry;
import org.gradle.internal.build.BuildStateRegistry;
import org.gradle.internal.build.IncludedBuildState;
import org.gradle.internal.concurrent.CompositeStoppable;
import org.gradle.internal.concurrent.ExecutorFactory;
import org.gradle.internal.concurrent.ManagedExecutor;
import org.gradle.internal.concurrent.Stoppable;
import org.gradle.internal.resources.ResourceLockCoordinationService;

import java.util.HashMap;
import java.util.Map;
import java.util.function.Consumer;

class DefaultIncludedBuildControllers implements Stoppable, IncludedBuildControllers {
    private final Map<BuildIdentifier, IncludedBuildController> buildControllers = new HashMap<>();
    private final ManagedExecutor executorService;
    private final ResourceLockCoordinationService coordinationService;
    private final ProjectStateRegistry projectStateRegistry;
    private final BuildStateRegistry buildRegistry;

    DefaultIncludedBuildControllers(ExecutorFactory executorFactory, BuildStateRegistry buildRegistry, ResourceLockCoordinationService coordinationService, ProjectStateRegistry projectStateRegistry) {
        this.buildRegistry = buildRegistry;
        this.executorService = executorFactory.create("included builds");
        this.coordinationService = coordinationService;
        this.projectStateRegistry = projectStateRegistry;
    }

    @Override
    public IncludedBuildController getBuildController(BuildIdentifier buildId) {
        IncludedBuildController buildController = buildControllers.get(buildId);
        if (buildController != null) {
            return buildController;
        }

        IncludedBuildState build = buildRegistry.getIncludedBuild(buildId);
        DefaultIncludedBuildController newBuildController = new DefaultIncludedBuildController(build, coordinationService, projectStateRegistry);
        buildControllers.put(buildId, newBuildController);
        return newBuildController;
    }

    @Override
    public void startTaskExecution() {
        for (IncludedBuildController buildController : buildControllers.values()) {
            buildController.startTaskExecution(executorService);
        }
    }

    @Override
    public void populateTaskGraphs() {
        boolean tasksDiscovered = true;
        while (tasksDiscovered) {
            tasksDiscovered = false;
            for (IncludedBuildController buildController : ImmutableList.copyOf(buildControllers.values())) {
                if (buildController.populateTaskGraph()) {
                    tasksDiscovered = true;
                }
            }
        }
    }

    @Override
    public void awaitTaskCompletion(Consumer<? super Throwable> taskFailures) {
        for (IncludedBuildController buildController : buildControllers.values()) {
            buildController.awaitTaskCompletion(taskFailures);
        }
    }

    @Override
    public void finishBuild(Consumer<? super Throwable> collector) {
        CompositeStoppable.stoppable(buildControllers.values()).stop();
        buildControllers.clear();
        for (IncludedBuildState includedBuild : buildRegistry.getIncludedBuilds()) {
            includedBuild.finishBuild(collector);
        }
    }

    @Override
    public void stop() {
        CompositeStoppable.stoppable(buildControllers.values()).stop();
        executorService.stop();
    }
}
