package org.springframework.uaa.client;

import java.util.List;

import org.springframework.uaa.client.util.Assert;

/**
 * Represents common product names and dependency information.
 * 
 * <p>
 * Spring UAA allows product names to be expressed as a string. This freedom is necessary as new products may be added
 * at any time and we shouldn't need to publish a new version of UAA just to support them.
 * 
 * <p>
 * Usually UAA clients should consult {@link UaaDetectedProducts#shouldReportUsage(String, String)} before reporting
 * usage to {@link UaaService#registerProductUsage()}. This is currently not enforced but reports of un-detected
 * products may be ignored on the UAA server.
 * 
 * @author Ben Alex
 * @author Christian Dupuis
 * @since 1.0.1
 */
public interface UaaDetectedProducts {

	public static final ProductInfo SPRING_UAA = new ProductInfo("Spring UAA", "org.springframework.uaa", "org.springframework.uaa.client", "org.springframework.uaa.client.UaaService");

	/**
	 * Returns a {@link ProductInfo} for the given <code>groupId</code> and <code>artifactId</code>.
	 * 
	 * @param groupId the groupId of the product
	 * @param artifactId the productId of the product
	 * @return the {@link ProductInfo} for given <code>groupId</code> and <code>artifactId</code> or <code>null</code>
	 * if no such product exists
	 */
	ProductInfo getDetectedProductInfo(String groupId, String artifactId);

	/**
	 * Returns the list of all detected products that UAA clients should report to {@link UaaService}.
	 * 
	 * @return the list of all detected products
	 */
	List<ProductInfo> getDetectedProductInfos();

	/**
	 * Verifies if a product identified by <code>groupId</code> and <code>artifactId</code> should be reported to
	 * {@link UaaService}.
	 * 
	 * @param groupId the groupId of the product
	 * @param artifactId the productId of the product
	 * @return <code>true</code> if UAA clients should present this product to {@link UaaService}
	 */
	boolean shouldReportUsage(String groupId, String artifactId);

	final static class ProductInfo {

		private String artifactId;

		private String groupId;

		private String productName;

		private String typeName;

		public ProductInfo(String productName, String groupId, String artifactId, String typeName) {
			Assert.hasLength(productName, "Product name required");
			Assert.hasLength(productName, "Group ID required");
			Assert.hasLength(productName, "Artifact ID required");
			Assert.notNull(typeName, "Type name cannot be null (but may be empty)");
			this.productName = productName;
			this.groupId = groupId;
			this.artifactId = artifactId;
			this.typeName = typeName;
		}

		@Override
		public boolean equals(Object obj) {
			if (this == obj)
				return true;
			if (obj == null)
				return false;
			if (getClass() != obj.getClass())
				return false;
			ProductInfo other = (ProductInfo) obj;
			if (artifactId == null) {
				if (other.artifactId != null)
					return false;
			}
			else if (!artifactId.equals(other.artifactId))
				return false;
			if (groupId == null) {
				if (other.groupId != null)
					return false;
			}
			else if (!groupId.equals(other.groupId))
				return false;
			if (productName == null) {
				if (other.productName != null)
					return false;
			}
			else if (!productName.equals(other.productName))
				return false;
			if (typeName == null) {
				if (other.typeName != null)
					return false;
			}
			else if (!typeName.equals(other.typeName))
				return false;
			return true;
		}

		public final String getArtifactId() {
			return artifactId;
		}

		public final String getGroupId() {
			return groupId;
		}

		public final String getProductName() {
			return productName;
		}

		/**
		 * @return the type name for fallback detection (this may be an empty string, but never null)
		 */
		public String getTypeName() {
			return typeName;
		}

		@Override
		public int hashCode() {
			final int prime = 31;
			int result = 1;
			result = prime * result + ((artifactId == null) ? 0 : artifactId.hashCode());
			result = prime * result + ((groupId == null) ? 0 : groupId.hashCode());
			result = prime * result + ((productName == null) ? 0 : productName.hashCode());
			result = prime * result + ((typeName == null) ? 0 : typeName.hashCode());
			return result;
		}

	}
}
