package org.springframework.uaa.client.util;

import java.io.IOException;
import java.io.StringWriter;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import org.json.simple.JSONArray;
import org.json.simple.JSONObject;
import org.json.simple.JSONValue;
import org.json.simple.parser.JSONParser;
import org.springframework.uaa.client.protobuf.UaaClient.FeatureUse;
import org.springframework.uaa.client.protobuf.UaaClient.ProductUse;
import org.springframework.uaa.client.protobuf.UaaClient.Project;
import org.springframework.uaa.client.protobuf.UaaClient.UaaEnvelope;
import org.springframework.uaa.client.protobuf.UaaClient.UserAgent;
import org.springframework.uaa.client.protobuf.UaaClient.Privacy.PrivacyLevel;

/**
 * Helper methods to present a {@link UserAgent} and {@link UaaEnvelope} in human-readable JSON
 * format. Note UAA itself doesn't use JSON, instead internally relying on Protocol Buffers byte
 * arrays. JSON is only used to display output for ease of reading by UAA users. Note some products
 * which embed UAA also elect to use JSON for encoding the "custom_data" which can be associated
 * with a ProductUse or FeatureUse record, however this is not a requirement of UAA.
 * 
 * @author Ben Alex
 * @author Christian Dupuis
 * @since 1.0.1
 */
@SuppressWarnings("unchecked")
public abstract class StringUtils {
	private static final SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
	
	public static String toString(UaaEnvelope ue) {
		if (ue == null) {
			return "";
		}
		
		Map<String, Object> o = new LinkedHashMap<String, Object>();
		o.put("installation_id", new UUID(ue.getInstallationIdentifierMostSignificantBits(), ue.getInstallationIdentifierLeastSignificantBits()).toString());
		o.put("privacy_level", ue.getPrivacy().getPrivacyLevel().toString());
		o.put("privacy_level_last_changed", sdf.format(new Date(ue.getPrivacy().getDateLastChanged())));
		PrivacyLevel pl = ue.getPrivacy().getPrivacyLevel();
		if (pl != PrivacyLevel.DECLINE_TOU && pl != PrivacyLevel.UNDECIDED_TOU) {
			o.put("usage_data", getUserAgent(ue.getUserAgent()));
		}
		StringWriter out = new StringWriter();
		try {
			JSONValue.writeJSONString(o, out);
		} catch (IOException ignore) {}
		return out.toString();
	}

	private static Map<String, Object> getUserAgent(UserAgent ua) {
		Map<String, Object> o = new LinkedHashMap<String, Object>();
		if (ua == null) {
			return o;
		}
		
		o.put("user_country", ua.getUserCountry());
		o.put("user_language", ua.getUserLanguage());
		
		if (ua.getProductUseCount() > 0) {
			JSONArray a = new JSONArray();
			
			List<ProductUse> productUseList = new ArrayList<ProductUse>(ua.getProductUseList());
			
			// Alphabetically sort the products so that it is easier to find things
			Collections.sort(productUseList, new Comparator<ProductUse>() {
				public int compare(ProductUse o1, ProductUse o2) {
					return o1.getProduct().getName().compareTo(o2.getProduct().getName());
				}
			});
			
			for (int i = 0; i < productUseList.size(); i++) {
				ProductUse pu = productUseList.get(i);
				a.add(getProductUse(pu));
			}
			
			o.put("product_use", a);
		}
			
		return o;
	}
	
	private static Map<String, Object> getProductUse(ProductUse pu) {
		Map<String, Object> o = new LinkedHashMap<String, Object>();
		o.put("product_name", JSONObject.escape(pu.getProduct().getName()));
		if (pu.getDateLastUsed() > 0) {
			o.put("product_last_used", sdf.format(new Date(pu.getDateLastUsed())));
		}
		o.put("product_major_version", pu.getProduct().getMajorVersion());
		o.put("product_minor_version", pu.getProduct().getMinorVersion());
		o.put("product_patch_version", pu.getProduct().getPatchVersion());
		o.put("product_release_qualifier", JSONObject.escape(pu.getProduct().getReleaseQualifier()));
		o.put("product_source_control_id", JSONObject.escape(pu.getProduct().getSourceControlIdentifier()));
		
		if (!pu.getProductData().isEmpty()) {
			ByteDataContainer c = getByteDataForDisplay(pu.getProductData().toByteArray());
			o.put("product_data_" + c.getKey(), c.getValue());
		}
		
		if (pu.getFeatureUseCount() > 0) {
			List<FeatureUse> featureUseList = pu.getFeatureUseList();
			JSONArray a = new JSONArray();
			for (int x = 0; x < featureUseList.size(); x++) {
				FeatureUse fu = featureUseList.get(x);
				a.add(getFeatureUse(fu));
			}
			o.put("feature_use", a);
		}
		
		if (pu.getProjectsUsingCount() > 0) {
			JSONArray a = new JSONArray();
			List<Project> projectsUsingList = pu.getProjectsUsingList();
			for (int x = 0; x < projectsUsingList.size(); x++) {
				Project p = projectsUsingList.get(x);
				a.add(getProjectUse(p));
			}
			o.put("projects_using", a);
		}
		
		return o;
	}
	
	private static Map<String, Object> getFeatureUse(FeatureUse fu) {
		Map<String, Object> o = new LinkedHashMap<String, Object>();
		o.put("feature_name", JSONObject.escape(fu.getName()));
		if (fu.getDateLastUsed() > 0) {
			o.put("feature_last_used", sdf.format(new Date(fu.getDateLastUsed())));
		}
		o.put("feature_major_version", fu.getMajorVersion());
		o.put("feature_minor_version", fu.getMinorVersion());
		o.put("feature_patch_version", fu.getPatchVersion());
		o.put("feature_release_qualifier", JSONObject.escape(fu.getReleaseQualifier()));
		o.put("feature_source_control_id", JSONObject.escape(fu.getSourceControlIdentifier()));
		if (!fu.getFeatureData().isEmpty()) {
			ByteDataContainer c = getByteDataForDisplay(fu.getFeatureData().toByteArray());
			o.put("feature_data_" + c.getKey(), c.getValue());
		}
		return o;
	}
	
	private static Map<String, Object> getProjectUse(Project p) {
		Map<String, Object> o = new LinkedHashMap<String, Object>();
		o.put("project_sha256_hash", HexUtils.toHex(p.getProjectSha256Hash().toByteArray()));
		o.put("project_last_used", sdf.format(new Date(p.getDateLastUsed())));
		return o;
	}

	private static ByteDataContainer getByteDataForDisplay(byte[] bytes) {
		ByteDataContainer result = new ByteDataContainer();
		if (isAscii(bytes)) {
			String data = new String(bytes);
			try {
				JSONParser parser = new JSONParser();
				result.key = "json";
				result.valueAsParsedByJson = parser.parse(data);
			}
			catch (Exception e) {
				result.key = "ascii";
				result.valueAsString = new String(bytes);
			}
		} else {
			result.key = "base64";
			result.valueAsString = Base64.encodeBytes(bytes);
		}
		return result;
	}
	
	// Package protected to enable unit tests
	static boolean isAscii(byte[] bytes) {
		for (byte b : bytes) {
			if (b < 32 || b > 126) {
				return false;
			}
		}
		return true;
	}
	
	private static class ByteDataContainer {
		private String key = null; // never null
		private Object valueAsParsedByJson = null; // may be null, but then the valueAsString won't be
		private String valueAsString = null; // may be null, but then the valueAsParsedByJson won't be
		public Object getValue() {
			return valueAsParsedByJson == null ? valueAsString : valueAsParsedByJson;
		}
		public String getKey() {
			return key;
		}
	}
}
