/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.collect;

import com.google.common.base.Function;
import com.google.common.base.Objects;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.base.Predicates;
import com.google.common.collect.AbstractIndexedListIterator;
import com.google.common.collect.AbstractIterator;
import com.google.common.collect.CollectPreconditions;
import com.google.common.collect.ConsumingQueueIterator;
import com.google.common.collect.MultitransformedIterator;
import com.google.common.collect.PeekingIterator;
import com.google.common.collect.TransformedIterator;
import com.google.common.collect.UnmodifiableIterator;
import com.google.common.collect.UnmodifiableListIterator;
import com.google.common.primitives.Ints;
import com.google.errorprone.annotations.CanIgnoreReturnValue;
import java.util.Collection;
import java.util.Iterator;
import java.util.NoSuchElementException;

public final class Iterators {
    static final UnmodifiableListIterator<Object> EMPTY_LIST_ITERATOR = new UnmodifiableListIterator<Object>(){

        @Override
        public boolean hasNext() {
            return false;
        }

        @Override
        public Object next() {
            throw new NoSuchElementException();
        }

        @Override
        public boolean hasPrevious() {
            return false;
        }

        @Override
        public Object previous() {
            throw new NoSuchElementException();
        }

        @Override
        public int nextIndex() {
            return 0;
        }

        @Override
        public int previousIndex() {
            return -1;
        }
    };
    private static final Iterator<Object> EMPTY_MODIFIABLE_ITERATOR = new Iterator<Object>(){

        @Override
        public boolean hasNext() {
            return false;
        }

        @Override
        public Object next() {
            throw new NoSuchElementException();
        }

        @Override
        public void remove() {
            CollectPreconditions.checkRemove(false);
        }
    };

    static <T> UnmodifiableIterator<T> emptyIterator() {
        return Iterators.emptyListIterator();
    }

    static <T> UnmodifiableListIterator<T> emptyListIterator() {
        return EMPTY_LIST_ITERATOR;
    }

    static <T> Iterator<T> emptyModifiableIterator() {
        return EMPTY_MODIFIABLE_ITERATOR;
    }

    public static <T> UnmodifiableIterator<T> unmodifiableIterator(final Iterator<? extends T> iterator2) {
        Preconditions.checkNotNull(iterator2);
        if (iterator2 instanceof UnmodifiableIterator) {
            UnmodifiableIterator result2 = (UnmodifiableIterator)iterator2;
            return result2;
        }
        return new UnmodifiableIterator<T>(){

            @Override
            public boolean hasNext() {
                return iterator2.hasNext();
            }

            @Override
            public T next() {
                return iterator2.next();
            }
        };
    }

    public static int size(Iterator<?> iterator2) {
        long count2 = 0L;
        while (iterator2.hasNext()) {
            iterator2.next();
            ++count2;
        }
        return Ints.saturatedCast(count2);
    }

    public static boolean contains(Iterator<?> iterator2, Object element) {
        return Iterators.any(iterator2, Predicates.equalTo(element));
    }

    @CanIgnoreReturnValue
    public static boolean removeAll(Iterator<?> removeFrom, Collection<?> elementsToRemove) {
        return Iterators.removeIf(removeFrom, Predicates.in(elementsToRemove));
    }

    @CanIgnoreReturnValue
    public static <T> boolean removeIf(Iterator<T> removeFrom, Predicate<? super T> predicate) {
        Preconditions.checkNotNull(predicate);
        boolean modified = false;
        while (removeFrom.hasNext()) {
            if (!predicate.apply(removeFrom.next())) continue;
            removeFrom.remove();
            modified = true;
        }
        return modified;
    }

    public static boolean elementsEqual(Iterator<?> iterator1, Iterator<?> iterator2) {
        while (iterator1.hasNext()) {
            Object o2;
            if (!iterator2.hasNext()) {
                return false;
            }
            Object o1 = iterator1.next();
            if (Objects.equal(o1, o2 = iterator2.next())) continue;
            return false;
        }
        return !iterator2.hasNext();
    }

    @CanIgnoreReturnValue
    public static <T> T getOnlyElement(Iterator<T> iterator2) {
        T first2 = iterator2.next();
        if (!iterator2.hasNext()) {
            return first2;
        }
        StringBuilder sb = new StringBuilder().append("expected one element but was: <").append(first2);
        for (int i = 0; i < 4 && iterator2.hasNext(); ++i) {
            sb.append(", ").append(iterator2.next());
        }
        if (iterator2.hasNext()) {
            sb.append(", ...");
        }
        sb.append('>');
        throw new IllegalArgumentException(sb.toString());
    }

    @CanIgnoreReturnValue
    public static <T> boolean addAll(Collection<T> addTo, Iterator<? extends T> iterator2) {
        Preconditions.checkNotNull(addTo);
        Preconditions.checkNotNull(iterator2);
        boolean wasModified = false;
        while (iterator2.hasNext()) {
            wasModified |= addTo.add(iterator2.next());
        }
        return wasModified;
    }

    public static <T> Iterator<T> concat(Iterator<? extends T> a, Iterator<? extends T> b) {
        Preconditions.checkNotNull(a);
        Preconditions.checkNotNull(b);
        return Iterators.concat(new ConsumingQueueIterator<Iterator>(a, b));
    }

    public static <T> Iterator<T> concat(Iterator<? extends Iterator<? extends T>> inputs) {
        return new ConcatenatedIterator(inputs);
    }

    public static <T> UnmodifiableIterator<T> filter(final Iterator<T> unfiltered, final Predicate<? super T> retainIfTrue) {
        Preconditions.checkNotNull(unfiltered);
        Preconditions.checkNotNull(retainIfTrue);
        return new AbstractIterator<T>(){

            @Override
            protected T computeNext() {
                while (unfiltered.hasNext()) {
                    Object element = unfiltered.next();
                    if (!retainIfTrue.apply(element)) continue;
                    return element;
                }
                return this.endOfData();
            }
        };
    }

    public static <T> boolean any(Iterator<T> iterator2, Predicate<? super T> predicate) {
        return Iterators.indexOf(iterator2, predicate) != -1;
    }

    public static <T> boolean all(Iterator<T> iterator2, Predicate<? super T> predicate) {
        Preconditions.checkNotNull(predicate);
        while (iterator2.hasNext()) {
            T element = iterator2.next();
            if (predicate.apply(element)) continue;
            return false;
        }
        return true;
    }

    public static <T> int indexOf(Iterator<T> iterator2, Predicate<? super T> predicate) {
        Preconditions.checkNotNull(predicate, "predicate");
        int i = 0;
        while (iterator2.hasNext()) {
            T current = iterator2.next();
            if (predicate.apply(current)) {
                return i;
            }
            ++i;
        }
        return -1;
    }

    public static <F, T> Iterator<T> transform(Iterator<F> fromIterator, final Function<? super F, ? extends T> function2) {
        Preconditions.checkNotNull(function2);
        return new TransformedIterator<F, T>(fromIterator){

            @Override
            T transform(F from) {
                return function2.apply(from);
            }
        };
    }

    public static <T> T getNext(Iterator<? extends T> iterator2, T defaultValue) {
        return iterator2.hasNext() ? iterator2.next() : defaultValue;
    }

    static <T> T pollNext(Iterator<T> iterator2) {
        if (iterator2.hasNext()) {
            T result2 = iterator2.next();
            iterator2.remove();
            return result2;
        }
        return null;
    }

    static void clear(Iterator<?> iterator2) {
        Preconditions.checkNotNull(iterator2);
        while (iterator2.hasNext()) {
            iterator2.next();
            iterator2.remove();
        }
    }

    @SafeVarargs
    public static <T> UnmodifiableIterator<T> forArray(T ... array) {
        return Iterators.forArray(array, 0, array.length, 0);
    }

    static <T> UnmodifiableListIterator<T> forArray(final T[] array, final int offset2, int length, int index2) {
        Preconditions.checkArgument(length >= 0);
        int end = offset2 + length;
        Preconditions.checkPositionIndexes(offset2, end, array.length);
        Preconditions.checkPositionIndex(index2, length);
        if (length == 0) {
            return Iterators.emptyListIterator();
        }
        return new AbstractIndexedListIterator<T>(length, index2){

            @Override
            protected T get(int index2) {
                return array[offset2 + index2];
            }
        };
    }

    public static <T> UnmodifiableIterator<T> singletonIterator(final T value) {
        return new UnmodifiableIterator<T>(){
            boolean done;

            @Override
            public boolean hasNext() {
                return !this.done;
            }

            @Override
            public T next() {
                if (this.done) {
                    throw new NoSuchElementException();
                }
                this.done = true;
                return value;
            }
        };
    }

    public static <T> PeekingIterator<T> peekingIterator(Iterator<? extends T> iterator2) {
        if (iterator2 instanceof PeekingImpl) {
            PeekingImpl peeking = (PeekingImpl)iterator2;
            return peeking;
        }
        return new PeekingImpl<T>(iterator2);
    }

    private static class ConcatenatedIterator<T>
    extends MultitransformedIterator<Iterator<? extends T>, T> {
        public ConcatenatedIterator(Iterator<? extends Iterator<? extends T>> iterators) {
            super(ConcatenatedIterator.getComponentIterators(iterators));
        }

        @Override
        Iterator<? extends T> transform(Iterator<? extends T> iterator2) {
            return iterator2;
        }

        private static <T> Iterator<Iterator<? extends T>> getComponentIterators(Iterator<? extends Iterator<? extends T>> iterators) {
            return new MultitransformedIterator<Iterator<? extends T>, Iterator<? extends T>>(iterators){

                @Override
                Iterator<? extends Iterator<? extends T>> transform(Iterator<? extends T> iterator2) {
                    if (iterator2 instanceof ConcatenatedIterator) {
                        ConcatenatedIterator concatIterator = (ConcatenatedIterator)iterator2;
                        return ConcatenatedIterator.getComponentIterators(concatIterator.backingIterator);
                    }
                    return Iterators.singletonIterator(iterator2);
                }
            };
        }
    }

    private static class PeekingImpl<E>
    implements PeekingIterator<E> {
        private final Iterator<? extends E> iterator;
        private boolean hasPeeked;
        private E peekedElement;

        public PeekingImpl(Iterator<? extends E> iterator2) {
            this.iterator = Preconditions.checkNotNull(iterator2);
        }

        @Override
        public boolean hasNext() {
            return this.hasPeeked || this.iterator.hasNext();
        }

        @Override
        public E next() {
            if (!this.hasPeeked) {
                return this.iterator.next();
            }
            E result2 = this.peekedElement;
            this.hasPeeked = false;
            this.peekedElement = null;
            return result2;
        }

        @Override
        public void remove() {
            Preconditions.checkState(!this.hasPeeked, "Can't remove after you've peeked at next");
            this.iterator.remove();
        }

        @Override
        public E peek() {
            if (!this.hasPeeked) {
                this.peekedElement = this.iterator.next();
                this.hasPeeked = true;
            }
            return this.peekedElement;
        }
    }
}

