/*
 * Copyright 2002-2008 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package test.feature.externalbean;

import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertSame;

import org.junit.Test;

import org.springframework.config.java.annotation.Bean;
import org.springframework.config.java.annotation.Configuration;
import org.springframework.config.java.annotation.ExternalBean;
import org.springframework.config.java.context.JavaConfigApplicationContext;

import test.common.beans.TestBean;


/**
 * Integration tests for scenarios where multiple ExternalBean annotations
 * depend on each other.
 *
 * @author  Costin Leau
 * @author  Chris Beams
 */
public class ExternalBeanDependenciesTests {

    @Test
    public void testCircularExternalBean() throws Exception {
        JavaConfigApplicationContext ctx =
            new JavaConfigApplicationContext(FirstConfiguration.class, SecondConfiguration.class);

        TestBean beanA = (TestBean) ctx.getBean("beanA");
        TestBean beanB = (TestBean) ctx.getBean("beanB");

        assertNotNull(beanA.getSpouse());
        assertNotNull(beanB.getSpouse());

        TestBean spouseA = (TestBean) ctx.getBean("spouseA");
        TestBean spouseB = (TestBean) ctx.getBean("spouseB");

        assertSame(spouseB, beanA.getSpouse());
        assertSame(spouseA, beanB.getSpouse());
    }

    @Test
    public void testNamedExternalBean() throws Exception {
        JavaConfigApplicationContext ctx =
            new JavaConfigApplicationContext(NamedExternalConfiguration.class, Foo.class);

        assertSame(Foo.obj, ctx.getBean("usesExternal"));
    }

    @Configuration
    static class FirstConfiguration {

        // beanA() depends on spouseB()
        @Bean
        public TestBean beanA() {
            TestBean bean = new TestBean("beanA");
            bean.setSpouse(spouseB());
            return bean;
        }

        @Bean
        public TestBean spouseA() { return new TestBean("spouseA"); }

        @ExternalBean
        public TestBean spouseB() { throw new IllegalArgumentException("method was not decorated"); }
    }

    @Configuration
    static class SecondConfiguration {

        // beanB() depends on spouseA()
        @Bean
        public TestBean beanB() {
            TestBean bean = new TestBean("beanB");
            bean.setSpouse(spouseA());
            return bean;
        }

        @Bean
        public TestBean spouseB() { return new TestBean("spouseB"); }

        @ExternalBean
        public TestBean spouseA() { throw new IllegalArgumentException("method was not decorated"); }
    }

    @Configuration
    abstract static class NamedExternalConfiguration {
        @Bean
        public Object usesExternal() { return namedExternal(); }

        @ExternalBean("nameOverride")
        protected abstract Object namedExternal();
    }

    @Configuration
    static class Foo {
        static Object obj = new Object();

        @Bean
        public Object nameOverride() { return obj; }
    }

}
