<!--
    Licensed to the Apache Software Foundation (ASF) under one
    or more contributor license agreements.  See the NOTICE file
    distributed with this work for additional information
    regarding copyright ownership.  The ASF licenses this file
    to you under the Apache License, Version 2.0 (the
    "License"); you may not use this file except in compliance
    with the License.  You may obtain a copy of the License at

      http://www.apache.org/licenses/LICENSE-2.0

    Unless required by applicable law or agreed to in writing,
    software distributed under the License is distributed on an
    "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
    KIND, either express or implied.  See the License for the
    specific language governing permissions and limitations
    under the License.
-->
# Custom Log Label Examples

The following are examples illustrating how to add custom labels to logs generated by UserALE.js. Custom labels are
useful in cases where one would prefer to use a more descriptive label for an element users interact with or when
logging information about a specific feature.

## Adding Custom Labels to Logs

#### Important Note
Please be aware that when adding custom logs without disabling raw logging for those specific logs, duplicate log entries will be generated: one for raw logs and another for custom logs.

### Example 1

Consider the following HTML:

```html

<div>
    <button>New Feature</button>
</div>
```

The following code snippet will add a custom field, `customLabel`, and send a log whenever the new feature button is
clicked:

```js
window.userale.addCallbacks({
    map(log, e) {
        // determine whether we want to add custom labels to the log
        if (e && e.target.innerHTML !== 'New Feature') {
            return log; // normal logging
        }
        // if the event occurred on the New Feature, add custom labeling
        return {
            ...log,
            customLabel: 'New Feature',
            logType: 'custom',
        }
    }
});
```

### Example 2

Let's say you want to generate logs on custom events, or events not currently supported by UserALE.js. In this case, you
can add an event listener and use the `customPackageLog` function as shown below:

```js
    document.addEventListener('customEvent', function (e) {
        window.userale.packageCustomLog({
            type: 'customEvent',
            customLabel: 'custom label',
            customField1: 'custom field',
        }, () => ({customDetails: Date.now()}), true);
    }
);
```

This event listener will now be invoked on the `customEvent` and send a custom log to the backend. Note that the
`packageCustomLog` function also adds the typical metadata used in the UserALE.js logs.

Note that we only advise adding custom event listeners in the following scenarios:

1. For events not captured natively by UserALE.js (as seen above)
2. For sending custom logs *only*
