/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.bigtable.grpc.io;

import com.google.api.client.util.Clock;
import com.google.api.core.InternalApi;
import com.google.cloud.bigtable.config.Logger;
import com.google.common.base.Preconditions;
import io.grpc.ClientCall;
import io.grpc.ForwardingClientCall;
import io.grpc.ForwardingClientCallListener;
import io.grpc.Metadata;
import io.grpc.Status;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import javax.annotation.concurrent.GuardedBy;

@InternalApi(value="For internal usage only")
public class Watchdog
implements Runnable {
    private static final Logger LOG = new Logger(Watchdog.class);
    private static final long DEFAULT_IDLE_TIMEOUT_MS = TimeUnit.MINUTES.toMillis(10L);
    private static final long MIN_CHECK_PERIOD_MS = TimeUnit.SECONDS.toMillis(10L);
    private static Object PRESENT = new Object();
    private final ConcurrentHashMap<WatchedCall<?, ?>, Object> openStreams = new ConcurrentHashMap();
    private final Clock clock;
    private final long waitTimeoutMs;
    private final long idleTimeoutMs;
    private ScheduledFuture<?> scheduledFuture;

    public Watchdog(Clock clock, long waitTimeoutMs) {
        this(clock, waitTimeoutMs, DEFAULT_IDLE_TIMEOUT_MS);
    }

    public Watchdog(Clock clock, long waitTimeoutMs, long idleTimeoutMs) {
        this.clock = (Clock)Preconditions.checkNotNull((Object)clock, (Object)"clock can't be null");
        this.waitTimeoutMs = waitTimeoutMs;
        this.idleTimeoutMs = idleTimeoutMs;
    }

    public <ReqT, RespT> ClientCall<ReqT, RespT> watch(ClientCall<ReqT, RespT> innerCall) {
        return new WatchedCall<ReqT, RespT>(innerCall);
    }

    @Override
    public void run() {
        try {
            this.runUnsafe();
        }
        catch (Throwable t) {
            LOG.error("Caught throwable in periodic Watchdog run. Continuing.", t, new Object[0]);
        }
    }

    private void runUnsafe() {
        Iterator<Map.Entry<WatchedCall<?, ?>, Object>> it = this.openStreams.entrySet().iterator();
        int count = 0;
        while (it.hasNext()) {
            WatchedCall<?, ?> stream = it.next().getKey();
            if (!((WatchedCall)stream).cancelIfStale()) continue;
            ++count;
            it.remove();
        }
        if (count > 0) {
            LOG.warn("Found %d stale streams and cancelled them", count);
        }
    }

    public void start(ScheduledExecutorService executor) {
        Preconditions.checkState((this.scheduledFuture == null ? 1 : 0) != 0, (Object)"Already started");
        long minTimeoutMs = Math.min(this.waitTimeoutMs, this.idleTimeoutMs);
        long checkPeriodMs = Math.max(minTimeoutMs / 2L, MIN_CHECK_PERIOD_MS);
        this.scheduledFuture = executor.scheduleAtFixedRate(this, checkPeriodMs, checkPeriodMs, TimeUnit.MILLISECONDS);
    }

    public void stop() {
        ScheduledFuture<?> tmp = this.scheduledFuture;
        this.scheduledFuture = null;
        if (tmp != null) {
            tmp.cancel(true);
        }
    }

    public static class StreamWaitTimeoutException
    extends RuntimeException {
        private final State state;
        private final long waitTimeMs;

        public StreamWaitTimeoutException(State state, long waitTimeMs) {
            this.state = state;
            this.waitTimeMs = waitTimeMs;
        }

        public State getState() {
            return this.state;
        }

        public long getWaitTimeMs() {
            return this.waitTimeMs;
        }
    }

    private class WatchedCall<ReqT, RespT>
    extends ForwardingClientCall.SimpleForwardingClientCall<ReqT, RespT> {
        private final Object lock;
        @GuardedBy(value="lock")
        private State state;
        @GuardedBy(value="lock")
        private int pendingCount;
        @GuardedBy(value="lock")
        private long lastActivityAt;

        WatchedCall(ClientCall<ReqT, RespT> delegate) {
            super(delegate);
            this.lock = new Object();
            this.state = State.NOT_STARTED;
            this.pendingCount = 0;
            this.lastActivityAt = Watchdog.this.clock.currentTimeMillis();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void start(ClientCall.Listener<RespT> listener, Metadata metadata) {
            ClientCall call = this.delegate();
            Object object = this.lock;
            synchronized (object) {
                Preconditions.checkState((this.state == State.NOT_STARTED ? 1 : 0) != 0, (Object)"Already started");
                this.setState(State.IDLE);
            }
            Watchdog.this.openStreams.put(this, PRESENT);
            call.start((ClientCall.Listener)new ForwardingClientCallListener.SimpleForwardingClientCallListener<RespT>(listener){

                /*
                 * WARNING - Removed try catching itself - possible behaviour change.
                 */
                public void onMessage(RespT message) {
                    WatchedCall.this.setState(State.DELIVERING);
                    try {
                        super.onMessage(message);
                    }
                    catch (Throwable throwable) {
                        Object object = WatchedCall.this.lock;
                        synchronized (object) {
                            WatchedCall.this.pendingCount--;
                            WatchedCall.this.setState(WatchedCall.this.pendingCount > 0 ? State.WAITING : State.IDLE);
                        }
                        throw throwable;
                    }
                    Object object = WatchedCall.this.lock;
                    synchronized (object) {
                        WatchedCall.this.pendingCount--;
                        WatchedCall.this.setState(WatchedCall.this.pendingCount > 0 ? State.WAITING : State.IDLE);
                    }
                }

                public void onClose(Status status, Metadata trailers) {
                    Watchdog.this.openStreams.remove((Object)WatchedCall.this);
                    super.onClose(status, trailers);
                }
            }, metadata);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void setState(State state) {
            Object object = this.lock;
            synchronized (object) {
                this.state = state;
                this.lastActivityAt = Watchdog.this.clock.currentTimeMillis();
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void request(int count) {
            Object object = this.lock;
            synchronized (object) {
                Preconditions.checkState((this.state != State.NOT_STARTED ? 1 : 0) != 0, (Object)"The Call was not started");
                int maxIncrement = Integer.MAX_VALUE - this.pendingCount;
                this.pendingCount += Math.min(maxIncrement, count);
                if (this.state == State.IDLE) {
                    this.setState(State.WAITING);
                }
            }
            super.request(count);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private boolean cancelIfStale() {
            Object object = this.lock;
            synchronized (object) {
                long waitTime = Watchdog.this.clock.currentTimeMillis() - this.lastActivityAt;
                switch (this.state) {
                    case NOT_STARTED: 
                    case IDLE: {
                        if (waitTime < Watchdog.this.idleTimeoutMs) break;
                        this.delegate().cancel("Canceled due to idle connection", (Throwable)new StreamWaitTimeoutException(this.state, waitTime));
                        return true;
                    }
                    case WAITING: {
                        if (waitTime < Watchdog.this.waitTimeoutMs) break;
                        this.delegate().cancel("Canceled due to timeout waiting for next response", (Throwable)new StreamWaitTimeoutException(this.state, waitTime));
                        return true;
                    }
                    case DELIVERING: {
                        break;
                    }
                    default: {
                        throw new IllegalStateException("Unknown state: " + (Object)((Object)this.state));
                    }
                }
                return false;
            }
        }
    }

    public static enum State {
        NOT_STARTED,
        IDLE,
        WAITING,
        DELIVERING;

    }
}

