/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.optaplanner.core.impl.domain.common.accessor.gizmo;

import java.util.HashMap;
import java.util.Map;

/**
 * Loads a class if we have the Gizmo-generated bytecode for it,
 * otherwise uses the current {@link Thread}'s context {@link ClassLoader}.
 * This implementation is thread-safe.
 */
public final class GizmoClassLoader extends ClassLoader {

    private final Map<String, byte[]> classNameToBytecodeMap = new HashMap<>();

    public GizmoClassLoader() {
        /*
         * As parent, Gizmo needs to use the same ClassLoader that loaded its own class.
         * Otherwise, issues will arise in Quarkus with MemberAccessors which were first loaded by Quarkus
         * and then loaded again by Gizmo, which uses the default parent ClassLoader.
         */
        super(GizmoClassLoader.class.getClassLoader());
    }

    @Override
    public String getName() {
        return "OptaPlanner Solver Gizmo ClassLoader";
    }

    @Override
    public Class<?> findClass(String name) throws ClassNotFoundException {
        byte[] byteCode = getBytecodeFor(name);
        if (byteCode == null) { // Not a Gizmo generated class; load from context class loader.
            return Thread.currentThread().getContextClassLoader().loadClass(name);
        } else { // Gizmo generated class.
            return defineClass(name, byteCode, 0, byteCode.length);
        }
    }

    public synchronized byte[] getBytecodeFor(String className) {
        return classNameToBytecodeMap.get(className);
    }

    public boolean hasBytecodeFor(String className) {
        return getBytecodeFor(className) != null;
    }

    public synchronized void storeBytecode(String className, byte[] bytecode) {
        classNameToBytecodeMap.put(className, bytecode);
    }

}
