/*
 * Decompiled with CFR 0.152.
 */
package org.apache.wiki.util;

import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.Base64;
import java.util.Random;

public final class CryptoUtil {
    private static final String SSHA = "{SSHA}";
    private static final String SHA1 = "{SHA-1}";
    private static final String SHA256 = "{SHA-256}";
    private static final Random RANDOM = new SecureRandom();
    private static final int DEFAULT_SALT_SIZE = 8;
    private static final Object HELP = "--help";
    private static final Object HASH = "--hash";
    private static final Object VERIFY = "--verify";

    private CryptoUtil() {
    }

    public static void main(String[] args) throws Exception {
        if (args.length == 0 || args.length == 1 && HELP.equals(args[0])) {
            System.out.println("Usage: CryptoUtil [options] ");
            System.out.println("   --hash   password algorithm             create hash for password");
            System.out.println("   --verify password digest algorithm      verify password for digest");
            System.out.println("Valid algorithm options are {SSHA} and {SHA-256}. If no algorithm is specified or an unsupported algorithm is specified, SHA-256 is used.");
        }
        if (HASH.equals(args[0])) {
            if (args.length < 2) {
                throw new IllegalArgumentException("Error: --hash requires a 'password' argument.");
            }
            String password = args[1].trim();
            String algorithm = args.length > 2 ? args[2].trim() : SHA256;
            System.out.println(CryptoUtil.getSaltedPassword(password.getBytes(StandardCharsets.UTF_8), algorithm));
        } else if (VERIFY.equals(args[0])) {
            if (args.length < 3) {
                throw new IllegalArgumentException("Error: --hash requires 'password' and 'digest' arguments.");
            }
            String password = args[1].trim();
            String digest = args[2].trim();
            System.out.println(CryptoUtil.verifySaltedPassword(password.getBytes(StandardCharsets.UTF_8), digest));
        } else {
            System.out.println("Wrong usage. Try --help.");
        }
    }

    public static String getSaltedPassword(byte[] password, String algorithm) throws NoSuchAlgorithmException {
        byte[] salt = new byte[8];
        RANDOM.nextBytes(salt);
        return CryptoUtil.getSaltedPassword(password, salt, algorithm);
    }

    static String getSaltedPassword(byte[] password, byte[] salt, String algorithm) throws NoSuchAlgorithmException {
        String algorithmToUse = algorithm.equals(SSHA) ? SHA1 : algorithm;
        MessageDigest digest = MessageDigest.getInstance(algorithmToUse.substring(1, algorithmToUse.length() - 1));
        digest.update(password);
        byte[] hash = digest.digest(salt);
        byte[] all = new byte[hash.length + salt.length];
        System.arraycopy(hash, 0, all, 0, hash.length);
        System.arraycopy(salt, 0, all, hash.length, salt.length);
        byte[] base64 = Base64.getEncoder().encode(all);
        return algorithm + new String(base64, StandardCharsets.UTF_8);
    }

    public static boolean verifySaltedPassword(byte[] password, String entry) throws NoSuchAlgorithmException {
        if (!entry.startsWith(SSHA) && !entry.startsWith(SHA256)) {
            throw new IllegalArgumentException("Hash not prefixed by expected algorithm; is it really a salted hash?");
        }
        String algorithm = entry.startsWith(SSHA) ? SSHA : SHA256;
        byte[] challenge = Base64.getDecoder().decode(entry.substring(algorithm.length()).getBytes(StandardCharsets.UTF_8));
        byte[] passwordHash = CryptoUtil.extractPasswordHash(challenge, algorithm.equals(SSHA) ? 20 : 32);
        byte[] salt = CryptoUtil.extractSalt(challenge, algorithm.equals(SSHA) ? 20 : 32);
        String algorithmToUse = algorithm.equals(SSHA) ? SHA1 : algorithm;
        MessageDigest digest = MessageDigest.getInstance(algorithmToUse.substring(1, algorithmToUse.length() - 1));
        digest.update(password);
        byte[] hash = digest.digest(salt);
        return MessageDigest.isEqual(passwordHash, hash);
    }

    static byte[] extractPasswordHash(byte[] digest, int hashLength) throws IllegalArgumentException {
        if (digest.length < hashLength) {
            throw new IllegalArgumentException("Hash was shorter than expected; could not extract password hash!");
        }
        byte[] hash = new byte[hashLength];
        System.arraycopy(digest, 0, hash, 0, hashLength);
        return hash;
    }

    static byte[] extractSalt(byte[] digest, int hashLength) throws IllegalArgumentException {
        if (digest.length <= hashLength) {
            throw new IllegalArgumentException("Hash was shorter than expected; we found no salt!");
        }
        byte[] salt = new byte[digest.length - hashLength];
        System.arraycopy(digest, hashLength, salt, 0, digest.length - hashLength);
        return salt;
    }
}

