/*
 * Decompiled with CFR 0.152.
 */
package org.apache.rocketmq.tieredstore.provider;

import java.nio.Buffer;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Semaphore;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.rocketmq.logging.org.slf4j.Logger;
import org.apache.rocketmq.logging.org.slf4j.LoggerFactory;
import org.apache.rocketmq.tieredstore.common.AppendResult;
import org.apache.rocketmq.tieredstore.common.FileSegmentType;
import org.apache.rocketmq.tieredstore.common.TieredMessageStoreConfig;
import org.apache.rocketmq.tieredstore.exception.TieredStoreErrorCode;
import org.apache.rocketmq.tieredstore.exception.TieredStoreException;
import org.apache.rocketmq.tieredstore.provider.TieredStoreProvider;
import org.apache.rocketmq.tieredstore.provider.stream.FileSegmentInputStream;
import org.apache.rocketmq.tieredstore.provider.stream.FileSegmentInputStreamFactory;
import org.apache.rocketmq.tieredstore.util.MessageBufferUtil;

public abstract class TieredFileSegment
implements Comparable<TieredFileSegment>,
TieredStoreProvider {
    private static final Logger logger = LoggerFactory.getLogger((String)"RocketmqTieredStore");
    protected final String filePath;
    protected final long baseOffset;
    protected final FileSegmentType fileType;
    protected final TieredMessageStoreConfig storeConfig;
    private final long maxSize;
    private final ReentrantLock bufferLock = new ReentrantLock();
    private final Semaphore commitLock = new Semaphore(1);
    private volatile boolean full = false;
    private volatile boolean closed = false;
    private volatile long minTimestamp = Long.MAX_VALUE;
    private volatile long maxTimestamp = Long.MAX_VALUE;
    private volatile long commitPosition = 0L;
    private volatile long appendPosition = 0L;
    private volatile long dispatchCommitOffset = 0L;
    private ByteBuffer codaBuffer;
    private List<ByteBuffer> bufferList = new ArrayList<ByteBuffer>();
    private FileSegmentInputStream fileSegmentInputStream;
    private CompletableFuture<Boolean> flightCommitRequest = CompletableFuture.completedFuture(false);

    public TieredFileSegment(TieredMessageStoreConfig storeConfig, FileSegmentType fileType, String filePath, long baseOffset) {
        this.storeConfig = storeConfig;
        this.fileType = fileType;
        this.filePath = filePath;
        this.baseOffset = baseOffset;
        this.maxSize = this.getMaxSizeByFileType();
    }

    protected long getMaxSizeByFileType() {
        switch (this.fileType) {
            case COMMIT_LOG: {
                return this.storeConfig.getTieredStoreCommitLogMaxSize();
            }
            case CONSUME_QUEUE: {
                return this.storeConfig.getTieredStoreConsumeQueueMaxSize();
            }
            case INDEX: {
                return Long.MAX_VALUE;
            }
        }
        throw new IllegalArgumentException("Unsupported file type: " + (Object)((Object)this.fileType));
    }

    @Override
    public int compareTo(TieredFileSegment o) {
        return Long.compare(this.baseOffset, o.baseOffset);
    }

    public long getBaseOffset() {
        return this.baseOffset;
    }

    public long getCommitOffset() {
        return this.baseOffset + this.commitPosition;
    }

    public long getCommitPosition() {
        return this.commitPosition;
    }

    public long getDispatchCommitOffset() {
        return this.dispatchCommitOffset;
    }

    public long getMaxOffset() {
        return this.baseOffset + this.appendPosition;
    }

    public long getMaxSize() {
        return this.maxSize;
    }

    public long getMinTimestamp() {
        return this.minTimestamp;
    }

    public void setMinTimestamp(long minTimestamp) {
        this.minTimestamp = minTimestamp;
    }

    public long getMaxTimestamp() {
        return this.maxTimestamp;
    }

    public void setMaxTimestamp(long maxTimestamp) {
        this.maxTimestamp = maxTimestamp;
    }

    public boolean isFull() {
        return this.full;
    }

    public void setFull() {
        this.setFull(true);
    }

    public void setFull(boolean appendCoda) {
        this.bufferLock.lock();
        try {
            this.full = true;
            if (this.fileType == FileSegmentType.COMMIT_LOG && appendCoda) {
                this.appendCoda();
            }
        }
        finally {
            this.bufferLock.unlock();
        }
    }

    public boolean isClosed() {
        return this.closed;
    }

    public void close() {
        this.closed = true;
    }

    public FileSegmentType getFileType() {
        return this.fileType;
    }

    public void initPosition(long pos) {
        this.commitPosition = pos;
        this.appendPosition = pos;
    }

    private List<ByteBuffer> borrowBuffer() {
        this.bufferLock.lock();
        try {
            List<ByteBuffer> tmp = this.bufferList;
            this.bufferList = new ArrayList<ByteBuffer>();
            List<ByteBuffer> list = tmp;
            return list;
        }
        finally {
            this.bufferLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public AppendResult append(ByteBuffer byteBuf, long timestamp) {
        if (this.closed) {
            return AppendResult.FILE_CLOSED;
        }
        this.bufferLock.lock();
        try {
            if (this.full || this.codaBuffer != null) {
                AppendResult appendResult = AppendResult.FILE_FULL;
                return appendResult;
            }
            if (this.fileType == FileSegmentType.INDEX) {
                this.minTimestamp = byteBuf.getLong(4);
                this.maxTimestamp = byteBuf.getLong(12);
                this.appendPosition += (long)byteBuf.remaining();
                this.bufferList.add(byteBuf);
                this.setFull();
                AppendResult appendResult = AppendResult.SUCCESS;
                return appendResult;
            }
            if (this.appendPosition + (long)byteBuf.remaining() > this.maxSize) {
                this.setFull();
                AppendResult appendResult = AppendResult.FILE_FULL;
                return appendResult;
            }
            if (this.bufferList.size() > this.storeConfig.getTieredStoreGroupCommitCount() || this.appendPosition - this.commitPosition > (long)this.storeConfig.getTieredStoreGroupCommitSize()) {
                this.commitAsync();
            }
            if (this.bufferList.size() > this.storeConfig.getTieredStoreMaxGroupCommitCount()) {
                logger.debug("File segment append buffer full, file: {}, buffer size: {}, pending bytes: {}", new Object[]{this.getPath(), this.bufferList.size(), this.appendPosition - this.commitPosition});
                AppendResult appendResult = AppendResult.BUFFER_FULL;
                return appendResult;
            }
            if (timestamp != Long.MAX_VALUE) {
                this.maxTimestamp = timestamp;
                if (this.minTimestamp == Long.MAX_VALUE) {
                    this.minTimestamp = timestamp;
                }
            }
            this.appendPosition += (long)byteBuf.remaining();
            ByteBuffer byteBuffer = ByteBuffer.allocateDirect(byteBuf.remaining());
            byteBuffer.put(byteBuf);
            byteBuffer.flip();
            byteBuf.rewind();
            this.bufferList.add(byteBuffer);
            AppendResult appendResult = AppendResult.SUCCESS;
            return appendResult;
        }
        finally {
            this.bufferLock.unlock();
        }
    }

    public void setCommitPosition(long commitPosition) {
        this.commitPosition = commitPosition;
    }

    public long getAppendPosition() {
        return this.appendPosition;
    }

    public void setAppendPosition(long appendPosition) {
        this.appendPosition = appendPosition;
    }

    private void appendCoda() {
        if (this.codaBuffer != null) {
            return;
        }
        this.codaBuffer = ByteBuffer.allocate(16);
        this.codaBuffer.putInt(16);
        this.codaBuffer.putInt(-875286124);
        this.codaBuffer.putLong(this.maxTimestamp);
        this.codaBuffer.flip();
        this.appendPosition += 16L;
    }

    public ByteBuffer read(long position, int length) {
        return this.readAsync(position, length).join();
    }

    public CompletableFuture<ByteBuffer> readAsync(long position, int length) {
        CompletableFuture<ByteBuffer> future = new CompletableFuture<ByteBuffer>();
        if (position < 0L || length < 0) {
            future.completeExceptionally(new TieredStoreException(TieredStoreErrorCode.ILLEGAL_PARAM, "position or length is negative"));
            return future;
        }
        if (length == 0) {
            future.completeExceptionally(new TieredStoreException(TieredStoreErrorCode.ILLEGAL_PARAM, "length is zero"));
            return future;
        }
        if (position >= this.commitPosition) {
            future.completeExceptionally(new TieredStoreException(TieredStoreErrorCode.ILLEGAL_PARAM, "position is illegal"));
            return future;
        }
        if (position + (long)length > this.commitPosition) {
            logger.warn("TieredFileSegment#readAsync request position + length is greater than commit position, correct length using commit position, file: {}, request position: {}, commit position:{}, change length from {} to {}", new Object[]{this.getPath(), position, this.commitPosition, length, this.commitPosition - position});
            length = (int)(this.commitPosition - position);
            if (length == 0) {
                future.completeExceptionally(new TieredStoreException(TieredStoreErrorCode.NO_NEW_DATA, "request position is equal to commit position"));
                return future;
            }
            if (this.fileType == FileSegmentType.CONSUME_QUEUE && length % 20 != 0) {
                future.completeExceptionally(new TieredStoreException(TieredStoreErrorCode.ILLEGAL_PARAM, "position and length is illegal"));
                return future;
            }
        }
        return this.read0(position, length);
    }

    public boolean needCommit() {
        return this.appendPosition > this.commitPosition;
    }

    public boolean commit() {
        if (this.closed) {
            return false;
        }
        Boolean result = this.commitAsync().join();
        if (!result.booleanValue()) {
            result = this.flightCommitRequest.join();
        }
        return result;
    }

    private void releaseCommitLock() {
        if (this.commitLock.availablePermits() == 0) {
            this.commitLock.release();
        } else {
            logger.error("[Bug] FileSegmentCommitAsync, lock is already released: available permits: {}", (Object)this.commitLock.availablePermits());
        }
    }

    private void updateDispatchCommitOffset(List<ByteBuffer> bufferList) {
        if (this.fileType == FileSegmentType.COMMIT_LOG && bufferList.size() > 0) {
            this.dispatchCommitOffset = MessageBufferUtil.getQueueOffset(bufferList.get(bufferList.size() - 1));
        }
    }

    public CompletableFuture<Boolean> commitAsync() {
        if (this.closed) {
            return CompletableFuture.completedFuture(false);
        }
        if (!this.needCommit()) {
            return CompletableFuture.completedFuture(true);
        }
        if (this.commitLock.drainPermits() <= 0) {
            return CompletableFuture.completedFuture(false);
        }
        try {
            int bufferSize;
            if (this.fileSegmentInputStream != null) {
                long fileSize = this.getSize();
                if (fileSize == -1L) {
                    logger.error("Get commit position error before commit, Commit: %d, Expect: %d, Current Max: %d, FileName: %s", new Object[]{this.commitPosition, this.commitPosition + (long)this.fileSegmentInputStream.getContentLength(), this.appendPosition, this.getPath()});
                    this.releaseCommitLock();
                    return CompletableFuture.completedFuture(false);
                }
                if (this.correctPosition(fileSize, null)) {
                    this.updateDispatchCommitOffset(this.fileSegmentInputStream.getBufferList());
                    this.fileSegmentInputStream = null;
                }
            }
            if (this.fileSegmentInputStream != null) {
                bufferSize = this.fileSegmentInputStream.available();
            } else {
                List<ByteBuffer> bufferList = this.borrowBuffer();
                bufferSize = bufferList.stream().mapToInt(Buffer::remaining).sum() + (this.codaBuffer != null ? this.codaBuffer.remaining() : 0);
                if (bufferSize == 0) {
                    this.releaseCommitLock();
                    return CompletableFuture.completedFuture(true);
                }
                this.fileSegmentInputStream = FileSegmentInputStreamFactory.build(this.fileType, this.baseOffset + this.commitPosition, bufferList, this.codaBuffer, bufferSize);
            }
            this.flightCommitRequest = ((CompletableFuture)((CompletableFuture)this.commit0(this.fileSegmentInputStream, this.commitPosition, bufferSize, this.fileType != FileSegmentType.INDEX).thenApply(result -> {
                if (result.booleanValue()) {
                    this.updateDispatchCommitOffset(this.fileSegmentInputStream.getBufferList());
                    this.commitPosition += (long)bufferSize;
                    this.fileSegmentInputStream = null;
                    return true;
                }
                this.fileSegmentInputStream.rewind();
                return false;
            })).exceptionally(this::handleCommitException)).whenComplete((result, e) -> this.releaseCommitLock());
            return this.flightCommitRequest;
        }
        catch (Exception e2) {
            this.handleCommitException(e2);
            this.releaseCommitLock();
            return CompletableFuture.completedFuture(false);
        }
    }

    private long getCorrectFileSize(Throwable throwable) {
        long fileSize;
        if (throwable instanceof TieredStoreException && (fileSize = ((TieredStoreException)throwable).getPosition()) > 0L) {
            return fileSize;
        }
        return this.getSize();
    }

    private boolean handleCommitException(Throwable e) {
        Throwable cause = e.getCause() != null ? e.getCause() : e;
        long fileSize = this.getCorrectFileSize(cause);
        if (fileSize == -1L) {
            logger.error("Get commit position error, Commit: %d, Expect: %d, Current Max: %d, FileName: %s", new Object[]{this.commitPosition, this.commitPosition + (long)this.fileSegmentInputStream.getContentLength(), this.appendPosition, this.getPath()});
            this.fileSegmentInputStream.rewind();
            return false;
        }
        if (this.correctPosition(fileSize, cause)) {
            this.updateDispatchCommitOffset(this.fileSegmentInputStream.getBufferList());
            this.fileSegmentInputStream = null;
            return true;
        }
        this.fileSegmentInputStream.rewind();
        return false;
    }

    private boolean correctPosition(long fileSize, Throwable throwable) {
        String handleInfo = throwable == null ? "before commit" : "after commit";
        long expectPosition = this.commitPosition + (long)this.fileSegmentInputStream.getContentLength();
        String offsetInfo = String.format("Correct Commit Position, %s, result=[{}], Commit: %d, Expect: %d, Current Max: %d, FileSize: %d, FileName: %s", handleInfo, this.commitPosition, expectPosition, this.appendPosition, fileSize, this.getPath());
        if (fileSize == expectPosition) {
            logger.info(offsetInfo, (Object)"Success", (Object)throwable);
            this.commitPosition = fileSize;
            return true;
        }
        if (fileSize < this.commitPosition) {
            logger.error(offsetInfo, (Object)"FileSizeIncorrect", (Object)throwable);
        } else if (fileSize == this.commitPosition) {
            logger.warn(offsetInfo, (Object)"CommitFailed", (Object)throwable);
        } else if (fileSize > this.commitPosition) {
            logger.warn(offsetInfo, (Object)"PartialSuccess", (Object)throwable);
        }
        this.commitPosition = fileSize;
        return false;
    }
}

