##
# Procedural model of an ASG29 electrical system.  Includes a
# preliminary battery charge/discharge model and realistic ammeter
# gauge modeling.
#
# Based on C172P electrical system.


##
# Initialize internal values
#

var vbus_volts = 0.0;
var ebus1_volts = 0.0;
var ebus2_volts = 0.0;

var ammeter_ave = 0.0;
##
# Battery model class.
#

var BatteryClass = {};

BatteryClass.new = func {
    var obj = { parents : [BatteryClass],
                ideal_volts : 24.0,
                ideal_amps : 30.0,
                amp_hours : 3.1875,
                charge_percent : getprop("/systems/electrical/battery-charge-percent") or 1.0,
                charge_amps : 7.0 };
    setprop("/systems/electrical/battery-charge-percent", obj.charge_percent);
    return obj;
}

##
# Passing in positive amps means the battery will be discharged.
# Negative amps indicates a battery charge.
#

BatteryClass.apply_load = func (amps, dt) {
    var old_charge_percent = getprop("/systems/electrical/battery-charge-percent");

    if (getprop("/sim/freeze/replay-state"))
        return me.amp_hours * old_charge_percent;

    var amphrs_used = amps * dt / 3600.0;
    var percent_used = amphrs_used / me.amp_hours;

    var new_charge_percent = std.max(0.0, std.min(old_charge_percent - percent_used, 1.0));

    if (new_charge_percent < 0.1 and old_charge_percent >= 0.1)
        gui.popupTip("Warning: Low battery! Enable alternator or apply external power to recharge battery!", 10);
    me.charge_percent = new_charge_percent;
    setprop("/systems/electrical/battery-charge-percent", new_charge_percent);
    return me.amp_hours * new_charge_percent;
}

##
# Return output volts based on percent charged.  Currently based on a simple
# polynomial percent charge vs. volts function.
#

BatteryClass.get_output_volts = func {
    var x = 1.0 - me.charge_percent;
    var tmp = -(3.0 * x - 1.0);
    var factor = (tmp*tmp*tmp*tmp*tmp + 32) / 32;
    return me.ideal_volts * factor;
}


##
# Return output amps available.  This function is totally wrong and should be
# fixed at some point with a more sensible function based on charge percent.
# There is probably some physical limits to the number of instantaneous amps
# a battery can produce (cold cranking amps?)
#

BatteryClass.get_output_amps = func {
    var x = 1.0 - me.charge_percent;
    var tmp = -(3.0 * x - 1.0);
    var factor = (tmp*tmp*tmp*tmp*tmp + 32) / 32;
    return me.ideal_amps * factor;
}

##
# Set the current charge instantly to 100 %.
#

BatteryClass.reset_to_full_charge = func {
    me.apply_load(-(1.0 - me.charge_percent) * me.amp_hours, 3600);
}


var battery1 = BatteryClass.new();
var battery2 = BatteryClass.new();
var battery3 = BatteryClass.new();
var battery4 = BatteryClass.new();

var reset_circuit_breakers = func {
	# Reset circuit breakers
    setprop("/controls/circuit-breakers/master", 1);
    setprop("/controls/circuit-breakers/vario", 1);
    setprop("/controls/circuit-breakers/radio", 1);
    setprop("/controls/circuit-breakers/turn", 1);
}

var recharge_battery = func {
    # Charge battery to 100 %
        battery1.reset_to_full_charge();
        battery2.reset_to_full_charge();
        battery3.reset_to_full_charge();
        battery4.reset_to_full_charge();
}
##
# This is the main electrical system update function.
#

var ElectricalSystemUpdater = {
    new : func {
        var m = {
            parents: [ElectricalSystemUpdater]
        };
        # Request that the update function be called each frame
        m.loop = updateloop.UpdateLoop.new(components: [m], update_period: 0.0, enable: 0);
        return m;
    },

    enable: func {
        me.loop.reset();
        me.loop.enable();
    },

    disable: func {
        me.loop.disable();
    },

    reset: func {
        # Do nothing
    },

    update: func (dt) {
        update_virtual_bus(dt);
    }
};

##
# Model the system of relays and connections that join the battery,master/alt switches.
#

var update_virtual_bus = func (dt) {
    var serviceable = getprop("/systems/electrical/serviceable");
    var load = 0.0;
    var battery_volts = 0.0;
    if ( serviceable ) {
        battery1_volts = battery1.get_output_volts();
        battery2_volts = battery2.get_output_volts();
        battery3_volts = battery3.get_output_volts();
        battery4_volts = battery4.get_output_volts();
    }

    # switch state
    var vario_knob=getprop("/controls/electric/vario-source");
    var radio_knob=getprop("/controls/electric/radio-source");
    var turn_knob=getprop("/controls/electric/turn-source");

    
    # determine power source
    if ( getprop("/controls/circuit-breakers/vario")){
	if(vario_knob==1){
		var vario_volts = battery1_volts;
		battery1.apply_load(battery1_volts/57, dt);
	}else if(vario_knob==2){
		var vario_volts = battery2_volts;
		battery2.apply_load(battery2_volts/57, dt);
	}else if(vario_knob==3){
		var vario_volts = battery3_volts;
		battery3.apply_load(battery3_volts/57, dt);
	}else if(vario_knob==4){
		var vario_volts = battery4_volts;
		battery4.apply_load(battery4_volts/57, dt);
	}else{
		var vario_volts = 0.0;
	}
    }else{
		var vario_volts = 0.0;
	}
    
    setprop("systems/electrical/outputs/vario", vario_volts);
    
    
    if ( getprop("/controls/circuit-breakers/radio")){
	if(radio_knob==1){
		var radio_volts = battery1_volts;
		battery1.apply_load(battery1_volts/57, dt);
	}else if(radio_knob==2){
		var radio_volts = battery2_volts;
		battery2.apply_load(battery2_volts/57, dt);
	}else if(radio_knob==3){
		var radio_volts = battery3_volts;
		battery3.apply_load(battery3_volts/57, dt);
	}else if(radio_knob==4){
		var radio_volts = battery4_volts;
		battery4.apply_load(battery4_volts/57, dt);
	}else{
		var radio_volts = 0.0;
	}
    }else{
		var radio_volts = 0.0;
	}
    
    setprop("systems/electrical/outputs/radio", radio_volts);
    
    
    
    
    if ( getprop("/controls/circuit-breakers/turn")){
	if(turn_knob==1){
		var turn_volts = battery1_volts;
		battery1.apply_load(battery1_volts/57, dt);
	}else if(turn_knob==2){
		var turn_volts = battery2_volts;
		battery2.apply_load(battery2_volts/57, dt);
	}else if(turn_knob==3){
		var turn_volts = battery3_volts;
		battery3.apply_load(battery3_volts/57, dt);
	}else if(turn_knob==4){
		var turn_volts = battery4_volts;
		battery4.apply_load(battery4_volts/57, dt);
	}else{
		var turn_volts = 0.0;
	}
    }else{
		var turn_volts = 0.0;
	}
    
    setprop("systems/electrical/outputs/turn", turn_volts);
    setprop("systems/electrical/outputs/turn-coordinator", turn_volts);
    
    #print( "virtual bus volts = ", bus_volts );

    # bus network (1. these must be called in the right order, 2. the
    # bus routine itself determins where it draws power from.)
    
    # swtich the master breaker off if load is out of limits
   # if ( load > 55 ) {
  #    bus_volts = 0;
  #  }

    # system loads and ammeter gauge
   # var ammeter = 0.0;
   # if ( bus_volts > 1.0 ) {
        # ammeter gauge
      #  if ( power_source == "battery" ) {
      #      ammeter = -load;
      #  } else {
      #      ammeter = battery.charge_amps;
      #  }
    #}
    # print( "ammeter = ", ammeter );


    # filter ammeter needle pos
   # ammeter_ave = 0.8 * ammeter_ave + 0.2 * ammeter;

    # outputs
#    setprop("/systems/electrical/amps", ammeter_ave);
#    setprop("/systems/electrical/volts", bus_volts);

    return load;
}


##
# Initialize the electrical system
#

var system_updater = ElectricalSystemUpdater.new();

setlistener("/sim/signals/fdm-initialized", func {
    # checking if battery should be automatically recharged
    if (!getprop("/systems/electrical/save-battery-charge")) {
        battery1.reset_to_full_charge();
        battery2.reset_to_full_charge();
        battery3.reset_to_full_charge();
        battery4.reset_to_full_charge();
    };

    system_updater.enable();
    print("Electrical system initialized");
});
