##
# Electric aircraft electrical system model
#

#Motor System: 
# max voltage: 398V, min voltage: 288V

##
# Buses:
# Avionics Bus (DC)
# Engine Bus (AC)
#

var dc_volts = 0.0;
var ac_volts = 0.0;

var dc_bus_volts = 0.0;
var eng_bus_volts = 0.0;

# Switches
var switches = {
	master:		props.globals.getNode("/controls/switches/master-bat", 1),
	avionics:	props.globals.getNode("/controls/switches/avionics",   1),
	batt_en:	props.globals.getNode("/controls/switches/batt-en",    1),
	pwr_en:		props.globals.getNode("/controls/switches/pwr-en",     1),
	#		Only Alpha Electro
	socket:		props.globals.getNode("/controls/switches/socket",     1),
	light:	{
		landing:	props.globals.getNode("/controls/lighting/landing-lights",	1),
		navac:		props.globals.getNode("/controls/lighting/nav-ac-lights",	1),
		cockpit:	props.globals.getNode("/controls/lighting/cockpit-light",	1),
	},
};

var ann_selftest = props.globals.getNode("/instrumentation/annunciator/selftest", 1);
var ann_selftest_int = props.globals.getNode("/instrumentation/annunciator/selftest-int", 1);
var ann_batt_front = props.globals.getNode("//instrumentation/annunciator/batt-front-int", 1);
var ann_batt_rear  = props.globals.getNode("//instrumentation/annunciator/batt-rear-int",  1);

var cb_path = props.globals.initNode("/controls/circuit-breakers");

var cb = {
	com:		cb_path.initNode("com", 	1, "BOOL"),
	gps:		cb_path.initNode("gps", 	1, "BOOL"),
	xpdr:		cb_path.initNode("xpdr", 	1, "BOOL"),
	instr:		cb_path.initNode("instr", 	1, "BOOL"),
	eis:		cb_path.initNode("eis", 	1, "BOOL"),	# "EPSI" on Velis Electro
	trim:		cb_path.initNode("trim",	1, "BOOL"),
	pump:		cb_path.initNode("pump",	1, "BOOL"),	# "INV PUMP" on Velis Electro
	pwrctrl:	cb_path.initNode("pwrctrl",	1, "BOOL"),
	mainsys:	cb_path.initNode("mainsys",	1, "BOOL"),	# "SYS CTRL" on Velis Electro
	chgr:		cb_path.initNode("chgr",	1, "BOOL"),	# "CHG" on Velis Electro
	batt_f:		cb_path.initNode("batt-f",	1, "BOOL"),	# "BATT FRONT" on Velis Electro
	batt_r:		cb_path.initNode("batt-r",	1, "BOOL"),	# "BATT REAR" on Velis Electro
	#		Only Velis Electro
	batt_fan:	cb_path.initNode("batt-fan",	1, "BOOL"),
	batt_pump_f:	cb_path.initNode("batt-pump-f",	1, "BOOL"),
	batt_pump_r:	cb_path.initNode("batt-pump-r",	1, "BOOL"),
};

var variant = 0;	# set on initialization; 0 = Alpha Electro; 1 = Velis Electro
	
#var ammeter_ave = 0.0;

# Lights
var anti_coll_lights = aircraft.light.new( "/systems/electrical/outputs/ac-lights-norm", [0.05, 0.8], "/systems/electrical/outputs/ac-lights-bool" );

##
# Battery model class.
#

var BatteryClass = {};

BatteryClass.new = func (volts, amps, amp_hrs, name) {
	var obj = { parents : [BatteryClass],
		ideal_volts : volts,
		ideal_amps : amps,
		amp_hours : amp_hrs,
		name : name,
		charge_percent : props.globals.initNode("/systems/electrical/battery-charge-percent-"~name, 0.0, "DOUBLE"),
		charge_amps : 7.0
	};
	return obj;
}

##
# Passing in positive amps means the battery will be discharged.
# Negative amps indicates a battery charge.
#

BatteryClass.apply_load = func (amps, dt) {
	var old_charge_percent = me.charge_percent.getDoubleValue();
	
	if (getprop("/sim/freeze/replay-state"))
		return me.amp_hours * old_charge_percent;
	
	var amphrs_used = amps * dt / 3600.0;
	var percent_used = amphrs_used / me.amp_hours;
	
	var new_charge_percent = std.max(0.0, std.min(old_charge_percent - percent_used, 1.0));
	
	if (new_charge_percent < 0.1 and old_charge_percent >= 0.1)
		gui.popupTip("Warning: Low battery! Apply external power to recharge battery!", 10);
	me.charge_percent.setDoubleValue( new_charge_percent );
	return me.amp_hours * new_charge_percent;
}

##
# Return output volts based on percent charged.  Currently based on a simple
# polynomial percent charge vs. volts function.
#

BatteryClass.get_output_volts = func {
	var x = 1.0 - me.charge_percent.getDoubleValue();
	var tmp = -(3.0 * x - 1.0);
	var factor = (tmp*tmp*tmp*tmp*tmp + 32) / 32;
	return me.ideal_volts * factor;
}


##
# Return output amps available.  This function is totally wrong and should be
# fixed at some point with a more sensible function based on charge percent.
# There is probably some physical limits to the number of instantaneous amps
# a battery can produce (cold cranking amps?)
#

BatteryClass.get_output_amps = func {
	var x = 1.0 - me.charge_percent.getDoubleValue();
	var tmp = -(3.0 * x - 1.0);
	var factor = (tmp*tmp*tmp*tmp*tmp + 32) / 32;
	return me.ideal_amps * factor;
}

##
# Set the current charge instantly to 100 %.
#

BatteryClass.reset_to_full_charge = func {
	me.apply_load(-(1.0 - me.charge_percent.getDoubleValue() ) * me.amp_hours, 3600);
}


var battery_12v = BatteryClass.new(12, 30.0, 50,"12v"); #Todo find out 12V Batt capacity
var battery_front = BatteryClass.new(350, 87.5, 70,"front");
var battery_rear = BatteryClass.new(350, 87.5, 70,"back");

var reset_battery_and_circuit_breakers = func {
	# Charge battery to 100 %
	battery_12v.reset_to_full_charge();
	battery_front.reset_to_full_charge();
	battery_rear.reset_to_full_charge();
	
	reset_circuit_breakers();
}

var reset_circuit_breakers = func {
	foreach( var i; keys(cb) ){
		cb[i].setBoolValue( 1 );
	}
}

##
# This is the main electrical system update function.
#

var ElectricalSystemUpdater = {
	new : func {
		var m = {
			parents: [ElectricalSystemUpdater]
		};
		# Request that the update function be called each frame
		m.loop = updateloop.UpdateLoop.new(components: [m], update_period: 0.0, enable: 0);
		return m;
	},
	
	enable: func {
		me.loop.reset();
		me.loop.enable();
	},
	
	disable: func {
		me.loop.disable();
	},
	
	reset: func {
		# Do nothing
	},
	
	update: func (dt) {
		update_dc_bus(dt);
		update_ac_bus(dt);
	}
};


var update_dc_bus = func (dt) {
	var serviceable = getprop("/systems/electrical/serviceable");
	var external_volts = 0.0;
	var load = 0.0;
	var battery_12v_volts = 0.0;
	if ( serviceable ) {
		battery_12v_volts = battery_12v.get_output_volts();
	}
	
	# switch state
	if (getprop("/controls/electric/charger"))
	{
		external_volts = 28;
	}
	
	# determine power source
	var bus_volts = 0.0;
	var power_source = nil;
	if ( switches.master.getBoolValue() ) {
		bus_volts = battery_12v_volts;
		power_source = "battery";
	}
	if ( external_volts > bus_volts ) {
		bus_volts = external_volts;
		power_source = "external";
	}
	#print( "virtual bus volts = ", bus_volts );
	
	# bus network (1. these must be called in the right order, 2. the
	# bus routine itself determins where it draws power from.)
	load += electrical_bus_1();
	load += avionics_bus_1();
	
	# swtich the master breaker off if load is out of limits
	if ( load > 55 ) {
		bus_volts = 0;
	}
	
	
	
	# charge/discharge the battery
	if ( power_source == "battery" ) {
		battery_12v.apply_load( load, dt );
	} elsif ( bus_volts > battery_12v_volts ) {
		battery_12v.apply_load( -battery.charge_amps, dt );
	}
	
	
	# outputs
	#setprop("/systems/electrical/dc-amps", ammeter_ave);
	setprop("/systems/electrical/dc-volts", bus_volts);
	if (bus_volts > 8)
		dc_bus_volts = bus_volts;
	else
		dc_bus_volts = 0.0;
	
	return load;
}

var update_ac_bus = func (dt) {
	var serviceable = getprop("/systems/electrical/ac-serviceable");
	var inverter_serviceable = getprop("/systems/electrical/inverter-serviceable");
	var load = 0.0;
	var battery_front_volts = 0.0;
	var battery_rear_volts = 0.0;
	if ( serviceable ) {
		battery_front_volts = battery_front.get_output_volts();
		battery_rear_volts = battery_rear.get_output_volts();
	}
	
	# determine power source
	var bus_volts = 0.0;
	var power_source = nil;
	if ( switches.batt_en.getBoolValue() ) {
		if(battery_front_volts > 0){
			bus_volts = battery_front_volts;
			power_source = "battery_front";
		}
		if(battery_rear_volts > 0){
			if(bus_volts > 0){
				bus_volts += battery_rear_volts;
				power_source = "both_battery_packs";
			}else{
				bus_volts = battery_rear_volts;
				power_source = "battery_rear";
			}
		}
	}
	#print( "virtual bus volts = ", bus_volts );
	
	# bus network (1. these must be called in the right order, 2. the
	# bus routine itself determins where it draws power from.)
	load += engine_bus_1(bus_volts, power_source);
	
	# swtich the master breaker off if load is out of limits
	if ( load > 150 ) {
		bus_volts = 0;
	}
	
	# Warning if one or more batteries is offline
	if ( bus_volts > 15 ) {
		if ( power_source == "battery_front" ){
			
		} else if ( power_source == "battery_rear" ){
			
		} else {
			
		}
	} else {
		
	}
	
	
	# charge/discharge the battery
	if ( power_source == "battery_front" ) {
		battery_front.apply_load( load, dt );
	}else if ( power_source == "battery_rear" ) {
		battery_rear.apply_load( load, dt );
	}else if (power_source == "both_battery_packs" ) {
		battery_front.apply_load( 0.5*load, dt );
		battery_rear.apply_load( 0.5*load, dt );
	}
	
	
	var amps = load;
	
	# outputs
	setprop("/systems/electrical/ac-amps", amps or 0);
	setprop("/systems/electrical/ac-volts", bus_volts);
	if (bus_volts > 200)
		eng_bus_volts = bus_volts;
	else
		eng_bus_volts = 0.0;
	
	return load;
}

var engine_bus_1 = func(bv, src) {
	var bus_volts = 0.0;
	var load = 0.0;
	
	# check master breaker
 	if ( cb.pwrctrl.getBoolValue() ) {
		# we are feed from the engine bus
		bus_volts = bv;
	}
	#print("Eng Bus volts: ", bus_volts);
	
	#Control Engine Throttle
	if( switches.pwr_en.getBoolValue() ){
		var throttle = getprop("controls/engines/engine[0]/throttle");
	}else{
		var throttle = 0.0;    
	}
	
	setprop("/systems/electrical/engine-volts", bus_volts);
	if ( src == "both_battery_packs" ) {
		setprop("/systems/electrical/engine-limit", 1.0);
	} else {
		setprop("/systems/electrical/engine-limit", 0.5833);
	}
	
	#Load is engine power divided by volts
	var eng_power = getprop("fdm/jsbsim/propulsion/engine[0]/power-hp") * 745.7 * 1.1236 * 0.25; #hp to watts * ( 1 / motor efficiency) * calibration factor to achieve 1h endurance at 20kW (useful capacity 20kWh)
	if(getprop("/engines/engine[0]/rpm")>50){
		load += eng_power / bus_volts;   #P=U*I -> I=P/U
	}
	
	# register bus voltage
	eng_bus1_volts = bus_volts;
	
	# return cumulative load
	return load;
}

var electrical_bus_1 = func() {
	var bus_volts = 0.0;
	var load = 0.0;
	
	# check master breaker
	if ( cb.mainsys.getBoolValue() ) {
		# we are feed from the virtual bus
		bus_volts = dc_bus_volts;
	}
	#print("Bus volts: ", bus_volts);
	
	if( bus_volts > 10 and ann_selftest_int.getBoolValue() ){
		ann_selftest.setBoolValue( 1 );
	} else if ( ann_selftest.getBoolValue() ){
		ann_selftest.setBoolValue( 0 );
	}
	
	# Trim Power
	if ( cb.trim.getBoolValue() ) {
		setprop("/systems/electrical/outputs/trim", bus_volts);
		load += bus_volts / 28;
	} else {
		setprop("/systems/electrical/outputs/trim", 0.0);
	}
	
	
	# Engine Information System Power
	if ( cb.trim.getBoolValue() ) {
		setprop("/systems/electrical/outputs/eis", bus_volts);
	} else {
		setprop("/systems/electrical/outputs/eis", 0.0);
	}
	
	# variant-specific consumers
	
	if( variant == 0 ){
		# Socket Power
		if ( switches.socket.getBoolValue() ) {
			setprop("systems/electrical/outputs/socket", bus_volts);
			load += bus_volts / 57;
		} else {
			setprop("systems/electrical/outputs/socket", 0.0);
		}
		
		# Cockpit lights
		if ( switches.light.cockpit.getBoolValue() ) {
			setprop("/systems/electrical/outputs/cockpit-light", bus_volts);
			load += bus_volts / 57;
		} else {
			setprop("/systems/electrical/outputs/cockpit-light", 0.0);
		}
		
		# Landing Light Power
		if ( switches.light.landing.getBoolValue() ) {
			setprop("/systems/electrical/outputs/landing-lights", bus_volts);
			load += bus_volts / 5;
		} else {
			setprop("/systems/electrical/outputs/landing-lights", 0.0 );
		}
		
		
		# Beacon Power
		if ( switches.light.navac.getBoolValue() ) {
			setprop("/systems/electrical/outputs/nav-ac-lights", bus_volts);
			if( bus_volts > 10 ) {
				setprop("/systems/electrical/outputs/nav-lights-norm", 1.0);
				setprop("/systems/electrical/outputs/ac-lights-bool", 1);
			} else {
				setprop("/systems/electrical/outputs/nav-lights-norm", 0.0);
				setprop("/systems/electrical/outputs/ac-lights-bool", 0);
			}
			load += bus_volts / 28;
		} else {
			setprop("/systems/electrical/outputs/nav-ac-lights", 0.0);
			setprop("/systems/electrical/outputs/nav-lights-norm", 0.0);
			setprop("/systems/electrical/outputs/ac-lights-bool", 0);
		}
	}
	
	# register bus voltage
	dc_volts = bus_volts;
	
	# return cumulative load
	return load;
}

var avionics_bus_1 = func() {
	var bus_volts = 0.0;
	var load = 0.0;
	
	# we are fed from the electrical bus 1
	var master_av = getprop("/controls/switches/master-avionics");
	if ( master_av ) {
		bus_volts = dc_volts;
	}
	
	load += bus_volts / 20.0;
	
	
	# Com and Nav Power
	if ( cb.com.getBoolValue() ) {
		setprop("systems/electrical/outputs/comm[0]", bus_volts);
	} else {
		setprop("systems/electrical/outputs/comm[0]", 0.0);
	}
	
	# GPS Power
	if ( cb.gps.getBoolValue() ) {
		setprop("systems/electrical/outputs/gps", bus_volts);
	} else {
		setprop("systems/electrical/outputs/gps", 0.0);
	}
	
	# XPDR Power
	if ( cb.xpdr.getBoolValue() ) {
		setprop("/systems/electrical/outputs/transponder", bus_volts);
	} else {
		setprop("/systems/electrical/outputs/transponder", 0.0);
	}
	
	# Other Instruments Power
	if ( cb.instr.getBoolValue() ) {
		setprop("/systems/electrical/outputs/instruments", bus_volts);
	} else {
		setprop("/systems/electrical/outputs/instruments", 0.0);
	}
	
	
	# return cumulative load
	return load;
}

##
# Initialize the electrical system
#

var system_updater = ElectricalSystemUpdater.new();

var el_ls = setlistener("/sim/signals/fdm-initialized", func {
	# Check variant
	if( getprop("/sim/aero") == "alphaelectro-jsb" ){
		variant = 0;
	} else if ( getprop("/sim/aero") == "veliselectro-jsb" ){
		variant = 1;
	} else {
		die( "Electrical system: FATAL: Unknown variant!: "~ getprop("/sim/aero") );
	}
	
	# checking if battery should be automatically recharged
	if (!getprop("/systems/electrical/save-battery-charge")) {
		battery_12v.reset_to_full_charge();
		battery_front.reset_to_full_charge();
		battery_rear.reset_to_full_charge();
	};
	reset_circuit_breakers();
	
	system_updater.enable();
	
	removelistener( el_ls );
	print("Electrical system initialized");
});

##
# Extra functions for the "seen zero" safety feature
#
var throttle_listener = nil;
var throttle_en = props.globals.getNode("fdm/jsbsim/fcs/epsi570/enable-throttle", 1);
var throttle_zero = props.globals.getNode("/fdm/jsbsim/fcs/epsi570/throttle-at-zero");

var enable_throttle = func () {
	if(throttle_zero.getValue() == 1){
		throttle_en.setValue( 1 * switches.pwr_en.getBoolValue() );
		if( throttle_listener != nil ){
			removelistener(throttle_listener);
		}
	}
}
var disable_throttle = func () {
	throttle_en.setValue( 0 );
}

var power_enable_button = func( i )  {
	if( i == 1 ){
		throttle_listener = setlistener(throttle_zero, func{ if( throttle_zero.getValue() == 1 ) { enable_throttle(); } } );
	} else {
		disable_throttle();
	}
}

setlistener(switches.pwr_en, func (i) {
	power_enable_button(i.getValue());
});

#	Annunciator Selftest: Activated when enabling the master switch (only on the Velis Electro)
setlistener(switches.master, func {
	if( variant == 1 ){
		if( switches.master.getBoolValue() ){
			ann_selftest_int.setBoolValue( 1 );
			settimer( func { ann_selftest_int.setBoolValue( 0 ) }, 2.2 );
		}
	}
});
