#    This file is part of the DG-505 simulation for the FlightGear Flight Simulator (FGFS)
#
#    dg505 is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 2 of the License, or
#    (at your option) any later version.
#
#    dg505 is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with dg505.  If not, see <http://www.gnu.org/licenses/>.
#
#	Authors:	Benedikt Wolf (D-ECHO) based on extra500 work by Dirk Dittman and Eric van den Berg

var font_mapper = func(family, weight) {
	#print(sprintf("Canvas font-mapper %s %s",family,weight));
	if (family =="Liberation Mono"){
		if (weight == "bold"){
			return "LiberationFonts/LiberationMono-Bold.ttf";
		}elsif(weight == "normal"){
			return "LiberationFonts/LiberationMono-Regular.ttf";
		}
	}elsif(family =="Liberation Sans"){
		if (weight == "bold"){
			return "LiberationFonts/LiberationSans-Bold.ttf";
		}elsif(weight == "normal"){
			return "LiberationFonts/LiberationSans-Regular.ttf";
		}
	}
	
}

var MyWindow = {
	# Constructor
	#
	# @param size ([width, height])
	new: func(size, type = nil, id = nil) {
		var ghost = canvas._newWindowGhost(id);
		var m = {
			parents: [MyWindow, canvas.PropertyElement, ghost],
			_node: props.wrapNode(ghost._node_ghost)
		};
		
		m.setInt("size[0]", size[0]);
		m.setInt("size[1]", size[1]);
		
		# TODO better default position
		m.move(0,0);
		
		# arg = [child, listener_node, mode, is_child_event]
		setlistener(m._node, func m._propCallback(arg[0], arg[2]), 0, 2);
		if( type )
			m.set("type", type);
		
		m._isOpen = 1;
		return m;
	},
	# Destructor
	del: func {
		me._node.remove();
		me._node = nil;
		
		if( me["_canvas"] != nil )
		{
			me._canvas.del();
			me._canvas = nil;
		}
		me._isOpen = 0;
	},
	# Create the canvas to be used for this Window
	#
	# @return The new canvas
	createCanvas: func() {
		var size = [
		me.get("size[0]"),
		me.get("size[1]")
		];
		
		me._canvas = canvas.new({
			size: [2 * size[0], 2 * size[1]],
			view: size,
			placement: {
				type: "window",
			  id: me.get("id")
			}
		});
		
		me._canvas.addEventListener("mousedown", func me.raise());
		return me._canvas;
	},
	# Set an existing canvas to be used for this Window
	setCanvas: func(canvas_) {
		if( !isa(canvas_, canvas.Canvas) )
			return debug.warn("Not a canvas.Canvas");
		
		canvas_.addPlacement({type: "window", index: me._node.getIndex()});
		me['_canvas'] = canvas_;
	},
	# Get the displayed canvas
	getCanvas: func() {
		return me['_canvas'];
	},
	getCanvasDecoration: func() {
		return canvas.wrapCanvas(me._getCanvasDecoration());
	},
	setPosition: func(x, y) {
		me.setInt("tf/t[0]", x);
		me.setInt("tf/t[1]", y);
	},
	move: func(x, y) {
		me.setInt("tf/t[0]", me.get("tf/t[0]", 10) + x);
		me.setInt("tf/t[1]", me.get("tf/t[1]", 30) + y);
	},
	# Raise to top of window stack
	raise: func() {
		# on writing the z-index the window always is moved to the top of all other
		# windows with the same z-index.
		me.setInt("z-index", me.get("z-index", 0));
	},
	# private:
	_propCallback: func(child, mode) {
		if( !me._node.equals(child.getParent()) )
			return;
		var name = child.getName();
		
		# support for CSS like position: absolute; with right and/or bottom margin
		if( name == "right" ){
			me._handlePositionAbsolute(child, mode, name, 0);
		}else if( name == "bottom" ){
			me._handlePositionAbsolute(child, mode, name, 1);
		}
		# update decoration on type change
		else if( name == "type" )
		{
			if( mode == 0 ) {	settimer(func me._updateDecoration(), 0);	}
		}
	},
	_handlePositionAbsolute: func(child, mode, name, index) {
		# mode
		#   -1 child removed
		#    0 value changed
		#    1 child added
		
		if( mode == 0 ) {
			me._updatePos(index, name);
		} else if( mode == 1 ){
			me["_listener_" ~ name] = [	
				setlistener("/sim/gui/canvas/size[" ~ index ~ "]", func me._updatePos(index, name) ),
				setlistener(me._node.getNode("size[" ~ index ~ "]"),func me._updatePos(index, name) )
			];
		} else if( mode == -1 ){
			for(var i = 0; i < 2; i += 1){
				removelistener(me["_listener_" ~ name][i]);
			}
		}
	},
	_updatePos: func(index, name){
		me.setInt("tf/t[" ~ index ~ "]", getprop("/sim/gui/canvas/size[" ~ index ~ "]")  - me.get(name)  - me.get("size[" ~ index ~ "]") );
	},
	_onClose : func(){
		me.del();
	},
	_updateDecoration: func(){
		var border_radius = 9;
		me.set("decoration-border", "25 1 1");
		me.set("shadow-inset", int((1 - math.cos(45 * D2R)) * border_radius + 0.5));
		me.set("shadow-radius", 5);
		me.setBool("update", 1);
		
		var canvas_deco = me.getCanvasDecoration();
		canvas_deco.addEventListener("mousedown", func me.raise());
		canvas_deco.set("blend-source-rgb", "src-alpha");
		canvas_deco.set("blend-destination-rgb", "one-minus-src-alpha");
		canvas_deco.set("blend-source-alpha", "one");
		canvas_deco.set("blend-destination-alpha", "one");
		
		var group_deco = canvas_deco.getGroup("decoration");
		var title_bar = group_deco.createChild("group", "title_bar");
		title_bar
		.rect( 0, 0,
		       me.get("size[0]"),
		       me.get("size[1]"), #25,
		       {"border-top-radius": border_radius} )
		.setColorFill(0.25,0.24,0.22)
		.setStrokeLineWidth(0);
		
		var style_dir = "gui/styles/AmbianceClassic/decoration/";
		
		# close icon
		var x = 10;
		var y = 3;
		var w = 19;
		var h = 19;
		var ico = title_bar.createChild("image", "icon-close")
		.set("file", style_dir ~ "close_focused_normal.png")
		.setTranslation(x,y);
		ico.addEventListener("click", func me._onClose());
		ico.addEventListener("mouseover", func ico.set("file", style_dir ~ "close_focused_prelight.png"));
		ico.addEventListener("mousedown", func ico.set("file", style_dir ~ "close_focused_pressed.png"));
		ico.addEventListener("mouseout",  func ico.set("file", style_dir ~ "close_focused_normal.png"));
		
		# title
		me._title = title_bar.createChild("text", "title")
		.set("alignment", "left-center")
		.set("character-size", 14)
		.set("font", "LiberationFonts/LiberationSans-Bold.ttf")
		.setTranslation( int(x + 1.5 * w + 0.5),
				 int(y + 0.5 * h + 0.5) );
		
		var title = me.get("title", "Canvas Dialog");
		me._node.getNode("title", 1).alias(me._title._node.getPath() ~ "/text");
		me.set("title", title);
		
		title_bar.addEventListener("drag", func(e) {
			if( !ico.equals(e.target) )
				me.move(e.deltaX, e.deltaY);
		});
	}
};

var COLOR = {};
COLOR["Red"] 			= "rgb(244,28,33)";
COLOR["Text_Default"] 			= "#404040ff";


var SvgWidget = {
	new: func(dialog,canvasGroup,name){
		var m = {parents:[SvgWidget]};
		m._class = "SvgWidget";
		m._dialog 	= dialog;
		m._listeners 	= [];
		m._name 	= name;
		m._group	= canvasGroup;
		return m;
	},
	removeListeners  :func(){
		foreach(l;me._listeners){
			removelistener(l);
		}
		me._listeners = [];
	},
	setListeners : func(instance) {
		
	},
	init : func(instance=me){
		
	},
	deinit : func(){
		me.removeListeners();	
	},
	
};

var WeightWidget = {
	new: func(dialog,canvasGroup,name,widgets){
		var m = {parents:[WeightWidget,SvgWidget.new(dialog,canvasGroup,name)]};
		m._class = "WeightWidget";
		m._widget = {};
		
		foreach(w;keys(widgets)){
			if(widgets[w] != nil){
				if(widgets[w]._class == "PayloadWidget"){
					m._widget[w] = widgets[w];
				}
			}
		}
		
		
		m._cCenterGravityX	 	= m._group.getElementById("CenterGravity_X");
		m._cCenterGravityXMarker	= m._group.getElementById("CenterGravity_X_Marker");
		m._cCenterGravityXWarning	= m._group.getElementById("CenterGravity_X_Warning");
		m._cWeightGross	 		= m._group.getElementById("Weight_Gross");
		m._cWeightWarning	 	= m._group.getElementById("Weight_Warning");
		m._cWeightLoad	 		= m._group.getElementById("Weight_Load");
		
		
		
		m._nGross 	= props.globals.initNode("/fdm/jsbsim/inertia/weight-lbs");
		m._nEmpty 	= props.globals.initNode("/fdm/jsbsim/inertia/empty-weight-lbs");
		
		
		m._cgX  	= 0;
		m._gross 	= 0;
		m._takeoff	= getprop("limits/mtow-lbs");

		m._mm_px	= 0.4541326;	# convert mm behind datum to px of the cg diagram
		
		return m;
	},
	setListeners : func(instance) {
		append(me._listeners, setlistener(fuelPayload._nGrossWeight,func(n){me._onGrossWeightChange(n);},1,0) );
		append(me._listeners, setlistener(fuelPayload._nCGx,func(n){me._onCGChange(n);},1,0) );
		
	},
	init : func(instance=me){
		me.setListeners(instance);
	},
	deinit : func(){
		me.removeListeners();	
	},
	_onGrossWeightChange : func(n){
		me._gross = me._nGross.getValue();
		me._cWeightLoad.setText(sprintf("%3d",( me._gross * LB2KG ) - 400 ) ~ " kg");
		me._cWeightGross.setText(sprintf("%3d",me._gross * LB2KG ) ~ " kg");
		
		
		if (me._gross > me._takeoff){
			me._cWeightWarning.show();
			me._cWeightGross.setColor(COLOR["Red"]);
		}else{
			me._cWeightWarning.hide();
			me._cWeightGross.setColor(COLOR["Text_Default"]);
		}
		
	},
	_onCGChange : func(n){
		me._cgX = fuelPayload._nCGx.getValue();
		me._cCenterGravityX.setText(sprintf("%3d",me._cgX) ~ "mm behind datum");
		
		me._cCenterGravityXMarker.setTranslation( me._cgX * me._mm_px,0);
		
		if(me._cgX>190 and me._cgX<440){
			me._cCenterGravityXWarning.hide();
		}else{
			me._cCenterGravityXWarning.show();
		}
		
	},
};

var PayloadWidget = {
	new: func(dialog,canvasGroup,name,index){
		var m = {parents:[PayloadWidget,SvgWidget.new(dialog,canvasGroup,name)]};
		m._class = "PayloadWidget";
		m._index 	= index;
		
		#debug.dump(m._listCategoryKeys);
		
		m._nRoot	= props.globals.getNode("/payload/weight["~m._index~"]");
		
		### HACK : listener on /payload/weight[0]/weight-lb not working
		###	   two props one for fdm(weight-lb) one for dialog(nt-weight-lb) listener
		m._nWeightFdm 	= m._nRoot.getNode("weight-lb", 1);
		m._weight	= m._nWeightFdm.getValue(); # lbs
		m._nWeight 	= m._nRoot.initNode("nt-weight-lb",m._weight,"DOUBLE");
		
		m._nCapacity 	= m._nRoot.initNode("max-lb",0.0,"DOUBLE");
		
		m._capacity	= m._nCapacity.getValue();
		m._fraction	= m._weight / m._capacity;
		
		m._cFrame 	= m._group.getElementById(m._name~"_Frame");
		m._cLevel 	= m._group.getElementById(m._name~"_Level");
		m._cKG	 	= m._group.getElementById(m._name~"_Kg");
		
		m._cKG.setText(sprintf("%3d",m._weight * LB2KG ) ~ " kg");
		
		
		m._down		= m._cFrame.get("coord[3]");
		m._up		= m._cFrame.get("coord[1]");
		m._height	= m._up - m._down;
		
		return m;
	},
	setListeners : func(instance) {
		### FIXME : use one property remove the HACK
		append(me._listeners, setlistener(me._nWeight,func(n){me._onWeightChange(n);},1,0) );
		
		me._cFrame.addEventListener("drag",func(e){me._onInputChange(e);});
		me._cLevel.addEventListener("drag",func(e){me._onInputChange(e);});
		me._cFrame.addEventListener("wheel",func(e){me._onInputChange(e);});
		me._cLevel.addEventListener("wheel",func(e){me._onInputChange(e);});
		
		
		
	},
	init : func(instance=me){
		me.setListeners(instance);
	},
	deinit : func(){
		me.removeListeners();	
	},
	setWeight : func(weight){
		me._weight = weight;
		
		### HACK : set two props 
		me._nWeight.setValue(me._weight);
		me._nWeightFdm.setValue(me._weight);
		
	},
	_onWeightChange : func(n){
		me._weight	= me._nWeight.getValue();
		
		me._fraction	= me._weight / me._capacity;
		
		me._cKG.setText(sprintf("%3d",me._weight * LB2KG ) ~ " kg");
		
		me._cLevel.set("coord[1]", me._down + (me._height * me._fraction));
		
	},
	_onInputChange : func(e){
		var newFraction =0;
		if(e.type == "wheel"){
			newFraction = me._fraction - (e.deltaY/me._height);
		}else{
			newFraction = me._fraction + (e.deltaY/me._height);
		}
		newFraction = math.clamp(newFraction,0.0,1.0);
		me._weight = me._capacity * newFraction;

		me.setWeight(me._weight);
	},
};


var BallastWidget = {
	new: func(dialog,canvasGroup,name,index,mass_per_heavy_weight_kg,mass_per_light_weight_kg){
		var m = {parents:[BallastWidget,SvgWidget.new(dialog,canvasGroup,name)]};
		m._class = "BallastWidget";
		m._index 	= index;

		m._nRoot	= props.globals.getNode("/payload/weight["~m._index~"]");

		m._nWeightFdm 	= m._nRoot.getNode("weight-lb", 1);

		m._nLightWeights = m._nRoot.getNode("n-light-weights");
		m._nHeavyWeights = m._nRoot.getNode("n-heavy-weights");

		m._mass_per_light_weight_kg = mass_per_light_weight_kg;
		m._mass_per_heavy_weight_kg = mass_per_heavy_weight_kg;

		m._cHeavy 	= m._group.getElementById(m._name~"_Heavy");
		m._cHeavyP 	= m._group.getElementById(m._name~"_Heavy_plus");
		m._cHeavyM 	= m._group.getElementById(m._name~"_Heavy_minus");
		m._cLight 	= m._group.getElementById(m._name~"_Light");
		m._cLightP 	= m._group.getElementById(m._name~"_Light_plus");
		m._cLightM 	= m._group.getElementById(m._name~"_Light_minus");
		m._cKG	 = m._group.getElementById(m._name~"_TotalWeight");

		m._cKG.setText(sprintf("%3.1f",m._nWeightFdm.getDoubleValue() * LB2KG ) ~ " kg");

		m._cHeavy.setText( sprintf("%1d", m._nHeavyWeights.getIntValue() ) ~ " * "~ sprintf("%2.1f", m._mass_per_heavy_weight_kg) ~"kg" );
		m._cLight.setText( sprintf("%1d", m._nLightWeights.getIntValue() ) ~ " * "~ sprintf("%2.1f", m._mass_per_light_weight_kg) ~"kg" );

		m._nWeightFdm.setValue( ( m._nHeavyWeights.getIntValue() * m._mass_per_heavy_weight_kg + m._nLightWeights.getIntValue() * m._mass_per_light_weight_kg ) * KG2LB );

		return m;
	},
	setListeners : func(instance) {
		append(me._listeners, setlistener(me._nLightWeights,func(n){me._onWeightChange(n);},1,0) );
		append(me._listeners, setlistener(me._nHeavyWeights,func(n){me._onWeightChange(n);},1,0) );

		me._cHeavyP.addEventListener("click",func(e){ me._nHeavyWeights.setIntValue( math.min( me._nHeavyWeights.getIntValue() + 1, 4 ) ); });
		me._cHeavyM.addEventListener("click",func(e){ me._nHeavyWeights.setIntValue( math.max( me._nHeavyWeights.getIntValue() - 1, 0 ) ); });
		me._cLightP.addEventListener("click",func(e){ me._nLightWeights.setIntValue( math.min( me._nLightWeights.getIntValue() + 1, 2 ) ); });
		me._cLightM.addEventListener("click",func(e){ me._nLightWeights.setIntValue( math.max( me._nLightWeights.getIntValue() - 1, 0 ) ); });
	},
	init : func(instance=me){
		me.setListeners(instance);
	},
	deinit : func(){
		me.removeListeners();
	},
	_onWeightChange : func(n){
		me._nWeightFdm.setValue( ( me._nHeavyWeights.getIntValue() * me._mass_per_heavy_weight_kg + me._nLightWeights.getIntValue() * me._mass_per_light_weight_kg ) * KG2LB );

		me._cKG.setText(sprintf("%3.1f",me._nWeightFdm.getDoubleValue() * LB2KG ) ~ " kg");

		me._cHeavy.setText( sprintf("%1d", me._nHeavyWeights.getIntValue() ) ~ " * "~ sprintf("%2.1f", me._mass_per_heavy_weight_kg) ~"kg" );
		me._cLight.setText( sprintf("%1d", me._nLightWeights.getIntValue() ) ~ " * "~ sprintf("%2.1f", me._mass_per_light_weight_kg) ~"kg" );

	},
};

var FuelPayloadClass = {
	new : func(){
		var m = {parents:[FuelPayloadClass]};
		
		m._nGrossWeight	= props.globals.initNode("/fdm/jsbsim/inertia/nt-weight-lbs",0.0,"DOUBLE"); #listener on weight-lbs not possible, set via system in Systems/fuelpayload.xml
		m._nCGx	= props.globals.initNode("/fdm/jsbsim/inertia/cg-x-mm",0.0,"DOUBLE");
		
		m._name  = "Weight and Balance";
		m._title = "Weight and Balance";
		
		
		m._listeners = [];
		m._dlg 		= nil;
		m._canvas 	= nil;
		
		m._isOpen = 0;
		m._isNotInitialized = 1;
		
		m._widget = {
			Tank	 	: nil,
			Pilot	 	: nil,
			Copilot		: nil,
			Ballast	 	: nil,
			weight	 	: nil,
		};
		
		
		return m;
	},
	toggle : func(){
		if(me._dlg != nil){
			if (me._dlg._isOpen){
				me.close();
			}else{
				me.open();	
			}
		}else{
			me.open();
		}
	},
	close : func(){
		me._dlg.del();
		me._dlg = nil;
	},
	removeListeners  :func(){
		foreach(l;me._listeners){
			removelistener(l);
		}
		me._listeners = [];
	},
	setListeners : func(instance) {
		
		
	},
	_onClose : func(){
		#print("FuelPayloadClass._onClose() ... ");
		me.removeListeners();
		me._dlg.del();
		
		foreach(widget;keys(me._widget)){
			if(me._widget[widget] != nil){
				me._widget[widget].deinit();
				me._widget[widget] = nil;
			}
		}
		
	},
	open : func(){
		me.openWB();
	},
	openWB : func(){
		
		
		me._dlg = MyWindow.new([1024,690], "dialog");
		me._dlg._onClose = func(){
			fuelPayload._onClose();
		}
		me._dlg.set("title", "Weight and Balance");
		me._dlg.move(100,100);
		
		
		me._canvas = me._dlg.createCanvas();
		me._canvas.set("background", "#FFFFFF");
		
		me._group = me._canvas.createGroup();
		
		canvas.parsesvg(me._group, "Aircraft/DG-1000/gui/dialogs/WeightBalance.svg",{"font-mapper": font_mapper});
		
		
		
		me._widget.Pilot			= PayloadWidget.new(me,me._group,"Pilot",0);
		me._widget.Copilot			= PayloadWidget.new(me,me._group,"Copilot",1);
		me._widget.Baggage			= PayloadWidget.new(me,me._group,"Baggage",2);
		me._widget.WingBallastLeft	= PayloadWidget.new(me,me._group,"WingBallastLeft",3);
		me._widget.WingBallastRight	= PayloadWidget.new(me,me._group,"WingBallastRight",4);
		me._widget.FinBallast		= BallastWidget.new(me,me._group,"FinBallast",5, 2.4, 1.2);
		
		me._widget.weight = WeightWidget.new(me,me._group,"Weight",me._widget);
		
		foreach(widget;keys(me._widget)){
			if(me._widget[widget] != nil){
				me._widget[widget].init();
			}
		}
		
		#me.setListeners();
		
		
	},
	_onNotifyChange : func(n){
		
	},
	
};

var fuelPayload = FuelPayloadClass.new();

gui.menuBind("fuel-and-payload", "dialogs.fuelPayload.toggle();");
