# Copyright (C) 2004,2005 by SICEm S.L. and Imendio AB
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

import gettext

import gtk

from gazpacho.dialogs import warning
from gazpacho.clipboard import clipboard
from gazpacho.gaction import GAction, GActionGroup
from gazpacho.sizegroup import safe_to_add_widget
from gazpacho.util import get_parent

_ = gettext.gettext

(COMMAND_CUT,
 COMMAND_PASTE) = range(2)

class CommandManager(object):
    """This class is the entry point accesing the commands.
    Every undoable action in Gazpacho is wrapped into a command.
    The stack of un/redoable actions is stored in the Project class
    so each project is independent on terms of undo/redo.
    
    The CommandManager knows how to perform the undo and redo actions.
    There is also a method for every type of command supported so
    you don't have to worry about creating the low level command
    objects.
    """
    def __init__(self):
        self._commands = {}

    def _get_command(self, command_name, *args, **kwargs):
        cmd_class = self._commands[command_name]
        return cmd_class(*args, **kwargs)

    def register(self, command_class, command):
        self._commands[command_class] = command
        
    def undo(self, project):
        """Undo the last command if there is such a command"""
        if project.undo_stack.has_undo():
            cmd = project.undo_stack.pop_undo()
            cmd.undo()
    
    def redo(self, project):
        """Redo the last undo command if there is such a command"""
        if project.undo_stack.has_redo():
            cmd = project.undo_stack.pop_redo()
            cmd.redo()

    #
    # for every possible command we have a method here
    #

    def delete(self, widget_instance):
        # internal children cannot be deleted. Should we notify the user?
        if widget_instance.internal_name is not None:
            return
        description = _("Delete %s") % widget_instance.name
        cmd = self._get_command('delete', widget_instance, None,
                                widget_instance.get_parent(), False,
                                description)
        cmd.execute()
        widget_instance.project.undo_stack.push_undo(cmd)

    def create(self, adaptor, placeholder, project, parent=None,
               interactive=True):
        """
        @return: the widget that was created
        @rtype: gazpatcho.widget.Widget
        """
        if placeholder:
            parent = get_parent(placeholder)
            if parent is None:
                return

        if project is None:
            project = parent.project

        from gazpacho.widget import Widget
        widget_instance = Widget(adaptor, project)
        widget_instance.create_gtk_widget(interactive)
        if widget_instance is None:
            return

        description = _("Create %s") % widget_instance.name
        cmd = self._get_command('create', widget_instance, placeholder,
                                parent, True, description)
        cmd.execute()
        widget_instance.project.undo_stack.push_undo(cmd)

        from gazpacho.palette import palette
        if not palette.persistent_mode:
            palette.unselect_widget()

        return widget_instance

    def delete_placeholder(self, placeholder):
        parent = get_parent(placeholder)
        if len(parent.gtk_widget.get_children()) == 1:
            return
        
        description = _("Delete placeholder")
        cmd = self._get_command('delete-placeholder', placeholder,
                                parent, description)
        cmd.execute()
        parent.project.undo_stack.push_undo(cmd)
        
    def box_insert_placeholder(self, box, pos):
        description = _("Insert after")

        cmd = self._get_command('insert-placeholder', box, pos, description)
        cmd.execute()
        box.project.undo_stack.push_undo(cmd)

    def set_property(self, prop, value):
        dsc = _('Setting %s of %s') % (prop.name, prop.get_object_name())
        cmd = self._get_command('set-property', prop, value, dsc)
        cmd.execute()
        project = prop.get_project()
        project.undo_stack.push_undo(cmd)
        
    def set_translatable_property(self, prop, value, comment,
                                  translatable, has_context):
        gwidget = prop.widget
        dsc = _('Setting %s of %s') % (prop.name, gwidget.name)
        cmd = self._get_command('set-translatable',
                                prop, value, comment, translatable,
                                has_context, dsc)
        cmd.execute()
        project = prop.get_project()
        project.changed = True
        project.undo_stack.push_undo(cmd)

    def add_signal(self, widget_instance, signal):
        dsc = _('Add signal handler %s') % signal['handler']
        cmd = self._get_command('add-signal',
                                True, signal, widget_instance, dsc)
        cmd.execute()
        widget_instance.project.undo_stack.push_undo(cmd)
        
    def remove_signal(self, widget_instance, signal):
        dsc = _('Remove signal handler %s') % signal['handler']
        cmd = self._get_command('add-signal',
                                False, signal, widget_instance, dsc)
        cmd.execute()
        widget_instance.project.undo_stack.push_undo(cmd)

    def change_signal(self, widget_instance, old_signal, new_signal):
        dsc = _('Change signal handler for signal "%s"') % old_signal['name']
        cmd = self._get_command('change-signal',
                                widget_instance, old_signal, new_signal,
                                dsc)
        cmd.execute()
        widget_instance.project.undo_stack.push_undo(cmd)

    def copy(self, widget_instance):
        """Add a copy of the widget to the clipboard.

        Note that it does not make sense to undo this operation
        """
        clipboard.add_widget(widget_instance)
        
    def cut(self, widget_instance):
        dsc = _('Cut widget %s into the clipboard') % widget_instance.name
        clipboard.add_widget(widget_instance)
        cmd = self._get_command('cut-paste', widget_instance,
                                widget_instance.project, None,
                                COMMAND_CUT, dsc)
        cmd.execute()
        widget_instance.project.undo_stack.push_undo(cmd)
        
    def paste(self, placeholder, project):
        if project is None:
            raise ValueError("No project has been specified. Cannot paste "
                             "the widget")

        gwidget = clipboard.get_selected_widget(project)

        dsc = _('Paste widget %s from the clipboard') % gwidget.name
        cmd = self._get_command('cut-paste', gwidget, project, placeholder,
                                COMMAND_PASTE, dsc)
        cmd.execute()
        project.undo_stack.push_undo(cmd)

    def add_action(self, values, parent, project):
        gact = GAction(parent, values['name'], values['label'],
                       values['short_label'], values['is_important'],
                       values['tooltip'], values['stock_id'],
                       values['callback'], values['accelerator'])
        dsc = _('Add action %s') % gact.name
        cmd = self._get_command('add-action', parent, gact, True, dsc)
        cmd.execute()
        project.undo_stack.push_undo(cmd)
    
    def remove_action(self, gact, project):
        dsc = _('Remove action %s') % gact.name
        cmd = self._get_command('add-action',
                                gact.parent, gact, False, dsc)
        cmd.execute()
        project.undo_stack.push_undo(cmd)
    
    def edit_action(self, gact, new_values, project):
        dsc = _('Edit action %s') % gact.name
        cmd = self._get_command('edit-action', gact, new_values,
                                project, dsc)
        cmd.execute()
        project.undo_stack.push_undo(cmd)

    def add_action_group(self, name, project):
        gaction_group = GActionGroup(name)
        dsc = _('Add action group %s') % gaction_group.name
        cmd = self._get_command('add-action-group',
                                gaction_group, project, True, dsc)
        cmd.execute()
        project.undo_stack.push_undo(cmd)
        return gaction_group
    
    def remove_action_group(self, gaction_group, project):
        dsc = _('Remove action group %s') % gaction_group.name
        cmd = self._get_command('remove-action-group',
                                gaction_group, project, False, dsc)
        cmd.execute()
        project.undo_stack.push_undo(cmd)
        
    def edit_action_group(self, gaction_group, new_name, project):
        dsc = _('Edit action group %s') % gaction_group.name
        cmd = self._get_command('edit-action-group',
                                gaction_group, new_name, project, dsc)
        cmd.execute()
        project.undo_stack.push_undo(cmd)

    def set_button_contents(self, gwidget, stock_id=None, notext=False,
                            label=None, image_path=None, position=-1,
                            icon_size=gtk.ICON_SIZE_BUTTON):
        dsc = _("Setting button %s contents") % gwidget.name
        cmd = self._get_command('button', gwidget, stock_id, notext,
                                label, image_path, position,
                                icon_size, dsc)
        cmd.execute()
        gwidget.project.undo_stack.push_undo(cmd)

    def execute_drag_drop(self, source_widget, target_placeholder):
        """Execute a drag and drop action, i.e. move a widget from one
        place to another. This method cannot be used to drag widgets
        between programs.

        @param source_widget: the widget that is dragged
        @type source_widget: gazpacho.widget.Widget
        @param target_placeholder: the placeholder onto which the
        widget is droped
        @type target_placeholder: gazpacho.placeholder.Placeholder
        """
        cmd = self._get_command('drag-drop', source_widget, target_placeholder,
                                _("Drag and Drop widget %s") % source_widget.name)
        cmd.execute()
        source_widget.project.undo_stack.push_undo(cmd)

    def execute_drop(self, widget_instance, target_placeholder):
        """Execute a drop action, i.e. add a widget.

        @param widget_instance: the widget that is droped
        @type widget_instance: gazpacho.widget.Widget
        @param target_placeholder: the placeholder onto which the
        widget is droped
        @type target_placeholder: gazpacho.placeholder.Placeholder
        """
        project = widget_instance.project
        cmd = self._get_command('cut-paste', widget_instance, project,
                                target_placeholder, COMMAND_PASTE,
                                _('Drop widget %s') % widget_instance.name)
        cmd.execute()
        project.undo_stack.push_undo(cmd)

    def execute_drag_extend(self, source_widget, target_widget,
                            location, keep_source):
        cmd = self._get_command('drag-extend', source_widget,
                                target_widget, location,
                                keep_source, _("Drag - extend"))
        cmd.execute()
        target_widget.project.undo_stack.push_undo(cmd)
        
    def execute_drag_append(self, source_widget, target_box, pos, keep_source):
        cmd = self._get_command('drag-append', source_widget, target_box, pos,
                                keep_source, _("Drag - append"))
        cmd.execute()
        target_box.project.undo_stack.push_undo(cmd)

    def add_sizegroup_widget(self, sizegroup, widget, project):
        """
        Add a widget to a size group. If a widget is already in the
        size group it will be ignored.

        It is not possible to add widgets that already exist in the
        sizegroup. If that should happend those widgets will be
        ignored. It is thus possible to end up with a command that
        doesn't do anything at all.

        @param sizegroup: the sizegroup that the widgets belong to
        @type sizegroup: gazpacho.sizegroup.GSizeGroup
        @param widget: the widget that should be added
        @type widget: gazpacho.widget.Widget
        @param project: the project that the sizegroup belongs to
        @type project: gazpacho.project.Project
        """
        add_widgets = []
        
        # We don't add a widget that's already in the sizegroup
        if not sizegroup.has_widget(widget):
            add_widgets.append(widget)

        if widget and not safe_to_add_widget(sizegroup, widget):
            warning(_("Cannot add the widet"),
                    _("It's not possible to add a widget who has an ancestor"
                      " or child in the sizegroup."))
            return
        
        dsc = _("Add widgets to size group '%s'") % sizegroup.name
        cmd = self._get_command('sizegroup', sizegroup, add_widgets,
                                project, True, dsc)
        cmd.execute()
        project.undo_stack.push_undo(cmd)

    def remove_sizegroup_widgets(self, sizegroup, widgets, project):
        """
        Remove widgets from a size group.

        @param sizegroup: the sizegroup that the widgets belong to
        @type sizegroup: gazpacho.sizegroup.GSizeGroup
        @param widgets: the widgets that should be removed
        @type widgets: list (of gazpacho.widget.Widget)
        @param project: the project that the sizegroup belongs to
        @type project: gazpacho.project.Project
        """
        dsc = _("Remove widgets from size group '%s'") % sizegroup.name
        cmd = self._get_command('sizegroup', sizegroup, widgets,
                                project, False, dsc)
        cmd.execute()
        project.undo_stack.push_undo(cmd)
        
    def remove_sizegroup(self, sizegroup, project):
        """
        Remove a size group.

        @param sizegroup: the sizegroup that should be removed
        @type sizegroup: gazpacho.sizegroup.GSizeGroup
        @param project: the project that the sizegroup belongs to
        @type project: gazpacho.project.Project
        """
        widgets = sizegroup.get_widgets()
        dsc = _("Remove size group '%s'") % sizegroup.name
        cmd = self._get_command('sizegroup', sizegroup, widgets,
                                project, False, dsc)
        cmd.execute()
        project.undo_stack.push_undo(cmd)

command_manager = CommandManager()
