/*===---- arm_neon.h - ARM Neon intrinsics ---------------------------------===
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *
 *===-----------------------------------------------------------------------===
 */

#ifndef __ARM_NEON_H
#define __ARM_NEON_H

#ifndef __ARM_FP
#error "NEON intrinsics not available with the soft-float ABI. Please use -mfloat-abi=softfp or -mfloat-abi=hard"
#else

#if !defined(__ARM_NEON)
#error "NEON support not enabled"
#else

#include <stdint.h>

#ifdef __ARM_FEATURE_BF16
#include <arm_bf16.h>
typedef __bf16 bfloat16_t;
#endif

typedef float float32_t;
typedef __fp16 float16_t;
#ifdef __aarch64__
typedef double float64_t;
#endif

#ifdef __aarch64__
typedef uint8_t poly8_t;
typedef uint16_t poly16_t;
typedef uint64_t poly64_t;
typedef __uint128_t poly128_t;
#else
typedef int8_t poly8_t;
typedef int16_t poly16_t;
typedef int64_t poly64_t;
#endif
typedef __attribute__((neon_vector_type(8))) int8_t int8x8_t;
typedef __attribute__((neon_vector_type(16))) int8_t int8x16_t;
typedef __attribute__((neon_vector_type(4))) int16_t int16x4_t;
typedef __attribute__((neon_vector_type(8))) int16_t int16x8_t;
typedef __attribute__((neon_vector_type(2))) int32_t int32x2_t;
typedef __attribute__((neon_vector_type(4))) int32_t int32x4_t;
typedef __attribute__((neon_vector_type(1))) int64_t int64x1_t;
typedef __attribute__((neon_vector_type(2))) int64_t int64x2_t;
typedef __attribute__((neon_vector_type(8))) uint8_t uint8x8_t;
typedef __attribute__((neon_vector_type(16))) uint8_t uint8x16_t;
typedef __attribute__((neon_vector_type(4))) uint16_t uint16x4_t;
typedef __attribute__((neon_vector_type(8))) uint16_t uint16x8_t;
typedef __attribute__((neon_vector_type(2))) uint32_t uint32x2_t;
typedef __attribute__((neon_vector_type(4))) uint32_t uint32x4_t;
typedef __attribute__((neon_vector_type(1))) uint64_t uint64x1_t;
typedef __attribute__((neon_vector_type(2))) uint64_t uint64x2_t;
typedef __attribute__((neon_vector_type(4))) float16_t float16x4_t;
typedef __attribute__((neon_vector_type(8))) float16_t float16x8_t;
typedef __attribute__((neon_vector_type(2))) float32_t float32x2_t;
typedef __attribute__((neon_vector_type(4))) float32_t float32x4_t;
#ifdef __aarch64__
typedef __attribute__((neon_vector_type(1))) float64_t float64x1_t;
typedef __attribute__((neon_vector_type(2))) float64_t float64x2_t;
#endif
typedef __attribute__((neon_polyvector_type(8))) poly8_t poly8x8_t;
typedef __attribute__((neon_polyvector_type(16))) poly8_t poly8x16_t;
typedef __attribute__((neon_polyvector_type(4))) poly16_t poly16x4_t;
typedef __attribute__((neon_polyvector_type(8))) poly16_t poly16x8_t;
typedef __attribute__((neon_polyvector_type(1))) poly64_t poly64x1_t;
typedef __attribute__((neon_polyvector_type(2))) poly64_t poly64x2_t;

typedef struct int8x8x2_t {
  int8x8_t val[2];
} int8x8x2_t;

typedef struct int8x16x2_t {
  int8x16_t val[2];
} int8x16x2_t;

typedef struct int16x4x2_t {
  int16x4_t val[2];
} int16x4x2_t;

typedef struct int16x8x2_t {
  int16x8_t val[2];
} int16x8x2_t;

typedef struct int32x2x2_t {
  int32x2_t val[2];
} int32x2x2_t;

typedef struct int32x4x2_t {
  int32x4_t val[2];
} int32x4x2_t;

typedef struct int64x1x2_t {
  int64x1_t val[2];
} int64x1x2_t;

typedef struct int64x2x2_t {
  int64x2_t val[2];
} int64x2x2_t;

typedef struct uint8x8x2_t {
  uint8x8_t val[2];
} uint8x8x2_t;

typedef struct uint8x16x2_t {
  uint8x16_t val[2];
} uint8x16x2_t;

typedef struct uint16x4x2_t {
  uint16x4_t val[2];
} uint16x4x2_t;

typedef struct uint16x8x2_t {
  uint16x8_t val[2];
} uint16x8x2_t;

typedef struct uint32x2x2_t {
  uint32x2_t val[2];
} uint32x2x2_t;

typedef struct uint32x4x2_t {
  uint32x4_t val[2];
} uint32x4x2_t;

typedef struct uint64x1x2_t {
  uint64x1_t val[2];
} uint64x1x2_t;

typedef struct uint64x2x2_t {
  uint64x2_t val[2];
} uint64x2x2_t;

typedef struct float16x4x2_t {
  float16x4_t val[2];
} float16x4x2_t;

typedef struct float16x8x2_t {
  float16x8_t val[2];
} float16x8x2_t;

typedef struct float32x2x2_t {
  float32x2_t val[2];
} float32x2x2_t;

typedef struct float32x4x2_t {
  float32x4_t val[2];
} float32x4x2_t;

#ifdef __aarch64__
typedef struct float64x1x2_t {
  float64x1_t val[2];
} float64x1x2_t;

typedef struct float64x2x2_t {
  float64x2_t val[2];
} float64x2x2_t;

#endif
typedef struct poly8x8x2_t {
  poly8x8_t val[2];
} poly8x8x2_t;

typedef struct poly8x16x2_t {
  poly8x16_t val[2];
} poly8x16x2_t;

typedef struct poly16x4x2_t {
  poly16x4_t val[2];
} poly16x4x2_t;

typedef struct poly16x8x2_t {
  poly16x8_t val[2];
} poly16x8x2_t;

typedef struct poly64x1x2_t {
  poly64x1_t val[2];
} poly64x1x2_t;

typedef struct poly64x2x2_t {
  poly64x2_t val[2];
} poly64x2x2_t;

typedef struct int8x8x3_t {
  int8x8_t val[3];
} int8x8x3_t;

typedef struct int8x16x3_t {
  int8x16_t val[3];
} int8x16x3_t;

typedef struct int16x4x3_t {
  int16x4_t val[3];
} int16x4x3_t;

typedef struct int16x8x3_t {
  int16x8_t val[3];
} int16x8x3_t;

typedef struct int32x2x3_t {
  int32x2_t val[3];
} int32x2x3_t;

typedef struct int32x4x3_t {
  int32x4_t val[3];
} int32x4x3_t;

typedef struct int64x1x3_t {
  int64x1_t val[3];
} int64x1x3_t;

typedef struct int64x2x3_t {
  int64x2_t val[3];
} int64x2x3_t;

typedef struct uint8x8x3_t {
  uint8x8_t val[3];
} uint8x8x3_t;

typedef struct uint8x16x3_t {
  uint8x16_t val[3];
} uint8x16x3_t;

typedef struct uint16x4x3_t {
  uint16x4_t val[3];
} uint16x4x3_t;

typedef struct uint16x8x3_t {
  uint16x8_t val[3];
} uint16x8x3_t;

typedef struct uint32x2x3_t {
  uint32x2_t val[3];
} uint32x2x3_t;

typedef struct uint32x4x3_t {
  uint32x4_t val[3];
} uint32x4x3_t;

typedef struct uint64x1x3_t {
  uint64x1_t val[3];
} uint64x1x3_t;

typedef struct uint64x2x3_t {
  uint64x2_t val[3];
} uint64x2x3_t;

typedef struct float16x4x3_t {
  float16x4_t val[3];
} float16x4x3_t;

typedef struct float16x8x3_t {
  float16x8_t val[3];
} float16x8x3_t;

typedef struct float32x2x3_t {
  float32x2_t val[3];
} float32x2x3_t;

typedef struct float32x4x3_t {
  float32x4_t val[3];
} float32x4x3_t;

#ifdef __aarch64__
typedef struct float64x1x3_t {
  float64x1_t val[3];
} float64x1x3_t;

typedef struct float64x2x3_t {
  float64x2_t val[3];
} float64x2x3_t;

#endif
typedef struct poly8x8x3_t {
  poly8x8_t val[3];
} poly8x8x3_t;

typedef struct poly8x16x3_t {
  poly8x16_t val[3];
} poly8x16x3_t;

typedef struct poly16x4x3_t {
  poly16x4_t val[3];
} poly16x4x3_t;

typedef struct poly16x8x3_t {
  poly16x8_t val[3];
} poly16x8x3_t;

typedef struct poly64x1x3_t {
  poly64x1_t val[3];
} poly64x1x3_t;

typedef struct poly64x2x3_t {
  poly64x2_t val[3];
} poly64x2x3_t;

typedef struct int8x8x4_t {
  int8x8_t val[4];
} int8x8x4_t;

typedef struct int8x16x4_t {
  int8x16_t val[4];
} int8x16x4_t;

typedef struct int16x4x4_t {
  int16x4_t val[4];
} int16x4x4_t;

typedef struct int16x8x4_t {
  int16x8_t val[4];
} int16x8x4_t;

typedef struct int32x2x4_t {
  int32x2_t val[4];
} int32x2x4_t;

typedef struct int32x4x4_t {
  int32x4_t val[4];
} int32x4x4_t;

typedef struct int64x1x4_t {
  int64x1_t val[4];
} int64x1x4_t;

typedef struct int64x2x4_t {
  int64x2_t val[4];
} int64x2x4_t;

typedef struct uint8x8x4_t {
  uint8x8_t val[4];
} uint8x8x4_t;

typedef struct uint8x16x4_t {
  uint8x16_t val[4];
} uint8x16x4_t;

typedef struct uint16x4x4_t {
  uint16x4_t val[4];
} uint16x4x4_t;

typedef struct uint16x8x4_t {
  uint16x8_t val[4];
} uint16x8x4_t;

typedef struct uint32x2x4_t {
  uint32x2_t val[4];
} uint32x2x4_t;

typedef struct uint32x4x4_t {
  uint32x4_t val[4];
} uint32x4x4_t;

typedef struct uint64x1x4_t {
  uint64x1_t val[4];
} uint64x1x4_t;

typedef struct uint64x2x4_t {
  uint64x2_t val[4];
} uint64x2x4_t;

typedef struct float16x4x4_t {
  float16x4_t val[4];
} float16x4x4_t;

typedef struct float16x8x4_t {
  float16x8_t val[4];
} float16x8x4_t;

typedef struct float32x2x4_t {
  float32x2_t val[4];
} float32x2x4_t;

typedef struct float32x4x4_t {
  float32x4_t val[4];
} float32x4x4_t;

#ifdef __aarch64__
typedef struct float64x1x4_t {
  float64x1_t val[4];
} float64x1x4_t;

typedef struct float64x2x4_t {
  float64x2_t val[4];
} float64x2x4_t;

#endif
typedef struct poly8x8x4_t {
  poly8x8_t val[4];
} poly8x8x4_t;

typedef struct poly8x16x4_t {
  poly8x16_t val[4];
} poly8x16x4_t;

typedef struct poly16x4x4_t {
  poly16x4_t val[4];
} poly16x4x4_t;

typedef struct poly16x8x4_t {
  poly16x8_t val[4];
} poly16x8x4_t;

typedef struct poly64x1x4_t {
  poly64x1_t val[4];
} poly64x1x4_t;

typedef struct poly64x2x4_t {
  poly64x2_t val[4];
} poly64x2x4_t;

#ifdef __ARM_FEATURE_BF16
typedef __attribute__((neon_vector_type(4))) bfloat16_t bfloat16x4_t;
typedef __attribute__((neon_vector_type(8))) bfloat16_t bfloat16x8_t;

typedef struct bfloat16x4x2_t {
  bfloat16x4_t val[2];
} bfloat16x4x2_t;

typedef struct bfloat16x8x2_t {
  bfloat16x8_t val[2];
} bfloat16x8x2_t;

typedef struct bfloat16x4x3_t {
  bfloat16x4_t val[3];
} bfloat16x4x3_t;

typedef struct bfloat16x8x3_t {
  bfloat16x8_t val[3];
} bfloat16x8x3_t;

typedef struct bfloat16x4x4_t {
  bfloat16x4_t val[4];
} bfloat16x4x4_t;

typedef struct bfloat16x8x4_t {
  bfloat16x8_t val[4];
} bfloat16x8x4_t;

#endif

#define __ai static __inline__ __attribute__((__always_inline__, __nodebug__))

#ifdef __LITTLE_ENDIAN__
#define splat_lane_p8(__p0, __p1) __extension__ ({ \
  poly8x8_t __s0 = __p0; \
  poly8x8_t __ret; \
  __ret = (poly8x8_t) __builtin_neon_splat_lane_v((int8x8_t)__s0, __p1, 4); \
  __ret; \
})
#else
#define splat_lane_p8(__p0, __p1) __extension__ ({ \
  poly8x8_t __s0 = __p0; \
  poly8x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  poly8x8_t __ret; \
  __ret = (poly8x8_t) __builtin_neon_splat_lane_v((int8x8_t)__rev0, __p1, 4); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_splat_lane_p8(__p0, __p1) __extension__ ({ \
  poly8x8_t __s0 = __p0; \
  poly8x8_t __ret; \
  __ret = (poly8x8_t) __builtin_neon_splat_lane_v((int8x8_t)__s0, __p1, 4); \
  __ret; \
})
#endif

#define splat_lane_p64(__p0, __p1) __extension__ ({ \
  poly64x1_t __s0 = __p0; \
  poly64x1_t __ret; \
  __ret = (poly64x1_t) __builtin_neon_splat_lane_v((int8x8_t)__s0, __p1, 6); \
  __ret; \
})
#ifdef __LITTLE_ENDIAN__
#define splat_lane_p16(__p0, __p1) __extension__ ({ \
  poly16x4_t __s0 = __p0; \
  poly16x4_t __ret; \
  __ret = (poly16x4_t) __builtin_neon_splat_lane_v((int8x8_t)__s0, __p1, 5); \
  __ret; \
})
#else
#define splat_lane_p16(__p0, __p1) __extension__ ({ \
  poly16x4_t __s0 = __p0; \
  poly16x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  poly16x4_t __ret; \
  __ret = (poly16x4_t) __builtin_neon_splat_lane_v((int8x8_t)__rev0, __p1, 5); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_splat_lane_p16(__p0, __p1) __extension__ ({ \
  poly16x4_t __s0 = __p0; \
  poly16x4_t __ret; \
  __ret = (poly16x4_t) __builtin_neon_splat_lane_v((int8x8_t)__s0, __p1, 5); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define splatq_lane_p8(__p0, __p1) __extension__ ({ \
  poly8x8_t __s0 = __p0; \
  poly8x16_t __ret; \
  __ret = (poly8x16_t) __builtin_neon_splatq_lane_v((int8x8_t)__s0, __p1, 4); \
  __ret; \
})
#else
#define splatq_lane_p8(__p0, __p1) __extension__ ({ \
  poly8x8_t __s0 = __p0; \
  poly8x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  poly8x16_t __ret; \
  __ret = (poly8x16_t) __builtin_neon_splatq_lane_v((int8x8_t)__rev0, __p1, 4); \
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_splatq_lane_p8(__p0, __p1) __extension__ ({ \
  poly8x8_t __s0 = __p0; \
  poly8x16_t __ret; \
  __ret = (poly8x16_t) __builtin_neon_splatq_lane_v((int8x8_t)__s0, __p1, 4); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define splatq_lane_p64(__p0, __p1) __extension__ ({ \
  poly64x1_t __s0 = __p0; \
  poly64x2_t __ret; \
  __ret = (poly64x2_t) __builtin_neon_splatq_lane_v((int8x8_t)__s0, __p1, 6); \
  __ret; \
})
#else
#define splatq_lane_p64(__p0, __p1) __extension__ ({ \
  poly64x1_t __s0 = __p0; \
  poly64x2_t __ret; \
  __ret = (poly64x2_t) __builtin_neon_splatq_lane_v((int8x8_t)__s0, __p1, 6); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#define __noswap_splatq_lane_p64(__p0, __p1) __extension__ ({ \
  poly64x1_t __s0 = __p0; \
  poly64x2_t __ret; \
  __ret = (poly64x2_t) __builtin_neon_splatq_lane_v((int8x8_t)__s0, __p1, 6); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define splatq_lane_p16(__p0, __p1) __extension__ ({ \
  poly16x4_t __s0 = __p0; \
  poly16x8_t __ret; \
  __ret = (poly16x8_t) __builtin_neon_splatq_lane_v((int8x8_t)__s0, __p1, 5); \
  __ret; \
})
#else
#define splatq_lane_p16(__p0, __p1) __extension__ ({ \
  poly16x4_t __s0 = __p0; \
  poly16x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  poly16x8_t __ret; \
  __ret = (poly16x8_t) __builtin_neon_splatq_lane_v((int8x8_t)__rev0, __p1, 5); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_splatq_lane_p16(__p0, __p1) __extension__ ({ \
  poly16x4_t __s0 = __p0; \
  poly16x8_t __ret; \
  __ret = (poly16x8_t) __builtin_neon_splatq_lane_v((int8x8_t)__s0, __p1, 5); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define splatq_lane_u8(__p0, __p1) __extension__ ({ \
  uint8x8_t __s0 = __p0; \
  uint8x16_t __ret; \
  __ret = (uint8x16_t) __builtin_neon_splatq_lane_v((int8x8_t)__s0, __p1, 16); \
  __ret; \
})
#else
#define splatq_lane_u8(__p0, __p1) __extension__ ({ \
  uint8x8_t __s0 = __p0; \
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint8x16_t __ret; \
  __ret = (uint8x16_t) __builtin_neon_splatq_lane_v((int8x8_t)__rev0, __p1, 16); \
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_splatq_lane_u8(__p0, __p1) __extension__ ({ \
  uint8x8_t __s0 = __p0; \
  uint8x16_t __ret; \
  __ret = (uint8x16_t) __builtin_neon_splatq_lane_v((int8x8_t)__s0, __p1, 16); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define splatq_lane_u32(__p0, __p1) __extension__ ({ \
  uint32x2_t __s0 = __p0; \
  uint32x4_t __ret; \
  __ret = (uint32x4_t) __builtin_neon_splatq_lane_v((int8x8_t)__s0, __p1, 18); \
  __ret; \
})
#else
#define splatq_lane_u32(__p0, __p1) __extension__ ({ \
  uint32x2_t __s0 = __p0; \
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  uint32x4_t __ret; \
  __ret = (uint32x4_t) __builtin_neon_splatq_lane_v((int8x8_t)__rev0, __p1, 18); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_splatq_lane_u32(__p0, __p1) __extension__ ({ \
  uint32x2_t __s0 = __p0; \
  uint32x4_t __ret; \
  __ret = (uint32x4_t) __builtin_neon_splatq_lane_v((int8x8_t)__s0, __p1, 18); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define splatq_lane_u64(__p0, __p1) __extension__ ({ \
  uint64x1_t __s0 = __p0; \
  uint64x2_t __ret; \
  __ret = (uint64x2_t) __builtin_neon_splatq_lane_v((int8x8_t)__s0, __p1, 19); \
  __ret; \
})
#else
#define splatq_lane_u64(__p0, __p1) __extension__ ({ \
  uint64x1_t __s0 = __p0; \
  uint64x2_t __ret; \
  __ret = (uint64x2_t) __builtin_neon_splatq_lane_v((int8x8_t)__s0, __p1, 19); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#define __noswap_splatq_lane_u64(__p0, __p1) __extension__ ({ \
  uint64x1_t __s0 = __p0; \
  uint64x2_t __ret; \
  __ret = (uint64x2_t) __builtin_neon_splatq_lane_v((int8x8_t)__s0, __p1, 19); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define splatq_lane_u16(__p0, __p1) __extension__ ({ \
  uint16x4_t __s0 = __p0; \
  uint16x8_t __ret; \
  __ret = (uint16x8_t) __builtin_neon_splatq_lane_v((int8x8_t)__s0, __p1, 17); \
  __ret; \
})
#else
#define splatq_lane_u16(__p0, __p1) __extension__ ({ \
  uint16x4_t __s0 = __p0; \
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  uint16x8_t __ret; \
  __ret = (uint16x8_t) __builtin_neon_splatq_lane_v((int8x8_t)__rev0, __p1, 17); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_splatq_lane_u16(__p0, __p1) __extension__ ({ \
  uint16x4_t __s0 = __p0; \
  uint16x8_t __ret; \
  __ret = (uint16x8_t) __builtin_neon_splatq_lane_v((int8x8_t)__s0, __p1, 17); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define splatq_lane_s8(__p0, __p1) __extension__ ({ \
  int8x8_t __s0 = __p0; \
  int8x16_t __ret; \
  __ret = (int8x16_t) __builtin_neon_splatq_lane_v((int8x8_t)__s0, __p1, 0); \
  __ret; \
})
#else
#define splatq_lane_s8(__p0, __p1) __extension__ ({ \
  int8x8_t __s0 = __p0; \
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  int8x16_t __ret; \
  __ret = (int8x16_t) __builtin_neon_splatq_lane_v((int8x8_t)__rev0, __p1, 0); \
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_splatq_lane_s8(__p0, __p1) __extension__ ({ \
  int8x8_t __s0 = __p0; \
  int8x16_t __ret; \
  __ret = (int8x16_t) __builtin_neon_splatq_lane_v((int8x8_t)__s0, __p1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define splatq_lane_f64(__p0, __p1) __extension__ ({ \
  float64x1_t __s0 = __p0; \
  float64x2_t __ret; \
  __ret = (float64x2_t) __builtin_neon_splatq_lane_v((int8x8_t)__s0, __p1, 10); \
  __ret; \
})
#else
#define splatq_lane_f64(__p0, __p1) __extension__ ({ \
  float64x1_t __s0 = __p0; \
  float64x2_t __ret; \
  __ret = (float64x2_t) __builtin_neon_splatq_lane_v((int8x8_t)__s0, __p1, 10); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#define __noswap_splatq_lane_f64(__p0, __p1) __extension__ ({ \
  float64x1_t __s0 = __p0; \
  float64x2_t __ret; \
  __ret = (float64x2_t) __builtin_neon_splatq_lane_v((int8x8_t)__s0, __p1, 10); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define splatq_lane_f32(__p0, __p1) __extension__ ({ \
  float32x2_t __s0 = __p0; \
  float32x4_t __ret; \
  __ret = (float32x4_t) __builtin_neon_splatq_lane_v((int8x8_t)__s0, __p1, 9); \
  __ret; \
})
#else
#define splatq_lane_f32(__p0, __p1) __extension__ ({ \
  float32x2_t __s0 = __p0; \
  float32x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  float32x4_t __ret; \
  __ret = (float32x4_t) __builtin_neon_splatq_lane_v((int8x8_t)__rev0, __p1, 9); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_splatq_lane_f32(__p0, __p1) __extension__ ({ \
  float32x2_t __s0 = __p0; \
  float32x4_t __ret; \
  __ret = (float32x4_t) __builtin_neon_splatq_lane_v((int8x8_t)__s0, __p1, 9); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define splatq_lane_f16(__p0, __p1) __extension__ ({ \
  float16x4_t __s0 = __p0; \
  float16x8_t __ret; \
  __ret = (float16x8_t) __builtin_neon_splatq_lane_v((int8x8_t)__s0, __p1, 8); \
  __ret; \
})
#else
#define splatq_lane_f16(__p0, __p1) __extension__ ({ \
  float16x4_t __s0 = __p0; \
  float16x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  float16x8_t __ret; \
  __ret = (float16x8_t) __builtin_neon_splatq_lane_v((int8x8_t)__rev0, __p1, 8); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_splatq_lane_f16(__p0, __p1) __extension__ ({ \
  float16x4_t __s0 = __p0; \
  float16x8_t __ret; \
  __ret = (float16x8_t) __builtin_neon_splatq_lane_v((int8x8_t)__s0, __p1, 8); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define splatq_lane_s32(__p0, __p1) __extension__ ({ \
  int32x2_t __s0 = __p0; \
  int32x4_t __ret; \
  __ret = (int32x4_t) __builtin_neon_splatq_lane_v((int8x8_t)__s0, __p1, 2); \
  __ret; \
})
#else
#define splatq_lane_s32(__p0, __p1) __extension__ ({ \
  int32x2_t __s0 = __p0; \
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  int32x4_t __ret; \
  __ret = (int32x4_t) __builtin_neon_splatq_lane_v((int8x8_t)__rev0, __p1, 2); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_splatq_lane_s32(__p0, __p1) __extension__ ({ \
  int32x2_t __s0 = __p0; \
  int32x4_t __ret; \
  __ret = (int32x4_t) __builtin_neon_splatq_lane_v((int8x8_t)__s0, __p1, 2); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define splatq_lane_s64(__p0, __p1) __extension__ ({ \
  int64x1_t __s0 = __p0; \
  int64x2_t __ret; \
  __ret = (int64x2_t) __builtin_neon_splatq_lane_v((int8x8_t)__s0, __p1, 3); \
  __ret; \
})
#else
#define splatq_lane_s64(__p0, __p1) __extension__ ({ \
  int64x1_t __s0 = __p0; \
  int64x2_t __ret; \
  __ret = (int64x2_t) __builtin_neon_splatq_lane_v((int8x8_t)__s0, __p1, 3); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#define __noswap_splatq_lane_s64(__p0, __p1) __extension__ ({ \
  int64x1_t __s0 = __p0; \
  int64x2_t __ret; \
  __ret = (int64x2_t) __builtin_neon_splatq_lane_v((int8x8_t)__s0, __p1, 3); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define splatq_lane_s16(__p0, __p1) __extension__ ({ \
  int16x4_t __s0 = __p0; \
  int16x8_t __ret; \
  __ret = (int16x8_t) __builtin_neon_splatq_lane_v((int8x8_t)__s0, __p1, 1); \
  __ret; \
})
#else
#define splatq_lane_s16(__p0, __p1) __extension__ ({ \
  int16x4_t __s0 = __p0; \
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  int16x8_t __ret; \
  __ret = (int16x8_t) __builtin_neon_splatq_lane_v((int8x8_t)__rev0, __p1, 1); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_splatq_lane_s16(__p0, __p1) __extension__ ({ \
  int16x4_t __s0 = __p0; \
  int16x8_t __ret; \
  __ret = (int16x8_t) __builtin_neon_splatq_lane_v((int8x8_t)__s0, __p1, 1); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define splat_lane_u8(__p0, __p1) __extension__ ({ \
  uint8x8_t __s0 = __p0; \
  uint8x8_t __ret; \
  __ret = (uint8x8_t) __builtin_neon_splat_lane_v((int8x8_t)__s0, __p1, 16); \
  __ret; \
})
#else
#define splat_lane_u8(__p0, __p1) __extension__ ({ \
  uint8x8_t __s0 = __p0; \
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint8x8_t __ret; \
  __ret = (uint8x8_t) __builtin_neon_splat_lane_v((int8x8_t)__rev0, __p1, 16); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_splat_lane_u8(__p0, __p1) __extension__ ({ \
  uint8x8_t __s0 = __p0; \
  uint8x8_t __ret; \
  __ret = (uint8x8_t) __builtin_neon_splat_lane_v((int8x8_t)__s0, __p1, 16); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define splat_lane_u32(__p0, __p1) __extension__ ({ \
  uint32x2_t __s0 = __p0; \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_splat_lane_v((int8x8_t)__s0, __p1, 18); \
  __ret; \
})
#else
#define splat_lane_u32(__p0, __p1) __extension__ ({ \
  uint32x2_t __s0 = __p0; \
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_splat_lane_v((int8x8_t)__rev0, __p1, 18); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#define __noswap_splat_lane_u32(__p0, __p1) __extension__ ({ \
  uint32x2_t __s0 = __p0; \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_splat_lane_v((int8x8_t)__s0, __p1, 18); \
  __ret; \
})
#endif

#define splat_lane_u64(__p0, __p1) __extension__ ({ \
  uint64x1_t __s0 = __p0; \
  uint64x1_t __ret; \
  __ret = (uint64x1_t) __builtin_neon_splat_lane_v((int8x8_t)__s0, __p1, 19); \
  __ret; \
})
#ifdef __LITTLE_ENDIAN__
#define splat_lane_u16(__p0, __p1) __extension__ ({ \
  uint16x4_t __s0 = __p0; \
  uint16x4_t __ret; \
  __ret = (uint16x4_t) __builtin_neon_splat_lane_v((int8x8_t)__s0, __p1, 17); \
  __ret; \
})
#else
#define splat_lane_u16(__p0, __p1) __extension__ ({ \
  uint16x4_t __s0 = __p0; \
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  uint16x4_t __ret; \
  __ret = (uint16x4_t) __builtin_neon_splat_lane_v((int8x8_t)__rev0, __p1, 17); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_splat_lane_u16(__p0, __p1) __extension__ ({ \
  uint16x4_t __s0 = __p0; \
  uint16x4_t __ret; \
  __ret = (uint16x4_t) __builtin_neon_splat_lane_v((int8x8_t)__s0, __p1, 17); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define splat_lane_s8(__p0, __p1) __extension__ ({ \
  int8x8_t __s0 = __p0; \
  int8x8_t __ret; \
  __ret = (int8x8_t) __builtin_neon_splat_lane_v((int8x8_t)__s0, __p1, 0); \
  __ret; \
})
#else
#define splat_lane_s8(__p0, __p1) __extension__ ({ \
  int8x8_t __s0 = __p0; \
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  int8x8_t __ret; \
  __ret = (int8x8_t) __builtin_neon_splat_lane_v((int8x8_t)__rev0, __p1, 0); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_splat_lane_s8(__p0, __p1) __extension__ ({ \
  int8x8_t __s0 = __p0; \
  int8x8_t __ret; \
  __ret = (int8x8_t) __builtin_neon_splat_lane_v((int8x8_t)__s0, __p1, 0); \
  __ret; \
})
#endif

#define splat_lane_f64(__p0, __p1) __extension__ ({ \
  float64x1_t __s0 = __p0; \
  float64x1_t __ret; \
  __ret = (float64x1_t) __builtin_neon_splat_lane_v((int8x8_t)__s0, __p1, 10); \
  __ret; \
})
#ifdef __LITTLE_ENDIAN__
#define splat_lane_f32(__p0, __p1) __extension__ ({ \
  float32x2_t __s0 = __p0; \
  float32x2_t __ret; \
  __ret = (float32x2_t) __builtin_neon_splat_lane_v((int8x8_t)__s0, __p1, 9); \
  __ret; \
})
#else
#define splat_lane_f32(__p0, __p1) __extension__ ({ \
  float32x2_t __s0 = __p0; \
  float32x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  float32x2_t __ret; \
  __ret = (float32x2_t) __builtin_neon_splat_lane_v((int8x8_t)__rev0, __p1, 9); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#define __noswap_splat_lane_f32(__p0, __p1) __extension__ ({ \
  float32x2_t __s0 = __p0; \
  float32x2_t __ret; \
  __ret = (float32x2_t) __builtin_neon_splat_lane_v((int8x8_t)__s0, __p1, 9); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define splat_lane_f16(__p0, __p1) __extension__ ({ \
  float16x4_t __s0 = __p0; \
  float16x4_t __ret; \
  __ret = (float16x4_t) __builtin_neon_splat_lane_v((int8x8_t)__s0, __p1, 8); \
  __ret; \
})
#else
#define splat_lane_f16(__p0, __p1) __extension__ ({ \
  float16x4_t __s0 = __p0; \
  float16x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  float16x4_t __ret; \
  __ret = (float16x4_t) __builtin_neon_splat_lane_v((int8x8_t)__rev0, __p1, 8); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_splat_lane_f16(__p0, __p1) __extension__ ({ \
  float16x4_t __s0 = __p0; \
  float16x4_t __ret; \
  __ret = (float16x4_t) __builtin_neon_splat_lane_v((int8x8_t)__s0, __p1, 8); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define splat_lane_s32(__p0, __p1) __extension__ ({ \
  int32x2_t __s0 = __p0; \
  int32x2_t __ret; \
  __ret = (int32x2_t) __builtin_neon_splat_lane_v((int8x8_t)__s0, __p1, 2); \
  __ret; \
})
#else
#define splat_lane_s32(__p0, __p1) __extension__ ({ \
  int32x2_t __s0 = __p0; \
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  int32x2_t __ret; \
  __ret = (int32x2_t) __builtin_neon_splat_lane_v((int8x8_t)__rev0, __p1, 2); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#define __noswap_splat_lane_s32(__p0, __p1) __extension__ ({ \
  int32x2_t __s0 = __p0; \
  int32x2_t __ret; \
  __ret = (int32x2_t) __builtin_neon_splat_lane_v((int8x8_t)__s0, __p1, 2); \
  __ret; \
})
#endif

#define splat_lane_s64(__p0, __p1) __extension__ ({ \
  int64x1_t __s0 = __p0; \
  int64x1_t __ret; \
  __ret = (int64x1_t) __builtin_neon_splat_lane_v((int8x8_t)__s0, __p1, 3); \
  __ret; \
})
#ifdef __LITTLE_ENDIAN__
#define splat_lane_s16(__p0, __p1) __extension__ ({ \
  int16x4_t __s0 = __p0; \
  int16x4_t __ret; \
  __ret = (int16x4_t) __builtin_neon_splat_lane_v((int8x8_t)__s0, __p1, 1); \
  __ret; \
})
#else
#define splat_lane_s16(__p0, __p1) __extension__ ({ \
  int16x4_t __s0 = __p0; \
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  int16x4_t __ret; \
  __ret = (int16x4_t) __builtin_neon_splat_lane_v((int8x8_t)__rev0, __p1, 1); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_splat_lane_s16(__p0, __p1) __extension__ ({ \
  int16x4_t __s0 = __p0; \
  int16x4_t __ret; \
  __ret = (int16x4_t) __builtin_neon_splat_lane_v((int8x8_t)__s0, __p1, 1); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define splat_laneq_p8(__p0, __p1) __extension__ ({ \
  poly8x16_t __s0 = __p0; \
  poly8x8_t __ret; \
  __ret = (poly8x8_t) __builtin_neon_splat_laneq_v((int8x16_t)__s0, __p1, 36); \
  __ret; \
})
#else
#define splat_laneq_p8(__p0, __p1) __extension__ ({ \
  poly8x16_t __s0 = __p0; \
  poly8x16_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  poly8x8_t __ret; \
  __ret = (poly8x8_t) __builtin_neon_splat_laneq_v((int8x16_t)__rev0, __p1, 36); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_splat_laneq_p8(__p0, __p1) __extension__ ({ \
  poly8x16_t __s0 = __p0; \
  poly8x8_t __ret; \
  __ret = (poly8x8_t) __builtin_neon_splat_laneq_v((int8x16_t)__s0, __p1, 36); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define splat_laneq_p64(__p0, __p1) __extension__ ({ \
  poly64x2_t __s0 = __p0; \
  poly64x1_t __ret; \
  __ret = (poly64x1_t) __builtin_neon_splat_laneq_v((int8x16_t)__s0, __p1, 38); \
  __ret; \
})
#else
#define splat_laneq_p64(__p0, __p1) __extension__ ({ \
  poly64x2_t __s0 = __p0; \
  poly64x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  poly64x1_t __ret; \
  __ret = (poly64x1_t) __builtin_neon_splat_laneq_v((int8x16_t)__rev0, __p1, 38); \
  __ret; \
})
#define __noswap_splat_laneq_p64(__p0, __p1) __extension__ ({ \
  poly64x2_t __s0 = __p0; \
  poly64x1_t __ret; \
  __ret = (poly64x1_t) __builtin_neon_splat_laneq_v((int8x16_t)__s0, __p1, 38); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define splat_laneq_p16(__p0, __p1) __extension__ ({ \
  poly16x8_t __s0 = __p0; \
  poly16x4_t __ret; \
  __ret = (poly16x4_t) __builtin_neon_splat_laneq_v((int8x16_t)__s0, __p1, 37); \
  __ret; \
})
#else
#define splat_laneq_p16(__p0, __p1) __extension__ ({ \
  poly16x8_t __s0 = __p0; \
  poly16x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  poly16x4_t __ret; \
  __ret = (poly16x4_t) __builtin_neon_splat_laneq_v((int8x16_t)__rev0, __p1, 37); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_splat_laneq_p16(__p0, __p1) __extension__ ({ \
  poly16x8_t __s0 = __p0; \
  poly16x4_t __ret; \
  __ret = (poly16x4_t) __builtin_neon_splat_laneq_v((int8x16_t)__s0, __p1, 37); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define splatq_laneq_p8(__p0, __p1) __extension__ ({ \
  poly8x16_t __s0 = __p0; \
  poly8x16_t __ret; \
  __ret = (poly8x16_t) __builtin_neon_splatq_laneq_v((int8x16_t)__s0, __p1, 36); \
  __ret; \
})
#else
#define splatq_laneq_p8(__p0, __p1) __extension__ ({ \
  poly8x16_t __s0 = __p0; \
  poly8x16_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  poly8x16_t __ret; \
  __ret = (poly8x16_t) __builtin_neon_splatq_laneq_v((int8x16_t)__rev0, __p1, 36); \
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_splatq_laneq_p8(__p0, __p1) __extension__ ({ \
  poly8x16_t __s0 = __p0; \
  poly8x16_t __ret; \
  __ret = (poly8x16_t) __builtin_neon_splatq_laneq_v((int8x16_t)__s0, __p1, 36); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define splatq_laneq_p64(__p0, __p1) __extension__ ({ \
  poly64x2_t __s0 = __p0; \
  poly64x2_t __ret; \
  __ret = (poly64x2_t) __builtin_neon_splatq_laneq_v((int8x16_t)__s0, __p1, 38); \
  __ret; \
})
#else
#define splatq_laneq_p64(__p0, __p1) __extension__ ({ \
  poly64x2_t __s0 = __p0; \
  poly64x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  poly64x2_t __ret; \
  __ret = (poly64x2_t) __builtin_neon_splatq_laneq_v((int8x16_t)__rev0, __p1, 38); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#define __noswap_splatq_laneq_p64(__p0, __p1) __extension__ ({ \
  poly64x2_t __s0 = __p0; \
  poly64x2_t __ret; \
  __ret = (poly64x2_t) __builtin_neon_splatq_laneq_v((int8x16_t)__s0, __p1, 38); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define splatq_laneq_p16(__p0, __p1) __extension__ ({ \
  poly16x8_t __s0 = __p0; \
  poly16x8_t __ret; \
  __ret = (poly16x8_t) __builtin_neon_splatq_laneq_v((int8x16_t)__s0, __p1, 37); \
  __ret; \
})
#else
#define splatq_laneq_p16(__p0, __p1) __extension__ ({ \
  poly16x8_t __s0 = __p0; \
  poly16x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  poly16x8_t __ret; \
  __ret = (poly16x8_t) __builtin_neon_splatq_laneq_v((int8x16_t)__rev0, __p1, 37); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_splatq_laneq_p16(__p0, __p1) __extension__ ({ \
  poly16x8_t __s0 = __p0; \
  poly16x8_t __ret; \
  __ret = (poly16x8_t) __builtin_neon_splatq_laneq_v((int8x16_t)__s0, __p1, 37); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define splatq_laneq_u8(__p0, __p1) __extension__ ({ \
  uint8x16_t __s0 = __p0; \
  uint8x16_t __ret; \
  __ret = (uint8x16_t) __builtin_neon_splatq_laneq_v((int8x16_t)__s0, __p1, 48); \
  __ret; \
})
#else
#define splatq_laneq_u8(__p0, __p1) __extension__ ({ \
  uint8x16_t __s0 = __p0; \
  uint8x16_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint8x16_t __ret; \
  __ret = (uint8x16_t) __builtin_neon_splatq_laneq_v((int8x16_t)__rev0, __p1, 48); \
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_splatq_laneq_u8(__p0, __p1) __extension__ ({ \
  uint8x16_t __s0 = __p0; \
  uint8x16_t __ret; \
  __ret = (uint8x16_t) __builtin_neon_splatq_laneq_v((int8x16_t)__s0, __p1, 48); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define splatq_laneq_u32(__p0, __p1) __extension__ ({ \
  uint32x4_t __s0 = __p0; \
  uint32x4_t __ret; \
  __ret = (uint32x4_t) __builtin_neon_splatq_laneq_v((int8x16_t)__s0, __p1, 50); \
  __ret; \
})
#else
#define splatq_laneq_u32(__p0, __p1) __extension__ ({ \
  uint32x4_t __s0 = __p0; \
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  uint32x4_t __ret; \
  __ret = (uint32x4_t) __builtin_neon_splatq_laneq_v((int8x16_t)__rev0, __p1, 50); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_splatq_laneq_u32(__p0, __p1) __extension__ ({ \
  uint32x4_t __s0 = __p0; \
  uint32x4_t __ret; \
  __ret = (uint32x4_t) __builtin_neon_splatq_laneq_v((int8x16_t)__s0, __p1, 50); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define splatq_laneq_u64(__p0, __p1) __extension__ ({ \
  uint64x2_t __s0 = __p0; \
  uint64x2_t __ret; \
  __ret = (uint64x2_t) __builtin_neon_splatq_laneq_v((int8x16_t)__s0, __p1, 51); \
  __ret; \
})
#else
#define splatq_laneq_u64(__p0, __p1) __extension__ ({ \
  uint64x2_t __s0 = __p0; \
  uint64x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  uint64x2_t __ret; \
  __ret = (uint64x2_t) __builtin_neon_splatq_laneq_v((int8x16_t)__rev0, __p1, 51); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#define __noswap_splatq_laneq_u64(__p0, __p1) __extension__ ({ \
  uint64x2_t __s0 = __p0; \
  uint64x2_t __ret; \
  __ret = (uint64x2_t) __builtin_neon_splatq_laneq_v((int8x16_t)__s0, __p1, 51); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define splatq_laneq_u16(__p0, __p1) __extension__ ({ \
  uint16x8_t __s0 = __p0; \
  uint16x8_t __ret; \
  __ret = (uint16x8_t) __builtin_neon_splatq_laneq_v((int8x16_t)__s0, __p1, 49); \
  __ret; \
})
#else
#define splatq_laneq_u16(__p0, __p1) __extension__ ({ \
  uint16x8_t __s0 = __p0; \
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint16x8_t __ret; \
  __ret = (uint16x8_t) __builtin_neon_splatq_laneq_v((int8x16_t)__rev0, __p1, 49); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_splatq_laneq_u16(__p0, __p1) __extension__ ({ \
  uint16x8_t __s0 = __p0; \
  uint16x8_t __ret; \
  __ret = (uint16x8_t) __builtin_neon_splatq_laneq_v((int8x16_t)__s0, __p1, 49); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define splatq_laneq_s8(__p0, __p1) __extension__ ({ \
  int8x16_t __s0 = __p0; \
  int8x16_t __ret; \
  __ret = (int8x16_t) __builtin_neon_splatq_laneq_v((int8x16_t)__s0, __p1, 32); \
  __ret; \
})
#else
#define splatq_laneq_s8(__p0, __p1) __extension__ ({ \
  int8x16_t __s0 = __p0; \
  int8x16_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  int8x16_t __ret; \
  __ret = (int8x16_t) __builtin_neon_splatq_laneq_v((int8x16_t)__rev0, __p1, 32); \
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_splatq_laneq_s8(__p0, __p1) __extension__ ({ \
  int8x16_t __s0 = __p0; \
  int8x16_t __ret; \
  __ret = (int8x16_t) __builtin_neon_splatq_laneq_v((int8x16_t)__s0, __p1, 32); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define splatq_laneq_f64(__p0, __p1) __extension__ ({ \
  float64x2_t __s0 = __p0; \
  float64x2_t __ret; \
  __ret = (float64x2_t) __builtin_neon_splatq_laneq_v((int8x16_t)__s0, __p1, 42); \
  __ret; \
})
#else
#define splatq_laneq_f64(__p0, __p1) __extension__ ({ \
  float64x2_t __s0 = __p0; \
  float64x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  float64x2_t __ret; \
  __ret = (float64x2_t) __builtin_neon_splatq_laneq_v((int8x16_t)__rev0, __p1, 42); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#define __noswap_splatq_laneq_f64(__p0, __p1) __extension__ ({ \
  float64x2_t __s0 = __p0; \
  float64x2_t __ret; \
  __ret = (float64x2_t) __builtin_neon_splatq_laneq_v((int8x16_t)__s0, __p1, 42); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define splatq_laneq_f32(__p0, __p1) __extension__ ({ \
  float32x4_t __s0 = __p0; \
  float32x4_t __ret; \
  __ret = (float32x4_t) __builtin_neon_splatq_laneq_v((int8x16_t)__s0, __p1, 41); \
  __ret; \
})
#else
#define splatq_laneq_f32(__p0, __p1) __extension__ ({ \
  float32x4_t __s0 = __p0; \
  float32x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  float32x4_t __ret; \
  __ret = (float32x4_t) __builtin_neon_splatq_laneq_v((int8x16_t)__rev0, __p1, 41); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_splatq_laneq_f32(__p0, __p1) __extension__ ({ \
  float32x4_t __s0 = __p0; \
  float32x4_t __ret; \
  __ret = (float32x4_t) __builtin_neon_splatq_laneq_v((int8x16_t)__s0, __p1, 41); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define splatq_laneq_f16(__p0, __p1) __extension__ ({ \
  float16x8_t __s0 = __p0; \
  float16x8_t __ret; \
  __ret = (float16x8_t) __builtin_neon_splatq_laneq_v((int8x16_t)__s0, __p1, 40); \
  __ret; \
})
#else
#define splatq_laneq_f16(__p0, __p1) __extension__ ({ \
  float16x8_t __s0 = __p0; \
  float16x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  float16x8_t __ret; \
  __ret = (float16x8_t) __builtin_neon_splatq_laneq_v((int8x16_t)__rev0, __p1, 40); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_splatq_laneq_f16(__p0, __p1) __extension__ ({ \
  float16x8_t __s0 = __p0; \
  float16x8_t __ret; \
  __ret = (float16x8_t) __builtin_neon_splatq_laneq_v((int8x16_t)__s0, __p1, 40); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define splatq_laneq_s32(__p0, __p1) __extension__ ({ \
  int32x4_t __s0 = __p0; \
  int32x4_t __ret; \
  __ret = (int32x4_t) __builtin_neon_splatq_laneq_v((int8x16_t)__s0, __p1, 34); \
  __ret; \
})
#else
#define splatq_laneq_s32(__p0, __p1) __extension__ ({ \
  int32x4_t __s0 = __p0; \
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  int32x4_t __ret; \
  __ret = (int32x4_t) __builtin_neon_splatq_laneq_v((int8x16_t)__rev0, __p1, 34); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_splatq_laneq_s32(__p0, __p1) __extension__ ({ \
  int32x4_t __s0 = __p0; \
  int32x4_t __ret; \
  __ret = (int32x4_t) __builtin_neon_splatq_laneq_v((int8x16_t)__s0, __p1, 34); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define splatq_laneq_s64(__p0, __p1) __extension__ ({ \
  int64x2_t __s0 = __p0; \
  int64x2_t __ret; \
  __ret = (int64x2_t) __builtin_neon_splatq_laneq_v((int8x16_t)__s0, __p1, 35); \
  __ret; \
})
#else
#define splatq_laneq_s64(__p0, __p1) __extension__ ({ \
  int64x2_t __s0 = __p0; \
  int64x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  int64x2_t __ret; \
  __ret = (int64x2_t) __builtin_neon_splatq_laneq_v((int8x16_t)__rev0, __p1, 35); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#define __noswap_splatq_laneq_s64(__p0, __p1) __extension__ ({ \
  int64x2_t __s0 = __p0; \
  int64x2_t __ret; \
  __ret = (int64x2_t) __builtin_neon_splatq_laneq_v((int8x16_t)__s0, __p1, 35); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define splatq_laneq_s16(__p0, __p1) __extension__ ({ \
  int16x8_t __s0 = __p0; \
  int16x8_t __ret; \
  __ret = (int16x8_t) __builtin_neon_splatq_laneq_v((int8x16_t)__s0, __p1, 33); \
  __ret; \
})
#else
#define splatq_laneq_s16(__p0, __p1) __extension__ ({ \
  int16x8_t __s0 = __p0; \
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  int16x8_t __ret; \
  __ret = (int16x8_t) __builtin_neon_splatq_laneq_v((int8x16_t)__rev0, __p1, 33); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_splatq_laneq_s16(__p0, __p1) __extension__ ({ \
  int16x8_t __s0 = __p0; \
  int16x8_t __ret; \
  __ret = (int16x8_t) __builtin_neon_splatq_laneq_v((int8x16_t)__s0, __p1, 33); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define splat_laneq_u8(__p0, __p1) __extension__ ({ \
  uint8x16_t __s0 = __p0; \
  uint8x8_t __ret; \
  __ret = (uint8x8_t) __builtin_neon_splat_laneq_v((int8x16_t)__s0, __p1, 48); \
  __ret; \
})
#else
#define splat_laneq_u8(__p0, __p1) __extension__ ({ \
  uint8x16_t __s0 = __p0; \
  uint8x16_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint8x8_t __ret; \
  __ret = (uint8x8_t) __builtin_neon_splat_laneq_v((int8x16_t)__rev0, __p1, 48); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_splat_laneq_u8(__p0, __p1) __extension__ ({ \
  uint8x16_t __s0 = __p0; \
  uint8x8_t __ret; \
  __ret = (uint8x8_t) __builtin_neon_splat_laneq_v((int8x16_t)__s0, __p1, 48); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define splat_laneq_u32(__p0, __p1) __extension__ ({ \
  uint32x4_t __s0 = __p0; \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_splat_laneq_v((int8x16_t)__s0, __p1, 50); \
  __ret; \
})
#else
#define splat_laneq_u32(__p0, __p1) __extension__ ({ \
  uint32x4_t __s0 = __p0; \
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_splat_laneq_v((int8x16_t)__rev0, __p1, 50); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#define __noswap_splat_laneq_u32(__p0, __p1) __extension__ ({ \
  uint32x4_t __s0 = __p0; \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_splat_laneq_v((int8x16_t)__s0, __p1, 50); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define splat_laneq_u64(__p0, __p1) __extension__ ({ \
  uint64x2_t __s0 = __p0; \
  uint64x1_t __ret; \
  __ret = (uint64x1_t) __builtin_neon_splat_laneq_v((int8x16_t)__s0, __p1, 51); \
  __ret; \
})
#else
#define splat_laneq_u64(__p0, __p1) __extension__ ({ \
  uint64x2_t __s0 = __p0; \
  uint64x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  uint64x1_t __ret; \
  __ret = (uint64x1_t) __builtin_neon_splat_laneq_v((int8x16_t)__rev0, __p1, 51); \
  __ret; \
})
#define __noswap_splat_laneq_u64(__p0, __p1) __extension__ ({ \
  uint64x2_t __s0 = __p0; \
  uint64x1_t __ret; \
  __ret = (uint64x1_t) __builtin_neon_splat_laneq_v((int8x16_t)__s0, __p1, 51); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define splat_laneq_u16(__p0, __p1) __extension__ ({ \
  uint16x8_t __s0 = __p0; \
  uint16x4_t __ret; \
  __ret = (uint16x4_t) __builtin_neon_splat_laneq_v((int8x16_t)__s0, __p1, 49); \
  __ret; \
})
#else
#define splat_laneq_u16(__p0, __p1) __extension__ ({ \
  uint16x8_t __s0 = __p0; \
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint16x4_t __ret; \
  __ret = (uint16x4_t) __builtin_neon_splat_laneq_v((int8x16_t)__rev0, __p1, 49); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_splat_laneq_u16(__p0, __p1) __extension__ ({ \
  uint16x8_t __s0 = __p0; \
  uint16x4_t __ret; \
  __ret = (uint16x4_t) __builtin_neon_splat_laneq_v((int8x16_t)__s0, __p1, 49); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define splat_laneq_s8(__p0, __p1) __extension__ ({ \
  int8x16_t __s0 = __p0; \
  int8x8_t __ret; \
  __ret = (int8x8_t) __builtin_neon_splat_laneq_v((int8x16_t)__s0, __p1, 32); \
  __ret; \
})
#else
#define splat_laneq_s8(__p0, __p1) __extension__ ({ \
  int8x16_t __s0 = __p0; \
  int8x16_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  int8x8_t __ret; \
  __ret = (int8x8_t) __builtin_neon_splat_laneq_v((int8x16_t)__rev0, __p1, 32); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_splat_laneq_s8(__p0, __p1) __extension__ ({ \
  int8x16_t __s0 = __p0; \
  int8x8_t __ret; \
  __ret = (int8x8_t) __builtin_neon_splat_laneq_v((int8x16_t)__s0, __p1, 32); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define splat_laneq_f64(__p0, __p1) __extension__ ({ \
  float64x2_t __s0 = __p0; \
  float64x1_t __ret; \
  __ret = (float64x1_t) __builtin_neon_splat_laneq_v((int8x16_t)__s0, __p1, 42); \
  __ret; \
})
#else
#define splat_laneq_f64(__p0, __p1) __extension__ ({ \
  float64x2_t __s0 = __p0; \
  float64x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  float64x1_t __ret; \
  __ret = (float64x1_t) __builtin_neon_splat_laneq_v((int8x16_t)__rev0, __p1, 42); \
  __ret; \
})
#define __noswap_splat_laneq_f64(__p0, __p1) __extension__ ({ \
  float64x2_t __s0 = __p0; \
  float64x1_t __ret; \
  __ret = (float64x1_t) __builtin_neon_splat_laneq_v((int8x16_t)__s0, __p1, 42); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define splat_laneq_f32(__p0, __p1) __extension__ ({ \
  float32x4_t __s0 = __p0; \
  float32x2_t __ret; \
  __ret = (float32x2_t) __builtin_neon_splat_laneq_v((int8x16_t)__s0, __p1, 41); \
  __ret; \
})
#else
#define splat_laneq_f32(__p0, __p1) __extension__ ({ \
  float32x4_t __s0 = __p0; \
  float32x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  float32x2_t __ret; \
  __ret = (float32x2_t) __builtin_neon_splat_laneq_v((int8x16_t)__rev0, __p1, 41); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#define __noswap_splat_laneq_f32(__p0, __p1) __extension__ ({ \
  float32x4_t __s0 = __p0; \
  float32x2_t __ret; \
  __ret = (float32x2_t) __builtin_neon_splat_laneq_v((int8x16_t)__s0, __p1, 41); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define splat_laneq_f16(__p0, __p1) __extension__ ({ \
  float16x8_t __s0 = __p0; \
  float16x4_t __ret; \
  __ret = (float16x4_t) __builtin_neon_splat_laneq_v((int8x16_t)__s0, __p1, 40); \
  __ret; \
})
#else
#define splat_laneq_f16(__p0, __p1) __extension__ ({ \
  float16x8_t __s0 = __p0; \
  float16x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  float16x4_t __ret; \
  __ret = (float16x4_t) __builtin_neon_splat_laneq_v((int8x16_t)__rev0, __p1, 40); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_splat_laneq_f16(__p0, __p1) __extension__ ({ \
  float16x8_t __s0 = __p0; \
  float16x4_t __ret; \
  __ret = (float16x4_t) __builtin_neon_splat_laneq_v((int8x16_t)__s0, __p1, 40); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define splat_laneq_s32(__p0, __p1) __extension__ ({ \
  int32x4_t __s0 = __p0; \
  int32x2_t __ret; \
  __ret = (int32x2_t) __builtin_neon_splat_laneq_v((int8x16_t)__s0, __p1, 34); \
  __ret; \
})
#else
#define splat_laneq_s32(__p0, __p1) __extension__ ({ \
  int32x4_t __s0 = __p0; \
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  int32x2_t __ret; \
  __ret = (int32x2_t) __builtin_neon_splat_laneq_v((int8x16_t)__rev0, __p1, 34); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#define __noswap_splat_laneq_s32(__p0, __p1) __extension__ ({ \
  int32x4_t __s0 = __p0; \
  int32x2_t __ret; \
  __ret = (int32x2_t) __builtin_neon_splat_laneq_v((int8x16_t)__s0, __p1, 34); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define splat_laneq_s64(__p0, __p1) __extension__ ({ \
  int64x2_t __s0 = __p0; \
  int64x1_t __ret; \
  __ret = (int64x1_t) __builtin_neon_splat_laneq_v((int8x16_t)__s0, __p1, 35); \
  __ret; \
})
#else
#define splat_laneq_s64(__p0, __p1) __extension__ ({ \
  int64x2_t __s0 = __p0; \
  int64x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  int64x1_t __ret; \
  __ret = (int64x1_t) __builtin_neon_splat_laneq_v((int8x16_t)__rev0, __p1, 35); \
  __ret; \
})
#define __noswap_splat_laneq_s64(__p0, __p1) __extension__ ({ \
  int64x2_t __s0 = __p0; \
  int64x1_t __ret; \
  __ret = (int64x1_t) __builtin_neon_splat_laneq_v((int8x16_t)__s0, __p1, 35); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define splat_laneq_s16(__p0, __p1) __extension__ ({ \
  int16x8_t __s0 = __p0; \
  int16x4_t __ret; \
  __ret = (int16x4_t) __builtin_neon_splat_laneq_v((int8x16_t)__s0, __p1, 33); \
  __ret; \
})
#else
#define splat_laneq_s16(__p0, __p1) __extension__ ({ \
  int16x8_t __s0 = __p0; \
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  int16x4_t __ret; \
  __ret = (int16x4_t) __builtin_neon_splat_laneq_v((int8x16_t)__rev0, __p1, 33); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_splat_laneq_s16(__p0, __p1) __extension__ ({ \
  int16x8_t __s0 = __p0; \
  int16x4_t __ret; \
  __ret = (int16x4_t) __builtin_neon_splat_laneq_v((int8x16_t)__s0, __p1, 33); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x16_t vabdq_u8(uint8x16_t __p0, uint8x16_t __p1) {
  uint8x16_t __ret;
  __ret = (uint8x16_t) __builtin_neon_vabdq_v((int8x16_t)__p0, (int8x16_t)__p1, 48);
  return __ret;
}
#else
__ai uint8x16_t vabdq_u8(uint8x16_t __p0, uint8x16_t __p1) {
  uint8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __ret;
  __ret = (uint8x16_t) __builtin_neon_vabdq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 48);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
__ai uint8x16_t __noswap_vabdq_u8(uint8x16_t __p0, uint8x16_t __p1) {
  uint8x16_t __ret;
  __ret = (uint8x16_t) __builtin_neon_vabdq_v((int8x16_t)__p0, (int8x16_t)__p1, 48);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vabdq_u32(uint32x4_t __p0, uint32x4_t __p1) {
  uint32x4_t __ret;
  __ret = (uint32x4_t) __builtin_neon_vabdq_v((int8x16_t)__p0, (int8x16_t)__p1, 50);
  return __ret;
}
#else
__ai uint32x4_t vabdq_u32(uint32x4_t __p0, uint32x4_t __p1) {
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = (uint32x4_t) __builtin_neon_vabdq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 50);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
__ai uint32x4_t __noswap_vabdq_u32(uint32x4_t __p0, uint32x4_t __p1) {
  uint32x4_t __ret;
  __ret = (uint32x4_t) __builtin_neon_vabdq_v((int8x16_t)__p0, (int8x16_t)__p1, 50);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x8_t vabdq_u16(uint16x8_t __p0, uint16x8_t __p1) {
  uint16x8_t __ret;
  __ret = (uint16x8_t) __builtin_neon_vabdq_v((int8x16_t)__p0, (int8x16_t)__p1, 49);
  return __ret;
}
#else
__ai uint16x8_t vabdq_u16(uint16x8_t __p0, uint16x8_t __p1) {
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __ret;
  __ret = (uint16x8_t) __builtin_neon_vabdq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 49);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
__ai uint16x8_t __noswap_vabdq_u16(uint16x8_t __p0, uint16x8_t __p1) {
  uint16x8_t __ret;
  __ret = (uint16x8_t) __builtin_neon_vabdq_v((int8x16_t)__p0, (int8x16_t)__p1, 49);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x16_t vabdq_s8(int8x16_t __p0, int8x16_t __p1) {
  int8x16_t __ret;
  __ret = (int8x16_t) __builtin_neon_vabdq_v((int8x16_t)__p0, (int8x16_t)__p1, 32);
  return __ret;
}
#else
__ai int8x16_t vabdq_s8(int8x16_t __p0, int8x16_t __p1) {
  int8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __ret;
  __ret = (int8x16_t) __builtin_neon_vabdq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 32);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
__ai int8x16_t __noswap_vabdq_s8(int8x16_t __p0, int8x16_t __p1) {
  int8x16_t __ret;
  __ret = (int8x16_t) __builtin_neon_vabdq_v((int8x16_t)__p0, (int8x16_t)__p1, 32);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai float32x4_t vabdq_f32(float32x4_t __p0, float32x4_t __p1) {
  float32x4_t __ret;
  __ret = (float32x4_t) __builtin_neon_vabdq_v((int8x16_t)__p0, (int8x16_t)__p1, 41);
  return __ret;
}
#else
__ai float32x4_t vabdq_f32(float32x4_t __p0, float32x4_t __p1) {
  float32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  float32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  float32x4_t __ret;
  __ret = (float32x4_t) __builtin_neon_vabdq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 41);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x4_t vabdq_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vabdq_v((int8x16_t)__p0, (int8x16_t)__p1, 34);
  return __ret;
}
#else
__ai int32x4_t vabdq_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vabdq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 34);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
__ai int32x4_t __noswap_vabdq_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vabdq_v((int8x16_t)__p0, (int8x16_t)__p1, 34);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x8_t vabdq_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vabdq_v((int8x16_t)__p0, (int8x16_t)__p1, 33);
  return __ret;
}
#else
__ai int16x8_t vabdq_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vabdq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 33);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
__ai int16x8_t __noswap_vabdq_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vabdq_v((int8x16_t)__p0, (int8x16_t)__p1, 33);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x8_t vabd_u8(uint8x8_t __p0, uint8x8_t __p1) {
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vabd_v((int8x8_t)__p0, (int8x8_t)__p1, 16);
  return __ret;
}
#else
__ai uint8x8_t vabd_u8(uint8x8_t __p0, uint8x8_t __p1) {
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vabd_v((int8x8_t)__rev0, (int8x8_t)__rev1, 16);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
__ai uint8x8_t __noswap_vabd_u8(uint8x8_t __p0, uint8x8_t __p1) {
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vabd_v((int8x8_t)__p0, (int8x8_t)__p1, 16);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vabd_u32(uint32x2_t __p0, uint32x2_t __p1) {
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vabd_v((int8x8_t)__p0, (int8x8_t)__p1, 18);
  return __ret;
}
#else
__ai uint32x2_t vabd_u32(uint32x2_t __p0, uint32x2_t __p1) {
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vabd_v((int8x8_t)__rev0, (int8x8_t)__rev1, 18);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
__ai uint32x2_t __noswap_vabd_u32(uint32x2_t __p0, uint32x2_t __p1) {
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vabd_v((int8x8_t)__p0, (int8x8_t)__p1, 18);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x4_t vabd_u16(uint16x4_t __p0, uint16x4_t __p1) {
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vabd_v((int8x8_t)__p0, (int8x8_t)__p1, 17);
  return __ret;
}
#else
__ai uint16x4_t vabd_u16(uint16x4_t __p0, uint16x4_t __p1) {
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vabd_v((int8x8_t)__rev0, (int8x8_t)__rev1, 17);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
__ai uint16x4_t __noswap_vabd_u16(uint16x4_t __p0, uint16x4_t __p1) {
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vabd_v((int8x8_t)__p0, (int8x8_t)__p1, 17);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x8_t vabd_s8(int8x8_t __p0, int8x8_t __p1) {
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vabd_v((int8x8_t)__p0, (int8x8_t)__p1, 0);
  return __ret;
}
#else
__ai int8x8_t vabd_s8(int8x8_t __p0, int8x8_t __p1) {
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vabd_v((int8x8_t)__rev0, (int8x8_t)__rev1, 0);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
__ai int8x8_t __noswap_vabd_s8(int8x8_t __p0, int8x8_t __p1) {
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vabd_v((int8x8_t)__p0, (int8x8_t)__p1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai float32x2_t vabd_f32(float32x2_t __p0, float32x2_t __p1) {
  float32x2_t __ret;
  __ret = (float32x2_t) __builtin_neon_vabd_v((int8x8_t)__p0, (int8x8_t)__p1, 9);
  return __ret;
}
#else
__ai float32x2_t vabd_f32(float32x2_t __p0, float32x2_t __p1) {
  float32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  float32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  float32x2_t __ret;
  __ret = (float32x2_t) __builtin_neon_vabd_v((int8x8_t)__rev0, (int8x8_t)__rev1, 9);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x2_t vabd_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vabd_v((int8x8_t)__p0, (int8x8_t)__p1, 2);
  return __ret;
}
#else
__ai int32x2_t vabd_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vabd_v((int8x8_t)__rev0, (int8x8_t)__rev1, 2);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
__ai int32x2_t __noswap_vabd_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vabd_v((int8x8_t)__p0, (int8x8_t)__p1, 2);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x4_t vabd_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vabd_v((int8x8_t)__p0, (int8x8_t)__p1, 1);
  return __ret;
}
#else
__ai int16x4_t vabd_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vabd_v((int8x8_t)__rev0, (int8x8_t)__rev1, 1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
__ai int16x4_t __noswap_vabd_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vabd_v((int8x8_t)__p0, (int8x8_t)__p1, 1);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x16_t vabsq_s8(int8x16_t __p0) {
  int8x16_t __ret;
  __ret = (int8x16_t) __builtin_neon_vabsq_v((int8x16_t)__p0, 32);
  return __ret;
}
#else
__ai int8x16_t vabsq_s8(int8x16_t __p0) {
  int8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __ret;
  __ret = (int8x16_t) __builtin_neon_vabsq_v((int8x16_t)__rev0, 32);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai float32x4_t vabsq_f32(float32x4_t __p0) {
  float32x4_t __ret;
  __ret = (float32x4_t) __builtin_neon_vabsq_v((int8x16_t)__p0, 41);
  return __ret;
}
#else
__ai float32x4_t vabsq_f32(float32x4_t __p0) {
  float32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  float32x4_t __ret;
  __ret = (float32x4_t) __builtin_neon_vabsq_v((int8x16_t)__rev0, 41);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x4_t vabsq_s32(int32x4_t __p0) {
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vabsq_v((int8x16_t)__p0, 34);
  return __ret;
}
#else
__ai int32x4_t vabsq_s32(int32x4_t __p0) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vabsq_v((int8x16_t)__rev0, 34);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x8_t vabsq_s16(int16x8_t __p0) {
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vabsq_v((int8x16_t)__p0, 33);
  return __ret;
}
#else
__ai int16x8_t vabsq_s16(int16x8_t __p0) {
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vabsq_v((int8x16_t)__rev0, 33);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x8_t vabs_s8(int8x8_t __p0) {
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vabs_v((int8x8_t)__p0, 0);
  return __ret;
}
#else
__ai int8x8_t vabs_s8(int8x8_t __p0) {
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vabs_v((int8x8_t)__rev0, 0);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai float32x2_t vabs_f32(float32x2_t __p0) {
  float32x2_t __ret;
  __ret = (float32x2_t) __builtin_neon_vabs_v((int8x8_t)__p0, 9);
  return __ret;
}
#else
__ai float32x2_t vabs_f32(float32x2_t __p0) {
  float32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  float32x2_t __ret;
  __ret = (float32x2_t) __builtin_neon_vabs_v((int8x8_t)__rev0, 9);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x2_t vabs_s32(int32x2_t __p0) {
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vabs_v((int8x8_t)__p0, 2);
  return __ret;
}
#else
__ai int32x2_t vabs_s32(int32x2_t __p0) {
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vabs_v((int8x8_t)__rev0, 2);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x4_t vabs_s16(int16x4_t __p0) {
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vabs_v((int8x8_t)__p0, 1);
  return __ret;
}
#else
__ai int16x4_t vabs_s16(int16x4_t __p0) {
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vabs_v((int8x8_t)__rev0, 1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x16_t vaddq_u8(uint8x16_t __p0, uint8x16_t __p1) {
  uint8x16_t __ret;
  __ret = __p0 + __p1;
  return __ret;
}
#else
__ai uint8x16_t vaddq_u8(uint8x16_t __p0, uint8x16_t __p1) {
  uint8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __ret;
  __ret = __rev0 + __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vaddq_u32(uint32x4_t __p0, uint32x4_t __p1) {
  uint32x4_t __ret;
  __ret = __p0 + __p1;
  return __ret;
}
#else
__ai uint32x4_t vaddq_u32(uint32x4_t __p0, uint32x4_t __p1) {
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = __rev0 + __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint64x2_t vaddq_u64(uint64x2_t __p0, uint64x2_t __p1) {
  uint64x2_t __ret;
  __ret = __p0 + __p1;
  return __ret;
}
#else
__ai uint64x2_t vaddq_u64(uint64x2_t __p0, uint64x2_t __p1) {
  uint64x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint64x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint64x2_t __ret;
  __ret = __rev0 + __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x8_t vaddq_u16(uint16x8_t __p0, uint16x8_t __p1) {
  uint16x8_t __ret;
  __ret = __p0 + __p1;
  return __ret;
}
#else
__ai uint16x8_t vaddq_u16(uint16x8_t __p0, uint16x8_t __p1) {
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __ret;
  __ret = __rev0 + __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x16_t vaddq_s8(int8x16_t __p0, int8x16_t __p1) {
  int8x16_t __ret;
  __ret = __p0 + __p1;
  return __ret;
}
#else
__ai int8x16_t vaddq_s8(int8x16_t __p0, int8x16_t __p1) {
  int8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __ret;
  __ret = __rev0 + __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai float32x4_t vaddq_f32(float32x4_t __p0, float32x4_t __p1) {
  float32x4_t __ret;
  __ret = __p0 + __p1;
  return __ret;
}
#else
__ai float32x4_t vaddq_f32(float32x4_t __p0, float32x4_t __p1) {
  float32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  float32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  float32x4_t __ret;
  __ret = __rev0 + __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x4_t vaddq_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __ret;
  __ret = __p0 + __p1;
  return __ret;
}
#else
__ai int32x4_t vaddq_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int32x4_t __ret;
  __ret = __rev0 + __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int64x2_t vaddq_s64(int64x2_t __p0, int64x2_t __p1) {
  int64x2_t __ret;
  __ret = __p0 + __p1;
  return __ret;
}
#else
__ai int64x2_t vaddq_s64(int64x2_t __p0, int64x2_t __p1) {
  int64x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int64x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  int64x2_t __ret;
  __ret = __rev0 + __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x8_t vaddq_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __ret;
  __ret = __p0 + __p1;
  return __ret;
}
#else
__ai int16x8_t vaddq_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __ret;
  __ret = __rev0 + __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x8_t vadd_u8(uint8x8_t __p0, uint8x8_t __p1) {
  uint8x8_t __ret;
  __ret = __p0 + __p1;
  return __ret;
}
#else
__ai uint8x8_t vadd_u8(uint8x8_t __p0, uint8x8_t __p1) {
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __ret;
  __ret = __rev0 + __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vadd_u32(uint32x2_t __p0, uint32x2_t __p1) {
  uint32x2_t __ret;
  __ret = __p0 + __p1;
  return __ret;
}
#else
__ai uint32x2_t vadd_u32(uint32x2_t __p0, uint32x2_t __p1) {
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint32x2_t __ret;
  __ret = __rev0 + __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

__ai uint64x1_t vadd_u64(uint64x1_t __p0, uint64x1_t __p1) {
  uint64x1_t __ret;
  __ret = __p0 + __p1;
  return __ret;
}
#ifdef __LITTLE_ENDIAN__
__ai uint16x4_t vadd_u16(uint16x4_t __p0, uint16x4_t __p1) {
  uint16x4_t __ret;
  __ret = __p0 + __p1;
  return __ret;
}
#else
__ai uint16x4_t vadd_u16(uint16x4_t __p0, uint16x4_t __p1) {
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint16x4_t __ret;
  __ret = __rev0 + __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x8_t vadd_s8(int8x8_t __p0, int8x8_t __p1) {
  int8x8_t __ret;
  __ret = __p0 + __p1;
  return __ret;
}
#else
__ai int8x8_t vadd_s8(int8x8_t __p0, int8x8_t __p1) {
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __ret;
  __ret = __rev0 + __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai float32x2_t vadd_f32(float32x2_t __p0, float32x2_t __p1) {
  float32x2_t __ret;
  __ret = __p0 + __p1;
  return __ret;
}
#else
__ai float32x2_t vadd_f32(float32x2_t __p0, float32x2_t __p1) {
  float32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  float32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  float32x2_t __ret;
  __ret = __rev0 + __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x2_t vadd_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __ret;
  __ret = __p0 + __p1;
  return __ret;
}
#else
__ai int32x2_t vadd_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  int32x2_t __ret;
  __ret = __rev0 + __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

__ai int64x1_t vadd_s64(int64x1_t __p0, int64x1_t __p1) {
  int64x1_t __ret;
  __ret = __p0 + __p1;
  return __ret;
}
#ifdef __LITTLE_ENDIAN__
__ai int16x4_t vadd_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __ret;
  __ret = __p0 + __p1;
  return __ret;
}
#else
__ai int16x4_t vadd_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int16x4_t __ret;
  __ret = __rev0 + __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x4_t vaddhn_u32(uint32x4_t __p0, uint32x4_t __p1) {
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vaddhn_v((int8x16_t)__p0, (int8x16_t)__p1, 17);
  return __ret;
}
#else
__ai uint16x4_t vaddhn_u32(uint32x4_t __p0, uint32x4_t __p1) {
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vaddhn_v((int8x16_t)__rev0, (int8x16_t)__rev1, 17);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
__ai uint16x4_t __noswap_vaddhn_u32(uint32x4_t __p0, uint32x4_t __p1) {
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vaddhn_v((int8x16_t)__p0, (int8x16_t)__p1, 17);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vaddhn_u64(uint64x2_t __p0, uint64x2_t __p1) {
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vaddhn_v((int8x16_t)__p0, (int8x16_t)__p1, 18);
  return __ret;
}
#else
__ai uint32x2_t vaddhn_u64(uint64x2_t __p0, uint64x2_t __p1) {
  uint64x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint64x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vaddhn_v((int8x16_t)__rev0, (int8x16_t)__rev1, 18);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
__ai uint32x2_t __noswap_vaddhn_u64(uint64x2_t __p0, uint64x2_t __p1) {
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vaddhn_v((int8x16_t)__p0, (int8x16_t)__p1, 18);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x8_t vaddhn_u16(uint16x8_t __p0, uint16x8_t __p1) {
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vaddhn_v((int8x16_t)__p0, (int8x16_t)__p1, 16);
  return __ret;
}
#else
__ai uint8x8_t vaddhn_u16(uint16x8_t __p0, uint16x8_t __p1) {
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vaddhn_v((int8x16_t)__rev0, (int8x16_t)__rev1, 16);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
__ai uint8x8_t __noswap_vaddhn_u16(uint16x8_t __p0, uint16x8_t __p1) {
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vaddhn_v((int8x16_t)__p0, (int8x16_t)__p1, 16);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x4_t vaddhn_s32(int32x4_t __p0, int32x4_t __p1) {
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vaddhn_v((int8x16_t)__p0, (int8x16_t)__p1, 1);
  return __ret;
}
#else
__ai int16x4_t vaddhn_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vaddhn_v((int8x16_t)__rev0, (int8x16_t)__rev1, 1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
__ai int16x4_t __noswap_vaddhn_s32(int32x4_t __p0, int32x4_t __p1) {
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vaddhn_v((int8x16_t)__p0, (int8x16_t)__p1, 1);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x2_t vaddhn_s64(int64x2_t __p0, int64x2_t __p1) {
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vaddhn_v((int8x16_t)__p0, (int8x16_t)__p1, 2);
  return __ret;
}
#else
__ai int32x2_t vaddhn_s64(int64x2_t __p0, int64x2_t __p1) {
  int64x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int64x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vaddhn_v((int8x16_t)__rev0, (int8x16_t)__rev1, 2);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
__ai int32x2_t __noswap_vaddhn_s64(int64x2_t __p0, int64x2_t __p1) {
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vaddhn_v((int8x16_t)__p0, (int8x16_t)__p1, 2);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x8_t vaddhn_s16(int16x8_t __p0, int16x8_t __p1) {
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vaddhn_v((int8x16_t)__p0, (int8x16_t)__p1, 0);
  return __ret;
}
#else
__ai int8x8_t vaddhn_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vaddhn_v((int8x16_t)__rev0, (int8x16_t)__rev1, 0);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
__ai int8x8_t __noswap_vaddhn_s16(int16x8_t __p0, int16x8_t __p1) {
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vaddhn_v((int8x16_t)__p0, (int8x16_t)__p1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x16_t vandq_u8(uint8x16_t __p0, uint8x16_t __p1) {
  uint8x16_t __ret;
  __ret = __p0 & __p1;
  return __ret;
}
#else
__ai uint8x16_t vandq_u8(uint8x16_t __p0, uint8x16_t __p1) {
  uint8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __ret;
  __ret = __rev0 & __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vandq_u32(uint32x4_t __p0, uint32x4_t __p1) {
  uint32x4_t __ret;
  __ret = __p0 & __p1;
  return __ret;
}
#else
__ai uint32x4_t vandq_u32(uint32x4_t __p0, uint32x4_t __p1) {
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = __rev0 & __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint64x2_t vandq_u64(uint64x2_t __p0, uint64x2_t __p1) {
  uint64x2_t __ret;
  __ret = __p0 & __p1;
  return __ret;
}
#else
__ai uint64x2_t vandq_u64(uint64x2_t __p0, uint64x2_t __p1) {
  uint64x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint64x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint64x2_t __ret;
  __ret = __rev0 & __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x8_t vandq_u16(uint16x8_t __p0, uint16x8_t __p1) {
  uint16x8_t __ret;
  __ret = __p0 & __p1;
  return __ret;
}
#else
__ai uint16x8_t vandq_u16(uint16x8_t __p0, uint16x8_t __p1) {
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __ret;
  __ret = __rev0 & __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x16_t vandq_s8(int8x16_t __p0, int8x16_t __p1) {
  int8x16_t __ret;
  __ret = __p0 & __p1;
  return __ret;
}
#else
__ai int8x16_t vandq_s8(int8x16_t __p0, int8x16_t __p1) {
  int8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __ret;
  __ret = __rev0 & __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x4_t vandq_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __ret;
  __ret = __p0 & __p1;
  return __ret;
}
#else
__ai int32x4_t vandq_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int32x4_t __ret;
  __ret = __rev0 & __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int64x2_t vandq_s64(int64x2_t __p0, int64x2_t __p1) {
  int64x2_t __ret;
  __ret = __p0 & __p1;
  return __ret;
}
#else
__ai int64x2_t vandq_s64(int64x2_t __p0, int64x2_t __p1) {
  int64x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int64x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  int64x2_t __ret;
  __ret = __rev0 & __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x8_t vandq_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __ret;
  __ret = __p0 & __p1;
  return __ret;
}
#else
__ai int16x8_t vandq_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __ret;
  __ret = __rev0 & __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x8_t vand_u8(uint8x8_t __p0, uint8x8_t __p1) {
  uint8x8_t __ret;
  __ret = __p0 & __p1;
  return __ret;
}
#else
__ai uint8x8_t vand_u8(uint8x8_t __p0, uint8x8_t __p1) {
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __ret;
  __ret = __rev0 & __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vand_u32(uint32x2_t __p0, uint32x2_t __p1) {
  uint32x2_t __ret;
  __ret = __p0 & __p1;
  return __ret;
}
#else
__ai uint32x2_t vand_u32(uint32x2_t __p0, uint32x2_t __p1) {
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint32x2_t __ret;
  __ret = __rev0 & __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

__ai uint64x1_t vand_u64(uint64x1_t __p0, uint64x1_t __p1) {
  uint64x1_t __ret;
  __ret = __p0 & __p1;
  return __ret;
}
#ifdef __LITTLE_ENDIAN__
__ai uint16x4_t vand_u16(uint16x4_t __p0, uint16x4_t __p1) {
  uint16x4_t __ret;
  __ret = __p0 & __p1;
  return __ret;
}
#else
__ai uint16x4_t vand_u16(uint16x4_t __p0, uint16x4_t __p1) {
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint16x4_t __ret;
  __ret = __rev0 & __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x8_t vand_s8(int8x8_t __p0, int8x8_t __p1) {
  int8x8_t __ret;
  __ret = __p0 & __p1;
  return __ret;
}
#else
__ai int8x8_t vand_s8(int8x8_t __p0, int8x8_t __p1) {
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __ret;
  __ret = __rev0 & __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x2_t vand_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __ret;
  __ret = __p0 & __p1;
  return __ret;
}
#else
__ai int32x2_t vand_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  int32x2_t __ret;
  __ret = __rev0 & __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

__ai int64x1_t vand_s64(int64x1_t __p0, int64x1_t __p1) {
  int64x1_t __ret;
  __ret = __p0 & __p1;
  return __ret;
}
#ifdef __LITTLE_ENDIAN__
__ai int16x4_t vand_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __ret;
  __ret = __p0 & __p1;
  return __ret;
}
#else
__ai int16x4_t vand_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int16x4_t __ret;
  __ret = __rev0 & __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x16_t vbicq_u8(uint8x16_t __p0, uint8x16_t __p1) {
  uint8x16_t __ret;
  __ret = __p0 & ~__p1;
  return __ret;
}
#else
__ai uint8x16_t vbicq_u8(uint8x16_t __p0, uint8x16_t __p1) {
  uint8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __ret;
  __ret = __rev0 & ~__rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vbicq_u32(uint32x4_t __p0, uint32x4_t __p1) {
  uint32x4_t __ret;
  __ret = __p0 & ~__p1;
  return __ret;
}
#else
__ai uint32x4_t vbicq_u32(uint32x4_t __p0, uint32x4_t __p1) {
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = __rev0 & ~__rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint64x2_t vbicq_u64(uint64x2_t __p0, uint64x2_t __p1) {
  uint64x2_t __ret;
  __ret = __p0 & ~__p1;
  return __ret;
}
#else
__ai uint64x2_t vbicq_u64(uint64x2_t __p0, uint64x2_t __p1) {
  uint64x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint64x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint64x2_t __ret;
  __ret = __rev0 & ~__rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x8_t vbicq_u16(uint16x8_t __p0, uint16x8_t __p1) {
  uint16x8_t __ret;
  __ret = __p0 & ~__p1;
  return __ret;
}
#else
__ai uint16x8_t vbicq_u16(uint16x8_t __p0, uint16x8_t __p1) {
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __ret;
  __ret = __rev0 & ~__rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x16_t vbicq_s8(int8x16_t __p0, int8x16_t __p1) {
  int8x16_t __ret;
  __ret = __p0 & ~__p1;
  return __ret;
}
#else
__ai int8x16_t vbicq_s8(int8x16_t __p0, int8x16_t __p1) {
  int8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __ret;
  __ret = __rev0 & ~__rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x4_t vbicq_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __ret;
  __ret = __p0 & ~__p1;
  return __ret;
}
#else
__ai int32x4_t vbicq_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int32x4_t __ret;
  __ret = __rev0 & ~__rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int64x2_t vbicq_s64(int64x2_t __p0, int64x2_t __p1) {
  int64x2_t __ret;
  __ret = __p0 & ~__p1;
  return __ret;
}
#else
__ai int64x2_t vbicq_s64(int64x2_t __p0, int64x2_t __p1) {
  int64x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int64x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  int64x2_t __ret;
  __ret = __rev0 & ~__rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x8_t vbicq_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __ret;
  __ret = __p0 & ~__p1;
  return __ret;
}
#else
__ai int16x8_t vbicq_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __ret;
  __ret = __rev0 & ~__rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x8_t vbic_u8(uint8x8_t __p0, uint8x8_t __p1) {
  uint8x8_t __ret;
  __ret = __p0 & ~__p1;
  return __ret;
}
#else
__ai uint8x8_t vbic_u8(uint8x8_t __p0, uint8x8_t __p1) {
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __ret;
  __ret = __rev0 & ~__rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vbic_u32(uint32x2_t __p0, uint32x2_t __p1) {
  uint32x2_t __ret;
  __ret = __p0 & ~__p1;
  return __ret;
}
#else
__ai uint32x2_t vbic_u32(uint32x2_t __p0, uint32x2_t __p1) {
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint32x2_t __ret;
  __ret = __rev0 & ~__rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

__ai uint64x1_t vbic_u64(uint64x1_t __p0, uint64x1_t __p1) {
  uint64x1_t __ret;
  __ret = __p0 & ~__p1;
  return __ret;
}
#ifdef __LITTLE_ENDIAN__
__ai uint16x4_t vbic_u16(uint16x4_t __p0, uint16x4_t __p1) {
  uint16x4_t __ret;
  __ret = __p0 & ~__p1;
  return __ret;
}
#else
__ai uint16x4_t vbic_u16(uint16x4_t __p0, uint16x4_t __p1) {
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint16x4_t __ret;
  __ret = __rev0 & ~__rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x8_t vbic_s8(int8x8_t __p0, int8x8_t __p1) {
  int8x8_t __ret;
  __ret = __p0 & ~__p1;
  return __ret;
}
#else
__ai int8x8_t vbic_s8(int8x8_t __p0, int8x8_t __p1) {
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __ret;
  __ret = __rev0 & ~__rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x2_t vbic_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __ret;
  __ret = __p0 & ~__p1;
  return __ret;
}
#else
__ai int32x2_t vbic_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  int32x2_t __ret;
  __ret = __rev0 & ~__rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

__ai int64x1_t vbic_s64(int64x1_t __p0, int64x1_t __p1) {
  int64x1_t __ret;
  __ret = __p0 & ~__p1;
  return __ret;
}
#ifdef __LITTLE_ENDIAN__
__ai int16x4_t vbic_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __ret;
  __ret = __p0 & ~__p1;
  return __ret;
}
#else
__ai int16x4_t vbic_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int16x4_t __ret;
  __ret = __rev0 & ~__rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai poly8x8_t vbsl_p8(uint8x8_t __p0, poly8x8_t __p1, poly8x8_t __p2) {
  poly8x8_t __ret;
  __ret = (poly8x8_t) __builtin_neon_vbsl_v((int8x8_t)__p0, (int8x8_t)__p1, (int8x8_t)__p2, 4);
  return __ret;
}
#else
__ai poly8x8_t vbsl_p8(uint8x8_t __p0, poly8x8_t __p1, poly8x8_t __p2) {
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  poly8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  poly8x8_t __rev2;  __rev2 = __builtin_shufflevector(__p2, __p2, 7, 6, 5, 4, 3, 2, 1, 0);
  poly8x8_t __ret;
  __ret = (poly8x8_t) __builtin_neon_vbsl_v((int8x8_t)__rev0, (int8x8_t)__rev1, (int8x8_t)__rev2, 4);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai poly16x4_t vbsl_p16(uint16x4_t __p0, poly16x4_t __p1, poly16x4_t __p2) {
  poly16x4_t __ret;
  __ret = (poly16x4_t) __builtin_neon_vbsl_v((int8x8_t)__p0, (int8x8_t)__p1, (int8x8_t)__p2, 5);
  return __ret;
}
#else
__ai poly16x4_t vbsl_p16(uint16x4_t __p0, poly16x4_t __p1, poly16x4_t __p2) {
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  poly16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  poly16x4_t __rev2;  __rev2 = __builtin_shufflevector(__p2, __p2, 3, 2, 1, 0);
  poly16x4_t __ret;
  __ret = (poly16x4_t) __builtin_neon_vbsl_v((int8x8_t)__rev0, (int8x8_t)__rev1, (int8x8_t)__rev2, 5);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai poly8x16_t vbslq_p8(uint8x16_t __p0, poly8x16_t __p1, poly8x16_t __p2) {
  poly8x16_t __ret;
  __ret = (poly8x16_t) __builtin_neon_vbslq_v((int8x16_t)__p0, (int8x16_t)__p1, (int8x16_t)__p2, 36);
  return __ret;
}
#else
__ai poly8x16_t vbslq_p8(uint8x16_t __p0, poly8x16_t __p1, poly8x16_t __p2) {
  uint8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  poly8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  poly8x16_t __rev2;  __rev2 = __builtin_shufflevector(__p2, __p2, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  poly8x16_t __ret;
  __ret = (poly8x16_t) __builtin_neon_vbslq_v((int8x16_t)__rev0, (int8x16_t)__rev1, (int8x16_t)__rev2, 36);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai poly16x8_t vbslq_p16(uint16x8_t __p0, poly16x8_t __p1, poly16x8_t __p2) {
  poly16x8_t __ret;
  __ret = (poly16x8_t) __builtin_neon_vbslq_v((int8x16_t)__p0, (int8x16_t)__p1, (int8x16_t)__p2, 37);
  return __ret;
}
#else
__ai poly16x8_t vbslq_p16(uint16x8_t __p0, poly16x8_t __p1, poly16x8_t __p2) {
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  poly16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  poly16x8_t __rev2;  __rev2 = __builtin_shufflevector(__p2, __p2, 7, 6, 5, 4, 3, 2, 1, 0);
  poly16x8_t __ret;
  __ret = (poly16x8_t) __builtin_neon_vbslq_v((int8x16_t)__rev0, (int8x16_t)__rev1, (int8x16_t)__rev2, 37);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x16_t vbslq_u8(uint8x16_t __p0, uint8x16_t __p1, uint8x16_t __p2) {
  uint8x16_t __ret;
  __ret = (uint8x16_t) __builtin_neon_vbslq_v((int8x16_t)__p0, (int8x16_t)__p1, (int8x16_t)__p2, 48);
  return __ret;
}
#else
__ai uint8x16_t vbslq_u8(uint8x16_t __p0, uint8x16_t __p1, uint8x16_t __p2) {
  uint8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __rev2;  __rev2 = __builtin_shufflevector(__p2, __p2, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __ret;
  __ret = (uint8x16_t) __builtin_neon_vbslq_v((int8x16_t)__rev0, (int8x16_t)__rev1, (int8x16_t)__rev2, 48);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vbslq_u32(uint32x4_t __p0, uint32x4_t __p1, uint32x4_t __p2) {
  uint32x4_t __ret;
  __ret = (uint32x4_t) __builtin_neon_vbslq_v((int8x16_t)__p0, (int8x16_t)__p1, (int8x16_t)__p2, 50);
  return __ret;
}
#else
__ai uint32x4_t vbslq_u32(uint32x4_t __p0, uint32x4_t __p1, uint32x4_t __p2) {
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint32x4_t __rev2;  __rev2 = __builtin_shufflevector(__p2, __p2, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = (uint32x4_t) __builtin_neon_vbslq_v((int8x16_t)__rev0, (int8x16_t)__rev1, (int8x16_t)__rev2, 50);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint64x2_t vbslq_u64(uint64x2_t __p0, uint64x2_t __p1, uint64x2_t __p2) {
  uint64x2_t __ret;
  __ret = (uint64x2_t) __builtin_neon_vbslq_v((int8x16_t)__p0, (int8x16_t)__p1, (int8x16_t)__p2, 51);
  return __ret;
}
#else
__ai uint64x2_t vbslq_u64(uint64x2_t __p0, uint64x2_t __p1, uint64x2_t __p2) {
  uint64x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint64x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint64x2_t __rev2;  __rev2 = __builtin_shufflevector(__p2, __p2, 1, 0);
  uint64x2_t __ret;
  __ret = (uint64x2_t) __builtin_neon_vbslq_v((int8x16_t)__rev0, (int8x16_t)__rev1, (int8x16_t)__rev2, 51);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x8_t vbslq_u16(uint16x8_t __p0, uint16x8_t __p1, uint16x8_t __p2) {
  uint16x8_t __ret;
  __ret = (uint16x8_t) __builtin_neon_vbslq_v((int8x16_t)__p0, (int8x16_t)__p1, (int8x16_t)__p2, 49);
  return __ret;
}
#else
__ai uint16x8_t vbslq_u16(uint16x8_t __p0, uint16x8_t __p1, uint16x8_t __p2) {
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __rev2;  __rev2 = __builtin_shufflevector(__p2, __p2, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __ret;
  __ret = (uint16x8_t) __builtin_neon_vbslq_v((int8x16_t)__rev0, (int8x16_t)__rev1, (int8x16_t)__rev2, 49);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x16_t vbslq_s8(uint8x16_t __p0, int8x16_t __p1, int8x16_t __p2) {
  int8x16_t __ret;
  __ret = (int8x16_t) __builtin_neon_vbslq_v((int8x16_t)__p0, (int8x16_t)__p1, (int8x16_t)__p2, 32);
  return __ret;
}
#else
__ai int8x16_t vbslq_s8(uint8x16_t __p0, int8x16_t __p1, int8x16_t __p2) {
  uint8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __rev2;  __rev2 = __builtin_shufflevector(__p2, __p2, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __ret;
  __ret = (int8x16_t) __builtin_neon_vbslq_v((int8x16_t)__rev0, (int8x16_t)__rev1, (int8x16_t)__rev2, 32);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai float32x4_t vbslq_f32(uint32x4_t __p0, float32x4_t __p1, float32x4_t __p2) {
  float32x4_t __ret;
  __ret = (float32x4_t) __builtin_neon_vbslq_v((int8x16_t)__p0, (int8x16_t)__p1, (int8x16_t)__p2, 41);
  return __ret;
}
#else
__ai float32x4_t vbslq_f32(uint32x4_t __p0, float32x4_t __p1, float32x4_t __p2) {
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  float32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  float32x4_t __rev2;  __rev2 = __builtin_shufflevector(__p2, __p2, 3, 2, 1, 0);
  float32x4_t __ret;
  __ret = (float32x4_t) __builtin_neon_vbslq_v((int8x16_t)__rev0, (int8x16_t)__rev1, (int8x16_t)__rev2, 41);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x4_t vbslq_s32(uint32x4_t __p0, int32x4_t __p1, int32x4_t __p2) {
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vbslq_v((int8x16_t)__p0, (int8x16_t)__p1, (int8x16_t)__p2, 34);
  return __ret;
}
#else
__ai int32x4_t vbslq_s32(uint32x4_t __p0, int32x4_t __p1, int32x4_t __p2) {
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int32x4_t __rev2;  __rev2 = __builtin_shufflevector(__p2, __p2, 3, 2, 1, 0);
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vbslq_v((int8x16_t)__rev0, (int8x16_t)__rev1, (int8x16_t)__rev2, 34);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int64x2_t vbslq_s64(uint64x2_t __p0, int64x2_t __p1, int64x2_t __p2) {
  int64x2_t __ret;
  __ret = (int64x2_t) __builtin_neon_vbslq_v((int8x16_t)__p0, (int8x16_t)__p1, (int8x16_t)__p2, 35);
  return __ret;
}
#else
__ai int64x2_t vbslq_s64(uint64x2_t __p0, int64x2_t __p1, int64x2_t __p2) {
  uint64x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int64x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  int64x2_t __rev2;  __rev2 = __builtin_shufflevector(__p2, __p2, 1, 0);
  int64x2_t __ret;
  __ret = (int64x2_t) __builtin_neon_vbslq_v((int8x16_t)__rev0, (int8x16_t)__rev1, (int8x16_t)__rev2, 35);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x8_t vbslq_s16(uint16x8_t __p0, int16x8_t __p1, int16x8_t __p2) {
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vbslq_v((int8x16_t)__p0, (int8x16_t)__p1, (int8x16_t)__p2, 33);
  return __ret;
}
#else
__ai int16x8_t vbslq_s16(uint16x8_t __p0, int16x8_t __p1, int16x8_t __p2) {
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __rev2;  __rev2 = __builtin_shufflevector(__p2, __p2, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vbslq_v((int8x16_t)__rev0, (int8x16_t)__rev1, (int8x16_t)__rev2, 33);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x8_t vbsl_u8(uint8x8_t __p0, uint8x8_t __p1, uint8x8_t __p2) {
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vbsl_v((int8x8_t)__p0, (int8x8_t)__p1, (int8x8_t)__p2, 16);
  return __ret;
}
#else
__ai uint8x8_t vbsl_u8(uint8x8_t __p0, uint8x8_t __p1, uint8x8_t __p2) {
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __rev2;  __rev2 = __builtin_shufflevector(__p2, __p2, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vbsl_v((int8x8_t)__rev0, (int8x8_t)__rev1, (int8x8_t)__rev2, 16);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vbsl_u32(uint32x2_t __p0, uint32x2_t __p1, uint32x2_t __p2) {
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vbsl_v((int8x8_t)__p0, (int8x8_t)__p1, (int8x8_t)__p2, 18);
  return __ret;
}
#else
__ai uint32x2_t vbsl_u32(uint32x2_t __p0, uint32x2_t __p1, uint32x2_t __p2) {
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint32x2_t __rev2;  __rev2 = __builtin_shufflevector(__p2, __p2, 1, 0);
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vbsl_v((int8x8_t)__rev0, (int8x8_t)__rev1, (int8x8_t)__rev2, 18);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

__ai uint64x1_t vbsl_u64(uint64x1_t __p0, uint64x1_t __p1, uint64x1_t __p2) {
  uint64x1_t __ret;
  __ret = (uint64x1_t) __builtin_neon_vbsl_v((int8x8_t)__p0, (int8x8_t)__p1, (int8x8_t)__p2, 19);
  return __ret;
}
#ifdef __LITTLE_ENDIAN__
__ai uint16x4_t vbsl_u16(uint16x4_t __p0, uint16x4_t __p1, uint16x4_t __p2) {
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vbsl_v((int8x8_t)__p0, (int8x8_t)__p1, (int8x8_t)__p2, 17);
  return __ret;
}
#else
__ai uint16x4_t vbsl_u16(uint16x4_t __p0, uint16x4_t __p1, uint16x4_t __p2) {
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint16x4_t __rev2;  __rev2 = __builtin_shufflevector(__p2, __p2, 3, 2, 1, 0);
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vbsl_v((int8x8_t)__rev0, (int8x8_t)__rev1, (int8x8_t)__rev2, 17);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x8_t vbsl_s8(uint8x8_t __p0, int8x8_t __p1, int8x8_t __p2) {
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vbsl_v((int8x8_t)__p0, (int8x8_t)__p1, (int8x8_t)__p2, 0);
  return __ret;
}
#else
__ai int8x8_t vbsl_s8(uint8x8_t __p0, int8x8_t __p1, int8x8_t __p2) {
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __rev2;  __rev2 = __builtin_shufflevector(__p2, __p2, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vbsl_v((int8x8_t)__rev0, (int8x8_t)__rev1, (int8x8_t)__rev2, 0);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai float32x2_t vbsl_f32(uint32x2_t __p0, float32x2_t __p1, float32x2_t __p2) {
  float32x2_t __ret;
  __ret = (float32x2_t) __builtin_neon_vbsl_v((int8x8_t)__p0, (int8x8_t)__p1, (int8x8_t)__p2, 9);
  return __ret;
}
#else
__ai float32x2_t vbsl_f32(uint32x2_t __p0, float32x2_t __p1, float32x2_t __p2) {
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  float32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  float32x2_t __rev2;  __rev2 = __builtin_shufflevector(__p2, __p2, 1, 0);
  float32x2_t __ret;
  __ret = (float32x2_t) __builtin_neon_vbsl_v((int8x8_t)__rev0, (int8x8_t)__rev1, (int8x8_t)__rev2, 9);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x2_t vbsl_s32(uint32x2_t __p0, int32x2_t __p1, int32x2_t __p2) {
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vbsl_v((int8x8_t)__p0, (int8x8_t)__p1, (int8x8_t)__p2, 2);
  return __ret;
}
#else
__ai int32x2_t vbsl_s32(uint32x2_t __p0, int32x2_t __p1, int32x2_t __p2) {
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  int32x2_t __rev2;  __rev2 = __builtin_shufflevector(__p2, __p2, 1, 0);
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vbsl_v((int8x8_t)__rev0, (int8x8_t)__rev1, (int8x8_t)__rev2, 2);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

__ai int64x1_t vbsl_s64(uint64x1_t __p0, int64x1_t __p1, int64x1_t __p2) {
  int64x1_t __ret;
  __ret = (int64x1_t) __builtin_neon_vbsl_v((int8x8_t)__p0, (int8x8_t)__p1, (int8x8_t)__p2, 3);
  return __ret;
}
#ifdef __LITTLE_ENDIAN__
__ai int16x4_t vbsl_s16(uint16x4_t __p0, int16x4_t __p1, int16x4_t __p2) {
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vbsl_v((int8x8_t)__p0, (int8x8_t)__p1, (int8x8_t)__p2, 1);
  return __ret;
}
#else
__ai int16x4_t vbsl_s16(uint16x4_t __p0, int16x4_t __p1, int16x4_t __p2) {
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int16x4_t __rev2;  __rev2 = __builtin_shufflevector(__p2, __p2, 3, 2, 1, 0);
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vbsl_v((int8x8_t)__rev0, (int8x8_t)__rev1, (int8x8_t)__rev2, 1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vcageq_f32(float32x4_t __p0, float32x4_t __p1) {
  uint32x4_t __ret;
  __ret = (uint32x4_t) __builtin_neon_vcageq_v((int8x16_t)__p0, (int8x16_t)__p1, 50);
  return __ret;
}
#else
__ai uint32x4_t vcageq_f32(float32x4_t __p0, float32x4_t __p1) {
  float32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  float32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = (uint32x4_t) __builtin_neon_vcageq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 50);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vcage_f32(float32x2_t __p0, float32x2_t __p1) {
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vcage_v((int8x8_t)__p0, (int8x8_t)__p1, 18);
  return __ret;
}
#else
__ai uint32x2_t vcage_f32(float32x2_t __p0, float32x2_t __p1) {
  float32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  float32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vcage_v((int8x8_t)__rev0, (int8x8_t)__rev1, 18);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vcagtq_f32(float32x4_t __p0, float32x4_t __p1) {
  uint32x4_t __ret;
  __ret = (uint32x4_t) __builtin_neon_vcagtq_v((int8x16_t)__p0, (int8x16_t)__p1, 50);
  return __ret;
}
#else
__ai uint32x4_t vcagtq_f32(float32x4_t __p0, float32x4_t __p1) {
  float32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  float32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = (uint32x4_t) __builtin_neon_vcagtq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 50);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vcagt_f32(float32x2_t __p0, float32x2_t __p1) {
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vcagt_v((int8x8_t)__p0, (int8x8_t)__p1, 18);
  return __ret;
}
#else
__ai uint32x2_t vcagt_f32(float32x2_t __p0, float32x2_t __p1) {
  float32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  float32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vcagt_v((int8x8_t)__rev0, (int8x8_t)__rev1, 18);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vcaleq_f32(float32x4_t __p0, float32x4_t __p1) {
  uint32x4_t __ret;
  __ret = (uint32x4_t) __builtin_neon_vcaleq_v((int8x16_t)__p0, (int8x16_t)__p1, 50);
  return __ret;
}
#else
__ai uint32x4_t vcaleq_f32(float32x4_t __p0, float32x4_t __p1) {
  float32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  float32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = (uint32x4_t) __builtin_neon_vcaleq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 50);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vcale_f32(float32x2_t __p0, float32x2_t __p1) {
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vcale_v((int8x8_t)__p0, (int8x8_t)__p1, 18);
  return __ret;
}
#else
__ai uint32x2_t vcale_f32(float32x2_t __p0, float32x2_t __p1) {
  float32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  float32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vcale_v((int8x8_t)__rev0, (int8x8_t)__rev1, 18);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vcaltq_f32(float32x4_t __p0, float32x4_t __p1) {
  uint32x4_t __ret;
  __ret = (uint32x4_t) __builtin_neon_vcaltq_v((int8x16_t)__p0, (int8x16_t)__p1, 50);
  return __ret;
}
#else
__ai uint32x4_t vcaltq_f32(float32x4_t __p0, float32x4_t __p1) {
  float32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  float32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = (uint32x4_t) __builtin_neon_vcaltq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 50);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vcalt_f32(float32x2_t __p0, float32x2_t __p1) {
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vcalt_v((int8x8_t)__p0, (int8x8_t)__p1, 18);
  return __ret;
}
#else
__ai uint32x2_t vcalt_f32(float32x2_t __p0, float32x2_t __p1) {
  float32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  float32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vcalt_v((int8x8_t)__rev0, (int8x8_t)__rev1, 18);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x8_t vceq_p8(poly8x8_t __p0, poly8x8_t __p1) {
  uint8x8_t __ret;
  __ret = (uint8x8_t)(__p0 == __p1);
  return __ret;
}
#else
__ai uint8x8_t vceq_p8(poly8x8_t __p0, poly8x8_t __p1) {
  poly8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  poly8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __ret;
  __ret = (uint8x8_t)(__rev0 == __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x16_t vceqq_p8(poly8x16_t __p0, poly8x16_t __p1) {
  uint8x16_t __ret;
  __ret = (uint8x16_t)(__p0 == __p1);
  return __ret;
}
#else
__ai uint8x16_t vceqq_p8(poly8x16_t __p0, poly8x16_t __p1) {
  poly8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  poly8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __ret;
  __ret = (uint8x16_t)(__rev0 == __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x16_t vceqq_u8(uint8x16_t __p0, uint8x16_t __p1) {
  uint8x16_t __ret;
  __ret = (uint8x16_t)(__p0 == __p1);
  return __ret;
}
#else
__ai uint8x16_t vceqq_u8(uint8x16_t __p0, uint8x16_t __p1) {
  uint8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __ret;
  __ret = (uint8x16_t)(__rev0 == __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vceqq_u32(uint32x4_t __p0, uint32x4_t __p1) {
  uint32x4_t __ret;
  __ret = (uint32x4_t)(__p0 == __p1);
  return __ret;
}
#else
__ai uint32x4_t vceqq_u32(uint32x4_t __p0, uint32x4_t __p1) {
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = (uint32x4_t)(__rev0 == __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x8_t vceqq_u16(uint16x8_t __p0, uint16x8_t __p1) {
  uint16x8_t __ret;
  __ret = (uint16x8_t)(__p0 == __p1);
  return __ret;
}
#else
__ai uint16x8_t vceqq_u16(uint16x8_t __p0, uint16x8_t __p1) {
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __ret;
  __ret = (uint16x8_t)(__rev0 == __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x16_t vceqq_s8(int8x16_t __p0, int8x16_t __p1) {
  uint8x16_t __ret;
  __ret = (uint8x16_t)(__p0 == __p1);
  return __ret;
}
#else
__ai uint8x16_t vceqq_s8(int8x16_t __p0, int8x16_t __p1) {
  int8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __ret;
  __ret = (uint8x16_t)(__rev0 == __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vceqq_f32(float32x4_t __p0, float32x4_t __p1) {
  uint32x4_t __ret;
  __ret = (uint32x4_t)(__p0 == __p1);
  return __ret;
}
#else
__ai uint32x4_t vceqq_f32(float32x4_t __p0, float32x4_t __p1) {
  float32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  float32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = (uint32x4_t)(__rev0 == __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vceqq_s32(int32x4_t __p0, int32x4_t __p1) {
  uint32x4_t __ret;
  __ret = (uint32x4_t)(__p0 == __p1);
  return __ret;
}
#else
__ai uint32x4_t vceqq_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = (uint32x4_t)(__rev0 == __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x8_t vceqq_s16(int16x8_t __p0, int16x8_t __p1) {
  uint16x8_t __ret;
  __ret = (uint16x8_t)(__p0 == __p1);
  return __ret;
}
#else
__ai uint16x8_t vceqq_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __ret;
  __ret = (uint16x8_t)(__rev0 == __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x8_t vceq_u8(uint8x8_t __p0, uint8x8_t __p1) {
  uint8x8_t __ret;
  __ret = (uint8x8_t)(__p0 == __p1);
  return __ret;
}
#else
__ai uint8x8_t vceq_u8(uint8x8_t __p0, uint8x8_t __p1) {
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __ret;
  __ret = (uint8x8_t)(__rev0 == __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vceq_u32(uint32x2_t __p0, uint32x2_t __p1) {
  uint32x2_t __ret;
  __ret = (uint32x2_t)(__p0 == __p1);
  return __ret;
}
#else
__ai uint32x2_t vceq_u32(uint32x2_t __p0, uint32x2_t __p1) {
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint32x2_t __ret;
  __ret = (uint32x2_t)(__rev0 == __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x4_t vceq_u16(uint16x4_t __p0, uint16x4_t __p1) {
  uint16x4_t __ret;
  __ret = (uint16x4_t)(__p0 == __p1);
  return __ret;
}
#else
__ai uint16x4_t vceq_u16(uint16x4_t __p0, uint16x4_t __p1) {
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint16x4_t __ret;
  __ret = (uint16x4_t)(__rev0 == __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x8_t vceq_s8(int8x8_t __p0, int8x8_t __p1) {
  uint8x8_t __ret;
  __ret = (uint8x8_t)(__p0 == __p1);
  return __ret;
}
#else
__ai uint8x8_t vceq_s8(int8x8_t __p0, int8x8_t __p1) {
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __ret;
  __ret = (uint8x8_t)(__rev0 == __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vceq_f32(float32x2_t __p0, float32x2_t __p1) {
  uint32x2_t __ret;
  __ret = (uint32x2_t)(__p0 == __p1);
  return __ret;
}
#else
__ai uint32x2_t vceq_f32(float32x2_t __p0, float32x2_t __p1) {
  float32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  float32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint32x2_t __ret;
  __ret = (uint32x2_t)(__rev0 == __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vceq_s32(int32x2_t __p0, int32x2_t __p1) {
  uint32x2_t __ret;
  __ret = (uint32x2_t)(__p0 == __p1);
  return __ret;
}
#else
__ai uint32x2_t vceq_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint32x2_t __ret;
  __ret = (uint32x2_t)(__rev0 == __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x4_t vceq_s16(int16x4_t __p0, int16x4_t __p1) {
  uint16x4_t __ret;
  __ret = (uint16x4_t)(__p0 == __p1);
  return __ret;
}
#else
__ai uint16x4_t vceq_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint16x4_t __ret;
  __ret = (uint16x4_t)(__rev0 == __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x16_t vcgeq_u8(uint8x16_t __p0, uint8x16_t __p1) {
  uint8x16_t __ret;
  __ret = (uint8x16_t)(__p0 >= __p1);
  return __ret;
}
#else
__ai uint8x16_t vcgeq_u8(uint8x16_t __p0, uint8x16_t __p1) {
  uint8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __ret;
  __ret = (uint8x16_t)(__rev0 >= __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vcgeq_u32(uint32x4_t __p0, uint32x4_t __p1) {
  uint32x4_t __ret;
  __ret = (uint32x4_t)(__p0 >= __p1);
  return __ret;
}
#else
__ai uint32x4_t vcgeq_u32(uint32x4_t __p0, uint32x4_t __p1) {
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = (uint32x4_t)(__rev0 >= __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x8_t vcgeq_u16(uint16x8_t __p0, uint16x8_t __p1) {
  uint16x8_t __ret;
  __ret = (uint16x8_t)(__p0 >= __p1);
  return __ret;
}
#else
__ai uint16x8_t vcgeq_u16(uint16x8_t __p0, uint16x8_t __p1) {
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __ret;
  __ret = (uint16x8_t)(__rev0 >= __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x16_t vcgeq_s8(int8x16_t __p0, int8x16_t __p1) {
  uint8x16_t __ret;
  __ret = (uint8x16_t)(__p0 >= __p1);
  return __ret;
}
#else
__ai uint8x16_t vcgeq_s8(int8x16_t __p0, int8x16_t __p1) {
  int8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __ret;
  __ret = (uint8x16_t)(__rev0 >= __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vcgeq_f32(float32x4_t __p0, float32x4_t __p1) {
  uint32x4_t __ret;
  __ret = (uint32x4_t)(__p0 >= __p1);
  return __ret;
}
#else
__ai uint32x4_t vcgeq_f32(float32x4_t __p0, float32x4_t __p1) {
  float32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  float32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = (uint32x4_t)(__rev0 >= __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vcgeq_s32(int32x4_t __p0, int32x4_t __p1) {
  uint32x4_t __ret;
  __ret = (uint32x4_t)(__p0 >= __p1);
  return __ret;
}
#else
__ai uint32x4_t vcgeq_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = (uint32x4_t)(__rev0 >= __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x8_t vcgeq_s16(int16x8_t __p0, int16x8_t __p1) {
  uint16x8_t __ret;
  __ret = (uint16x8_t)(__p0 >= __p1);
  return __ret;
}
#else
__ai uint16x8_t vcgeq_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __ret;
  __ret = (uint16x8_t)(__rev0 >= __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x8_t vcge_u8(uint8x8_t __p0, uint8x8_t __p1) {
  uint8x8_t __ret;
  __ret = (uint8x8_t)(__p0 >= __p1);
  return __ret;
}
#else
__ai uint8x8_t vcge_u8(uint8x8_t __p0, uint8x8_t __p1) {
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __ret;
  __ret = (uint8x8_t)(__rev0 >= __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vcge_u32(uint32x2_t __p0, uint32x2_t __p1) {
  uint32x2_t __ret;
  __ret = (uint32x2_t)(__p0 >= __p1);
  return __ret;
}
#else
__ai uint32x2_t vcge_u32(uint32x2_t __p0, uint32x2_t __p1) {
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint32x2_t __ret;
  __ret = (uint32x2_t)(__rev0 >= __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x4_t vcge_u16(uint16x4_t __p0, uint16x4_t __p1) {
  uint16x4_t __ret;
  __ret = (uint16x4_t)(__p0 >= __p1);
  return __ret;
}
#else
__ai uint16x4_t vcge_u16(uint16x4_t __p0, uint16x4_t __p1) {
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint16x4_t __ret;
  __ret = (uint16x4_t)(__rev0 >= __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x8_t vcge_s8(int8x8_t __p0, int8x8_t __p1) {
  uint8x8_t __ret;
  __ret = (uint8x8_t)(__p0 >= __p1);
  return __ret;
}
#else
__ai uint8x8_t vcge_s8(int8x8_t __p0, int8x8_t __p1) {
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __ret;
  __ret = (uint8x8_t)(__rev0 >= __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vcge_f32(float32x2_t __p0, float32x2_t __p1) {
  uint32x2_t __ret;
  __ret = (uint32x2_t)(__p0 >= __p1);
  return __ret;
}
#else
__ai uint32x2_t vcge_f32(float32x2_t __p0, float32x2_t __p1) {
  float32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  float32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint32x2_t __ret;
  __ret = (uint32x2_t)(__rev0 >= __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vcge_s32(int32x2_t __p0, int32x2_t __p1) {
  uint32x2_t __ret;
  __ret = (uint32x2_t)(__p0 >= __p1);
  return __ret;
}
#else
__ai uint32x2_t vcge_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint32x2_t __ret;
  __ret = (uint32x2_t)(__rev0 >= __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x4_t vcge_s16(int16x4_t __p0, int16x4_t __p1) {
  uint16x4_t __ret;
  __ret = (uint16x4_t)(__p0 >= __p1);
  return __ret;
}
#else
__ai uint16x4_t vcge_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint16x4_t __ret;
  __ret = (uint16x4_t)(__rev0 >= __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x16_t vcgtq_u8(uint8x16_t __p0, uint8x16_t __p1) {
  uint8x16_t __ret;
  __ret = (uint8x16_t)(__p0 > __p1);
  return __ret;
}
#else
__ai uint8x16_t vcgtq_u8(uint8x16_t __p0, uint8x16_t __p1) {
  uint8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __ret;
  __ret = (uint8x16_t)(__rev0 > __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vcgtq_u32(uint32x4_t __p0, uint32x4_t __p1) {
  uint32x4_t __ret;
  __ret = (uint32x4_t)(__p0 > __p1);
  return __ret;
}
#else
__ai uint32x4_t vcgtq_u32(uint32x4_t __p0, uint32x4_t __p1) {
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = (uint32x4_t)(__rev0 > __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x8_t vcgtq_u16(uint16x8_t __p0, uint16x8_t __p1) {
  uint16x8_t __ret;
  __ret = (uint16x8_t)(__p0 > __p1);
  return __ret;
}
#else
__ai uint16x8_t vcgtq_u16(uint16x8_t __p0, uint16x8_t __p1) {
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __ret;
  __ret = (uint16x8_t)(__rev0 > __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x16_t vcgtq_s8(int8x16_t __p0, int8x16_t __p1) {
  uint8x16_t __ret;
  __ret = (uint8x16_t)(__p0 > __p1);
  return __ret;
}
#else
__ai uint8x16_t vcgtq_s8(int8x16_t __p0, int8x16_t __p1) {
  int8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __ret;
  __ret = (uint8x16_t)(__rev0 > __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vcgtq_f32(float32x4_t __p0, float32x4_t __p1) {
  uint32x4_t __ret;
  __ret = (uint32x4_t)(__p0 > __p1);
  return __ret;
}
#else
__ai uint32x4_t vcgtq_f32(float32x4_t __p0, float32x4_t __p1) {
  float32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  float32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = (uint32x4_t)(__rev0 > __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vcgtq_s32(int32x4_t __p0, int32x4_t __p1) {
  uint32x4_t __ret;
  __ret = (uint32x4_t)(__p0 > __p1);
  return __ret;
}
#else
__ai uint32x4_t vcgtq_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = (uint32x4_t)(__rev0 > __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x8_t vcgtq_s16(int16x8_t __p0, int16x8_t __p1) {
  uint16x8_t __ret;
  __ret = (uint16x8_t)(__p0 > __p1);
  return __ret;
}
#else
__ai uint16x8_t vcgtq_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __ret;
  __ret = (uint16x8_t)(__rev0 > __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x8_t vcgt_u8(uint8x8_t __p0, uint8x8_t __p1) {
  uint8x8_t __ret;
  __ret = (uint8x8_t)(__p0 > __p1);
  return __ret;
}
#else
__ai uint8x8_t vcgt_u8(uint8x8_t __p0, uint8x8_t __p1) {
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __ret;
  __ret = (uint8x8_t)(__rev0 > __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vcgt_u32(uint32x2_t __p0, uint32x2_t __p1) {
  uint32x2_t __ret;
  __ret = (uint32x2_t)(__p0 > __p1);
  return __ret;
}
#else
__ai uint32x2_t vcgt_u32(uint32x2_t __p0, uint32x2_t __p1) {
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint32x2_t __ret;
  __ret = (uint32x2_t)(__rev0 > __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x4_t vcgt_u16(uint16x4_t __p0, uint16x4_t __p1) {
  uint16x4_t __ret;
  __ret = (uint16x4_t)(__p0 > __p1);
  return __ret;
}
#else
__ai uint16x4_t vcgt_u16(uint16x4_t __p0, uint16x4_t __p1) {
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint16x4_t __ret;
  __ret = (uint16x4_t)(__rev0 > __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x8_t vcgt_s8(int8x8_t __p0, int8x8_t __p1) {
  uint8x8_t __ret;
  __ret = (uint8x8_t)(__p0 > __p1);
  return __ret;
}
#else
__ai uint8x8_t vcgt_s8(int8x8_t __p0, int8x8_t __p1) {
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __ret;
  __ret = (uint8x8_t)(__rev0 > __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vcgt_f32(float32x2_t __p0, float32x2_t __p1) {
  uint32x2_t __ret;
  __ret = (uint32x2_t)(__p0 > __p1);
  return __ret;
}
#else
__ai uint32x2_t vcgt_f32(float32x2_t __p0, float32x2_t __p1) {
  float32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  float32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint32x2_t __ret;
  __ret = (uint32x2_t)(__rev0 > __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vcgt_s32(int32x2_t __p0, int32x2_t __p1) {
  uint32x2_t __ret;
  __ret = (uint32x2_t)(__p0 > __p1);
  return __ret;
}
#else
__ai uint32x2_t vcgt_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint32x2_t __ret;
  __ret = (uint32x2_t)(__rev0 > __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x4_t vcgt_s16(int16x4_t __p0, int16x4_t __p1) {
  uint16x4_t __ret;
  __ret = (uint16x4_t)(__p0 > __p1);
  return __ret;
}
#else
__ai uint16x4_t vcgt_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint16x4_t __ret;
  __ret = (uint16x4_t)(__rev0 > __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x16_t vcleq_u8(uint8x16_t __p0, uint8x16_t __p1) {
  uint8x16_t __ret;
  __ret = (uint8x16_t)(__p0 <= __p1);
  return __ret;
}
#else
__ai uint8x16_t vcleq_u8(uint8x16_t __p0, uint8x16_t __p1) {
  uint8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __ret;
  __ret = (uint8x16_t)(__rev0 <= __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vcleq_u32(uint32x4_t __p0, uint32x4_t __p1) {
  uint32x4_t __ret;
  __ret = (uint32x4_t)(__p0 <= __p1);
  return __ret;
}
#else
__ai uint32x4_t vcleq_u32(uint32x4_t __p0, uint32x4_t __p1) {
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = (uint32x4_t)(__rev0 <= __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x8_t vcleq_u16(uint16x8_t __p0, uint16x8_t __p1) {
  uint16x8_t __ret;
  __ret = (uint16x8_t)(__p0 <= __p1);
  return __ret;
}
#else
__ai uint16x8_t vcleq_u16(uint16x8_t __p0, uint16x8_t __p1) {
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __ret;
  __ret = (uint16x8_t)(__rev0 <= __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x16_t vcleq_s8(int8x16_t __p0, int8x16_t __p1) {
  uint8x16_t __ret;
  __ret = (uint8x16_t)(__p0 <= __p1);
  return __ret;
}
#else
__ai uint8x16_t vcleq_s8(int8x16_t __p0, int8x16_t __p1) {
  int8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __ret;
  __ret = (uint8x16_t)(__rev0 <= __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vcleq_f32(float32x4_t __p0, float32x4_t __p1) {
  uint32x4_t __ret;
  __ret = (uint32x4_t)(__p0 <= __p1);
  return __ret;
}
#else
__ai uint32x4_t vcleq_f32(float32x4_t __p0, float32x4_t __p1) {
  float32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  float32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = (uint32x4_t)(__rev0 <= __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vcleq_s32(int32x4_t __p0, int32x4_t __p1) {
  uint32x4_t __ret;
  __ret = (uint32x4_t)(__p0 <= __p1);
  return __ret;
}
#else
__ai uint32x4_t vcleq_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = (uint32x4_t)(__rev0 <= __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x8_t vcleq_s16(int16x8_t __p0, int16x8_t __p1) {
  uint16x8_t __ret;
  __ret = (uint16x8_t)(__p0 <= __p1);
  return __ret;
}
#else
__ai uint16x8_t vcleq_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __ret;
  __ret = (uint16x8_t)(__rev0 <= __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x8_t vcle_u8(uint8x8_t __p0, uint8x8_t __p1) {
  uint8x8_t __ret;
  __ret = (uint8x8_t)(__p0 <= __p1);
  return __ret;
}
#else
__ai uint8x8_t vcle_u8(uint8x8_t __p0, uint8x8_t __p1) {
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __ret;
  __ret = (uint8x8_t)(__rev0 <= __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vcle_u32(uint32x2_t __p0, uint32x2_t __p1) {
  uint32x2_t __ret;
  __ret = (uint32x2_t)(__p0 <= __p1);
  return __ret;
}
#else
__ai uint32x2_t vcle_u32(uint32x2_t __p0, uint32x2_t __p1) {
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint32x2_t __ret;
  __ret = (uint32x2_t)(__rev0 <= __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x4_t vcle_u16(uint16x4_t __p0, uint16x4_t __p1) {
  uint16x4_t __ret;
  __ret = (uint16x4_t)(__p0 <= __p1);
  return __ret;
}
#else
__ai uint16x4_t vcle_u16(uint16x4_t __p0, uint16x4_t __p1) {
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint16x4_t __ret;
  __ret = (uint16x4_t)(__rev0 <= __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x8_t vcle_s8(int8x8_t __p0, int8x8_t __p1) {
  uint8x8_t __ret;
  __ret = (uint8x8_t)(__p0 <= __p1);
  return __ret;
}
#else
__ai uint8x8_t vcle_s8(int8x8_t __p0, int8x8_t __p1) {
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __ret;
  __ret = (uint8x8_t)(__rev0 <= __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vcle_f32(float32x2_t __p0, float32x2_t __p1) {
  uint32x2_t __ret;
  __ret = (uint32x2_t)(__p0 <= __p1);
  return __ret;
}
#else
__ai uint32x2_t vcle_f32(float32x2_t __p0, float32x2_t __p1) {
  float32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  float32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint32x2_t __ret;
  __ret = (uint32x2_t)(__rev0 <= __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vcle_s32(int32x2_t __p0, int32x2_t __p1) {
  uint32x2_t __ret;
  __ret = (uint32x2_t)(__p0 <= __p1);
  return __ret;
}
#else
__ai uint32x2_t vcle_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint32x2_t __ret;
  __ret = (uint32x2_t)(__rev0 <= __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x4_t vcle_s16(int16x4_t __p0, int16x4_t __p1) {
  uint16x4_t __ret;
  __ret = (uint16x4_t)(__p0 <= __p1);
  return __ret;
}
#else
__ai uint16x4_t vcle_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint16x4_t __ret;
  __ret = (uint16x4_t)(__rev0 <= __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x16_t vclsq_s8(int8x16_t __p0) {
  int8x16_t __ret;
  __ret = (int8x16_t) __builtin_neon_vclsq_v((int8x16_t)__p0, 32);
  return __ret;
}
#else
__ai int8x16_t vclsq_s8(int8x16_t __p0) {
  int8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __ret;
  __ret = (int8x16_t) __builtin_neon_vclsq_v((int8x16_t)__rev0, 32);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x4_t vclsq_s32(int32x4_t __p0) {
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vclsq_v((int8x16_t)__p0, 34);
  return __ret;
}
#else
__ai int32x4_t vclsq_s32(int32x4_t __p0) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vclsq_v((int8x16_t)__rev0, 34);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x8_t vclsq_s16(int16x8_t __p0) {
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vclsq_v((int8x16_t)__p0, 33);
  return __ret;
}
#else
__ai int16x8_t vclsq_s16(int16x8_t __p0) {
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vclsq_v((int8x16_t)__rev0, 33);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x8_t vcls_s8(int8x8_t __p0) {
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vcls_v((int8x8_t)__p0, 0);
  return __ret;
}
#else
__ai int8x8_t vcls_s8(int8x8_t __p0) {
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vcls_v((int8x8_t)__rev0, 0);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x2_t vcls_s32(int32x2_t __p0) {
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vcls_v((int8x8_t)__p0, 2);
  return __ret;
}
#else
__ai int32x2_t vcls_s32(int32x2_t __p0) {
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vcls_v((int8x8_t)__rev0, 2);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x4_t vcls_s16(int16x4_t __p0) {
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vcls_v((int8x8_t)__p0, 1);
  return __ret;
}
#else
__ai int16x4_t vcls_s16(int16x4_t __p0) {
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vcls_v((int8x8_t)__rev0, 1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x16_t vcltq_u8(uint8x16_t __p0, uint8x16_t __p1) {
  uint8x16_t __ret;
  __ret = (uint8x16_t)(__p0 < __p1);
  return __ret;
}
#else
__ai uint8x16_t vcltq_u8(uint8x16_t __p0, uint8x16_t __p1) {
  uint8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __ret;
  __ret = (uint8x16_t)(__rev0 < __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vcltq_u32(uint32x4_t __p0, uint32x4_t __p1) {
  uint32x4_t __ret;
  __ret = (uint32x4_t)(__p0 < __p1);
  return __ret;
}
#else
__ai uint32x4_t vcltq_u32(uint32x4_t __p0, uint32x4_t __p1) {
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = (uint32x4_t)(__rev0 < __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x8_t vcltq_u16(uint16x8_t __p0, uint16x8_t __p1) {
  uint16x8_t __ret;
  __ret = (uint16x8_t)(__p0 < __p1);
  return __ret;
}
#else
__ai uint16x8_t vcltq_u16(uint16x8_t __p0, uint16x8_t __p1) {
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __ret;
  __ret = (uint16x8_t)(__rev0 < __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x16_t vcltq_s8(int8x16_t __p0, int8x16_t __p1) {
  uint8x16_t __ret;
  __ret = (uint8x16_t)(__p0 < __p1);
  return __ret;
}
#else
__ai uint8x16_t vcltq_s8(int8x16_t __p0, int8x16_t __p1) {
  int8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __ret;
  __ret = (uint8x16_t)(__rev0 < __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vcltq_f32(float32x4_t __p0, float32x4_t __p1) {
  uint32x4_t __ret;
  __ret = (uint32x4_t)(__p0 < __p1);
  return __ret;
}
#else
__ai uint32x4_t vcltq_f32(float32x4_t __p0, float32x4_t __p1) {
  float32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  float32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = (uint32x4_t)(__rev0 < __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vcltq_s32(int32x4_t __p0, int32x4_t __p1) {
  uint32x4_t __ret;
  __ret = (uint32x4_t)(__p0 < __p1);
  return __ret;
}
#else
__ai uint32x4_t vcltq_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = (uint32x4_t)(__rev0 < __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x8_t vcltq_s16(int16x8_t __p0, int16x8_t __p1) {
  uint16x8_t __ret;
  __ret = (uint16x8_t)(__p0 < __p1);
  return __ret;
}
#else
__ai uint16x8_t vcltq_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __ret;
  __ret = (uint16x8_t)(__rev0 < __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x8_t vclt_u8(uint8x8_t __p0, uint8x8_t __p1) {
  uint8x8_t __ret;
  __ret = (uint8x8_t)(__p0 < __p1);
  return __ret;
}
#else
__ai uint8x8_t vclt_u8(uint8x8_t __p0, uint8x8_t __p1) {
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __ret;
  __ret = (uint8x8_t)(__rev0 < __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vclt_u32(uint32x2_t __p0, uint32x2_t __p1) {
  uint32x2_t __ret;
  __ret = (uint32x2_t)(__p0 < __p1);
  return __ret;
}
#else
__ai uint32x2_t vclt_u32(uint32x2_t __p0, uint32x2_t __p1) {
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint32x2_t __ret;
  __ret = (uint32x2_t)(__rev0 < __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x4_t vclt_u16(uint16x4_t __p0, uint16x4_t __p1) {
  uint16x4_t __ret;
  __ret = (uint16x4_t)(__p0 < __p1);
  return __ret;
}
#else
__ai uint16x4_t vclt_u16(uint16x4_t __p0, uint16x4_t __p1) {
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint16x4_t __ret;
  __ret = (uint16x4_t)(__rev0 < __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x8_t vclt_s8(int8x8_t __p0, int8x8_t __p1) {
  uint8x8_t __ret;
  __ret = (uint8x8_t)(__p0 < __p1);
  return __ret;
}
#else
__ai uint8x8_t vclt_s8(int8x8_t __p0, int8x8_t __p1) {
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __ret;
  __ret = (uint8x8_t)(__rev0 < __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vclt_f32(float32x2_t __p0, float32x2_t __p1) {
  uint32x2_t __ret;
  __ret = (uint32x2_t)(__p0 < __p1);
  return __ret;
}
#else
__ai uint32x2_t vclt_f32(float32x2_t __p0, float32x2_t __p1) {
  float32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  float32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint32x2_t __ret;
  __ret = (uint32x2_t)(__rev0 < __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vclt_s32(int32x2_t __p0, int32x2_t __p1) {
  uint32x2_t __ret;
  __ret = (uint32x2_t)(__p0 < __p1);
  return __ret;
}
#else
__ai uint32x2_t vclt_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint32x2_t __ret;
  __ret = (uint32x2_t)(__rev0 < __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x4_t vclt_s16(int16x4_t __p0, int16x4_t __p1) {
  uint16x4_t __ret;
  __ret = (uint16x4_t)(__p0 < __p1);
  return __ret;
}
#else
__ai uint16x4_t vclt_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint16x4_t __ret;
  __ret = (uint16x4_t)(__rev0 < __rev1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x16_t vclzq_u8(uint8x16_t __p0) {
  uint8x16_t __ret;
  __ret = (uint8x16_t) __builtin_neon_vclzq_v((int8x16_t)__p0, 48);
  return __ret;
}
#else
__ai uint8x16_t vclzq_u8(uint8x16_t __p0) {
  uint8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __ret;
  __ret = (uint8x16_t) __builtin_neon_vclzq_v((int8x16_t)__rev0, 48);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vclzq_u32(uint32x4_t __p0) {
  uint32x4_t __ret;
  __ret = (uint32x4_t) __builtin_neon_vclzq_v((int8x16_t)__p0, 50);
  return __ret;
}
#else
__ai uint32x4_t vclzq_u32(uint32x4_t __p0) {
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = (uint32x4_t) __builtin_neon_vclzq_v((int8x16_t)__rev0, 50);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x8_t vclzq_u16(uint16x8_t __p0) {
  uint16x8_t __ret;
  __ret = (uint16x8_t) __builtin_neon_vclzq_v((int8x16_t)__p0, 49);
  return __ret;
}
#else
__ai uint16x8_t vclzq_u16(uint16x8_t __p0) {
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __ret;
  __ret = (uint16x8_t) __builtin_neon_vclzq_v((int8x16_t)__rev0, 49);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x16_t vclzq_s8(int8x16_t __p0) {
  int8x16_t __ret;
  __ret = (int8x16_t) __builtin_neon_vclzq_v((int8x16_t)__p0, 32);
  return __ret;
}
#else
__ai int8x16_t vclzq_s8(int8x16_t __p0) {
  int8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __ret;
  __ret = (int8x16_t) __builtin_neon_vclzq_v((int8x16_t)__rev0, 32);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x4_t vclzq_s32(int32x4_t __p0) {
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vclzq_v((int8x16_t)__p0, 34);
  return __ret;
}
#else
__ai int32x4_t vclzq_s32(int32x4_t __p0) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vclzq_v((int8x16_t)__rev0, 34);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x8_t vclzq_s16(int16x8_t __p0) {
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vclzq_v((int8x16_t)__p0, 33);
  return __ret;
}
#else
__ai int16x8_t vclzq_s16(int16x8_t __p0) {
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vclzq_v((int8x16_t)__rev0, 33);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x8_t vclz_u8(uint8x8_t __p0) {
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vclz_v((int8x8_t)__p0, 16);
  return __ret;
}
#else
__ai uint8x8_t vclz_u8(uint8x8_t __p0) {
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vclz_v((int8x8_t)__rev0, 16);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vclz_u32(uint32x2_t __p0) {
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vclz_v((int8x8_t)__p0, 18);
  return __ret;
}
#else
__ai uint32x2_t vclz_u32(uint32x2_t __p0) {
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vclz_v((int8x8_t)__rev0, 18);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x4_t vclz_u16(uint16x4_t __p0) {
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vclz_v((int8x8_t)__p0, 17);
  return __ret;
}
#else
__ai uint16x4_t vclz_u16(uint16x4_t __p0) {
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vclz_v((int8x8_t)__rev0, 17);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x8_t vclz_s8(int8x8_t __p0) {
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vclz_v((int8x8_t)__p0, 0);
  return __ret;
}
#else
__ai int8x8_t vclz_s8(int8x8_t __p0) {
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vclz_v((int8x8_t)__rev0, 0);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x2_t vclz_s32(int32x2_t __p0) {
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vclz_v((int8x8_t)__p0, 2);
  return __ret;
}
#else
__ai int32x2_t vclz_s32(int32x2_t __p0) {
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vclz_v((int8x8_t)__rev0, 2);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x4_t vclz_s16(int16x4_t __p0) {
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vclz_v((int8x8_t)__p0, 1);
  return __ret;
}
#else
__ai int16x4_t vclz_s16(int16x4_t __p0) {
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vclz_v((int8x8_t)__rev0, 1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai poly8x8_t vcnt_p8(poly8x8_t __p0) {
  poly8x8_t __ret;
  __ret = (poly8x8_t) __builtin_neon_vcnt_v((int8x8_t)__p0, 4);
  return __ret;
}
#else
__ai poly8x8_t vcnt_p8(poly8x8_t __p0) {
  poly8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  poly8x8_t __ret;
  __ret = (poly8x8_t) __builtin_neon_vcnt_v((int8x8_t)__rev0, 4);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai poly8x16_t vcntq_p8(poly8x16_t __p0) {
  poly8x16_t __ret;
  __ret = (poly8x16_t) __builtin_neon_vcntq_v((int8x16_t)__p0, 36);
  return __ret;
}
#else
__ai poly8x16_t vcntq_p8(poly8x16_t __p0) {
  poly8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  poly8x16_t __ret;
  __ret = (poly8x16_t) __builtin_neon_vcntq_v((int8x16_t)__rev0, 36);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x16_t vcntq_u8(uint8x16_t __p0) {
  uint8x16_t __ret;
  __ret = (uint8x16_t) __builtin_neon_vcntq_v((int8x16_t)__p0, 48);
  return __ret;
}
#else
__ai uint8x16_t vcntq_u8(uint8x16_t __p0) {
  uint8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __ret;
  __ret = (uint8x16_t) __builtin_neon_vcntq_v((int8x16_t)__rev0, 48);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x16_t vcntq_s8(int8x16_t __p0) {
  int8x16_t __ret;
  __ret = (int8x16_t) __builtin_neon_vcntq_v((int8x16_t)__p0, 32);
  return __ret;
}
#else
__ai int8x16_t vcntq_s8(int8x16_t __p0) {
  int8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __ret;
  __ret = (int8x16_t) __builtin_neon_vcntq_v((int8x16_t)__rev0, 32);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x8_t vcnt_u8(uint8x8_t __p0) {
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vcnt_v((int8x8_t)__p0, 16);
  return __ret;
}
#else
__ai uint8x8_t vcnt_u8(uint8x8_t __p0) {
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vcnt_v((int8x8_t)__rev0, 16);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x8_t vcnt_s8(int8x8_t __p0) {
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vcnt_v((int8x8_t)__p0, 0);
  return __ret;
}
#else
__ai int8x8_t vcnt_s8(int8x8_t __p0) {
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vcnt_v((int8x8_t)__rev0, 0);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai poly8x16_t vcombine_p8(poly8x8_t __p0, poly8x8_t __p1) {
  poly8x16_t __ret;
  __ret = __builtin_shufflevector(__p0, __p1, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15);
  return __ret;
}
#else
__ai poly8x16_t vcombine_p8(poly8x8_t __p0, poly8x8_t __p1) {
  poly8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  poly8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  poly8x16_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev1, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai poly16x8_t vcombine_p16(poly16x4_t __p0, poly16x4_t __p1) {
  poly16x8_t __ret;
  __ret = __builtin_shufflevector(__p0, __p1, 0, 1, 2, 3, 4, 5, 6, 7);
  return __ret;
}
#else
__ai poly16x8_t vcombine_p16(poly16x4_t __p0, poly16x4_t __p1) {
  poly16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  poly16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  poly16x8_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev1, 0, 1, 2, 3, 4, 5, 6, 7);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x16_t vcombine_u8(uint8x8_t __p0, uint8x8_t __p1) {
  uint8x16_t __ret;
  __ret = __builtin_shufflevector(__p0, __p1, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15);
  return __ret;
}
#else
__ai uint8x16_t vcombine_u8(uint8x8_t __p0, uint8x8_t __p1) {
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev1, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
__ai uint8x16_t __noswap_vcombine_u8(uint8x8_t __p0, uint8x8_t __p1) {
  uint8x16_t __ret;
  __ret = __builtin_shufflevector(__p0, __p1, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vcombine_u32(uint32x2_t __p0, uint32x2_t __p1) {
  uint32x4_t __ret;
  __ret = __builtin_shufflevector(__p0, __p1, 0, 1, 2, 3);
  return __ret;
}
#else
__ai uint32x4_t vcombine_u32(uint32x2_t __p0, uint32x2_t __p1) {
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint32x4_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev1, 0, 1, 2, 3);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
__ai uint32x4_t __noswap_vcombine_u32(uint32x2_t __p0, uint32x2_t __p1) {
  uint32x4_t __ret;
  __ret = __builtin_shufflevector(__p0, __p1, 0, 1, 2, 3);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint64x2_t vcombine_u64(uint64x1_t __p0, uint64x1_t __p1) {
  uint64x2_t __ret;
  __ret = __builtin_shufflevector(__p0, __p1, 0, 1);
  return __ret;
}
#else
__ai uint64x2_t vcombine_u64(uint64x1_t __p0, uint64x1_t __p1) {
  uint64x2_t __ret;
  __ret = __builtin_shufflevector(__p0, __p1, 0, 1);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x8_t vcombine_u16(uint16x4_t __p0, uint16x4_t __p1) {
  uint16x8_t __ret;
  __ret = __builtin_shufflevector(__p0, __p1, 0, 1, 2, 3, 4, 5, 6, 7);
  return __ret;
}
#else
__ai uint16x8_t vcombine_u16(uint16x4_t __p0, uint16x4_t __p1) {
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint16x8_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev1, 0, 1, 2, 3, 4, 5, 6, 7);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
__ai uint16x8_t __noswap_vcombine_u16(uint16x4_t __p0, uint16x4_t __p1) {
  uint16x8_t __ret;
  __ret = __builtin_shufflevector(__p0, __p1, 0, 1, 2, 3, 4, 5, 6, 7);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x16_t vcombine_s8(int8x8_t __p0, int8x8_t __p1) {
  int8x16_t __ret;
  __ret = __builtin_shufflevector(__p0, __p1, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15);
  return __ret;
}
#else
__ai int8x16_t vcombine_s8(int8x8_t __p0, int8x8_t __p1) {
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev1, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
__ai int8x16_t __noswap_vcombine_s8(int8x8_t __p0, int8x8_t __p1) {
  int8x16_t __ret;
  __ret = __builtin_shufflevector(__p0, __p1, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai float32x4_t vcombine_f32(float32x2_t __p0, float32x2_t __p1) {
  float32x4_t __ret;
  __ret = __builtin_shufflevector(__p0, __p1, 0, 1, 2, 3);
  return __ret;
}
#else
__ai float32x4_t vcombine_f32(float32x2_t __p0, float32x2_t __p1) {
  float32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  float32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  float32x4_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev1, 0, 1, 2, 3);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
__ai float32x4_t __noswap_vcombine_f32(float32x2_t __p0, float32x2_t __p1) {
  float32x4_t __ret;
  __ret = __builtin_shufflevector(__p0, __p1, 0, 1, 2, 3);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai float16x8_t vcombine_f16(float16x4_t __p0, float16x4_t __p1) {
  float16x8_t __ret;
  __ret = __builtin_shufflevector(__p0, __p1, 0, 1, 2, 3, 4, 5, 6, 7);
  return __ret;
}
#else
__ai float16x8_t vcombine_f16(float16x4_t __p0, float16x4_t __p1) {
  float16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  float16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  float16x8_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev1, 0, 1, 2, 3, 4, 5, 6, 7);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
__ai float16x8_t __noswap_vcombine_f16(float16x4_t __p0, float16x4_t __p1) {
  float16x8_t __ret;
  __ret = __builtin_shufflevector(__p0, __p1, 0, 1, 2, 3, 4, 5, 6, 7);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x4_t vcombine_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x4_t __ret;
  __ret = __builtin_shufflevector(__p0, __p1, 0, 1, 2, 3);
  return __ret;
}
#else
__ai int32x4_t vcombine_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  int32x4_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev1, 0, 1, 2, 3);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
__ai int32x4_t __noswap_vcombine_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x4_t __ret;
  __ret = __builtin_shufflevector(__p0, __p1, 0, 1, 2, 3);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int64x2_t vcombine_s64(int64x1_t __p0, int64x1_t __p1) {
  int64x2_t __ret;
  __ret = __builtin_shufflevector(__p0, __p1, 0, 1);
  return __ret;
}
#else
__ai int64x2_t vcombine_s64(int64x1_t __p0, int64x1_t __p1) {
  int64x2_t __ret;
  __ret = __builtin_shufflevector(__p0, __p1, 0, 1);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x8_t vcombine_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x8_t __ret;
  __ret = __builtin_shufflevector(__p0, __p1, 0, 1, 2, 3, 4, 5, 6, 7);
  return __ret;
}
#else
__ai int16x8_t vcombine_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int16x8_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev1, 0, 1, 2, 3, 4, 5, 6, 7);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
__ai int16x8_t __noswap_vcombine_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x8_t __ret;
  __ret = __builtin_shufflevector(__p0, __p1, 0, 1, 2, 3, 4, 5, 6, 7);
  return __ret;
}
#endif

#define vcreate_p8(__p0) __extension__ ({ \
  poly8x8_t __ret; \
  uint64_t __promote = __p0; \
  __ret = (poly8x8_t)(__promote); \
  __ret; \
})
#define vcreate_p16(__p0) __extension__ ({ \
  poly16x4_t __ret; \
  uint64_t __promote = __p0; \
  __ret = (poly16x4_t)(__promote); \
  __ret; \
})
#define vcreate_u8(__p0) __extension__ ({ \
  uint8x8_t __ret; \
  uint64_t __promote = __p0; \
  __ret = (uint8x8_t)(__promote); \
  __ret; \
})
#define vcreate_u32(__p0) __extension__ ({ \
  uint32x2_t __ret; \
  uint64_t __promote = __p0; \
  __ret = (uint32x2_t)(__promote); \
  __ret; \
})
#define vcreate_u64(__p0) __extension__ ({ \
  uint64x1_t __ret; \
  uint64_t __promote = __p0; \
  __ret = (uint64x1_t)(__promote); \
  __ret; \
})
#define vcreate_u16(__p0) __extension__ ({ \
  uint16x4_t __ret; \
  uint64_t __promote = __p0; \
  __ret = (uint16x4_t)(__promote); \
  __ret; \
})
#define vcreate_s8(__p0) __extension__ ({ \
  int8x8_t __ret; \
  uint64_t __promote = __p0; \
  __ret = (int8x8_t)(__promote); \
  __ret; \
})
#define vcreate_f32(__p0) __extension__ ({ \
  float32x2_t __ret; \
  uint64_t __promote = __p0; \
  __ret = (float32x2_t)(__promote); \
  __ret; \
})
#define vcreate_f16(__p0) __extension__ ({ \
  float16x4_t __ret; \
  uint64_t __promote = __p0; \
  __ret = (float16x4_t)(__promote); \
  __ret; \
})
#define vcreate_s32(__p0) __extension__ ({ \
  int32x2_t __ret; \
  uint64_t __promote = __p0; \
  __ret = (int32x2_t)(__promote); \
  __ret; \
})
#define vcreate_s64(__p0) __extension__ ({ \
  int64x1_t __ret; \
  uint64_t __promote = __p0; \
  __ret = (int64x1_t)(__promote); \
  __ret; \
})
#define vcreate_s16(__p0) __extension__ ({ \
  int16x4_t __ret; \
  uint64_t __promote = __p0; \
  __ret = (int16x4_t)(__promote); \
  __ret; \
})
#ifdef __LITTLE_ENDIAN__
__ai float32x4_t vcvtq_f32_u32(uint32x4_t __p0) {
  float32x4_t __ret;
  __ret = (float32x4_t) __builtin_neon_vcvtq_f32_v((int8x16_t)__p0, 50);
  return __ret;
}
#else
__ai float32x4_t vcvtq_f32_u32(uint32x4_t __p0) {
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  float32x4_t __ret;
  __ret = (float32x4_t) __builtin_neon_vcvtq_f32_v((int8x16_t)__rev0, 50);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai float32x4_t vcvtq_f32_s32(int32x4_t __p0) {
  float32x4_t __ret;
  __ret = (float32x4_t) __builtin_neon_vcvtq_f32_v((int8x16_t)__p0, 34);
  return __ret;
}
#else
__ai float32x4_t vcvtq_f32_s32(int32x4_t __p0) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  float32x4_t __ret;
  __ret = (float32x4_t) __builtin_neon_vcvtq_f32_v((int8x16_t)__rev0, 34);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai float32x2_t vcvt_f32_u32(uint32x2_t __p0) {
  float32x2_t __ret;
  __ret = (float32x2_t) __builtin_neon_vcvt_f32_v((int8x8_t)__p0, 18);
  return __ret;
}
#else
__ai float32x2_t vcvt_f32_u32(uint32x2_t __p0) {
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  float32x2_t __ret;
  __ret = (float32x2_t) __builtin_neon_vcvt_f32_v((int8x8_t)__rev0, 18);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai float32x2_t vcvt_f32_s32(int32x2_t __p0) {
  float32x2_t __ret;
  __ret = (float32x2_t) __builtin_neon_vcvt_f32_v((int8x8_t)__p0, 2);
  return __ret;
}
#else
__ai float32x2_t vcvt_f32_s32(int32x2_t __p0) {
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  float32x2_t __ret;
  __ret = (float32x2_t) __builtin_neon_vcvt_f32_v((int8x8_t)__rev0, 2);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
#define vcvtq_n_f32_u32(__p0, __p1) __extension__ ({ \
  uint32x4_t __s0 = __p0; \
  float32x4_t __ret; \
  __ret = (float32x4_t) __builtin_neon_vcvtq_n_f32_v((int8x16_t)__s0, __p1, 50); \
  __ret; \
})
#else
#define vcvtq_n_f32_u32(__p0, __p1) __extension__ ({ \
  uint32x4_t __s0 = __p0; \
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  float32x4_t __ret; \
  __ret = (float32x4_t) __builtin_neon_vcvtq_n_f32_v((int8x16_t)__rev0, __p1, 50); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vcvtq_n_f32_s32(__p0, __p1) __extension__ ({ \
  int32x4_t __s0 = __p0; \
  float32x4_t __ret; \
  __ret = (float32x4_t) __builtin_neon_vcvtq_n_f32_v((int8x16_t)__s0, __p1, 34); \
  __ret; \
})
#else
#define vcvtq_n_f32_s32(__p0, __p1) __extension__ ({ \
  int32x4_t __s0 = __p0; \
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  float32x4_t __ret; \
  __ret = (float32x4_t) __builtin_neon_vcvtq_n_f32_v((int8x16_t)__rev0, __p1, 34); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vcvt_n_f32_u32(__p0, __p1) __extension__ ({ \
  uint32x2_t __s0 = __p0; \
  float32x2_t __ret; \
  __ret = (float32x2_t) __builtin_neon_vcvt_n_f32_v((int8x8_t)__s0, __p1, 18); \
  __ret; \
})
#else
#define vcvt_n_f32_u32(__p0, __p1) __extension__ ({ \
  uint32x2_t __s0 = __p0; \
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  float32x2_t __ret; \
  __ret = (float32x2_t) __builtin_neon_vcvt_n_f32_v((int8x8_t)__rev0, __p1, 18); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vcvt_n_f32_s32(__p0, __p1) __extension__ ({ \
  int32x2_t __s0 = __p0; \
  float32x2_t __ret; \
  __ret = (float32x2_t) __builtin_neon_vcvt_n_f32_v((int8x8_t)__s0, __p1, 2); \
  __ret; \
})
#else
#define vcvt_n_f32_s32(__p0, __p1) __extension__ ({ \
  int32x2_t __s0 = __p0; \
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  float32x2_t __ret; \
  __ret = (float32x2_t) __builtin_neon_vcvt_n_f32_v((int8x8_t)__rev0, __p1, 2); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vcvtq_n_s32_f32(__p0, __p1) __extension__ ({ \
  float32x4_t __s0 = __p0; \
  int32x4_t __ret; \
  __ret = (int32x4_t) __builtin_neon_vcvtq_n_s32_v((int8x16_t)__s0, __p1, 34); \
  __ret; \
})
#else
#define vcvtq_n_s32_f32(__p0, __p1) __extension__ ({ \
  float32x4_t __s0 = __p0; \
  float32x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  int32x4_t __ret; \
  __ret = (int32x4_t) __builtin_neon_vcvtq_n_s32_v((int8x16_t)__rev0, __p1, 34); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vcvt_n_s32_f32(__p0, __p1) __extension__ ({ \
  float32x2_t __s0 = __p0; \
  int32x2_t __ret; \
  __ret = (int32x2_t) __builtin_neon_vcvt_n_s32_v((int8x8_t)__s0, __p1, 2); \
  __ret; \
})
#else
#define vcvt_n_s32_f32(__p0, __p1) __extension__ ({ \
  float32x2_t __s0 = __p0; \
  float32x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  int32x2_t __ret; \
  __ret = (int32x2_t) __builtin_neon_vcvt_n_s32_v((int8x8_t)__rev0, __p1, 2); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vcvtq_n_u32_f32(__p0, __p1) __extension__ ({ \
  float32x4_t __s0 = __p0; \
  uint32x4_t __ret; \
  __ret = (uint32x4_t) __builtin_neon_vcvtq_n_u32_v((int8x16_t)__s0, __p1, 50); \
  __ret; \
})
#else
#define vcvtq_n_u32_f32(__p0, __p1) __extension__ ({ \
  float32x4_t __s0 = __p0; \
  float32x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  uint32x4_t __ret; \
  __ret = (uint32x4_t) __builtin_neon_vcvtq_n_u32_v((int8x16_t)__rev0, __p1, 50); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vcvt_n_u32_f32(__p0, __p1) __extension__ ({ \
  float32x2_t __s0 = __p0; \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_vcvt_n_u32_v((int8x8_t)__s0, __p1, 18); \
  __ret; \
})
#else
#define vcvt_n_u32_f32(__p0, __p1) __extension__ ({ \
  float32x2_t __s0 = __p0; \
  float32x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_vcvt_n_u32_v((int8x8_t)__rev0, __p1, 18); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x4_t vcvtq_s32_f32(float32x4_t __p0) {
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vcvtq_s32_v((int8x16_t)__p0, 34);
  return __ret;
}
#else
__ai int32x4_t vcvtq_s32_f32(float32x4_t __p0) {
  float32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vcvtq_s32_v((int8x16_t)__rev0, 34);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x2_t vcvt_s32_f32(float32x2_t __p0) {
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vcvt_s32_v((int8x8_t)__p0, 2);
  return __ret;
}
#else
__ai int32x2_t vcvt_s32_f32(float32x2_t __p0) {
  float32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vcvt_s32_v((int8x8_t)__rev0, 2);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vcvtq_u32_f32(float32x4_t __p0) {
  uint32x4_t __ret;
  __ret = (uint32x4_t) __builtin_neon_vcvtq_u32_v((int8x16_t)__p0, 50);
  return __ret;
}
#else
__ai uint32x4_t vcvtq_u32_f32(float32x4_t __p0) {
  float32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = (uint32x4_t) __builtin_neon_vcvtq_u32_v((int8x16_t)__rev0, 50);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vcvt_u32_f32(float32x2_t __p0) {
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vcvt_u32_v((int8x8_t)__p0, 18);
  return __ret;
}
#else
__ai uint32x2_t vcvt_u32_f32(float32x2_t __p0) {
  float32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vcvt_u32_v((int8x8_t)__rev0, 18);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
#define vdup_lane_p8(__p0_0, __p1_0) __extension__ ({ \
  poly8x8_t __s0_0 = __p0_0; \
  poly8x8_t __ret_0; \
  __ret_0 = splat_lane_p8(__s0_0, __p1_0); \
  __ret_0; \
})
#else
#define vdup_lane_p8(__p0_1, __p1_1) __extension__ ({ \
  poly8x8_t __s0_1 = __p0_1; \
  poly8x8_t __rev0_1;  __rev0_1 = __builtin_shufflevector(__s0_1, __s0_1, 7, 6, 5, 4, 3, 2, 1, 0); \
  poly8x8_t __ret_1; \
  __ret_1 = __noswap_splat_lane_p8(__rev0_1, __p1_1); \
  __ret_1 = __builtin_shufflevector(__ret_1, __ret_1, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret_1; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vdup_lane_p16(__p0_2, __p1_2) __extension__ ({ \
  poly16x4_t __s0_2 = __p0_2; \
  poly16x4_t __ret_2; \
  __ret_2 = splat_lane_p16(__s0_2, __p1_2); \
  __ret_2; \
})
#else
#define vdup_lane_p16(__p0_3, __p1_3) __extension__ ({ \
  poly16x4_t __s0_3 = __p0_3; \
  poly16x4_t __rev0_3;  __rev0_3 = __builtin_shufflevector(__s0_3, __s0_3, 3, 2, 1, 0); \
  poly16x4_t __ret_3; \
  __ret_3 = __noswap_splat_lane_p16(__rev0_3, __p1_3); \
  __ret_3 = __builtin_shufflevector(__ret_3, __ret_3, 3, 2, 1, 0); \
  __ret_3; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vdupq_lane_p8(__p0_4, __p1_4) __extension__ ({ \
  poly8x8_t __s0_4 = __p0_4; \
  poly8x16_t __ret_4; \
  __ret_4 = splatq_lane_p8(__s0_4, __p1_4); \
  __ret_4; \
})
#else
#define vdupq_lane_p8(__p0_5, __p1_5) __extension__ ({ \
  poly8x8_t __s0_5 = __p0_5; \
  poly8x8_t __rev0_5;  __rev0_5 = __builtin_shufflevector(__s0_5, __s0_5, 7, 6, 5, 4, 3, 2, 1, 0); \
  poly8x16_t __ret_5; \
  __ret_5 = __noswap_splatq_lane_p8(__rev0_5, __p1_5); \
  __ret_5 = __builtin_shufflevector(__ret_5, __ret_5, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret_5; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vdupq_lane_p16(__p0_6, __p1_6) __extension__ ({ \
  poly16x4_t __s0_6 = __p0_6; \
  poly16x8_t __ret_6; \
  __ret_6 = splatq_lane_p16(__s0_6, __p1_6); \
  __ret_6; \
})
#else
#define vdupq_lane_p16(__p0_7, __p1_7) __extension__ ({ \
  poly16x4_t __s0_7 = __p0_7; \
  poly16x4_t __rev0_7;  __rev0_7 = __builtin_shufflevector(__s0_7, __s0_7, 3, 2, 1, 0); \
  poly16x8_t __ret_7; \
  __ret_7 = __noswap_splatq_lane_p16(__rev0_7, __p1_7); \
  __ret_7 = __builtin_shufflevector(__ret_7, __ret_7, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret_7; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vdupq_lane_u8(__p0_8, __p1_8) __extension__ ({ \
  uint8x8_t __s0_8 = __p0_8; \
  uint8x16_t __ret_8; \
  __ret_8 = splatq_lane_u8(__s0_8, __p1_8); \
  __ret_8; \
})
#else
#define vdupq_lane_u8(__p0_9, __p1_9) __extension__ ({ \
  uint8x8_t __s0_9 = __p0_9; \
  uint8x8_t __rev0_9;  __rev0_9 = __builtin_shufflevector(__s0_9, __s0_9, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint8x16_t __ret_9; \
  __ret_9 = __noswap_splatq_lane_u8(__rev0_9, __p1_9); \
  __ret_9 = __builtin_shufflevector(__ret_9, __ret_9, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret_9; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vdupq_lane_u32(__p0_10, __p1_10) __extension__ ({ \
  uint32x2_t __s0_10 = __p0_10; \
  uint32x4_t __ret_10; \
  __ret_10 = splatq_lane_u32(__s0_10, __p1_10); \
  __ret_10; \
})
#else
#define vdupq_lane_u32(__p0_11, __p1_11) __extension__ ({ \
  uint32x2_t __s0_11 = __p0_11; \
  uint32x2_t __rev0_11;  __rev0_11 = __builtin_shufflevector(__s0_11, __s0_11, 1, 0); \
  uint32x4_t __ret_11; \
  __ret_11 = __noswap_splatq_lane_u32(__rev0_11, __p1_11); \
  __ret_11 = __builtin_shufflevector(__ret_11, __ret_11, 3, 2, 1, 0); \
  __ret_11; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vdupq_lane_u64(__p0_12, __p1_12) __extension__ ({ \
  uint64x1_t __s0_12 = __p0_12; \
  uint64x2_t __ret_12; \
  __ret_12 = splatq_lane_u64(__s0_12, __p1_12); \
  __ret_12; \
})
#else
#define vdupq_lane_u64(__p0_13, __p1_13) __extension__ ({ \
  uint64x1_t __s0_13 = __p0_13; \
  uint64x2_t __ret_13; \
  __ret_13 = __noswap_splatq_lane_u64(__s0_13, __p1_13); \
  __ret_13 = __builtin_shufflevector(__ret_13, __ret_13, 1, 0); \
  __ret_13; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vdupq_lane_u16(__p0_14, __p1_14) __extension__ ({ \
  uint16x4_t __s0_14 = __p0_14; \
  uint16x8_t __ret_14; \
  __ret_14 = splatq_lane_u16(__s0_14, __p1_14); \
  __ret_14; \
})
#else
#define vdupq_lane_u16(__p0_15, __p1_15) __extension__ ({ \
  uint16x4_t __s0_15 = __p0_15; \
  uint16x4_t __rev0_15;  __rev0_15 = __builtin_shufflevector(__s0_15, __s0_15, 3, 2, 1, 0); \
  uint16x8_t __ret_15; \
  __ret_15 = __noswap_splatq_lane_u16(__rev0_15, __p1_15); \
  __ret_15 = __builtin_shufflevector(__ret_15, __ret_15, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret_15; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vdupq_lane_s8(__p0_16, __p1_16) __extension__ ({ \
  int8x8_t __s0_16 = __p0_16; \
  int8x16_t __ret_16; \
  __ret_16 = splatq_lane_s8(__s0_16, __p1_16); \
  __ret_16; \
})
#else
#define vdupq_lane_s8(__p0_17, __p1_17) __extension__ ({ \
  int8x8_t __s0_17 = __p0_17; \
  int8x8_t __rev0_17;  __rev0_17 = __builtin_shufflevector(__s0_17, __s0_17, 7, 6, 5, 4, 3, 2, 1, 0); \
  int8x16_t __ret_17; \
  __ret_17 = __noswap_splatq_lane_s8(__rev0_17, __p1_17); \
  __ret_17 = __builtin_shufflevector(__ret_17, __ret_17, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret_17; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vdupq_lane_f32(__p0_18, __p1_18) __extension__ ({ \
  float32x2_t __s0_18 = __p0_18; \
  float32x4_t __ret_18; \
  __ret_18 = splatq_lane_f32(__s0_18, __p1_18); \
  __ret_18; \
})
#else
#define vdupq_lane_f32(__p0_19, __p1_19) __extension__ ({ \
  float32x2_t __s0_19 = __p0_19; \
  float32x2_t __rev0_19;  __rev0_19 = __builtin_shufflevector(__s0_19, __s0_19, 1, 0); \
  float32x4_t __ret_19; \
  __ret_19 = __noswap_splatq_lane_f32(__rev0_19, __p1_19); \
  __ret_19 = __builtin_shufflevector(__ret_19, __ret_19, 3, 2, 1, 0); \
  __ret_19; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vdupq_lane_s32(__p0_20, __p1_20) __extension__ ({ \
  int32x2_t __s0_20 = __p0_20; \
  int32x4_t __ret_20; \
  __ret_20 = splatq_lane_s32(__s0_20, __p1_20); \
  __ret_20; \
})
#else
#define vdupq_lane_s32(__p0_21, __p1_21) __extension__ ({ \
  int32x2_t __s0_21 = __p0_21; \
  int32x2_t __rev0_21;  __rev0_21 = __builtin_shufflevector(__s0_21, __s0_21, 1, 0); \
  int32x4_t __ret_21; \
  __ret_21 = __noswap_splatq_lane_s32(__rev0_21, __p1_21); \
  __ret_21 = __builtin_shufflevector(__ret_21, __ret_21, 3, 2, 1, 0); \
  __ret_21; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vdupq_lane_s64(__p0_22, __p1_22) __extension__ ({ \
  int64x1_t __s0_22 = __p0_22; \
  int64x2_t __ret_22; \
  __ret_22 = splatq_lane_s64(__s0_22, __p1_22); \
  __ret_22; \
})
#else
#define vdupq_lane_s64(__p0_23, __p1_23) __extension__ ({ \
  int64x1_t __s0_23 = __p0_23; \
  int64x2_t __ret_23; \
  __ret_23 = __noswap_splatq_lane_s64(__s0_23, __p1_23); \
  __ret_23 = __builtin_shufflevector(__ret_23, __ret_23, 1, 0); \
  __ret_23; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vdupq_lane_s16(__p0_24, __p1_24) __extension__ ({ \
  int16x4_t __s0_24 = __p0_24; \
  int16x8_t __ret_24; \
  __ret_24 = splatq_lane_s16(__s0_24, __p1_24); \
  __ret_24; \
})
#else
#define vdupq_lane_s16(__p0_25, __p1_25) __extension__ ({ \
  int16x4_t __s0_25 = __p0_25; \
  int16x4_t __rev0_25;  __rev0_25 = __builtin_shufflevector(__s0_25, __s0_25, 3, 2, 1, 0); \
  int16x8_t __ret_25; \
  __ret_25 = __noswap_splatq_lane_s16(__rev0_25, __p1_25); \
  __ret_25 = __builtin_shufflevector(__ret_25, __ret_25, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret_25; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vdup_lane_u8(__p0_26, __p1_26) __extension__ ({ \
  uint8x8_t __s0_26 = __p0_26; \
  uint8x8_t __ret_26; \
  __ret_26 = splat_lane_u8(__s0_26, __p1_26); \
  __ret_26; \
})
#else
#define vdup_lane_u8(__p0_27, __p1_27) __extension__ ({ \
  uint8x8_t __s0_27 = __p0_27; \
  uint8x8_t __rev0_27;  __rev0_27 = __builtin_shufflevector(__s0_27, __s0_27, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint8x8_t __ret_27; \
  __ret_27 = __noswap_splat_lane_u8(__rev0_27, __p1_27); \
  __ret_27 = __builtin_shufflevector(__ret_27, __ret_27, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret_27; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vdup_lane_u32(__p0_28, __p1_28) __extension__ ({ \
  uint32x2_t __s0_28 = __p0_28; \
  uint32x2_t __ret_28; \
  __ret_28 = splat_lane_u32(__s0_28, __p1_28); \
  __ret_28; \
})
#else
#define vdup_lane_u32(__p0_29, __p1_29) __extension__ ({ \
  uint32x2_t __s0_29 = __p0_29; \
  uint32x2_t __rev0_29;  __rev0_29 = __builtin_shufflevector(__s0_29, __s0_29, 1, 0); \
  uint32x2_t __ret_29; \
  __ret_29 = __noswap_splat_lane_u32(__rev0_29, __p1_29); \
  __ret_29 = __builtin_shufflevector(__ret_29, __ret_29, 1, 0); \
  __ret_29; \
})
#endif

#define vdup_lane_u64(__p0_30, __p1_30) __extension__ ({ \
  uint64x1_t __s0_30 = __p0_30; \
  uint64x1_t __ret_30; \
  __ret_30 = splat_lane_u64(__s0_30, __p1_30); \
  __ret_30; \
})
#ifdef __LITTLE_ENDIAN__
#define vdup_lane_u16(__p0_31, __p1_31) __extension__ ({ \
  uint16x4_t __s0_31 = __p0_31; \
  uint16x4_t __ret_31; \
  __ret_31 = splat_lane_u16(__s0_31, __p1_31); \
  __ret_31; \
})
#else
#define vdup_lane_u16(__p0_32, __p1_32) __extension__ ({ \
  uint16x4_t __s0_32 = __p0_32; \
  uint16x4_t __rev0_32;  __rev0_32 = __builtin_shufflevector(__s0_32, __s0_32, 3, 2, 1, 0); \
  uint16x4_t __ret_32; \
  __ret_32 = __noswap_splat_lane_u16(__rev0_32, __p1_32); \
  __ret_32 = __builtin_shufflevector(__ret_32, __ret_32, 3, 2, 1, 0); \
  __ret_32; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vdup_lane_s8(__p0_33, __p1_33) __extension__ ({ \
  int8x8_t __s0_33 = __p0_33; \
  int8x8_t __ret_33; \
  __ret_33 = splat_lane_s8(__s0_33, __p1_33); \
  __ret_33; \
})
#else
#define vdup_lane_s8(__p0_34, __p1_34) __extension__ ({ \
  int8x8_t __s0_34 = __p0_34; \
  int8x8_t __rev0_34;  __rev0_34 = __builtin_shufflevector(__s0_34, __s0_34, 7, 6, 5, 4, 3, 2, 1, 0); \
  int8x8_t __ret_34; \
  __ret_34 = __noswap_splat_lane_s8(__rev0_34, __p1_34); \
  __ret_34 = __builtin_shufflevector(__ret_34, __ret_34, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret_34; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vdup_lane_f32(__p0_35, __p1_35) __extension__ ({ \
  float32x2_t __s0_35 = __p0_35; \
  float32x2_t __ret_35; \
  __ret_35 = splat_lane_f32(__s0_35, __p1_35); \
  __ret_35; \
})
#else
#define vdup_lane_f32(__p0_36, __p1_36) __extension__ ({ \
  float32x2_t __s0_36 = __p0_36; \
  float32x2_t __rev0_36;  __rev0_36 = __builtin_shufflevector(__s0_36, __s0_36, 1, 0); \
  float32x2_t __ret_36; \
  __ret_36 = __noswap_splat_lane_f32(__rev0_36, __p1_36); \
  __ret_36 = __builtin_shufflevector(__ret_36, __ret_36, 1, 0); \
  __ret_36; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vdup_lane_s32(__p0_37, __p1_37) __extension__ ({ \
  int32x2_t __s0_37 = __p0_37; \
  int32x2_t __ret_37; \
  __ret_37 = splat_lane_s32(__s0_37, __p1_37); \
  __ret_37; \
})
#else
#define vdup_lane_s32(__p0_38, __p1_38) __extension__ ({ \
  int32x2_t __s0_38 = __p0_38; \
  int32x2_t __rev0_38;  __rev0_38 = __builtin_shufflevector(__s0_38, __s0_38, 1, 0); \
  int32x2_t __ret_38; \
  __ret_38 = __noswap_splat_lane_s32(__rev0_38, __p1_38); \
  __ret_38 = __builtin_shufflevector(__ret_38, __ret_38, 1, 0); \
  __ret_38; \
})
#endif

#define vdup_lane_s64(__p0_39, __p1_39) __extension__ ({ \
  int64x1_t __s0_39 = __p0_39; \
  int64x1_t __ret_39; \
  __ret_39 = splat_lane_s64(__s0_39, __p1_39); \
  __ret_39; \
})
#ifdef __LITTLE_ENDIAN__
#define vdup_lane_s16(__p0_40, __p1_40) __extension__ ({ \
  int16x4_t __s0_40 = __p0_40; \
  int16x4_t __ret_40; \
  __ret_40 = splat_lane_s16(__s0_40, __p1_40); \
  __ret_40; \
})
#else
#define vdup_lane_s16(__p0_41, __p1_41) __extension__ ({ \
  int16x4_t __s0_41 = __p0_41; \
  int16x4_t __rev0_41;  __rev0_41 = __builtin_shufflevector(__s0_41, __s0_41, 3, 2, 1, 0); \
  int16x4_t __ret_41; \
  __ret_41 = __noswap_splat_lane_s16(__rev0_41, __p1_41); \
  __ret_41 = __builtin_shufflevector(__ret_41, __ret_41, 3, 2, 1, 0); \
  __ret_41; \
})
#endif

#ifdef __LITTLE_ENDIAN__
__ai poly8x8_t vdup_n_p8(poly8_t __p0) {
  poly8x8_t __ret;
  __ret = (poly8x8_t) {__p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0};
  return __ret;
}
#else
__ai poly8x8_t vdup_n_p8(poly8_t __p0) {
  poly8x8_t __ret;
  __ret = (poly8x8_t) {__p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0};
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai poly16x4_t vdup_n_p16(poly16_t __p0) {
  poly16x4_t __ret;
  __ret = (poly16x4_t) {__p0, __p0, __p0, __p0};
  return __ret;
}
#else
__ai poly16x4_t vdup_n_p16(poly16_t __p0) {
  poly16x4_t __ret;
  __ret = (poly16x4_t) {__p0, __p0, __p0, __p0};
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai poly8x16_t vdupq_n_p8(poly8_t __p0) {
  poly8x16_t __ret;
  __ret = (poly8x16_t) {__p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0};
  return __ret;
}
#else
__ai poly8x16_t vdupq_n_p8(poly8_t __p0) {
  poly8x16_t __ret;
  __ret = (poly8x16_t) {__p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0};
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai poly16x8_t vdupq_n_p16(poly16_t __p0) {
  poly16x8_t __ret;
  __ret = (poly16x8_t) {__p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0};
  return __ret;
}
#else
__ai poly16x8_t vdupq_n_p16(poly16_t __p0) {
  poly16x8_t __ret;
  __ret = (poly16x8_t) {__p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0};
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x16_t vdupq_n_u8(uint8_t __p0) {
  uint8x16_t __ret;
  __ret = (uint8x16_t) {__p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0};
  return __ret;
}
#else
__ai uint8x16_t vdupq_n_u8(uint8_t __p0) {
  uint8x16_t __ret;
  __ret = (uint8x16_t) {__p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0};
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vdupq_n_u32(uint32_t __p0) {
  uint32x4_t __ret;
  __ret = (uint32x4_t) {__p0, __p0, __p0, __p0};
  return __ret;
}
#else
__ai uint32x4_t vdupq_n_u32(uint32_t __p0) {
  uint32x4_t __ret;
  __ret = (uint32x4_t) {__p0, __p0, __p0, __p0};
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint64x2_t vdupq_n_u64(uint64_t __p0) {
  uint64x2_t __ret;
  __ret = (uint64x2_t) {__p0, __p0};
  return __ret;
}
#else
__ai uint64x2_t vdupq_n_u64(uint64_t __p0) {
  uint64x2_t __ret;
  __ret = (uint64x2_t) {__p0, __p0};
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x8_t vdupq_n_u16(uint16_t __p0) {
  uint16x8_t __ret;
  __ret = (uint16x8_t) {__p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0};
  return __ret;
}
#else
__ai uint16x8_t vdupq_n_u16(uint16_t __p0) {
  uint16x8_t __ret;
  __ret = (uint16x8_t) {__p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0};
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x16_t vdupq_n_s8(int8_t __p0) {
  int8x16_t __ret;
  __ret = (int8x16_t) {__p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0};
  return __ret;
}
#else
__ai int8x16_t vdupq_n_s8(int8_t __p0) {
  int8x16_t __ret;
  __ret = (int8x16_t) {__p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0};
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai float32x4_t vdupq_n_f32(float32_t __p0) {
  float32x4_t __ret;
  __ret = (float32x4_t) {__p0, __p0, __p0, __p0};
  return __ret;
}
#else
__ai float32x4_t vdupq_n_f32(float32_t __p0) {
  float32x4_t __ret;
  __ret = (float32x4_t) {__p0, __p0, __p0, __p0};
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
#define vdupq_n_f16(__p0) __extension__ ({ \
  float16_t __s0 = __p0; \
  float16x8_t __ret; \
  __ret = (float16x8_t) {__s0, __s0, __s0, __s0, __s0, __s0, __s0, __s0}; \
  __ret; \
})
#else
#define vdupq_n_f16(__p0) __extension__ ({ \
  float16_t __s0 = __p0; \
  float16x8_t __ret; \
  __ret = (float16x8_t) {__s0, __s0, __s0, __s0, __s0, __s0, __s0, __s0}; \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x4_t vdupq_n_s32(int32_t __p0) {
  int32x4_t __ret;
  __ret = (int32x4_t) {__p0, __p0, __p0, __p0};
  return __ret;
}
#else
__ai int32x4_t vdupq_n_s32(int32_t __p0) {
  int32x4_t __ret;
  __ret = (int32x4_t) {__p0, __p0, __p0, __p0};
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int64x2_t vdupq_n_s64(int64_t __p0) {
  int64x2_t __ret;
  __ret = (int64x2_t) {__p0, __p0};
  return __ret;
}
#else
__ai int64x2_t vdupq_n_s64(int64_t __p0) {
  int64x2_t __ret;
  __ret = (int64x2_t) {__p0, __p0};
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x8_t vdupq_n_s16(int16_t __p0) {
  int16x8_t __ret;
  __ret = (int16x8_t) {__p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0};
  return __ret;
}
#else
__ai int16x8_t vdupq_n_s16(int16_t __p0) {
  int16x8_t __ret;
  __ret = (int16x8_t) {__p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0};
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x8_t vdup_n_u8(uint8_t __p0) {
  uint8x8_t __ret;
  __ret = (uint8x8_t) {__p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0};
  return __ret;
}
#else
__ai uint8x8_t vdup_n_u8(uint8_t __p0) {
  uint8x8_t __ret;
  __ret = (uint8x8_t) {__p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0};
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vdup_n_u32(uint32_t __p0) {
  uint32x2_t __ret;
  __ret = (uint32x2_t) {__p0, __p0};
  return __ret;
}
#else
__ai uint32x2_t vdup_n_u32(uint32_t __p0) {
  uint32x2_t __ret;
  __ret = (uint32x2_t) {__p0, __p0};
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

__ai uint64x1_t vdup_n_u64(uint64_t __p0) {
  uint64x1_t __ret;
  __ret = (uint64x1_t) {__p0};
  return __ret;
}
#ifdef __LITTLE_ENDIAN__
__ai uint16x4_t vdup_n_u16(uint16_t __p0) {
  uint16x4_t __ret;
  __ret = (uint16x4_t) {__p0, __p0, __p0, __p0};
  return __ret;
}
#else
__ai uint16x4_t vdup_n_u16(uint16_t __p0) {
  uint16x4_t __ret;
  __ret = (uint16x4_t) {__p0, __p0, __p0, __p0};
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x8_t vdup_n_s8(int8_t __p0) {
  int8x8_t __ret;
  __ret = (int8x8_t) {__p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0};
  return __ret;
}
#else
__ai int8x8_t vdup_n_s8(int8_t __p0) {
  int8x8_t __ret;
  __ret = (int8x8_t) {__p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0};
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai float32x2_t vdup_n_f32(float32_t __p0) {
  float32x2_t __ret;
  __ret = (float32x2_t) {__p0, __p0};
  return __ret;
}
#else
__ai float32x2_t vdup_n_f32(float32_t __p0) {
  float32x2_t __ret;
  __ret = (float32x2_t) {__p0, __p0};
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
#define vdup_n_f16(__p0) __extension__ ({ \
  float16_t __s0 = __p0; \
  float16x4_t __ret; \
  __ret = (float16x4_t) {__s0, __s0, __s0, __s0}; \
  __ret; \
})
#else
#define vdup_n_f16(__p0) __extension__ ({ \
  float16_t __s0 = __p0; \
  float16x4_t __ret; \
  __ret = (float16x4_t) {__s0, __s0, __s0, __s0}; \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x2_t vdup_n_s32(int32_t __p0) {
  int32x2_t __ret;
  __ret = (int32x2_t) {__p0, __p0};
  return __ret;
}
#else
__ai int32x2_t vdup_n_s32(int32_t __p0) {
  int32x2_t __ret;
  __ret = (int32x2_t) {__p0, __p0};
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

__ai int64x1_t vdup_n_s64(int64_t __p0) {
  int64x1_t __ret;
  __ret = (int64x1_t) {__p0};
  return __ret;
}
#ifdef __LITTLE_ENDIAN__
__ai int16x4_t vdup_n_s16(int16_t __p0) {
  int16x4_t __ret;
  __ret = (int16x4_t) {__p0, __p0, __p0, __p0};
  return __ret;
}
#else
__ai int16x4_t vdup_n_s16(int16_t __p0) {
  int16x4_t __ret;
  __ret = (int16x4_t) {__p0, __p0, __p0, __p0};
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x16_t veorq_u8(uint8x16_t __p0, uint8x16_t __p1) {
  uint8x16_t __ret;
  __ret = __p0 ^ __p1;
  return __ret;
}
#else
__ai uint8x16_t veorq_u8(uint8x16_t __p0, uint8x16_t __p1) {
  uint8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __ret;
  __ret = __rev0 ^ __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t veorq_u32(uint32x4_t __p0, uint32x4_t __p1) {
  uint32x4_t __ret;
  __ret = __p0 ^ __p1;
  return __ret;
}
#else
__ai uint32x4_t veorq_u32(uint32x4_t __p0, uint32x4_t __p1) {
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = __rev0 ^ __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint64x2_t veorq_u64(uint64x2_t __p0, uint64x2_t __p1) {
  uint64x2_t __ret;
  __ret = __p0 ^ __p1;
  return __ret;
}
#else
__ai uint64x2_t veorq_u64(uint64x2_t __p0, uint64x2_t __p1) {
  uint64x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint64x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint64x2_t __ret;
  __ret = __rev0 ^ __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x8_t veorq_u16(uint16x8_t __p0, uint16x8_t __p1) {
  uint16x8_t __ret;
  __ret = __p0 ^ __p1;
  return __ret;
}
#else
__ai uint16x8_t veorq_u16(uint16x8_t __p0, uint16x8_t __p1) {
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __ret;
  __ret = __rev0 ^ __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x16_t veorq_s8(int8x16_t __p0, int8x16_t __p1) {
  int8x16_t __ret;
  __ret = __p0 ^ __p1;
  return __ret;
}
#else
__ai int8x16_t veorq_s8(int8x16_t __p0, int8x16_t __p1) {
  int8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __ret;
  __ret = __rev0 ^ __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x4_t veorq_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __ret;
  __ret = __p0 ^ __p1;
  return __ret;
}
#else
__ai int32x4_t veorq_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int32x4_t __ret;
  __ret = __rev0 ^ __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int64x2_t veorq_s64(int64x2_t __p0, int64x2_t __p1) {
  int64x2_t __ret;
  __ret = __p0 ^ __p1;
  return __ret;
}
#else
__ai int64x2_t veorq_s64(int64x2_t __p0, int64x2_t __p1) {
  int64x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int64x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  int64x2_t __ret;
  __ret = __rev0 ^ __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x8_t veorq_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __ret;
  __ret = __p0 ^ __p1;
  return __ret;
}
#else
__ai int16x8_t veorq_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __ret;
  __ret = __rev0 ^ __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x8_t veor_u8(uint8x8_t __p0, uint8x8_t __p1) {
  uint8x8_t __ret;
  __ret = __p0 ^ __p1;
  return __ret;
}
#else
__ai uint8x8_t veor_u8(uint8x8_t __p0, uint8x8_t __p1) {
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __ret;
  __ret = __rev0 ^ __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t veor_u32(uint32x2_t __p0, uint32x2_t __p1) {
  uint32x2_t __ret;
  __ret = __p0 ^ __p1;
  return __ret;
}
#else
__ai uint32x2_t veor_u32(uint32x2_t __p0, uint32x2_t __p1) {
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint32x2_t __ret;
  __ret = __rev0 ^ __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

__ai uint64x1_t veor_u64(uint64x1_t __p0, uint64x1_t __p1) {
  uint64x1_t __ret;
  __ret = __p0 ^ __p1;
  return __ret;
}
#ifdef __LITTLE_ENDIAN__
__ai uint16x4_t veor_u16(uint16x4_t __p0, uint16x4_t __p1) {
  uint16x4_t __ret;
  __ret = __p0 ^ __p1;
  return __ret;
}
#else
__ai uint16x4_t veor_u16(uint16x4_t __p0, uint16x4_t __p1) {
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint16x4_t __ret;
  __ret = __rev0 ^ __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x8_t veor_s8(int8x8_t __p0, int8x8_t __p1) {
  int8x8_t __ret;
  __ret = __p0 ^ __p1;
  return __ret;
}
#else
__ai int8x8_t veor_s8(int8x8_t __p0, int8x8_t __p1) {
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __ret;
  __ret = __rev0 ^ __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x2_t veor_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __ret;
  __ret = __p0 ^ __p1;
  return __ret;
}
#else
__ai int32x2_t veor_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  int32x2_t __ret;
  __ret = __rev0 ^ __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

__ai int64x1_t veor_s64(int64x1_t __p0, int64x1_t __p1) {
  int64x1_t __ret;
  __ret = __p0 ^ __p1;
  return __ret;
}
#ifdef __LITTLE_ENDIAN__
__ai int16x4_t veor_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __ret;
  __ret = __p0 ^ __p1;
  return __ret;
}
#else
__ai int16x4_t veor_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int16x4_t __ret;
  __ret = __rev0 ^ __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
#define vext_p8(__p0, __p1, __p2) __extension__ ({ \
  poly8x8_t __s0 = __p0; \
  poly8x8_t __s1 = __p1; \
  poly8x8_t __ret; \
  __ret = (poly8x8_t) __builtin_neon_vext_v((int8x8_t)__s0, (int8x8_t)__s1, __p2, 4); \
  __ret; \
})
#else
#define vext_p8(__p0, __p1, __p2) __extension__ ({ \
  poly8x8_t __s0 = __p0; \
  poly8x8_t __s1 = __p1; \
  poly8x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  poly8x8_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 7, 6, 5, 4, 3, 2, 1, 0); \
  poly8x8_t __ret; \
  __ret = (poly8x8_t) __builtin_neon_vext_v((int8x8_t)__rev0, (int8x8_t)__rev1, __p2, 4); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vext_p16(__p0, __p1, __p2) __extension__ ({ \
  poly16x4_t __s0 = __p0; \
  poly16x4_t __s1 = __p1; \
  poly16x4_t __ret; \
  __ret = (poly16x4_t) __builtin_neon_vext_v((int8x8_t)__s0, (int8x8_t)__s1, __p2, 5); \
  __ret; \
})
#else
#define vext_p16(__p0, __p1, __p2) __extension__ ({ \
  poly16x4_t __s0 = __p0; \
  poly16x4_t __s1 = __p1; \
  poly16x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  poly16x4_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 3, 2, 1, 0); \
  poly16x4_t __ret; \
  __ret = (poly16x4_t) __builtin_neon_vext_v((int8x8_t)__rev0, (int8x8_t)__rev1, __p2, 5); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vextq_p8(__p0, __p1, __p2) __extension__ ({ \
  poly8x16_t __s0 = __p0; \
  poly8x16_t __s1 = __p1; \
  poly8x16_t __ret; \
  __ret = (poly8x16_t) __builtin_neon_vextq_v((int8x16_t)__s0, (int8x16_t)__s1, __p2, 36); \
  __ret; \
})
#else
#define vextq_p8(__p0, __p1, __p2) __extension__ ({ \
  poly8x16_t __s0 = __p0; \
  poly8x16_t __s1 = __p1; \
  poly8x16_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  poly8x16_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  poly8x16_t __ret; \
  __ret = (poly8x16_t) __builtin_neon_vextq_v((int8x16_t)__rev0, (int8x16_t)__rev1, __p2, 36); \
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vextq_p16(__p0, __p1, __p2) __extension__ ({ \
  poly16x8_t __s0 = __p0; \
  poly16x8_t __s1 = __p1; \
  poly16x8_t __ret; \
  __ret = (poly16x8_t) __builtin_neon_vextq_v((int8x16_t)__s0, (int8x16_t)__s1, __p2, 37); \
  __ret; \
})
#else
#define vextq_p16(__p0, __p1, __p2) __extension__ ({ \
  poly16x8_t __s0 = __p0; \
  poly16x8_t __s1 = __p1; \
  poly16x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  poly16x8_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 7, 6, 5, 4, 3, 2, 1, 0); \
  poly16x8_t __ret; \
  __ret = (poly16x8_t) __builtin_neon_vextq_v((int8x16_t)__rev0, (int8x16_t)__rev1, __p2, 37); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vextq_u8(__p0, __p1, __p2) __extension__ ({ \
  uint8x16_t __s0 = __p0; \
  uint8x16_t __s1 = __p1; \
  uint8x16_t __ret; \
  __ret = (uint8x16_t) __builtin_neon_vextq_v((int8x16_t)__s0, (int8x16_t)__s1, __p2, 48); \
  __ret; \
})
#else
#define vextq_u8(__p0, __p1, __p2) __extension__ ({ \
  uint8x16_t __s0 = __p0; \
  uint8x16_t __s1 = __p1; \
  uint8x16_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint8x16_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint8x16_t __ret; \
  __ret = (uint8x16_t) __builtin_neon_vextq_v((int8x16_t)__rev0, (int8x16_t)__rev1, __p2, 48); \
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vextq_u32(__p0, __p1, __p2) __extension__ ({ \
  uint32x4_t __s0 = __p0; \
  uint32x4_t __s1 = __p1; \
  uint32x4_t __ret; \
  __ret = (uint32x4_t) __builtin_neon_vextq_v((int8x16_t)__s0, (int8x16_t)__s1, __p2, 50); \
  __ret; \
})
#else
#define vextq_u32(__p0, __p1, __p2) __extension__ ({ \
  uint32x4_t __s0 = __p0; \
  uint32x4_t __s1 = __p1; \
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  uint32x4_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 3, 2, 1, 0); \
  uint32x4_t __ret; \
  __ret = (uint32x4_t) __builtin_neon_vextq_v((int8x16_t)__rev0, (int8x16_t)__rev1, __p2, 50); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vextq_u64(__p0, __p1, __p2) __extension__ ({ \
  uint64x2_t __s0 = __p0; \
  uint64x2_t __s1 = __p1; \
  uint64x2_t __ret; \
  __ret = (uint64x2_t) __builtin_neon_vextq_v((int8x16_t)__s0, (int8x16_t)__s1, __p2, 51); \
  __ret; \
})
#else
#define vextq_u64(__p0, __p1, __p2) __extension__ ({ \
  uint64x2_t __s0 = __p0; \
  uint64x2_t __s1 = __p1; \
  uint64x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  uint64x2_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 1, 0); \
  uint64x2_t __ret; \
  __ret = (uint64x2_t) __builtin_neon_vextq_v((int8x16_t)__rev0, (int8x16_t)__rev1, __p2, 51); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vextq_u16(__p0, __p1, __p2) __extension__ ({ \
  uint16x8_t __s0 = __p0; \
  uint16x8_t __s1 = __p1; \
  uint16x8_t __ret; \
  __ret = (uint16x8_t) __builtin_neon_vextq_v((int8x16_t)__s0, (int8x16_t)__s1, __p2, 49); \
  __ret; \
})
#else
#define vextq_u16(__p0, __p1, __p2) __extension__ ({ \
  uint16x8_t __s0 = __p0; \
  uint16x8_t __s1 = __p1; \
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint16x8_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint16x8_t __ret; \
  __ret = (uint16x8_t) __builtin_neon_vextq_v((int8x16_t)__rev0, (int8x16_t)__rev1, __p2, 49); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vextq_s8(__p0, __p1, __p2) __extension__ ({ \
  int8x16_t __s0 = __p0; \
  int8x16_t __s1 = __p1; \
  int8x16_t __ret; \
  __ret = (int8x16_t) __builtin_neon_vextq_v((int8x16_t)__s0, (int8x16_t)__s1, __p2, 32); \
  __ret; \
})
#else
#define vextq_s8(__p0, __p1, __p2) __extension__ ({ \
  int8x16_t __s0 = __p0; \
  int8x16_t __s1 = __p1; \
  int8x16_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  int8x16_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  int8x16_t __ret; \
  __ret = (int8x16_t) __builtin_neon_vextq_v((int8x16_t)__rev0, (int8x16_t)__rev1, __p2, 32); \
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vextq_f32(__p0, __p1, __p2) __extension__ ({ \
  float32x4_t __s0 = __p0; \
  float32x4_t __s1 = __p1; \
  float32x4_t __ret; \
  __ret = (float32x4_t) __builtin_neon_vextq_v((int8x16_t)__s0, (int8x16_t)__s1, __p2, 41); \
  __ret; \
})
#else
#define vextq_f32(__p0, __p1, __p2) __extension__ ({ \
  float32x4_t __s0 = __p0; \
  float32x4_t __s1 = __p1; \
  float32x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  float32x4_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 3, 2, 1, 0); \
  float32x4_t __ret; \
  __ret = (float32x4_t) __builtin_neon_vextq_v((int8x16_t)__rev0, (int8x16_t)__rev1, __p2, 41); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vextq_s32(__p0, __p1, __p2) __extension__ ({ \
  int32x4_t __s0 = __p0; \
  int32x4_t __s1 = __p1; \
  int32x4_t __ret; \
  __ret = (int32x4_t) __builtin_neon_vextq_v((int8x16_t)__s0, (int8x16_t)__s1, __p2, 34); \
  __ret; \
})
#else
#define vextq_s32(__p0, __p1, __p2) __extension__ ({ \
  int32x4_t __s0 = __p0; \
  int32x4_t __s1 = __p1; \
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  int32x4_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 3, 2, 1, 0); \
  int32x4_t __ret; \
  __ret = (int32x4_t) __builtin_neon_vextq_v((int8x16_t)__rev0, (int8x16_t)__rev1, __p2, 34); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vextq_s64(__p0, __p1, __p2) __extension__ ({ \
  int64x2_t __s0 = __p0; \
  int64x2_t __s1 = __p1; \
  int64x2_t __ret; \
  __ret = (int64x2_t) __builtin_neon_vextq_v((int8x16_t)__s0, (int8x16_t)__s1, __p2, 35); \
  __ret; \
})
#else
#define vextq_s64(__p0, __p1, __p2) __extension__ ({ \
  int64x2_t __s0 = __p0; \
  int64x2_t __s1 = __p1; \
  int64x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  int64x2_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 1, 0); \
  int64x2_t __ret; \
  __ret = (int64x2_t) __builtin_neon_vextq_v((int8x16_t)__rev0, (int8x16_t)__rev1, __p2, 35); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vextq_s16(__p0, __p1, __p2) __extension__ ({ \
  int16x8_t __s0 = __p0; \
  int16x8_t __s1 = __p1; \
  int16x8_t __ret; \
  __ret = (int16x8_t) __builtin_neon_vextq_v((int8x16_t)__s0, (int8x16_t)__s1, __p2, 33); \
  __ret; \
})
#else
#define vextq_s16(__p0, __p1, __p2) __extension__ ({ \
  int16x8_t __s0 = __p0; \
  int16x8_t __s1 = __p1; \
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  int16x8_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 7, 6, 5, 4, 3, 2, 1, 0); \
  int16x8_t __ret; \
  __ret = (int16x8_t) __builtin_neon_vextq_v((int8x16_t)__rev0, (int8x16_t)__rev1, __p2, 33); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vext_u8(__p0, __p1, __p2) __extension__ ({ \
  uint8x8_t __s0 = __p0; \
  uint8x8_t __s1 = __p1; \
  uint8x8_t __ret; \
  __ret = (uint8x8_t) __builtin_neon_vext_v((int8x8_t)__s0, (int8x8_t)__s1, __p2, 16); \
  __ret; \
})
#else
#define vext_u8(__p0, __p1, __p2) __extension__ ({ \
  uint8x8_t __s0 = __p0; \
  uint8x8_t __s1 = __p1; \
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint8x8_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint8x8_t __ret; \
  __ret = (uint8x8_t) __builtin_neon_vext_v((int8x8_t)__rev0, (int8x8_t)__rev1, __p2, 16); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vext_u32(__p0, __p1, __p2) __extension__ ({ \
  uint32x2_t __s0 = __p0; \
  uint32x2_t __s1 = __p1; \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_vext_v((int8x8_t)__s0, (int8x8_t)__s1, __p2, 18); \
  __ret; \
})
#else
#define vext_u32(__p0, __p1, __p2) __extension__ ({ \
  uint32x2_t __s0 = __p0; \
  uint32x2_t __s1 = __p1; \
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  uint32x2_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 1, 0); \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_vext_v((int8x8_t)__rev0, (int8x8_t)__rev1, __p2, 18); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#define vext_u64(__p0, __p1, __p2) __extension__ ({ \
  uint64x1_t __s0 = __p0; \
  uint64x1_t __s1 = __p1; \
  uint64x1_t __ret; \
  __ret = (uint64x1_t) __builtin_neon_vext_v((int8x8_t)__s0, (int8x8_t)__s1, __p2, 19); \
  __ret; \
})
#ifdef __LITTLE_ENDIAN__
#define vext_u16(__p0, __p1, __p2) __extension__ ({ \
  uint16x4_t __s0 = __p0; \
  uint16x4_t __s1 = __p1; \
  uint16x4_t __ret; \
  __ret = (uint16x4_t) __builtin_neon_vext_v((int8x8_t)__s0, (int8x8_t)__s1, __p2, 17); \
  __ret; \
})
#else
#define vext_u16(__p0, __p1, __p2) __extension__ ({ \
  uint16x4_t __s0 = __p0; \
  uint16x4_t __s1 = __p1; \
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  uint16x4_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 3, 2, 1, 0); \
  uint16x4_t __ret; \
  __ret = (uint16x4_t) __builtin_neon_vext_v((int8x8_t)__rev0, (int8x8_t)__rev1, __p2, 17); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vext_s8(__p0, __p1, __p2) __extension__ ({ \
  int8x8_t __s0 = __p0; \
  int8x8_t __s1 = __p1; \
  int8x8_t __ret; \
  __ret = (int8x8_t) __builtin_neon_vext_v((int8x8_t)__s0, (int8x8_t)__s1, __p2, 0); \
  __ret; \
})
#else
#define vext_s8(__p0, __p1, __p2) __extension__ ({ \
  int8x8_t __s0 = __p0; \
  int8x8_t __s1 = __p1; \
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  int8x8_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 7, 6, 5, 4, 3, 2, 1, 0); \
  int8x8_t __ret; \
  __ret = (int8x8_t) __builtin_neon_vext_v((int8x8_t)__rev0, (int8x8_t)__rev1, __p2, 0); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vext_f32(__p0, __p1, __p2) __extension__ ({ \
  float32x2_t __s0 = __p0; \
  float32x2_t __s1 = __p1; \
  float32x2_t __ret; \
  __ret = (float32x2_t) __builtin_neon_vext_v((int8x8_t)__s0, (int8x8_t)__s1, __p2, 9); \
  __ret; \
})
#else
#define vext_f32(__p0, __p1, __p2) __extension__ ({ \
  float32x2_t __s0 = __p0; \
  float32x2_t __s1 = __p1; \
  float32x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  float32x2_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 1, 0); \
  float32x2_t __ret; \
  __ret = (float32x2_t) __builtin_neon_vext_v((int8x8_t)__rev0, (int8x8_t)__rev1, __p2, 9); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vext_s32(__p0, __p1, __p2) __extension__ ({ \
  int32x2_t __s0 = __p0; \
  int32x2_t __s1 = __p1; \
  int32x2_t __ret; \
  __ret = (int32x2_t) __builtin_neon_vext_v((int8x8_t)__s0, (int8x8_t)__s1, __p2, 2); \
  __ret; \
})
#else
#define vext_s32(__p0, __p1, __p2) __extension__ ({ \
  int32x2_t __s0 = __p0; \
  int32x2_t __s1 = __p1; \
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 1, 0); \
  int32x2_t __ret; \
  __ret = (int32x2_t) __builtin_neon_vext_v((int8x8_t)__rev0, (int8x8_t)__rev1, __p2, 2); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#define vext_s64(__p0, __p1, __p2) __extension__ ({ \
  int64x1_t __s0 = __p0; \
  int64x1_t __s1 = __p1; \
  int64x1_t __ret; \
  __ret = (int64x1_t) __builtin_neon_vext_v((int8x8_t)__s0, (int8x8_t)__s1, __p2, 3); \
  __ret; \
})
#ifdef __LITTLE_ENDIAN__
#define vext_s16(__p0, __p1, __p2) __extension__ ({ \
  int16x4_t __s0 = __p0; \
  int16x4_t __s1 = __p1; \
  int16x4_t __ret; \
  __ret = (int16x4_t) __builtin_neon_vext_v((int8x8_t)__s0, (int8x8_t)__s1, __p2, 1); \
  __ret; \
})
#else
#define vext_s16(__p0, __p1, __p2) __extension__ ({ \
  int16x4_t __s0 = __p0; \
  int16x4_t __s1 = __p1; \
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 3, 2, 1, 0); \
  int16x4_t __ret; \
  __ret = (int16x4_t) __builtin_neon_vext_v((int8x8_t)__rev0, (int8x8_t)__rev1, __p2, 1); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
__ai poly8x8_t vget_high_p8(poly8x16_t __p0) {
  poly8x8_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 8, 9, 10, 11, 12, 13, 14, 15);
  return __ret;
}
#else
__ai poly8x8_t vget_high_p8(poly8x16_t __p0) {
  poly8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  poly8x8_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 8, 9, 10, 11, 12, 13, 14, 15);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
__ai poly8x8_t __noswap_vget_high_p8(poly8x16_t __p0) {
  poly8x8_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 8, 9, 10, 11, 12, 13, 14, 15);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai poly16x4_t vget_high_p16(poly16x8_t __p0) {
  poly16x4_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 4, 5, 6, 7);
  return __ret;
}
#else
__ai poly16x4_t vget_high_p16(poly16x8_t __p0) {
  poly16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  poly16x4_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 4, 5, 6, 7);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x8_t vget_high_u8(uint8x16_t __p0) {
  uint8x8_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 8, 9, 10, 11, 12, 13, 14, 15);
  return __ret;
}
#else
__ai uint8x8_t vget_high_u8(uint8x16_t __p0) {
  uint8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 8, 9, 10, 11, 12, 13, 14, 15);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
__ai uint8x8_t __noswap_vget_high_u8(uint8x16_t __p0) {
  uint8x8_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 8, 9, 10, 11, 12, 13, 14, 15);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vget_high_u32(uint32x4_t __p0) {
  uint32x2_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 2, 3);
  return __ret;
}
#else
__ai uint32x2_t vget_high_u32(uint32x4_t __p0) {
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint32x2_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 2, 3);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
__ai uint32x2_t __noswap_vget_high_u32(uint32x4_t __p0) {
  uint32x2_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 2, 3);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint64x1_t vget_high_u64(uint64x2_t __p0) {
  uint64x1_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 1);
  return __ret;
}
#else
__ai uint64x1_t vget_high_u64(uint64x2_t __p0) {
  uint64x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint64x1_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 1);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x4_t vget_high_u16(uint16x8_t __p0) {
  uint16x4_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 4, 5, 6, 7);
  return __ret;
}
#else
__ai uint16x4_t vget_high_u16(uint16x8_t __p0) {
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x4_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 4, 5, 6, 7);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
__ai uint16x4_t __noswap_vget_high_u16(uint16x8_t __p0) {
  uint16x4_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 4, 5, 6, 7);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x8_t vget_high_s8(int8x16_t __p0) {
  int8x8_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 8, 9, 10, 11, 12, 13, 14, 15);
  return __ret;
}
#else
__ai int8x8_t vget_high_s8(int8x16_t __p0) {
  int8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 8, 9, 10, 11, 12, 13, 14, 15);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
__ai int8x8_t __noswap_vget_high_s8(int8x16_t __p0) {
  int8x8_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 8, 9, 10, 11, 12, 13, 14, 15);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai float32x2_t vget_high_f32(float32x4_t __p0) {
  float32x2_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 2, 3);
  return __ret;
}
#else
__ai float32x2_t vget_high_f32(float32x4_t __p0) {
  float32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  float32x2_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 2, 3);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
__ai float32x2_t __noswap_vget_high_f32(float32x4_t __p0) {
  float32x2_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 2, 3);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai float16x4_t vget_high_f16(float16x8_t __p0) {
  float16x4_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 4, 5, 6, 7);
  return __ret;
}
#else
__ai float16x4_t vget_high_f16(float16x8_t __p0) {
  float16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  float16x4_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 4, 5, 6, 7);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
__ai float16x4_t __noswap_vget_high_f16(float16x8_t __p0) {
  float16x4_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 4, 5, 6, 7);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x2_t vget_high_s32(int32x4_t __p0) {
  int32x2_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 2, 3);
  return __ret;
}
#else
__ai int32x2_t vget_high_s32(int32x4_t __p0) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x2_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 2, 3);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
__ai int32x2_t __noswap_vget_high_s32(int32x4_t __p0) {
  int32x2_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 2, 3);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int64x1_t vget_high_s64(int64x2_t __p0) {
  int64x1_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 1);
  return __ret;
}
#else
__ai int64x1_t vget_high_s64(int64x2_t __p0) {
  int64x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int64x1_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 1);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x4_t vget_high_s16(int16x8_t __p0) {
  int16x4_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 4, 5, 6, 7);
  return __ret;
}
#else
__ai int16x4_t vget_high_s16(int16x8_t __p0) {
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x4_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 4, 5, 6, 7);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
__ai int16x4_t __noswap_vget_high_s16(int16x8_t __p0) {
  int16x4_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 4, 5, 6, 7);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
#define vget_lane_p8(__p0, __p1) __extension__ ({ \
  poly8x8_t __s0 = __p0; \
  poly8_t __ret; \
  __ret = (poly8_t) __builtin_neon_vget_lane_i8((poly8x8_t)__s0, __p1); \
  __ret; \
})
#else
#define vget_lane_p8(__p0, __p1) __extension__ ({ \
  poly8x8_t __s0 = __p0; \
  poly8x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  poly8_t __ret; \
  __ret = (poly8_t) __builtin_neon_vget_lane_i8((poly8x8_t)__rev0, __p1); \
  __ret; \
})
#define __noswap_vget_lane_p8(__p0, __p1) __extension__ ({ \
  poly8x8_t __s0 = __p0; \
  poly8_t __ret; \
  __ret = (poly8_t) __builtin_neon_vget_lane_i8((poly8x8_t)__s0, __p1); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vget_lane_p16(__p0, __p1) __extension__ ({ \
  poly16x4_t __s0 = __p0; \
  poly16_t __ret; \
  __ret = (poly16_t) __builtin_neon_vget_lane_i16((poly16x4_t)__s0, __p1); \
  __ret; \
})
#else
#define vget_lane_p16(__p0, __p1) __extension__ ({ \
  poly16x4_t __s0 = __p0; \
  poly16x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  poly16_t __ret; \
  __ret = (poly16_t) __builtin_neon_vget_lane_i16((poly16x4_t)__rev0, __p1); \
  __ret; \
})
#define __noswap_vget_lane_p16(__p0, __p1) __extension__ ({ \
  poly16x4_t __s0 = __p0; \
  poly16_t __ret; \
  __ret = (poly16_t) __builtin_neon_vget_lane_i16((poly16x4_t)__s0, __p1); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vgetq_lane_p8(__p0, __p1) __extension__ ({ \
  poly8x16_t __s0 = __p0; \
  poly8_t __ret; \
  __ret = (poly8_t) __builtin_neon_vgetq_lane_i8((poly8x16_t)__s0, __p1); \
  __ret; \
})
#else
#define vgetq_lane_p8(__p0, __p1) __extension__ ({ \
  poly8x16_t __s0 = __p0; \
  poly8x16_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  poly8_t __ret; \
  __ret = (poly8_t) __builtin_neon_vgetq_lane_i8((poly8x16_t)__rev0, __p1); \
  __ret; \
})
#define __noswap_vgetq_lane_p8(__p0, __p1) __extension__ ({ \
  poly8x16_t __s0 = __p0; \
  poly8_t __ret; \
  __ret = (poly8_t) __builtin_neon_vgetq_lane_i8((poly8x16_t)__s0, __p1); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vgetq_lane_p16(__p0, __p1) __extension__ ({ \
  poly16x8_t __s0 = __p0; \
  poly16_t __ret; \
  __ret = (poly16_t) __builtin_neon_vgetq_lane_i16((poly16x8_t)__s0, __p1); \
  __ret; \
})
#else
#define vgetq_lane_p16(__p0, __p1) __extension__ ({ \
  poly16x8_t __s0 = __p0; \
  poly16x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  poly16_t __ret; \
  __ret = (poly16_t) __builtin_neon_vgetq_lane_i16((poly16x8_t)__rev0, __p1); \
  __ret; \
})
#define __noswap_vgetq_lane_p16(__p0, __p1) __extension__ ({ \
  poly16x8_t __s0 = __p0; \
  poly16_t __ret; \
  __ret = (poly16_t) __builtin_neon_vgetq_lane_i16((poly16x8_t)__s0, __p1); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vgetq_lane_u8(__p0, __p1) __extension__ ({ \
  uint8x16_t __s0 = __p0; \
  uint8_t __ret; \
  __ret = (uint8_t) __builtin_neon_vgetq_lane_i8((int8x16_t)__s0, __p1); \
  __ret; \
})
#else
#define vgetq_lane_u8(__p0, __p1) __extension__ ({ \
  uint8x16_t __s0 = __p0; \
  uint8x16_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint8_t __ret; \
  __ret = (uint8_t) __builtin_neon_vgetq_lane_i8((int8x16_t)__rev0, __p1); \
  __ret; \
})
#define __noswap_vgetq_lane_u8(__p0, __p1) __extension__ ({ \
  uint8x16_t __s0 = __p0; \
  uint8_t __ret; \
  __ret = (uint8_t) __builtin_neon_vgetq_lane_i8((int8x16_t)__s0, __p1); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vgetq_lane_u32(__p0, __p1) __extension__ ({ \
  uint32x4_t __s0 = __p0; \
  uint32_t __ret; \
  __ret = (uint32_t) __builtin_neon_vgetq_lane_i32((int32x4_t)__s0, __p1); \
  __ret; \
})
#else
#define vgetq_lane_u32(__p0, __p1) __extension__ ({ \
  uint32x4_t __s0 = __p0; \
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  uint32_t __ret; \
  __ret = (uint32_t) __builtin_neon_vgetq_lane_i32((int32x4_t)__rev0, __p1); \
  __ret; \
})
#define __noswap_vgetq_lane_u32(__p0, __p1) __extension__ ({ \
  uint32x4_t __s0 = __p0; \
  uint32_t __ret; \
  __ret = (uint32_t) __builtin_neon_vgetq_lane_i32((int32x4_t)__s0, __p1); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vgetq_lane_u64(__p0, __p1) __extension__ ({ \
  uint64x2_t __s0 = __p0; \
  uint64_t __ret; \
  __ret = (uint64_t) __builtin_neon_vgetq_lane_i64((int64x2_t)__s0, __p1); \
  __ret; \
})
#else
#define vgetq_lane_u64(__p0, __p1) __extension__ ({ \
  uint64x2_t __s0 = __p0; \
  uint64x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  uint64_t __ret; \
  __ret = (uint64_t) __builtin_neon_vgetq_lane_i64((int64x2_t)__rev0, __p1); \
  __ret; \
})
#define __noswap_vgetq_lane_u64(__p0, __p1) __extension__ ({ \
  uint64x2_t __s0 = __p0; \
  uint64_t __ret; \
  __ret = (uint64_t) __builtin_neon_vgetq_lane_i64((int64x2_t)__s0, __p1); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vgetq_lane_u16(__p0, __p1) __extension__ ({ \
  uint16x8_t __s0 = __p0; \
  uint16_t __ret; \
  __ret = (uint16_t) __builtin_neon_vgetq_lane_i16((int16x8_t)__s0, __p1); \
  __ret; \
})
#else
#define vgetq_lane_u16(__p0, __p1) __extension__ ({ \
  uint16x8_t __s0 = __p0; \
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint16_t __ret; \
  __ret = (uint16_t) __builtin_neon_vgetq_lane_i16((int16x8_t)__rev0, __p1); \
  __ret; \
})
#define __noswap_vgetq_lane_u16(__p0, __p1) __extension__ ({ \
  uint16x8_t __s0 = __p0; \
  uint16_t __ret; \
  __ret = (uint16_t) __builtin_neon_vgetq_lane_i16((int16x8_t)__s0, __p1); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vgetq_lane_s8(__p0, __p1) __extension__ ({ \
  int8x16_t __s0 = __p0; \
  int8_t __ret; \
  __ret = (int8_t) __builtin_neon_vgetq_lane_i8((int8x16_t)__s0, __p1); \
  __ret; \
})
#else
#define vgetq_lane_s8(__p0, __p1) __extension__ ({ \
  int8x16_t __s0 = __p0; \
  int8x16_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  int8_t __ret; \
  __ret = (int8_t) __builtin_neon_vgetq_lane_i8((int8x16_t)__rev0, __p1); \
  __ret; \
})
#define __noswap_vgetq_lane_s8(__p0, __p1) __extension__ ({ \
  int8x16_t __s0 = __p0; \
  int8_t __ret; \
  __ret = (int8_t) __builtin_neon_vgetq_lane_i8((int8x16_t)__s0, __p1); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vgetq_lane_f32(__p0, __p1) __extension__ ({ \
  float32x4_t __s0 = __p0; \
  float32_t __ret; \
  __ret = (float32_t) __builtin_neon_vgetq_lane_f32((float32x4_t)__s0, __p1); \
  __ret; \
})
#else
#define vgetq_lane_f32(__p0, __p1) __extension__ ({ \
  float32x4_t __s0 = __p0; \
  float32x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  float32_t __ret; \
  __ret = (float32_t) __builtin_neon_vgetq_lane_f32((float32x4_t)__rev0, __p1); \
  __ret; \
})
#define __noswap_vgetq_lane_f32(__p0, __p1) __extension__ ({ \
  float32x4_t __s0 = __p0; \
  float32_t __ret; \
  __ret = (float32_t) __builtin_neon_vgetq_lane_f32((float32x4_t)__s0, __p1); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vgetq_lane_s32(__p0, __p1) __extension__ ({ \
  int32x4_t __s0 = __p0; \
  int32_t __ret; \
  __ret = (int32_t) __builtin_neon_vgetq_lane_i32((int32x4_t)__s0, __p1); \
  __ret; \
})
#else
#define vgetq_lane_s32(__p0, __p1) __extension__ ({ \
  int32x4_t __s0 = __p0; \
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  int32_t __ret; \
  __ret = (int32_t) __builtin_neon_vgetq_lane_i32((int32x4_t)__rev0, __p1); \
  __ret; \
})
#define __noswap_vgetq_lane_s32(__p0, __p1) __extension__ ({ \
  int32x4_t __s0 = __p0; \
  int32_t __ret; \
  __ret = (int32_t) __builtin_neon_vgetq_lane_i32((int32x4_t)__s0, __p1); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vgetq_lane_s64(__p0, __p1) __extension__ ({ \
  int64x2_t __s0 = __p0; \
  int64_t __ret; \
  __ret = (int64_t) __builtin_neon_vgetq_lane_i64((int64x2_t)__s0, __p1); \
  __ret; \
})
#else
#define vgetq_lane_s64(__p0, __p1) __extension__ ({ \
  int64x2_t __s0 = __p0; \
  int64x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  int64_t __ret; \
  __ret = (int64_t) __builtin_neon_vgetq_lane_i64((int64x2_t)__rev0, __p1); \
  __ret; \
})
#define __noswap_vgetq_lane_s64(__p0, __p1) __extension__ ({ \
  int64x2_t __s0 = __p0; \
  int64_t __ret; \
  __ret = (int64_t) __builtin_neon_vgetq_lane_i64((int64x2_t)__s0, __p1); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vgetq_lane_s16(__p0, __p1) __extension__ ({ \
  int16x8_t __s0 = __p0; \
  int16_t __ret; \
  __ret = (int16_t) __builtin_neon_vgetq_lane_i16((int16x8_t)__s0, __p1); \
  __ret; \
})
#else
#define vgetq_lane_s16(__p0, __p1) __extension__ ({ \
  int16x8_t __s0 = __p0; \
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  int16_t __ret; \
  __ret = (int16_t) __builtin_neon_vgetq_lane_i16((int16x8_t)__rev0, __p1); \
  __ret; \
})
#define __noswap_vgetq_lane_s16(__p0, __p1) __extension__ ({ \
  int16x8_t __s0 = __p0; \
  int16_t __ret; \
  __ret = (int16_t) __builtin_neon_vgetq_lane_i16((int16x8_t)__s0, __p1); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vget_lane_u8(__p0, __p1) __extension__ ({ \
  uint8x8_t __s0 = __p0; \
  uint8_t __ret; \
  __ret = (uint8_t) __builtin_neon_vget_lane_i8((int8x8_t)__s0, __p1); \
  __ret; \
})
#else
#define vget_lane_u8(__p0, __p1) __extension__ ({ \
  uint8x8_t __s0 = __p0; \
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint8_t __ret; \
  __ret = (uint8_t) __builtin_neon_vget_lane_i8((int8x8_t)__rev0, __p1); \
  __ret; \
})
#define __noswap_vget_lane_u8(__p0, __p1) __extension__ ({ \
  uint8x8_t __s0 = __p0; \
  uint8_t __ret; \
  __ret = (uint8_t) __builtin_neon_vget_lane_i8((int8x8_t)__s0, __p1); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vget_lane_u32(__p0, __p1) __extension__ ({ \
  uint32x2_t __s0 = __p0; \
  uint32_t __ret; \
  __ret = (uint32_t) __builtin_neon_vget_lane_i32((int32x2_t)__s0, __p1); \
  __ret; \
})
#else
#define vget_lane_u32(__p0, __p1) __extension__ ({ \
  uint32x2_t __s0 = __p0; \
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  uint32_t __ret; \
  __ret = (uint32_t) __builtin_neon_vget_lane_i32((int32x2_t)__rev0, __p1); \
  __ret; \
})
#define __noswap_vget_lane_u32(__p0, __p1) __extension__ ({ \
  uint32x2_t __s0 = __p0; \
  uint32_t __ret; \
  __ret = (uint32_t) __builtin_neon_vget_lane_i32((int32x2_t)__s0, __p1); \
  __ret; \
})
#endif

#define vget_lane_u64(__p0, __p1) __extension__ ({ \
  uint64x1_t __s0 = __p0; \
  uint64_t __ret; \
  __ret = (uint64_t) __builtin_neon_vget_lane_i64((int64x1_t)__s0, __p1); \
  __ret; \
})
#ifdef __LITTLE_ENDIAN__
#define vget_lane_u16(__p0, __p1) __extension__ ({ \
  uint16x4_t __s0 = __p0; \
  uint16_t __ret; \
  __ret = (uint16_t) __builtin_neon_vget_lane_i16((int16x4_t)__s0, __p1); \
  __ret; \
})
#else
#define vget_lane_u16(__p0, __p1) __extension__ ({ \
  uint16x4_t __s0 = __p0; \
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  uint16_t __ret; \
  __ret = (uint16_t) __builtin_neon_vget_lane_i16((int16x4_t)__rev0, __p1); \
  __ret; \
})
#define __noswap_vget_lane_u16(__p0, __p1) __extension__ ({ \
  uint16x4_t __s0 = __p0; \
  uint16_t __ret; \
  __ret = (uint16_t) __builtin_neon_vget_lane_i16((int16x4_t)__s0, __p1); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vget_lane_s8(__p0, __p1) __extension__ ({ \
  int8x8_t __s0 = __p0; \
  int8_t __ret; \
  __ret = (int8_t) __builtin_neon_vget_lane_i8((int8x8_t)__s0, __p1); \
  __ret; \
})
#else
#define vget_lane_s8(__p0, __p1) __extension__ ({ \
  int8x8_t __s0 = __p0; \
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  int8_t __ret; \
  __ret = (int8_t) __builtin_neon_vget_lane_i8((int8x8_t)__rev0, __p1); \
  __ret; \
})
#define __noswap_vget_lane_s8(__p0, __p1) __extension__ ({ \
  int8x8_t __s0 = __p0; \
  int8_t __ret; \
  __ret = (int8_t) __builtin_neon_vget_lane_i8((int8x8_t)__s0, __p1); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vget_lane_f32(__p0, __p1) __extension__ ({ \
  float32x2_t __s0 = __p0; \
  float32_t __ret; \
  __ret = (float32_t) __builtin_neon_vget_lane_f32((float32x2_t)__s0, __p1); \
  __ret; \
})
#else
#define vget_lane_f32(__p0, __p1) __extension__ ({ \
  float32x2_t __s0 = __p0; \
  float32x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  float32_t __ret; \
  __ret = (float32_t) __builtin_neon_vget_lane_f32((float32x2_t)__rev0, __p1); \
  __ret; \
})
#define __noswap_vget_lane_f32(__p0, __p1) __extension__ ({ \
  float32x2_t __s0 = __p0; \
  float32_t __ret; \
  __ret = (float32_t) __builtin_neon_vget_lane_f32((float32x2_t)__s0, __p1); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vget_lane_s32(__p0, __p1) __extension__ ({ \
  int32x2_t __s0 = __p0; \
  int32_t __ret; \
  __ret = (int32_t) __builtin_neon_vget_lane_i32((int32x2_t)__s0, __p1); \
  __ret; \
})
#else
#define vget_lane_s32(__p0, __p1) __extension__ ({ \
  int32x2_t __s0 = __p0; \
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  int32_t __ret; \
  __ret = (int32_t) __builtin_neon_vget_lane_i32((int32x2_t)__rev0, __p1); \
  __ret; \
})
#define __noswap_vget_lane_s32(__p0, __p1) __extension__ ({ \
  int32x2_t __s0 = __p0; \
  int32_t __ret; \
  __ret = (int32_t) __builtin_neon_vget_lane_i32((int32x2_t)__s0, __p1); \
  __ret; \
})
#endif

#define vget_lane_s64(__p0, __p1) __extension__ ({ \
  int64x1_t __s0 = __p0; \
  int64_t __ret; \
  __ret = (int64_t) __builtin_neon_vget_lane_i64((int64x1_t)__s0, __p1); \
  __ret; \
})
#ifdef __LITTLE_ENDIAN__
#define vget_lane_s16(__p0, __p1) __extension__ ({ \
  int16x4_t __s0 = __p0; \
  int16_t __ret; \
  __ret = (int16_t) __builtin_neon_vget_lane_i16((int16x4_t)__s0, __p1); \
  __ret; \
})
#else
#define vget_lane_s16(__p0, __p1) __extension__ ({ \
  int16x4_t __s0 = __p0; \
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  int16_t __ret; \
  __ret = (int16_t) __builtin_neon_vget_lane_i16((int16x4_t)__rev0, __p1); \
  __ret; \
})
#define __noswap_vget_lane_s16(__p0, __p1) __extension__ ({ \
  int16x4_t __s0 = __p0; \
  int16_t __ret; \
  __ret = (int16_t) __builtin_neon_vget_lane_i16((int16x4_t)__s0, __p1); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
__ai poly8x8_t vget_low_p8(poly8x16_t __p0) {
  poly8x8_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 0, 1, 2, 3, 4, 5, 6, 7);
  return __ret;
}
#else
__ai poly8x8_t vget_low_p8(poly8x16_t __p0) {
  poly8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  poly8x8_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 0, 1, 2, 3, 4, 5, 6, 7);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai poly16x4_t vget_low_p16(poly16x8_t __p0) {
  poly16x4_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 0, 1, 2, 3);
  return __ret;
}
#else
__ai poly16x4_t vget_low_p16(poly16x8_t __p0) {
  poly16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  poly16x4_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 0, 1, 2, 3);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x8_t vget_low_u8(uint8x16_t __p0) {
  uint8x8_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 0, 1, 2, 3, 4, 5, 6, 7);
  return __ret;
}
#else
__ai uint8x8_t vget_low_u8(uint8x16_t __p0) {
  uint8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 0, 1, 2, 3, 4, 5, 6, 7);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vget_low_u32(uint32x4_t __p0) {
  uint32x2_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 0, 1);
  return __ret;
}
#else
__ai uint32x2_t vget_low_u32(uint32x4_t __p0) {
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint32x2_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 0, 1);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint64x1_t vget_low_u64(uint64x2_t __p0) {
  uint64x1_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 0);
  return __ret;
}
#else
__ai uint64x1_t vget_low_u64(uint64x2_t __p0) {
  uint64x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint64x1_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x4_t vget_low_u16(uint16x8_t __p0) {
  uint16x4_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 0, 1, 2, 3);
  return __ret;
}
#else
__ai uint16x4_t vget_low_u16(uint16x8_t __p0) {
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x4_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 0, 1, 2, 3);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x8_t vget_low_s8(int8x16_t __p0) {
  int8x8_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 0, 1, 2, 3, 4, 5, 6, 7);
  return __ret;
}
#else
__ai int8x8_t vget_low_s8(int8x16_t __p0) {
  int8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 0, 1, 2, 3, 4, 5, 6, 7);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai float32x2_t vget_low_f32(float32x4_t __p0) {
  float32x2_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 0, 1);
  return __ret;
}
#else
__ai float32x2_t vget_low_f32(float32x4_t __p0) {
  float32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  float32x2_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 0, 1);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai float16x4_t vget_low_f16(float16x8_t __p0) {
  float16x4_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 0, 1, 2, 3);
  return __ret;
}
#else
__ai float16x4_t vget_low_f16(float16x8_t __p0) {
  float16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  float16x4_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 0, 1, 2, 3);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x2_t vget_low_s32(int32x4_t __p0) {
  int32x2_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 0, 1);
  return __ret;
}
#else
__ai int32x2_t vget_low_s32(int32x4_t __p0) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x2_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 0, 1);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int64x1_t vget_low_s64(int64x2_t __p0) {
  int64x1_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 0);
  return __ret;
}
#else
__ai int64x1_t vget_low_s64(int64x2_t __p0) {
  int64x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int64x1_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x4_t vget_low_s16(int16x8_t __p0) {
  int16x4_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 0, 1, 2, 3);
  return __ret;
}
#else
__ai int16x4_t vget_low_s16(int16x8_t __p0) {
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x4_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 0, 1, 2, 3);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x16_t vhaddq_u8(uint8x16_t __p0, uint8x16_t __p1) {
  uint8x16_t __ret;
  __ret = (uint8x16_t) __builtin_neon_vhaddq_v((int8x16_t)__p0, (int8x16_t)__p1, 48);
  return __ret;
}
#else
__ai uint8x16_t vhaddq_u8(uint8x16_t __p0, uint8x16_t __p1) {
  uint8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __ret;
  __ret = (uint8x16_t) __builtin_neon_vhaddq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 48);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vhaddq_u32(uint32x4_t __p0, uint32x4_t __p1) {
  uint32x4_t __ret;
  __ret = (uint32x4_t) __builtin_neon_vhaddq_v((int8x16_t)__p0, (int8x16_t)__p1, 50);
  return __ret;
}
#else
__ai uint32x4_t vhaddq_u32(uint32x4_t __p0, uint32x4_t __p1) {
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = (uint32x4_t) __builtin_neon_vhaddq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 50);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x8_t vhaddq_u16(uint16x8_t __p0, uint16x8_t __p1) {
  uint16x8_t __ret;
  __ret = (uint16x8_t) __builtin_neon_vhaddq_v((int8x16_t)__p0, (int8x16_t)__p1, 49);
  return __ret;
}
#else
__ai uint16x8_t vhaddq_u16(uint16x8_t __p0, uint16x8_t __p1) {
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __ret;
  __ret = (uint16x8_t) __builtin_neon_vhaddq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 49);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x16_t vhaddq_s8(int8x16_t __p0, int8x16_t __p1) {
  int8x16_t __ret;
  __ret = (int8x16_t) __builtin_neon_vhaddq_v((int8x16_t)__p0, (int8x16_t)__p1, 32);
  return __ret;
}
#else
__ai int8x16_t vhaddq_s8(int8x16_t __p0, int8x16_t __p1) {
  int8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __ret;
  __ret = (int8x16_t) __builtin_neon_vhaddq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 32);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x4_t vhaddq_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vhaddq_v((int8x16_t)__p0, (int8x16_t)__p1, 34);
  return __ret;
}
#else
__ai int32x4_t vhaddq_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vhaddq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 34);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x8_t vhaddq_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vhaddq_v((int8x16_t)__p0, (int8x16_t)__p1, 33);
  return __ret;
}
#else
__ai int16x8_t vhaddq_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vhaddq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 33);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x8_t vhadd_u8(uint8x8_t __p0, uint8x8_t __p1) {
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vhadd_v((int8x8_t)__p0, (int8x8_t)__p1, 16);
  return __ret;
}
#else
__ai uint8x8_t vhadd_u8(uint8x8_t __p0, uint8x8_t __p1) {
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vhadd_v((int8x8_t)__rev0, (int8x8_t)__rev1, 16);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vhadd_u32(uint32x2_t __p0, uint32x2_t __p1) {
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vhadd_v((int8x8_t)__p0, (int8x8_t)__p1, 18);
  return __ret;
}
#else
__ai uint32x2_t vhadd_u32(uint32x2_t __p0, uint32x2_t __p1) {
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vhadd_v((int8x8_t)__rev0, (int8x8_t)__rev1, 18);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x4_t vhadd_u16(uint16x4_t __p0, uint16x4_t __p1) {
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vhadd_v((int8x8_t)__p0, (int8x8_t)__p1, 17);
  return __ret;
}
#else
__ai uint16x4_t vhadd_u16(uint16x4_t __p0, uint16x4_t __p1) {
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vhadd_v((int8x8_t)__rev0, (int8x8_t)__rev1, 17);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x8_t vhadd_s8(int8x8_t __p0, int8x8_t __p1) {
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vhadd_v((int8x8_t)__p0, (int8x8_t)__p1, 0);
  return __ret;
}
#else
__ai int8x8_t vhadd_s8(int8x8_t __p0, int8x8_t __p1) {
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vhadd_v((int8x8_t)__rev0, (int8x8_t)__rev1, 0);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x2_t vhadd_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vhadd_v((int8x8_t)__p0, (int8x8_t)__p1, 2);
  return __ret;
}
#else
__ai int32x2_t vhadd_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vhadd_v((int8x8_t)__rev0, (int8x8_t)__rev1, 2);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x4_t vhadd_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vhadd_v((int8x8_t)__p0, (int8x8_t)__p1, 1);
  return __ret;
}
#else
__ai int16x4_t vhadd_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vhadd_v((int8x8_t)__rev0, (int8x8_t)__rev1, 1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x16_t vhsubq_u8(uint8x16_t __p0, uint8x16_t __p1) {
  uint8x16_t __ret;
  __ret = (uint8x16_t) __builtin_neon_vhsubq_v((int8x16_t)__p0, (int8x16_t)__p1, 48);
  return __ret;
}
#else
__ai uint8x16_t vhsubq_u8(uint8x16_t __p0, uint8x16_t __p1) {
  uint8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __ret;
  __ret = (uint8x16_t) __builtin_neon_vhsubq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 48);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vhsubq_u32(uint32x4_t __p0, uint32x4_t __p1) {
  uint32x4_t __ret;
  __ret = (uint32x4_t) __builtin_neon_vhsubq_v((int8x16_t)__p0, (int8x16_t)__p1, 50);
  return __ret;
}
#else
__ai uint32x4_t vhsubq_u32(uint32x4_t __p0, uint32x4_t __p1) {
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = (uint32x4_t) __builtin_neon_vhsubq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 50);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x8_t vhsubq_u16(uint16x8_t __p0, uint16x8_t __p1) {
  uint16x8_t __ret;
  __ret = (uint16x8_t) __builtin_neon_vhsubq_v((int8x16_t)__p0, (int8x16_t)__p1, 49);
  return __ret;
}
#else
__ai uint16x8_t vhsubq_u16(uint16x8_t __p0, uint16x8_t __p1) {
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __ret;
  __ret = (uint16x8_t) __builtin_neon_vhsubq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 49);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x16_t vhsubq_s8(int8x16_t __p0, int8x16_t __p1) {
  int8x16_t __ret;
  __ret = (int8x16_t) __builtin_neon_vhsubq_v((int8x16_t)__p0, (int8x16_t)__p1, 32);
  return __ret;
}
#else
__ai int8x16_t vhsubq_s8(int8x16_t __p0, int8x16_t __p1) {
  int8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __ret;
  __ret = (int8x16_t) __builtin_neon_vhsubq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 32);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x4_t vhsubq_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vhsubq_v((int8x16_t)__p0, (int8x16_t)__p1, 34);
  return __ret;
}
#else
__ai int32x4_t vhsubq_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vhsubq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 34);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x8_t vhsubq_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vhsubq_v((int8x16_t)__p0, (int8x16_t)__p1, 33);
  return __ret;
}
#else
__ai int16x8_t vhsubq_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vhsubq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 33);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x8_t vhsub_u8(uint8x8_t __p0, uint8x8_t __p1) {
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vhsub_v((int8x8_t)__p0, (int8x8_t)__p1, 16);
  return __ret;
}
#else
__ai uint8x8_t vhsub_u8(uint8x8_t __p0, uint8x8_t __p1) {
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vhsub_v((int8x8_t)__rev0, (int8x8_t)__rev1, 16);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vhsub_u32(uint32x2_t __p0, uint32x2_t __p1) {
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vhsub_v((int8x8_t)__p0, (int8x8_t)__p1, 18);
  return __ret;
}
#else
__ai uint32x2_t vhsub_u32(uint32x2_t __p0, uint32x2_t __p1) {
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vhsub_v((int8x8_t)__rev0, (int8x8_t)__rev1, 18);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x4_t vhsub_u16(uint16x4_t __p0, uint16x4_t __p1) {
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vhsub_v((int8x8_t)__p0, (int8x8_t)__p1, 17);
  return __ret;
}
#else
__ai uint16x4_t vhsub_u16(uint16x4_t __p0, uint16x4_t __p1) {
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vhsub_v((int8x8_t)__rev0, (int8x8_t)__rev1, 17);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x8_t vhsub_s8(int8x8_t __p0, int8x8_t __p1) {
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vhsub_v((int8x8_t)__p0, (int8x8_t)__p1, 0);
  return __ret;
}
#else
__ai int8x8_t vhsub_s8(int8x8_t __p0, int8x8_t __p1) {
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vhsub_v((int8x8_t)__rev0, (int8x8_t)__rev1, 0);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x2_t vhsub_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vhsub_v((int8x8_t)__p0, (int8x8_t)__p1, 2);
  return __ret;
}
#else
__ai int32x2_t vhsub_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vhsub_v((int8x8_t)__rev0, (int8x8_t)__rev1, 2);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x4_t vhsub_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vhsub_v((int8x8_t)__p0, (int8x8_t)__p1, 1);
  return __ret;
}
#else
__ai int16x4_t vhsub_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vhsub_v((int8x8_t)__rev0, (int8x8_t)__rev1, 1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1_p8(__p0) __extension__ ({ \
  poly8x8_t __ret; \
  __ret = (poly8x8_t) __builtin_neon_vld1_v(__p0, 4); \
  __ret; \
})
#else
#define vld1_p8(__p0) __extension__ ({ \
  poly8x8_t __ret; \
  __ret = (poly8x8_t) __builtin_neon_vld1_v(__p0, 4); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1_p16(__p0) __extension__ ({ \
  poly16x4_t __ret; \
  __ret = (poly16x4_t) __builtin_neon_vld1_v(__p0, 5); \
  __ret; \
})
#else
#define vld1_p16(__p0) __extension__ ({ \
  poly16x4_t __ret; \
  __ret = (poly16x4_t) __builtin_neon_vld1_v(__p0, 5); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_p8(__p0) __extension__ ({ \
  poly8x16_t __ret; \
  __ret = (poly8x16_t) __builtin_neon_vld1q_v(__p0, 36); \
  __ret; \
})
#else
#define vld1q_p8(__p0) __extension__ ({ \
  poly8x16_t __ret; \
  __ret = (poly8x16_t) __builtin_neon_vld1q_v(__p0, 36); \
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_p16(__p0) __extension__ ({ \
  poly16x8_t __ret; \
  __ret = (poly16x8_t) __builtin_neon_vld1q_v(__p0, 37); \
  __ret; \
})
#else
#define vld1q_p16(__p0) __extension__ ({ \
  poly16x8_t __ret; \
  __ret = (poly16x8_t) __builtin_neon_vld1q_v(__p0, 37); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_u8(__p0) __extension__ ({ \
  uint8x16_t __ret; \
  __ret = (uint8x16_t) __builtin_neon_vld1q_v(__p0, 48); \
  __ret; \
})
#else
#define vld1q_u8(__p0) __extension__ ({ \
  uint8x16_t __ret; \
  __ret = (uint8x16_t) __builtin_neon_vld1q_v(__p0, 48); \
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_u32(__p0) __extension__ ({ \
  uint32x4_t __ret; \
  __ret = (uint32x4_t) __builtin_neon_vld1q_v(__p0, 50); \
  __ret; \
})
#else
#define vld1q_u32(__p0) __extension__ ({ \
  uint32x4_t __ret; \
  __ret = (uint32x4_t) __builtin_neon_vld1q_v(__p0, 50); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_u64(__p0) __extension__ ({ \
  uint64x2_t __ret; \
  __ret = (uint64x2_t) __builtin_neon_vld1q_v(__p0, 51); \
  __ret; \
})
#else
#define vld1q_u64(__p0) __extension__ ({ \
  uint64x2_t __ret; \
  __ret = (uint64x2_t) __builtin_neon_vld1q_v(__p0, 51); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_u16(__p0) __extension__ ({ \
  uint16x8_t __ret; \
  __ret = (uint16x8_t) __builtin_neon_vld1q_v(__p0, 49); \
  __ret; \
})
#else
#define vld1q_u16(__p0) __extension__ ({ \
  uint16x8_t __ret; \
  __ret = (uint16x8_t) __builtin_neon_vld1q_v(__p0, 49); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_s8(__p0) __extension__ ({ \
  int8x16_t __ret; \
  __ret = (int8x16_t) __builtin_neon_vld1q_v(__p0, 32); \
  __ret; \
})
#else
#define vld1q_s8(__p0) __extension__ ({ \
  int8x16_t __ret; \
  __ret = (int8x16_t) __builtin_neon_vld1q_v(__p0, 32); \
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_f32(__p0) __extension__ ({ \
  float32x4_t __ret; \
  __ret = (float32x4_t) __builtin_neon_vld1q_v(__p0, 41); \
  __ret; \
})
#else
#define vld1q_f32(__p0) __extension__ ({ \
  float32x4_t __ret; \
  __ret = (float32x4_t) __builtin_neon_vld1q_v(__p0, 41); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_s32(__p0) __extension__ ({ \
  int32x4_t __ret; \
  __ret = (int32x4_t) __builtin_neon_vld1q_v(__p0, 34); \
  __ret; \
})
#else
#define vld1q_s32(__p0) __extension__ ({ \
  int32x4_t __ret; \
  __ret = (int32x4_t) __builtin_neon_vld1q_v(__p0, 34); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_s64(__p0) __extension__ ({ \
  int64x2_t __ret; \
  __ret = (int64x2_t) __builtin_neon_vld1q_v(__p0, 35); \
  __ret; \
})
#else
#define vld1q_s64(__p0) __extension__ ({ \
  int64x2_t __ret; \
  __ret = (int64x2_t) __builtin_neon_vld1q_v(__p0, 35); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_s16(__p0) __extension__ ({ \
  int16x8_t __ret; \
  __ret = (int16x8_t) __builtin_neon_vld1q_v(__p0, 33); \
  __ret; \
})
#else
#define vld1q_s16(__p0) __extension__ ({ \
  int16x8_t __ret; \
  __ret = (int16x8_t) __builtin_neon_vld1q_v(__p0, 33); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1_u8(__p0) __extension__ ({ \
  uint8x8_t __ret; \
  __ret = (uint8x8_t) __builtin_neon_vld1_v(__p0, 16); \
  __ret; \
})
#else
#define vld1_u8(__p0) __extension__ ({ \
  uint8x8_t __ret; \
  __ret = (uint8x8_t) __builtin_neon_vld1_v(__p0, 16); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1_u32(__p0) __extension__ ({ \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_vld1_v(__p0, 18); \
  __ret; \
})
#else
#define vld1_u32(__p0) __extension__ ({ \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_vld1_v(__p0, 18); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#define vld1_u64(__p0) __extension__ ({ \
  uint64x1_t __ret; \
  __ret = (uint64x1_t) __builtin_neon_vld1_v(__p0, 19); \
  __ret; \
})
#ifdef __LITTLE_ENDIAN__
#define vld1_u16(__p0) __extension__ ({ \
  uint16x4_t __ret; \
  __ret = (uint16x4_t) __builtin_neon_vld1_v(__p0, 17); \
  __ret; \
})
#else
#define vld1_u16(__p0) __extension__ ({ \
  uint16x4_t __ret; \
  __ret = (uint16x4_t) __builtin_neon_vld1_v(__p0, 17); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1_s8(__p0) __extension__ ({ \
  int8x8_t __ret; \
  __ret = (int8x8_t) __builtin_neon_vld1_v(__p0, 0); \
  __ret; \
})
#else
#define vld1_s8(__p0) __extension__ ({ \
  int8x8_t __ret; \
  __ret = (int8x8_t) __builtin_neon_vld1_v(__p0, 0); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1_f32(__p0) __extension__ ({ \
  float32x2_t __ret; \
  __ret = (float32x2_t) __builtin_neon_vld1_v(__p0, 9); \
  __ret; \
})
#else
#define vld1_f32(__p0) __extension__ ({ \
  float32x2_t __ret; \
  __ret = (float32x2_t) __builtin_neon_vld1_v(__p0, 9); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1_s32(__p0) __extension__ ({ \
  int32x2_t __ret; \
  __ret = (int32x2_t) __builtin_neon_vld1_v(__p0, 2); \
  __ret; \
})
#else
#define vld1_s32(__p0) __extension__ ({ \
  int32x2_t __ret; \
  __ret = (int32x2_t) __builtin_neon_vld1_v(__p0, 2); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#define vld1_s64(__p0) __extension__ ({ \
  int64x1_t __ret; \
  __ret = (int64x1_t) __builtin_neon_vld1_v(__p0, 3); \
  __ret; \
})
#ifdef __LITTLE_ENDIAN__
#define vld1_s16(__p0) __extension__ ({ \
  int16x4_t __ret; \
  __ret = (int16x4_t) __builtin_neon_vld1_v(__p0, 1); \
  __ret; \
})
#else
#define vld1_s16(__p0) __extension__ ({ \
  int16x4_t __ret; \
  __ret = (int16x4_t) __builtin_neon_vld1_v(__p0, 1); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1_dup_p8(__p0) __extension__ ({ \
  poly8x8_t __ret; \
  __ret = (poly8x8_t) __builtin_neon_vld1_dup_v(__p0, 4); \
  __ret; \
})
#else
#define vld1_dup_p8(__p0) __extension__ ({ \
  poly8x8_t __ret; \
  __ret = (poly8x8_t) __builtin_neon_vld1_dup_v(__p0, 4); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1_dup_p16(__p0) __extension__ ({ \
  poly16x4_t __ret; \
  __ret = (poly16x4_t) __builtin_neon_vld1_dup_v(__p0, 5); \
  __ret; \
})
#else
#define vld1_dup_p16(__p0) __extension__ ({ \
  poly16x4_t __ret; \
  __ret = (poly16x4_t) __builtin_neon_vld1_dup_v(__p0, 5); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_dup_p8(__p0) __extension__ ({ \
  poly8x16_t __ret; \
  __ret = (poly8x16_t) __builtin_neon_vld1q_dup_v(__p0, 36); \
  __ret; \
})
#else
#define vld1q_dup_p8(__p0) __extension__ ({ \
  poly8x16_t __ret; \
  __ret = (poly8x16_t) __builtin_neon_vld1q_dup_v(__p0, 36); \
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_dup_p16(__p0) __extension__ ({ \
  poly16x8_t __ret; \
  __ret = (poly16x8_t) __builtin_neon_vld1q_dup_v(__p0, 37); \
  __ret; \
})
#else
#define vld1q_dup_p16(__p0) __extension__ ({ \
  poly16x8_t __ret; \
  __ret = (poly16x8_t) __builtin_neon_vld1q_dup_v(__p0, 37); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_dup_u8(__p0) __extension__ ({ \
  uint8x16_t __ret; \
  __ret = (uint8x16_t) __builtin_neon_vld1q_dup_v(__p0, 48); \
  __ret; \
})
#else
#define vld1q_dup_u8(__p0) __extension__ ({ \
  uint8x16_t __ret; \
  __ret = (uint8x16_t) __builtin_neon_vld1q_dup_v(__p0, 48); \
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_dup_u32(__p0) __extension__ ({ \
  uint32x4_t __ret; \
  __ret = (uint32x4_t) __builtin_neon_vld1q_dup_v(__p0, 50); \
  __ret; \
})
#else
#define vld1q_dup_u32(__p0) __extension__ ({ \
  uint32x4_t __ret; \
  __ret = (uint32x4_t) __builtin_neon_vld1q_dup_v(__p0, 50); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_dup_u64(__p0) __extension__ ({ \
  uint64x2_t __ret; \
  __ret = (uint64x2_t) __builtin_neon_vld1q_dup_v(__p0, 51); \
  __ret; \
})
#else
#define vld1q_dup_u64(__p0) __extension__ ({ \
  uint64x2_t __ret; \
  __ret = (uint64x2_t) __builtin_neon_vld1q_dup_v(__p0, 51); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_dup_u16(__p0) __extension__ ({ \
  uint16x8_t __ret; \
  __ret = (uint16x8_t) __builtin_neon_vld1q_dup_v(__p0, 49); \
  __ret; \
})
#else
#define vld1q_dup_u16(__p0) __extension__ ({ \
  uint16x8_t __ret; \
  __ret = (uint16x8_t) __builtin_neon_vld1q_dup_v(__p0, 49); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_dup_s8(__p0) __extension__ ({ \
  int8x16_t __ret; \
  __ret = (int8x16_t) __builtin_neon_vld1q_dup_v(__p0, 32); \
  __ret; \
})
#else
#define vld1q_dup_s8(__p0) __extension__ ({ \
  int8x16_t __ret; \
  __ret = (int8x16_t) __builtin_neon_vld1q_dup_v(__p0, 32); \
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_dup_f32(__p0) __extension__ ({ \
  float32x4_t __ret; \
  __ret = (float32x4_t) __builtin_neon_vld1q_dup_v(__p0, 41); \
  __ret; \
})
#else
#define vld1q_dup_f32(__p0) __extension__ ({ \
  float32x4_t __ret; \
  __ret = (float32x4_t) __builtin_neon_vld1q_dup_v(__p0, 41); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_dup_s32(__p0) __extension__ ({ \
  int32x4_t __ret; \
  __ret = (int32x4_t) __builtin_neon_vld1q_dup_v(__p0, 34); \
  __ret; \
})
#else
#define vld1q_dup_s32(__p0) __extension__ ({ \
  int32x4_t __ret; \
  __ret = (int32x4_t) __builtin_neon_vld1q_dup_v(__p0, 34); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_dup_s64(__p0) __extension__ ({ \
  int64x2_t __ret; \
  __ret = (int64x2_t) __builtin_neon_vld1q_dup_v(__p0, 35); \
  __ret; \
})
#else
#define vld1q_dup_s64(__p0) __extension__ ({ \
  int64x2_t __ret; \
  __ret = (int64x2_t) __builtin_neon_vld1q_dup_v(__p0, 35); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_dup_s16(__p0) __extension__ ({ \
  int16x8_t __ret; \
  __ret = (int16x8_t) __builtin_neon_vld1q_dup_v(__p0, 33); \
  __ret; \
})
#else
#define vld1q_dup_s16(__p0) __extension__ ({ \
  int16x8_t __ret; \
  __ret = (int16x8_t) __builtin_neon_vld1q_dup_v(__p0, 33); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1_dup_u8(__p0) __extension__ ({ \
  uint8x8_t __ret; \
  __ret = (uint8x8_t) __builtin_neon_vld1_dup_v(__p0, 16); \
  __ret; \
})
#else
#define vld1_dup_u8(__p0) __extension__ ({ \
  uint8x8_t __ret; \
  __ret = (uint8x8_t) __builtin_neon_vld1_dup_v(__p0, 16); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1_dup_u32(__p0) __extension__ ({ \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_vld1_dup_v(__p0, 18); \
  __ret; \
})
#else
#define vld1_dup_u32(__p0) __extension__ ({ \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_vld1_dup_v(__p0, 18); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#define vld1_dup_u64(__p0) __extension__ ({ \
  uint64x1_t __ret; \
  __ret = (uint64x1_t) __builtin_neon_vld1_dup_v(__p0, 19); \
  __ret; \
})
#ifdef __LITTLE_ENDIAN__
#define vld1_dup_u16(__p0) __extension__ ({ \
  uint16x4_t __ret; \
  __ret = (uint16x4_t) __builtin_neon_vld1_dup_v(__p0, 17); \
  __ret; \
})
#else
#define vld1_dup_u16(__p0) __extension__ ({ \
  uint16x4_t __ret; \
  __ret = (uint16x4_t) __builtin_neon_vld1_dup_v(__p0, 17); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1_dup_s8(__p0) __extension__ ({ \
  int8x8_t __ret; \
  __ret = (int8x8_t) __builtin_neon_vld1_dup_v(__p0, 0); \
  __ret; \
})
#else
#define vld1_dup_s8(__p0) __extension__ ({ \
  int8x8_t __ret; \
  __ret = (int8x8_t) __builtin_neon_vld1_dup_v(__p0, 0); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1_dup_f32(__p0) __extension__ ({ \
  float32x2_t __ret; \
  __ret = (float32x2_t) __builtin_neon_vld1_dup_v(__p0, 9); \
  __ret; \
})
#else
#define vld1_dup_f32(__p0) __extension__ ({ \
  float32x2_t __ret; \
  __ret = (float32x2_t) __builtin_neon_vld1_dup_v(__p0, 9); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1_dup_s32(__p0) __extension__ ({ \
  int32x2_t __ret; \
  __ret = (int32x2_t) __builtin_neon_vld1_dup_v(__p0, 2); \
  __ret; \
})
#else
#define vld1_dup_s32(__p0) __extension__ ({ \
  int32x2_t __ret; \
  __ret = (int32x2_t) __builtin_neon_vld1_dup_v(__p0, 2); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#define vld1_dup_s64(__p0) __extension__ ({ \
  int64x1_t __ret; \
  __ret = (int64x1_t) __builtin_neon_vld1_dup_v(__p0, 3); \
  __ret; \
})
#ifdef __LITTLE_ENDIAN__
#define vld1_dup_s16(__p0) __extension__ ({ \
  int16x4_t __ret; \
  __ret = (int16x4_t) __builtin_neon_vld1_dup_v(__p0, 1); \
  __ret; \
})
#else
#define vld1_dup_s16(__p0) __extension__ ({ \
  int16x4_t __ret; \
  __ret = (int16x4_t) __builtin_neon_vld1_dup_v(__p0, 1); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1_lane_p8(__p0, __p1, __p2) __extension__ ({ \
  poly8x8_t __s1 = __p1; \
  poly8x8_t __ret; \
  __ret = (poly8x8_t) __builtin_neon_vld1_lane_v(__p0, (int8x8_t)__s1, __p2, 4); \
  __ret; \
})
#else
#define vld1_lane_p8(__p0, __p1, __p2) __extension__ ({ \
  poly8x8_t __s1 = __p1; \
  poly8x8_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 7, 6, 5, 4, 3, 2, 1, 0); \
  poly8x8_t __ret; \
  __ret = (poly8x8_t) __builtin_neon_vld1_lane_v(__p0, (int8x8_t)__rev1, __p2, 4); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1_lane_p16(__p0, __p1, __p2) __extension__ ({ \
  poly16x4_t __s1 = __p1; \
  poly16x4_t __ret; \
  __ret = (poly16x4_t) __builtin_neon_vld1_lane_v(__p0, (int8x8_t)__s1, __p2, 5); \
  __ret; \
})
#else
#define vld1_lane_p16(__p0, __p1, __p2) __extension__ ({ \
  poly16x4_t __s1 = __p1; \
  poly16x4_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 3, 2, 1, 0); \
  poly16x4_t __ret; \
  __ret = (poly16x4_t) __builtin_neon_vld1_lane_v(__p0, (int8x8_t)__rev1, __p2, 5); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_lane_p8(__p0, __p1, __p2) __extension__ ({ \
  poly8x16_t __s1 = __p1; \
  poly8x16_t __ret; \
  __ret = (poly8x16_t) __builtin_neon_vld1q_lane_v(__p0, (int8x16_t)__s1, __p2, 36); \
  __ret; \
})
#else
#define vld1q_lane_p8(__p0, __p1, __p2) __extension__ ({ \
  poly8x16_t __s1 = __p1; \
  poly8x16_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  poly8x16_t __ret; \
  __ret = (poly8x16_t) __builtin_neon_vld1q_lane_v(__p0, (int8x16_t)__rev1, __p2, 36); \
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_lane_p16(__p0, __p1, __p2) __extension__ ({ \
  poly16x8_t __s1 = __p1; \
  poly16x8_t __ret; \
  __ret = (poly16x8_t) __builtin_neon_vld1q_lane_v(__p0, (int8x16_t)__s1, __p2, 37); \
  __ret; \
})
#else
#define vld1q_lane_p16(__p0, __p1, __p2) __extension__ ({ \
  poly16x8_t __s1 = __p1; \
  poly16x8_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 7, 6, 5, 4, 3, 2, 1, 0); \
  poly16x8_t __ret; \
  __ret = (poly16x8_t) __builtin_neon_vld1q_lane_v(__p0, (int8x16_t)__rev1, __p2, 37); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_lane_u8(__p0, __p1, __p2) __extension__ ({ \
  uint8x16_t __s1 = __p1; \
  uint8x16_t __ret; \
  __ret = (uint8x16_t) __builtin_neon_vld1q_lane_v(__p0, (int8x16_t)__s1, __p2, 48); \
  __ret; \
})
#else
#define vld1q_lane_u8(__p0, __p1, __p2) __extension__ ({ \
  uint8x16_t __s1 = __p1; \
  uint8x16_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint8x16_t __ret; \
  __ret = (uint8x16_t) __builtin_neon_vld1q_lane_v(__p0, (int8x16_t)__rev1, __p2, 48); \
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_lane_u32(__p0, __p1, __p2) __extension__ ({ \
  uint32x4_t __s1 = __p1; \
  uint32x4_t __ret; \
  __ret = (uint32x4_t) __builtin_neon_vld1q_lane_v(__p0, (int8x16_t)__s1, __p2, 50); \
  __ret; \
})
#else
#define vld1q_lane_u32(__p0, __p1, __p2) __extension__ ({ \
  uint32x4_t __s1 = __p1; \
  uint32x4_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 3, 2, 1, 0); \
  uint32x4_t __ret; \
  __ret = (uint32x4_t) __builtin_neon_vld1q_lane_v(__p0, (int8x16_t)__rev1, __p2, 50); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_lane_u64(__p0, __p1, __p2) __extension__ ({ \
  uint64x2_t __s1 = __p1; \
  uint64x2_t __ret; \
  __ret = (uint64x2_t) __builtin_neon_vld1q_lane_v(__p0, (int8x16_t)__s1, __p2, 51); \
  __ret; \
})
#else
#define vld1q_lane_u64(__p0, __p1, __p2) __extension__ ({ \
  uint64x2_t __s1 = __p1; \
  uint64x2_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 1, 0); \
  uint64x2_t __ret; \
  __ret = (uint64x2_t) __builtin_neon_vld1q_lane_v(__p0, (int8x16_t)__rev1, __p2, 51); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_lane_u16(__p0, __p1, __p2) __extension__ ({ \
  uint16x8_t __s1 = __p1; \
  uint16x8_t __ret; \
  __ret = (uint16x8_t) __builtin_neon_vld1q_lane_v(__p0, (int8x16_t)__s1, __p2, 49); \
  __ret; \
})
#else
#define vld1q_lane_u16(__p0, __p1, __p2) __extension__ ({ \
  uint16x8_t __s1 = __p1; \
  uint16x8_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint16x8_t __ret; \
  __ret = (uint16x8_t) __builtin_neon_vld1q_lane_v(__p0, (int8x16_t)__rev1, __p2, 49); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_lane_s8(__p0, __p1, __p2) __extension__ ({ \
  int8x16_t __s1 = __p1; \
  int8x16_t __ret; \
  __ret = (int8x16_t) __builtin_neon_vld1q_lane_v(__p0, (int8x16_t)__s1, __p2, 32); \
  __ret; \
})
#else
#define vld1q_lane_s8(__p0, __p1, __p2) __extension__ ({ \
  int8x16_t __s1 = __p1; \
  int8x16_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  int8x16_t __ret; \
  __ret = (int8x16_t) __builtin_neon_vld1q_lane_v(__p0, (int8x16_t)__rev1, __p2, 32); \
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_lane_f32(__p0, __p1, __p2) __extension__ ({ \
  float32x4_t __s1 = __p1; \
  float32x4_t __ret; \
  __ret = (float32x4_t) __builtin_neon_vld1q_lane_v(__p0, (int8x16_t)__s1, __p2, 41); \
  __ret; \
})
#else
#define vld1q_lane_f32(__p0, __p1, __p2) __extension__ ({ \
  float32x4_t __s1 = __p1; \
  float32x4_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 3, 2, 1, 0); \
  float32x4_t __ret; \
  __ret = (float32x4_t) __builtin_neon_vld1q_lane_v(__p0, (int8x16_t)__rev1, __p2, 41); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_lane_s32(__p0, __p1, __p2) __extension__ ({ \
  int32x4_t __s1 = __p1; \
  int32x4_t __ret; \
  __ret = (int32x4_t) __builtin_neon_vld1q_lane_v(__p0, (int8x16_t)__s1, __p2, 34); \
  __ret; \
})
#else
#define vld1q_lane_s32(__p0, __p1, __p2) __extension__ ({ \
  int32x4_t __s1 = __p1; \
  int32x4_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 3, 2, 1, 0); \
  int32x4_t __ret; \
  __ret = (int32x4_t) __builtin_neon_vld1q_lane_v(__p0, (int8x16_t)__rev1, __p2, 34); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_lane_s64(__p0, __p1, __p2) __extension__ ({ \
  int64x2_t __s1 = __p1; \
  int64x2_t __ret; \
  __ret = (int64x2_t) __builtin_neon_vld1q_lane_v(__p0, (int8x16_t)__s1, __p2, 35); \
  __ret; \
})
#else
#define vld1q_lane_s64(__p0, __p1, __p2) __extension__ ({ \
  int64x2_t __s1 = __p1; \
  int64x2_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 1, 0); \
  int64x2_t __ret; \
  __ret = (int64x2_t) __builtin_neon_vld1q_lane_v(__p0, (int8x16_t)__rev1, __p2, 35); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_lane_s16(__p0, __p1, __p2) __extension__ ({ \
  int16x8_t __s1 = __p1; \
  int16x8_t __ret; \
  __ret = (int16x8_t) __builtin_neon_vld1q_lane_v(__p0, (int8x16_t)__s1, __p2, 33); \
  __ret; \
})
#else
#define vld1q_lane_s16(__p0, __p1, __p2) __extension__ ({ \
  int16x8_t __s1 = __p1; \
  int16x8_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 7, 6, 5, 4, 3, 2, 1, 0); \
  int16x8_t __ret; \
  __ret = (int16x8_t) __builtin_neon_vld1q_lane_v(__p0, (int8x16_t)__rev1, __p2, 33); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1_lane_u8(__p0, __p1, __p2) __extension__ ({ \
  uint8x8_t __s1 = __p1; \
  uint8x8_t __ret; \
  __ret = (uint8x8_t) __builtin_neon_vld1_lane_v(__p0, (int8x8_t)__s1, __p2, 16); \
  __ret; \
})
#else
#define vld1_lane_u8(__p0, __p1, __p2) __extension__ ({ \
  uint8x8_t __s1 = __p1; \
  uint8x8_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint8x8_t __ret; \
  __ret = (uint8x8_t) __builtin_neon_vld1_lane_v(__p0, (int8x8_t)__rev1, __p2, 16); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1_lane_u32(__p0, __p1, __p2) __extension__ ({ \
  uint32x2_t __s1 = __p1; \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_vld1_lane_v(__p0, (int8x8_t)__s1, __p2, 18); \
  __ret; \
})
#else
#define vld1_lane_u32(__p0, __p1, __p2) __extension__ ({ \
  uint32x2_t __s1 = __p1; \
  uint32x2_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 1, 0); \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_vld1_lane_v(__p0, (int8x8_t)__rev1, __p2, 18); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#define vld1_lane_u64(__p0, __p1, __p2) __extension__ ({ \
  uint64x1_t __s1 = __p1; \
  uint64x1_t __ret; \
  __ret = (uint64x1_t) __builtin_neon_vld1_lane_v(__p0, (int8x8_t)__s1, __p2, 19); \
  __ret; \
})
#ifdef __LITTLE_ENDIAN__
#define vld1_lane_u16(__p0, __p1, __p2) __extension__ ({ \
  uint16x4_t __s1 = __p1; \
  uint16x4_t __ret; \
  __ret = (uint16x4_t) __builtin_neon_vld1_lane_v(__p0, (int8x8_t)__s1, __p2, 17); \
  __ret; \
})
#else
#define vld1_lane_u16(__p0, __p1, __p2) __extension__ ({ \
  uint16x4_t __s1 = __p1; \
  uint16x4_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 3, 2, 1, 0); \
  uint16x4_t __ret; \
  __ret = (uint16x4_t) __builtin_neon_vld1_lane_v(__p0, (int8x8_t)__rev1, __p2, 17); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1_lane_s8(__p0, __p1, __p2) __extension__ ({ \
  int8x8_t __s1 = __p1; \
  int8x8_t __ret; \
  __ret = (int8x8_t) __builtin_neon_vld1_lane_v(__p0, (int8x8_t)__s1, __p2, 0); \
  __ret; \
})
#else
#define vld1_lane_s8(__p0, __p1, __p2) __extension__ ({ \
  int8x8_t __s1 = __p1; \
  int8x8_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 7, 6, 5, 4, 3, 2, 1, 0); \
  int8x8_t __ret; \
  __ret = (int8x8_t) __builtin_neon_vld1_lane_v(__p0, (int8x8_t)__rev1, __p2, 0); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1_lane_f32(__p0, __p1, __p2) __extension__ ({ \
  float32x2_t __s1 = __p1; \
  float32x2_t __ret; \
  __ret = (float32x2_t) __builtin_neon_vld1_lane_v(__p0, (int8x8_t)__s1, __p2, 9); \
  __ret; \
})
#else
#define vld1_lane_f32(__p0, __p1, __p2) __extension__ ({ \
  float32x2_t __s1 = __p1; \
  float32x2_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 1, 0); \
  float32x2_t __ret; \
  __ret = (float32x2_t) __builtin_neon_vld1_lane_v(__p0, (int8x8_t)__rev1, __p2, 9); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1_lane_s32(__p0, __p1, __p2) __extension__ ({ \
  int32x2_t __s1 = __p1; \
  int32x2_t __ret; \
  __ret = (int32x2_t) __builtin_neon_vld1_lane_v(__p0, (int8x8_t)__s1, __p2, 2); \
  __ret; \
})
#else
#define vld1_lane_s32(__p0, __p1, __p2) __extension__ ({ \
  int32x2_t __s1 = __p1; \
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 1, 0); \
  int32x2_t __ret; \
  __ret = (int32x2_t) __builtin_neon_vld1_lane_v(__p0, (int8x8_t)__rev1, __p2, 2); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#define vld1_lane_s64(__p0, __p1, __p2) __extension__ ({ \
  int64x1_t __s1 = __p1; \
  int64x1_t __ret; \
  __ret = (int64x1_t) __builtin_neon_vld1_lane_v(__p0, (int8x8_t)__s1, __p2, 3); \
  __ret; \
})
#ifdef __LITTLE_ENDIAN__
#define vld1_lane_s16(__p0, __p1, __p2) __extension__ ({ \
  int16x4_t __s1 = __p1; \
  int16x4_t __ret; \
  __ret = (int16x4_t) __builtin_neon_vld1_lane_v(__p0, (int8x8_t)__s1, __p2, 1); \
  __ret; \
})
#else
#define vld1_lane_s16(__p0, __p1, __p2) __extension__ ({ \
  int16x4_t __s1 = __p1; \
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 3, 2, 1, 0); \
  int16x4_t __ret; \
  __ret = (int16x4_t) __builtin_neon_vld1_lane_v(__p0, (int8x8_t)__rev1, __p2, 1); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1_p8_x2(__p0) __extension__ ({ \
  poly8x8x2_t __ret; \
  __builtin_neon_vld1_x2_v(&__ret, __p0, 4); \
  __ret; \
})
#else
#define vld1_p8_x2(__p0) __extension__ ({ \
  poly8x8x2_t __ret; \
  __builtin_neon_vld1_x2_v(&__ret, __p0, 4); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1_p16_x2(__p0) __extension__ ({ \
  poly16x4x2_t __ret; \
  __builtin_neon_vld1_x2_v(&__ret, __p0, 5); \
  __ret; \
})
#else
#define vld1_p16_x2(__p0) __extension__ ({ \
  poly16x4x2_t __ret; \
  __builtin_neon_vld1_x2_v(&__ret, __p0, 5); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_p8_x2(__p0) __extension__ ({ \
  poly8x16x2_t __ret; \
  __builtin_neon_vld1q_x2_v(&__ret, __p0, 36); \
  __ret; \
})
#else
#define vld1q_p8_x2(__p0) __extension__ ({ \
  poly8x16x2_t __ret; \
  __builtin_neon_vld1q_x2_v(&__ret, __p0, 36); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_p16_x2(__p0) __extension__ ({ \
  poly16x8x2_t __ret; \
  __builtin_neon_vld1q_x2_v(&__ret, __p0, 37); \
  __ret; \
})
#else
#define vld1q_p16_x2(__p0) __extension__ ({ \
  poly16x8x2_t __ret; \
  __builtin_neon_vld1q_x2_v(&__ret, __p0, 37); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_u8_x2(__p0) __extension__ ({ \
  uint8x16x2_t __ret; \
  __builtin_neon_vld1q_x2_v(&__ret, __p0, 48); \
  __ret; \
})
#else
#define vld1q_u8_x2(__p0) __extension__ ({ \
  uint8x16x2_t __ret; \
  __builtin_neon_vld1q_x2_v(&__ret, __p0, 48); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_u32_x2(__p0) __extension__ ({ \
  uint32x4x2_t __ret; \
  __builtin_neon_vld1q_x2_v(&__ret, __p0, 50); \
  __ret; \
})
#else
#define vld1q_u32_x2(__p0) __extension__ ({ \
  uint32x4x2_t __ret; \
  __builtin_neon_vld1q_x2_v(&__ret, __p0, 50); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_u64_x2(__p0) __extension__ ({ \
  uint64x2x2_t __ret; \
  __builtin_neon_vld1q_x2_v(&__ret, __p0, 51); \
  __ret; \
})
#else
#define vld1q_u64_x2(__p0) __extension__ ({ \
  uint64x2x2_t __ret; \
  __builtin_neon_vld1q_x2_v(&__ret, __p0, 51); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_u16_x2(__p0) __extension__ ({ \
  uint16x8x2_t __ret; \
  __builtin_neon_vld1q_x2_v(&__ret, __p0, 49); \
  __ret; \
})
#else
#define vld1q_u16_x2(__p0) __extension__ ({ \
  uint16x8x2_t __ret; \
  __builtin_neon_vld1q_x2_v(&__ret, __p0, 49); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_s8_x2(__p0) __extension__ ({ \
  int8x16x2_t __ret; \
  __builtin_neon_vld1q_x2_v(&__ret, __p0, 32); \
  __ret; \
})
#else
#define vld1q_s8_x2(__p0) __extension__ ({ \
  int8x16x2_t __ret; \
  __builtin_neon_vld1q_x2_v(&__ret, __p0, 32); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_f32_x2(__p0) __extension__ ({ \
  float32x4x2_t __ret; \
  __builtin_neon_vld1q_x2_v(&__ret, __p0, 41); \
  __ret; \
})
#else
#define vld1q_f32_x2(__p0) __extension__ ({ \
  float32x4x2_t __ret; \
  __builtin_neon_vld1q_x2_v(&__ret, __p0, 41); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_s32_x2(__p0) __extension__ ({ \
  int32x4x2_t __ret; \
  __builtin_neon_vld1q_x2_v(&__ret, __p0, 34); \
  __ret; \
})
#else
#define vld1q_s32_x2(__p0) __extension__ ({ \
  int32x4x2_t __ret; \
  __builtin_neon_vld1q_x2_v(&__ret, __p0, 34); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_s64_x2(__p0) __extension__ ({ \
  int64x2x2_t __ret; \
  __builtin_neon_vld1q_x2_v(&__ret, __p0, 35); \
  __ret; \
})
#else
#define vld1q_s64_x2(__p0) __extension__ ({ \
  int64x2x2_t __ret; \
  __builtin_neon_vld1q_x2_v(&__ret, __p0, 35); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_s16_x2(__p0) __extension__ ({ \
  int16x8x2_t __ret; \
  __builtin_neon_vld1q_x2_v(&__ret, __p0, 33); \
  __ret; \
})
#else
#define vld1q_s16_x2(__p0) __extension__ ({ \
  int16x8x2_t __ret; \
  __builtin_neon_vld1q_x2_v(&__ret, __p0, 33); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1_u8_x2(__p0) __extension__ ({ \
  uint8x8x2_t __ret; \
  __builtin_neon_vld1_x2_v(&__ret, __p0, 16); \
  __ret; \
})
#else
#define vld1_u8_x2(__p0) __extension__ ({ \
  uint8x8x2_t __ret; \
  __builtin_neon_vld1_x2_v(&__ret, __p0, 16); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1_u32_x2(__p0) __extension__ ({ \
  uint32x2x2_t __ret; \
  __builtin_neon_vld1_x2_v(&__ret, __p0, 18); \
  __ret; \
})
#else
#define vld1_u32_x2(__p0) __extension__ ({ \
  uint32x2x2_t __ret; \
  __builtin_neon_vld1_x2_v(&__ret, __p0, 18); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 1, 0); \
  __ret; \
})
#endif

#define vld1_u64_x2(__p0) __extension__ ({ \
  uint64x1x2_t __ret; \
  __builtin_neon_vld1_x2_v(&__ret, __p0, 19); \
  __ret; \
})
#ifdef __LITTLE_ENDIAN__
#define vld1_u16_x2(__p0) __extension__ ({ \
  uint16x4x2_t __ret; \
  __builtin_neon_vld1_x2_v(&__ret, __p0, 17); \
  __ret; \
})
#else
#define vld1_u16_x2(__p0) __extension__ ({ \
  uint16x4x2_t __ret; \
  __builtin_neon_vld1_x2_v(&__ret, __p0, 17); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1_s8_x2(__p0) __extension__ ({ \
  int8x8x2_t __ret; \
  __builtin_neon_vld1_x2_v(&__ret, __p0, 0); \
  __ret; \
})
#else
#define vld1_s8_x2(__p0) __extension__ ({ \
  int8x8x2_t __ret; \
  __builtin_neon_vld1_x2_v(&__ret, __p0, 0); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1_f32_x2(__p0) __extension__ ({ \
  float32x2x2_t __ret; \
  __builtin_neon_vld1_x2_v(&__ret, __p0, 9); \
  __ret; \
})
#else
#define vld1_f32_x2(__p0) __extension__ ({ \
  float32x2x2_t __ret; \
  __builtin_neon_vld1_x2_v(&__ret, __p0, 9); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1_s32_x2(__p0) __extension__ ({ \
  int32x2x2_t __ret; \
  __builtin_neon_vld1_x2_v(&__ret, __p0, 2); \
  __ret; \
})
#else
#define vld1_s32_x2(__p0) __extension__ ({ \
  int32x2x2_t __ret; \
  __builtin_neon_vld1_x2_v(&__ret, __p0, 2); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 1, 0); \
  __ret; \
})
#endif

#define vld1_s64_x2(__p0) __extension__ ({ \
  int64x1x2_t __ret; \
  __builtin_neon_vld1_x2_v(&__ret, __p0, 3); \
  __ret; \
})
#ifdef __LITTLE_ENDIAN__
#define vld1_s16_x2(__p0) __extension__ ({ \
  int16x4x2_t __ret; \
  __builtin_neon_vld1_x2_v(&__ret, __p0, 1); \
  __ret; \
})
#else
#define vld1_s16_x2(__p0) __extension__ ({ \
  int16x4x2_t __ret; \
  __builtin_neon_vld1_x2_v(&__ret, __p0, 1); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1_p8_x3(__p0) __extension__ ({ \
  poly8x8x3_t __ret; \
  __builtin_neon_vld1_x3_v(&__ret, __p0, 4); \
  __ret; \
})
#else
#define vld1_p8_x3(__p0) __extension__ ({ \
  poly8x8x3_t __ret; \
  __builtin_neon_vld1_x3_v(&__ret, __p0, 4); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1_p16_x3(__p0) __extension__ ({ \
  poly16x4x3_t __ret; \
  __builtin_neon_vld1_x3_v(&__ret, __p0, 5); \
  __ret; \
})
#else
#define vld1_p16_x3(__p0) __extension__ ({ \
  poly16x4x3_t __ret; \
  __builtin_neon_vld1_x3_v(&__ret, __p0, 5); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_p8_x3(__p0) __extension__ ({ \
  poly8x16x3_t __ret; \
  __builtin_neon_vld1q_x3_v(&__ret, __p0, 36); \
  __ret; \
})
#else
#define vld1q_p8_x3(__p0) __extension__ ({ \
  poly8x16x3_t __ret; \
  __builtin_neon_vld1q_x3_v(&__ret, __p0, 36); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_p16_x3(__p0) __extension__ ({ \
  poly16x8x3_t __ret; \
  __builtin_neon_vld1q_x3_v(&__ret, __p0, 37); \
  __ret; \
})
#else
#define vld1q_p16_x3(__p0) __extension__ ({ \
  poly16x8x3_t __ret; \
  __builtin_neon_vld1q_x3_v(&__ret, __p0, 37); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_u8_x3(__p0) __extension__ ({ \
  uint8x16x3_t __ret; \
  __builtin_neon_vld1q_x3_v(&__ret, __p0, 48); \
  __ret; \
})
#else
#define vld1q_u8_x3(__p0) __extension__ ({ \
  uint8x16x3_t __ret; \
  __builtin_neon_vld1q_x3_v(&__ret, __p0, 48); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_u32_x3(__p0) __extension__ ({ \
  uint32x4x3_t __ret; \
  __builtin_neon_vld1q_x3_v(&__ret, __p0, 50); \
  __ret; \
})
#else
#define vld1q_u32_x3(__p0) __extension__ ({ \
  uint32x4x3_t __ret; \
  __builtin_neon_vld1q_x3_v(&__ret, __p0, 50); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_u64_x3(__p0) __extension__ ({ \
  uint64x2x3_t __ret; \
  __builtin_neon_vld1q_x3_v(&__ret, __p0, 51); \
  __ret; \
})
#else
#define vld1q_u64_x3(__p0) __extension__ ({ \
  uint64x2x3_t __ret; \
  __builtin_neon_vld1q_x3_v(&__ret, __p0, 51); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_u16_x3(__p0) __extension__ ({ \
  uint16x8x3_t __ret; \
  __builtin_neon_vld1q_x3_v(&__ret, __p0, 49); \
  __ret; \
})
#else
#define vld1q_u16_x3(__p0) __extension__ ({ \
  uint16x8x3_t __ret; \
  __builtin_neon_vld1q_x3_v(&__ret, __p0, 49); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_s8_x3(__p0) __extension__ ({ \
  int8x16x3_t __ret; \
  __builtin_neon_vld1q_x3_v(&__ret, __p0, 32); \
  __ret; \
})
#else
#define vld1q_s8_x3(__p0) __extension__ ({ \
  int8x16x3_t __ret; \
  __builtin_neon_vld1q_x3_v(&__ret, __p0, 32); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_f32_x3(__p0) __extension__ ({ \
  float32x4x3_t __ret; \
  __builtin_neon_vld1q_x3_v(&__ret, __p0, 41); \
  __ret; \
})
#else
#define vld1q_f32_x3(__p0) __extension__ ({ \
  float32x4x3_t __ret; \
  __builtin_neon_vld1q_x3_v(&__ret, __p0, 41); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_s32_x3(__p0) __extension__ ({ \
  int32x4x3_t __ret; \
  __builtin_neon_vld1q_x3_v(&__ret, __p0, 34); \
  __ret; \
})
#else
#define vld1q_s32_x3(__p0) __extension__ ({ \
  int32x4x3_t __ret; \
  __builtin_neon_vld1q_x3_v(&__ret, __p0, 34); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_s64_x3(__p0) __extension__ ({ \
  int64x2x3_t __ret; \
  __builtin_neon_vld1q_x3_v(&__ret, __p0, 35); \
  __ret; \
})
#else
#define vld1q_s64_x3(__p0) __extension__ ({ \
  int64x2x3_t __ret; \
  __builtin_neon_vld1q_x3_v(&__ret, __p0, 35); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_s16_x3(__p0) __extension__ ({ \
  int16x8x3_t __ret; \
  __builtin_neon_vld1q_x3_v(&__ret, __p0, 33); \
  __ret; \
})
#else
#define vld1q_s16_x3(__p0) __extension__ ({ \
  int16x8x3_t __ret; \
  __builtin_neon_vld1q_x3_v(&__ret, __p0, 33); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1_u8_x3(__p0) __extension__ ({ \
  uint8x8x3_t __ret; \
  __builtin_neon_vld1_x3_v(&__ret, __p0, 16); \
  __ret; \
})
#else
#define vld1_u8_x3(__p0) __extension__ ({ \
  uint8x8x3_t __ret; \
  __builtin_neon_vld1_x3_v(&__ret, __p0, 16); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1_u32_x3(__p0) __extension__ ({ \
  uint32x2x3_t __ret; \
  __builtin_neon_vld1_x3_v(&__ret, __p0, 18); \
  __ret; \
})
#else
#define vld1_u32_x3(__p0) __extension__ ({ \
  uint32x2x3_t __ret; \
  __builtin_neon_vld1_x3_v(&__ret, __p0, 18); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 1, 0); \
  __ret; \
})
#endif

#define vld1_u64_x3(__p0) __extension__ ({ \
  uint64x1x3_t __ret; \
  __builtin_neon_vld1_x3_v(&__ret, __p0, 19); \
  __ret; \
})
#ifdef __LITTLE_ENDIAN__
#define vld1_u16_x3(__p0) __extension__ ({ \
  uint16x4x3_t __ret; \
  __builtin_neon_vld1_x3_v(&__ret, __p0, 17); \
  __ret; \
})
#else
#define vld1_u16_x3(__p0) __extension__ ({ \
  uint16x4x3_t __ret; \
  __builtin_neon_vld1_x3_v(&__ret, __p0, 17); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1_s8_x3(__p0) __extension__ ({ \
  int8x8x3_t __ret; \
  __builtin_neon_vld1_x3_v(&__ret, __p0, 0); \
  __ret; \
})
#else
#define vld1_s8_x3(__p0) __extension__ ({ \
  int8x8x3_t __ret; \
  __builtin_neon_vld1_x3_v(&__ret, __p0, 0); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1_f32_x3(__p0) __extension__ ({ \
  float32x2x3_t __ret; \
  __builtin_neon_vld1_x3_v(&__ret, __p0, 9); \
  __ret; \
})
#else
#define vld1_f32_x3(__p0) __extension__ ({ \
  float32x2x3_t __ret; \
  __builtin_neon_vld1_x3_v(&__ret, __p0, 9); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1_s32_x3(__p0) __extension__ ({ \
  int32x2x3_t __ret; \
  __builtin_neon_vld1_x3_v(&__ret, __p0, 2); \
  __ret; \
})
#else
#define vld1_s32_x3(__p0) __extension__ ({ \
  int32x2x3_t __ret; \
  __builtin_neon_vld1_x3_v(&__ret, __p0, 2); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 1, 0); \
  __ret; \
})
#endif

#define vld1_s64_x3(__p0) __extension__ ({ \
  int64x1x3_t __ret; \
  __builtin_neon_vld1_x3_v(&__ret, __p0, 3); \
  __ret; \
})
#ifdef __LITTLE_ENDIAN__
#define vld1_s16_x3(__p0) __extension__ ({ \
  int16x4x3_t __ret; \
  __builtin_neon_vld1_x3_v(&__ret, __p0, 1); \
  __ret; \
})
#else
#define vld1_s16_x3(__p0) __extension__ ({ \
  int16x4x3_t __ret; \
  __builtin_neon_vld1_x3_v(&__ret, __p0, 1); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1_p8_x4(__p0) __extension__ ({ \
  poly8x8x4_t __ret; \
  __builtin_neon_vld1_x4_v(&__ret, __p0, 4); \
  __ret; \
})
#else
#define vld1_p8_x4(__p0) __extension__ ({ \
  poly8x8x4_t __ret; \
  __builtin_neon_vld1_x4_v(&__ret, __p0, 4); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1_p16_x4(__p0) __extension__ ({ \
  poly16x4x4_t __ret; \
  __builtin_neon_vld1_x4_v(&__ret, __p0, 5); \
  __ret; \
})
#else
#define vld1_p16_x4(__p0) __extension__ ({ \
  poly16x4x4_t __ret; \
  __builtin_neon_vld1_x4_v(&__ret, __p0, 5); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_p8_x4(__p0) __extension__ ({ \
  poly8x16x4_t __ret; \
  __builtin_neon_vld1q_x4_v(&__ret, __p0, 36); \
  __ret; \
})
#else
#define vld1q_p8_x4(__p0) __extension__ ({ \
  poly8x16x4_t __ret; \
  __builtin_neon_vld1q_x4_v(&__ret, __p0, 36); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_p16_x4(__p0) __extension__ ({ \
  poly16x8x4_t __ret; \
  __builtin_neon_vld1q_x4_v(&__ret, __p0, 37); \
  __ret; \
})
#else
#define vld1q_p16_x4(__p0) __extension__ ({ \
  poly16x8x4_t __ret; \
  __builtin_neon_vld1q_x4_v(&__ret, __p0, 37); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_u8_x4(__p0) __extension__ ({ \
  uint8x16x4_t __ret; \
  __builtin_neon_vld1q_x4_v(&__ret, __p0, 48); \
  __ret; \
})
#else
#define vld1q_u8_x4(__p0) __extension__ ({ \
  uint8x16x4_t __ret; \
  __builtin_neon_vld1q_x4_v(&__ret, __p0, 48); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_u32_x4(__p0) __extension__ ({ \
  uint32x4x4_t __ret; \
  __builtin_neon_vld1q_x4_v(&__ret, __p0, 50); \
  __ret; \
})
#else
#define vld1q_u32_x4(__p0) __extension__ ({ \
  uint32x4x4_t __ret; \
  __builtin_neon_vld1q_x4_v(&__ret, __p0, 50); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_u64_x4(__p0) __extension__ ({ \
  uint64x2x4_t __ret; \
  __builtin_neon_vld1q_x4_v(&__ret, __p0, 51); \
  __ret; \
})
#else
#define vld1q_u64_x4(__p0) __extension__ ({ \
  uint64x2x4_t __ret; \
  __builtin_neon_vld1q_x4_v(&__ret, __p0, 51); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_u16_x4(__p0) __extension__ ({ \
  uint16x8x4_t __ret; \
  __builtin_neon_vld1q_x4_v(&__ret, __p0, 49); \
  __ret; \
})
#else
#define vld1q_u16_x4(__p0) __extension__ ({ \
  uint16x8x4_t __ret; \
  __builtin_neon_vld1q_x4_v(&__ret, __p0, 49); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_s8_x4(__p0) __extension__ ({ \
  int8x16x4_t __ret; \
  __builtin_neon_vld1q_x4_v(&__ret, __p0, 32); \
  __ret; \
})
#else
#define vld1q_s8_x4(__p0) __extension__ ({ \
  int8x16x4_t __ret; \
  __builtin_neon_vld1q_x4_v(&__ret, __p0, 32); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_f32_x4(__p0) __extension__ ({ \
  float32x4x4_t __ret; \
  __builtin_neon_vld1q_x4_v(&__ret, __p0, 41); \
  __ret; \
})
#else
#define vld1q_f32_x4(__p0) __extension__ ({ \
  float32x4x4_t __ret; \
  __builtin_neon_vld1q_x4_v(&__ret, __p0, 41); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_s32_x4(__p0) __extension__ ({ \
  int32x4x4_t __ret; \
  __builtin_neon_vld1q_x4_v(&__ret, __p0, 34); \
  __ret; \
})
#else
#define vld1q_s32_x4(__p0) __extension__ ({ \
  int32x4x4_t __ret; \
  __builtin_neon_vld1q_x4_v(&__ret, __p0, 34); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_s64_x4(__p0) __extension__ ({ \
  int64x2x4_t __ret; \
  __builtin_neon_vld1q_x4_v(&__ret, __p0, 35); \
  __ret; \
})
#else
#define vld1q_s64_x4(__p0) __extension__ ({ \
  int64x2x4_t __ret; \
  __builtin_neon_vld1q_x4_v(&__ret, __p0, 35); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1q_s16_x4(__p0) __extension__ ({ \
  int16x8x4_t __ret; \
  __builtin_neon_vld1q_x4_v(&__ret, __p0, 33); \
  __ret; \
})
#else
#define vld1q_s16_x4(__p0) __extension__ ({ \
  int16x8x4_t __ret; \
  __builtin_neon_vld1q_x4_v(&__ret, __p0, 33); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1_u8_x4(__p0) __extension__ ({ \
  uint8x8x4_t __ret; \
  __builtin_neon_vld1_x4_v(&__ret, __p0, 16); \
  __ret; \
})
#else
#define vld1_u8_x4(__p0) __extension__ ({ \
  uint8x8x4_t __ret; \
  __builtin_neon_vld1_x4_v(&__ret, __p0, 16); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1_u32_x4(__p0) __extension__ ({ \
  uint32x2x4_t __ret; \
  __builtin_neon_vld1_x4_v(&__ret, __p0, 18); \
  __ret; \
})
#else
#define vld1_u32_x4(__p0) __extension__ ({ \
  uint32x2x4_t __ret; \
  __builtin_neon_vld1_x4_v(&__ret, __p0, 18); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 1, 0); \
  __ret; \
})
#endif

#define vld1_u64_x4(__p0) __extension__ ({ \
  uint64x1x4_t __ret; \
  __builtin_neon_vld1_x4_v(&__ret, __p0, 19); \
  __ret; \
})
#ifdef __LITTLE_ENDIAN__
#define vld1_u16_x4(__p0) __extension__ ({ \
  uint16x4x4_t __ret; \
  __builtin_neon_vld1_x4_v(&__ret, __p0, 17); \
  __ret; \
})
#else
#define vld1_u16_x4(__p0) __extension__ ({ \
  uint16x4x4_t __ret; \
  __builtin_neon_vld1_x4_v(&__ret, __p0, 17); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1_s8_x4(__p0) __extension__ ({ \
  int8x8x4_t __ret; \
  __builtin_neon_vld1_x4_v(&__ret, __p0, 0); \
  __ret; \
})
#else
#define vld1_s8_x4(__p0) __extension__ ({ \
  int8x8x4_t __ret; \
  __builtin_neon_vld1_x4_v(&__ret, __p0, 0); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1_f32_x4(__p0) __extension__ ({ \
  float32x2x4_t __ret; \
  __builtin_neon_vld1_x4_v(&__ret, __p0, 9); \
  __ret; \
})
#else
#define vld1_f32_x4(__p0) __extension__ ({ \
  float32x2x4_t __ret; \
  __builtin_neon_vld1_x4_v(&__ret, __p0, 9); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld1_s32_x4(__p0) __extension__ ({ \
  int32x2x4_t __ret; \
  __builtin_neon_vld1_x4_v(&__ret, __p0, 2); \
  __ret; \
})
#else
#define vld1_s32_x4(__p0) __extension__ ({ \
  int32x2x4_t __ret; \
  __builtin_neon_vld1_x4_v(&__ret, __p0, 2); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 1, 0); \
  __ret; \
})
#endif

#define vld1_s64_x4(__p0) __extension__ ({ \
  int64x1x4_t __ret; \
  __builtin_neon_vld1_x4_v(&__ret, __p0, 3); \
  __ret; \
})
#ifdef __LITTLE_ENDIAN__
#define vld1_s16_x4(__p0) __extension__ ({ \
  int16x4x4_t __ret; \
  __builtin_neon_vld1_x4_v(&__ret, __p0, 1); \
  __ret; \
})
#else
#define vld1_s16_x4(__p0) __extension__ ({ \
  int16x4x4_t __ret; \
  __builtin_neon_vld1_x4_v(&__ret, __p0, 1); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld2_p8(__p0) __extension__ ({ \
  poly8x8x2_t __ret; \
  __builtin_neon_vld2_v(&__ret, __p0, 4); \
  __ret; \
})
#else
#define vld2_p8(__p0) __extension__ ({ \
  poly8x8x2_t __ret; \
  __builtin_neon_vld2_v(&__ret, __p0, 4); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld2_p16(__p0) __extension__ ({ \
  poly16x4x2_t __ret; \
  __builtin_neon_vld2_v(&__ret, __p0, 5); \
  __ret; \
})
#else
#define vld2_p16(__p0) __extension__ ({ \
  poly16x4x2_t __ret; \
  __builtin_neon_vld2_v(&__ret, __p0, 5); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld2q_p8(__p0) __extension__ ({ \
  poly8x16x2_t __ret; \
  __builtin_neon_vld2q_v(&__ret, __p0, 36); \
  __ret; \
})
#else
#define vld2q_p8(__p0) __extension__ ({ \
  poly8x16x2_t __ret; \
  __builtin_neon_vld2q_v(&__ret, __p0, 36); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld2q_p16(__p0) __extension__ ({ \
  poly16x8x2_t __ret; \
  __builtin_neon_vld2q_v(&__ret, __p0, 37); \
  __ret; \
})
#else
#define vld2q_p16(__p0) __extension__ ({ \
  poly16x8x2_t __ret; \
  __builtin_neon_vld2q_v(&__ret, __p0, 37); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld2q_u8(__p0) __extension__ ({ \
  uint8x16x2_t __ret; \
  __builtin_neon_vld2q_v(&__ret, __p0, 48); \
  __ret; \
})
#else
#define vld2q_u8(__p0) __extension__ ({ \
  uint8x16x2_t __ret; \
  __builtin_neon_vld2q_v(&__ret, __p0, 48); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld2q_u32(__p0) __extension__ ({ \
  uint32x4x2_t __ret; \
  __builtin_neon_vld2q_v(&__ret, __p0, 50); \
  __ret; \
})
#else
#define vld2q_u32(__p0) __extension__ ({ \
  uint32x4x2_t __ret; \
  __builtin_neon_vld2q_v(&__ret, __p0, 50); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld2q_u16(__p0) __extension__ ({ \
  uint16x8x2_t __ret; \
  __builtin_neon_vld2q_v(&__ret, __p0, 49); \
  __ret; \
})
#else
#define vld2q_u16(__p0) __extension__ ({ \
  uint16x8x2_t __ret; \
  __builtin_neon_vld2q_v(&__ret, __p0, 49); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld2q_s8(__p0) __extension__ ({ \
  int8x16x2_t __ret; \
  __builtin_neon_vld2q_v(&__ret, __p0, 32); \
  __ret; \
})
#else
#define vld2q_s8(__p0) __extension__ ({ \
  int8x16x2_t __ret; \
  __builtin_neon_vld2q_v(&__ret, __p0, 32); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld2q_f32(__p0) __extension__ ({ \
  float32x4x2_t __ret; \
  __builtin_neon_vld2q_v(&__ret, __p0, 41); \
  __ret; \
})
#else
#define vld2q_f32(__p0) __extension__ ({ \
  float32x4x2_t __ret; \
  __builtin_neon_vld2q_v(&__ret, __p0, 41); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld2q_s32(__p0) __extension__ ({ \
  int32x4x2_t __ret; \
  __builtin_neon_vld2q_v(&__ret, __p0, 34); \
  __ret; \
})
#else
#define vld2q_s32(__p0) __extension__ ({ \
  int32x4x2_t __ret; \
  __builtin_neon_vld2q_v(&__ret, __p0, 34); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld2q_s16(__p0) __extension__ ({ \
  int16x8x2_t __ret; \
  __builtin_neon_vld2q_v(&__ret, __p0, 33); \
  __ret; \
})
#else
#define vld2q_s16(__p0) __extension__ ({ \
  int16x8x2_t __ret; \
  __builtin_neon_vld2q_v(&__ret, __p0, 33); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld2_u8(__p0) __extension__ ({ \
  uint8x8x2_t __ret; \
  __builtin_neon_vld2_v(&__ret, __p0, 16); \
  __ret; \
})
#else
#define vld2_u8(__p0) __extension__ ({ \
  uint8x8x2_t __ret; \
  __builtin_neon_vld2_v(&__ret, __p0, 16); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld2_u32(__p0) __extension__ ({ \
  uint32x2x2_t __ret; \
  __builtin_neon_vld2_v(&__ret, __p0, 18); \
  __ret; \
})
#else
#define vld2_u32(__p0) __extension__ ({ \
  uint32x2x2_t __ret; \
  __builtin_neon_vld2_v(&__ret, __p0, 18); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 1, 0); \
  __ret; \
})
#endif

#define vld2_u64(__p0) __extension__ ({ \
  uint64x1x2_t __ret; \
  __builtin_neon_vld2_v(&__ret, __p0, 19); \
  __ret; \
})
#ifdef __LITTLE_ENDIAN__
#define vld2_u16(__p0) __extension__ ({ \
  uint16x4x2_t __ret; \
  __builtin_neon_vld2_v(&__ret, __p0, 17); \
  __ret; \
})
#else
#define vld2_u16(__p0) __extension__ ({ \
  uint16x4x2_t __ret; \
  __builtin_neon_vld2_v(&__ret, __p0, 17); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld2_s8(__p0) __extension__ ({ \
  int8x8x2_t __ret; \
  __builtin_neon_vld2_v(&__ret, __p0, 0); \
  __ret; \
})
#else
#define vld2_s8(__p0) __extension__ ({ \
  int8x8x2_t __ret; \
  __builtin_neon_vld2_v(&__ret, __p0, 0); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld2_f32(__p0) __extension__ ({ \
  float32x2x2_t __ret; \
  __builtin_neon_vld2_v(&__ret, __p0, 9); \
  __ret; \
})
#else
#define vld2_f32(__p0) __extension__ ({ \
  float32x2x2_t __ret; \
  __builtin_neon_vld2_v(&__ret, __p0, 9); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld2_s32(__p0) __extension__ ({ \
  int32x2x2_t __ret; \
  __builtin_neon_vld2_v(&__ret, __p0, 2); \
  __ret; \
})
#else
#define vld2_s32(__p0) __extension__ ({ \
  int32x2x2_t __ret; \
  __builtin_neon_vld2_v(&__ret, __p0, 2); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 1, 0); \
  __ret; \
})
#endif

#define vld2_s64(__p0) __extension__ ({ \
  int64x1x2_t __ret; \
  __builtin_neon_vld2_v(&__ret, __p0, 3); \
  __ret; \
})
#ifdef __LITTLE_ENDIAN__
#define vld2_s16(__p0) __extension__ ({ \
  int16x4x2_t __ret; \
  __builtin_neon_vld2_v(&__ret, __p0, 1); \
  __ret; \
})
#else
#define vld2_s16(__p0) __extension__ ({ \
  int16x4x2_t __ret; \
  __builtin_neon_vld2_v(&__ret, __p0, 1); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld2_dup_p8(__p0) __extension__ ({ \
  poly8x8x2_t __ret; \
  __builtin_neon_vld2_dup_v(&__ret, __p0, 4); \
  __ret; \
})
#else
#define vld2_dup_p8(__p0) __extension__ ({ \
  poly8x8x2_t __ret; \
  __builtin_neon_vld2_dup_v(&__ret, __p0, 4); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld2_dup_p16(__p0) __extension__ ({ \
  poly16x4x2_t __ret; \
  __builtin_neon_vld2_dup_v(&__ret, __p0, 5); \
  __ret; \
})
#else
#define vld2_dup_p16(__p0) __extension__ ({ \
  poly16x4x2_t __ret; \
  __builtin_neon_vld2_dup_v(&__ret, __p0, 5); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld2q_dup_p8(__p0) __extension__ ({ \
  poly8x16x2_t __ret; \
  __builtin_neon_vld2q_dup_v(&__ret, __p0, 36); \
  __ret; \
})
#else
#define vld2q_dup_p8(__p0) __extension__ ({ \
  poly8x16x2_t __ret; \
  __builtin_neon_vld2q_dup_v(&__ret, __p0, 36); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld2q_dup_p16(__p0) __extension__ ({ \
  poly16x8x2_t __ret; \
  __builtin_neon_vld2q_dup_v(&__ret, __p0, 37); \
  __ret; \
})
#else
#define vld2q_dup_p16(__p0) __extension__ ({ \
  poly16x8x2_t __ret; \
  __builtin_neon_vld2q_dup_v(&__ret, __p0, 37); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld2q_dup_u8(__p0) __extension__ ({ \
  uint8x16x2_t __ret; \
  __builtin_neon_vld2q_dup_v(&__ret, __p0, 48); \
  __ret; \
})
#else
#define vld2q_dup_u8(__p0) __extension__ ({ \
  uint8x16x2_t __ret; \
  __builtin_neon_vld2q_dup_v(&__ret, __p0, 48); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld2q_dup_u32(__p0) __extension__ ({ \
  uint32x4x2_t __ret; \
  __builtin_neon_vld2q_dup_v(&__ret, __p0, 50); \
  __ret; \
})
#else
#define vld2q_dup_u32(__p0) __extension__ ({ \
  uint32x4x2_t __ret; \
  __builtin_neon_vld2q_dup_v(&__ret, __p0, 50); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld2q_dup_u64(__p0) __extension__ ({ \
  uint64x2x2_t __ret; \
  __builtin_neon_vld2q_dup_v(&__ret, __p0, 51); \
  __ret; \
})
#else
#define vld2q_dup_u64(__p0) __extension__ ({ \
  uint64x2x2_t __ret; \
  __builtin_neon_vld2q_dup_v(&__ret, __p0, 51); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld2q_dup_u16(__p0) __extension__ ({ \
  uint16x8x2_t __ret; \
  __builtin_neon_vld2q_dup_v(&__ret, __p0, 49); \
  __ret; \
})
#else
#define vld2q_dup_u16(__p0) __extension__ ({ \
  uint16x8x2_t __ret; \
  __builtin_neon_vld2q_dup_v(&__ret, __p0, 49); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld2q_dup_s8(__p0) __extension__ ({ \
  int8x16x2_t __ret; \
  __builtin_neon_vld2q_dup_v(&__ret, __p0, 32); \
  __ret; \
})
#else
#define vld2q_dup_s8(__p0) __extension__ ({ \
  int8x16x2_t __ret; \
  __builtin_neon_vld2q_dup_v(&__ret, __p0, 32); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld2q_dup_f32(__p0) __extension__ ({ \
  float32x4x2_t __ret; \
  __builtin_neon_vld2q_dup_v(&__ret, __p0, 41); \
  __ret; \
})
#else
#define vld2q_dup_f32(__p0) __extension__ ({ \
  float32x4x2_t __ret; \
  __builtin_neon_vld2q_dup_v(&__ret, __p0, 41); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld2q_dup_s32(__p0) __extension__ ({ \
  int32x4x2_t __ret; \
  __builtin_neon_vld2q_dup_v(&__ret, __p0, 34); \
  __ret; \
})
#else
#define vld2q_dup_s32(__p0) __extension__ ({ \
  int32x4x2_t __ret; \
  __builtin_neon_vld2q_dup_v(&__ret, __p0, 34); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld2q_dup_s64(__p0) __extension__ ({ \
  int64x2x2_t __ret; \
  __builtin_neon_vld2q_dup_v(&__ret, __p0, 35); \
  __ret; \
})
#else
#define vld2q_dup_s64(__p0) __extension__ ({ \
  int64x2x2_t __ret; \
  __builtin_neon_vld2q_dup_v(&__ret, __p0, 35); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld2q_dup_s16(__p0) __extension__ ({ \
  int16x8x2_t __ret; \
  __builtin_neon_vld2q_dup_v(&__ret, __p0, 33); \
  __ret; \
})
#else
#define vld2q_dup_s16(__p0) __extension__ ({ \
  int16x8x2_t __ret; \
  __builtin_neon_vld2q_dup_v(&__ret, __p0, 33); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld2_dup_u8(__p0) __extension__ ({ \
  uint8x8x2_t __ret; \
  __builtin_neon_vld2_dup_v(&__ret, __p0, 16); \
  __ret; \
})
#else
#define vld2_dup_u8(__p0) __extension__ ({ \
  uint8x8x2_t __ret; \
  __builtin_neon_vld2_dup_v(&__ret, __p0, 16); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld2_dup_u32(__p0) __extension__ ({ \
  uint32x2x2_t __ret; \
  __builtin_neon_vld2_dup_v(&__ret, __p0, 18); \
  __ret; \
})
#else
#define vld2_dup_u32(__p0) __extension__ ({ \
  uint32x2x2_t __ret; \
  __builtin_neon_vld2_dup_v(&__ret, __p0, 18); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 1, 0); \
  __ret; \
})
#endif

#define vld2_dup_u64(__p0) __extension__ ({ \
  uint64x1x2_t __ret; \
  __builtin_neon_vld2_dup_v(&__ret, __p0, 19); \
  __ret; \
})
#ifdef __LITTLE_ENDIAN__
#define vld2_dup_u16(__p0) __extension__ ({ \
  uint16x4x2_t __ret; \
  __builtin_neon_vld2_dup_v(&__ret, __p0, 17); \
  __ret; \
})
#else
#define vld2_dup_u16(__p0) __extension__ ({ \
  uint16x4x2_t __ret; \
  __builtin_neon_vld2_dup_v(&__ret, __p0, 17); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld2_dup_s8(__p0) __extension__ ({ \
  int8x8x2_t __ret; \
  __builtin_neon_vld2_dup_v(&__ret, __p0, 0); \
  __ret; \
})
#else
#define vld2_dup_s8(__p0) __extension__ ({ \
  int8x8x2_t __ret; \
  __builtin_neon_vld2_dup_v(&__ret, __p0, 0); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld2_dup_f32(__p0) __extension__ ({ \
  float32x2x2_t __ret; \
  __builtin_neon_vld2_dup_v(&__ret, __p0, 9); \
  __ret; \
})
#else
#define vld2_dup_f32(__p0) __extension__ ({ \
  float32x2x2_t __ret; \
  __builtin_neon_vld2_dup_v(&__ret, __p0, 9); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld2_dup_s32(__p0) __extension__ ({ \
  int32x2x2_t __ret; \
  __builtin_neon_vld2_dup_v(&__ret, __p0, 2); \
  __ret; \
})
#else
#define vld2_dup_s32(__p0) __extension__ ({ \
  int32x2x2_t __ret; \
  __builtin_neon_vld2_dup_v(&__ret, __p0, 2); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 1, 0); \
  __ret; \
})
#endif

#define vld2_dup_s64(__p0) __extension__ ({ \
  int64x1x2_t __ret; \
  __builtin_neon_vld2_dup_v(&__ret, __p0, 3); \
  __ret; \
})
#ifdef __LITTLE_ENDIAN__
#define vld2_dup_s16(__p0) __extension__ ({ \
  int16x4x2_t __ret; \
  __builtin_neon_vld2_dup_v(&__ret, __p0, 1); \
  __ret; \
})
#else
#define vld2_dup_s16(__p0) __extension__ ({ \
  int16x4x2_t __ret; \
  __builtin_neon_vld2_dup_v(&__ret, __p0, 1); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld2_lane_p8(__p0, __p1, __p2) __extension__ ({ \
  poly8x8x2_t __s1 = __p1; \
  poly8x8x2_t __ret; \
  __builtin_neon_vld2_lane_v(&__ret, __p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], __p2, 4); \
  __ret; \
})
#else
#define vld2_lane_p8(__p0, __p1, __p2) __extension__ ({ \
  poly8x8x2_t __s1 = __p1; \
  poly8x8x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  poly8x8x2_t __ret; \
  __builtin_neon_vld2_lane_v(&__ret, __p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], __p2, 4); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld2_lane_p16(__p0, __p1, __p2) __extension__ ({ \
  poly16x4x2_t __s1 = __p1; \
  poly16x4x2_t __ret; \
  __builtin_neon_vld2_lane_v(&__ret, __p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], __p2, 5); \
  __ret; \
})
#else
#define vld2_lane_p16(__p0, __p1, __p2) __extension__ ({ \
  poly16x4x2_t __s1 = __p1; \
  poly16x4x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 3, 2, 1, 0); \
  poly16x4x2_t __ret; \
  __builtin_neon_vld2_lane_v(&__ret, __p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], __p2, 5); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld2q_lane_p16(__p0, __p1, __p2) __extension__ ({ \
  poly16x8x2_t __s1 = __p1; \
  poly16x8x2_t __ret; \
  __builtin_neon_vld2q_lane_v(&__ret, __p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], __p2, 37); \
  __ret; \
})
#else
#define vld2q_lane_p16(__p0, __p1, __p2) __extension__ ({ \
  poly16x8x2_t __s1 = __p1; \
  poly16x8x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  poly16x8x2_t __ret; \
  __builtin_neon_vld2q_lane_v(&__ret, __p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], __p2, 37); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld2q_lane_u32(__p0, __p1, __p2) __extension__ ({ \
  uint32x4x2_t __s1 = __p1; \
  uint32x4x2_t __ret; \
  __builtin_neon_vld2q_lane_v(&__ret, __p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], __p2, 50); \
  __ret; \
})
#else
#define vld2q_lane_u32(__p0, __p1, __p2) __extension__ ({ \
  uint32x4x2_t __s1 = __p1; \
  uint32x4x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 3, 2, 1, 0); \
  uint32x4x2_t __ret; \
  __builtin_neon_vld2q_lane_v(&__ret, __p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], __p2, 50); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld2q_lane_u16(__p0, __p1, __p2) __extension__ ({ \
  uint16x8x2_t __s1 = __p1; \
  uint16x8x2_t __ret; \
  __builtin_neon_vld2q_lane_v(&__ret, __p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], __p2, 49); \
  __ret; \
})
#else
#define vld2q_lane_u16(__p0, __p1, __p2) __extension__ ({ \
  uint16x8x2_t __s1 = __p1; \
  uint16x8x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  uint16x8x2_t __ret; \
  __builtin_neon_vld2q_lane_v(&__ret, __p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], __p2, 49); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld2q_lane_f32(__p0, __p1, __p2) __extension__ ({ \
  float32x4x2_t __s1 = __p1; \
  float32x4x2_t __ret; \
  __builtin_neon_vld2q_lane_v(&__ret, __p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], __p2, 41); \
  __ret; \
})
#else
#define vld2q_lane_f32(__p0, __p1, __p2) __extension__ ({ \
  float32x4x2_t __s1 = __p1; \
  float32x4x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 3, 2, 1, 0); \
  float32x4x2_t __ret; \
  __builtin_neon_vld2q_lane_v(&__ret, __p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], __p2, 41); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld2q_lane_s32(__p0, __p1, __p2) __extension__ ({ \
  int32x4x2_t __s1 = __p1; \
  int32x4x2_t __ret; \
  __builtin_neon_vld2q_lane_v(&__ret, __p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], __p2, 34); \
  __ret; \
})
#else
#define vld2q_lane_s32(__p0, __p1, __p2) __extension__ ({ \
  int32x4x2_t __s1 = __p1; \
  int32x4x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 3, 2, 1, 0); \
  int32x4x2_t __ret; \
  __builtin_neon_vld2q_lane_v(&__ret, __p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], __p2, 34); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld2q_lane_s16(__p0, __p1, __p2) __extension__ ({ \
  int16x8x2_t __s1 = __p1; \
  int16x8x2_t __ret; \
  __builtin_neon_vld2q_lane_v(&__ret, __p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], __p2, 33); \
  __ret; \
})
#else
#define vld2q_lane_s16(__p0, __p1, __p2) __extension__ ({ \
  int16x8x2_t __s1 = __p1; \
  int16x8x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  int16x8x2_t __ret; \
  __builtin_neon_vld2q_lane_v(&__ret, __p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], __p2, 33); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld2_lane_u8(__p0, __p1, __p2) __extension__ ({ \
  uint8x8x2_t __s1 = __p1; \
  uint8x8x2_t __ret; \
  __builtin_neon_vld2_lane_v(&__ret, __p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], __p2, 16); \
  __ret; \
})
#else
#define vld2_lane_u8(__p0, __p1, __p2) __extension__ ({ \
  uint8x8x2_t __s1 = __p1; \
  uint8x8x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  uint8x8x2_t __ret; \
  __builtin_neon_vld2_lane_v(&__ret, __p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], __p2, 16); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld2_lane_u32(__p0, __p1, __p2) __extension__ ({ \
  uint32x2x2_t __s1 = __p1; \
  uint32x2x2_t __ret; \
  __builtin_neon_vld2_lane_v(&__ret, __p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], __p2, 18); \
  __ret; \
})
#else
#define vld2_lane_u32(__p0, __p1, __p2) __extension__ ({ \
  uint32x2x2_t __s1 = __p1; \
  uint32x2x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 1, 0); \
  uint32x2x2_t __ret; \
  __builtin_neon_vld2_lane_v(&__ret, __p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], __p2, 18); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld2_lane_u16(__p0, __p1, __p2) __extension__ ({ \
  uint16x4x2_t __s1 = __p1; \
  uint16x4x2_t __ret; \
  __builtin_neon_vld2_lane_v(&__ret, __p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], __p2, 17); \
  __ret; \
})
#else
#define vld2_lane_u16(__p0, __p1, __p2) __extension__ ({ \
  uint16x4x2_t __s1 = __p1; \
  uint16x4x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 3, 2, 1, 0); \
  uint16x4x2_t __ret; \
  __builtin_neon_vld2_lane_v(&__ret, __p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], __p2, 17); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld2_lane_s8(__p0, __p1, __p2) __extension__ ({ \
  int8x8x2_t __s1 = __p1; \
  int8x8x2_t __ret; \
  __builtin_neon_vld2_lane_v(&__ret, __p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], __p2, 0); \
  __ret; \
})
#else
#define vld2_lane_s8(__p0, __p1, __p2) __extension__ ({ \
  int8x8x2_t __s1 = __p1; \
  int8x8x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  int8x8x2_t __ret; \
  __builtin_neon_vld2_lane_v(&__ret, __p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], __p2, 0); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld2_lane_f32(__p0, __p1, __p2) __extension__ ({ \
  float32x2x2_t __s1 = __p1; \
  float32x2x2_t __ret; \
  __builtin_neon_vld2_lane_v(&__ret, __p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], __p2, 9); \
  __ret; \
})
#else
#define vld2_lane_f32(__p0, __p1, __p2) __extension__ ({ \
  float32x2x2_t __s1 = __p1; \
  float32x2x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 1, 0); \
  float32x2x2_t __ret; \
  __builtin_neon_vld2_lane_v(&__ret, __p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], __p2, 9); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld2_lane_s32(__p0, __p1, __p2) __extension__ ({ \
  int32x2x2_t __s1 = __p1; \
  int32x2x2_t __ret; \
  __builtin_neon_vld2_lane_v(&__ret, __p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], __p2, 2); \
  __ret; \
})
#else
#define vld2_lane_s32(__p0, __p1, __p2) __extension__ ({ \
  int32x2x2_t __s1 = __p1; \
  int32x2x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 1, 0); \
  int32x2x2_t __ret; \
  __builtin_neon_vld2_lane_v(&__ret, __p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], __p2, 2); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld2_lane_s16(__p0, __p1, __p2) __extension__ ({ \
  int16x4x2_t __s1 = __p1; \
  int16x4x2_t __ret; \
  __builtin_neon_vld2_lane_v(&__ret, __p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], __p2, 1); \
  __ret; \
})
#else
#define vld2_lane_s16(__p0, __p1, __p2) __extension__ ({ \
  int16x4x2_t __s1 = __p1; \
  int16x4x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 3, 2, 1, 0); \
  int16x4x2_t __ret; \
  __builtin_neon_vld2_lane_v(&__ret, __p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], __p2, 1); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld3_p8(__p0) __extension__ ({ \
  poly8x8x3_t __ret; \
  __builtin_neon_vld3_v(&__ret, __p0, 4); \
  __ret; \
})
#else
#define vld3_p8(__p0) __extension__ ({ \
  poly8x8x3_t __ret; \
  __builtin_neon_vld3_v(&__ret, __p0, 4); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld3_p16(__p0) __extension__ ({ \
  poly16x4x3_t __ret; \
  __builtin_neon_vld3_v(&__ret, __p0, 5); \
  __ret; \
})
#else
#define vld3_p16(__p0) __extension__ ({ \
  poly16x4x3_t __ret; \
  __builtin_neon_vld3_v(&__ret, __p0, 5); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld3q_p8(__p0) __extension__ ({ \
  poly8x16x3_t __ret; \
  __builtin_neon_vld3q_v(&__ret, __p0, 36); \
  __ret; \
})
#else
#define vld3q_p8(__p0) __extension__ ({ \
  poly8x16x3_t __ret; \
  __builtin_neon_vld3q_v(&__ret, __p0, 36); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld3q_p16(__p0) __extension__ ({ \
  poly16x8x3_t __ret; \
  __builtin_neon_vld3q_v(&__ret, __p0, 37); \
  __ret; \
})
#else
#define vld3q_p16(__p0) __extension__ ({ \
  poly16x8x3_t __ret; \
  __builtin_neon_vld3q_v(&__ret, __p0, 37); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld3q_u8(__p0) __extension__ ({ \
  uint8x16x3_t __ret; \
  __builtin_neon_vld3q_v(&__ret, __p0, 48); \
  __ret; \
})
#else
#define vld3q_u8(__p0) __extension__ ({ \
  uint8x16x3_t __ret; \
  __builtin_neon_vld3q_v(&__ret, __p0, 48); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld3q_u32(__p0) __extension__ ({ \
  uint32x4x3_t __ret; \
  __builtin_neon_vld3q_v(&__ret, __p0, 50); \
  __ret; \
})
#else
#define vld3q_u32(__p0) __extension__ ({ \
  uint32x4x3_t __ret; \
  __builtin_neon_vld3q_v(&__ret, __p0, 50); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld3q_u16(__p0) __extension__ ({ \
  uint16x8x3_t __ret; \
  __builtin_neon_vld3q_v(&__ret, __p0, 49); \
  __ret; \
})
#else
#define vld3q_u16(__p0) __extension__ ({ \
  uint16x8x3_t __ret; \
  __builtin_neon_vld3q_v(&__ret, __p0, 49); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld3q_s8(__p0) __extension__ ({ \
  int8x16x3_t __ret; \
  __builtin_neon_vld3q_v(&__ret, __p0, 32); \
  __ret; \
})
#else
#define vld3q_s8(__p0) __extension__ ({ \
  int8x16x3_t __ret; \
  __builtin_neon_vld3q_v(&__ret, __p0, 32); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld3q_f32(__p0) __extension__ ({ \
  float32x4x3_t __ret; \
  __builtin_neon_vld3q_v(&__ret, __p0, 41); \
  __ret; \
})
#else
#define vld3q_f32(__p0) __extension__ ({ \
  float32x4x3_t __ret; \
  __builtin_neon_vld3q_v(&__ret, __p0, 41); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld3q_s32(__p0) __extension__ ({ \
  int32x4x3_t __ret; \
  __builtin_neon_vld3q_v(&__ret, __p0, 34); \
  __ret; \
})
#else
#define vld3q_s32(__p0) __extension__ ({ \
  int32x4x3_t __ret; \
  __builtin_neon_vld3q_v(&__ret, __p0, 34); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld3q_s16(__p0) __extension__ ({ \
  int16x8x3_t __ret; \
  __builtin_neon_vld3q_v(&__ret, __p0, 33); \
  __ret; \
})
#else
#define vld3q_s16(__p0) __extension__ ({ \
  int16x8x3_t __ret; \
  __builtin_neon_vld3q_v(&__ret, __p0, 33); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld3_u8(__p0) __extension__ ({ \
  uint8x8x3_t __ret; \
  __builtin_neon_vld3_v(&__ret, __p0, 16); \
  __ret; \
})
#else
#define vld3_u8(__p0) __extension__ ({ \
  uint8x8x3_t __ret; \
  __builtin_neon_vld3_v(&__ret, __p0, 16); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld3_u32(__p0) __extension__ ({ \
  uint32x2x3_t __ret; \
  __builtin_neon_vld3_v(&__ret, __p0, 18); \
  __ret; \
})
#else
#define vld3_u32(__p0) __extension__ ({ \
  uint32x2x3_t __ret; \
  __builtin_neon_vld3_v(&__ret, __p0, 18); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 1, 0); \
  __ret; \
})
#endif

#define vld3_u64(__p0) __extension__ ({ \
  uint64x1x3_t __ret; \
  __builtin_neon_vld3_v(&__ret, __p0, 19); \
  __ret; \
})
#ifdef __LITTLE_ENDIAN__
#define vld3_u16(__p0) __extension__ ({ \
  uint16x4x3_t __ret; \
  __builtin_neon_vld3_v(&__ret, __p0, 17); \
  __ret; \
})
#else
#define vld3_u16(__p0) __extension__ ({ \
  uint16x4x3_t __ret; \
  __builtin_neon_vld3_v(&__ret, __p0, 17); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld3_s8(__p0) __extension__ ({ \
  int8x8x3_t __ret; \
  __builtin_neon_vld3_v(&__ret, __p0, 0); \
  __ret; \
})
#else
#define vld3_s8(__p0) __extension__ ({ \
  int8x8x3_t __ret; \
  __builtin_neon_vld3_v(&__ret, __p0, 0); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld3_f32(__p0) __extension__ ({ \
  float32x2x3_t __ret; \
  __builtin_neon_vld3_v(&__ret, __p0, 9); \
  __ret; \
})
#else
#define vld3_f32(__p0) __extension__ ({ \
  float32x2x3_t __ret; \
  __builtin_neon_vld3_v(&__ret, __p0, 9); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld3_s32(__p0) __extension__ ({ \
  int32x2x3_t __ret; \
  __builtin_neon_vld3_v(&__ret, __p0, 2); \
  __ret; \
})
#else
#define vld3_s32(__p0) __extension__ ({ \
  int32x2x3_t __ret; \
  __builtin_neon_vld3_v(&__ret, __p0, 2); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 1, 0); \
  __ret; \
})
#endif

#define vld3_s64(__p0) __extension__ ({ \
  int64x1x3_t __ret; \
  __builtin_neon_vld3_v(&__ret, __p0, 3); \
  __ret; \
})
#ifdef __LITTLE_ENDIAN__
#define vld3_s16(__p0) __extension__ ({ \
  int16x4x3_t __ret; \
  __builtin_neon_vld3_v(&__ret, __p0, 1); \
  __ret; \
})
#else
#define vld3_s16(__p0) __extension__ ({ \
  int16x4x3_t __ret; \
  __builtin_neon_vld3_v(&__ret, __p0, 1); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld3_dup_p8(__p0) __extension__ ({ \
  poly8x8x3_t __ret; \
  __builtin_neon_vld3_dup_v(&__ret, __p0, 4); \
  __ret; \
})
#else
#define vld3_dup_p8(__p0) __extension__ ({ \
  poly8x8x3_t __ret; \
  __builtin_neon_vld3_dup_v(&__ret, __p0, 4); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld3_dup_p16(__p0) __extension__ ({ \
  poly16x4x3_t __ret; \
  __builtin_neon_vld3_dup_v(&__ret, __p0, 5); \
  __ret; \
})
#else
#define vld3_dup_p16(__p0) __extension__ ({ \
  poly16x4x3_t __ret; \
  __builtin_neon_vld3_dup_v(&__ret, __p0, 5); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld3q_dup_p8(__p0) __extension__ ({ \
  poly8x16x3_t __ret; \
  __builtin_neon_vld3q_dup_v(&__ret, __p0, 36); \
  __ret; \
})
#else
#define vld3q_dup_p8(__p0) __extension__ ({ \
  poly8x16x3_t __ret; \
  __builtin_neon_vld3q_dup_v(&__ret, __p0, 36); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld3q_dup_p16(__p0) __extension__ ({ \
  poly16x8x3_t __ret; \
  __builtin_neon_vld3q_dup_v(&__ret, __p0, 37); \
  __ret; \
})
#else
#define vld3q_dup_p16(__p0) __extension__ ({ \
  poly16x8x3_t __ret; \
  __builtin_neon_vld3q_dup_v(&__ret, __p0, 37); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld3q_dup_u8(__p0) __extension__ ({ \
  uint8x16x3_t __ret; \
  __builtin_neon_vld3q_dup_v(&__ret, __p0, 48); \
  __ret; \
})
#else
#define vld3q_dup_u8(__p0) __extension__ ({ \
  uint8x16x3_t __ret; \
  __builtin_neon_vld3q_dup_v(&__ret, __p0, 48); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld3q_dup_u32(__p0) __extension__ ({ \
  uint32x4x3_t __ret; \
  __builtin_neon_vld3q_dup_v(&__ret, __p0, 50); \
  __ret; \
})
#else
#define vld3q_dup_u32(__p0) __extension__ ({ \
  uint32x4x3_t __ret; \
  __builtin_neon_vld3q_dup_v(&__ret, __p0, 50); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld3q_dup_u64(__p0) __extension__ ({ \
  uint64x2x3_t __ret; \
  __builtin_neon_vld3q_dup_v(&__ret, __p0, 51); \
  __ret; \
})
#else
#define vld3q_dup_u64(__p0) __extension__ ({ \
  uint64x2x3_t __ret; \
  __builtin_neon_vld3q_dup_v(&__ret, __p0, 51); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld3q_dup_u16(__p0) __extension__ ({ \
  uint16x8x3_t __ret; \
  __builtin_neon_vld3q_dup_v(&__ret, __p0, 49); \
  __ret; \
})
#else
#define vld3q_dup_u16(__p0) __extension__ ({ \
  uint16x8x3_t __ret; \
  __builtin_neon_vld3q_dup_v(&__ret, __p0, 49); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld3q_dup_s8(__p0) __extension__ ({ \
  int8x16x3_t __ret; \
  __builtin_neon_vld3q_dup_v(&__ret, __p0, 32); \
  __ret; \
})
#else
#define vld3q_dup_s8(__p0) __extension__ ({ \
  int8x16x3_t __ret; \
  __builtin_neon_vld3q_dup_v(&__ret, __p0, 32); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld3q_dup_f32(__p0) __extension__ ({ \
  float32x4x3_t __ret; \
  __builtin_neon_vld3q_dup_v(&__ret, __p0, 41); \
  __ret; \
})
#else
#define vld3q_dup_f32(__p0) __extension__ ({ \
  float32x4x3_t __ret; \
  __builtin_neon_vld3q_dup_v(&__ret, __p0, 41); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld3q_dup_s32(__p0) __extension__ ({ \
  int32x4x3_t __ret; \
  __builtin_neon_vld3q_dup_v(&__ret, __p0, 34); \
  __ret; \
})
#else
#define vld3q_dup_s32(__p0) __extension__ ({ \
  int32x4x3_t __ret; \
  __builtin_neon_vld3q_dup_v(&__ret, __p0, 34); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld3q_dup_s64(__p0) __extension__ ({ \
  int64x2x3_t __ret; \
  __builtin_neon_vld3q_dup_v(&__ret, __p0, 35); \
  __ret; \
})
#else
#define vld3q_dup_s64(__p0) __extension__ ({ \
  int64x2x3_t __ret; \
  __builtin_neon_vld3q_dup_v(&__ret, __p0, 35); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld3q_dup_s16(__p0) __extension__ ({ \
  int16x8x3_t __ret; \
  __builtin_neon_vld3q_dup_v(&__ret, __p0, 33); \
  __ret; \
})
#else
#define vld3q_dup_s16(__p0) __extension__ ({ \
  int16x8x3_t __ret; \
  __builtin_neon_vld3q_dup_v(&__ret, __p0, 33); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld3_dup_u8(__p0) __extension__ ({ \
  uint8x8x3_t __ret; \
  __builtin_neon_vld3_dup_v(&__ret, __p0, 16); \
  __ret; \
})
#else
#define vld3_dup_u8(__p0) __extension__ ({ \
  uint8x8x3_t __ret; \
  __builtin_neon_vld3_dup_v(&__ret, __p0, 16); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld3_dup_u32(__p0) __extension__ ({ \
  uint32x2x3_t __ret; \
  __builtin_neon_vld3_dup_v(&__ret, __p0, 18); \
  __ret; \
})
#else
#define vld3_dup_u32(__p0) __extension__ ({ \
  uint32x2x3_t __ret; \
  __builtin_neon_vld3_dup_v(&__ret, __p0, 18); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 1, 0); \
  __ret; \
})
#endif

#define vld3_dup_u64(__p0) __extension__ ({ \
  uint64x1x3_t __ret; \
  __builtin_neon_vld3_dup_v(&__ret, __p0, 19); \
  __ret; \
})
#ifdef __LITTLE_ENDIAN__
#define vld3_dup_u16(__p0) __extension__ ({ \
  uint16x4x3_t __ret; \
  __builtin_neon_vld3_dup_v(&__ret, __p0, 17); \
  __ret; \
})
#else
#define vld3_dup_u16(__p0) __extension__ ({ \
  uint16x4x3_t __ret; \
  __builtin_neon_vld3_dup_v(&__ret, __p0, 17); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld3_dup_s8(__p0) __extension__ ({ \
  int8x8x3_t __ret; \
  __builtin_neon_vld3_dup_v(&__ret, __p0, 0); \
  __ret; \
})
#else
#define vld3_dup_s8(__p0) __extension__ ({ \
  int8x8x3_t __ret; \
  __builtin_neon_vld3_dup_v(&__ret, __p0, 0); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld3_dup_f32(__p0) __extension__ ({ \
  float32x2x3_t __ret; \
  __builtin_neon_vld3_dup_v(&__ret, __p0, 9); \
  __ret; \
})
#else
#define vld3_dup_f32(__p0) __extension__ ({ \
  float32x2x3_t __ret; \
  __builtin_neon_vld3_dup_v(&__ret, __p0, 9); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld3_dup_s32(__p0) __extension__ ({ \
  int32x2x3_t __ret; \
  __builtin_neon_vld3_dup_v(&__ret, __p0, 2); \
  __ret; \
})
#else
#define vld3_dup_s32(__p0) __extension__ ({ \
  int32x2x3_t __ret; \
  __builtin_neon_vld3_dup_v(&__ret, __p0, 2); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 1, 0); \
  __ret; \
})
#endif

#define vld3_dup_s64(__p0) __extension__ ({ \
  int64x1x3_t __ret; \
  __builtin_neon_vld3_dup_v(&__ret, __p0, 3); \
  __ret; \
})
#ifdef __LITTLE_ENDIAN__
#define vld3_dup_s16(__p0) __extension__ ({ \
  int16x4x3_t __ret; \
  __builtin_neon_vld3_dup_v(&__ret, __p0, 1); \
  __ret; \
})
#else
#define vld3_dup_s16(__p0) __extension__ ({ \
  int16x4x3_t __ret; \
  __builtin_neon_vld3_dup_v(&__ret, __p0, 1); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld3_lane_p8(__p0, __p1, __p2) __extension__ ({ \
  poly8x8x3_t __s1 = __p1; \
  poly8x8x3_t __ret; \
  __builtin_neon_vld3_lane_v(&__ret, __p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], (int8x8_t)__s1.val[2], __p2, 4); \
  __ret; \
})
#else
#define vld3_lane_p8(__p0, __p1, __p2) __extension__ ({ \
  poly8x8x3_t __s1 = __p1; \
  poly8x8x3_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  poly8x8x3_t __ret; \
  __builtin_neon_vld3_lane_v(&__ret, __p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], (int8x8_t)__rev1.val[2], __p2, 4); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld3_lane_p16(__p0, __p1, __p2) __extension__ ({ \
  poly16x4x3_t __s1 = __p1; \
  poly16x4x3_t __ret; \
  __builtin_neon_vld3_lane_v(&__ret, __p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], (int8x8_t)__s1.val[2], __p2, 5); \
  __ret; \
})
#else
#define vld3_lane_p16(__p0, __p1, __p2) __extension__ ({ \
  poly16x4x3_t __s1 = __p1; \
  poly16x4x3_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 3, 2, 1, 0); \
  poly16x4x3_t __ret; \
  __builtin_neon_vld3_lane_v(&__ret, __p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], (int8x8_t)__rev1.val[2], __p2, 5); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld3q_lane_p16(__p0, __p1, __p2) __extension__ ({ \
  poly16x8x3_t __s1 = __p1; \
  poly16x8x3_t __ret; \
  __builtin_neon_vld3q_lane_v(&__ret, __p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], (int8x16_t)__s1.val[2], __p2, 37); \
  __ret; \
})
#else
#define vld3q_lane_p16(__p0, __p1, __p2) __extension__ ({ \
  poly16x8x3_t __s1 = __p1; \
  poly16x8x3_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  poly16x8x3_t __ret; \
  __builtin_neon_vld3q_lane_v(&__ret, __p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], (int8x16_t)__rev1.val[2], __p2, 37); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld3q_lane_u32(__p0, __p1, __p2) __extension__ ({ \
  uint32x4x3_t __s1 = __p1; \
  uint32x4x3_t __ret; \
  __builtin_neon_vld3q_lane_v(&__ret, __p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], (int8x16_t)__s1.val[2], __p2, 50); \
  __ret; \
})
#else
#define vld3q_lane_u32(__p0, __p1, __p2) __extension__ ({ \
  uint32x4x3_t __s1 = __p1; \
  uint32x4x3_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 3, 2, 1, 0); \
  uint32x4x3_t __ret; \
  __builtin_neon_vld3q_lane_v(&__ret, __p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], (int8x16_t)__rev1.val[2], __p2, 50); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld3q_lane_u16(__p0, __p1, __p2) __extension__ ({ \
  uint16x8x3_t __s1 = __p1; \
  uint16x8x3_t __ret; \
  __builtin_neon_vld3q_lane_v(&__ret, __p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], (int8x16_t)__s1.val[2], __p2, 49); \
  __ret; \
})
#else
#define vld3q_lane_u16(__p0, __p1, __p2) __extension__ ({ \
  uint16x8x3_t __s1 = __p1; \
  uint16x8x3_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  uint16x8x3_t __ret; \
  __builtin_neon_vld3q_lane_v(&__ret, __p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], (int8x16_t)__rev1.val[2], __p2, 49); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld3q_lane_f32(__p0, __p1, __p2) __extension__ ({ \
  float32x4x3_t __s1 = __p1; \
  float32x4x3_t __ret; \
  __builtin_neon_vld3q_lane_v(&__ret, __p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], (int8x16_t)__s1.val[2], __p2, 41); \
  __ret; \
})
#else
#define vld3q_lane_f32(__p0, __p1, __p2) __extension__ ({ \
  float32x4x3_t __s1 = __p1; \
  float32x4x3_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 3, 2, 1, 0); \
  float32x4x3_t __ret; \
  __builtin_neon_vld3q_lane_v(&__ret, __p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], (int8x16_t)__rev1.val[2], __p2, 41); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld3q_lane_s32(__p0, __p1, __p2) __extension__ ({ \
  int32x4x3_t __s1 = __p1; \
  int32x4x3_t __ret; \
  __builtin_neon_vld3q_lane_v(&__ret, __p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], (int8x16_t)__s1.val[2], __p2, 34); \
  __ret; \
})
#else
#define vld3q_lane_s32(__p0, __p1, __p2) __extension__ ({ \
  int32x4x3_t __s1 = __p1; \
  int32x4x3_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 3, 2, 1, 0); \
  int32x4x3_t __ret; \
  __builtin_neon_vld3q_lane_v(&__ret, __p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], (int8x16_t)__rev1.val[2], __p2, 34); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld3q_lane_s16(__p0, __p1, __p2) __extension__ ({ \
  int16x8x3_t __s1 = __p1; \
  int16x8x3_t __ret; \
  __builtin_neon_vld3q_lane_v(&__ret, __p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], (int8x16_t)__s1.val[2], __p2, 33); \
  __ret; \
})
#else
#define vld3q_lane_s16(__p0, __p1, __p2) __extension__ ({ \
  int16x8x3_t __s1 = __p1; \
  int16x8x3_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  int16x8x3_t __ret; \
  __builtin_neon_vld3q_lane_v(&__ret, __p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], (int8x16_t)__rev1.val[2], __p2, 33); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld3_lane_u8(__p0, __p1, __p2) __extension__ ({ \
  uint8x8x3_t __s1 = __p1; \
  uint8x8x3_t __ret; \
  __builtin_neon_vld3_lane_v(&__ret, __p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], (int8x8_t)__s1.val[2], __p2, 16); \
  __ret; \
})
#else
#define vld3_lane_u8(__p0, __p1, __p2) __extension__ ({ \
  uint8x8x3_t __s1 = __p1; \
  uint8x8x3_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  uint8x8x3_t __ret; \
  __builtin_neon_vld3_lane_v(&__ret, __p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], (int8x8_t)__rev1.val[2], __p2, 16); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld3_lane_u32(__p0, __p1, __p2) __extension__ ({ \
  uint32x2x3_t __s1 = __p1; \
  uint32x2x3_t __ret; \
  __builtin_neon_vld3_lane_v(&__ret, __p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], (int8x8_t)__s1.val[2], __p2, 18); \
  __ret; \
})
#else
#define vld3_lane_u32(__p0, __p1, __p2) __extension__ ({ \
  uint32x2x3_t __s1 = __p1; \
  uint32x2x3_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 1, 0); \
  uint32x2x3_t __ret; \
  __builtin_neon_vld3_lane_v(&__ret, __p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], (int8x8_t)__rev1.val[2], __p2, 18); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld3_lane_u16(__p0, __p1, __p2) __extension__ ({ \
  uint16x4x3_t __s1 = __p1; \
  uint16x4x3_t __ret; \
  __builtin_neon_vld3_lane_v(&__ret, __p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], (int8x8_t)__s1.val[2], __p2, 17); \
  __ret; \
})
#else
#define vld3_lane_u16(__p0, __p1, __p2) __extension__ ({ \
  uint16x4x3_t __s1 = __p1; \
  uint16x4x3_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 3, 2, 1, 0); \
  uint16x4x3_t __ret; \
  __builtin_neon_vld3_lane_v(&__ret, __p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], (int8x8_t)__rev1.val[2], __p2, 17); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld3_lane_s8(__p0, __p1, __p2) __extension__ ({ \
  int8x8x3_t __s1 = __p1; \
  int8x8x3_t __ret; \
  __builtin_neon_vld3_lane_v(&__ret, __p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], (int8x8_t)__s1.val[2], __p2, 0); \
  __ret; \
})
#else
#define vld3_lane_s8(__p0, __p1, __p2) __extension__ ({ \
  int8x8x3_t __s1 = __p1; \
  int8x8x3_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  int8x8x3_t __ret; \
  __builtin_neon_vld3_lane_v(&__ret, __p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], (int8x8_t)__rev1.val[2], __p2, 0); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld3_lane_f32(__p0, __p1, __p2) __extension__ ({ \
  float32x2x3_t __s1 = __p1; \
  float32x2x3_t __ret; \
  __builtin_neon_vld3_lane_v(&__ret, __p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], (int8x8_t)__s1.val[2], __p2, 9); \
  __ret; \
})
#else
#define vld3_lane_f32(__p0, __p1, __p2) __extension__ ({ \
  float32x2x3_t __s1 = __p1; \
  float32x2x3_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 1, 0); \
  float32x2x3_t __ret; \
  __builtin_neon_vld3_lane_v(&__ret, __p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], (int8x8_t)__rev1.val[2], __p2, 9); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld3_lane_s32(__p0, __p1, __p2) __extension__ ({ \
  int32x2x3_t __s1 = __p1; \
  int32x2x3_t __ret; \
  __builtin_neon_vld3_lane_v(&__ret, __p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], (int8x8_t)__s1.val[2], __p2, 2); \
  __ret; \
})
#else
#define vld3_lane_s32(__p0, __p1, __p2) __extension__ ({ \
  int32x2x3_t __s1 = __p1; \
  int32x2x3_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 1, 0); \
  int32x2x3_t __ret; \
  __builtin_neon_vld3_lane_v(&__ret, __p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], (int8x8_t)__rev1.val[2], __p2, 2); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld3_lane_s16(__p0, __p1, __p2) __extension__ ({ \
  int16x4x3_t __s1 = __p1; \
  int16x4x3_t __ret; \
  __builtin_neon_vld3_lane_v(&__ret, __p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], (int8x8_t)__s1.val[2], __p2, 1); \
  __ret; \
})
#else
#define vld3_lane_s16(__p0, __p1, __p2) __extension__ ({ \
  int16x4x3_t __s1 = __p1; \
  int16x4x3_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 3, 2, 1, 0); \
  int16x4x3_t __ret; \
  __builtin_neon_vld3_lane_v(&__ret, __p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], (int8x8_t)__rev1.val[2], __p2, 1); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld4_p8(__p0) __extension__ ({ \
  poly8x8x4_t __ret; \
  __builtin_neon_vld4_v(&__ret, __p0, 4); \
  __ret; \
})
#else
#define vld4_p8(__p0) __extension__ ({ \
  poly8x8x4_t __ret; \
  __builtin_neon_vld4_v(&__ret, __p0, 4); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld4_p16(__p0) __extension__ ({ \
  poly16x4x4_t __ret; \
  __builtin_neon_vld4_v(&__ret, __p0, 5); \
  __ret; \
})
#else
#define vld4_p16(__p0) __extension__ ({ \
  poly16x4x4_t __ret; \
  __builtin_neon_vld4_v(&__ret, __p0, 5); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld4q_p8(__p0) __extension__ ({ \
  poly8x16x4_t __ret; \
  __builtin_neon_vld4q_v(&__ret, __p0, 36); \
  __ret; \
})
#else
#define vld4q_p8(__p0) __extension__ ({ \
  poly8x16x4_t __ret; \
  __builtin_neon_vld4q_v(&__ret, __p0, 36); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld4q_p16(__p0) __extension__ ({ \
  poly16x8x4_t __ret; \
  __builtin_neon_vld4q_v(&__ret, __p0, 37); \
  __ret; \
})
#else
#define vld4q_p16(__p0) __extension__ ({ \
  poly16x8x4_t __ret; \
  __builtin_neon_vld4q_v(&__ret, __p0, 37); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld4q_u8(__p0) __extension__ ({ \
  uint8x16x4_t __ret; \
  __builtin_neon_vld4q_v(&__ret, __p0, 48); \
  __ret; \
})
#else
#define vld4q_u8(__p0) __extension__ ({ \
  uint8x16x4_t __ret; \
  __builtin_neon_vld4q_v(&__ret, __p0, 48); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld4q_u32(__p0) __extension__ ({ \
  uint32x4x4_t __ret; \
  __builtin_neon_vld4q_v(&__ret, __p0, 50); \
  __ret; \
})
#else
#define vld4q_u32(__p0) __extension__ ({ \
  uint32x4x4_t __ret; \
  __builtin_neon_vld4q_v(&__ret, __p0, 50); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld4q_u16(__p0) __extension__ ({ \
  uint16x8x4_t __ret; \
  __builtin_neon_vld4q_v(&__ret, __p0, 49); \
  __ret; \
})
#else
#define vld4q_u16(__p0) __extension__ ({ \
  uint16x8x4_t __ret; \
  __builtin_neon_vld4q_v(&__ret, __p0, 49); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld4q_s8(__p0) __extension__ ({ \
  int8x16x4_t __ret; \
  __builtin_neon_vld4q_v(&__ret, __p0, 32); \
  __ret; \
})
#else
#define vld4q_s8(__p0) __extension__ ({ \
  int8x16x4_t __ret; \
  __builtin_neon_vld4q_v(&__ret, __p0, 32); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld4q_f32(__p0) __extension__ ({ \
  float32x4x4_t __ret; \
  __builtin_neon_vld4q_v(&__ret, __p0, 41); \
  __ret; \
})
#else
#define vld4q_f32(__p0) __extension__ ({ \
  float32x4x4_t __ret; \
  __builtin_neon_vld4q_v(&__ret, __p0, 41); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld4q_s32(__p0) __extension__ ({ \
  int32x4x4_t __ret; \
  __builtin_neon_vld4q_v(&__ret, __p0, 34); \
  __ret; \
})
#else
#define vld4q_s32(__p0) __extension__ ({ \
  int32x4x4_t __ret; \
  __builtin_neon_vld4q_v(&__ret, __p0, 34); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld4q_s16(__p0) __extension__ ({ \
  int16x8x4_t __ret; \
  __builtin_neon_vld4q_v(&__ret, __p0, 33); \
  __ret; \
})
#else
#define vld4q_s16(__p0) __extension__ ({ \
  int16x8x4_t __ret; \
  __builtin_neon_vld4q_v(&__ret, __p0, 33); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld4_u8(__p0) __extension__ ({ \
  uint8x8x4_t __ret; \
  __builtin_neon_vld4_v(&__ret, __p0, 16); \
  __ret; \
})
#else
#define vld4_u8(__p0) __extension__ ({ \
  uint8x8x4_t __ret; \
  __builtin_neon_vld4_v(&__ret, __p0, 16); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld4_u32(__p0) __extension__ ({ \
  uint32x2x4_t __ret; \
  __builtin_neon_vld4_v(&__ret, __p0, 18); \
  __ret; \
})
#else
#define vld4_u32(__p0) __extension__ ({ \
  uint32x2x4_t __ret; \
  __builtin_neon_vld4_v(&__ret, __p0, 18); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 1, 0); \
  __ret; \
})
#endif

#define vld4_u64(__p0) __extension__ ({ \
  uint64x1x4_t __ret; \
  __builtin_neon_vld4_v(&__ret, __p0, 19); \
  __ret; \
})
#ifdef __LITTLE_ENDIAN__
#define vld4_u16(__p0) __extension__ ({ \
  uint16x4x4_t __ret; \
  __builtin_neon_vld4_v(&__ret, __p0, 17); \
  __ret; \
})
#else
#define vld4_u16(__p0) __extension__ ({ \
  uint16x4x4_t __ret; \
  __builtin_neon_vld4_v(&__ret, __p0, 17); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld4_s8(__p0) __extension__ ({ \
  int8x8x4_t __ret; \
  __builtin_neon_vld4_v(&__ret, __p0, 0); \
  __ret; \
})
#else
#define vld4_s8(__p0) __extension__ ({ \
  int8x8x4_t __ret; \
  __builtin_neon_vld4_v(&__ret, __p0, 0); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld4_f32(__p0) __extension__ ({ \
  float32x2x4_t __ret; \
  __builtin_neon_vld4_v(&__ret, __p0, 9); \
  __ret; \
})
#else
#define vld4_f32(__p0) __extension__ ({ \
  float32x2x4_t __ret; \
  __builtin_neon_vld4_v(&__ret, __p0, 9); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld4_s32(__p0) __extension__ ({ \
  int32x2x4_t __ret; \
  __builtin_neon_vld4_v(&__ret, __p0, 2); \
  __ret; \
})
#else
#define vld4_s32(__p0) __extension__ ({ \
  int32x2x4_t __ret; \
  __builtin_neon_vld4_v(&__ret, __p0, 2); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 1, 0); \
  __ret; \
})
#endif

#define vld4_s64(__p0) __extension__ ({ \
  int64x1x4_t __ret; \
  __builtin_neon_vld4_v(&__ret, __p0, 3); \
  __ret; \
})
#ifdef __LITTLE_ENDIAN__
#define vld4_s16(__p0) __extension__ ({ \
  int16x4x4_t __ret; \
  __builtin_neon_vld4_v(&__ret, __p0, 1); \
  __ret; \
})
#else
#define vld4_s16(__p0) __extension__ ({ \
  int16x4x4_t __ret; \
  __builtin_neon_vld4_v(&__ret, __p0, 1); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld4_dup_p8(__p0) __extension__ ({ \
  poly8x8x4_t __ret; \
  __builtin_neon_vld4_dup_v(&__ret, __p0, 4); \
  __ret; \
})
#else
#define vld4_dup_p8(__p0) __extension__ ({ \
  poly8x8x4_t __ret; \
  __builtin_neon_vld4_dup_v(&__ret, __p0, 4); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld4_dup_p16(__p0) __extension__ ({ \
  poly16x4x4_t __ret; \
  __builtin_neon_vld4_dup_v(&__ret, __p0, 5); \
  __ret; \
})
#else
#define vld4_dup_p16(__p0) __extension__ ({ \
  poly16x4x4_t __ret; \
  __builtin_neon_vld4_dup_v(&__ret, __p0, 5); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld4q_dup_p8(__p0) __extension__ ({ \
  poly8x16x4_t __ret; \
  __builtin_neon_vld4q_dup_v(&__ret, __p0, 36); \
  __ret; \
})
#else
#define vld4q_dup_p8(__p0) __extension__ ({ \
  poly8x16x4_t __ret; \
  __builtin_neon_vld4q_dup_v(&__ret, __p0, 36); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld4q_dup_p16(__p0) __extension__ ({ \
  poly16x8x4_t __ret; \
  __builtin_neon_vld4q_dup_v(&__ret, __p0, 37); \
  __ret; \
})
#else
#define vld4q_dup_p16(__p0) __extension__ ({ \
  poly16x8x4_t __ret; \
  __builtin_neon_vld4q_dup_v(&__ret, __p0, 37); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld4q_dup_u8(__p0) __extension__ ({ \
  uint8x16x4_t __ret; \
  __builtin_neon_vld4q_dup_v(&__ret, __p0, 48); \
  __ret; \
})
#else
#define vld4q_dup_u8(__p0) __extension__ ({ \
  uint8x16x4_t __ret; \
  __builtin_neon_vld4q_dup_v(&__ret, __p0, 48); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld4q_dup_u32(__p0) __extension__ ({ \
  uint32x4x4_t __ret; \
  __builtin_neon_vld4q_dup_v(&__ret, __p0, 50); \
  __ret; \
})
#else
#define vld4q_dup_u32(__p0) __extension__ ({ \
  uint32x4x4_t __ret; \
  __builtin_neon_vld4q_dup_v(&__ret, __p0, 50); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld4q_dup_u64(__p0) __extension__ ({ \
  uint64x2x4_t __ret; \
  __builtin_neon_vld4q_dup_v(&__ret, __p0, 51); \
  __ret; \
})
#else
#define vld4q_dup_u64(__p0) __extension__ ({ \
  uint64x2x4_t __ret; \
  __builtin_neon_vld4q_dup_v(&__ret, __p0, 51); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld4q_dup_u16(__p0) __extension__ ({ \
  uint16x8x4_t __ret; \
  __builtin_neon_vld4q_dup_v(&__ret, __p0, 49); \
  __ret; \
})
#else
#define vld4q_dup_u16(__p0) __extension__ ({ \
  uint16x8x4_t __ret; \
  __builtin_neon_vld4q_dup_v(&__ret, __p0, 49); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld4q_dup_s8(__p0) __extension__ ({ \
  int8x16x4_t __ret; \
  __builtin_neon_vld4q_dup_v(&__ret, __p0, 32); \
  __ret; \
})
#else
#define vld4q_dup_s8(__p0) __extension__ ({ \
  int8x16x4_t __ret; \
  __builtin_neon_vld4q_dup_v(&__ret, __p0, 32); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld4q_dup_f32(__p0) __extension__ ({ \
  float32x4x4_t __ret; \
  __builtin_neon_vld4q_dup_v(&__ret, __p0, 41); \
  __ret; \
})
#else
#define vld4q_dup_f32(__p0) __extension__ ({ \
  float32x4x4_t __ret; \
  __builtin_neon_vld4q_dup_v(&__ret, __p0, 41); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld4q_dup_s32(__p0) __extension__ ({ \
  int32x4x4_t __ret; \
  __builtin_neon_vld4q_dup_v(&__ret, __p0, 34); \
  __ret; \
})
#else
#define vld4q_dup_s32(__p0) __extension__ ({ \
  int32x4x4_t __ret; \
  __builtin_neon_vld4q_dup_v(&__ret, __p0, 34); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld4q_dup_s64(__p0) __extension__ ({ \
  int64x2x4_t __ret; \
  __builtin_neon_vld4q_dup_v(&__ret, __p0, 35); \
  __ret; \
})
#else
#define vld4q_dup_s64(__p0) __extension__ ({ \
  int64x2x4_t __ret; \
  __builtin_neon_vld4q_dup_v(&__ret, __p0, 35); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld4q_dup_s16(__p0) __extension__ ({ \
  int16x8x4_t __ret; \
  __builtin_neon_vld4q_dup_v(&__ret, __p0, 33); \
  __ret; \
})
#else
#define vld4q_dup_s16(__p0) __extension__ ({ \
  int16x8x4_t __ret; \
  __builtin_neon_vld4q_dup_v(&__ret, __p0, 33); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld4_dup_u8(__p0) __extension__ ({ \
  uint8x8x4_t __ret; \
  __builtin_neon_vld4_dup_v(&__ret, __p0, 16); \
  __ret; \
})
#else
#define vld4_dup_u8(__p0) __extension__ ({ \
  uint8x8x4_t __ret; \
  __builtin_neon_vld4_dup_v(&__ret, __p0, 16); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld4_dup_u32(__p0) __extension__ ({ \
  uint32x2x4_t __ret; \
  __builtin_neon_vld4_dup_v(&__ret, __p0, 18); \
  __ret; \
})
#else
#define vld4_dup_u32(__p0) __extension__ ({ \
  uint32x2x4_t __ret; \
  __builtin_neon_vld4_dup_v(&__ret, __p0, 18); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 1, 0); \
  __ret; \
})
#endif

#define vld4_dup_u64(__p0) __extension__ ({ \
  uint64x1x4_t __ret; \
  __builtin_neon_vld4_dup_v(&__ret, __p0, 19); \
  __ret; \
})
#ifdef __LITTLE_ENDIAN__
#define vld4_dup_u16(__p0) __extension__ ({ \
  uint16x4x4_t __ret; \
  __builtin_neon_vld4_dup_v(&__ret, __p0, 17); \
  __ret; \
})
#else
#define vld4_dup_u16(__p0) __extension__ ({ \
  uint16x4x4_t __ret; \
  __builtin_neon_vld4_dup_v(&__ret, __p0, 17); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld4_dup_s8(__p0) __extension__ ({ \
  int8x8x4_t __ret; \
  __builtin_neon_vld4_dup_v(&__ret, __p0, 0); \
  __ret; \
})
#else
#define vld4_dup_s8(__p0) __extension__ ({ \
  int8x8x4_t __ret; \
  __builtin_neon_vld4_dup_v(&__ret, __p0, 0); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld4_dup_f32(__p0) __extension__ ({ \
  float32x2x4_t __ret; \
  __builtin_neon_vld4_dup_v(&__ret, __p0, 9); \
  __ret; \
})
#else
#define vld4_dup_f32(__p0) __extension__ ({ \
  float32x2x4_t __ret; \
  __builtin_neon_vld4_dup_v(&__ret, __p0, 9); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld4_dup_s32(__p0) __extension__ ({ \
  int32x2x4_t __ret; \
  __builtin_neon_vld4_dup_v(&__ret, __p0, 2); \
  __ret; \
})
#else
#define vld4_dup_s32(__p0) __extension__ ({ \
  int32x2x4_t __ret; \
  __builtin_neon_vld4_dup_v(&__ret, __p0, 2); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 1, 0); \
  __ret; \
})
#endif

#define vld4_dup_s64(__p0) __extension__ ({ \
  int64x1x4_t __ret; \
  __builtin_neon_vld4_dup_v(&__ret, __p0, 3); \
  __ret; \
})
#ifdef __LITTLE_ENDIAN__
#define vld4_dup_s16(__p0) __extension__ ({ \
  int16x4x4_t __ret; \
  __builtin_neon_vld4_dup_v(&__ret, __p0, 1); \
  __ret; \
})
#else
#define vld4_dup_s16(__p0) __extension__ ({ \
  int16x4x4_t __ret; \
  __builtin_neon_vld4_dup_v(&__ret, __p0, 1); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld4_lane_p8(__p0, __p1, __p2) __extension__ ({ \
  poly8x8x4_t __s1 = __p1; \
  poly8x8x4_t __ret; \
  __builtin_neon_vld4_lane_v(&__ret, __p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], (int8x8_t)__s1.val[2], (int8x8_t)__s1.val[3], __p2, 4); \
  __ret; \
})
#else
#define vld4_lane_p8(__p0, __p1, __p2) __extension__ ({ \
  poly8x8x4_t __s1 = __p1; \
  poly8x8x4_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[3] = __builtin_shufflevector(__s1.val[3], __s1.val[3], 7, 6, 5, 4, 3, 2, 1, 0); \
  poly8x8x4_t __ret; \
  __builtin_neon_vld4_lane_v(&__ret, __p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], (int8x8_t)__rev1.val[2], (int8x8_t)__rev1.val[3], __p2, 4); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld4_lane_p16(__p0, __p1, __p2) __extension__ ({ \
  poly16x4x4_t __s1 = __p1; \
  poly16x4x4_t __ret; \
  __builtin_neon_vld4_lane_v(&__ret, __p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], (int8x8_t)__s1.val[2], (int8x8_t)__s1.val[3], __p2, 5); \
  __ret; \
})
#else
#define vld4_lane_p16(__p0, __p1, __p2) __extension__ ({ \
  poly16x4x4_t __s1 = __p1; \
  poly16x4x4_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 3, 2, 1, 0); \
  __rev1.val[3] = __builtin_shufflevector(__s1.val[3], __s1.val[3], 3, 2, 1, 0); \
  poly16x4x4_t __ret; \
  __builtin_neon_vld4_lane_v(&__ret, __p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], (int8x8_t)__rev1.val[2], (int8x8_t)__rev1.val[3], __p2, 5); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld4q_lane_p16(__p0, __p1, __p2) __extension__ ({ \
  poly16x8x4_t __s1 = __p1; \
  poly16x8x4_t __ret; \
  __builtin_neon_vld4q_lane_v(&__ret, __p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], (int8x16_t)__s1.val[2], (int8x16_t)__s1.val[3], __p2, 37); \
  __ret; \
})
#else
#define vld4q_lane_p16(__p0, __p1, __p2) __extension__ ({ \
  poly16x8x4_t __s1 = __p1; \
  poly16x8x4_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[3] = __builtin_shufflevector(__s1.val[3], __s1.val[3], 7, 6, 5, 4, 3, 2, 1, 0); \
  poly16x8x4_t __ret; \
  __builtin_neon_vld4q_lane_v(&__ret, __p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], (int8x16_t)__rev1.val[2], (int8x16_t)__rev1.val[3], __p2, 37); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld4q_lane_u32(__p0, __p1, __p2) __extension__ ({ \
  uint32x4x4_t __s1 = __p1; \
  uint32x4x4_t __ret; \
  __builtin_neon_vld4q_lane_v(&__ret, __p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], (int8x16_t)__s1.val[2], (int8x16_t)__s1.val[3], __p2, 50); \
  __ret; \
})
#else
#define vld4q_lane_u32(__p0, __p1, __p2) __extension__ ({ \
  uint32x4x4_t __s1 = __p1; \
  uint32x4x4_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 3, 2, 1, 0); \
  __rev1.val[3] = __builtin_shufflevector(__s1.val[3], __s1.val[3], 3, 2, 1, 0); \
  uint32x4x4_t __ret; \
  __builtin_neon_vld4q_lane_v(&__ret, __p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], (int8x16_t)__rev1.val[2], (int8x16_t)__rev1.val[3], __p2, 50); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld4q_lane_u16(__p0, __p1, __p2) __extension__ ({ \
  uint16x8x4_t __s1 = __p1; \
  uint16x8x4_t __ret; \
  __builtin_neon_vld4q_lane_v(&__ret, __p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], (int8x16_t)__s1.val[2], (int8x16_t)__s1.val[3], __p2, 49); \
  __ret; \
})
#else
#define vld4q_lane_u16(__p0, __p1, __p2) __extension__ ({ \
  uint16x8x4_t __s1 = __p1; \
  uint16x8x4_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[3] = __builtin_shufflevector(__s1.val[3], __s1.val[3], 7, 6, 5, 4, 3, 2, 1, 0); \
  uint16x8x4_t __ret; \
  __builtin_neon_vld4q_lane_v(&__ret, __p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], (int8x16_t)__rev1.val[2], (int8x16_t)__rev1.val[3], __p2, 49); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld4q_lane_f32(__p0, __p1, __p2) __extension__ ({ \
  float32x4x4_t __s1 = __p1; \
  float32x4x4_t __ret; \
  __builtin_neon_vld4q_lane_v(&__ret, __p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], (int8x16_t)__s1.val[2], (int8x16_t)__s1.val[3], __p2, 41); \
  __ret; \
})
#else
#define vld4q_lane_f32(__p0, __p1, __p2) __extension__ ({ \
  float32x4x4_t __s1 = __p1; \
  float32x4x4_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 3, 2, 1, 0); \
  __rev1.val[3] = __builtin_shufflevector(__s1.val[3], __s1.val[3], 3, 2, 1, 0); \
  float32x4x4_t __ret; \
  __builtin_neon_vld4q_lane_v(&__ret, __p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], (int8x16_t)__rev1.val[2], (int8x16_t)__rev1.val[3], __p2, 41); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld4q_lane_s32(__p0, __p1, __p2) __extension__ ({ \
  int32x4x4_t __s1 = __p1; \
  int32x4x4_t __ret; \
  __builtin_neon_vld4q_lane_v(&__ret, __p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], (int8x16_t)__s1.val[2], (int8x16_t)__s1.val[3], __p2, 34); \
  __ret; \
})
#else
#define vld4q_lane_s32(__p0, __p1, __p2) __extension__ ({ \
  int32x4x4_t __s1 = __p1; \
  int32x4x4_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 3, 2, 1, 0); \
  __rev1.val[3] = __builtin_shufflevector(__s1.val[3], __s1.val[3], 3, 2, 1, 0); \
  int32x4x4_t __ret; \
  __builtin_neon_vld4q_lane_v(&__ret, __p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], (int8x16_t)__rev1.val[2], (int8x16_t)__rev1.val[3], __p2, 34); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld4q_lane_s16(__p0, __p1, __p2) __extension__ ({ \
  int16x8x4_t __s1 = __p1; \
  int16x8x4_t __ret; \
  __builtin_neon_vld4q_lane_v(&__ret, __p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], (int8x16_t)__s1.val[2], (int8x16_t)__s1.val[3], __p2, 33); \
  __ret; \
})
#else
#define vld4q_lane_s16(__p0, __p1, __p2) __extension__ ({ \
  int16x8x4_t __s1 = __p1; \
  int16x8x4_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[3] = __builtin_shufflevector(__s1.val[3], __s1.val[3], 7, 6, 5, 4, 3, 2, 1, 0); \
  int16x8x4_t __ret; \
  __builtin_neon_vld4q_lane_v(&__ret, __p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], (int8x16_t)__rev1.val[2], (int8x16_t)__rev1.val[3], __p2, 33); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld4_lane_u8(__p0, __p1, __p2) __extension__ ({ \
  uint8x8x4_t __s1 = __p1; \
  uint8x8x4_t __ret; \
  __builtin_neon_vld4_lane_v(&__ret, __p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], (int8x8_t)__s1.val[2], (int8x8_t)__s1.val[3], __p2, 16); \
  __ret; \
})
#else
#define vld4_lane_u8(__p0, __p1, __p2) __extension__ ({ \
  uint8x8x4_t __s1 = __p1; \
  uint8x8x4_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[3] = __builtin_shufflevector(__s1.val[3], __s1.val[3], 7, 6, 5, 4, 3, 2, 1, 0); \
  uint8x8x4_t __ret; \
  __builtin_neon_vld4_lane_v(&__ret, __p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], (int8x8_t)__rev1.val[2], (int8x8_t)__rev1.val[3], __p2, 16); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld4_lane_u32(__p0, __p1, __p2) __extension__ ({ \
  uint32x2x4_t __s1 = __p1; \
  uint32x2x4_t __ret; \
  __builtin_neon_vld4_lane_v(&__ret, __p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], (int8x8_t)__s1.val[2], (int8x8_t)__s1.val[3], __p2, 18); \
  __ret; \
})
#else
#define vld4_lane_u32(__p0, __p1, __p2) __extension__ ({ \
  uint32x2x4_t __s1 = __p1; \
  uint32x2x4_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 1, 0); \
  __rev1.val[3] = __builtin_shufflevector(__s1.val[3], __s1.val[3], 1, 0); \
  uint32x2x4_t __ret; \
  __builtin_neon_vld4_lane_v(&__ret, __p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], (int8x8_t)__rev1.val[2], (int8x8_t)__rev1.val[3], __p2, 18); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld4_lane_u16(__p0, __p1, __p2) __extension__ ({ \
  uint16x4x4_t __s1 = __p1; \
  uint16x4x4_t __ret; \
  __builtin_neon_vld4_lane_v(&__ret, __p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], (int8x8_t)__s1.val[2], (int8x8_t)__s1.val[3], __p2, 17); \
  __ret; \
})
#else
#define vld4_lane_u16(__p0, __p1, __p2) __extension__ ({ \
  uint16x4x4_t __s1 = __p1; \
  uint16x4x4_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 3, 2, 1, 0); \
  __rev1.val[3] = __builtin_shufflevector(__s1.val[3], __s1.val[3], 3, 2, 1, 0); \
  uint16x4x4_t __ret; \
  __builtin_neon_vld4_lane_v(&__ret, __p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], (int8x8_t)__rev1.val[2], (int8x8_t)__rev1.val[3], __p2, 17); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld4_lane_s8(__p0, __p1, __p2) __extension__ ({ \
  int8x8x4_t __s1 = __p1; \
  int8x8x4_t __ret; \
  __builtin_neon_vld4_lane_v(&__ret, __p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], (int8x8_t)__s1.val[2], (int8x8_t)__s1.val[3], __p2, 0); \
  __ret; \
})
#else
#define vld4_lane_s8(__p0, __p1, __p2) __extension__ ({ \
  int8x8x4_t __s1 = __p1; \
  int8x8x4_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[3] = __builtin_shufflevector(__s1.val[3], __s1.val[3], 7, 6, 5, 4, 3, 2, 1, 0); \
  int8x8x4_t __ret; \
  __builtin_neon_vld4_lane_v(&__ret, __p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], (int8x8_t)__rev1.val[2], (int8x8_t)__rev1.val[3], __p2, 0); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld4_lane_f32(__p0, __p1, __p2) __extension__ ({ \
  float32x2x4_t __s1 = __p1; \
  float32x2x4_t __ret; \
  __builtin_neon_vld4_lane_v(&__ret, __p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], (int8x8_t)__s1.val[2], (int8x8_t)__s1.val[3], __p2, 9); \
  __ret; \
})
#else
#define vld4_lane_f32(__p0, __p1, __p2) __extension__ ({ \
  float32x2x4_t __s1 = __p1; \
  float32x2x4_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 1, 0); \
  __rev1.val[3] = __builtin_shufflevector(__s1.val[3], __s1.val[3], 1, 0); \
  float32x2x4_t __ret; \
  __builtin_neon_vld4_lane_v(&__ret, __p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], (int8x8_t)__rev1.val[2], (int8x8_t)__rev1.val[3], __p2, 9); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld4_lane_s32(__p0, __p1, __p2) __extension__ ({ \
  int32x2x4_t __s1 = __p1; \
  int32x2x4_t __ret; \
  __builtin_neon_vld4_lane_v(&__ret, __p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], (int8x8_t)__s1.val[2], (int8x8_t)__s1.val[3], __p2, 2); \
  __ret; \
})
#else
#define vld4_lane_s32(__p0, __p1, __p2) __extension__ ({ \
  int32x2x4_t __s1 = __p1; \
  int32x2x4_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 1, 0); \
  __rev1.val[3] = __builtin_shufflevector(__s1.val[3], __s1.val[3], 1, 0); \
  int32x2x4_t __ret; \
  __builtin_neon_vld4_lane_v(&__ret, __p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], (int8x8_t)__rev1.val[2], (int8x8_t)__rev1.val[3], __p2, 2); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vld4_lane_s16(__p0, __p1, __p2) __extension__ ({ \
  int16x4x4_t __s1 = __p1; \
  int16x4x4_t __ret; \
  __builtin_neon_vld4_lane_v(&__ret, __p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], (int8x8_t)__s1.val[2], (int8x8_t)__s1.val[3], __p2, 1); \
  __ret; \
})
#else
#define vld4_lane_s16(__p0, __p1, __p2) __extension__ ({ \
  int16x4x4_t __s1 = __p1; \
  int16x4x4_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 3, 2, 1, 0); \
  __rev1.val[3] = __builtin_shufflevector(__s1.val[3], __s1.val[3], 3, 2, 1, 0); \
  int16x4x4_t __ret; \
  __builtin_neon_vld4_lane_v(&__ret, __p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], (int8x8_t)__rev1.val[2], (int8x8_t)__rev1.val[3], __p2, 1); \
 \
  __ret.val[0] = __builtin_shufflevector(__ret.val[0], __ret.val[0], 3, 2, 1, 0); \
  __ret.val[1] = __builtin_shufflevector(__ret.val[1], __ret.val[1], 3, 2, 1, 0); \
  __ret.val[2] = __builtin_shufflevector(__ret.val[2], __ret.val[2], 3, 2, 1, 0); \
  __ret.val[3] = __builtin_shufflevector(__ret.val[3], __ret.val[3], 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x16_t vmaxq_u8(uint8x16_t __p0, uint8x16_t __p1) {
  uint8x16_t __ret;
  __ret = (uint8x16_t) __builtin_neon_vmaxq_v((int8x16_t)__p0, (int8x16_t)__p1, 48);
  return __ret;
}
#else
__ai uint8x16_t vmaxq_u8(uint8x16_t __p0, uint8x16_t __p1) {
  uint8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __ret;
  __ret = (uint8x16_t) __builtin_neon_vmaxq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 48);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vmaxq_u32(uint32x4_t __p0, uint32x4_t __p1) {
  uint32x4_t __ret;
  __ret = (uint32x4_t) __builtin_neon_vmaxq_v((int8x16_t)__p0, (int8x16_t)__p1, 50);
  return __ret;
}
#else
__ai uint32x4_t vmaxq_u32(uint32x4_t __p0, uint32x4_t __p1) {
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = (uint32x4_t) __builtin_neon_vmaxq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 50);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x8_t vmaxq_u16(uint16x8_t __p0, uint16x8_t __p1) {
  uint16x8_t __ret;
  __ret = (uint16x8_t) __builtin_neon_vmaxq_v((int8x16_t)__p0, (int8x16_t)__p1, 49);
  return __ret;
}
#else
__ai uint16x8_t vmaxq_u16(uint16x8_t __p0, uint16x8_t __p1) {
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __ret;
  __ret = (uint16x8_t) __builtin_neon_vmaxq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 49);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x16_t vmaxq_s8(int8x16_t __p0, int8x16_t __p1) {
  int8x16_t __ret;
  __ret = (int8x16_t) __builtin_neon_vmaxq_v((int8x16_t)__p0, (int8x16_t)__p1, 32);
  return __ret;
}
#else
__ai int8x16_t vmaxq_s8(int8x16_t __p0, int8x16_t __p1) {
  int8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __ret;
  __ret = (int8x16_t) __builtin_neon_vmaxq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 32);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai float32x4_t vmaxq_f32(float32x4_t __p0, float32x4_t __p1) {
  float32x4_t __ret;
  __ret = (float32x4_t) __builtin_neon_vmaxq_v((int8x16_t)__p0, (int8x16_t)__p1, 41);
  return __ret;
}
#else
__ai float32x4_t vmaxq_f32(float32x4_t __p0, float32x4_t __p1) {
  float32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  float32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  float32x4_t __ret;
  __ret = (float32x4_t) __builtin_neon_vmaxq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 41);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x4_t vmaxq_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vmaxq_v((int8x16_t)__p0, (int8x16_t)__p1, 34);
  return __ret;
}
#else
__ai int32x4_t vmaxq_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vmaxq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 34);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x8_t vmaxq_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vmaxq_v((int8x16_t)__p0, (int8x16_t)__p1, 33);
  return __ret;
}
#else
__ai int16x8_t vmaxq_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vmaxq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 33);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x8_t vmax_u8(uint8x8_t __p0, uint8x8_t __p1) {
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vmax_v((int8x8_t)__p0, (int8x8_t)__p1, 16);
  return __ret;
}
#else
__ai uint8x8_t vmax_u8(uint8x8_t __p0, uint8x8_t __p1) {
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vmax_v((int8x8_t)__rev0, (int8x8_t)__rev1, 16);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vmax_u32(uint32x2_t __p0, uint32x2_t __p1) {
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vmax_v((int8x8_t)__p0, (int8x8_t)__p1, 18);
  return __ret;
}
#else
__ai uint32x2_t vmax_u32(uint32x2_t __p0, uint32x2_t __p1) {
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vmax_v((int8x8_t)__rev0, (int8x8_t)__rev1, 18);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x4_t vmax_u16(uint16x4_t __p0, uint16x4_t __p1) {
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vmax_v((int8x8_t)__p0, (int8x8_t)__p1, 17);
  return __ret;
}
#else
__ai uint16x4_t vmax_u16(uint16x4_t __p0, uint16x4_t __p1) {
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vmax_v((int8x8_t)__rev0, (int8x8_t)__rev1, 17);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x8_t vmax_s8(int8x8_t __p0, int8x8_t __p1) {
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vmax_v((int8x8_t)__p0, (int8x8_t)__p1, 0);
  return __ret;
}
#else
__ai int8x8_t vmax_s8(int8x8_t __p0, int8x8_t __p1) {
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vmax_v((int8x8_t)__rev0, (int8x8_t)__rev1, 0);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai float32x2_t vmax_f32(float32x2_t __p0, float32x2_t __p1) {
  float32x2_t __ret;
  __ret = (float32x2_t) __builtin_neon_vmax_v((int8x8_t)__p0, (int8x8_t)__p1, 9);
  return __ret;
}
#else
__ai float32x2_t vmax_f32(float32x2_t __p0, float32x2_t __p1) {
  float32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  float32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  float32x2_t __ret;
  __ret = (float32x2_t) __builtin_neon_vmax_v((int8x8_t)__rev0, (int8x8_t)__rev1, 9);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x2_t vmax_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vmax_v((int8x8_t)__p0, (int8x8_t)__p1, 2);
  return __ret;
}
#else
__ai int32x2_t vmax_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vmax_v((int8x8_t)__rev0, (int8x8_t)__rev1, 2);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x4_t vmax_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vmax_v((int8x8_t)__p0, (int8x8_t)__p1, 1);
  return __ret;
}
#else
__ai int16x4_t vmax_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vmax_v((int8x8_t)__rev0, (int8x8_t)__rev1, 1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x16_t vminq_u8(uint8x16_t __p0, uint8x16_t __p1) {
  uint8x16_t __ret;
  __ret = (uint8x16_t) __builtin_neon_vminq_v((int8x16_t)__p0, (int8x16_t)__p1, 48);
  return __ret;
}
#else
__ai uint8x16_t vminq_u8(uint8x16_t __p0, uint8x16_t __p1) {
  uint8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __ret;
  __ret = (uint8x16_t) __builtin_neon_vminq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 48);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vminq_u32(uint32x4_t __p0, uint32x4_t __p1) {
  uint32x4_t __ret;
  __ret = (uint32x4_t) __builtin_neon_vminq_v((int8x16_t)__p0, (int8x16_t)__p1, 50);
  return __ret;
}
#else
__ai uint32x4_t vminq_u32(uint32x4_t __p0, uint32x4_t __p1) {
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = (uint32x4_t) __builtin_neon_vminq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 50);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x8_t vminq_u16(uint16x8_t __p0, uint16x8_t __p1) {
  uint16x8_t __ret;
  __ret = (uint16x8_t) __builtin_neon_vminq_v((int8x16_t)__p0, (int8x16_t)__p1, 49);
  return __ret;
}
#else
__ai uint16x8_t vminq_u16(uint16x8_t __p0, uint16x8_t __p1) {
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __ret;
  __ret = (uint16x8_t) __builtin_neon_vminq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 49);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x16_t vminq_s8(int8x16_t __p0, int8x16_t __p1) {
  int8x16_t __ret;
  __ret = (int8x16_t) __builtin_neon_vminq_v((int8x16_t)__p0, (int8x16_t)__p1, 32);
  return __ret;
}
#else
__ai int8x16_t vminq_s8(int8x16_t __p0, int8x16_t __p1) {
  int8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __ret;
  __ret = (int8x16_t) __builtin_neon_vminq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 32);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai float32x4_t vminq_f32(float32x4_t __p0, float32x4_t __p1) {
  float32x4_t __ret;
  __ret = (float32x4_t) __builtin_neon_vminq_v((int8x16_t)__p0, (int8x16_t)__p1, 41);
  return __ret;
}
#else
__ai float32x4_t vminq_f32(float32x4_t __p0, float32x4_t __p1) {
  float32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  float32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  float32x4_t __ret;
  __ret = (float32x4_t) __builtin_neon_vminq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 41);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x4_t vminq_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vminq_v((int8x16_t)__p0, (int8x16_t)__p1, 34);
  return __ret;
}
#else
__ai int32x4_t vminq_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vminq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 34);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x8_t vminq_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vminq_v((int8x16_t)__p0, (int8x16_t)__p1, 33);
  return __ret;
}
#else
__ai int16x8_t vminq_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vminq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 33);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x8_t vmin_u8(uint8x8_t __p0, uint8x8_t __p1) {
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vmin_v((int8x8_t)__p0, (int8x8_t)__p1, 16);
  return __ret;
}
#else
__ai uint8x8_t vmin_u8(uint8x8_t __p0, uint8x8_t __p1) {
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vmin_v((int8x8_t)__rev0, (int8x8_t)__rev1, 16);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vmin_u32(uint32x2_t __p0, uint32x2_t __p1) {
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vmin_v((int8x8_t)__p0, (int8x8_t)__p1, 18);
  return __ret;
}
#else
__ai uint32x2_t vmin_u32(uint32x2_t __p0, uint32x2_t __p1) {
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vmin_v((int8x8_t)__rev0, (int8x8_t)__rev1, 18);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x4_t vmin_u16(uint16x4_t __p0, uint16x4_t __p1) {
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vmin_v((int8x8_t)__p0, (int8x8_t)__p1, 17);
  return __ret;
}
#else
__ai uint16x4_t vmin_u16(uint16x4_t __p0, uint16x4_t __p1) {
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vmin_v((int8x8_t)__rev0, (int8x8_t)__rev1, 17);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x8_t vmin_s8(int8x8_t __p0, int8x8_t __p1) {
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vmin_v((int8x8_t)__p0, (int8x8_t)__p1, 0);
  return __ret;
}
#else
__ai int8x8_t vmin_s8(int8x8_t __p0, int8x8_t __p1) {
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vmin_v((int8x8_t)__rev0, (int8x8_t)__rev1, 0);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai float32x2_t vmin_f32(float32x2_t __p0, float32x2_t __p1) {
  float32x2_t __ret;
  __ret = (float32x2_t) __builtin_neon_vmin_v((int8x8_t)__p0, (int8x8_t)__p1, 9);
  return __ret;
}
#else
__ai float32x2_t vmin_f32(float32x2_t __p0, float32x2_t __p1) {
  float32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  float32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  float32x2_t __ret;
  __ret = (float32x2_t) __builtin_neon_vmin_v((int8x8_t)__rev0, (int8x8_t)__rev1, 9);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x2_t vmin_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vmin_v((int8x8_t)__p0, (int8x8_t)__p1, 2);
  return __ret;
}
#else
__ai int32x2_t vmin_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vmin_v((int8x8_t)__rev0, (int8x8_t)__rev1, 2);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x4_t vmin_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vmin_v((int8x8_t)__p0, (int8x8_t)__p1, 1);
  return __ret;
}
#else
__ai int16x4_t vmin_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vmin_v((int8x8_t)__rev0, (int8x8_t)__rev1, 1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x16_t vmlaq_u8(uint8x16_t __p0, uint8x16_t __p1, uint8x16_t __p2) {
  uint8x16_t __ret;
  __ret = __p0 + __p1 * __p2;
  return __ret;
}
#else
__ai uint8x16_t vmlaq_u8(uint8x16_t __p0, uint8x16_t __p1, uint8x16_t __p2) {
  uint8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __rev2;  __rev2 = __builtin_shufflevector(__p2, __p2, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __ret;
  __ret = __rev0 + __rev1 * __rev2;
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vmlaq_u32(uint32x4_t __p0, uint32x4_t __p1, uint32x4_t __p2) {
  uint32x4_t __ret;
  __ret = __p0 + __p1 * __p2;
  return __ret;
}
#else
__ai uint32x4_t vmlaq_u32(uint32x4_t __p0, uint32x4_t __p1, uint32x4_t __p2) {
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint32x4_t __rev2;  __rev2 = __builtin_shufflevector(__p2, __p2, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = __rev0 + __rev1 * __rev2;
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x8_t vmlaq_u16(uint16x8_t __p0, uint16x8_t __p1, uint16x8_t __p2) {
  uint16x8_t __ret;
  __ret = __p0 + __p1 * __p2;
  return __ret;
}
#else
__ai uint16x8_t vmlaq_u16(uint16x8_t __p0, uint16x8_t __p1, uint16x8_t __p2) {
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __rev2;  __rev2 = __builtin_shufflevector(__p2, __p2, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __ret;
  __ret = __rev0 + __rev1 * __rev2;
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x16_t vmlaq_s8(int8x16_t __p0, int8x16_t __p1, int8x16_t __p2) {
  int8x16_t __ret;
  __ret = __p0 + __p1 * __p2;
  return __ret;
}
#else
__ai int8x16_t vmlaq_s8(int8x16_t __p0, int8x16_t __p1, int8x16_t __p2) {
  int8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __rev2;  __rev2 = __builtin_shufflevector(__p2, __p2, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __ret;
  __ret = __rev0 + __rev1 * __rev2;
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai float32x4_t vmlaq_f32(float32x4_t __p0, float32x4_t __p1, float32x4_t __p2) {
  float32x4_t __ret;
  __ret = __p0 + __p1 * __p2;
  return __ret;
}
#else
__ai float32x4_t vmlaq_f32(float32x4_t __p0, float32x4_t __p1, float32x4_t __p2) {
  float32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  float32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  float32x4_t __rev2;  __rev2 = __builtin_shufflevector(__p2, __p2, 3, 2, 1, 0);
  float32x4_t __ret;
  __ret = __rev0 + __rev1 * __rev2;
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x4_t vmlaq_s32(int32x4_t __p0, int32x4_t __p1, int32x4_t __p2) {
  int32x4_t __ret;
  __ret = __p0 + __p1 * __p2;
  return __ret;
}
#else
__ai int32x4_t vmlaq_s32(int32x4_t __p0, int32x4_t __p1, int32x4_t __p2) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int32x4_t __rev2;  __rev2 = __builtin_shufflevector(__p2, __p2, 3, 2, 1, 0);
  int32x4_t __ret;
  __ret = __rev0 + __rev1 * __rev2;
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x8_t vmlaq_s16(int16x8_t __p0, int16x8_t __p1, int16x8_t __p2) {
  int16x8_t __ret;
  __ret = __p0 + __p1 * __p2;
  return __ret;
}
#else
__ai int16x8_t vmlaq_s16(int16x8_t __p0, int16x8_t __p1, int16x8_t __p2) {
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __rev2;  __rev2 = __builtin_shufflevector(__p2, __p2, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __ret;
  __ret = __rev0 + __rev1 * __rev2;
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x8_t vmla_u8(uint8x8_t __p0, uint8x8_t __p1, uint8x8_t __p2) {
  uint8x8_t __ret;
  __ret = __p0 + __p1 * __p2;
  return __ret;
}
#else
__ai uint8x8_t vmla_u8(uint8x8_t __p0, uint8x8_t __p1, uint8x8_t __p2) {
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __rev2;  __rev2 = __builtin_shufflevector(__p2, __p2, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __ret;
  __ret = __rev0 + __rev1 * __rev2;
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vmla_u32(uint32x2_t __p0, uint32x2_t __p1, uint32x2_t __p2) {
  uint32x2_t __ret;
  __ret = __p0 + __p1 * __p2;
  return __ret;
}
#else
__ai uint32x2_t vmla_u32(uint32x2_t __p0, uint32x2_t __p1, uint32x2_t __p2) {
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint32x2_t __rev2;  __rev2 = __builtin_shufflevector(__p2, __p2, 1, 0);
  uint32x2_t __ret;
  __ret = __rev0 + __rev1 * __rev2;
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x4_t vmla_u16(uint16x4_t __p0, uint16x4_t __p1, uint16x4_t __p2) {
  uint16x4_t __ret;
  __ret = __p0 + __p1 * __p2;
  return __ret;
}
#else
__ai uint16x4_t vmla_u16(uint16x4_t __p0, uint16x4_t __p1, uint16x4_t __p2) {
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint16x4_t __rev2;  __rev2 = __builtin_shufflevector(__p2, __p2, 3, 2, 1, 0);
  uint16x4_t __ret;
  __ret = __rev0 + __rev1 * __rev2;
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x8_t vmla_s8(int8x8_t __p0, int8x8_t __p1, int8x8_t __p2) {
  int8x8_t __ret;
  __ret = __p0 + __p1 * __p2;
  return __ret;
}
#else
__ai int8x8_t vmla_s8(int8x8_t __p0, int8x8_t __p1, int8x8_t __p2) {
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __rev2;  __rev2 = __builtin_shufflevector(__p2, __p2, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __ret;
  __ret = __rev0 + __rev1 * __rev2;
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai float32x2_t vmla_f32(float32x2_t __p0, float32x2_t __p1, float32x2_t __p2) {
  float32x2_t __ret;
  __ret = __p0 + __p1 * __p2;
  return __ret;
}
#else
__ai float32x2_t vmla_f32(float32x2_t __p0, float32x2_t __p1, float32x2_t __p2) {
  float32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  float32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  float32x2_t __rev2;  __rev2 = __builtin_shufflevector(__p2, __p2, 1, 0);
  float32x2_t __ret;
  __ret = __rev0 + __rev1 * __rev2;
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x2_t vmla_s32(int32x2_t __p0, int32x2_t __p1, int32x2_t __p2) {
  int32x2_t __ret;
  __ret = __p0 + __p1 * __p2;
  return __ret;
}
#else
__ai int32x2_t vmla_s32(int32x2_t __p0, int32x2_t __p1, int32x2_t __p2) {
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  int32x2_t __rev2;  __rev2 = __builtin_shufflevector(__p2, __p2, 1, 0);
  int32x2_t __ret;
  __ret = __rev0 + __rev1 * __rev2;
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x4_t vmla_s16(int16x4_t __p0, int16x4_t __p1, int16x4_t __p2) {
  int16x4_t __ret;
  __ret = __p0 + __p1 * __p2;
  return __ret;
}
#else
__ai int16x4_t vmla_s16(int16x4_t __p0, int16x4_t __p1, int16x4_t __p2) {
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int16x4_t __rev2;  __rev2 = __builtin_shufflevector(__p2, __p2, 3, 2, 1, 0);
  int16x4_t __ret;
  __ret = __rev0 + __rev1 * __rev2;
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
#define vmlaq_lane_u32(__p0_42, __p1_42, __p2_42, __p3_42) __extension__ ({ \
  uint32x4_t __s0_42 = __p0_42; \
  uint32x4_t __s1_42 = __p1_42; \
  uint32x2_t __s2_42 = __p2_42; \
  uint32x4_t __ret_42; \
  __ret_42 = __s0_42 + __s1_42 * splatq_lane_u32(__s2_42, __p3_42); \
  __ret_42; \
})
#else
#define vmlaq_lane_u32(__p0_43, __p1_43, __p2_43, __p3_43) __extension__ ({ \
  uint32x4_t __s0_43 = __p0_43; \
  uint32x4_t __s1_43 = __p1_43; \
  uint32x2_t __s2_43 = __p2_43; \
  uint32x4_t __rev0_43;  __rev0_43 = __builtin_shufflevector(__s0_43, __s0_43, 3, 2, 1, 0); \
  uint32x4_t __rev1_43;  __rev1_43 = __builtin_shufflevector(__s1_43, __s1_43, 3, 2, 1, 0); \
  uint32x2_t __rev2_43;  __rev2_43 = __builtin_shufflevector(__s2_43, __s2_43, 1, 0); \
  uint32x4_t __ret_43; \
  __ret_43 = __rev0_43 + __rev1_43 * __noswap_splatq_lane_u32(__rev2_43, __p3_43); \
  __ret_43 = __builtin_shufflevector(__ret_43, __ret_43, 3, 2, 1, 0); \
  __ret_43; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vmlaq_lane_u16(__p0_44, __p1_44, __p2_44, __p3_44) __extension__ ({ \
  uint16x8_t __s0_44 = __p0_44; \
  uint16x8_t __s1_44 = __p1_44; \
  uint16x4_t __s2_44 = __p2_44; \
  uint16x8_t __ret_44; \
  __ret_44 = __s0_44 + __s1_44 * splatq_lane_u16(__s2_44, __p3_44); \
  __ret_44; \
})
#else
#define vmlaq_lane_u16(__p0_45, __p1_45, __p2_45, __p3_45) __extension__ ({ \
  uint16x8_t __s0_45 = __p0_45; \
  uint16x8_t __s1_45 = __p1_45; \
  uint16x4_t __s2_45 = __p2_45; \
  uint16x8_t __rev0_45;  __rev0_45 = __builtin_shufflevector(__s0_45, __s0_45, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint16x8_t __rev1_45;  __rev1_45 = __builtin_shufflevector(__s1_45, __s1_45, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint16x4_t __rev2_45;  __rev2_45 = __builtin_shufflevector(__s2_45, __s2_45, 3, 2, 1, 0); \
  uint16x8_t __ret_45; \
  __ret_45 = __rev0_45 + __rev1_45 * __noswap_splatq_lane_u16(__rev2_45, __p3_45); \
  __ret_45 = __builtin_shufflevector(__ret_45, __ret_45, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret_45; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vmlaq_lane_f32(__p0_46, __p1_46, __p2_46, __p3_46) __extension__ ({ \
  float32x4_t __s0_46 = __p0_46; \
  float32x4_t __s1_46 = __p1_46; \
  float32x2_t __s2_46 = __p2_46; \
  float32x4_t __ret_46; \
  __ret_46 = __s0_46 + __s1_46 * splatq_lane_f32(__s2_46, __p3_46); \
  __ret_46; \
})
#else
#define vmlaq_lane_f32(__p0_47, __p1_47, __p2_47, __p3_47) __extension__ ({ \
  float32x4_t __s0_47 = __p0_47; \
  float32x4_t __s1_47 = __p1_47; \
  float32x2_t __s2_47 = __p2_47; \
  float32x4_t __rev0_47;  __rev0_47 = __builtin_shufflevector(__s0_47, __s0_47, 3, 2, 1, 0); \
  float32x4_t __rev1_47;  __rev1_47 = __builtin_shufflevector(__s1_47, __s1_47, 3, 2, 1, 0); \
  float32x2_t __rev2_47;  __rev2_47 = __builtin_shufflevector(__s2_47, __s2_47, 1, 0); \
  float32x4_t __ret_47; \
  __ret_47 = __rev0_47 + __rev1_47 * __noswap_splatq_lane_f32(__rev2_47, __p3_47); \
  __ret_47 = __builtin_shufflevector(__ret_47, __ret_47, 3, 2, 1, 0); \
  __ret_47; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vmlaq_lane_s32(__p0_48, __p1_48, __p2_48, __p3_48) __extension__ ({ \
  int32x4_t __s0_48 = __p0_48; \
  int32x4_t __s1_48 = __p1_48; \
  int32x2_t __s2_48 = __p2_48; \
  int32x4_t __ret_48; \
  __ret_48 = __s0_48 + __s1_48 * splatq_lane_s32(__s2_48, __p3_48); \
  __ret_48; \
})
#else
#define vmlaq_lane_s32(__p0_49, __p1_49, __p2_49, __p3_49) __extension__ ({ \
  int32x4_t __s0_49 = __p0_49; \
  int32x4_t __s1_49 = __p1_49; \
  int32x2_t __s2_49 = __p2_49; \
  int32x4_t __rev0_49;  __rev0_49 = __builtin_shufflevector(__s0_49, __s0_49, 3, 2, 1, 0); \
  int32x4_t __rev1_49;  __rev1_49 = __builtin_shufflevector(__s1_49, __s1_49, 3, 2, 1, 0); \
  int32x2_t __rev2_49;  __rev2_49 = __builtin_shufflevector(__s2_49, __s2_49, 1, 0); \
  int32x4_t __ret_49; \
  __ret_49 = __rev0_49 + __rev1_49 * __noswap_splatq_lane_s32(__rev2_49, __p3_49); \
  __ret_49 = __builtin_shufflevector(__ret_49, __ret_49, 3, 2, 1, 0); \
  __ret_49; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vmlaq_lane_s16(__p0_50, __p1_50, __p2_50, __p3_50) __extension__ ({ \
  int16x8_t __s0_50 = __p0_50; \
  int16x8_t __s1_50 = __p1_50; \
  int16x4_t __s2_50 = __p2_50; \
  int16x8_t __ret_50; \
  __ret_50 = __s0_50 + __s1_50 * splatq_lane_s16(__s2_50, __p3_50); \
  __ret_50; \
})
#else
#define vmlaq_lane_s16(__p0_51, __p1_51, __p2_51, __p3_51) __extension__ ({ \
  int16x8_t __s0_51 = __p0_51; \
  int16x8_t __s1_51 = __p1_51; \
  int16x4_t __s2_51 = __p2_51; \
  int16x8_t __rev0_51;  __rev0_51 = __builtin_shufflevector(__s0_51, __s0_51, 7, 6, 5, 4, 3, 2, 1, 0); \
  int16x8_t __rev1_51;  __rev1_51 = __builtin_shufflevector(__s1_51, __s1_51, 7, 6, 5, 4, 3, 2, 1, 0); \
  int16x4_t __rev2_51;  __rev2_51 = __builtin_shufflevector(__s2_51, __s2_51, 3, 2, 1, 0); \
  int16x8_t __ret_51; \
  __ret_51 = __rev0_51 + __rev1_51 * __noswap_splatq_lane_s16(__rev2_51, __p3_51); \
  __ret_51 = __builtin_shufflevector(__ret_51, __ret_51, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret_51; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vmla_lane_u32(__p0_52, __p1_52, __p2_52, __p3_52) __extension__ ({ \
  uint32x2_t __s0_52 = __p0_52; \
  uint32x2_t __s1_52 = __p1_52; \
  uint32x2_t __s2_52 = __p2_52; \
  uint32x2_t __ret_52; \
  __ret_52 = __s0_52 + __s1_52 * splat_lane_u32(__s2_52, __p3_52); \
  __ret_52; \
})
#else
#define vmla_lane_u32(__p0_53, __p1_53, __p2_53, __p3_53) __extension__ ({ \
  uint32x2_t __s0_53 = __p0_53; \
  uint32x2_t __s1_53 = __p1_53; \
  uint32x2_t __s2_53 = __p2_53; \
  uint32x2_t __rev0_53;  __rev0_53 = __builtin_shufflevector(__s0_53, __s0_53, 1, 0); \
  uint32x2_t __rev1_53;  __rev1_53 = __builtin_shufflevector(__s1_53, __s1_53, 1, 0); \
  uint32x2_t __rev2_53;  __rev2_53 = __builtin_shufflevector(__s2_53, __s2_53, 1, 0); \
  uint32x2_t __ret_53; \
  __ret_53 = __rev0_53 + __rev1_53 * __noswap_splat_lane_u32(__rev2_53, __p3_53); \
  __ret_53 = __builtin_shufflevector(__ret_53, __ret_53, 1, 0); \
  __ret_53; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vmla_lane_u16(__p0_54, __p1_54, __p2_54, __p3_54) __extension__ ({ \
  uint16x4_t __s0_54 = __p0_54; \
  uint16x4_t __s1_54 = __p1_54; \
  uint16x4_t __s2_54 = __p2_54; \
  uint16x4_t __ret_54; \
  __ret_54 = __s0_54 + __s1_54 * splat_lane_u16(__s2_54, __p3_54); \
  __ret_54; \
})
#else
#define vmla_lane_u16(__p0_55, __p1_55, __p2_55, __p3_55) __extension__ ({ \
  uint16x4_t __s0_55 = __p0_55; \
  uint16x4_t __s1_55 = __p1_55; \
  uint16x4_t __s2_55 = __p2_55; \
  uint16x4_t __rev0_55;  __rev0_55 = __builtin_shufflevector(__s0_55, __s0_55, 3, 2, 1, 0); \
  uint16x4_t __rev1_55;  __rev1_55 = __builtin_shufflevector(__s1_55, __s1_55, 3, 2, 1, 0); \
  uint16x4_t __rev2_55;  __rev2_55 = __builtin_shufflevector(__s2_55, __s2_55, 3, 2, 1, 0); \
  uint16x4_t __ret_55; \
  __ret_55 = __rev0_55 + __rev1_55 * __noswap_splat_lane_u16(__rev2_55, __p3_55); \
  __ret_55 = __builtin_shufflevector(__ret_55, __ret_55, 3, 2, 1, 0); \
  __ret_55; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vmla_lane_f32(__p0_56, __p1_56, __p2_56, __p3_56) __extension__ ({ \
  float32x2_t __s0_56 = __p0_56; \
  float32x2_t __s1_56 = __p1_56; \
  float32x2_t __s2_56 = __p2_56; \
  float32x2_t __ret_56; \
  __ret_56 = __s0_56 + __s1_56 * splat_lane_f32(__s2_56, __p3_56); \
  __ret_56; \
})
#else
#define vmla_lane_f32(__p0_57, __p1_57, __p2_57, __p3_57) __extension__ ({ \
  float32x2_t __s0_57 = __p0_57; \
  float32x2_t __s1_57 = __p1_57; \
  float32x2_t __s2_57 = __p2_57; \
  float32x2_t __rev0_57;  __rev0_57 = __builtin_shufflevector(__s0_57, __s0_57, 1, 0); \
  float32x2_t __rev1_57;  __rev1_57 = __builtin_shufflevector(__s1_57, __s1_57, 1, 0); \
  float32x2_t __rev2_57;  __rev2_57 = __builtin_shufflevector(__s2_57, __s2_57, 1, 0); \
  float32x2_t __ret_57; \
  __ret_57 = __rev0_57 + __rev1_57 * __noswap_splat_lane_f32(__rev2_57, __p3_57); \
  __ret_57 = __builtin_shufflevector(__ret_57, __ret_57, 1, 0); \
  __ret_57; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vmla_lane_s32(__p0_58, __p1_58, __p2_58, __p3_58) __extension__ ({ \
  int32x2_t __s0_58 = __p0_58; \
  int32x2_t __s1_58 = __p1_58; \
  int32x2_t __s2_58 = __p2_58; \
  int32x2_t __ret_58; \
  __ret_58 = __s0_58 + __s1_58 * splat_lane_s32(__s2_58, __p3_58); \
  __ret_58; \
})
#else
#define vmla_lane_s32(__p0_59, __p1_59, __p2_59, __p3_59) __extension__ ({ \
  int32x2_t __s0_59 = __p0_59; \
  int32x2_t __s1_59 = __p1_59; \
  int32x2_t __s2_59 = __p2_59; \
  int32x2_t __rev0_59;  __rev0_59 = __builtin_shufflevector(__s0_59, __s0_59, 1, 0); \
  int32x2_t __rev1_59;  __rev1_59 = __builtin_shufflevector(__s1_59, __s1_59, 1, 0); \
  int32x2_t __rev2_59;  __rev2_59 = __builtin_shufflevector(__s2_59, __s2_59, 1, 0); \
  int32x2_t __ret_59; \
  __ret_59 = __rev0_59 + __rev1_59 * __noswap_splat_lane_s32(__rev2_59, __p3_59); \
  __ret_59 = __builtin_shufflevector(__ret_59, __ret_59, 1, 0); \
  __ret_59; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vmla_lane_s16(__p0_60, __p1_60, __p2_60, __p3_60) __extension__ ({ \
  int16x4_t __s0_60 = __p0_60; \
  int16x4_t __s1_60 = __p1_60; \
  int16x4_t __s2_60 = __p2_60; \
  int16x4_t __ret_60; \
  __ret_60 = __s0_60 + __s1_60 * splat_lane_s16(__s2_60, __p3_60); \
  __ret_60; \
})
#else
#define vmla_lane_s16(__p0_61, __p1_61, __p2_61, __p3_61) __extension__ ({ \
  int16x4_t __s0_61 = __p0_61; \
  int16x4_t __s1_61 = __p1_61; \
  int16x4_t __s2_61 = __p2_61; \
  int16x4_t __rev0_61;  __rev0_61 = __builtin_shufflevector(__s0_61, __s0_61, 3, 2, 1, 0); \
  int16x4_t __rev1_61;  __rev1_61 = __builtin_shufflevector(__s1_61, __s1_61, 3, 2, 1, 0); \
  int16x4_t __rev2_61;  __rev2_61 = __builtin_shufflevector(__s2_61, __s2_61, 3, 2, 1, 0); \
  int16x4_t __ret_61; \
  __ret_61 = __rev0_61 + __rev1_61 * __noswap_splat_lane_s16(__rev2_61, __p3_61); \
  __ret_61 = __builtin_shufflevector(__ret_61, __ret_61, 3, 2, 1, 0); \
  __ret_61; \
})
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vmlaq_n_u32(uint32x4_t __p0, uint32x4_t __p1, uint32_t __p2) {
  uint32x4_t __ret;
  __ret = __p0 + __p1 * (uint32x4_t) {__p2, __p2, __p2, __p2};
  return __ret;
}
#else
__ai uint32x4_t vmlaq_n_u32(uint32x4_t __p0, uint32x4_t __p1, uint32_t __p2) {
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = __rev0 + __rev1 * (uint32x4_t) {__p2, __p2, __p2, __p2};
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x8_t vmlaq_n_u16(uint16x8_t __p0, uint16x8_t __p1, uint16_t __p2) {
  uint16x8_t __ret;
  __ret = __p0 + __p1 * (uint16x8_t) {__p2, __p2, __p2, __p2, __p2, __p2, __p2, __p2};
  return __ret;
}
#else
__ai uint16x8_t vmlaq_n_u16(uint16x8_t __p0, uint16x8_t __p1, uint16_t __p2) {
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __ret;
  __ret = __rev0 + __rev1 * (uint16x8_t) {__p2, __p2, __p2, __p2, __p2, __p2, __p2, __p2};
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai float32x4_t vmlaq_n_f32(float32x4_t __p0, float32x4_t __p1, float32_t __p2) {
  float32x4_t __ret;
  __ret = __p0 + __p1 * (float32x4_t) {__p2, __p2, __p2, __p2};
  return __ret;
}
#else
__ai float32x4_t vmlaq_n_f32(float32x4_t __p0, float32x4_t __p1, float32_t __p2) {
  float32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  float32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  float32x4_t __ret;
  __ret = __rev0 + __rev1 * (float32x4_t) {__p2, __p2, __p2, __p2};
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x4_t vmlaq_n_s32(int32x4_t __p0, int32x4_t __p1, int32_t __p2) {
  int32x4_t __ret;
  __ret = __p0 + __p1 * (int32x4_t) {__p2, __p2, __p2, __p2};
  return __ret;
}
#else
__ai int32x4_t vmlaq_n_s32(int32x4_t __p0, int32x4_t __p1, int32_t __p2) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int32x4_t __ret;
  __ret = __rev0 + __rev1 * (int32x4_t) {__p2, __p2, __p2, __p2};
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x8_t vmlaq_n_s16(int16x8_t __p0, int16x8_t __p1, int16_t __p2) {
  int16x8_t __ret;
  __ret = __p0 + __p1 * (int16x8_t) {__p2, __p2, __p2, __p2, __p2, __p2, __p2, __p2};
  return __ret;
}
#else
__ai int16x8_t vmlaq_n_s16(int16x8_t __p0, int16x8_t __p1, int16_t __p2) {
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __ret;
  __ret = __rev0 + __rev1 * (int16x8_t) {__p2, __p2, __p2, __p2, __p2, __p2, __p2, __p2};
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vmla_n_u32(uint32x2_t __p0, uint32x2_t __p1, uint32_t __p2) {
  uint32x2_t __ret;
  __ret = __p0 + __p1 * (uint32x2_t) {__p2, __p2};
  return __ret;
}
#else
__ai uint32x2_t vmla_n_u32(uint32x2_t __p0, uint32x2_t __p1, uint32_t __p2) {
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint32x2_t __ret;
  __ret = __rev0 + __rev1 * (uint32x2_t) {__p2, __p2};
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x4_t vmla_n_u16(uint16x4_t __p0, uint16x4_t __p1, uint16_t __p2) {
  uint16x4_t __ret;
  __ret = __p0 + __p1 * (uint16x4_t) {__p2, __p2, __p2, __p2};
  return __ret;
}
#else
__ai uint16x4_t vmla_n_u16(uint16x4_t __p0, uint16x4_t __p1, uint16_t __p2) {
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint16x4_t __ret;
  __ret = __rev0 + __rev1 * (uint16x4_t) {__p2, __p2, __p2, __p2};
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai float32x2_t vmla_n_f32(float32x2_t __p0, float32x2_t __p1, float32_t __p2) {
  float32x2_t __ret;
  __ret = __p0 + __p1 * (float32x2_t) {__p2, __p2};
  return __ret;
}
#else
__ai float32x2_t vmla_n_f32(float32x2_t __p0, float32x2_t __p1, float32_t __p2) {
  float32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  float32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  float32x2_t __ret;
  __ret = __rev0 + __rev1 * (float32x2_t) {__p2, __p2};
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x2_t vmla_n_s32(int32x2_t __p0, int32x2_t __p1, int32_t __p2) {
  int32x2_t __ret;
  __ret = __p0 + __p1 * (int32x2_t) {__p2, __p2};
  return __ret;
}
#else
__ai int32x2_t vmla_n_s32(int32x2_t __p0, int32x2_t __p1, int32_t __p2) {
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  int32x2_t __ret;
  __ret = __rev0 + __rev1 * (int32x2_t) {__p2, __p2};
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x4_t vmla_n_s16(int16x4_t __p0, int16x4_t __p1, int16_t __p2) {
  int16x4_t __ret;
  __ret = __p0 + __p1 * (int16x4_t) {__p2, __p2, __p2, __p2};
  return __ret;
}
#else
__ai int16x4_t vmla_n_s16(int16x4_t __p0, int16x4_t __p1, int16_t __p2) {
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int16x4_t __ret;
  __ret = __rev0 + __rev1 * (int16x4_t) {__p2, __p2, __p2, __p2};
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x16_t vmlsq_u8(uint8x16_t __p0, uint8x16_t __p1, uint8x16_t __p2) {
  uint8x16_t __ret;
  __ret = __p0 - __p1 * __p2;
  return __ret;
}
#else
__ai uint8x16_t vmlsq_u8(uint8x16_t __p0, uint8x16_t __p1, uint8x16_t __p2) {
  uint8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __rev2;  __rev2 = __builtin_shufflevector(__p2, __p2, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __ret;
  __ret = __rev0 - __rev1 * __rev2;
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vmlsq_u32(uint32x4_t __p0, uint32x4_t __p1, uint32x4_t __p2) {
  uint32x4_t __ret;
  __ret = __p0 - __p1 * __p2;
  return __ret;
}
#else
__ai uint32x4_t vmlsq_u32(uint32x4_t __p0, uint32x4_t __p1, uint32x4_t __p2) {
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint32x4_t __rev2;  __rev2 = __builtin_shufflevector(__p2, __p2, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = __rev0 - __rev1 * __rev2;
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x8_t vmlsq_u16(uint16x8_t __p0, uint16x8_t __p1, uint16x8_t __p2) {
  uint16x8_t __ret;
  __ret = __p0 - __p1 * __p2;
  return __ret;
}
#else
__ai uint16x8_t vmlsq_u16(uint16x8_t __p0, uint16x8_t __p1, uint16x8_t __p2) {
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __rev2;  __rev2 = __builtin_shufflevector(__p2, __p2, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __ret;
  __ret = __rev0 - __rev1 * __rev2;
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x16_t vmlsq_s8(int8x16_t __p0, int8x16_t __p1, int8x16_t __p2) {
  int8x16_t __ret;
  __ret = __p0 - __p1 * __p2;
  return __ret;
}
#else
__ai int8x16_t vmlsq_s8(int8x16_t __p0, int8x16_t __p1, int8x16_t __p2) {
  int8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __rev2;  __rev2 = __builtin_shufflevector(__p2, __p2, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __ret;
  __ret = __rev0 - __rev1 * __rev2;
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai float32x4_t vmlsq_f32(float32x4_t __p0, float32x4_t __p1, float32x4_t __p2) {
  float32x4_t __ret;
  __ret = __p0 - __p1 * __p2;
  return __ret;
}
#else
__ai float32x4_t vmlsq_f32(float32x4_t __p0, float32x4_t __p1, float32x4_t __p2) {
  float32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  float32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  float32x4_t __rev2;  __rev2 = __builtin_shufflevector(__p2, __p2, 3, 2, 1, 0);
  float32x4_t __ret;
  __ret = __rev0 - __rev1 * __rev2;
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x4_t vmlsq_s32(int32x4_t __p0, int32x4_t __p1, int32x4_t __p2) {
  int32x4_t __ret;
  __ret = __p0 - __p1 * __p2;
  return __ret;
}
#else
__ai int32x4_t vmlsq_s32(int32x4_t __p0, int32x4_t __p1, int32x4_t __p2) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int32x4_t __rev2;  __rev2 = __builtin_shufflevector(__p2, __p2, 3, 2, 1, 0);
  int32x4_t __ret;
  __ret = __rev0 - __rev1 * __rev2;
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x8_t vmlsq_s16(int16x8_t __p0, int16x8_t __p1, int16x8_t __p2) {
  int16x8_t __ret;
  __ret = __p0 - __p1 * __p2;
  return __ret;
}
#else
__ai int16x8_t vmlsq_s16(int16x8_t __p0, int16x8_t __p1, int16x8_t __p2) {
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __rev2;  __rev2 = __builtin_shufflevector(__p2, __p2, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __ret;
  __ret = __rev0 - __rev1 * __rev2;
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x8_t vmls_u8(uint8x8_t __p0, uint8x8_t __p1, uint8x8_t __p2) {
  uint8x8_t __ret;
  __ret = __p0 - __p1 * __p2;
  return __ret;
}
#else
__ai uint8x8_t vmls_u8(uint8x8_t __p0, uint8x8_t __p1, uint8x8_t __p2) {
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __rev2;  __rev2 = __builtin_shufflevector(__p2, __p2, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __ret;
  __ret = __rev0 - __rev1 * __rev2;
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vmls_u32(uint32x2_t __p0, uint32x2_t __p1, uint32x2_t __p2) {
  uint32x2_t __ret;
  __ret = __p0 - __p1 * __p2;
  return __ret;
}
#else
__ai uint32x2_t vmls_u32(uint32x2_t __p0, uint32x2_t __p1, uint32x2_t __p2) {
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint32x2_t __rev2;  __rev2 = __builtin_shufflevector(__p2, __p2, 1, 0);
  uint32x2_t __ret;
  __ret = __rev0 - __rev1 * __rev2;
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x4_t vmls_u16(uint16x4_t __p0, uint16x4_t __p1, uint16x4_t __p2) {
  uint16x4_t __ret;
  __ret = __p0 - __p1 * __p2;
  return __ret;
}
#else
__ai uint16x4_t vmls_u16(uint16x4_t __p0, uint16x4_t __p1, uint16x4_t __p2) {
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint16x4_t __rev2;  __rev2 = __builtin_shufflevector(__p2, __p2, 3, 2, 1, 0);
  uint16x4_t __ret;
  __ret = __rev0 - __rev1 * __rev2;
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x8_t vmls_s8(int8x8_t __p0, int8x8_t __p1, int8x8_t __p2) {
  int8x8_t __ret;
  __ret = __p0 - __p1 * __p2;
  return __ret;
}
#else
__ai int8x8_t vmls_s8(int8x8_t __p0, int8x8_t __p1, int8x8_t __p2) {
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __rev2;  __rev2 = __builtin_shufflevector(__p2, __p2, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __ret;
  __ret = __rev0 - __rev1 * __rev2;
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai float32x2_t vmls_f32(float32x2_t __p0, float32x2_t __p1, float32x2_t __p2) {
  float32x2_t __ret;
  __ret = __p0 - __p1 * __p2;
  return __ret;
}
#else
__ai float32x2_t vmls_f32(float32x2_t __p0, float32x2_t __p1, float32x2_t __p2) {
  float32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  float32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  float32x2_t __rev2;  __rev2 = __builtin_shufflevector(__p2, __p2, 1, 0);
  float32x2_t __ret;
  __ret = __rev0 - __rev1 * __rev2;
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x2_t vmls_s32(int32x2_t __p0, int32x2_t __p1, int32x2_t __p2) {
  int32x2_t __ret;
  __ret = __p0 - __p1 * __p2;
  return __ret;
}
#else
__ai int32x2_t vmls_s32(int32x2_t __p0, int32x2_t __p1, int32x2_t __p2) {
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  int32x2_t __rev2;  __rev2 = __builtin_shufflevector(__p2, __p2, 1, 0);
  int32x2_t __ret;
  __ret = __rev0 - __rev1 * __rev2;
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x4_t vmls_s16(int16x4_t __p0, int16x4_t __p1, int16x4_t __p2) {
  int16x4_t __ret;
  __ret = __p0 - __p1 * __p2;
  return __ret;
}
#else
__ai int16x4_t vmls_s16(int16x4_t __p0, int16x4_t __p1, int16x4_t __p2) {
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int16x4_t __rev2;  __rev2 = __builtin_shufflevector(__p2, __p2, 3, 2, 1, 0);
  int16x4_t __ret;
  __ret = __rev0 - __rev1 * __rev2;
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
#define vmlsq_lane_u32(__p0_62, __p1_62, __p2_62, __p3_62) __extension__ ({ \
  uint32x4_t __s0_62 = __p0_62; \
  uint32x4_t __s1_62 = __p1_62; \
  uint32x2_t __s2_62 = __p2_62; \
  uint32x4_t __ret_62; \
  __ret_62 = __s0_62 - __s1_62 * splatq_lane_u32(__s2_62, __p3_62); \
  __ret_62; \
})
#else
#define vmlsq_lane_u32(__p0_63, __p1_63, __p2_63, __p3_63) __extension__ ({ \
  uint32x4_t __s0_63 = __p0_63; \
  uint32x4_t __s1_63 = __p1_63; \
  uint32x2_t __s2_63 = __p2_63; \
  uint32x4_t __rev0_63;  __rev0_63 = __builtin_shufflevector(__s0_63, __s0_63, 3, 2, 1, 0); \
  uint32x4_t __rev1_63;  __rev1_63 = __builtin_shufflevector(__s1_63, __s1_63, 3, 2, 1, 0); \
  uint32x2_t __rev2_63;  __rev2_63 = __builtin_shufflevector(__s2_63, __s2_63, 1, 0); \
  uint32x4_t __ret_63; \
  __ret_63 = __rev0_63 - __rev1_63 * __noswap_splatq_lane_u32(__rev2_63, __p3_63); \
  __ret_63 = __builtin_shufflevector(__ret_63, __ret_63, 3, 2, 1, 0); \
  __ret_63; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vmlsq_lane_u16(__p0_64, __p1_64, __p2_64, __p3_64) __extension__ ({ \
  uint16x8_t __s0_64 = __p0_64; \
  uint16x8_t __s1_64 = __p1_64; \
  uint16x4_t __s2_64 = __p2_64; \
  uint16x8_t __ret_64; \
  __ret_64 = __s0_64 - __s1_64 * splatq_lane_u16(__s2_64, __p3_64); \
  __ret_64; \
})
#else
#define vmlsq_lane_u16(__p0_65, __p1_65, __p2_65, __p3_65) __extension__ ({ \
  uint16x8_t __s0_65 = __p0_65; \
  uint16x8_t __s1_65 = __p1_65; \
  uint16x4_t __s2_65 = __p2_65; \
  uint16x8_t __rev0_65;  __rev0_65 = __builtin_shufflevector(__s0_65, __s0_65, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint16x8_t __rev1_65;  __rev1_65 = __builtin_shufflevector(__s1_65, __s1_65, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint16x4_t __rev2_65;  __rev2_65 = __builtin_shufflevector(__s2_65, __s2_65, 3, 2, 1, 0); \
  uint16x8_t __ret_65; \
  __ret_65 = __rev0_65 - __rev1_65 * __noswap_splatq_lane_u16(__rev2_65, __p3_65); \
  __ret_65 = __builtin_shufflevector(__ret_65, __ret_65, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret_65; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vmlsq_lane_f32(__p0_66, __p1_66, __p2_66, __p3_66) __extension__ ({ \
  float32x4_t __s0_66 = __p0_66; \
  float32x4_t __s1_66 = __p1_66; \
  float32x2_t __s2_66 = __p2_66; \
  float32x4_t __ret_66; \
  __ret_66 = __s0_66 - __s1_66 * splatq_lane_f32(__s2_66, __p3_66); \
  __ret_66; \
})
#else
#define vmlsq_lane_f32(__p0_67, __p1_67, __p2_67, __p3_67) __extension__ ({ \
  float32x4_t __s0_67 = __p0_67; \
  float32x4_t __s1_67 = __p1_67; \
  float32x2_t __s2_67 = __p2_67; \
  float32x4_t __rev0_67;  __rev0_67 = __builtin_shufflevector(__s0_67, __s0_67, 3, 2, 1, 0); \
  float32x4_t __rev1_67;  __rev1_67 = __builtin_shufflevector(__s1_67, __s1_67, 3, 2, 1, 0); \
  float32x2_t __rev2_67;  __rev2_67 = __builtin_shufflevector(__s2_67, __s2_67, 1, 0); \
  float32x4_t __ret_67; \
  __ret_67 = __rev0_67 - __rev1_67 * __noswap_splatq_lane_f32(__rev2_67, __p3_67); \
  __ret_67 = __builtin_shufflevector(__ret_67, __ret_67, 3, 2, 1, 0); \
  __ret_67; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vmlsq_lane_s32(__p0_68, __p1_68, __p2_68, __p3_68) __extension__ ({ \
  int32x4_t __s0_68 = __p0_68; \
  int32x4_t __s1_68 = __p1_68; \
  int32x2_t __s2_68 = __p2_68; \
  int32x4_t __ret_68; \
  __ret_68 = __s0_68 - __s1_68 * splatq_lane_s32(__s2_68, __p3_68); \
  __ret_68; \
})
#else
#define vmlsq_lane_s32(__p0_69, __p1_69, __p2_69, __p3_69) __extension__ ({ \
  int32x4_t __s0_69 = __p0_69; \
  int32x4_t __s1_69 = __p1_69; \
  int32x2_t __s2_69 = __p2_69; \
  int32x4_t __rev0_69;  __rev0_69 = __builtin_shufflevector(__s0_69, __s0_69, 3, 2, 1, 0); \
  int32x4_t __rev1_69;  __rev1_69 = __builtin_shufflevector(__s1_69, __s1_69, 3, 2, 1, 0); \
  int32x2_t __rev2_69;  __rev2_69 = __builtin_shufflevector(__s2_69, __s2_69, 1, 0); \
  int32x4_t __ret_69; \
  __ret_69 = __rev0_69 - __rev1_69 * __noswap_splatq_lane_s32(__rev2_69, __p3_69); \
  __ret_69 = __builtin_shufflevector(__ret_69, __ret_69, 3, 2, 1, 0); \
  __ret_69; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vmlsq_lane_s16(__p0_70, __p1_70, __p2_70, __p3_70) __extension__ ({ \
  int16x8_t __s0_70 = __p0_70; \
  int16x8_t __s1_70 = __p1_70; \
  int16x4_t __s2_70 = __p2_70; \
  int16x8_t __ret_70; \
  __ret_70 = __s0_70 - __s1_70 * splatq_lane_s16(__s2_70, __p3_70); \
  __ret_70; \
})
#else
#define vmlsq_lane_s16(__p0_71, __p1_71, __p2_71, __p3_71) __extension__ ({ \
  int16x8_t __s0_71 = __p0_71; \
  int16x8_t __s1_71 = __p1_71; \
  int16x4_t __s2_71 = __p2_71; \
  int16x8_t __rev0_71;  __rev0_71 = __builtin_shufflevector(__s0_71, __s0_71, 7, 6, 5, 4, 3, 2, 1, 0); \
  int16x8_t __rev1_71;  __rev1_71 = __builtin_shufflevector(__s1_71, __s1_71, 7, 6, 5, 4, 3, 2, 1, 0); \
  int16x4_t __rev2_71;  __rev2_71 = __builtin_shufflevector(__s2_71, __s2_71, 3, 2, 1, 0); \
  int16x8_t __ret_71; \
  __ret_71 = __rev0_71 - __rev1_71 * __noswap_splatq_lane_s16(__rev2_71, __p3_71); \
  __ret_71 = __builtin_shufflevector(__ret_71, __ret_71, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret_71; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vmls_lane_u32(__p0_72, __p1_72, __p2_72, __p3_72) __extension__ ({ \
  uint32x2_t __s0_72 = __p0_72; \
  uint32x2_t __s1_72 = __p1_72; \
  uint32x2_t __s2_72 = __p2_72; \
  uint32x2_t __ret_72; \
  __ret_72 = __s0_72 - __s1_72 * splat_lane_u32(__s2_72, __p3_72); \
  __ret_72; \
})
#else
#define vmls_lane_u32(__p0_73, __p1_73, __p2_73, __p3_73) __extension__ ({ \
  uint32x2_t __s0_73 = __p0_73; \
  uint32x2_t __s1_73 = __p1_73; \
  uint32x2_t __s2_73 = __p2_73; \
  uint32x2_t __rev0_73;  __rev0_73 = __builtin_shufflevector(__s0_73, __s0_73, 1, 0); \
  uint32x2_t __rev1_73;  __rev1_73 = __builtin_shufflevector(__s1_73, __s1_73, 1, 0); \
  uint32x2_t __rev2_73;  __rev2_73 = __builtin_shufflevector(__s2_73, __s2_73, 1, 0); \
  uint32x2_t __ret_73; \
  __ret_73 = __rev0_73 - __rev1_73 * __noswap_splat_lane_u32(__rev2_73, __p3_73); \
  __ret_73 = __builtin_shufflevector(__ret_73, __ret_73, 1, 0); \
  __ret_73; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vmls_lane_u16(__p0_74, __p1_74, __p2_74, __p3_74) __extension__ ({ \
  uint16x4_t __s0_74 = __p0_74; \
  uint16x4_t __s1_74 = __p1_74; \
  uint16x4_t __s2_74 = __p2_74; \
  uint16x4_t __ret_74; \
  __ret_74 = __s0_74 - __s1_74 * splat_lane_u16(__s2_74, __p3_74); \
  __ret_74; \
})
#else
#define vmls_lane_u16(__p0_75, __p1_75, __p2_75, __p3_75) __extension__ ({ \
  uint16x4_t __s0_75 = __p0_75; \
  uint16x4_t __s1_75 = __p1_75; \
  uint16x4_t __s2_75 = __p2_75; \
  uint16x4_t __rev0_75;  __rev0_75 = __builtin_shufflevector(__s0_75, __s0_75, 3, 2, 1, 0); \
  uint16x4_t __rev1_75;  __rev1_75 = __builtin_shufflevector(__s1_75, __s1_75, 3, 2, 1, 0); \
  uint16x4_t __rev2_75;  __rev2_75 = __builtin_shufflevector(__s2_75, __s2_75, 3, 2, 1, 0); \
  uint16x4_t __ret_75; \
  __ret_75 = __rev0_75 - __rev1_75 * __noswap_splat_lane_u16(__rev2_75, __p3_75); \
  __ret_75 = __builtin_shufflevector(__ret_75, __ret_75, 3, 2, 1, 0); \
  __ret_75; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vmls_lane_f32(__p0_76, __p1_76, __p2_76, __p3_76) __extension__ ({ \
  float32x2_t __s0_76 = __p0_76; \
  float32x2_t __s1_76 = __p1_76; \
  float32x2_t __s2_76 = __p2_76; \
  float32x2_t __ret_76; \
  __ret_76 = __s0_76 - __s1_76 * splat_lane_f32(__s2_76, __p3_76); \
  __ret_76; \
})
#else
#define vmls_lane_f32(__p0_77, __p1_77, __p2_77, __p3_77) __extension__ ({ \
  float32x2_t __s0_77 = __p0_77; \
  float32x2_t __s1_77 = __p1_77; \
  float32x2_t __s2_77 = __p2_77; \
  float32x2_t __rev0_77;  __rev0_77 = __builtin_shufflevector(__s0_77, __s0_77, 1, 0); \
  float32x2_t __rev1_77;  __rev1_77 = __builtin_shufflevector(__s1_77, __s1_77, 1, 0); \
  float32x2_t __rev2_77;  __rev2_77 = __builtin_shufflevector(__s2_77, __s2_77, 1, 0); \
  float32x2_t __ret_77; \
  __ret_77 = __rev0_77 - __rev1_77 * __noswap_splat_lane_f32(__rev2_77, __p3_77); \
  __ret_77 = __builtin_shufflevector(__ret_77, __ret_77, 1, 0); \
  __ret_77; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vmls_lane_s32(__p0_78, __p1_78, __p2_78, __p3_78) __extension__ ({ \
  int32x2_t __s0_78 = __p0_78; \
  int32x2_t __s1_78 = __p1_78; \
  int32x2_t __s2_78 = __p2_78; \
  int32x2_t __ret_78; \
  __ret_78 = __s0_78 - __s1_78 * splat_lane_s32(__s2_78, __p3_78); \
  __ret_78; \
})
#else
#define vmls_lane_s32(__p0_79, __p1_79, __p2_79, __p3_79) __extension__ ({ \
  int32x2_t __s0_79 = __p0_79; \
  int32x2_t __s1_79 = __p1_79; \
  int32x2_t __s2_79 = __p2_79; \
  int32x2_t __rev0_79;  __rev0_79 = __builtin_shufflevector(__s0_79, __s0_79, 1, 0); \
  int32x2_t __rev1_79;  __rev1_79 = __builtin_shufflevector(__s1_79, __s1_79, 1, 0); \
  int32x2_t __rev2_79;  __rev2_79 = __builtin_shufflevector(__s2_79, __s2_79, 1, 0); \
  int32x2_t __ret_79; \
  __ret_79 = __rev0_79 - __rev1_79 * __noswap_splat_lane_s32(__rev2_79, __p3_79); \
  __ret_79 = __builtin_shufflevector(__ret_79, __ret_79, 1, 0); \
  __ret_79; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vmls_lane_s16(__p0_80, __p1_80, __p2_80, __p3_80) __extension__ ({ \
  int16x4_t __s0_80 = __p0_80; \
  int16x4_t __s1_80 = __p1_80; \
  int16x4_t __s2_80 = __p2_80; \
  int16x4_t __ret_80; \
  __ret_80 = __s0_80 - __s1_80 * splat_lane_s16(__s2_80, __p3_80); \
  __ret_80; \
})
#else
#define vmls_lane_s16(__p0_81, __p1_81, __p2_81, __p3_81) __extension__ ({ \
  int16x4_t __s0_81 = __p0_81; \
  int16x4_t __s1_81 = __p1_81; \
  int16x4_t __s2_81 = __p2_81; \
  int16x4_t __rev0_81;  __rev0_81 = __builtin_shufflevector(__s0_81, __s0_81, 3, 2, 1, 0); \
  int16x4_t __rev1_81;  __rev1_81 = __builtin_shufflevector(__s1_81, __s1_81, 3, 2, 1, 0); \
  int16x4_t __rev2_81;  __rev2_81 = __builtin_shufflevector(__s2_81, __s2_81, 3, 2, 1, 0); \
  int16x4_t __ret_81; \
  __ret_81 = __rev0_81 - __rev1_81 * __noswap_splat_lane_s16(__rev2_81, __p3_81); \
  __ret_81 = __builtin_shufflevector(__ret_81, __ret_81, 3, 2, 1, 0); \
  __ret_81; \
})
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vmlsq_n_u32(uint32x4_t __p0, uint32x4_t __p1, uint32_t __p2) {
  uint32x4_t __ret;
  __ret = __p0 - __p1 * (uint32x4_t) {__p2, __p2, __p2, __p2};
  return __ret;
}
#else
__ai uint32x4_t vmlsq_n_u32(uint32x4_t __p0, uint32x4_t __p1, uint32_t __p2) {
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = __rev0 - __rev1 * (uint32x4_t) {__p2, __p2, __p2, __p2};
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x8_t vmlsq_n_u16(uint16x8_t __p0, uint16x8_t __p1, uint16_t __p2) {
  uint16x8_t __ret;
  __ret = __p0 - __p1 * (uint16x8_t) {__p2, __p2, __p2, __p2, __p2, __p2, __p2, __p2};
  return __ret;
}
#else
__ai uint16x8_t vmlsq_n_u16(uint16x8_t __p0, uint16x8_t __p1, uint16_t __p2) {
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __ret;
  __ret = __rev0 - __rev1 * (uint16x8_t) {__p2, __p2, __p2, __p2, __p2, __p2, __p2, __p2};
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai float32x4_t vmlsq_n_f32(float32x4_t __p0, float32x4_t __p1, float32_t __p2) {
  float32x4_t __ret;
  __ret = __p0 - __p1 * (float32x4_t) {__p2, __p2, __p2, __p2};
  return __ret;
}
#else
__ai float32x4_t vmlsq_n_f32(float32x4_t __p0, float32x4_t __p1, float32_t __p2) {
  float32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  float32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  float32x4_t __ret;
  __ret = __rev0 - __rev1 * (float32x4_t) {__p2, __p2, __p2, __p2};
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x4_t vmlsq_n_s32(int32x4_t __p0, int32x4_t __p1, int32_t __p2) {
  int32x4_t __ret;
  __ret = __p0 - __p1 * (int32x4_t) {__p2, __p2, __p2, __p2};
  return __ret;
}
#else
__ai int32x4_t vmlsq_n_s32(int32x4_t __p0, int32x4_t __p1, int32_t __p2) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int32x4_t __ret;
  __ret = __rev0 - __rev1 * (int32x4_t) {__p2, __p2, __p2, __p2};
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x8_t vmlsq_n_s16(int16x8_t __p0, int16x8_t __p1, int16_t __p2) {
  int16x8_t __ret;
  __ret = __p0 - __p1 * (int16x8_t) {__p2, __p2, __p2, __p2, __p2, __p2, __p2, __p2};
  return __ret;
}
#else
__ai int16x8_t vmlsq_n_s16(int16x8_t __p0, int16x8_t __p1, int16_t __p2) {
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __ret;
  __ret = __rev0 - __rev1 * (int16x8_t) {__p2, __p2, __p2, __p2, __p2, __p2, __p2, __p2};
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vmls_n_u32(uint32x2_t __p0, uint32x2_t __p1, uint32_t __p2) {
  uint32x2_t __ret;
  __ret = __p0 - __p1 * (uint32x2_t) {__p2, __p2};
  return __ret;
}
#else
__ai uint32x2_t vmls_n_u32(uint32x2_t __p0, uint32x2_t __p1, uint32_t __p2) {
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint32x2_t __ret;
  __ret = __rev0 - __rev1 * (uint32x2_t) {__p2, __p2};
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x4_t vmls_n_u16(uint16x4_t __p0, uint16x4_t __p1, uint16_t __p2) {
  uint16x4_t __ret;
  __ret = __p0 - __p1 * (uint16x4_t) {__p2, __p2, __p2, __p2};
  return __ret;
}
#else
__ai uint16x4_t vmls_n_u16(uint16x4_t __p0, uint16x4_t __p1, uint16_t __p2) {
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint16x4_t __ret;
  __ret = __rev0 - __rev1 * (uint16x4_t) {__p2, __p2, __p2, __p2};
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai float32x2_t vmls_n_f32(float32x2_t __p0, float32x2_t __p1, float32_t __p2) {
  float32x2_t __ret;
  __ret = __p0 - __p1 * (float32x2_t) {__p2, __p2};
  return __ret;
}
#else
__ai float32x2_t vmls_n_f32(float32x2_t __p0, float32x2_t __p1, float32_t __p2) {
  float32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  float32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  float32x2_t __ret;
  __ret = __rev0 - __rev1 * (float32x2_t) {__p2, __p2};
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x2_t vmls_n_s32(int32x2_t __p0, int32x2_t __p1, int32_t __p2) {
  int32x2_t __ret;
  __ret = __p0 - __p1 * (int32x2_t) {__p2, __p2};
  return __ret;
}
#else
__ai int32x2_t vmls_n_s32(int32x2_t __p0, int32x2_t __p1, int32_t __p2) {
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  int32x2_t __ret;
  __ret = __rev0 - __rev1 * (int32x2_t) {__p2, __p2};
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x4_t vmls_n_s16(int16x4_t __p0, int16x4_t __p1, int16_t __p2) {
  int16x4_t __ret;
  __ret = __p0 - __p1 * (int16x4_t) {__p2, __p2, __p2, __p2};
  return __ret;
}
#else
__ai int16x4_t vmls_n_s16(int16x4_t __p0, int16x4_t __p1, int16_t __p2) {
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int16x4_t __ret;
  __ret = __rev0 - __rev1 * (int16x4_t) {__p2, __p2, __p2, __p2};
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai poly8x8_t vmov_n_p8(poly8_t __p0) {
  poly8x8_t __ret;
  __ret = (poly8x8_t) {__p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0};
  return __ret;
}
#else
__ai poly8x8_t vmov_n_p8(poly8_t __p0) {
  poly8x8_t __ret;
  __ret = (poly8x8_t) {__p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0};
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai poly16x4_t vmov_n_p16(poly16_t __p0) {
  poly16x4_t __ret;
  __ret = (poly16x4_t) {__p0, __p0, __p0, __p0};
  return __ret;
}
#else
__ai poly16x4_t vmov_n_p16(poly16_t __p0) {
  poly16x4_t __ret;
  __ret = (poly16x4_t) {__p0, __p0, __p0, __p0};
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai poly8x16_t vmovq_n_p8(poly8_t __p0) {
  poly8x16_t __ret;
  __ret = (poly8x16_t) {__p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0};
  return __ret;
}
#else
__ai poly8x16_t vmovq_n_p8(poly8_t __p0) {
  poly8x16_t __ret;
  __ret = (poly8x16_t) {__p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0};
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai poly16x8_t vmovq_n_p16(poly16_t __p0) {
  poly16x8_t __ret;
  __ret = (poly16x8_t) {__p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0};
  return __ret;
}
#else
__ai poly16x8_t vmovq_n_p16(poly16_t __p0) {
  poly16x8_t __ret;
  __ret = (poly16x8_t) {__p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0};
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x16_t vmovq_n_u8(uint8_t __p0) {
  uint8x16_t __ret;
  __ret = (uint8x16_t) {__p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0};
  return __ret;
}
#else
__ai uint8x16_t vmovq_n_u8(uint8_t __p0) {
  uint8x16_t __ret;
  __ret = (uint8x16_t) {__p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0};
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vmovq_n_u32(uint32_t __p0) {
  uint32x4_t __ret;
  __ret = (uint32x4_t) {__p0, __p0, __p0, __p0};
  return __ret;
}
#else
__ai uint32x4_t vmovq_n_u32(uint32_t __p0) {
  uint32x4_t __ret;
  __ret = (uint32x4_t) {__p0, __p0, __p0, __p0};
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint64x2_t vmovq_n_u64(uint64_t __p0) {
  uint64x2_t __ret;
  __ret = (uint64x2_t) {__p0, __p0};
  return __ret;
}
#else
__ai uint64x2_t vmovq_n_u64(uint64_t __p0) {
  uint64x2_t __ret;
  __ret = (uint64x2_t) {__p0, __p0};
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x8_t vmovq_n_u16(uint16_t __p0) {
  uint16x8_t __ret;
  __ret = (uint16x8_t) {__p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0};
  return __ret;
}
#else
__ai uint16x8_t vmovq_n_u16(uint16_t __p0) {
  uint16x8_t __ret;
  __ret = (uint16x8_t) {__p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0};
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x16_t vmovq_n_s8(int8_t __p0) {
  int8x16_t __ret;
  __ret = (int8x16_t) {__p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0};
  return __ret;
}
#else
__ai int8x16_t vmovq_n_s8(int8_t __p0) {
  int8x16_t __ret;
  __ret = (int8x16_t) {__p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0};
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai float32x4_t vmovq_n_f32(float32_t __p0) {
  float32x4_t __ret;
  __ret = (float32x4_t) {__p0, __p0, __p0, __p0};
  return __ret;
}
#else
__ai float32x4_t vmovq_n_f32(float32_t __p0) {
  float32x4_t __ret;
  __ret = (float32x4_t) {__p0, __p0, __p0, __p0};
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
#define vmovq_n_f16(__p0) __extension__ ({ \
  float16_t __s0 = __p0; \
  float16x8_t __ret; \
  __ret = (float16x8_t) {__s0, __s0, __s0, __s0, __s0, __s0, __s0, __s0}; \
  __ret; \
})
#else
#define vmovq_n_f16(__p0) __extension__ ({ \
  float16_t __s0 = __p0; \
  float16x8_t __ret; \
  __ret = (float16x8_t) {__s0, __s0, __s0, __s0, __s0, __s0, __s0, __s0}; \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x4_t vmovq_n_s32(int32_t __p0) {
  int32x4_t __ret;
  __ret = (int32x4_t) {__p0, __p0, __p0, __p0};
  return __ret;
}
#else
__ai int32x4_t vmovq_n_s32(int32_t __p0) {
  int32x4_t __ret;
  __ret = (int32x4_t) {__p0, __p0, __p0, __p0};
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int64x2_t vmovq_n_s64(int64_t __p0) {
  int64x2_t __ret;
  __ret = (int64x2_t) {__p0, __p0};
  return __ret;
}
#else
__ai int64x2_t vmovq_n_s64(int64_t __p0) {
  int64x2_t __ret;
  __ret = (int64x2_t) {__p0, __p0};
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x8_t vmovq_n_s16(int16_t __p0) {
  int16x8_t __ret;
  __ret = (int16x8_t) {__p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0};
  return __ret;
}
#else
__ai int16x8_t vmovq_n_s16(int16_t __p0) {
  int16x8_t __ret;
  __ret = (int16x8_t) {__p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0};
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x8_t vmov_n_u8(uint8_t __p0) {
  uint8x8_t __ret;
  __ret = (uint8x8_t) {__p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0};
  return __ret;
}
#else
__ai uint8x8_t vmov_n_u8(uint8_t __p0) {
  uint8x8_t __ret;
  __ret = (uint8x8_t) {__p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0};
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vmov_n_u32(uint32_t __p0) {
  uint32x2_t __ret;
  __ret = (uint32x2_t) {__p0, __p0};
  return __ret;
}
#else
__ai uint32x2_t vmov_n_u32(uint32_t __p0) {
  uint32x2_t __ret;
  __ret = (uint32x2_t) {__p0, __p0};
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

__ai uint64x1_t vmov_n_u64(uint64_t __p0) {
  uint64x1_t __ret;
  __ret = (uint64x1_t) {__p0};
  return __ret;
}
#ifdef __LITTLE_ENDIAN__
__ai uint16x4_t vmov_n_u16(uint16_t __p0) {
  uint16x4_t __ret;
  __ret = (uint16x4_t) {__p0, __p0, __p0, __p0};
  return __ret;
}
#else
__ai uint16x4_t vmov_n_u16(uint16_t __p0) {
  uint16x4_t __ret;
  __ret = (uint16x4_t) {__p0, __p0, __p0, __p0};
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x8_t vmov_n_s8(int8_t __p0) {
  int8x8_t __ret;
  __ret = (int8x8_t) {__p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0};
  return __ret;
}
#else
__ai int8x8_t vmov_n_s8(int8_t __p0) {
  int8x8_t __ret;
  __ret = (int8x8_t) {__p0, __p0, __p0, __p0, __p0, __p0, __p0, __p0};
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai float32x2_t vmov_n_f32(float32_t __p0) {
  float32x2_t __ret;
  __ret = (float32x2_t) {__p0, __p0};
  return __ret;
}
#else
__ai float32x2_t vmov_n_f32(float32_t __p0) {
  float32x2_t __ret;
  __ret = (float32x2_t) {__p0, __p0};
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
#define vmov_n_f16(__p0) __extension__ ({ \
  float16_t __s0 = __p0; \
  float16x4_t __ret; \
  __ret = (float16x4_t) {__s0, __s0, __s0, __s0}; \
  __ret; \
})
#else
#define vmov_n_f16(__p0) __extension__ ({ \
  float16_t __s0 = __p0; \
  float16x4_t __ret; \
  __ret = (float16x4_t) {__s0, __s0, __s0, __s0}; \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x2_t vmov_n_s32(int32_t __p0) {
  int32x2_t __ret;
  __ret = (int32x2_t) {__p0, __p0};
  return __ret;
}
#else
__ai int32x2_t vmov_n_s32(int32_t __p0) {
  int32x2_t __ret;
  __ret = (int32x2_t) {__p0, __p0};
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

__ai int64x1_t vmov_n_s64(int64_t __p0) {
  int64x1_t __ret;
  __ret = (int64x1_t) {__p0};
  return __ret;
}
#ifdef __LITTLE_ENDIAN__
__ai int16x4_t vmov_n_s16(int16_t __p0) {
  int16x4_t __ret;
  __ret = (int16x4_t) {__p0, __p0, __p0, __p0};
  return __ret;
}
#else
__ai int16x4_t vmov_n_s16(int16_t __p0) {
  int16x4_t __ret;
  __ret = (int16x4_t) {__p0, __p0, __p0, __p0};
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x8_t vmovl_u8(uint8x8_t __p0) {
  uint16x8_t __ret;
  __ret = (uint16x8_t) __builtin_neon_vmovl_v((int8x8_t)__p0, 49);
  return __ret;
}
#else
__ai uint16x8_t vmovl_u8(uint8x8_t __p0) {
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __ret;
  __ret = (uint16x8_t) __builtin_neon_vmovl_v((int8x8_t)__rev0, 49);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
__ai uint16x8_t __noswap_vmovl_u8(uint8x8_t __p0) {
  uint16x8_t __ret;
  __ret = (uint16x8_t) __builtin_neon_vmovl_v((int8x8_t)__p0, 49);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint64x2_t vmovl_u32(uint32x2_t __p0) {
  uint64x2_t __ret;
  __ret = (uint64x2_t) __builtin_neon_vmovl_v((int8x8_t)__p0, 51);
  return __ret;
}
#else
__ai uint64x2_t vmovl_u32(uint32x2_t __p0) {
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint64x2_t __ret;
  __ret = (uint64x2_t) __builtin_neon_vmovl_v((int8x8_t)__rev0, 51);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
__ai uint64x2_t __noswap_vmovl_u32(uint32x2_t __p0) {
  uint64x2_t __ret;
  __ret = (uint64x2_t) __builtin_neon_vmovl_v((int8x8_t)__p0, 51);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vmovl_u16(uint16x4_t __p0) {
  uint32x4_t __ret;
  __ret = (uint32x4_t) __builtin_neon_vmovl_v((int8x8_t)__p0, 50);
  return __ret;
}
#else
__ai uint32x4_t vmovl_u16(uint16x4_t __p0) {
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = (uint32x4_t) __builtin_neon_vmovl_v((int8x8_t)__rev0, 50);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
__ai uint32x4_t __noswap_vmovl_u16(uint16x4_t __p0) {
  uint32x4_t __ret;
  __ret = (uint32x4_t) __builtin_neon_vmovl_v((int8x8_t)__p0, 50);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x8_t vmovl_s8(int8x8_t __p0) {
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vmovl_v((int8x8_t)__p0, 33);
  return __ret;
}
#else
__ai int16x8_t vmovl_s8(int8x8_t __p0) {
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vmovl_v((int8x8_t)__rev0, 33);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
__ai int16x8_t __noswap_vmovl_s8(int8x8_t __p0) {
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vmovl_v((int8x8_t)__p0, 33);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int64x2_t vmovl_s32(int32x2_t __p0) {
  int64x2_t __ret;
  __ret = (int64x2_t) __builtin_neon_vmovl_v((int8x8_t)__p0, 35);
  return __ret;
}
#else
__ai int64x2_t vmovl_s32(int32x2_t __p0) {
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int64x2_t __ret;
  __ret = (int64x2_t) __builtin_neon_vmovl_v((int8x8_t)__rev0, 35);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
__ai int64x2_t __noswap_vmovl_s32(int32x2_t __p0) {
  int64x2_t __ret;
  __ret = (int64x2_t) __builtin_neon_vmovl_v((int8x8_t)__p0, 35);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x4_t vmovl_s16(int16x4_t __p0) {
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vmovl_v((int8x8_t)__p0, 34);
  return __ret;
}
#else
__ai int32x4_t vmovl_s16(int16x4_t __p0) {
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vmovl_v((int8x8_t)__rev0, 34);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
__ai int32x4_t __noswap_vmovl_s16(int16x4_t __p0) {
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vmovl_v((int8x8_t)__p0, 34);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x4_t vmovn_u32(uint32x4_t __p0) {
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vmovn_v((int8x16_t)__p0, 17);
  return __ret;
}
#else
__ai uint16x4_t vmovn_u32(uint32x4_t __p0) {
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vmovn_v((int8x16_t)__rev0, 17);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
__ai uint16x4_t __noswap_vmovn_u32(uint32x4_t __p0) {
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vmovn_v((int8x16_t)__p0, 17);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vmovn_u64(uint64x2_t __p0) {
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vmovn_v((int8x16_t)__p0, 18);
  return __ret;
}
#else
__ai uint32x2_t vmovn_u64(uint64x2_t __p0) {
  uint64x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vmovn_v((int8x16_t)__rev0, 18);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
__ai uint32x2_t __noswap_vmovn_u64(uint64x2_t __p0) {
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vmovn_v((int8x16_t)__p0, 18);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x8_t vmovn_u16(uint16x8_t __p0) {
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vmovn_v((int8x16_t)__p0, 16);
  return __ret;
}
#else
__ai uint8x8_t vmovn_u16(uint16x8_t __p0) {
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vmovn_v((int8x16_t)__rev0, 16);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
__ai uint8x8_t __noswap_vmovn_u16(uint16x8_t __p0) {
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vmovn_v((int8x16_t)__p0, 16);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x4_t vmovn_s32(int32x4_t __p0) {
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vmovn_v((int8x16_t)__p0, 1);
  return __ret;
}
#else
__ai int16x4_t vmovn_s32(int32x4_t __p0) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vmovn_v((int8x16_t)__rev0, 1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
__ai int16x4_t __noswap_vmovn_s32(int32x4_t __p0) {
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vmovn_v((int8x16_t)__p0, 1);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x2_t vmovn_s64(int64x2_t __p0) {
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vmovn_v((int8x16_t)__p0, 2);
  return __ret;
}
#else
__ai int32x2_t vmovn_s64(int64x2_t __p0) {
  int64x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vmovn_v((int8x16_t)__rev0, 2);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
__ai int32x2_t __noswap_vmovn_s64(int64x2_t __p0) {
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vmovn_v((int8x16_t)__p0, 2);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x8_t vmovn_s16(int16x8_t __p0) {
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vmovn_v((int8x16_t)__p0, 0);
  return __ret;
}
#else
__ai int8x8_t vmovn_s16(int16x8_t __p0) {
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vmovn_v((int8x16_t)__rev0, 0);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
__ai int8x8_t __noswap_vmovn_s16(int16x8_t __p0) {
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vmovn_v((int8x16_t)__p0, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x16_t vmulq_u8(uint8x16_t __p0, uint8x16_t __p1) {
  uint8x16_t __ret;
  __ret = __p0 * __p1;
  return __ret;
}
#else
__ai uint8x16_t vmulq_u8(uint8x16_t __p0, uint8x16_t __p1) {
  uint8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __ret;
  __ret = __rev0 * __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vmulq_u32(uint32x4_t __p0, uint32x4_t __p1) {
  uint32x4_t __ret;
  __ret = __p0 * __p1;
  return __ret;
}
#else
__ai uint32x4_t vmulq_u32(uint32x4_t __p0, uint32x4_t __p1) {
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = __rev0 * __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x8_t vmulq_u16(uint16x8_t __p0, uint16x8_t __p1) {
  uint16x8_t __ret;
  __ret = __p0 * __p1;
  return __ret;
}
#else
__ai uint16x8_t vmulq_u16(uint16x8_t __p0, uint16x8_t __p1) {
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __ret;
  __ret = __rev0 * __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x16_t vmulq_s8(int8x16_t __p0, int8x16_t __p1) {
  int8x16_t __ret;
  __ret = __p0 * __p1;
  return __ret;
}
#else
__ai int8x16_t vmulq_s8(int8x16_t __p0, int8x16_t __p1) {
  int8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __ret;
  __ret = __rev0 * __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai float32x4_t vmulq_f32(float32x4_t __p0, float32x4_t __p1) {
  float32x4_t __ret;
  __ret = __p0 * __p1;
  return __ret;
}
#else
__ai float32x4_t vmulq_f32(float32x4_t __p0, float32x4_t __p1) {
  float32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  float32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  float32x4_t __ret;
  __ret = __rev0 * __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x4_t vmulq_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __ret;
  __ret = __p0 * __p1;
  return __ret;
}
#else
__ai int32x4_t vmulq_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int32x4_t __ret;
  __ret = __rev0 * __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x8_t vmulq_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __ret;
  __ret = __p0 * __p1;
  return __ret;
}
#else
__ai int16x8_t vmulq_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __ret;
  __ret = __rev0 * __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x8_t vmul_u8(uint8x8_t __p0, uint8x8_t __p1) {
  uint8x8_t __ret;
  __ret = __p0 * __p1;
  return __ret;
}
#else
__ai uint8x8_t vmul_u8(uint8x8_t __p0, uint8x8_t __p1) {
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __ret;
  __ret = __rev0 * __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vmul_u32(uint32x2_t __p0, uint32x2_t __p1) {
  uint32x2_t __ret;
  __ret = __p0 * __p1;
  return __ret;
}
#else
__ai uint32x2_t vmul_u32(uint32x2_t __p0, uint32x2_t __p1) {
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint32x2_t __ret;
  __ret = __rev0 * __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x4_t vmul_u16(uint16x4_t __p0, uint16x4_t __p1) {
  uint16x4_t __ret;
  __ret = __p0 * __p1;
  return __ret;
}
#else
__ai uint16x4_t vmul_u16(uint16x4_t __p0, uint16x4_t __p1) {
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint16x4_t __ret;
  __ret = __rev0 * __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x8_t vmul_s8(int8x8_t __p0, int8x8_t __p1) {
  int8x8_t __ret;
  __ret = __p0 * __p1;
  return __ret;
}
#else
__ai int8x8_t vmul_s8(int8x8_t __p0, int8x8_t __p1) {
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __ret;
  __ret = __rev0 * __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai float32x2_t vmul_f32(float32x2_t __p0, float32x2_t __p1) {
  float32x2_t __ret;
  __ret = __p0 * __p1;
  return __ret;
}
#else
__ai float32x2_t vmul_f32(float32x2_t __p0, float32x2_t __p1) {
  float32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  float32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  float32x2_t __ret;
  __ret = __rev0 * __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x2_t vmul_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __ret;
  __ret = __p0 * __p1;
  return __ret;
}
#else
__ai int32x2_t vmul_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  int32x2_t __ret;
  __ret = __rev0 * __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x4_t vmul_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __ret;
  __ret = __p0 * __p1;
  return __ret;
}
#else
__ai int16x4_t vmul_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int16x4_t __ret;
  __ret = __rev0 * __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai poly8x8_t vmul_p8(poly8x8_t __p0, poly8x8_t __p1) {
  poly8x8_t __ret;
  __ret = (poly8x8_t) __builtin_neon_vmul_v((int8x8_t)__p0, (int8x8_t)__p1, 4);
  return __ret;
}
#else
__ai poly8x8_t vmul_p8(poly8x8_t __p0, poly8x8_t __p1) {
  poly8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  poly8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  poly8x8_t __ret;
  __ret = (poly8x8_t) __builtin_neon_vmul_v((int8x8_t)__rev0, (int8x8_t)__rev1, 4);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai poly8x16_t vmulq_p8(poly8x16_t __p0, poly8x16_t __p1) {
  poly8x16_t __ret;
  __ret = (poly8x16_t) __builtin_neon_vmulq_v((int8x16_t)__p0, (int8x16_t)__p1, 36);
  return __ret;
}
#else
__ai poly8x16_t vmulq_p8(poly8x16_t __p0, poly8x16_t __p1) {
  poly8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  poly8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  poly8x16_t __ret;
  __ret = (poly8x16_t) __builtin_neon_vmulq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 36);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
#define vmulq_lane_u32(__p0_82, __p1_82, __p2_82) __extension__ ({ \
  uint32x4_t __s0_82 = __p0_82; \
  uint32x2_t __s1_82 = __p1_82; \
  uint32x4_t __ret_82; \
  __ret_82 = __s0_82 * splatq_lane_u32(__s1_82, __p2_82); \
  __ret_82; \
})
#else
#define vmulq_lane_u32(__p0_83, __p1_83, __p2_83) __extension__ ({ \
  uint32x4_t __s0_83 = __p0_83; \
  uint32x2_t __s1_83 = __p1_83; \
  uint32x4_t __rev0_83;  __rev0_83 = __builtin_shufflevector(__s0_83, __s0_83, 3, 2, 1, 0); \
  uint32x2_t __rev1_83;  __rev1_83 = __builtin_shufflevector(__s1_83, __s1_83, 1, 0); \
  uint32x4_t __ret_83; \
  __ret_83 = __rev0_83 * __noswap_splatq_lane_u32(__rev1_83, __p2_83); \
  __ret_83 = __builtin_shufflevector(__ret_83, __ret_83, 3, 2, 1, 0); \
  __ret_83; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vmulq_lane_u16(__p0_84, __p1_84, __p2_84) __extension__ ({ \
  uint16x8_t __s0_84 = __p0_84; \
  uint16x4_t __s1_84 = __p1_84; \
  uint16x8_t __ret_84; \
  __ret_84 = __s0_84 * splatq_lane_u16(__s1_84, __p2_84); \
  __ret_84; \
})
#else
#define vmulq_lane_u16(__p0_85, __p1_85, __p2_85) __extension__ ({ \
  uint16x8_t __s0_85 = __p0_85; \
  uint16x4_t __s1_85 = __p1_85; \
  uint16x8_t __rev0_85;  __rev0_85 = __builtin_shufflevector(__s0_85, __s0_85, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint16x4_t __rev1_85;  __rev1_85 = __builtin_shufflevector(__s1_85, __s1_85, 3, 2, 1, 0); \
  uint16x8_t __ret_85; \
  __ret_85 = __rev0_85 * __noswap_splatq_lane_u16(__rev1_85, __p2_85); \
  __ret_85 = __builtin_shufflevector(__ret_85, __ret_85, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret_85; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vmulq_lane_f32(__p0_86, __p1_86, __p2_86) __extension__ ({ \
  float32x4_t __s0_86 = __p0_86; \
  float32x2_t __s1_86 = __p1_86; \
  float32x4_t __ret_86; \
  __ret_86 = __s0_86 * splatq_lane_f32(__s1_86, __p2_86); \
  __ret_86; \
})
#else
#define vmulq_lane_f32(__p0_87, __p1_87, __p2_87) __extension__ ({ \
  float32x4_t __s0_87 = __p0_87; \
  float32x2_t __s1_87 = __p1_87; \
  float32x4_t __rev0_87;  __rev0_87 = __builtin_shufflevector(__s0_87, __s0_87, 3, 2, 1, 0); \
  float32x2_t __rev1_87;  __rev1_87 = __builtin_shufflevector(__s1_87, __s1_87, 1, 0); \
  float32x4_t __ret_87; \
  __ret_87 = __rev0_87 * __noswap_splatq_lane_f32(__rev1_87, __p2_87); \
  __ret_87 = __builtin_shufflevector(__ret_87, __ret_87, 3, 2, 1, 0); \
  __ret_87; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vmulq_lane_s32(__p0_88, __p1_88, __p2_88) __extension__ ({ \
  int32x4_t __s0_88 = __p0_88; \
  int32x2_t __s1_88 = __p1_88; \
  int32x4_t __ret_88; \
  __ret_88 = __s0_88 * splatq_lane_s32(__s1_88, __p2_88); \
  __ret_88; \
})
#else
#define vmulq_lane_s32(__p0_89, __p1_89, __p2_89) __extension__ ({ \
  int32x4_t __s0_89 = __p0_89; \
  int32x2_t __s1_89 = __p1_89; \
  int32x4_t __rev0_89;  __rev0_89 = __builtin_shufflevector(__s0_89, __s0_89, 3, 2, 1, 0); \
  int32x2_t __rev1_89;  __rev1_89 = __builtin_shufflevector(__s1_89, __s1_89, 1, 0); \
  int32x4_t __ret_89; \
  __ret_89 = __rev0_89 * __noswap_splatq_lane_s32(__rev1_89, __p2_89); \
  __ret_89 = __builtin_shufflevector(__ret_89, __ret_89, 3, 2, 1, 0); \
  __ret_89; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vmulq_lane_s16(__p0_90, __p1_90, __p2_90) __extension__ ({ \
  int16x8_t __s0_90 = __p0_90; \
  int16x4_t __s1_90 = __p1_90; \
  int16x8_t __ret_90; \
  __ret_90 = __s0_90 * splatq_lane_s16(__s1_90, __p2_90); \
  __ret_90; \
})
#else
#define vmulq_lane_s16(__p0_91, __p1_91, __p2_91) __extension__ ({ \
  int16x8_t __s0_91 = __p0_91; \
  int16x4_t __s1_91 = __p1_91; \
  int16x8_t __rev0_91;  __rev0_91 = __builtin_shufflevector(__s0_91, __s0_91, 7, 6, 5, 4, 3, 2, 1, 0); \
  int16x4_t __rev1_91;  __rev1_91 = __builtin_shufflevector(__s1_91, __s1_91, 3, 2, 1, 0); \
  int16x8_t __ret_91; \
  __ret_91 = __rev0_91 * __noswap_splatq_lane_s16(__rev1_91, __p2_91); \
  __ret_91 = __builtin_shufflevector(__ret_91, __ret_91, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret_91; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vmul_lane_u32(__p0_92, __p1_92, __p2_92) __extension__ ({ \
  uint32x2_t __s0_92 = __p0_92; \
  uint32x2_t __s1_92 = __p1_92; \
  uint32x2_t __ret_92; \
  __ret_92 = __s0_92 * splat_lane_u32(__s1_92, __p2_92); \
  __ret_92; \
})
#else
#define vmul_lane_u32(__p0_93, __p1_93, __p2_93) __extension__ ({ \
  uint32x2_t __s0_93 = __p0_93; \
  uint32x2_t __s1_93 = __p1_93; \
  uint32x2_t __rev0_93;  __rev0_93 = __builtin_shufflevector(__s0_93, __s0_93, 1, 0); \
  uint32x2_t __rev1_93;  __rev1_93 = __builtin_shufflevector(__s1_93, __s1_93, 1, 0); \
  uint32x2_t __ret_93; \
  __ret_93 = __rev0_93 * __noswap_splat_lane_u32(__rev1_93, __p2_93); \
  __ret_93 = __builtin_shufflevector(__ret_93, __ret_93, 1, 0); \
  __ret_93; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vmul_lane_u16(__p0_94, __p1_94, __p2_94) __extension__ ({ \
  uint16x4_t __s0_94 = __p0_94; \
  uint16x4_t __s1_94 = __p1_94; \
  uint16x4_t __ret_94; \
  __ret_94 = __s0_94 * splat_lane_u16(__s1_94, __p2_94); \
  __ret_94; \
})
#else
#define vmul_lane_u16(__p0_95, __p1_95, __p2_95) __extension__ ({ \
  uint16x4_t __s0_95 = __p0_95; \
  uint16x4_t __s1_95 = __p1_95; \
  uint16x4_t __rev0_95;  __rev0_95 = __builtin_shufflevector(__s0_95, __s0_95, 3, 2, 1, 0); \
  uint16x4_t __rev1_95;  __rev1_95 = __builtin_shufflevector(__s1_95, __s1_95, 3, 2, 1, 0); \
  uint16x4_t __ret_95; \
  __ret_95 = __rev0_95 * __noswap_splat_lane_u16(__rev1_95, __p2_95); \
  __ret_95 = __builtin_shufflevector(__ret_95, __ret_95, 3, 2, 1, 0); \
  __ret_95; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vmul_lane_f32(__p0_96, __p1_96, __p2_96) __extension__ ({ \
  float32x2_t __s0_96 = __p0_96; \
  float32x2_t __s1_96 = __p1_96; \
  float32x2_t __ret_96; \
  __ret_96 = __s0_96 * splat_lane_f32(__s1_96, __p2_96); \
  __ret_96; \
})
#else
#define vmul_lane_f32(__p0_97, __p1_97, __p2_97) __extension__ ({ \
  float32x2_t __s0_97 = __p0_97; \
  float32x2_t __s1_97 = __p1_97; \
  float32x2_t __rev0_97;  __rev0_97 = __builtin_shufflevector(__s0_97, __s0_97, 1, 0); \
  float32x2_t __rev1_97;  __rev1_97 = __builtin_shufflevector(__s1_97, __s1_97, 1, 0); \
  float32x2_t __ret_97; \
  __ret_97 = __rev0_97 * __noswap_splat_lane_f32(__rev1_97, __p2_97); \
  __ret_97 = __builtin_shufflevector(__ret_97, __ret_97, 1, 0); \
  __ret_97; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vmul_lane_s32(__p0_98, __p1_98, __p2_98) __extension__ ({ \
  int32x2_t __s0_98 = __p0_98; \
  int32x2_t __s1_98 = __p1_98; \
  int32x2_t __ret_98; \
  __ret_98 = __s0_98 * splat_lane_s32(__s1_98, __p2_98); \
  __ret_98; \
})
#else
#define vmul_lane_s32(__p0_99, __p1_99, __p2_99) __extension__ ({ \
  int32x2_t __s0_99 = __p0_99; \
  int32x2_t __s1_99 = __p1_99; \
  int32x2_t __rev0_99;  __rev0_99 = __builtin_shufflevector(__s0_99, __s0_99, 1, 0); \
  int32x2_t __rev1_99;  __rev1_99 = __builtin_shufflevector(__s1_99, __s1_99, 1, 0); \
  int32x2_t __ret_99; \
  __ret_99 = __rev0_99 * __noswap_splat_lane_s32(__rev1_99, __p2_99); \
  __ret_99 = __builtin_shufflevector(__ret_99, __ret_99, 1, 0); \
  __ret_99; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vmul_lane_s16(__p0_100, __p1_100, __p2_100) __extension__ ({ \
  int16x4_t __s0_100 = __p0_100; \
  int16x4_t __s1_100 = __p1_100; \
  int16x4_t __ret_100; \
  __ret_100 = __s0_100 * splat_lane_s16(__s1_100, __p2_100); \
  __ret_100; \
})
#else
#define vmul_lane_s16(__p0_101, __p1_101, __p2_101) __extension__ ({ \
  int16x4_t __s0_101 = __p0_101; \
  int16x4_t __s1_101 = __p1_101; \
  int16x4_t __rev0_101;  __rev0_101 = __builtin_shufflevector(__s0_101, __s0_101, 3, 2, 1, 0); \
  int16x4_t __rev1_101;  __rev1_101 = __builtin_shufflevector(__s1_101, __s1_101, 3, 2, 1, 0); \
  int16x4_t __ret_101; \
  __ret_101 = __rev0_101 * __noswap_splat_lane_s16(__rev1_101, __p2_101); \
  __ret_101 = __builtin_shufflevector(__ret_101, __ret_101, 3, 2, 1, 0); \
  __ret_101; \
})
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vmulq_n_u32(uint32x4_t __p0, uint32_t __p1) {
  uint32x4_t __ret;
  __ret = __p0 * (uint32x4_t) {__p1, __p1, __p1, __p1};
  return __ret;
}
#else
__ai uint32x4_t vmulq_n_u32(uint32x4_t __p0, uint32_t __p1) {
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = __rev0 * (uint32x4_t) {__p1, __p1, __p1, __p1};
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x8_t vmulq_n_u16(uint16x8_t __p0, uint16_t __p1) {
  uint16x8_t __ret;
  __ret = __p0 * (uint16x8_t) {__p1, __p1, __p1, __p1, __p1, __p1, __p1, __p1};
  return __ret;
}
#else
__ai uint16x8_t vmulq_n_u16(uint16x8_t __p0, uint16_t __p1) {
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __ret;
  __ret = __rev0 * (uint16x8_t) {__p1, __p1, __p1, __p1, __p1, __p1, __p1, __p1};
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai float32x4_t vmulq_n_f32(float32x4_t __p0, float32_t __p1) {
  float32x4_t __ret;
  __ret = __p0 * (float32x4_t) {__p1, __p1, __p1, __p1};
  return __ret;
}
#else
__ai float32x4_t vmulq_n_f32(float32x4_t __p0, float32_t __p1) {
  float32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  float32x4_t __ret;
  __ret = __rev0 * (float32x4_t) {__p1, __p1, __p1, __p1};
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x4_t vmulq_n_s32(int32x4_t __p0, int32_t __p1) {
  int32x4_t __ret;
  __ret = __p0 * (int32x4_t) {__p1, __p1, __p1, __p1};
  return __ret;
}
#else
__ai int32x4_t vmulq_n_s32(int32x4_t __p0, int32_t __p1) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x4_t __ret;
  __ret = __rev0 * (int32x4_t) {__p1, __p1, __p1, __p1};
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x8_t vmulq_n_s16(int16x8_t __p0, int16_t __p1) {
  int16x8_t __ret;
  __ret = __p0 * (int16x8_t) {__p1, __p1, __p1, __p1, __p1, __p1, __p1, __p1};
  return __ret;
}
#else
__ai int16x8_t vmulq_n_s16(int16x8_t __p0, int16_t __p1) {
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __ret;
  __ret = __rev0 * (int16x8_t) {__p1, __p1, __p1, __p1, __p1, __p1, __p1, __p1};
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vmul_n_u32(uint32x2_t __p0, uint32_t __p1) {
  uint32x2_t __ret;
  __ret = __p0 * (uint32x2_t) {__p1, __p1};
  return __ret;
}
#else
__ai uint32x2_t vmul_n_u32(uint32x2_t __p0, uint32_t __p1) {
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint32x2_t __ret;
  __ret = __rev0 * (uint32x2_t) {__p1, __p1};
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x4_t vmul_n_u16(uint16x4_t __p0, uint16_t __p1) {
  uint16x4_t __ret;
  __ret = __p0 * (uint16x4_t) {__p1, __p1, __p1, __p1};
  return __ret;
}
#else
__ai uint16x4_t vmul_n_u16(uint16x4_t __p0, uint16_t __p1) {
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint16x4_t __ret;
  __ret = __rev0 * (uint16x4_t) {__p1, __p1, __p1, __p1};
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai float32x2_t vmul_n_f32(float32x2_t __p0, float32_t __p1) {
  float32x2_t __ret;
  __ret = __p0 * (float32x2_t) {__p1, __p1};
  return __ret;
}
#else
__ai float32x2_t vmul_n_f32(float32x2_t __p0, float32_t __p1) {
  float32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  float32x2_t __ret;
  __ret = __rev0 * (float32x2_t) {__p1, __p1};
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x2_t vmul_n_s32(int32x2_t __p0, int32_t __p1) {
  int32x2_t __ret;
  __ret = __p0 * (int32x2_t) {__p1, __p1};
  return __ret;
}
#else
__ai int32x2_t vmul_n_s32(int32x2_t __p0, int32_t __p1) {
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __ret;
  __ret = __rev0 * (int32x2_t) {__p1, __p1};
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x4_t vmul_n_s16(int16x4_t __p0, int16_t __p1) {
  int16x4_t __ret;
  __ret = __p0 * (int16x4_t) {__p1, __p1, __p1, __p1};
  return __ret;
}
#else
__ai int16x4_t vmul_n_s16(int16x4_t __p0, int16_t __p1) {
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __ret;
  __ret = __rev0 * (int16x4_t) {__p1, __p1, __p1, __p1};
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai poly16x8_t vmull_p8(poly8x8_t __p0, poly8x8_t __p1) {
  poly16x8_t __ret;
  __ret = (poly16x8_t) __builtin_neon_vmull_v((int8x8_t)__p0, (int8x8_t)__p1, 37);
  return __ret;
}
#else
__ai poly16x8_t vmull_p8(poly8x8_t __p0, poly8x8_t __p1) {
  poly8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  poly8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  poly16x8_t __ret;
  __ret = (poly16x8_t) __builtin_neon_vmull_v((int8x8_t)__rev0, (int8x8_t)__rev1, 37);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
__ai poly16x8_t __noswap_vmull_p8(poly8x8_t __p0, poly8x8_t __p1) {
  poly16x8_t __ret;
  __ret = (poly16x8_t) __builtin_neon_vmull_v((int8x8_t)__p0, (int8x8_t)__p1, 37);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x8_t vmull_u8(uint8x8_t __p0, uint8x8_t __p1) {
  uint16x8_t __ret;
  __ret = (uint16x8_t) __builtin_neon_vmull_v((int8x8_t)__p0, (int8x8_t)__p1, 49);
  return __ret;
}
#else
__ai uint16x8_t vmull_u8(uint8x8_t __p0, uint8x8_t __p1) {
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __ret;
  __ret = (uint16x8_t) __builtin_neon_vmull_v((int8x8_t)__rev0, (int8x8_t)__rev1, 49);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
__ai uint16x8_t __noswap_vmull_u8(uint8x8_t __p0, uint8x8_t __p1) {
  uint16x8_t __ret;
  __ret = (uint16x8_t) __builtin_neon_vmull_v((int8x8_t)__p0, (int8x8_t)__p1, 49);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint64x2_t vmull_u32(uint32x2_t __p0, uint32x2_t __p1) {
  uint64x2_t __ret;
  __ret = (uint64x2_t) __builtin_neon_vmull_v((int8x8_t)__p0, (int8x8_t)__p1, 51);
  return __ret;
}
#else
__ai uint64x2_t vmull_u32(uint32x2_t __p0, uint32x2_t __p1) {
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint64x2_t __ret;
  __ret = (uint64x2_t) __builtin_neon_vmull_v((int8x8_t)__rev0, (int8x8_t)__rev1, 51);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
__ai uint64x2_t __noswap_vmull_u32(uint32x2_t __p0, uint32x2_t __p1) {
  uint64x2_t __ret;
  __ret = (uint64x2_t) __builtin_neon_vmull_v((int8x8_t)__p0, (int8x8_t)__p1, 51);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vmull_u16(uint16x4_t __p0, uint16x4_t __p1) {
  uint32x4_t __ret;
  __ret = (uint32x4_t) __builtin_neon_vmull_v((int8x8_t)__p0, (int8x8_t)__p1, 50);
  return __ret;
}
#else
__ai uint32x4_t vmull_u16(uint16x4_t __p0, uint16x4_t __p1) {
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = (uint32x4_t) __builtin_neon_vmull_v((int8x8_t)__rev0, (int8x8_t)__rev1, 50);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
__ai uint32x4_t __noswap_vmull_u16(uint16x4_t __p0, uint16x4_t __p1) {
  uint32x4_t __ret;
  __ret = (uint32x4_t) __builtin_neon_vmull_v((int8x8_t)__p0, (int8x8_t)__p1, 50);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x8_t vmull_s8(int8x8_t __p0, int8x8_t __p1) {
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vmull_v((int8x8_t)__p0, (int8x8_t)__p1, 33);
  return __ret;
}
#else
__ai int16x8_t vmull_s8(int8x8_t __p0, int8x8_t __p1) {
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vmull_v((int8x8_t)__rev0, (int8x8_t)__rev1, 33);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
__ai int16x8_t __noswap_vmull_s8(int8x8_t __p0, int8x8_t __p1) {
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vmull_v((int8x8_t)__p0, (int8x8_t)__p1, 33);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int64x2_t vmull_s32(int32x2_t __p0, int32x2_t __p1) {
  int64x2_t __ret;
  __ret = (int64x2_t) __builtin_neon_vmull_v((int8x8_t)__p0, (int8x8_t)__p1, 35);
  return __ret;
}
#else
__ai int64x2_t vmull_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  int64x2_t __ret;
  __ret = (int64x2_t) __builtin_neon_vmull_v((int8x8_t)__rev0, (int8x8_t)__rev1, 35);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
__ai int64x2_t __noswap_vmull_s32(int32x2_t __p0, int32x2_t __p1) {
  int64x2_t __ret;
  __ret = (int64x2_t) __builtin_neon_vmull_v((int8x8_t)__p0, (int8x8_t)__p1, 35);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x4_t vmull_s16(int16x4_t __p0, int16x4_t __p1) {
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vmull_v((int8x8_t)__p0, (int8x8_t)__p1, 34);
  return __ret;
}
#else
__ai int32x4_t vmull_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vmull_v((int8x8_t)__rev0, (int8x8_t)__rev1, 34);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
__ai int32x4_t __noswap_vmull_s16(int16x4_t __p0, int16x4_t __p1) {
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vmull_v((int8x8_t)__p0, (int8x8_t)__p1, 34);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
#define vmull_lane_u32(__p0_102, __p1_102, __p2_102) __extension__ ({ \
  uint32x2_t __s0_102 = __p0_102; \
  uint32x2_t __s1_102 = __p1_102; \
  uint64x2_t __ret_102; \
  __ret_102 = vmull_u32(__s0_102, splat_lane_u32(__s1_102, __p2_102)); \
  __ret_102; \
})
#else
#define vmull_lane_u32(__p0_103, __p1_103, __p2_103) __extension__ ({ \
  uint32x2_t __s0_103 = __p0_103; \
  uint32x2_t __s1_103 = __p1_103; \
  uint32x2_t __rev0_103;  __rev0_103 = __builtin_shufflevector(__s0_103, __s0_103, 1, 0); \
  uint32x2_t __rev1_103;  __rev1_103 = __builtin_shufflevector(__s1_103, __s1_103, 1, 0); \
  uint64x2_t __ret_103; \
  __ret_103 = __noswap_vmull_u32(__rev0_103, __noswap_splat_lane_u32(__rev1_103, __p2_103)); \
  __ret_103 = __builtin_shufflevector(__ret_103, __ret_103, 1, 0); \
  __ret_103; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vmull_lane_u16(__p0_104, __p1_104, __p2_104) __extension__ ({ \
  uint16x4_t __s0_104 = __p0_104; \
  uint16x4_t __s1_104 = __p1_104; \
  uint32x4_t __ret_104; \
  __ret_104 = vmull_u16(__s0_104, splat_lane_u16(__s1_104, __p2_104)); \
  __ret_104; \
})
#else
#define vmull_lane_u16(__p0_105, __p1_105, __p2_105) __extension__ ({ \
  uint16x4_t __s0_105 = __p0_105; \
  uint16x4_t __s1_105 = __p1_105; \
  uint16x4_t __rev0_105;  __rev0_105 = __builtin_shufflevector(__s0_105, __s0_105, 3, 2, 1, 0); \
  uint16x4_t __rev1_105;  __rev1_105 = __builtin_shufflevector(__s1_105, __s1_105, 3, 2, 1, 0); \
  uint32x4_t __ret_105; \
  __ret_105 = __noswap_vmull_u16(__rev0_105, __noswap_splat_lane_u16(__rev1_105, __p2_105)); \
  __ret_105 = __builtin_shufflevector(__ret_105, __ret_105, 3, 2, 1, 0); \
  __ret_105; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vmull_lane_s32(__p0_106, __p1_106, __p2_106) __extension__ ({ \
  int32x2_t __s0_106 = __p0_106; \
  int32x2_t __s1_106 = __p1_106; \
  int64x2_t __ret_106; \
  __ret_106 = vmull_s32(__s0_106, splat_lane_s32(__s1_106, __p2_106)); \
  __ret_106; \
})
#else
#define vmull_lane_s32(__p0_107, __p1_107, __p2_107) __extension__ ({ \
  int32x2_t __s0_107 = __p0_107; \
  int32x2_t __s1_107 = __p1_107; \
  int32x2_t __rev0_107;  __rev0_107 = __builtin_shufflevector(__s0_107, __s0_107, 1, 0); \
  int32x2_t __rev1_107;  __rev1_107 = __builtin_shufflevector(__s1_107, __s1_107, 1, 0); \
  int64x2_t __ret_107; \
  __ret_107 = __noswap_vmull_s32(__rev0_107, __noswap_splat_lane_s32(__rev1_107, __p2_107)); \
  __ret_107 = __builtin_shufflevector(__ret_107, __ret_107, 1, 0); \
  __ret_107; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vmull_lane_s16(__p0_108, __p1_108, __p2_108) __extension__ ({ \
  int16x4_t __s0_108 = __p0_108; \
  int16x4_t __s1_108 = __p1_108; \
  int32x4_t __ret_108; \
  __ret_108 = vmull_s16(__s0_108, splat_lane_s16(__s1_108, __p2_108)); \
  __ret_108; \
})
#else
#define vmull_lane_s16(__p0_109, __p1_109, __p2_109) __extension__ ({ \
  int16x4_t __s0_109 = __p0_109; \
  int16x4_t __s1_109 = __p1_109; \
  int16x4_t __rev0_109;  __rev0_109 = __builtin_shufflevector(__s0_109, __s0_109, 3, 2, 1, 0); \
  int16x4_t __rev1_109;  __rev1_109 = __builtin_shufflevector(__s1_109, __s1_109, 3, 2, 1, 0); \
  int32x4_t __ret_109; \
  __ret_109 = __noswap_vmull_s16(__rev0_109, __noswap_splat_lane_s16(__rev1_109, __p2_109)); \
  __ret_109 = __builtin_shufflevector(__ret_109, __ret_109, 3, 2, 1, 0); \
  __ret_109; \
})
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint64x2_t vmull_n_u32(uint32x2_t __p0, uint32_t __p1) {
  uint64x2_t __ret;
  __ret = vmull_u32(__p0, (uint32x2_t) {__p1, __p1});
  return __ret;
}
#else
__ai uint64x2_t vmull_n_u32(uint32x2_t __p0, uint32_t __p1) {
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint64x2_t __ret;
  __ret = __noswap_vmull_u32(__rev0, (uint32x2_t) {__p1, __p1});
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
__ai uint64x2_t __noswap_vmull_n_u32(uint32x2_t __p0, uint32_t __p1) {
  uint64x2_t __ret;
  __ret = __noswap_vmull_u32(__p0, (uint32x2_t) {__p1, __p1});
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vmull_n_u16(uint16x4_t __p0, uint16_t __p1) {
  uint32x4_t __ret;
  __ret = vmull_u16(__p0, (uint16x4_t) {__p1, __p1, __p1, __p1});
  return __ret;
}
#else
__ai uint32x4_t vmull_n_u16(uint16x4_t __p0, uint16_t __p1) {
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = __noswap_vmull_u16(__rev0, (uint16x4_t) {__p1, __p1, __p1, __p1});
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
__ai uint32x4_t __noswap_vmull_n_u16(uint16x4_t __p0, uint16_t __p1) {
  uint32x4_t __ret;
  __ret = __noswap_vmull_u16(__p0, (uint16x4_t) {__p1, __p1, __p1, __p1});
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int64x2_t vmull_n_s32(int32x2_t __p0, int32_t __p1) {
  int64x2_t __ret;
  __ret = vmull_s32(__p0, (int32x2_t) {__p1, __p1});
  return __ret;
}
#else
__ai int64x2_t vmull_n_s32(int32x2_t __p0, int32_t __p1) {
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int64x2_t __ret;
  __ret = __noswap_vmull_s32(__rev0, (int32x2_t) {__p1, __p1});
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
__ai int64x2_t __noswap_vmull_n_s32(int32x2_t __p0, int32_t __p1) {
  int64x2_t __ret;
  __ret = __noswap_vmull_s32(__p0, (int32x2_t) {__p1, __p1});
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x4_t vmull_n_s16(int16x4_t __p0, int16_t __p1) {
  int32x4_t __ret;
  __ret = vmull_s16(__p0, (int16x4_t) {__p1, __p1, __p1, __p1});
  return __ret;
}
#else
__ai int32x4_t vmull_n_s16(int16x4_t __p0, int16_t __p1) {
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x4_t __ret;
  __ret = __noswap_vmull_s16(__rev0, (int16x4_t) {__p1, __p1, __p1, __p1});
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
__ai int32x4_t __noswap_vmull_n_s16(int16x4_t __p0, int16_t __p1) {
  int32x4_t __ret;
  __ret = __noswap_vmull_s16(__p0, (int16x4_t) {__p1, __p1, __p1, __p1});
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai poly8x8_t vmvn_p8(poly8x8_t __p0) {
  poly8x8_t __ret;
  __ret = ~__p0;
  return __ret;
}
#else
__ai poly8x8_t vmvn_p8(poly8x8_t __p0) {
  poly8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  poly8x8_t __ret;
  __ret = ~__rev0;
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai poly8x16_t vmvnq_p8(poly8x16_t __p0) {
  poly8x16_t __ret;
  __ret = ~__p0;
  return __ret;
}
#else
__ai poly8x16_t vmvnq_p8(poly8x16_t __p0) {
  poly8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  poly8x16_t __ret;
  __ret = ~__rev0;
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x16_t vmvnq_u8(uint8x16_t __p0) {
  uint8x16_t __ret;
  __ret = ~__p0;
  return __ret;
}
#else
__ai uint8x16_t vmvnq_u8(uint8x16_t __p0) {
  uint8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __ret;
  __ret = ~__rev0;
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vmvnq_u32(uint32x4_t __p0) {
  uint32x4_t __ret;
  __ret = ~__p0;
  return __ret;
}
#else
__ai uint32x4_t vmvnq_u32(uint32x4_t __p0) {
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = ~__rev0;
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x8_t vmvnq_u16(uint16x8_t __p0) {
  uint16x8_t __ret;
  __ret = ~__p0;
  return __ret;
}
#else
__ai uint16x8_t vmvnq_u16(uint16x8_t __p0) {
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __ret;
  __ret = ~__rev0;
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x16_t vmvnq_s8(int8x16_t __p0) {
  int8x16_t __ret;
  __ret = ~__p0;
  return __ret;
}
#else
__ai int8x16_t vmvnq_s8(int8x16_t __p0) {
  int8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __ret;
  __ret = ~__rev0;
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x4_t vmvnq_s32(int32x4_t __p0) {
  int32x4_t __ret;
  __ret = ~__p0;
  return __ret;
}
#else
__ai int32x4_t vmvnq_s32(int32x4_t __p0) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x4_t __ret;
  __ret = ~__rev0;
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x8_t vmvnq_s16(int16x8_t __p0) {
  int16x8_t __ret;
  __ret = ~__p0;
  return __ret;
}
#else
__ai int16x8_t vmvnq_s16(int16x8_t __p0) {
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __ret;
  __ret = ~__rev0;
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x8_t vmvn_u8(uint8x8_t __p0) {
  uint8x8_t __ret;
  __ret = ~__p0;
  return __ret;
}
#else
__ai uint8x8_t vmvn_u8(uint8x8_t __p0) {
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __ret;
  __ret = ~__rev0;
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vmvn_u32(uint32x2_t __p0) {
  uint32x2_t __ret;
  __ret = ~__p0;
  return __ret;
}
#else
__ai uint32x2_t vmvn_u32(uint32x2_t __p0) {
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint32x2_t __ret;
  __ret = ~__rev0;
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x4_t vmvn_u16(uint16x4_t __p0) {
  uint16x4_t __ret;
  __ret = ~__p0;
  return __ret;
}
#else
__ai uint16x4_t vmvn_u16(uint16x4_t __p0) {
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint16x4_t __ret;
  __ret = ~__rev0;
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x8_t vmvn_s8(int8x8_t __p0) {
  int8x8_t __ret;
  __ret = ~__p0;
  return __ret;
}
#else
__ai int8x8_t vmvn_s8(int8x8_t __p0) {
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __ret;
  __ret = ~__rev0;
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x2_t vmvn_s32(int32x2_t __p0) {
  int32x2_t __ret;
  __ret = ~__p0;
  return __ret;
}
#else
__ai int32x2_t vmvn_s32(int32x2_t __p0) {
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __ret;
  __ret = ~__rev0;
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x4_t vmvn_s16(int16x4_t __p0) {
  int16x4_t __ret;
  __ret = ~__p0;
  return __ret;
}
#else
__ai int16x4_t vmvn_s16(int16x4_t __p0) {
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __ret;
  __ret = ~__rev0;
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x16_t vnegq_s8(int8x16_t __p0) {
  int8x16_t __ret;
  __ret = -__p0;
  return __ret;
}
#else
__ai int8x16_t vnegq_s8(int8x16_t __p0) {
  int8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __ret;
  __ret = -__rev0;
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai float32x4_t vnegq_f32(float32x4_t __p0) {
  float32x4_t __ret;
  __ret = -__p0;
  return __ret;
}
#else
__ai float32x4_t vnegq_f32(float32x4_t __p0) {
  float32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  float32x4_t __ret;
  __ret = -__rev0;
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x4_t vnegq_s32(int32x4_t __p0) {
  int32x4_t __ret;
  __ret = -__p0;
  return __ret;
}
#else
__ai int32x4_t vnegq_s32(int32x4_t __p0) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x4_t __ret;
  __ret = -__rev0;
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x8_t vnegq_s16(int16x8_t __p0) {
  int16x8_t __ret;
  __ret = -__p0;
  return __ret;
}
#else
__ai int16x8_t vnegq_s16(int16x8_t __p0) {
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __ret;
  __ret = -__rev0;
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x8_t vneg_s8(int8x8_t __p0) {
  int8x8_t __ret;
  __ret = -__p0;
  return __ret;
}
#else
__ai int8x8_t vneg_s8(int8x8_t __p0) {
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __ret;
  __ret = -__rev0;
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai float32x2_t vneg_f32(float32x2_t __p0) {
  float32x2_t __ret;
  __ret = -__p0;
  return __ret;
}
#else
__ai float32x2_t vneg_f32(float32x2_t __p0) {
  float32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  float32x2_t __ret;
  __ret = -__rev0;
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x2_t vneg_s32(int32x2_t __p0) {
  int32x2_t __ret;
  __ret = -__p0;
  return __ret;
}
#else
__ai int32x2_t vneg_s32(int32x2_t __p0) {
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __ret;
  __ret = -__rev0;
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x4_t vneg_s16(int16x4_t __p0) {
  int16x4_t __ret;
  __ret = -__p0;
  return __ret;
}
#else
__ai int16x4_t vneg_s16(int16x4_t __p0) {
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __ret;
  __ret = -__rev0;
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x16_t vornq_u8(uint8x16_t __p0, uint8x16_t __p1) {
  uint8x16_t __ret;
  __ret = __p0 | ~__p1;
  return __ret;
}
#else
__ai uint8x16_t vornq_u8(uint8x16_t __p0, uint8x16_t __p1) {
  uint8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __ret;
  __ret = __rev0 | ~__rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vornq_u32(uint32x4_t __p0, uint32x4_t __p1) {
  uint32x4_t __ret;
  __ret = __p0 | ~__p1;
  return __ret;
}
#else
__ai uint32x4_t vornq_u32(uint32x4_t __p0, uint32x4_t __p1) {
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = __rev0 | ~__rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint64x2_t vornq_u64(uint64x2_t __p0, uint64x2_t __p1) {
  uint64x2_t __ret;
  __ret = __p0 | ~__p1;
  return __ret;
}
#else
__ai uint64x2_t vornq_u64(uint64x2_t __p0, uint64x2_t __p1) {
  uint64x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint64x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint64x2_t __ret;
  __ret = __rev0 | ~__rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x8_t vornq_u16(uint16x8_t __p0, uint16x8_t __p1) {
  uint16x8_t __ret;
  __ret = __p0 | ~__p1;
  return __ret;
}
#else
__ai uint16x8_t vornq_u16(uint16x8_t __p0, uint16x8_t __p1) {
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __ret;
  __ret = __rev0 | ~__rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x16_t vornq_s8(int8x16_t __p0, int8x16_t __p1) {
  int8x16_t __ret;
  __ret = __p0 | ~__p1;
  return __ret;
}
#else
__ai int8x16_t vornq_s8(int8x16_t __p0, int8x16_t __p1) {
  int8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __ret;
  __ret = __rev0 | ~__rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x4_t vornq_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __ret;
  __ret = __p0 | ~__p1;
  return __ret;
}
#else
__ai int32x4_t vornq_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int32x4_t __ret;
  __ret = __rev0 | ~__rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int64x2_t vornq_s64(int64x2_t __p0, int64x2_t __p1) {
  int64x2_t __ret;
  __ret = __p0 | ~__p1;
  return __ret;
}
#else
__ai int64x2_t vornq_s64(int64x2_t __p0, int64x2_t __p1) {
  int64x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int64x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  int64x2_t __ret;
  __ret = __rev0 | ~__rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x8_t vornq_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __ret;
  __ret = __p0 | ~__p1;
  return __ret;
}
#else
__ai int16x8_t vornq_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __ret;
  __ret = __rev0 | ~__rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x8_t vorn_u8(uint8x8_t __p0, uint8x8_t __p1) {
  uint8x8_t __ret;
  __ret = __p0 | ~__p1;
  return __ret;
}
#else
__ai uint8x8_t vorn_u8(uint8x8_t __p0, uint8x8_t __p1) {
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __ret;
  __ret = __rev0 | ~__rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vorn_u32(uint32x2_t __p0, uint32x2_t __p1) {
  uint32x2_t __ret;
  __ret = __p0 | ~__p1;
  return __ret;
}
#else
__ai uint32x2_t vorn_u32(uint32x2_t __p0, uint32x2_t __p1) {
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint32x2_t __ret;
  __ret = __rev0 | ~__rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

__ai uint64x1_t vorn_u64(uint64x1_t __p0, uint64x1_t __p1) {
  uint64x1_t __ret;
  __ret = __p0 | ~__p1;
  return __ret;
}
#ifdef __LITTLE_ENDIAN__
__ai uint16x4_t vorn_u16(uint16x4_t __p0, uint16x4_t __p1) {
  uint16x4_t __ret;
  __ret = __p0 | ~__p1;
  return __ret;
}
#else
__ai uint16x4_t vorn_u16(uint16x4_t __p0, uint16x4_t __p1) {
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint16x4_t __ret;
  __ret = __rev0 | ~__rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x8_t vorn_s8(int8x8_t __p0, int8x8_t __p1) {
  int8x8_t __ret;
  __ret = __p0 | ~__p1;
  return __ret;
}
#else
__ai int8x8_t vorn_s8(int8x8_t __p0, int8x8_t __p1) {
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __ret;
  __ret = __rev0 | ~__rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x2_t vorn_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __ret;
  __ret = __p0 | ~__p1;
  return __ret;
}
#else
__ai int32x2_t vorn_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  int32x2_t __ret;
  __ret = __rev0 | ~__rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

__ai int64x1_t vorn_s64(int64x1_t __p0, int64x1_t __p1) {
  int64x1_t __ret;
  __ret = __p0 | ~__p1;
  return __ret;
}
#ifdef __LITTLE_ENDIAN__
__ai int16x4_t vorn_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __ret;
  __ret = __p0 | ~__p1;
  return __ret;
}
#else
__ai int16x4_t vorn_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int16x4_t __ret;
  __ret = __rev0 | ~__rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x16_t vorrq_u8(uint8x16_t __p0, uint8x16_t __p1) {
  uint8x16_t __ret;
  __ret = __p0 | __p1;
  return __ret;
}
#else
__ai uint8x16_t vorrq_u8(uint8x16_t __p0, uint8x16_t __p1) {
  uint8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __ret;
  __ret = __rev0 | __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vorrq_u32(uint32x4_t __p0, uint32x4_t __p1) {
  uint32x4_t __ret;
  __ret = __p0 | __p1;
  return __ret;
}
#else
__ai uint32x4_t vorrq_u32(uint32x4_t __p0, uint32x4_t __p1) {
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = __rev0 | __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint64x2_t vorrq_u64(uint64x2_t __p0, uint64x2_t __p1) {
  uint64x2_t __ret;
  __ret = __p0 | __p1;
  return __ret;
}
#else
__ai uint64x2_t vorrq_u64(uint64x2_t __p0, uint64x2_t __p1) {
  uint64x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint64x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint64x2_t __ret;
  __ret = __rev0 | __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x8_t vorrq_u16(uint16x8_t __p0, uint16x8_t __p1) {
  uint16x8_t __ret;
  __ret = __p0 | __p1;
  return __ret;
}
#else
__ai uint16x8_t vorrq_u16(uint16x8_t __p0, uint16x8_t __p1) {
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __ret;
  __ret = __rev0 | __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x16_t vorrq_s8(int8x16_t __p0, int8x16_t __p1) {
  int8x16_t __ret;
  __ret = __p0 | __p1;
  return __ret;
}
#else
__ai int8x16_t vorrq_s8(int8x16_t __p0, int8x16_t __p1) {
  int8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __ret;
  __ret = __rev0 | __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x4_t vorrq_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __ret;
  __ret = __p0 | __p1;
  return __ret;
}
#else
__ai int32x4_t vorrq_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int32x4_t __ret;
  __ret = __rev0 | __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int64x2_t vorrq_s64(int64x2_t __p0, int64x2_t __p1) {
  int64x2_t __ret;
  __ret = __p0 | __p1;
  return __ret;
}
#else
__ai int64x2_t vorrq_s64(int64x2_t __p0, int64x2_t __p1) {
  int64x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int64x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  int64x2_t __ret;
  __ret = __rev0 | __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x8_t vorrq_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __ret;
  __ret = __p0 | __p1;
  return __ret;
}
#else
__ai int16x8_t vorrq_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __ret;
  __ret = __rev0 | __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x8_t vorr_u8(uint8x8_t __p0, uint8x8_t __p1) {
  uint8x8_t __ret;
  __ret = __p0 | __p1;
  return __ret;
}
#else
__ai uint8x8_t vorr_u8(uint8x8_t __p0, uint8x8_t __p1) {
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __ret;
  __ret = __rev0 | __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vorr_u32(uint32x2_t __p0, uint32x2_t __p1) {
  uint32x2_t __ret;
  __ret = __p0 | __p1;
  return __ret;
}
#else
__ai uint32x2_t vorr_u32(uint32x2_t __p0, uint32x2_t __p1) {
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint32x2_t __ret;
  __ret = __rev0 | __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

__ai uint64x1_t vorr_u64(uint64x1_t __p0, uint64x1_t __p1) {
  uint64x1_t __ret;
  __ret = __p0 | __p1;
  return __ret;
}
#ifdef __LITTLE_ENDIAN__
__ai uint16x4_t vorr_u16(uint16x4_t __p0, uint16x4_t __p1) {
  uint16x4_t __ret;
  __ret = __p0 | __p1;
  return __ret;
}
#else
__ai uint16x4_t vorr_u16(uint16x4_t __p0, uint16x4_t __p1) {
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint16x4_t __ret;
  __ret = __rev0 | __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x8_t vorr_s8(int8x8_t __p0, int8x8_t __p1) {
  int8x8_t __ret;
  __ret = __p0 | __p1;
  return __ret;
}
#else
__ai int8x8_t vorr_s8(int8x8_t __p0, int8x8_t __p1) {
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __ret;
  __ret = __rev0 | __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x2_t vorr_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __ret;
  __ret = __p0 | __p1;
  return __ret;
}
#else
__ai int32x2_t vorr_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  int32x2_t __ret;
  __ret = __rev0 | __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

__ai int64x1_t vorr_s64(int64x1_t __p0, int64x1_t __p1) {
  int64x1_t __ret;
  __ret = __p0 | __p1;
  return __ret;
}
#ifdef __LITTLE_ENDIAN__
__ai int16x4_t vorr_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __ret;
  __ret = __p0 | __p1;
  return __ret;
}
#else
__ai int16x4_t vorr_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int16x4_t __ret;
  __ret = __rev0 | __rev1;
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x8_t vpadalq_u8(uint16x8_t __p0, uint8x16_t __p1) {
  uint16x8_t __ret;
  __ret = (uint16x8_t) __builtin_neon_vpadalq_v((int8x16_t)__p0, (int8x16_t)__p1, 49);
  return __ret;
}
#else
__ai uint16x8_t vpadalq_u8(uint16x8_t __p0, uint8x16_t __p1) {
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __ret;
  __ret = (uint16x8_t) __builtin_neon_vpadalq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 49);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint64x2_t vpadalq_u32(uint64x2_t __p0, uint32x4_t __p1) {
  uint64x2_t __ret;
  __ret = (uint64x2_t) __builtin_neon_vpadalq_v((int8x16_t)__p0, (int8x16_t)__p1, 51);
  return __ret;
}
#else
__ai uint64x2_t vpadalq_u32(uint64x2_t __p0, uint32x4_t __p1) {
  uint64x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint64x2_t __ret;
  __ret = (uint64x2_t) __builtin_neon_vpadalq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 51);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vpadalq_u16(uint32x4_t __p0, uint16x8_t __p1) {
  uint32x4_t __ret;
  __ret = (uint32x4_t) __builtin_neon_vpadalq_v((int8x16_t)__p0, (int8x16_t)__p1, 50);
  return __ret;
}
#else
__ai uint32x4_t vpadalq_u16(uint32x4_t __p0, uint16x8_t __p1) {
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = (uint32x4_t) __builtin_neon_vpadalq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 50);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x8_t vpadalq_s8(int16x8_t __p0, int8x16_t __p1) {
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vpadalq_v((int8x16_t)__p0, (int8x16_t)__p1, 33);
  return __ret;
}
#else
__ai int16x8_t vpadalq_s8(int16x8_t __p0, int8x16_t __p1) {
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vpadalq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 33);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int64x2_t vpadalq_s32(int64x2_t __p0, int32x4_t __p1) {
  int64x2_t __ret;
  __ret = (int64x2_t) __builtin_neon_vpadalq_v((int8x16_t)__p0, (int8x16_t)__p1, 35);
  return __ret;
}
#else
__ai int64x2_t vpadalq_s32(int64x2_t __p0, int32x4_t __p1) {
  int64x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int64x2_t __ret;
  __ret = (int64x2_t) __builtin_neon_vpadalq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 35);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x4_t vpadalq_s16(int32x4_t __p0, int16x8_t __p1) {
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vpadalq_v((int8x16_t)__p0, (int8x16_t)__p1, 34);
  return __ret;
}
#else
__ai int32x4_t vpadalq_s16(int32x4_t __p0, int16x8_t __p1) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vpadalq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 34);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x4_t vpadal_u8(uint16x4_t __p0, uint8x8_t __p1) {
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vpadal_v((int8x8_t)__p0, (int8x8_t)__p1, 17);
  return __ret;
}
#else
__ai uint16x4_t vpadal_u8(uint16x4_t __p0, uint8x8_t __p1) {
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vpadal_v((int8x8_t)__rev0, (int8x8_t)__rev1, 17);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint64x1_t vpadal_u32(uint64x1_t __p0, uint32x2_t __p1) {
  uint64x1_t __ret;
  __ret = (uint64x1_t) __builtin_neon_vpadal_v((int8x8_t)__p0, (int8x8_t)__p1, 19);
  return __ret;
}
#else
__ai uint64x1_t vpadal_u32(uint64x1_t __p0, uint32x2_t __p1) {
  uint32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint64x1_t __ret;
  __ret = (uint64x1_t) __builtin_neon_vpadal_v((int8x8_t)__p0, (int8x8_t)__rev1, 19);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vpadal_u16(uint32x2_t __p0, uint16x4_t __p1) {
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vpadal_v((int8x8_t)__p0, (int8x8_t)__p1, 18);
  return __ret;
}
#else
__ai uint32x2_t vpadal_u16(uint32x2_t __p0, uint16x4_t __p1) {
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vpadal_v((int8x8_t)__rev0, (int8x8_t)__rev1, 18);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x4_t vpadal_s8(int16x4_t __p0, int8x8_t __p1) {
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vpadal_v((int8x8_t)__p0, (int8x8_t)__p1, 1);
  return __ret;
}
#else
__ai int16x4_t vpadal_s8(int16x4_t __p0, int8x8_t __p1) {
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vpadal_v((int8x8_t)__rev0, (int8x8_t)__rev1, 1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int64x1_t vpadal_s32(int64x1_t __p0, int32x2_t __p1) {
  int64x1_t __ret;
  __ret = (int64x1_t) __builtin_neon_vpadal_v((int8x8_t)__p0, (int8x8_t)__p1, 3);
  return __ret;
}
#else
__ai int64x1_t vpadal_s32(int64x1_t __p0, int32x2_t __p1) {
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  int64x1_t __ret;
  __ret = (int64x1_t) __builtin_neon_vpadal_v((int8x8_t)__p0, (int8x8_t)__rev1, 3);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x2_t vpadal_s16(int32x2_t __p0, int16x4_t __p1) {
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vpadal_v((int8x8_t)__p0, (int8x8_t)__p1, 2);
  return __ret;
}
#else
__ai int32x2_t vpadal_s16(int32x2_t __p0, int16x4_t __p1) {
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vpadal_v((int8x8_t)__rev0, (int8x8_t)__rev1, 2);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x8_t vpadd_u8(uint8x8_t __p0, uint8x8_t __p1) {
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vpadd_v((int8x8_t)__p0, (int8x8_t)__p1, 16);
  return __ret;
}
#else
__ai uint8x8_t vpadd_u8(uint8x8_t __p0, uint8x8_t __p1) {
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vpadd_v((int8x8_t)__rev0, (int8x8_t)__rev1, 16);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vpadd_u32(uint32x2_t __p0, uint32x2_t __p1) {
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vpadd_v((int8x8_t)__p0, (int8x8_t)__p1, 18);
  return __ret;
}
#else
__ai uint32x2_t vpadd_u32(uint32x2_t __p0, uint32x2_t __p1) {
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vpadd_v((int8x8_t)__rev0, (int8x8_t)__rev1, 18);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x4_t vpadd_u16(uint16x4_t __p0, uint16x4_t __p1) {
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vpadd_v((int8x8_t)__p0, (int8x8_t)__p1, 17);
  return __ret;
}
#else
__ai uint16x4_t vpadd_u16(uint16x4_t __p0, uint16x4_t __p1) {
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vpadd_v((int8x8_t)__rev0, (int8x8_t)__rev1, 17);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x8_t vpadd_s8(int8x8_t __p0, int8x8_t __p1) {
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vpadd_v((int8x8_t)__p0, (int8x8_t)__p1, 0);
  return __ret;
}
#else
__ai int8x8_t vpadd_s8(int8x8_t __p0, int8x8_t __p1) {
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vpadd_v((int8x8_t)__rev0, (int8x8_t)__rev1, 0);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai float32x2_t vpadd_f32(float32x2_t __p0, float32x2_t __p1) {
  float32x2_t __ret;
  __ret = (float32x2_t) __builtin_neon_vpadd_v((int8x8_t)__p0, (int8x8_t)__p1, 9);
  return __ret;
}
#else
__ai float32x2_t vpadd_f32(float32x2_t __p0, float32x2_t __p1) {
  float32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  float32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  float32x2_t __ret;
  __ret = (float32x2_t) __builtin_neon_vpadd_v((int8x8_t)__rev0, (int8x8_t)__rev1, 9);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x2_t vpadd_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vpadd_v((int8x8_t)__p0, (int8x8_t)__p1, 2);
  return __ret;
}
#else
__ai int32x2_t vpadd_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vpadd_v((int8x8_t)__rev0, (int8x8_t)__rev1, 2);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x4_t vpadd_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vpadd_v((int8x8_t)__p0, (int8x8_t)__p1, 1);
  return __ret;
}
#else
__ai int16x4_t vpadd_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vpadd_v((int8x8_t)__rev0, (int8x8_t)__rev1, 1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x8_t vpaddlq_u8(uint8x16_t __p0) {
  uint16x8_t __ret;
  __ret = (uint16x8_t) __builtin_neon_vpaddlq_v((int8x16_t)__p0, 49);
  return __ret;
}
#else
__ai uint16x8_t vpaddlq_u8(uint8x16_t __p0) {
  uint8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __ret;
  __ret = (uint16x8_t) __builtin_neon_vpaddlq_v((int8x16_t)__rev0, 49);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint64x2_t vpaddlq_u32(uint32x4_t __p0) {
  uint64x2_t __ret;
  __ret = (uint64x2_t) __builtin_neon_vpaddlq_v((int8x16_t)__p0, 51);
  return __ret;
}
#else
__ai uint64x2_t vpaddlq_u32(uint32x4_t __p0) {
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint64x2_t __ret;
  __ret = (uint64x2_t) __builtin_neon_vpaddlq_v((int8x16_t)__rev0, 51);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vpaddlq_u16(uint16x8_t __p0) {
  uint32x4_t __ret;
  __ret = (uint32x4_t) __builtin_neon_vpaddlq_v((int8x16_t)__p0, 50);
  return __ret;
}
#else
__ai uint32x4_t vpaddlq_u16(uint16x8_t __p0) {
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = (uint32x4_t) __builtin_neon_vpaddlq_v((int8x16_t)__rev0, 50);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x8_t vpaddlq_s8(int8x16_t __p0) {
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vpaddlq_v((int8x16_t)__p0, 33);
  return __ret;
}
#else
__ai int16x8_t vpaddlq_s8(int8x16_t __p0) {
  int8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vpaddlq_v((int8x16_t)__rev0, 33);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int64x2_t vpaddlq_s32(int32x4_t __p0) {
  int64x2_t __ret;
  __ret = (int64x2_t) __builtin_neon_vpaddlq_v((int8x16_t)__p0, 35);
  return __ret;
}
#else
__ai int64x2_t vpaddlq_s32(int32x4_t __p0) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int64x2_t __ret;
  __ret = (int64x2_t) __builtin_neon_vpaddlq_v((int8x16_t)__rev0, 35);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x4_t vpaddlq_s16(int16x8_t __p0) {
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vpaddlq_v((int8x16_t)__p0, 34);
  return __ret;
}
#else
__ai int32x4_t vpaddlq_s16(int16x8_t __p0) {
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vpaddlq_v((int8x16_t)__rev0, 34);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x4_t vpaddl_u8(uint8x8_t __p0) {
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vpaddl_v((int8x8_t)__p0, 17);
  return __ret;
}
#else
__ai uint16x4_t vpaddl_u8(uint8x8_t __p0) {
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vpaddl_v((int8x8_t)__rev0, 17);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint64x1_t vpaddl_u32(uint32x2_t __p0) {
  uint64x1_t __ret;
  __ret = (uint64x1_t) __builtin_neon_vpaddl_v((int8x8_t)__p0, 19);
  return __ret;
}
#else
__ai uint64x1_t vpaddl_u32(uint32x2_t __p0) {
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint64x1_t __ret;
  __ret = (uint64x1_t) __builtin_neon_vpaddl_v((int8x8_t)__rev0, 19);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vpaddl_u16(uint16x4_t __p0) {
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vpaddl_v((int8x8_t)__p0, 18);
  return __ret;
}
#else
__ai uint32x2_t vpaddl_u16(uint16x4_t __p0) {
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vpaddl_v((int8x8_t)__rev0, 18);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x4_t vpaddl_s8(int8x8_t __p0) {
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vpaddl_v((int8x8_t)__p0, 1);
  return __ret;
}
#else
__ai int16x4_t vpaddl_s8(int8x8_t __p0) {
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vpaddl_v((int8x8_t)__rev0, 1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int64x1_t vpaddl_s32(int32x2_t __p0) {
  int64x1_t __ret;
  __ret = (int64x1_t) __builtin_neon_vpaddl_v((int8x8_t)__p0, 3);
  return __ret;
}
#else
__ai int64x1_t vpaddl_s32(int32x2_t __p0) {
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int64x1_t __ret;
  __ret = (int64x1_t) __builtin_neon_vpaddl_v((int8x8_t)__rev0, 3);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x2_t vpaddl_s16(int16x4_t __p0) {
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vpaddl_v((int8x8_t)__p0, 2);
  return __ret;
}
#else
__ai int32x2_t vpaddl_s16(int16x4_t __p0) {
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vpaddl_v((int8x8_t)__rev0, 2);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x8_t vpmax_u8(uint8x8_t __p0, uint8x8_t __p1) {
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vpmax_v((int8x8_t)__p0, (int8x8_t)__p1, 16);
  return __ret;
}
#else
__ai uint8x8_t vpmax_u8(uint8x8_t __p0, uint8x8_t __p1) {
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vpmax_v((int8x8_t)__rev0, (int8x8_t)__rev1, 16);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vpmax_u32(uint32x2_t __p0, uint32x2_t __p1) {
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vpmax_v((int8x8_t)__p0, (int8x8_t)__p1, 18);
  return __ret;
}
#else
__ai uint32x2_t vpmax_u32(uint32x2_t __p0, uint32x2_t __p1) {
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vpmax_v((int8x8_t)__rev0, (int8x8_t)__rev1, 18);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x4_t vpmax_u16(uint16x4_t __p0, uint16x4_t __p1) {
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vpmax_v((int8x8_t)__p0, (int8x8_t)__p1, 17);
  return __ret;
}
#else
__ai uint16x4_t vpmax_u16(uint16x4_t __p0, uint16x4_t __p1) {
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vpmax_v((int8x8_t)__rev0, (int8x8_t)__rev1, 17);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x8_t vpmax_s8(int8x8_t __p0, int8x8_t __p1) {
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vpmax_v((int8x8_t)__p0, (int8x8_t)__p1, 0);
  return __ret;
}
#else
__ai int8x8_t vpmax_s8(int8x8_t __p0, int8x8_t __p1) {
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vpmax_v((int8x8_t)__rev0, (int8x8_t)__rev1, 0);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai float32x2_t vpmax_f32(float32x2_t __p0, float32x2_t __p1) {
  float32x2_t __ret;
  __ret = (float32x2_t) __builtin_neon_vpmax_v((int8x8_t)__p0, (int8x8_t)__p1, 9);
  return __ret;
}
#else
__ai float32x2_t vpmax_f32(float32x2_t __p0, float32x2_t __p1) {
  float32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  float32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  float32x2_t __ret;
  __ret = (float32x2_t) __builtin_neon_vpmax_v((int8x8_t)__rev0, (int8x8_t)__rev1, 9);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x2_t vpmax_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vpmax_v((int8x8_t)__p0, (int8x8_t)__p1, 2);
  return __ret;
}
#else
__ai int32x2_t vpmax_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vpmax_v((int8x8_t)__rev0, (int8x8_t)__rev1, 2);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x4_t vpmax_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vpmax_v((int8x8_t)__p0, (int8x8_t)__p1, 1);
  return __ret;
}
#else
__ai int16x4_t vpmax_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vpmax_v((int8x8_t)__rev0, (int8x8_t)__rev1, 1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x8_t vpmin_u8(uint8x8_t __p0, uint8x8_t __p1) {
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vpmin_v((int8x8_t)__p0, (int8x8_t)__p1, 16);
  return __ret;
}
#else
__ai uint8x8_t vpmin_u8(uint8x8_t __p0, uint8x8_t __p1) {
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vpmin_v((int8x8_t)__rev0, (int8x8_t)__rev1, 16);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vpmin_u32(uint32x2_t __p0, uint32x2_t __p1) {
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vpmin_v((int8x8_t)__p0, (int8x8_t)__p1, 18);
  return __ret;
}
#else
__ai uint32x2_t vpmin_u32(uint32x2_t __p0, uint32x2_t __p1) {
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vpmin_v((int8x8_t)__rev0, (int8x8_t)__rev1, 18);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x4_t vpmin_u16(uint16x4_t __p0, uint16x4_t __p1) {
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vpmin_v((int8x8_t)__p0, (int8x8_t)__p1, 17);
  return __ret;
}
#else
__ai uint16x4_t vpmin_u16(uint16x4_t __p0, uint16x4_t __p1) {
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vpmin_v((int8x8_t)__rev0, (int8x8_t)__rev1, 17);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x8_t vpmin_s8(int8x8_t __p0, int8x8_t __p1) {
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vpmin_v((int8x8_t)__p0, (int8x8_t)__p1, 0);
  return __ret;
}
#else
__ai int8x8_t vpmin_s8(int8x8_t __p0, int8x8_t __p1) {
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vpmin_v((int8x8_t)__rev0, (int8x8_t)__rev1, 0);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai float32x2_t vpmin_f32(float32x2_t __p0, float32x2_t __p1) {
  float32x2_t __ret;
  __ret = (float32x2_t) __builtin_neon_vpmin_v((int8x8_t)__p0, (int8x8_t)__p1, 9);
  return __ret;
}
#else
__ai float32x2_t vpmin_f32(float32x2_t __p0, float32x2_t __p1) {
  float32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  float32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  float32x2_t __ret;
  __ret = (float32x2_t) __builtin_neon_vpmin_v((int8x8_t)__rev0, (int8x8_t)__rev1, 9);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x2_t vpmin_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vpmin_v((int8x8_t)__p0, (int8x8_t)__p1, 2);
  return __ret;
}
#else
__ai int32x2_t vpmin_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vpmin_v((int8x8_t)__rev0, (int8x8_t)__rev1, 2);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x4_t vpmin_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vpmin_v((int8x8_t)__p0, (int8x8_t)__p1, 1);
  return __ret;
}
#else
__ai int16x4_t vpmin_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vpmin_v((int8x8_t)__rev0, (int8x8_t)__rev1, 1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x16_t vqabsq_s8(int8x16_t __p0) {
  int8x16_t __ret;
  __ret = (int8x16_t) __builtin_neon_vqabsq_v((int8x16_t)__p0, 32);
  return __ret;
}
#else
__ai int8x16_t vqabsq_s8(int8x16_t __p0) {
  int8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __ret;
  __ret = (int8x16_t) __builtin_neon_vqabsq_v((int8x16_t)__rev0, 32);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x4_t vqabsq_s32(int32x4_t __p0) {
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vqabsq_v((int8x16_t)__p0, 34);
  return __ret;
}
#else
__ai int32x4_t vqabsq_s32(int32x4_t __p0) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vqabsq_v((int8x16_t)__rev0, 34);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x8_t vqabsq_s16(int16x8_t __p0) {
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vqabsq_v((int8x16_t)__p0, 33);
  return __ret;
}
#else
__ai int16x8_t vqabsq_s16(int16x8_t __p0) {
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vqabsq_v((int8x16_t)__rev0, 33);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x8_t vqabs_s8(int8x8_t __p0) {
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vqabs_v((int8x8_t)__p0, 0);
  return __ret;
}
#else
__ai int8x8_t vqabs_s8(int8x8_t __p0) {
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vqabs_v((int8x8_t)__rev0, 0);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x2_t vqabs_s32(int32x2_t __p0) {
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vqabs_v((int8x8_t)__p0, 2);
  return __ret;
}
#else
__ai int32x2_t vqabs_s32(int32x2_t __p0) {
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vqabs_v((int8x8_t)__rev0, 2);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x4_t vqabs_s16(int16x4_t __p0) {
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vqabs_v((int8x8_t)__p0, 1);
  return __ret;
}
#else
__ai int16x4_t vqabs_s16(int16x4_t __p0) {
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vqabs_v((int8x8_t)__rev0, 1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x16_t vqaddq_u8(uint8x16_t __p0, uint8x16_t __p1) {
  uint8x16_t __ret;
  __ret = (uint8x16_t) __builtin_neon_vqaddq_v((int8x16_t)__p0, (int8x16_t)__p1, 48);
  return __ret;
}
#else
__ai uint8x16_t vqaddq_u8(uint8x16_t __p0, uint8x16_t __p1) {
  uint8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __ret;
  __ret = (uint8x16_t) __builtin_neon_vqaddq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 48);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vqaddq_u32(uint32x4_t __p0, uint32x4_t __p1) {
  uint32x4_t __ret;
  __ret = (uint32x4_t) __builtin_neon_vqaddq_v((int8x16_t)__p0, (int8x16_t)__p1, 50);
  return __ret;
}
#else
__ai uint32x4_t vqaddq_u32(uint32x4_t __p0, uint32x4_t __p1) {
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = (uint32x4_t) __builtin_neon_vqaddq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 50);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint64x2_t vqaddq_u64(uint64x2_t __p0, uint64x2_t __p1) {
  uint64x2_t __ret;
  __ret = (uint64x2_t) __builtin_neon_vqaddq_v((int8x16_t)__p0, (int8x16_t)__p1, 51);
  return __ret;
}
#else
__ai uint64x2_t vqaddq_u64(uint64x2_t __p0, uint64x2_t __p1) {
  uint64x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint64x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint64x2_t __ret;
  __ret = (uint64x2_t) __builtin_neon_vqaddq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 51);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x8_t vqaddq_u16(uint16x8_t __p0, uint16x8_t __p1) {
  uint16x8_t __ret;
  __ret = (uint16x8_t) __builtin_neon_vqaddq_v((int8x16_t)__p0, (int8x16_t)__p1, 49);
  return __ret;
}
#else
__ai uint16x8_t vqaddq_u16(uint16x8_t __p0, uint16x8_t __p1) {
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __ret;
  __ret = (uint16x8_t) __builtin_neon_vqaddq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 49);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x16_t vqaddq_s8(int8x16_t __p0, int8x16_t __p1) {
  int8x16_t __ret;
  __ret = (int8x16_t) __builtin_neon_vqaddq_v((int8x16_t)__p0, (int8x16_t)__p1, 32);
  return __ret;
}
#else
__ai int8x16_t vqaddq_s8(int8x16_t __p0, int8x16_t __p1) {
  int8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __ret;
  __ret = (int8x16_t) __builtin_neon_vqaddq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 32);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x4_t vqaddq_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vqaddq_v((int8x16_t)__p0, (int8x16_t)__p1, 34);
  return __ret;
}
#else
__ai int32x4_t vqaddq_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vqaddq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 34);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
__ai int32x4_t __noswap_vqaddq_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vqaddq_v((int8x16_t)__p0, (int8x16_t)__p1, 34);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int64x2_t vqaddq_s64(int64x2_t __p0, int64x2_t __p1) {
  int64x2_t __ret;
  __ret = (int64x2_t) __builtin_neon_vqaddq_v((int8x16_t)__p0, (int8x16_t)__p1, 35);
  return __ret;
}
#else
__ai int64x2_t vqaddq_s64(int64x2_t __p0, int64x2_t __p1) {
  int64x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int64x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  int64x2_t __ret;
  __ret = (int64x2_t) __builtin_neon_vqaddq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 35);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x8_t vqaddq_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vqaddq_v((int8x16_t)__p0, (int8x16_t)__p1, 33);
  return __ret;
}
#else
__ai int16x8_t vqaddq_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vqaddq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 33);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
__ai int16x8_t __noswap_vqaddq_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vqaddq_v((int8x16_t)__p0, (int8x16_t)__p1, 33);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x8_t vqadd_u8(uint8x8_t __p0, uint8x8_t __p1) {
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vqadd_v((int8x8_t)__p0, (int8x8_t)__p1, 16);
  return __ret;
}
#else
__ai uint8x8_t vqadd_u8(uint8x8_t __p0, uint8x8_t __p1) {
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vqadd_v((int8x8_t)__rev0, (int8x8_t)__rev1, 16);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vqadd_u32(uint32x2_t __p0, uint32x2_t __p1) {
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vqadd_v((int8x8_t)__p0, (int8x8_t)__p1, 18);
  return __ret;
}
#else
__ai uint32x2_t vqadd_u32(uint32x2_t __p0, uint32x2_t __p1) {
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vqadd_v((int8x8_t)__rev0, (int8x8_t)__rev1, 18);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

__ai uint64x1_t vqadd_u64(uint64x1_t __p0, uint64x1_t __p1) {
  uint64x1_t __ret;
  __ret = (uint64x1_t) __builtin_neon_vqadd_v((int8x8_t)__p0, (int8x8_t)__p1, 19);
  return __ret;
}
#ifdef __LITTLE_ENDIAN__
__ai uint16x4_t vqadd_u16(uint16x4_t __p0, uint16x4_t __p1) {
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vqadd_v((int8x8_t)__p0, (int8x8_t)__p1, 17);
  return __ret;
}
#else
__ai uint16x4_t vqadd_u16(uint16x4_t __p0, uint16x4_t __p1) {
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vqadd_v((int8x8_t)__rev0, (int8x8_t)__rev1, 17);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x8_t vqadd_s8(int8x8_t __p0, int8x8_t __p1) {
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vqadd_v((int8x8_t)__p0, (int8x8_t)__p1, 0);
  return __ret;
}
#else
__ai int8x8_t vqadd_s8(int8x8_t __p0, int8x8_t __p1) {
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vqadd_v((int8x8_t)__rev0, (int8x8_t)__rev1, 0);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x2_t vqadd_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vqadd_v((int8x8_t)__p0, (int8x8_t)__p1, 2);
  return __ret;
}
#else
__ai int32x2_t vqadd_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vqadd_v((int8x8_t)__rev0, (int8x8_t)__rev1, 2);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
__ai int32x2_t __noswap_vqadd_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vqadd_v((int8x8_t)__p0, (int8x8_t)__p1, 2);
  return __ret;
}
#endif

__ai int64x1_t vqadd_s64(int64x1_t __p0, int64x1_t __p1) {
  int64x1_t __ret;
  __ret = (int64x1_t) __builtin_neon_vqadd_v((int8x8_t)__p0, (int8x8_t)__p1, 3);
  return __ret;
}
#ifdef __LITTLE_ENDIAN__
__ai int16x4_t vqadd_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vqadd_v((int8x8_t)__p0, (int8x8_t)__p1, 1);
  return __ret;
}
#else
__ai int16x4_t vqadd_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vqadd_v((int8x8_t)__rev0, (int8x8_t)__rev1, 1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
__ai int16x4_t __noswap_vqadd_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vqadd_v((int8x8_t)__p0, (int8x8_t)__p1, 1);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int64x2_t vqdmlal_s32(int64x2_t __p0, int32x2_t __p1, int32x2_t __p2) {
  int64x2_t __ret;
  __ret = (int64x2_t) __builtin_neon_vqdmlal_v((int8x16_t)__p0, (int8x8_t)__p1, (int8x8_t)__p2, 35);
  return __ret;
}
#else
__ai int64x2_t vqdmlal_s32(int64x2_t __p0, int32x2_t __p1, int32x2_t __p2) {
  int64x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  int32x2_t __rev2;  __rev2 = __builtin_shufflevector(__p2, __p2, 1, 0);
  int64x2_t __ret;
  __ret = (int64x2_t) __builtin_neon_vqdmlal_v((int8x16_t)__rev0, (int8x8_t)__rev1, (int8x8_t)__rev2, 35);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
__ai int64x2_t __noswap_vqdmlal_s32(int64x2_t __p0, int32x2_t __p1, int32x2_t __p2) {
  int64x2_t __ret;
  __ret = (int64x2_t) __builtin_neon_vqdmlal_v((int8x16_t)__p0, (int8x8_t)__p1, (int8x8_t)__p2, 35);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x4_t vqdmlal_s16(int32x4_t __p0, int16x4_t __p1, int16x4_t __p2) {
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vqdmlal_v((int8x16_t)__p0, (int8x8_t)__p1, (int8x8_t)__p2, 34);
  return __ret;
}
#else
__ai int32x4_t vqdmlal_s16(int32x4_t __p0, int16x4_t __p1, int16x4_t __p2) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int16x4_t __rev2;  __rev2 = __builtin_shufflevector(__p2, __p2, 3, 2, 1, 0);
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vqdmlal_v((int8x16_t)__rev0, (int8x8_t)__rev1, (int8x8_t)__rev2, 34);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
__ai int32x4_t __noswap_vqdmlal_s16(int32x4_t __p0, int16x4_t __p1, int16x4_t __p2) {
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vqdmlal_v((int8x16_t)__p0, (int8x8_t)__p1, (int8x8_t)__p2, 34);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
#define vqdmlal_lane_s32(__p0_110, __p1_110, __p2_110, __p3_110) __extension__ ({ \
  int64x2_t __s0_110 = __p0_110; \
  int32x2_t __s1_110 = __p1_110; \
  int32x2_t __s2_110 = __p2_110; \
  int64x2_t __ret_110; \
  __ret_110 = vqdmlal_s32(__s0_110, __s1_110, splat_lane_s32(__s2_110, __p3_110)); \
  __ret_110; \
})
#else
#define vqdmlal_lane_s32(__p0_111, __p1_111, __p2_111, __p3_111) __extension__ ({ \
  int64x2_t __s0_111 = __p0_111; \
  int32x2_t __s1_111 = __p1_111; \
  int32x2_t __s2_111 = __p2_111; \
  int64x2_t __rev0_111;  __rev0_111 = __builtin_shufflevector(__s0_111, __s0_111, 1, 0); \
  int32x2_t __rev1_111;  __rev1_111 = __builtin_shufflevector(__s1_111, __s1_111, 1, 0); \
  int32x2_t __rev2_111;  __rev2_111 = __builtin_shufflevector(__s2_111, __s2_111, 1, 0); \
  int64x2_t __ret_111; \
  __ret_111 = __noswap_vqdmlal_s32(__rev0_111, __rev1_111, __noswap_splat_lane_s32(__rev2_111, __p3_111)); \
  __ret_111 = __builtin_shufflevector(__ret_111, __ret_111, 1, 0); \
  __ret_111; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vqdmlal_lane_s16(__p0_112, __p1_112, __p2_112, __p3_112) __extension__ ({ \
  int32x4_t __s0_112 = __p0_112; \
  int16x4_t __s1_112 = __p1_112; \
  int16x4_t __s2_112 = __p2_112; \
  int32x4_t __ret_112; \
  __ret_112 = vqdmlal_s16(__s0_112, __s1_112, splat_lane_s16(__s2_112, __p3_112)); \
  __ret_112; \
})
#else
#define vqdmlal_lane_s16(__p0_113, __p1_113, __p2_113, __p3_113) __extension__ ({ \
  int32x4_t __s0_113 = __p0_113; \
  int16x4_t __s1_113 = __p1_113; \
  int16x4_t __s2_113 = __p2_113; \
  int32x4_t __rev0_113;  __rev0_113 = __builtin_shufflevector(__s0_113, __s0_113, 3, 2, 1, 0); \
  int16x4_t __rev1_113;  __rev1_113 = __builtin_shufflevector(__s1_113, __s1_113, 3, 2, 1, 0); \
  int16x4_t __rev2_113;  __rev2_113 = __builtin_shufflevector(__s2_113, __s2_113, 3, 2, 1, 0); \
  int32x4_t __ret_113; \
  __ret_113 = __noswap_vqdmlal_s16(__rev0_113, __rev1_113, __noswap_splat_lane_s16(__rev2_113, __p3_113)); \
  __ret_113 = __builtin_shufflevector(__ret_113, __ret_113, 3, 2, 1, 0); \
  __ret_113; \
})
#endif

#ifdef __LITTLE_ENDIAN__
__ai int64x2_t vqdmlal_n_s32(int64x2_t __p0, int32x2_t __p1, int32_t __p2) {
  int64x2_t __ret;
  __ret = vqdmlal_s32(__p0, __p1, (int32x2_t) {__p2, __p2});
  return __ret;
}
#else
__ai int64x2_t vqdmlal_n_s32(int64x2_t __p0, int32x2_t __p1, int32_t __p2) {
  int64x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  int64x2_t __ret;
  __ret = __noswap_vqdmlal_s32(__rev0, __rev1, (int32x2_t) {__p2, __p2});
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
__ai int64x2_t __noswap_vqdmlal_n_s32(int64x2_t __p0, int32x2_t __p1, int32_t __p2) {
  int64x2_t __ret;
  __ret = __noswap_vqdmlal_s32(__p0, __p1, (int32x2_t) {__p2, __p2});
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x4_t vqdmlal_n_s16(int32x4_t __p0, int16x4_t __p1, int16_t __p2) {
  int32x4_t __ret;
  __ret = vqdmlal_s16(__p0, __p1, (int16x4_t) {__p2, __p2, __p2, __p2});
  return __ret;
}
#else
__ai int32x4_t vqdmlal_n_s16(int32x4_t __p0, int16x4_t __p1, int16_t __p2) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int32x4_t __ret;
  __ret = __noswap_vqdmlal_s16(__rev0, __rev1, (int16x4_t) {__p2, __p2, __p2, __p2});
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
__ai int32x4_t __noswap_vqdmlal_n_s16(int32x4_t __p0, int16x4_t __p1, int16_t __p2) {
  int32x4_t __ret;
  __ret = __noswap_vqdmlal_s16(__p0, __p1, (int16x4_t) {__p2, __p2, __p2, __p2});
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int64x2_t vqdmlsl_s32(int64x2_t __p0, int32x2_t __p1, int32x2_t __p2) {
  int64x2_t __ret;
  __ret = (int64x2_t) __builtin_neon_vqdmlsl_v((int8x16_t)__p0, (int8x8_t)__p1, (int8x8_t)__p2, 35);
  return __ret;
}
#else
__ai int64x2_t vqdmlsl_s32(int64x2_t __p0, int32x2_t __p1, int32x2_t __p2) {
  int64x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  int32x2_t __rev2;  __rev2 = __builtin_shufflevector(__p2, __p2, 1, 0);
  int64x2_t __ret;
  __ret = (int64x2_t) __builtin_neon_vqdmlsl_v((int8x16_t)__rev0, (int8x8_t)__rev1, (int8x8_t)__rev2, 35);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
__ai int64x2_t __noswap_vqdmlsl_s32(int64x2_t __p0, int32x2_t __p1, int32x2_t __p2) {
  int64x2_t __ret;
  __ret = (int64x2_t) __builtin_neon_vqdmlsl_v((int8x16_t)__p0, (int8x8_t)__p1, (int8x8_t)__p2, 35);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x4_t vqdmlsl_s16(int32x4_t __p0, int16x4_t __p1, int16x4_t __p2) {
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vqdmlsl_v((int8x16_t)__p0, (int8x8_t)__p1, (int8x8_t)__p2, 34);
  return __ret;
}
#else
__ai int32x4_t vqdmlsl_s16(int32x4_t __p0, int16x4_t __p1, int16x4_t __p2) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int16x4_t __rev2;  __rev2 = __builtin_shufflevector(__p2, __p2, 3, 2, 1, 0);
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vqdmlsl_v((int8x16_t)__rev0, (int8x8_t)__rev1, (int8x8_t)__rev2, 34);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
__ai int32x4_t __noswap_vqdmlsl_s16(int32x4_t __p0, int16x4_t __p1, int16x4_t __p2) {
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vqdmlsl_v((int8x16_t)__p0, (int8x8_t)__p1, (int8x8_t)__p2, 34);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
#define vqdmlsl_lane_s32(__p0_114, __p1_114, __p2_114, __p3_114) __extension__ ({ \
  int64x2_t __s0_114 = __p0_114; \
  int32x2_t __s1_114 = __p1_114; \
  int32x2_t __s2_114 = __p2_114; \
  int64x2_t __ret_114; \
  __ret_114 = vqdmlsl_s32(__s0_114, __s1_114, splat_lane_s32(__s2_114, __p3_114)); \
  __ret_114; \
})
#else
#define vqdmlsl_lane_s32(__p0_115, __p1_115, __p2_115, __p3_115) __extension__ ({ \
  int64x2_t __s0_115 = __p0_115; \
  int32x2_t __s1_115 = __p1_115; \
  int32x2_t __s2_115 = __p2_115; \
  int64x2_t __rev0_115;  __rev0_115 = __builtin_shufflevector(__s0_115, __s0_115, 1, 0); \
  int32x2_t __rev1_115;  __rev1_115 = __builtin_shufflevector(__s1_115, __s1_115, 1, 0); \
  int32x2_t __rev2_115;  __rev2_115 = __builtin_shufflevector(__s2_115, __s2_115, 1, 0); \
  int64x2_t __ret_115; \
  __ret_115 = __noswap_vqdmlsl_s32(__rev0_115, __rev1_115, __noswap_splat_lane_s32(__rev2_115, __p3_115)); \
  __ret_115 = __builtin_shufflevector(__ret_115, __ret_115, 1, 0); \
  __ret_115; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vqdmlsl_lane_s16(__p0_116, __p1_116, __p2_116, __p3_116) __extension__ ({ \
  int32x4_t __s0_116 = __p0_116; \
  int16x4_t __s1_116 = __p1_116; \
  int16x4_t __s2_116 = __p2_116; \
  int32x4_t __ret_116; \
  __ret_116 = vqdmlsl_s16(__s0_116, __s1_116, splat_lane_s16(__s2_116, __p3_116)); \
  __ret_116; \
})
#else
#define vqdmlsl_lane_s16(__p0_117, __p1_117, __p2_117, __p3_117) __extension__ ({ \
  int32x4_t __s0_117 = __p0_117; \
  int16x4_t __s1_117 = __p1_117; \
  int16x4_t __s2_117 = __p2_117; \
  int32x4_t __rev0_117;  __rev0_117 = __builtin_shufflevector(__s0_117, __s0_117, 3, 2, 1, 0); \
  int16x4_t __rev1_117;  __rev1_117 = __builtin_shufflevector(__s1_117, __s1_117, 3, 2, 1, 0); \
  int16x4_t __rev2_117;  __rev2_117 = __builtin_shufflevector(__s2_117, __s2_117, 3, 2, 1, 0); \
  int32x4_t __ret_117; \
  __ret_117 = __noswap_vqdmlsl_s16(__rev0_117, __rev1_117, __noswap_splat_lane_s16(__rev2_117, __p3_117)); \
  __ret_117 = __builtin_shufflevector(__ret_117, __ret_117, 3, 2, 1, 0); \
  __ret_117; \
})
#endif

#ifdef __LITTLE_ENDIAN__
__ai int64x2_t vqdmlsl_n_s32(int64x2_t __p0, int32x2_t __p1, int32_t __p2) {
  int64x2_t __ret;
  __ret = vqdmlsl_s32(__p0, __p1, (int32x2_t) {__p2, __p2});
  return __ret;
}
#else
__ai int64x2_t vqdmlsl_n_s32(int64x2_t __p0, int32x2_t __p1, int32_t __p2) {
  int64x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  int64x2_t __ret;
  __ret = __noswap_vqdmlsl_s32(__rev0, __rev1, (int32x2_t) {__p2, __p2});
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
__ai int64x2_t __noswap_vqdmlsl_n_s32(int64x2_t __p0, int32x2_t __p1, int32_t __p2) {
  int64x2_t __ret;
  __ret = __noswap_vqdmlsl_s32(__p0, __p1, (int32x2_t) {__p2, __p2});
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x4_t vqdmlsl_n_s16(int32x4_t __p0, int16x4_t __p1, int16_t __p2) {
  int32x4_t __ret;
  __ret = vqdmlsl_s16(__p0, __p1, (int16x4_t) {__p2, __p2, __p2, __p2});
  return __ret;
}
#else
__ai int32x4_t vqdmlsl_n_s16(int32x4_t __p0, int16x4_t __p1, int16_t __p2) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int32x4_t __ret;
  __ret = __noswap_vqdmlsl_s16(__rev0, __rev1, (int16x4_t) {__p2, __p2, __p2, __p2});
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
__ai int32x4_t __noswap_vqdmlsl_n_s16(int32x4_t __p0, int16x4_t __p1, int16_t __p2) {
  int32x4_t __ret;
  __ret = __noswap_vqdmlsl_s16(__p0, __p1, (int16x4_t) {__p2, __p2, __p2, __p2});
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x4_t vqdmulhq_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vqdmulhq_v((int8x16_t)__p0, (int8x16_t)__p1, 34);
  return __ret;
}
#else
__ai int32x4_t vqdmulhq_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vqdmulhq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 34);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
__ai int32x4_t __noswap_vqdmulhq_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vqdmulhq_v((int8x16_t)__p0, (int8x16_t)__p1, 34);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x8_t vqdmulhq_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vqdmulhq_v((int8x16_t)__p0, (int8x16_t)__p1, 33);
  return __ret;
}
#else
__ai int16x8_t vqdmulhq_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vqdmulhq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 33);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
__ai int16x8_t __noswap_vqdmulhq_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vqdmulhq_v((int8x16_t)__p0, (int8x16_t)__p1, 33);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x2_t vqdmulh_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vqdmulh_v((int8x8_t)__p0, (int8x8_t)__p1, 2);
  return __ret;
}
#else
__ai int32x2_t vqdmulh_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vqdmulh_v((int8x8_t)__rev0, (int8x8_t)__rev1, 2);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
__ai int32x2_t __noswap_vqdmulh_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vqdmulh_v((int8x8_t)__p0, (int8x8_t)__p1, 2);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x4_t vqdmulh_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vqdmulh_v((int8x8_t)__p0, (int8x8_t)__p1, 1);
  return __ret;
}
#else
__ai int16x4_t vqdmulh_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vqdmulh_v((int8x8_t)__rev0, (int8x8_t)__rev1, 1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
__ai int16x4_t __noswap_vqdmulh_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vqdmulh_v((int8x8_t)__p0, (int8x8_t)__p1, 1);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x4_t vqdmulhq_n_s32(int32x4_t __p0, int32_t __p1) {
  int32x4_t __ret;
  __ret = vqdmulhq_s32(__p0, (int32x4_t) {__p1, __p1, __p1, __p1});
  return __ret;
}
#else
__ai int32x4_t vqdmulhq_n_s32(int32x4_t __p0, int32_t __p1) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x4_t __ret;
  __ret = __noswap_vqdmulhq_s32(__rev0, (int32x4_t) {__p1, __p1, __p1, __p1});
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x8_t vqdmulhq_n_s16(int16x8_t __p0, int16_t __p1) {
  int16x8_t __ret;
  __ret = vqdmulhq_s16(__p0, (int16x8_t) {__p1, __p1, __p1, __p1, __p1, __p1, __p1, __p1});
  return __ret;
}
#else
__ai int16x8_t vqdmulhq_n_s16(int16x8_t __p0, int16_t __p1) {
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __ret;
  __ret = __noswap_vqdmulhq_s16(__rev0, (int16x8_t) {__p1, __p1, __p1, __p1, __p1, __p1, __p1, __p1});
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x2_t vqdmulh_n_s32(int32x2_t __p0, int32_t __p1) {
  int32x2_t __ret;
  __ret = vqdmulh_s32(__p0, (int32x2_t) {__p1, __p1});
  return __ret;
}
#else
__ai int32x2_t vqdmulh_n_s32(int32x2_t __p0, int32_t __p1) {
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __ret;
  __ret = __noswap_vqdmulh_s32(__rev0, (int32x2_t) {__p1, __p1});
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x4_t vqdmulh_n_s16(int16x4_t __p0, int16_t __p1) {
  int16x4_t __ret;
  __ret = vqdmulh_s16(__p0, (int16x4_t) {__p1, __p1, __p1, __p1});
  return __ret;
}
#else
__ai int16x4_t vqdmulh_n_s16(int16x4_t __p0, int16_t __p1) {
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __ret;
  __ret = __noswap_vqdmulh_s16(__rev0, (int16x4_t) {__p1, __p1, __p1, __p1});
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int64x2_t vqdmull_s32(int32x2_t __p0, int32x2_t __p1) {
  int64x2_t __ret;
  __ret = (int64x2_t) __builtin_neon_vqdmull_v((int8x8_t)__p0, (int8x8_t)__p1, 35);
  return __ret;
}
#else
__ai int64x2_t vqdmull_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  int64x2_t __ret;
  __ret = (int64x2_t) __builtin_neon_vqdmull_v((int8x8_t)__rev0, (int8x8_t)__rev1, 35);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
__ai int64x2_t __noswap_vqdmull_s32(int32x2_t __p0, int32x2_t __p1) {
  int64x2_t __ret;
  __ret = (int64x2_t) __builtin_neon_vqdmull_v((int8x8_t)__p0, (int8x8_t)__p1, 35);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x4_t vqdmull_s16(int16x4_t __p0, int16x4_t __p1) {
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vqdmull_v((int8x8_t)__p0, (int8x8_t)__p1, 34);
  return __ret;
}
#else
__ai int32x4_t vqdmull_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vqdmull_v((int8x8_t)__rev0, (int8x8_t)__rev1, 34);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
__ai int32x4_t __noswap_vqdmull_s16(int16x4_t __p0, int16x4_t __p1) {
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vqdmull_v((int8x8_t)__p0, (int8x8_t)__p1, 34);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
#define vqdmull_lane_s32(__p0_118, __p1_118, __p2_118) __extension__ ({ \
  int32x2_t __s0_118 = __p0_118; \
  int32x2_t __s1_118 = __p1_118; \
  int64x2_t __ret_118; \
  __ret_118 = vqdmull_s32(__s0_118, splat_lane_s32(__s1_118, __p2_118)); \
  __ret_118; \
})
#else
#define vqdmull_lane_s32(__p0_119, __p1_119, __p2_119) __extension__ ({ \
  int32x2_t __s0_119 = __p0_119; \
  int32x2_t __s1_119 = __p1_119; \
  int32x2_t __rev0_119;  __rev0_119 = __builtin_shufflevector(__s0_119, __s0_119, 1, 0); \
  int32x2_t __rev1_119;  __rev1_119 = __builtin_shufflevector(__s1_119, __s1_119, 1, 0); \
  int64x2_t __ret_119; \
  __ret_119 = __noswap_vqdmull_s32(__rev0_119, __noswap_splat_lane_s32(__rev1_119, __p2_119)); \
  __ret_119 = __builtin_shufflevector(__ret_119, __ret_119, 1, 0); \
  __ret_119; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vqdmull_lane_s16(__p0_120, __p1_120, __p2_120) __extension__ ({ \
  int16x4_t __s0_120 = __p0_120; \
  int16x4_t __s1_120 = __p1_120; \
  int32x4_t __ret_120; \
  __ret_120 = vqdmull_s16(__s0_120, splat_lane_s16(__s1_120, __p2_120)); \
  __ret_120; \
})
#else
#define vqdmull_lane_s16(__p0_121, __p1_121, __p2_121) __extension__ ({ \
  int16x4_t __s0_121 = __p0_121; \
  int16x4_t __s1_121 = __p1_121; \
  int16x4_t __rev0_121;  __rev0_121 = __builtin_shufflevector(__s0_121, __s0_121, 3, 2, 1, 0); \
  int16x4_t __rev1_121;  __rev1_121 = __builtin_shufflevector(__s1_121, __s1_121, 3, 2, 1, 0); \
  int32x4_t __ret_121; \
  __ret_121 = __noswap_vqdmull_s16(__rev0_121, __noswap_splat_lane_s16(__rev1_121, __p2_121)); \
  __ret_121 = __builtin_shufflevector(__ret_121, __ret_121, 3, 2, 1, 0); \
  __ret_121; \
})
#endif

#ifdef __LITTLE_ENDIAN__
__ai int64x2_t vqdmull_n_s32(int32x2_t __p0, int32_t __p1) {
  int64x2_t __ret;
  __ret = vqdmull_s32(__p0, (int32x2_t) {__p1, __p1});
  return __ret;
}
#else
__ai int64x2_t vqdmull_n_s32(int32x2_t __p0, int32_t __p1) {
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int64x2_t __ret;
  __ret = __noswap_vqdmull_s32(__rev0, (int32x2_t) {__p1, __p1});
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
__ai int64x2_t __noswap_vqdmull_n_s32(int32x2_t __p0, int32_t __p1) {
  int64x2_t __ret;
  __ret = __noswap_vqdmull_s32(__p0, (int32x2_t) {__p1, __p1});
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x4_t vqdmull_n_s16(int16x4_t __p0, int16_t __p1) {
  int32x4_t __ret;
  __ret = vqdmull_s16(__p0, (int16x4_t) {__p1, __p1, __p1, __p1});
  return __ret;
}
#else
__ai int32x4_t vqdmull_n_s16(int16x4_t __p0, int16_t __p1) {
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x4_t __ret;
  __ret = __noswap_vqdmull_s16(__rev0, (int16x4_t) {__p1, __p1, __p1, __p1});
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
__ai int32x4_t __noswap_vqdmull_n_s16(int16x4_t __p0, int16_t __p1) {
  int32x4_t __ret;
  __ret = __noswap_vqdmull_s16(__p0, (int16x4_t) {__p1, __p1, __p1, __p1});
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x4_t vqmovn_u32(uint32x4_t __p0) {
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vqmovn_v((int8x16_t)__p0, 17);
  return __ret;
}
#else
__ai uint16x4_t vqmovn_u32(uint32x4_t __p0) {
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vqmovn_v((int8x16_t)__rev0, 17);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
__ai uint16x4_t __noswap_vqmovn_u32(uint32x4_t __p0) {
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vqmovn_v((int8x16_t)__p0, 17);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vqmovn_u64(uint64x2_t __p0) {
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vqmovn_v((int8x16_t)__p0, 18);
  return __ret;
}
#else
__ai uint32x2_t vqmovn_u64(uint64x2_t __p0) {
  uint64x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vqmovn_v((int8x16_t)__rev0, 18);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
__ai uint32x2_t __noswap_vqmovn_u64(uint64x2_t __p0) {
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vqmovn_v((int8x16_t)__p0, 18);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x8_t vqmovn_u16(uint16x8_t __p0) {
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vqmovn_v((int8x16_t)__p0, 16);
  return __ret;
}
#else
__ai uint8x8_t vqmovn_u16(uint16x8_t __p0) {
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vqmovn_v((int8x16_t)__rev0, 16);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
__ai uint8x8_t __noswap_vqmovn_u16(uint16x8_t __p0) {
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vqmovn_v((int8x16_t)__p0, 16);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x4_t vqmovn_s32(int32x4_t __p0) {
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vqmovn_v((int8x16_t)__p0, 1);
  return __ret;
}
#else
__ai int16x4_t vqmovn_s32(int32x4_t __p0) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vqmovn_v((int8x16_t)__rev0, 1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
__ai int16x4_t __noswap_vqmovn_s32(int32x4_t __p0) {
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vqmovn_v((int8x16_t)__p0, 1);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x2_t vqmovn_s64(int64x2_t __p0) {
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vqmovn_v((int8x16_t)__p0, 2);
  return __ret;
}
#else
__ai int32x2_t vqmovn_s64(int64x2_t __p0) {
  int64x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vqmovn_v((int8x16_t)__rev0, 2);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
__ai int32x2_t __noswap_vqmovn_s64(int64x2_t __p0) {
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vqmovn_v((int8x16_t)__p0, 2);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x8_t vqmovn_s16(int16x8_t __p0) {
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vqmovn_v((int8x16_t)__p0, 0);
  return __ret;
}
#else
__ai int8x8_t vqmovn_s16(int16x8_t __p0) {
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vqmovn_v((int8x16_t)__rev0, 0);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
__ai int8x8_t __noswap_vqmovn_s16(int16x8_t __p0) {
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vqmovn_v((int8x16_t)__p0, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x4_t vqmovun_s32(int32x4_t __p0) {
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vqmovun_v((int8x16_t)__p0, 17);
  return __ret;
}
#else
__ai uint16x4_t vqmovun_s32(int32x4_t __p0) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vqmovun_v((int8x16_t)__rev0, 17);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
__ai uint16x4_t __noswap_vqmovun_s32(int32x4_t __p0) {
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vqmovun_v((int8x16_t)__p0, 17);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vqmovun_s64(int64x2_t __p0) {
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vqmovun_v((int8x16_t)__p0, 18);
  return __ret;
}
#else
__ai uint32x2_t vqmovun_s64(int64x2_t __p0) {
  int64x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vqmovun_v((int8x16_t)__rev0, 18);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
__ai uint32x2_t __noswap_vqmovun_s64(int64x2_t __p0) {
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vqmovun_v((int8x16_t)__p0, 18);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x8_t vqmovun_s16(int16x8_t __p0) {
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vqmovun_v((int8x16_t)__p0, 16);
  return __ret;
}
#else
__ai uint8x8_t vqmovun_s16(int16x8_t __p0) {
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vqmovun_v((int8x16_t)__rev0, 16);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
__ai uint8x8_t __noswap_vqmovun_s16(int16x8_t __p0) {
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vqmovun_v((int8x16_t)__p0, 16);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x16_t vqnegq_s8(int8x16_t __p0) {
  int8x16_t __ret;
  __ret = (int8x16_t) __builtin_neon_vqnegq_v((int8x16_t)__p0, 32);
  return __ret;
}
#else
__ai int8x16_t vqnegq_s8(int8x16_t __p0) {
  int8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __ret;
  __ret = (int8x16_t) __builtin_neon_vqnegq_v((int8x16_t)__rev0, 32);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x4_t vqnegq_s32(int32x4_t __p0) {
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vqnegq_v((int8x16_t)__p0, 34);
  return __ret;
}
#else
__ai int32x4_t vqnegq_s32(int32x4_t __p0) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vqnegq_v((int8x16_t)__rev0, 34);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x8_t vqnegq_s16(int16x8_t __p0) {
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vqnegq_v((int8x16_t)__p0, 33);
  return __ret;
}
#else
__ai int16x8_t vqnegq_s16(int16x8_t __p0) {
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vqnegq_v((int8x16_t)__rev0, 33);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x8_t vqneg_s8(int8x8_t __p0) {
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vqneg_v((int8x8_t)__p0, 0);
  return __ret;
}
#else
__ai int8x8_t vqneg_s8(int8x8_t __p0) {
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vqneg_v((int8x8_t)__rev0, 0);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x2_t vqneg_s32(int32x2_t __p0) {
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vqneg_v((int8x8_t)__p0, 2);
  return __ret;
}
#else
__ai int32x2_t vqneg_s32(int32x2_t __p0) {
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vqneg_v((int8x8_t)__rev0, 2);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x4_t vqneg_s16(int16x4_t __p0) {
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vqneg_v((int8x8_t)__p0, 1);
  return __ret;
}
#else
__ai int16x4_t vqneg_s16(int16x4_t __p0) {
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vqneg_v((int8x8_t)__rev0, 1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x4_t vqrdmulhq_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vqrdmulhq_v((int8x16_t)__p0, (int8x16_t)__p1, 34);
  return __ret;
}
#else
__ai int32x4_t vqrdmulhq_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vqrdmulhq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 34);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
__ai int32x4_t __noswap_vqrdmulhq_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vqrdmulhq_v((int8x16_t)__p0, (int8x16_t)__p1, 34);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x8_t vqrdmulhq_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vqrdmulhq_v((int8x16_t)__p0, (int8x16_t)__p1, 33);
  return __ret;
}
#else
__ai int16x8_t vqrdmulhq_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vqrdmulhq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 33);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
__ai int16x8_t __noswap_vqrdmulhq_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vqrdmulhq_v((int8x16_t)__p0, (int8x16_t)__p1, 33);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x2_t vqrdmulh_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vqrdmulh_v((int8x8_t)__p0, (int8x8_t)__p1, 2);
  return __ret;
}
#else
__ai int32x2_t vqrdmulh_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vqrdmulh_v((int8x8_t)__rev0, (int8x8_t)__rev1, 2);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
__ai int32x2_t __noswap_vqrdmulh_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vqrdmulh_v((int8x8_t)__p0, (int8x8_t)__p1, 2);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x4_t vqrdmulh_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vqrdmulh_v((int8x8_t)__p0, (int8x8_t)__p1, 1);
  return __ret;
}
#else
__ai int16x4_t vqrdmulh_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vqrdmulh_v((int8x8_t)__rev0, (int8x8_t)__rev1, 1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
__ai int16x4_t __noswap_vqrdmulh_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vqrdmulh_v((int8x8_t)__p0, (int8x8_t)__p1, 1);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x4_t vqrdmulhq_n_s32(int32x4_t __p0, int32_t __p1) {
  int32x4_t __ret;
  __ret = vqrdmulhq_s32(__p0, (int32x4_t) {__p1, __p1, __p1, __p1});
  return __ret;
}
#else
__ai int32x4_t vqrdmulhq_n_s32(int32x4_t __p0, int32_t __p1) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x4_t __ret;
  __ret = __noswap_vqrdmulhq_s32(__rev0, (int32x4_t) {__p1, __p1, __p1, __p1});
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x8_t vqrdmulhq_n_s16(int16x8_t __p0, int16_t __p1) {
  int16x8_t __ret;
  __ret = vqrdmulhq_s16(__p0, (int16x8_t) {__p1, __p1, __p1, __p1, __p1, __p1, __p1, __p1});
  return __ret;
}
#else
__ai int16x8_t vqrdmulhq_n_s16(int16x8_t __p0, int16_t __p1) {
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __ret;
  __ret = __noswap_vqrdmulhq_s16(__rev0, (int16x8_t) {__p1, __p1, __p1, __p1, __p1, __p1, __p1, __p1});
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x2_t vqrdmulh_n_s32(int32x2_t __p0, int32_t __p1) {
  int32x2_t __ret;
  __ret = vqrdmulh_s32(__p0, (int32x2_t) {__p1, __p1});
  return __ret;
}
#else
__ai int32x2_t vqrdmulh_n_s32(int32x2_t __p0, int32_t __p1) {
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __ret;
  __ret = __noswap_vqrdmulh_s32(__rev0, (int32x2_t) {__p1, __p1});
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x4_t vqrdmulh_n_s16(int16x4_t __p0, int16_t __p1) {
  int16x4_t __ret;
  __ret = vqrdmulh_s16(__p0, (int16x4_t) {__p1, __p1, __p1, __p1});
  return __ret;
}
#else
__ai int16x4_t vqrdmulh_n_s16(int16x4_t __p0, int16_t __p1) {
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __ret;
  __ret = __noswap_vqrdmulh_s16(__rev0, (int16x4_t) {__p1, __p1, __p1, __p1});
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x16_t vqrshlq_u8(uint8x16_t __p0, int8x16_t __p1) {
  uint8x16_t __ret;
  __ret = (uint8x16_t) __builtin_neon_vqrshlq_v((int8x16_t)__p0, (int8x16_t)__p1, 48);
  return __ret;
}
#else
__ai uint8x16_t vqrshlq_u8(uint8x16_t __p0, int8x16_t __p1) {
  uint8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __ret;
  __ret = (uint8x16_t) __builtin_neon_vqrshlq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 48);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vqrshlq_u32(uint32x4_t __p0, int32x4_t __p1) {
  uint32x4_t __ret;
  __ret = (uint32x4_t) __builtin_neon_vqrshlq_v((int8x16_t)__p0, (int8x16_t)__p1, 50);
  return __ret;
}
#else
__ai uint32x4_t vqrshlq_u32(uint32x4_t __p0, int32x4_t __p1) {
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = (uint32x4_t) __builtin_neon_vqrshlq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 50);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint64x2_t vqrshlq_u64(uint64x2_t __p0, int64x2_t __p1) {
  uint64x2_t __ret;
  __ret = (uint64x2_t) __builtin_neon_vqrshlq_v((int8x16_t)__p0, (int8x16_t)__p1, 51);
  return __ret;
}
#else
__ai uint64x2_t vqrshlq_u64(uint64x2_t __p0, int64x2_t __p1) {
  uint64x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int64x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint64x2_t __ret;
  __ret = (uint64x2_t) __builtin_neon_vqrshlq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 51);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x8_t vqrshlq_u16(uint16x8_t __p0, int16x8_t __p1) {
  uint16x8_t __ret;
  __ret = (uint16x8_t) __builtin_neon_vqrshlq_v((int8x16_t)__p0, (int8x16_t)__p1, 49);
  return __ret;
}
#else
__ai uint16x8_t vqrshlq_u16(uint16x8_t __p0, int16x8_t __p1) {
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __ret;
  __ret = (uint16x8_t) __builtin_neon_vqrshlq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 49);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x16_t vqrshlq_s8(int8x16_t __p0, int8x16_t __p1) {
  int8x16_t __ret;
  __ret = (int8x16_t) __builtin_neon_vqrshlq_v((int8x16_t)__p0, (int8x16_t)__p1, 32);
  return __ret;
}
#else
__ai int8x16_t vqrshlq_s8(int8x16_t __p0, int8x16_t __p1) {
  int8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __ret;
  __ret = (int8x16_t) __builtin_neon_vqrshlq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 32);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x4_t vqrshlq_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vqrshlq_v((int8x16_t)__p0, (int8x16_t)__p1, 34);
  return __ret;
}
#else
__ai int32x4_t vqrshlq_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vqrshlq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 34);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int64x2_t vqrshlq_s64(int64x2_t __p0, int64x2_t __p1) {
  int64x2_t __ret;
  __ret = (int64x2_t) __builtin_neon_vqrshlq_v((int8x16_t)__p0, (int8x16_t)__p1, 35);
  return __ret;
}
#else
__ai int64x2_t vqrshlq_s64(int64x2_t __p0, int64x2_t __p1) {
  int64x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int64x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  int64x2_t __ret;
  __ret = (int64x2_t) __builtin_neon_vqrshlq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 35);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x8_t vqrshlq_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vqrshlq_v((int8x16_t)__p0, (int8x16_t)__p1, 33);
  return __ret;
}
#else
__ai int16x8_t vqrshlq_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vqrshlq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 33);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x8_t vqrshl_u8(uint8x8_t __p0, int8x8_t __p1) {
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vqrshl_v((int8x8_t)__p0, (int8x8_t)__p1, 16);
  return __ret;
}
#else
__ai uint8x8_t vqrshl_u8(uint8x8_t __p0, int8x8_t __p1) {
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vqrshl_v((int8x8_t)__rev0, (int8x8_t)__rev1, 16);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vqrshl_u32(uint32x2_t __p0, int32x2_t __p1) {
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vqrshl_v((int8x8_t)__p0, (int8x8_t)__p1, 18);
  return __ret;
}
#else
__ai uint32x2_t vqrshl_u32(uint32x2_t __p0, int32x2_t __p1) {
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vqrshl_v((int8x8_t)__rev0, (int8x8_t)__rev1, 18);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

__ai uint64x1_t vqrshl_u64(uint64x1_t __p0, int64x1_t __p1) {
  uint64x1_t __ret;
  __ret = (uint64x1_t) __builtin_neon_vqrshl_v((int8x8_t)__p0, (int8x8_t)__p1, 19);
  return __ret;
}
#ifdef __LITTLE_ENDIAN__
__ai uint16x4_t vqrshl_u16(uint16x4_t __p0, int16x4_t __p1) {
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vqrshl_v((int8x8_t)__p0, (int8x8_t)__p1, 17);
  return __ret;
}
#else
__ai uint16x4_t vqrshl_u16(uint16x4_t __p0, int16x4_t __p1) {
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vqrshl_v((int8x8_t)__rev0, (int8x8_t)__rev1, 17);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x8_t vqrshl_s8(int8x8_t __p0, int8x8_t __p1) {
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vqrshl_v((int8x8_t)__p0, (int8x8_t)__p1, 0);
  return __ret;
}
#else
__ai int8x8_t vqrshl_s8(int8x8_t __p0, int8x8_t __p1) {
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vqrshl_v((int8x8_t)__rev0, (int8x8_t)__rev1, 0);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x2_t vqrshl_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vqrshl_v((int8x8_t)__p0, (int8x8_t)__p1, 2);
  return __ret;
}
#else
__ai int32x2_t vqrshl_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vqrshl_v((int8x8_t)__rev0, (int8x8_t)__rev1, 2);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

__ai int64x1_t vqrshl_s64(int64x1_t __p0, int64x1_t __p1) {
  int64x1_t __ret;
  __ret = (int64x1_t) __builtin_neon_vqrshl_v((int8x8_t)__p0, (int8x8_t)__p1, 3);
  return __ret;
}
#ifdef __LITTLE_ENDIAN__
__ai int16x4_t vqrshl_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vqrshl_v((int8x8_t)__p0, (int8x8_t)__p1, 1);
  return __ret;
}
#else
__ai int16x4_t vqrshl_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vqrshl_v((int8x8_t)__rev0, (int8x8_t)__rev1, 1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
#define vqrshrn_n_u32(__p0, __p1) __extension__ ({ \
  uint32x4_t __s0 = __p0; \
  uint16x4_t __ret; \
  __ret = (uint16x4_t) __builtin_neon_vqrshrn_n_v((int8x16_t)__s0, __p1, 17); \
  __ret; \
})
#else
#define vqrshrn_n_u32(__p0, __p1) __extension__ ({ \
  uint32x4_t __s0 = __p0; \
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  uint16x4_t __ret; \
  __ret = (uint16x4_t) __builtin_neon_vqrshrn_n_v((int8x16_t)__rev0, __p1, 17); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_vqrshrn_n_u32(__p0, __p1) __extension__ ({ \
  uint32x4_t __s0 = __p0; \
  uint16x4_t __ret; \
  __ret = (uint16x4_t) __builtin_neon_vqrshrn_n_v((int8x16_t)__s0, __p1, 17); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vqrshrn_n_u64(__p0, __p1) __extension__ ({ \
  uint64x2_t __s0 = __p0; \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_vqrshrn_n_v((int8x16_t)__s0, __p1, 18); \
  __ret; \
})
#else
#define vqrshrn_n_u64(__p0, __p1) __extension__ ({ \
  uint64x2_t __s0 = __p0; \
  uint64x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_vqrshrn_n_v((int8x16_t)__rev0, __p1, 18); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#define __noswap_vqrshrn_n_u64(__p0, __p1) __extension__ ({ \
  uint64x2_t __s0 = __p0; \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_vqrshrn_n_v((int8x16_t)__s0, __p1, 18); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vqrshrn_n_u16(__p0, __p1) __extension__ ({ \
  uint16x8_t __s0 = __p0; \
  uint8x8_t __ret; \
  __ret = (uint8x8_t) __builtin_neon_vqrshrn_n_v((int8x16_t)__s0, __p1, 16); \
  __ret; \
})
#else
#define vqrshrn_n_u16(__p0, __p1) __extension__ ({ \
  uint16x8_t __s0 = __p0; \
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint8x8_t __ret; \
  __ret = (uint8x8_t) __builtin_neon_vqrshrn_n_v((int8x16_t)__rev0, __p1, 16); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_vqrshrn_n_u16(__p0, __p1) __extension__ ({ \
  uint16x8_t __s0 = __p0; \
  uint8x8_t __ret; \
  __ret = (uint8x8_t) __builtin_neon_vqrshrn_n_v((int8x16_t)__s0, __p1, 16); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vqrshrn_n_s32(__p0, __p1) __extension__ ({ \
  int32x4_t __s0 = __p0; \
  int16x4_t __ret; \
  __ret = (int16x4_t) __builtin_neon_vqrshrn_n_v((int8x16_t)__s0, __p1, 1); \
  __ret; \
})
#else
#define vqrshrn_n_s32(__p0, __p1) __extension__ ({ \
  int32x4_t __s0 = __p0; \
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  int16x4_t __ret; \
  __ret = (int16x4_t) __builtin_neon_vqrshrn_n_v((int8x16_t)__rev0, __p1, 1); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_vqrshrn_n_s32(__p0, __p1) __extension__ ({ \
  int32x4_t __s0 = __p0; \
  int16x4_t __ret; \
  __ret = (int16x4_t) __builtin_neon_vqrshrn_n_v((int8x16_t)__s0, __p1, 1); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vqrshrn_n_s64(__p0, __p1) __extension__ ({ \
  int64x2_t __s0 = __p0; \
  int32x2_t __ret; \
  __ret = (int32x2_t) __builtin_neon_vqrshrn_n_v((int8x16_t)__s0, __p1, 2); \
  __ret; \
})
#else
#define vqrshrn_n_s64(__p0, __p1) __extension__ ({ \
  int64x2_t __s0 = __p0; \
  int64x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  int32x2_t __ret; \
  __ret = (int32x2_t) __builtin_neon_vqrshrn_n_v((int8x16_t)__rev0, __p1, 2); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#define __noswap_vqrshrn_n_s64(__p0, __p1) __extension__ ({ \
  int64x2_t __s0 = __p0; \
  int32x2_t __ret; \
  __ret = (int32x2_t) __builtin_neon_vqrshrn_n_v((int8x16_t)__s0, __p1, 2); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vqrshrn_n_s16(__p0, __p1) __extension__ ({ \
  int16x8_t __s0 = __p0; \
  int8x8_t __ret; \
  __ret = (int8x8_t) __builtin_neon_vqrshrn_n_v((int8x16_t)__s0, __p1, 0); \
  __ret; \
})
#else
#define vqrshrn_n_s16(__p0, __p1) __extension__ ({ \
  int16x8_t __s0 = __p0; \
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  int8x8_t __ret; \
  __ret = (int8x8_t) __builtin_neon_vqrshrn_n_v((int8x16_t)__rev0, __p1, 0); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_vqrshrn_n_s16(__p0, __p1) __extension__ ({ \
  int16x8_t __s0 = __p0; \
  int8x8_t __ret; \
  __ret = (int8x8_t) __builtin_neon_vqrshrn_n_v((int8x16_t)__s0, __p1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vqrshrun_n_s32(__p0, __p1) __extension__ ({ \
  int32x4_t __s0 = __p0; \
  uint16x4_t __ret; \
  __ret = (uint16x4_t) __builtin_neon_vqrshrun_n_v((int8x16_t)__s0, __p1, 17); \
  __ret; \
})
#else
#define vqrshrun_n_s32(__p0, __p1) __extension__ ({ \
  int32x4_t __s0 = __p0; \
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  uint16x4_t __ret; \
  __ret = (uint16x4_t) __builtin_neon_vqrshrun_n_v((int8x16_t)__rev0, __p1, 17); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_vqrshrun_n_s32(__p0, __p1) __extension__ ({ \
  int32x4_t __s0 = __p0; \
  uint16x4_t __ret; \
  __ret = (uint16x4_t) __builtin_neon_vqrshrun_n_v((int8x16_t)__s0, __p1, 17); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vqrshrun_n_s64(__p0, __p1) __extension__ ({ \
  int64x2_t __s0 = __p0; \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_vqrshrun_n_v((int8x16_t)__s0, __p1, 18); \
  __ret; \
})
#else
#define vqrshrun_n_s64(__p0, __p1) __extension__ ({ \
  int64x2_t __s0 = __p0; \
  int64x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_vqrshrun_n_v((int8x16_t)__rev0, __p1, 18); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#define __noswap_vqrshrun_n_s64(__p0, __p1) __extension__ ({ \
  int64x2_t __s0 = __p0; \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_vqrshrun_n_v((int8x16_t)__s0, __p1, 18); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vqrshrun_n_s16(__p0, __p1) __extension__ ({ \
  int16x8_t __s0 = __p0; \
  uint8x8_t __ret; \
  __ret = (uint8x8_t) __builtin_neon_vqrshrun_n_v((int8x16_t)__s0, __p1, 16); \
  __ret; \
})
#else
#define vqrshrun_n_s16(__p0, __p1) __extension__ ({ \
  int16x8_t __s0 = __p0; \
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint8x8_t __ret; \
  __ret = (uint8x8_t) __builtin_neon_vqrshrun_n_v((int8x16_t)__rev0, __p1, 16); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_vqrshrun_n_s16(__p0, __p1) __extension__ ({ \
  int16x8_t __s0 = __p0; \
  uint8x8_t __ret; \
  __ret = (uint8x8_t) __builtin_neon_vqrshrun_n_v((int8x16_t)__s0, __p1, 16); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x16_t vqshlq_u8(uint8x16_t __p0, int8x16_t __p1) {
  uint8x16_t __ret;
  __ret = (uint8x16_t) __builtin_neon_vqshlq_v((int8x16_t)__p0, (int8x16_t)__p1, 48);
  return __ret;
}
#else
__ai uint8x16_t vqshlq_u8(uint8x16_t __p0, int8x16_t __p1) {
  uint8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __ret;
  __ret = (uint8x16_t) __builtin_neon_vqshlq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 48);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vqshlq_u32(uint32x4_t __p0, int32x4_t __p1) {
  uint32x4_t __ret;
  __ret = (uint32x4_t) __builtin_neon_vqshlq_v((int8x16_t)__p0, (int8x16_t)__p1, 50);
  return __ret;
}
#else
__ai uint32x4_t vqshlq_u32(uint32x4_t __p0, int32x4_t __p1) {
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = (uint32x4_t) __builtin_neon_vqshlq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 50);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint64x2_t vqshlq_u64(uint64x2_t __p0, int64x2_t __p1) {
  uint64x2_t __ret;
  __ret = (uint64x2_t) __builtin_neon_vqshlq_v((int8x16_t)__p0, (int8x16_t)__p1, 51);
  return __ret;
}
#else
__ai uint64x2_t vqshlq_u64(uint64x2_t __p0, int64x2_t __p1) {
  uint64x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int64x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint64x2_t __ret;
  __ret = (uint64x2_t) __builtin_neon_vqshlq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 51);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x8_t vqshlq_u16(uint16x8_t __p0, int16x8_t __p1) {
  uint16x8_t __ret;
  __ret = (uint16x8_t) __builtin_neon_vqshlq_v((int8x16_t)__p0, (int8x16_t)__p1, 49);
  return __ret;
}
#else
__ai uint16x8_t vqshlq_u16(uint16x8_t __p0, int16x8_t __p1) {
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __ret;
  __ret = (uint16x8_t) __builtin_neon_vqshlq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 49);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x16_t vqshlq_s8(int8x16_t __p0, int8x16_t __p1) {
  int8x16_t __ret;
  __ret = (int8x16_t) __builtin_neon_vqshlq_v((int8x16_t)__p0, (int8x16_t)__p1, 32);
  return __ret;
}
#else
__ai int8x16_t vqshlq_s8(int8x16_t __p0, int8x16_t __p1) {
  int8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __ret;
  __ret = (int8x16_t) __builtin_neon_vqshlq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 32);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x4_t vqshlq_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vqshlq_v((int8x16_t)__p0, (int8x16_t)__p1, 34);
  return __ret;
}
#else
__ai int32x4_t vqshlq_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vqshlq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 34);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int64x2_t vqshlq_s64(int64x2_t __p0, int64x2_t __p1) {
  int64x2_t __ret;
  __ret = (int64x2_t) __builtin_neon_vqshlq_v((int8x16_t)__p0, (int8x16_t)__p1, 35);
  return __ret;
}
#else
__ai int64x2_t vqshlq_s64(int64x2_t __p0, int64x2_t __p1) {
  int64x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int64x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  int64x2_t __ret;
  __ret = (int64x2_t) __builtin_neon_vqshlq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 35);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x8_t vqshlq_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vqshlq_v((int8x16_t)__p0, (int8x16_t)__p1, 33);
  return __ret;
}
#else
__ai int16x8_t vqshlq_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vqshlq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 33);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x8_t vqshl_u8(uint8x8_t __p0, int8x8_t __p1) {
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vqshl_v((int8x8_t)__p0, (int8x8_t)__p1, 16);
  return __ret;
}
#else
__ai uint8x8_t vqshl_u8(uint8x8_t __p0, int8x8_t __p1) {
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vqshl_v((int8x8_t)__rev0, (int8x8_t)__rev1, 16);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vqshl_u32(uint32x2_t __p0, int32x2_t __p1) {
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vqshl_v((int8x8_t)__p0, (int8x8_t)__p1, 18);
  return __ret;
}
#else
__ai uint32x2_t vqshl_u32(uint32x2_t __p0, int32x2_t __p1) {
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vqshl_v((int8x8_t)__rev0, (int8x8_t)__rev1, 18);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

__ai uint64x1_t vqshl_u64(uint64x1_t __p0, int64x1_t __p1) {
  uint64x1_t __ret;
  __ret = (uint64x1_t) __builtin_neon_vqshl_v((int8x8_t)__p0, (int8x8_t)__p1, 19);
  return __ret;
}
#ifdef __LITTLE_ENDIAN__
__ai uint16x4_t vqshl_u16(uint16x4_t __p0, int16x4_t __p1) {
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vqshl_v((int8x8_t)__p0, (int8x8_t)__p1, 17);
  return __ret;
}
#else
__ai uint16x4_t vqshl_u16(uint16x4_t __p0, int16x4_t __p1) {
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vqshl_v((int8x8_t)__rev0, (int8x8_t)__rev1, 17);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x8_t vqshl_s8(int8x8_t __p0, int8x8_t __p1) {
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vqshl_v((int8x8_t)__p0, (int8x8_t)__p1, 0);
  return __ret;
}
#else
__ai int8x8_t vqshl_s8(int8x8_t __p0, int8x8_t __p1) {
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vqshl_v((int8x8_t)__rev0, (int8x8_t)__rev1, 0);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x2_t vqshl_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vqshl_v((int8x8_t)__p0, (int8x8_t)__p1, 2);
  return __ret;
}
#else
__ai int32x2_t vqshl_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vqshl_v((int8x8_t)__rev0, (int8x8_t)__rev1, 2);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

__ai int64x1_t vqshl_s64(int64x1_t __p0, int64x1_t __p1) {
  int64x1_t __ret;
  __ret = (int64x1_t) __builtin_neon_vqshl_v((int8x8_t)__p0, (int8x8_t)__p1, 3);
  return __ret;
}
#ifdef __LITTLE_ENDIAN__
__ai int16x4_t vqshl_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vqshl_v((int8x8_t)__p0, (int8x8_t)__p1, 1);
  return __ret;
}
#else
__ai int16x4_t vqshl_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vqshl_v((int8x8_t)__rev0, (int8x8_t)__rev1, 1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
#define vqshlq_n_u8(__p0, __p1) __extension__ ({ \
  uint8x16_t __s0 = __p0; \
  uint8x16_t __ret; \
  __ret = (uint8x16_t) __builtin_neon_vqshlq_n_v((int8x16_t)__s0, __p1, 48); \
  __ret; \
})
#else
#define vqshlq_n_u8(__p0, __p1) __extension__ ({ \
  uint8x16_t __s0 = __p0; \
  uint8x16_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint8x16_t __ret; \
  __ret = (uint8x16_t) __builtin_neon_vqshlq_n_v((int8x16_t)__rev0, __p1, 48); \
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vqshlq_n_u32(__p0, __p1) __extension__ ({ \
  uint32x4_t __s0 = __p0; \
  uint32x4_t __ret; \
  __ret = (uint32x4_t) __builtin_neon_vqshlq_n_v((int8x16_t)__s0, __p1, 50); \
  __ret; \
})
#else
#define vqshlq_n_u32(__p0, __p1) __extension__ ({ \
  uint32x4_t __s0 = __p0; \
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  uint32x4_t __ret; \
  __ret = (uint32x4_t) __builtin_neon_vqshlq_n_v((int8x16_t)__rev0, __p1, 50); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vqshlq_n_u64(__p0, __p1) __extension__ ({ \
  uint64x2_t __s0 = __p0; \
  uint64x2_t __ret; \
  __ret = (uint64x2_t) __builtin_neon_vqshlq_n_v((int8x16_t)__s0, __p1, 51); \
  __ret; \
})
#else
#define vqshlq_n_u64(__p0, __p1) __extension__ ({ \
  uint64x2_t __s0 = __p0; \
  uint64x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  uint64x2_t __ret; \
  __ret = (uint64x2_t) __builtin_neon_vqshlq_n_v((int8x16_t)__rev0, __p1, 51); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vqshlq_n_u16(__p0, __p1) __extension__ ({ \
  uint16x8_t __s0 = __p0; \
  uint16x8_t __ret; \
  __ret = (uint16x8_t) __builtin_neon_vqshlq_n_v((int8x16_t)__s0, __p1, 49); \
  __ret; \
})
#else
#define vqshlq_n_u16(__p0, __p1) __extension__ ({ \
  uint16x8_t __s0 = __p0; \
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint16x8_t __ret; \
  __ret = (uint16x8_t) __builtin_neon_vqshlq_n_v((int8x16_t)__rev0, __p1, 49); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vqshlq_n_s8(__p0, __p1) __extension__ ({ \
  int8x16_t __s0 = __p0; \
  int8x16_t __ret; \
  __ret = (int8x16_t) __builtin_neon_vqshlq_n_v((int8x16_t)__s0, __p1, 32); \
  __ret; \
})
#else
#define vqshlq_n_s8(__p0, __p1) __extension__ ({ \
  int8x16_t __s0 = __p0; \
  int8x16_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  int8x16_t __ret; \
  __ret = (int8x16_t) __builtin_neon_vqshlq_n_v((int8x16_t)__rev0, __p1, 32); \
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vqshlq_n_s32(__p0, __p1) __extension__ ({ \
  int32x4_t __s0 = __p0; \
  int32x4_t __ret; \
  __ret = (int32x4_t) __builtin_neon_vqshlq_n_v((int8x16_t)__s0, __p1, 34); \
  __ret; \
})
#else
#define vqshlq_n_s32(__p0, __p1) __extension__ ({ \
  int32x4_t __s0 = __p0; \
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  int32x4_t __ret; \
  __ret = (int32x4_t) __builtin_neon_vqshlq_n_v((int8x16_t)__rev0, __p1, 34); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vqshlq_n_s64(__p0, __p1) __extension__ ({ \
  int64x2_t __s0 = __p0; \
  int64x2_t __ret; \
  __ret = (int64x2_t) __builtin_neon_vqshlq_n_v((int8x16_t)__s0, __p1, 35); \
  __ret; \
})
#else
#define vqshlq_n_s64(__p0, __p1) __extension__ ({ \
  int64x2_t __s0 = __p0; \
  int64x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  int64x2_t __ret; \
  __ret = (int64x2_t) __builtin_neon_vqshlq_n_v((int8x16_t)__rev0, __p1, 35); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vqshlq_n_s16(__p0, __p1) __extension__ ({ \
  int16x8_t __s0 = __p0; \
  int16x8_t __ret; \
  __ret = (int16x8_t) __builtin_neon_vqshlq_n_v((int8x16_t)__s0, __p1, 33); \
  __ret; \
})
#else
#define vqshlq_n_s16(__p0, __p1) __extension__ ({ \
  int16x8_t __s0 = __p0; \
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  int16x8_t __ret; \
  __ret = (int16x8_t) __builtin_neon_vqshlq_n_v((int8x16_t)__rev0, __p1, 33); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vqshl_n_u8(__p0, __p1) __extension__ ({ \
  uint8x8_t __s0 = __p0; \
  uint8x8_t __ret; \
  __ret = (uint8x8_t) __builtin_neon_vqshl_n_v((int8x8_t)__s0, __p1, 16); \
  __ret; \
})
#else
#define vqshl_n_u8(__p0, __p1) __extension__ ({ \
  uint8x8_t __s0 = __p0; \
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint8x8_t __ret; \
  __ret = (uint8x8_t) __builtin_neon_vqshl_n_v((int8x8_t)__rev0, __p1, 16); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vqshl_n_u32(__p0, __p1) __extension__ ({ \
  uint32x2_t __s0 = __p0; \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_vqshl_n_v((int8x8_t)__s0, __p1, 18); \
  __ret; \
})
#else
#define vqshl_n_u32(__p0, __p1) __extension__ ({ \
  uint32x2_t __s0 = __p0; \
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_vqshl_n_v((int8x8_t)__rev0, __p1, 18); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#define vqshl_n_u64(__p0, __p1) __extension__ ({ \
  uint64x1_t __s0 = __p0; \
  uint64x1_t __ret; \
  __ret = (uint64x1_t) __builtin_neon_vqshl_n_v((int8x8_t)__s0, __p1, 19); \
  __ret; \
})
#ifdef __LITTLE_ENDIAN__
#define vqshl_n_u16(__p0, __p1) __extension__ ({ \
  uint16x4_t __s0 = __p0; \
  uint16x4_t __ret; \
  __ret = (uint16x4_t) __builtin_neon_vqshl_n_v((int8x8_t)__s0, __p1, 17); \
  __ret; \
})
#else
#define vqshl_n_u16(__p0, __p1) __extension__ ({ \
  uint16x4_t __s0 = __p0; \
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  uint16x4_t __ret; \
  __ret = (uint16x4_t) __builtin_neon_vqshl_n_v((int8x8_t)__rev0, __p1, 17); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vqshl_n_s8(__p0, __p1) __extension__ ({ \
  int8x8_t __s0 = __p0; \
  int8x8_t __ret; \
  __ret = (int8x8_t) __builtin_neon_vqshl_n_v((int8x8_t)__s0, __p1, 0); \
  __ret; \
})
#else
#define vqshl_n_s8(__p0, __p1) __extension__ ({ \
  int8x8_t __s0 = __p0; \
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  int8x8_t __ret; \
  __ret = (int8x8_t) __builtin_neon_vqshl_n_v((int8x8_t)__rev0, __p1, 0); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vqshl_n_s32(__p0, __p1) __extension__ ({ \
  int32x2_t __s0 = __p0; \
  int32x2_t __ret; \
  __ret = (int32x2_t) __builtin_neon_vqshl_n_v((int8x8_t)__s0, __p1, 2); \
  __ret; \
})
#else
#define vqshl_n_s32(__p0, __p1) __extension__ ({ \
  int32x2_t __s0 = __p0; \
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  int32x2_t __ret; \
  __ret = (int32x2_t) __builtin_neon_vqshl_n_v((int8x8_t)__rev0, __p1, 2); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#define vqshl_n_s64(__p0, __p1) __extension__ ({ \
  int64x1_t __s0 = __p0; \
  int64x1_t __ret; \
  __ret = (int64x1_t) __builtin_neon_vqshl_n_v((int8x8_t)__s0, __p1, 3); \
  __ret; \
})
#ifdef __LITTLE_ENDIAN__
#define vqshl_n_s16(__p0, __p1) __extension__ ({ \
  int16x4_t __s0 = __p0; \
  int16x4_t __ret; \
  __ret = (int16x4_t) __builtin_neon_vqshl_n_v((int8x8_t)__s0, __p1, 1); \
  __ret; \
})
#else
#define vqshl_n_s16(__p0, __p1) __extension__ ({ \
  int16x4_t __s0 = __p0; \
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  int16x4_t __ret; \
  __ret = (int16x4_t) __builtin_neon_vqshl_n_v((int8x8_t)__rev0, __p1, 1); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vqshluq_n_s8(__p0, __p1) __extension__ ({ \
  int8x16_t __s0 = __p0; \
  uint8x16_t __ret; \
  __ret = (uint8x16_t) __builtin_neon_vqshluq_n_v((int8x16_t)__s0, __p1, 48); \
  __ret; \
})
#else
#define vqshluq_n_s8(__p0, __p1) __extension__ ({ \
  int8x16_t __s0 = __p0; \
  int8x16_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint8x16_t __ret; \
  __ret = (uint8x16_t) __builtin_neon_vqshluq_n_v((int8x16_t)__rev0, __p1, 48); \
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vqshluq_n_s32(__p0, __p1) __extension__ ({ \
  int32x4_t __s0 = __p0; \
  uint32x4_t __ret; \
  __ret = (uint32x4_t) __builtin_neon_vqshluq_n_v((int8x16_t)__s0, __p1, 50); \
  __ret; \
})
#else
#define vqshluq_n_s32(__p0, __p1) __extension__ ({ \
  int32x4_t __s0 = __p0; \
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  uint32x4_t __ret; \
  __ret = (uint32x4_t) __builtin_neon_vqshluq_n_v((int8x16_t)__rev0, __p1, 50); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vqshluq_n_s64(__p0, __p1) __extension__ ({ \
  int64x2_t __s0 = __p0; \
  uint64x2_t __ret; \
  __ret = (uint64x2_t) __builtin_neon_vqshluq_n_v((int8x16_t)__s0, __p1, 51); \
  __ret; \
})
#else
#define vqshluq_n_s64(__p0, __p1) __extension__ ({ \
  int64x2_t __s0 = __p0; \
  int64x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  uint64x2_t __ret; \
  __ret = (uint64x2_t) __builtin_neon_vqshluq_n_v((int8x16_t)__rev0, __p1, 51); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vqshluq_n_s16(__p0, __p1) __extension__ ({ \
  int16x8_t __s0 = __p0; \
  uint16x8_t __ret; \
  __ret = (uint16x8_t) __builtin_neon_vqshluq_n_v((int8x16_t)__s0, __p1, 49); \
  __ret; \
})
#else
#define vqshluq_n_s16(__p0, __p1) __extension__ ({ \
  int16x8_t __s0 = __p0; \
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint16x8_t __ret; \
  __ret = (uint16x8_t) __builtin_neon_vqshluq_n_v((int8x16_t)__rev0, __p1, 49); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vqshlu_n_s8(__p0, __p1) __extension__ ({ \
  int8x8_t __s0 = __p0; \
  uint8x8_t __ret; \
  __ret = (uint8x8_t) __builtin_neon_vqshlu_n_v((int8x8_t)__s0, __p1, 16); \
  __ret; \
})
#else
#define vqshlu_n_s8(__p0, __p1) __extension__ ({ \
  int8x8_t __s0 = __p0; \
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint8x8_t __ret; \
  __ret = (uint8x8_t) __builtin_neon_vqshlu_n_v((int8x8_t)__rev0, __p1, 16); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vqshlu_n_s32(__p0, __p1) __extension__ ({ \
  int32x2_t __s0 = __p0; \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_vqshlu_n_v((int8x8_t)__s0, __p1, 18); \
  __ret; \
})
#else
#define vqshlu_n_s32(__p0, __p1) __extension__ ({ \
  int32x2_t __s0 = __p0; \
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_vqshlu_n_v((int8x8_t)__rev0, __p1, 18); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#define vqshlu_n_s64(__p0, __p1) __extension__ ({ \
  int64x1_t __s0 = __p0; \
  uint64x1_t __ret; \
  __ret = (uint64x1_t) __builtin_neon_vqshlu_n_v((int8x8_t)__s0, __p1, 19); \
  __ret; \
})
#ifdef __LITTLE_ENDIAN__
#define vqshlu_n_s16(__p0, __p1) __extension__ ({ \
  int16x4_t __s0 = __p0; \
  uint16x4_t __ret; \
  __ret = (uint16x4_t) __builtin_neon_vqshlu_n_v((int8x8_t)__s0, __p1, 17); \
  __ret; \
})
#else
#define vqshlu_n_s16(__p0, __p1) __extension__ ({ \
  int16x4_t __s0 = __p0; \
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  uint16x4_t __ret; \
  __ret = (uint16x4_t) __builtin_neon_vqshlu_n_v((int8x8_t)__rev0, __p1, 17); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vqshrn_n_u32(__p0, __p1) __extension__ ({ \
  uint32x4_t __s0 = __p0; \
  uint16x4_t __ret; \
  __ret = (uint16x4_t) __builtin_neon_vqshrn_n_v((int8x16_t)__s0, __p1, 17); \
  __ret; \
})
#else
#define vqshrn_n_u32(__p0, __p1) __extension__ ({ \
  uint32x4_t __s0 = __p0; \
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  uint16x4_t __ret; \
  __ret = (uint16x4_t) __builtin_neon_vqshrn_n_v((int8x16_t)__rev0, __p1, 17); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_vqshrn_n_u32(__p0, __p1) __extension__ ({ \
  uint32x4_t __s0 = __p0; \
  uint16x4_t __ret; \
  __ret = (uint16x4_t) __builtin_neon_vqshrn_n_v((int8x16_t)__s0, __p1, 17); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vqshrn_n_u64(__p0, __p1) __extension__ ({ \
  uint64x2_t __s0 = __p0; \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_vqshrn_n_v((int8x16_t)__s0, __p1, 18); \
  __ret; \
})
#else
#define vqshrn_n_u64(__p0, __p1) __extension__ ({ \
  uint64x2_t __s0 = __p0; \
  uint64x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_vqshrn_n_v((int8x16_t)__rev0, __p1, 18); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#define __noswap_vqshrn_n_u64(__p0, __p1) __extension__ ({ \
  uint64x2_t __s0 = __p0; \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_vqshrn_n_v((int8x16_t)__s0, __p1, 18); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vqshrn_n_u16(__p0, __p1) __extension__ ({ \
  uint16x8_t __s0 = __p0; \
  uint8x8_t __ret; \
  __ret = (uint8x8_t) __builtin_neon_vqshrn_n_v((int8x16_t)__s0, __p1, 16); \
  __ret; \
})
#else
#define vqshrn_n_u16(__p0, __p1) __extension__ ({ \
  uint16x8_t __s0 = __p0; \
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint8x8_t __ret; \
  __ret = (uint8x8_t) __builtin_neon_vqshrn_n_v((int8x16_t)__rev0, __p1, 16); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_vqshrn_n_u16(__p0, __p1) __extension__ ({ \
  uint16x8_t __s0 = __p0; \
  uint8x8_t __ret; \
  __ret = (uint8x8_t) __builtin_neon_vqshrn_n_v((int8x16_t)__s0, __p1, 16); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vqshrn_n_s32(__p0, __p1) __extension__ ({ \
  int32x4_t __s0 = __p0; \
  int16x4_t __ret; \
  __ret = (int16x4_t) __builtin_neon_vqshrn_n_v((int8x16_t)__s0, __p1, 1); \
  __ret; \
})
#else
#define vqshrn_n_s32(__p0, __p1) __extension__ ({ \
  int32x4_t __s0 = __p0; \
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  int16x4_t __ret; \
  __ret = (int16x4_t) __builtin_neon_vqshrn_n_v((int8x16_t)__rev0, __p1, 1); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_vqshrn_n_s32(__p0, __p1) __extension__ ({ \
  int32x4_t __s0 = __p0; \
  int16x4_t __ret; \
  __ret = (int16x4_t) __builtin_neon_vqshrn_n_v((int8x16_t)__s0, __p1, 1); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vqshrn_n_s64(__p0, __p1) __extension__ ({ \
  int64x2_t __s0 = __p0; \
  int32x2_t __ret; \
  __ret = (int32x2_t) __builtin_neon_vqshrn_n_v((int8x16_t)__s0, __p1, 2); \
  __ret; \
})
#else
#define vqshrn_n_s64(__p0, __p1) __extension__ ({ \
  int64x2_t __s0 = __p0; \
  int64x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  int32x2_t __ret; \
  __ret = (int32x2_t) __builtin_neon_vqshrn_n_v((int8x16_t)__rev0, __p1, 2); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#define __noswap_vqshrn_n_s64(__p0, __p1) __extension__ ({ \
  int64x2_t __s0 = __p0; \
  int32x2_t __ret; \
  __ret = (int32x2_t) __builtin_neon_vqshrn_n_v((int8x16_t)__s0, __p1, 2); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vqshrn_n_s16(__p0, __p1) __extension__ ({ \
  int16x8_t __s0 = __p0; \
  int8x8_t __ret; \
  __ret = (int8x8_t) __builtin_neon_vqshrn_n_v((int8x16_t)__s0, __p1, 0); \
  __ret; \
})
#else
#define vqshrn_n_s16(__p0, __p1) __extension__ ({ \
  int16x8_t __s0 = __p0; \
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  int8x8_t __ret; \
  __ret = (int8x8_t) __builtin_neon_vqshrn_n_v((int8x16_t)__rev0, __p1, 0); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_vqshrn_n_s16(__p0, __p1) __extension__ ({ \
  int16x8_t __s0 = __p0; \
  int8x8_t __ret; \
  __ret = (int8x8_t) __builtin_neon_vqshrn_n_v((int8x16_t)__s0, __p1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vqshrun_n_s32(__p0, __p1) __extension__ ({ \
  int32x4_t __s0 = __p0; \
  uint16x4_t __ret; \
  __ret = (uint16x4_t) __builtin_neon_vqshrun_n_v((int8x16_t)__s0, __p1, 17); \
  __ret; \
})
#else
#define vqshrun_n_s32(__p0, __p1) __extension__ ({ \
  int32x4_t __s0 = __p0; \
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  uint16x4_t __ret; \
  __ret = (uint16x4_t) __builtin_neon_vqshrun_n_v((int8x16_t)__rev0, __p1, 17); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_vqshrun_n_s32(__p0, __p1) __extension__ ({ \
  int32x4_t __s0 = __p0; \
  uint16x4_t __ret; \
  __ret = (uint16x4_t) __builtin_neon_vqshrun_n_v((int8x16_t)__s0, __p1, 17); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vqshrun_n_s64(__p0, __p1) __extension__ ({ \
  int64x2_t __s0 = __p0; \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_vqshrun_n_v((int8x16_t)__s0, __p1, 18); \
  __ret; \
})
#else
#define vqshrun_n_s64(__p0, __p1) __extension__ ({ \
  int64x2_t __s0 = __p0; \
  int64x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_vqshrun_n_v((int8x16_t)__rev0, __p1, 18); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#define __noswap_vqshrun_n_s64(__p0, __p1) __extension__ ({ \
  int64x2_t __s0 = __p0; \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_vqshrun_n_v((int8x16_t)__s0, __p1, 18); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vqshrun_n_s16(__p0, __p1) __extension__ ({ \
  int16x8_t __s0 = __p0; \
  uint8x8_t __ret; \
  __ret = (uint8x8_t) __builtin_neon_vqshrun_n_v((int8x16_t)__s0, __p1, 16); \
  __ret; \
})
#else
#define vqshrun_n_s16(__p0, __p1) __extension__ ({ \
  int16x8_t __s0 = __p0; \
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint8x8_t __ret; \
  __ret = (uint8x8_t) __builtin_neon_vqshrun_n_v((int8x16_t)__rev0, __p1, 16); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_vqshrun_n_s16(__p0, __p1) __extension__ ({ \
  int16x8_t __s0 = __p0; \
  uint8x8_t __ret; \
  __ret = (uint8x8_t) __builtin_neon_vqshrun_n_v((int8x16_t)__s0, __p1, 16); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x16_t vqsubq_u8(uint8x16_t __p0, uint8x16_t __p1) {
  uint8x16_t __ret;
  __ret = (uint8x16_t) __builtin_neon_vqsubq_v((int8x16_t)__p0, (int8x16_t)__p1, 48);
  return __ret;
}
#else
__ai uint8x16_t vqsubq_u8(uint8x16_t __p0, uint8x16_t __p1) {
  uint8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __ret;
  __ret = (uint8x16_t) __builtin_neon_vqsubq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 48);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vqsubq_u32(uint32x4_t __p0, uint32x4_t __p1) {
  uint32x4_t __ret;
  __ret = (uint32x4_t) __builtin_neon_vqsubq_v((int8x16_t)__p0, (int8x16_t)__p1, 50);
  return __ret;
}
#else
__ai uint32x4_t vqsubq_u32(uint32x4_t __p0, uint32x4_t __p1) {
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = (uint32x4_t) __builtin_neon_vqsubq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 50);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint64x2_t vqsubq_u64(uint64x2_t __p0, uint64x2_t __p1) {
  uint64x2_t __ret;
  __ret = (uint64x2_t) __builtin_neon_vqsubq_v((int8x16_t)__p0, (int8x16_t)__p1, 51);
  return __ret;
}
#else
__ai uint64x2_t vqsubq_u64(uint64x2_t __p0, uint64x2_t __p1) {
  uint64x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint64x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint64x2_t __ret;
  __ret = (uint64x2_t) __builtin_neon_vqsubq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 51);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x8_t vqsubq_u16(uint16x8_t __p0, uint16x8_t __p1) {
  uint16x8_t __ret;
  __ret = (uint16x8_t) __builtin_neon_vqsubq_v((int8x16_t)__p0, (int8x16_t)__p1, 49);
  return __ret;
}
#else
__ai uint16x8_t vqsubq_u16(uint16x8_t __p0, uint16x8_t __p1) {
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __ret;
  __ret = (uint16x8_t) __builtin_neon_vqsubq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 49);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x16_t vqsubq_s8(int8x16_t __p0, int8x16_t __p1) {
  int8x16_t __ret;
  __ret = (int8x16_t) __builtin_neon_vqsubq_v((int8x16_t)__p0, (int8x16_t)__p1, 32);
  return __ret;
}
#else
__ai int8x16_t vqsubq_s8(int8x16_t __p0, int8x16_t __p1) {
  int8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __ret;
  __ret = (int8x16_t) __builtin_neon_vqsubq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 32);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x4_t vqsubq_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vqsubq_v((int8x16_t)__p0, (int8x16_t)__p1, 34);
  return __ret;
}
#else
__ai int32x4_t vqsubq_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vqsubq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 34);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
__ai int32x4_t __noswap_vqsubq_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vqsubq_v((int8x16_t)__p0, (int8x16_t)__p1, 34);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int64x2_t vqsubq_s64(int64x2_t __p0, int64x2_t __p1) {
  int64x2_t __ret;
  __ret = (int64x2_t) __builtin_neon_vqsubq_v((int8x16_t)__p0, (int8x16_t)__p1, 35);
  return __ret;
}
#else
__ai int64x2_t vqsubq_s64(int64x2_t __p0, int64x2_t __p1) {
  int64x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int64x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  int64x2_t __ret;
  __ret = (int64x2_t) __builtin_neon_vqsubq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 35);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x8_t vqsubq_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vqsubq_v((int8x16_t)__p0, (int8x16_t)__p1, 33);
  return __ret;
}
#else
__ai int16x8_t vqsubq_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vqsubq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 33);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
__ai int16x8_t __noswap_vqsubq_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vqsubq_v((int8x16_t)__p0, (int8x16_t)__p1, 33);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x8_t vqsub_u8(uint8x8_t __p0, uint8x8_t __p1) {
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vqsub_v((int8x8_t)__p0, (int8x8_t)__p1, 16);
  return __ret;
}
#else
__ai uint8x8_t vqsub_u8(uint8x8_t __p0, uint8x8_t __p1) {
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vqsub_v((int8x8_t)__rev0, (int8x8_t)__rev1, 16);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vqsub_u32(uint32x2_t __p0, uint32x2_t __p1) {
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vqsub_v((int8x8_t)__p0, (int8x8_t)__p1, 18);
  return __ret;
}
#else
__ai uint32x2_t vqsub_u32(uint32x2_t __p0, uint32x2_t __p1) {
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vqsub_v((int8x8_t)__rev0, (int8x8_t)__rev1, 18);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

__ai uint64x1_t vqsub_u64(uint64x1_t __p0, uint64x1_t __p1) {
  uint64x1_t __ret;
  __ret = (uint64x1_t) __builtin_neon_vqsub_v((int8x8_t)__p0, (int8x8_t)__p1, 19);
  return __ret;
}
#ifdef __LITTLE_ENDIAN__
__ai uint16x4_t vqsub_u16(uint16x4_t __p0, uint16x4_t __p1) {
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vqsub_v((int8x8_t)__p0, (int8x8_t)__p1, 17);
  return __ret;
}
#else
__ai uint16x4_t vqsub_u16(uint16x4_t __p0, uint16x4_t __p1) {
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vqsub_v((int8x8_t)__rev0, (int8x8_t)__rev1, 17);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x8_t vqsub_s8(int8x8_t __p0, int8x8_t __p1) {
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vqsub_v((int8x8_t)__p0, (int8x8_t)__p1, 0);
  return __ret;
}
#else
__ai int8x8_t vqsub_s8(int8x8_t __p0, int8x8_t __p1) {
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vqsub_v((int8x8_t)__rev0, (int8x8_t)__rev1, 0);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x2_t vqsub_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vqsub_v((int8x8_t)__p0, (int8x8_t)__p1, 2);
  return __ret;
}
#else
__ai int32x2_t vqsub_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vqsub_v((int8x8_t)__rev0, (int8x8_t)__rev1, 2);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
__ai int32x2_t __noswap_vqsub_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vqsub_v((int8x8_t)__p0, (int8x8_t)__p1, 2);
  return __ret;
}
#endif

__ai int64x1_t vqsub_s64(int64x1_t __p0, int64x1_t __p1) {
  int64x1_t __ret;
  __ret = (int64x1_t) __builtin_neon_vqsub_v((int8x8_t)__p0, (int8x8_t)__p1, 3);
  return __ret;
}
#ifdef __LITTLE_ENDIAN__
__ai int16x4_t vqsub_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vqsub_v((int8x8_t)__p0, (int8x8_t)__p1, 1);
  return __ret;
}
#else
__ai int16x4_t vqsub_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vqsub_v((int8x8_t)__rev0, (int8x8_t)__rev1, 1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
__ai int16x4_t __noswap_vqsub_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vqsub_v((int8x8_t)__p0, (int8x8_t)__p1, 1);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x4_t vraddhn_u32(uint32x4_t __p0, uint32x4_t __p1) {
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vraddhn_v((int8x16_t)__p0, (int8x16_t)__p1, 17);
  return __ret;
}
#else
__ai uint16x4_t vraddhn_u32(uint32x4_t __p0, uint32x4_t __p1) {
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vraddhn_v((int8x16_t)__rev0, (int8x16_t)__rev1, 17);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
__ai uint16x4_t __noswap_vraddhn_u32(uint32x4_t __p0, uint32x4_t __p1) {
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vraddhn_v((int8x16_t)__p0, (int8x16_t)__p1, 17);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vraddhn_u64(uint64x2_t __p0, uint64x2_t __p1) {
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vraddhn_v((int8x16_t)__p0, (int8x16_t)__p1, 18);
  return __ret;
}
#else
__ai uint32x2_t vraddhn_u64(uint64x2_t __p0, uint64x2_t __p1) {
  uint64x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint64x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vraddhn_v((int8x16_t)__rev0, (int8x16_t)__rev1, 18);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
__ai uint32x2_t __noswap_vraddhn_u64(uint64x2_t __p0, uint64x2_t __p1) {
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vraddhn_v((int8x16_t)__p0, (int8x16_t)__p1, 18);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x8_t vraddhn_u16(uint16x8_t __p0, uint16x8_t __p1) {
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vraddhn_v((int8x16_t)__p0, (int8x16_t)__p1, 16);
  return __ret;
}
#else
__ai uint8x8_t vraddhn_u16(uint16x8_t __p0, uint16x8_t __p1) {
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vraddhn_v((int8x16_t)__rev0, (int8x16_t)__rev1, 16);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
__ai uint8x8_t __noswap_vraddhn_u16(uint16x8_t __p0, uint16x8_t __p1) {
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vraddhn_v((int8x16_t)__p0, (int8x16_t)__p1, 16);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x4_t vraddhn_s32(int32x4_t __p0, int32x4_t __p1) {
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vraddhn_v((int8x16_t)__p0, (int8x16_t)__p1, 1);
  return __ret;
}
#else
__ai int16x4_t vraddhn_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vraddhn_v((int8x16_t)__rev0, (int8x16_t)__rev1, 1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
__ai int16x4_t __noswap_vraddhn_s32(int32x4_t __p0, int32x4_t __p1) {
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vraddhn_v((int8x16_t)__p0, (int8x16_t)__p1, 1);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x2_t vraddhn_s64(int64x2_t __p0, int64x2_t __p1) {
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vraddhn_v((int8x16_t)__p0, (int8x16_t)__p1, 2);
  return __ret;
}
#else
__ai int32x2_t vraddhn_s64(int64x2_t __p0, int64x2_t __p1) {
  int64x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int64x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vraddhn_v((int8x16_t)__rev0, (int8x16_t)__rev1, 2);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
__ai int32x2_t __noswap_vraddhn_s64(int64x2_t __p0, int64x2_t __p1) {
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vraddhn_v((int8x16_t)__p0, (int8x16_t)__p1, 2);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x8_t vraddhn_s16(int16x8_t __p0, int16x8_t __p1) {
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vraddhn_v((int8x16_t)__p0, (int8x16_t)__p1, 0);
  return __ret;
}
#else
__ai int8x8_t vraddhn_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vraddhn_v((int8x16_t)__rev0, (int8x16_t)__rev1, 0);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
__ai int8x8_t __noswap_vraddhn_s16(int16x8_t __p0, int16x8_t __p1) {
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vraddhn_v((int8x16_t)__p0, (int8x16_t)__p1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vrecpeq_u32(uint32x4_t __p0) {
  uint32x4_t __ret;
  __ret = (uint32x4_t) __builtin_neon_vrecpeq_v((int8x16_t)__p0, 50);
  return __ret;
}
#else
__ai uint32x4_t vrecpeq_u32(uint32x4_t __p0) {
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = (uint32x4_t) __builtin_neon_vrecpeq_v((int8x16_t)__rev0, 50);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai float32x4_t vrecpeq_f32(float32x4_t __p0) {
  float32x4_t __ret;
  __ret = (float32x4_t) __builtin_neon_vrecpeq_v((int8x16_t)__p0, 41);
  return __ret;
}
#else
__ai float32x4_t vrecpeq_f32(float32x4_t __p0) {
  float32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  float32x4_t __ret;
  __ret = (float32x4_t) __builtin_neon_vrecpeq_v((int8x16_t)__rev0, 41);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vrecpe_u32(uint32x2_t __p0) {
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vrecpe_v((int8x8_t)__p0, 18);
  return __ret;
}
#else
__ai uint32x2_t vrecpe_u32(uint32x2_t __p0) {
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vrecpe_v((int8x8_t)__rev0, 18);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai float32x2_t vrecpe_f32(float32x2_t __p0) {
  float32x2_t __ret;
  __ret = (float32x2_t) __builtin_neon_vrecpe_v((int8x8_t)__p0, 9);
  return __ret;
}
#else
__ai float32x2_t vrecpe_f32(float32x2_t __p0) {
  float32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  float32x2_t __ret;
  __ret = (float32x2_t) __builtin_neon_vrecpe_v((int8x8_t)__rev0, 9);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai float32x4_t vrecpsq_f32(float32x4_t __p0, float32x4_t __p1) {
  float32x4_t __ret;
  __ret = (float32x4_t) __builtin_neon_vrecpsq_v((int8x16_t)__p0, (int8x16_t)__p1, 41);
  return __ret;
}
#else
__ai float32x4_t vrecpsq_f32(float32x4_t __p0, float32x4_t __p1) {
  float32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  float32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  float32x4_t __ret;
  __ret = (float32x4_t) __builtin_neon_vrecpsq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 41);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai float32x2_t vrecps_f32(float32x2_t __p0, float32x2_t __p1) {
  float32x2_t __ret;
  __ret = (float32x2_t) __builtin_neon_vrecps_v((int8x8_t)__p0, (int8x8_t)__p1, 9);
  return __ret;
}
#else
__ai float32x2_t vrecps_f32(float32x2_t __p0, float32x2_t __p1) {
  float32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  float32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  float32x2_t __ret;
  __ret = (float32x2_t) __builtin_neon_vrecps_v((int8x8_t)__rev0, (int8x8_t)__rev1, 9);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai poly8x8_t vrev16_p8(poly8x8_t __p0) {
  poly8x8_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 1, 0, 3, 2, 5, 4, 7, 6);
  return __ret;
}
#else
__ai poly8x8_t vrev16_p8(poly8x8_t __p0) {
  poly8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  poly8x8_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 1, 0, 3, 2, 5, 4, 7, 6);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai poly8x16_t vrev16q_p8(poly8x16_t __p0) {
  poly8x16_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 1, 0, 3, 2, 5, 4, 7, 6, 9, 8, 11, 10, 13, 12, 15, 14);
  return __ret;
}
#else
__ai poly8x16_t vrev16q_p8(poly8x16_t __p0) {
  poly8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  poly8x16_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 1, 0, 3, 2, 5, 4, 7, 6, 9, 8, 11, 10, 13, 12, 15, 14);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x16_t vrev16q_u8(uint8x16_t __p0) {
  uint8x16_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 1, 0, 3, 2, 5, 4, 7, 6, 9, 8, 11, 10, 13, 12, 15, 14);
  return __ret;
}
#else
__ai uint8x16_t vrev16q_u8(uint8x16_t __p0) {
  uint8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 1, 0, 3, 2, 5, 4, 7, 6, 9, 8, 11, 10, 13, 12, 15, 14);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x16_t vrev16q_s8(int8x16_t __p0) {
  int8x16_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 1, 0, 3, 2, 5, 4, 7, 6, 9, 8, 11, 10, 13, 12, 15, 14);
  return __ret;
}
#else
__ai int8x16_t vrev16q_s8(int8x16_t __p0) {
  int8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 1, 0, 3, 2, 5, 4, 7, 6, 9, 8, 11, 10, 13, 12, 15, 14);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x8_t vrev16_u8(uint8x8_t __p0) {
  uint8x8_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 1, 0, 3, 2, 5, 4, 7, 6);
  return __ret;
}
#else
__ai uint8x8_t vrev16_u8(uint8x8_t __p0) {
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 1, 0, 3, 2, 5, 4, 7, 6);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x8_t vrev16_s8(int8x8_t __p0) {
  int8x8_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 1, 0, 3, 2, 5, 4, 7, 6);
  return __ret;
}
#else
__ai int8x8_t vrev16_s8(int8x8_t __p0) {
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 1, 0, 3, 2, 5, 4, 7, 6);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai poly8x8_t vrev32_p8(poly8x8_t __p0) {
  poly8x8_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0, 7, 6, 5, 4);
  return __ret;
}
#else
__ai poly8x8_t vrev32_p8(poly8x8_t __p0) {
  poly8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  poly8x8_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 3, 2, 1, 0, 7, 6, 5, 4);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai poly16x4_t vrev32_p16(poly16x4_t __p0) {
  poly16x4_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 1, 0, 3, 2);
  return __ret;
}
#else
__ai poly16x4_t vrev32_p16(poly16x4_t __p0) {
  poly16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  poly16x4_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 1, 0, 3, 2);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai poly8x16_t vrev32q_p8(poly8x16_t __p0) {
  poly8x16_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0, 7, 6, 5, 4, 11, 10, 9, 8, 15, 14, 13, 12);
  return __ret;
}
#else
__ai poly8x16_t vrev32q_p8(poly8x16_t __p0) {
  poly8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  poly8x16_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 3, 2, 1, 0, 7, 6, 5, 4, 11, 10, 9, 8, 15, 14, 13, 12);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai poly16x8_t vrev32q_p16(poly16x8_t __p0) {
  poly16x8_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 1, 0, 3, 2, 5, 4, 7, 6);
  return __ret;
}
#else
__ai poly16x8_t vrev32q_p16(poly16x8_t __p0) {
  poly16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  poly16x8_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 1, 0, 3, 2, 5, 4, 7, 6);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x16_t vrev32q_u8(uint8x16_t __p0) {
  uint8x16_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0, 7, 6, 5, 4, 11, 10, 9, 8, 15, 14, 13, 12);
  return __ret;
}
#else
__ai uint8x16_t vrev32q_u8(uint8x16_t __p0) {
  uint8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 3, 2, 1, 0, 7, 6, 5, 4, 11, 10, 9, 8, 15, 14, 13, 12);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x8_t vrev32q_u16(uint16x8_t __p0) {
  uint16x8_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 1, 0, 3, 2, 5, 4, 7, 6);
  return __ret;
}
#else
__ai uint16x8_t vrev32q_u16(uint16x8_t __p0) {
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 1, 0, 3, 2, 5, 4, 7, 6);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x16_t vrev32q_s8(int8x16_t __p0) {
  int8x16_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0, 7, 6, 5, 4, 11, 10, 9, 8, 15, 14, 13, 12);
  return __ret;
}
#else
__ai int8x16_t vrev32q_s8(int8x16_t __p0) {
  int8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 3, 2, 1, 0, 7, 6, 5, 4, 11, 10, 9, 8, 15, 14, 13, 12);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x8_t vrev32q_s16(int16x8_t __p0) {
  int16x8_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 1, 0, 3, 2, 5, 4, 7, 6);
  return __ret;
}
#else
__ai int16x8_t vrev32q_s16(int16x8_t __p0) {
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 1, 0, 3, 2, 5, 4, 7, 6);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x8_t vrev32_u8(uint8x8_t __p0) {
  uint8x8_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0, 7, 6, 5, 4);
  return __ret;
}
#else
__ai uint8x8_t vrev32_u8(uint8x8_t __p0) {
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 3, 2, 1, 0, 7, 6, 5, 4);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x4_t vrev32_u16(uint16x4_t __p0) {
  uint16x4_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 1, 0, 3, 2);
  return __ret;
}
#else
__ai uint16x4_t vrev32_u16(uint16x4_t __p0) {
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint16x4_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 1, 0, 3, 2);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x8_t vrev32_s8(int8x8_t __p0) {
  int8x8_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0, 7, 6, 5, 4);
  return __ret;
}
#else
__ai int8x8_t vrev32_s8(int8x8_t __p0) {
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 3, 2, 1, 0, 7, 6, 5, 4);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x4_t vrev32_s16(int16x4_t __p0) {
  int16x4_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 1, 0, 3, 2);
  return __ret;
}
#else
__ai int16x4_t vrev32_s16(int16x4_t __p0) {
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 1, 0, 3, 2);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai poly8x8_t vrev64_p8(poly8x8_t __p0) {
  poly8x8_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#else
__ai poly8x8_t vrev64_p8(poly8x8_t __p0) {
  poly8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  poly8x8_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 7, 6, 5, 4, 3, 2, 1, 0);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai poly16x4_t vrev64_p16(poly16x4_t __p0) {
  poly16x4_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  return __ret;
}
#else
__ai poly16x4_t vrev64_p16(poly16x4_t __p0) {
  poly16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  poly16x4_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 3, 2, 1, 0);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai poly8x16_t vrev64q_p8(poly8x16_t __p0) {
  poly8x16_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0, 15, 14, 13, 12, 11, 10, 9, 8);
  return __ret;
}
#else
__ai poly8x16_t vrev64q_p8(poly8x16_t __p0) {
  poly8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  poly8x16_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 7, 6, 5, 4, 3, 2, 1, 0, 15, 14, 13, 12, 11, 10, 9, 8);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai poly16x8_t vrev64q_p16(poly16x8_t __p0) {
  poly16x8_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0, 7, 6, 5, 4);
  return __ret;
}
#else
__ai poly16x8_t vrev64q_p16(poly16x8_t __p0) {
  poly16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  poly16x8_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 3, 2, 1, 0, 7, 6, 5, 4);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x16_t vrev64q_u8(uint8x16_t __p0) {
  uint8x16_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0, 15, 14, 13, 12, 11, 10, 9, 8);
  return __ret;
}
#else
__ai uint8x16_t vrev64q_u8(uint8x16_t __p0) {
  uint8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 7, 6, 5, 4, 3, 2, 1, 0, 15, 14, 13, 12, 11, 10, 9, 8);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vrev64q_u32(uint32x4_t __p0) {
  uint32x4_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 1, 0, 3, 2);
  return __ret;
}
#else
__ai uint32x4_t vrev64q_u32(uint32x4_t __p0) {
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 1, 0, 3, 2);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x8_t vrev64q_u16(uint16x8_t __p0) {
  uint16x8_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0, 7, 6, 5, 4);
  return __ret;
}
#else
__ai uint16x8_t vrev64q_u16(uint16x8_t __p0) {
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 3, 2, 1, 0, 7, 6, 5, 4);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x16_t vrev64q_s8(int8x16_t __p0) {
  int8x16_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0, 15, 14, 13, 12, 11, 10, 9, 8);
  return __ret;
}
#else
__ai int8x16_t vrev64q_s8(int8x16_t __p0) {
  int8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 7, 6, 5, 4, 3, 2, 1, 0, 15, 14, 13, 12, 11, 10, 9, 8);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai float32x4_t vrev64q_f32(float32x4_t __p0) {
  float32x4_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 1, 0, 3, 2);
  return __ret;
}
#else
__ai float32x4_t vrev64q_f32(float32x4_t __p0) {
  float32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  float32x4_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 1, 0, 3, 2);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x4_t vrev64q_s32(int32x4_t __p0) {
  int32x4_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 1, 0, 3, 2);
  return __ret;
}
#else
__ai int32x4_t vrev64q_s32(int32x4_t __p0) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x4_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 1, 0, 3, 2);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x8_t vrev64q_s16(int16x8_t __p0) {
  int16x8_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0, 7, 6, 5, 4);
  return __ret;
}
#else
__ai int16x8_t vrev64q_s16(int16x8_t __p0) {
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 3, 2, 1, 0, 7, 6, 5, 4);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x8_t vrev64_u8(uint8x8_t __p0) {
  uint8x8_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#else
__ai uint8x8_t vrev64_u8(uint8x8_t __p0) {
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 7, 6, 5, 4, 3, 2, 1, 0);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vrev64_u32(uint32x2_t __p0) {
  uint32x2_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 1, 0);
  return __ret;
}
#else
__ai uint32x2_t vrev64_u32(uint32x2_t __p0) {
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint32x2_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 1, 0);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x4_t vrev64_u16(uint16x4_t __p0) {
  uint16x4_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  return __ret;
}
#else
__ai uint16x4_t vrev64_u16(uint16x4_t __p0) {
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint16x4_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 3, 2, 1, 0);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x8_t vrev64_s8(int8x8_t __p0) {
  int8x8_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#else
__ai int8x8_t vrev64_s8(int8x8_t __p0) {
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 7, 6, 5, 4, 3, 2, 1, 0);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai float32x2_t vrev64_f32(float32x2_t __p0) {
  float32x2_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 1, 0);
  return __ret;
}
#else
__ai float32x2_t vrev64_f32(float32x2_t __p0) {
  float32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  float32x2_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 1, 0);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x2_t vrev64_s32(int32x2_t __p0) {
  int32x2_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 1, 0);
  return __ret;
}
#else
__ai int32x2_t vrev64_s32(int32x2_t __p0) {
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 1, 0);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x4_t vrev64_s16(int16x4_t __p0) {
  int16x4_t __ret;
  __ret = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  return __ret;
}
#else
__ai int16x4_t vrev64_s16(int16x4_t __p0) {
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __ret;
  __ret = __builtin_shufflevector(__rev0, __rev0, 3, 2, 1, 0);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x16_t vrhaddq_u8(uint8x16_t __p0, uint8x16_t __p1) {
  uint8x16_t __ret;
  __ret = (uint8x16_t) __builtin_neon_vrhaddq_v((int8x16_t)__p0, (int8x16_t)__p1, 48);
  return __ret;
}
#else
__ai uint8x16_t vrhaddq_u8(uint8x16_t __p0, uint8x16_t __p1) {
  uint8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __ret;
  __ret = (uint8x16_t) __builtin_neon_vrhaddq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 48);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vrhaddq_u32(uint32x4_t __p0, uint32x4_t __p1) {
  uint32x4_t __ret;
  __ret = (uint32x4_t) __builtin_neon_vrhaddq_v((int8x16_t)__p0, (int8x16_t)__p1, 50);
  return __ret;
}
#else
__ai uint32x4_t vrhaddq_u32(uint32x4_t __p0, uint32x4_t __p1) {
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = (uint32x4_t) __builtin_neon_vrhaddq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 50);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x8_t vrhaddq_u16(uint16x8_t __p0, uint16x8_t __p1) {
  uint16x8_t __ret;
  __ret = (uint16x8_t) __builtin_neon_vrhaddq_v((int8x16_t)__p0, (int8x16_t)__p1, 49);
  return __ret;
}
#else
__ai uint16x8_t vrhaddq_u16(uint16x8_t __p0, uint16x8_t __p1) {
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __ret;
  __ret = (uint16x8_t) __builtin_neon_vrhaddq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 49);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x16_t vrhaddq_s8(int8x16_t __p0, int8x16_t __p1) {
  int8x16_t __ret;
  __ret = (int8x16_t) __builtin_neon_vrhaddq_v((int8x16_t)__p0, (int8x16_t)__p1, 32);
  return __ret;
}
#else
__ai int8x16_t vrhaddq_s8(int8x16_t __p0, int8x16_t __p1) {
  int8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __ret;
  __ret = (int8x16_t) __builtin_neon_vrhaddq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 32);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x4_t vrhaddq_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vrhaddq_v((int8x16_t)__p0, (int8x16_t)__p1, 34);
  return __ret;
}
#else
__ai int32x4_t vrhaddq_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vrhaddq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 34);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x8_t vrhaddq_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vrhaddq_v((int8x16_t)__p0, (int8x16_t)__p1, 33);
  return __ret;
}
#else
__ai int16x8_t vrhaddq_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vrhaddq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 33);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x8_t vrhadd_u8(uint8x8_t __p0, uint8x8_t __p1) {
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vrhadd_v((int8x8_t)__p0, (int8x8_t)__p1, 16);
  return __ret;
}
#else
__ai uint8x8_t vrhadd_u8(uint8x8_t __p0, uint8x8_t __p1) {
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vrhadd_v((int8x8_t)__rev0, (int8x8_t)__rev1, 16);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vrhadd_u32(uint32x2_t __p0, uint32x2_t __p1) {
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vrhadd_v((int8x8_t)__p0, (int8x8_t)__p1, 18);
  return __ret;
}
#else
__ai uint32x2_t vrhadd_u32(uint32x2_t __p0, uint32x2_t __p1) {
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vrhadd_v((int8x8_t)__rev0, (int8x8_t)__rev1, 18);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x4_t vrhadd_u16(uint16x4_t __p0, uint16x4_t __p1) {
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vrhadd_v((int8x8_t)__p0, (int8x8_t)__p1, 17);
  return __ret;
}
#else
__ai uint16x4_t vrhadd_u16(uint16x4_t __p0, uint16x4_t __p1) {
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vrhadd_v((int8x8_t)__rev0, (int8x8_t)__rev1, 17);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x8_t vrhadd_s8(int8x8_t __p0, int8x8_t __p1) {
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vrhadd_v((int8x8_t)__p0, (int8x8_t)__p1, 0);
  return __ret;
}
#else
__ai int8x8_t vrhadd_s8(int8x8_t __p0, int8x8_t __p1) {
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vrhadd_v((int8x8_t)__rev0, (int8x8_t)__rev1, 0);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x2_t vrhadd_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vrhadd_v((int8x8_t)__p0, (int8x8_t)__p1, 2);
  return __ret;
}
#else
__ai int32x2_t vrhadd_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vrhadd_v((int8x8_t)__rev0, (int8x8_t)__rev1, 2);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x4_t vrhadd_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vrhadd_v((int8x8_t)__p0, (int8x8_t)__p1, 1);
  return __ret;
}
#else
__ai int16x4_t vrhadd_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vrhadd_v((int8x8_t)__rev0, (int8x8_t)__rev1, 1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x16_t vrshlq_u8(uint8x16_t __p0, int8x16_t __p1) {
  uint8x16_t __ret;
  __ret = (uint8x16_t) __builtin_neon_vrshlq_v((int8x16_t)__p0, (int8x16_t)__p1, 48);
  return __ret;
}
#else
__ai uint8x16_t vrshlq_u8(uint8x16_t __p0, int8x16_t __p1) {
  uint8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __ret;
  __ret = (uint8x16_t) __builtin_neon_vrshlq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 48);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vrshlq_u32(uint32x4_t __p0, int32x4_t __p1) {
  uint32x4_t __ret;
  __ret = (uint32x4_t) __builtin_neon_vrshlq_v((int8x16_t)__p0, (int8x16_t)__p1, 50);
  return __ret;
}
#else
__ai uint32x4_t vrshlq_u32(uint32x4_t __p0, int32x4_t __p1) {
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = (uint32x4_t) __builtin_neon_vrshlq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 50);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint64x2_t vrshlq_u64(uint64x2_t __p0, int64x2_t __p1) {
  uint64x2_t __ret;
  __ret = (uint64x2_t) __builtin_neon_vrshlq_v((int8x16_t)__p0, (int8x16_t)__p1, 51);
  return __ret;
}
#else
__ai uint64x2_t vrshlq_u64(uint64x2_t __p0, int64x2_t __p1) {
  uint64x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int64x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint64x2_t __ret;
  __ret = (uint64x2_t) __builtin_neon_vrshlq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 51);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x8_t vrshlq_u16(uint16x8_t __p0, int16x8_t __p1) {
  uint16x8_t __ret;
  __ret = (uint16x8_t) __builtin_neon_vrshlq_v((int8x16_t)__p0, (int8x16_t)__p1, 49);
  return __ret;
}
#else
__ai uint16x8_t vrshlq_u16(uint16x8_t __p0, int16x8_t __p1) {
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __ret;
  __ret = (uint16x8_t) __builtin_neon_vrshlq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 49);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x16_t vrshlq_s8(int8x16_t __p0, int8x16_t __p1) {
  int8x16_t __ret;
  __ret = (int8x16_t) __builtin_neon_vrshlq_v((int8x16_t)__p0, (int8x16_t)__p1, 32);
  return __ret;
}
#else
__ai int8x16_t vrshlq_s8(int8x16_t __p0, int8x16_t __p1) {
  int8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __ret;
  __ret = (int8x16_t) __builtin_neon_vrshlq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 32);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x4_t vrshlq_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vrshlq_v((int8x16_t)__p0, (int8x16_t)__p1, 34);
  return __ret;
}
#else
__ai int32x4_t vrshlq_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vrshlq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 34);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int64x2_t vrshlq_s64(int64x2_t __p0, int64x2_t __p1) {
  int64x2_t __ret;
  __ret = (int64x2_t) __builtin_neon_vrshlq_v((int8x16_t)__p0, (int8x16_t)__p1, 35);
  return __ret;
}
#else
__ai int64x2_t vrshlq_s64(int64x2_t __p0, int64x2_t __p1) {
  int64x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int64x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  int64x2_t __ret;
  __ret = (int64x2_t) __builtin_neon_vrshlq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 35);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x8_t vrshlq_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vrshlq_v((int8x16_t)__p0, (int8x16_t)__p1, 33);
  return __ret;
}
#else
__ai int16x8_t vrshlq_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vrshlq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 33);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x8_t vrshl_u8(uint8x8_t __p0, int8x8_t __p1) {
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vrshl_v((int8x8_t)__p0, (int8x8_t)__p1, 16);
  return __ret;
}
#else
__ai uint8x8_t vrshl_u8(uint8x8_t __p0, int8x8_t __p1) {
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vrshl_v((int8x8_t)__rev0, (int8x8_t)__rev1, 16);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vrshl_u32(uint32x2_t __p0, int32x2_t __p1) {
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vrshl_v((int8x8_t)__p0, (int8x8_t)__p1, 18);
  return __ret;
}
#else
__ai uint32x2_t vrshl_u32(uint32x2_t __p0, int32x2_t __p1) {
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vrshl_v((int8x8_t)__rev0, (int8x8_t)__rev1, 18);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

__ai uint64x1_t vrshl_u64(uint64x1_t __p0, int64x1_t __p1) {
  uint64x1_t __ret;
  __ret = (uint64x1_t) __builtin_neon_vrshl_v((int8x8_t)__p0, (int8x8_t)__p1, 19);
  return __ret;
}
#ifdef __LITTLE_ENDIAN__
__ai uint16x4_t vrshl_u16(uint16x4_t __p0, int16x4_t __p1) {
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vrshl_v((int8x8_t)__p0, (int8x8_t)__p1, 17);
  return __ret;
}
#else
__ai uint16x4_t vrshl_u16(uint16x4_t __p0, int16x4_t __p1) {
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vrshl_v((int8x8_t)__rev0, (int8x8_t)__rev1, 17);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x8_t vrshl_s8(int8x8_t __p0, int8x8_t __p1) {
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vrshl_v((int8x8_t)__p0, (int8x8_t)__p1, 0);
  return __ret;
}
#else
__ai int8x8_t vrshl_s8(int8x8_t __p0, int8x8_t __p1) {
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vrshl_v((int8x8_t)__rev0, (int8x8_t)__rev1, 0);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x2_t vrshl_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vrshl_v((int8x8_t)__p0, (int8x8_t)__p1, 2);
  return __ret;
}
#else
__ai int32x2_t vrshl_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vrshl_v((int8x8_t)__rev0, (int8x8_t)__rev1, 2);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

__ai int64x1_t vrshl_s64(int64x1_t __p0, int64x1_t __p1) {
  int64x1_t __ret;
  __ret = (int64x1_t) __builtin_neon_vrshl_v((int8x8_t)__p0, (int8x8_t)__p1, 3);
  return __ret;
}
#ifdef __LITTLE_ENDIAN__
__ai int16x4_t vrshl_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vrshl_v((int8x8_t)__p0, (int8x8_t)__p1, 1);
  return __ret;
}
#else
__ai int16x4_t vrshl_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vrshl_v((int8x8_t)__rev0, (int8x8_t)__rev1, 1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
#define vrshrq_n_u8(__p0, __p1) __extension__ ({ \
  uint8x16_t __s0 = __p0; \
  uint8x16_t __ret; \
  __ret = (uint8x16_t) __builtin_neon_vrshrq_n_v((int8x16_t)__s0, __p1, 48); \
  __ret; \
})
#else
#define vrshrq_n_u8(__p0, __p1) __extension__ ({ \
  uint8x16_t __s0 = __p0; \
  uint8x16_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint8x16_t __ret; \
  __ret = (uint8x16_t) __builtin_neon_vrshrq_n_v((int8x16_t)__rev0, __p1, 48); \
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vrshrq_n_u32(__p0, __p1) __extension__ ({ \
  uint32x4_t __s0 = __p0; \
  uint32x4_t __ret; \
  __ret = (uint32x4_t) __builtin_neon_vrshrq_n_v((int8x16_t)__s0, __p1, 50); \
  __ret; \
})
#else
#define vrshrq_n_u32(__p0, __p1) __extension__ ({ \
  uint32x4_t __s0 = __p0; \
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  uint32x4_t __ret; \
  __ret = (uint32x4_t) __builtin_neon_vrshrq_n_v((int8x16_t)__rev0, __p1, 50); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vrshrq_n_u64(__p0, __p1) __extension__ ({ \
  uint64x2_t __s0 = __p0; \
  uint64x2_t __ret; \
  __ret = (uint64x2_t) __builtin_neon_vrshrq_n_v((int8x16_t)__s0, __p1, 51); \
  __ret; \
})
#else
#define vrshrq_n_u64(__p0, __p1) __extension__ ({ \
  uint64x2_t __s0 = __p0; \
  uint64x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  uint64x2_t __ret; \
  __ret = (uint64x2_t) __builtin_neon_vrshrq_n_v((int8x16_t)__rev0, __p1, 51); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vrshrq_n_u16(__p0, __p1) __extension__ ({ \
  uint16x8_t __s0 = __p0; \
  uint16x8_t __ret; \
  __ret = (uint16x8_t) __builtin_neon_vrshrq_n_v((int8x16_t)__s0, __p1, 49); \
  __ret; \
})
#else
#define vrshrq_n_u16(__p0, __p1) __extension__ ({ \
  uint16x8_t __s0 = __p0; \
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint16x8_t __ret; \
  __ret = (uint16x8_t) __builtin_neon_vrshrq_n_v((int8x16_t)__rev0, __p1, 49); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vrshrq_n_s8(__p0, __p1) __extension__ ({ \
  int8x16_t __s0 = __p0; \
  int8x16_t __ret; \
  __ret = (int8x16_t) __builtin_neon_vrshrq_n_v((int8x16_t)__s0, __p1, 32); \
  __ret; \
})
#else
#define vrshrq_n_s8(__p0, __p1) __extension__ ({ \
  int8x16_t __s0 = __p0; \
  int8x16_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  int8x16_t __ret; \
  __ret = (int8x16_t) __builtin_neon_vrshrq_n_v((int8x16_t)__rev0, __p1, 32); \
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vrshrq_n_s32(__p0, __p1) __extension__ ({ \
  int32x4_t __s0 = __p0; \
  int32x4_t __ret; \
  __ret = (int32x4_t) __builtin_neon_vrshrq_n_v((int8x16_t)__s0, __p1, 34); \
  __ret; \
})
#else
#define vrshrq_n_s32(__p0, __p1) __extension__ ({ \
  int32x4_t __s0 = __p0; \
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  int32x4_t __ret; \
  __ret = (int32x4_t) __builtin_neon_vrshrq_n_v((int8x16_t)__rev0, __p1, 34); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vrshrq_n_s64(__p0, __p1) __extension__ ({ \
  int64x2_t __s0 = __p0; \
  int64x2_t __ret; \
  __ret = (int64x2_t) __builtin_neon_vrshrq_n_v((int8x16_t)__s0, __p1, 35); \
  __ret; \
})
#else
#define vrshrq_n_s64(__p0, __p1) __extension__ ({ \
  int64x2_t __s0 = __p0; \
  int64x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  int64x2_t __ret; \
  __ret = (int64x2_t) __builtin_neon_vrshrq_n_v((int8x16_t)__rev0, __p1, 35); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vrshrq_n_s16(__p0, __p1) __extension__ ({ \
  int16x8_t __s0 = __p0; \
  int16x8_t __ret; \
  __ret = (int16x8_t) __builtin_neon_vrshrq_n_v((int8x16_t)__s0, __p1, 33); \
  __ret; \
})
#else
#define vrshrq_n_s16(__p0, __p1) __extension__ ({ \
  int16x8_t __s0 = __p0; \
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  int16x8_t __ret; \
  __ret = (int16x8_t) __builtin_neon_vrshrq_n_v((int8x16_t)__rev0, __p1, 33); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vrshr_n_u8(__p0, __p1) __extension__ ({ \
  uint8x8_t __s0 = __p0; \
  uint8x8_t __ret; \
  __ret = (uint8x8_t) __builtin_neon_vrshr_n_v((int8x8_t)__s0, __p1, 16); \
  __ret; \
})
#else
#define vrshr_n_u8(__p0, __p1) __extension__ ({ \
  uint8x8_t __s0 = __p0; \
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint8x8_t __ret; \
  __ret = (uint8x8_t) __builtin_neon_vrshr_n_v((int8x8_t)__rev0, __p1, 16); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vrshr_n_u32(__p0, __p1) __extension__ ({ \
  uint32x2_t __s0 = __p0; \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_vrshr_n_v((int8x8_t)__s0, __p1, 18); \
  __ret; \
})
#else
#define vrshr_n_u32(__p0, __p1) __extension__ ({ \
  uint32x2_t __s0 = __p0; \
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_vrshr_n_v((int8x8_t)__rev0, __p1, 18); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#define vrshr_n_u64(__p0, __p1) __extension__ ({ \
  uint64x1_t __s0 = __p0; \
  uint64x1_t __ret; \
  __ret = (uint64x1_t) __builtin_neon_vrshr_n_v((int8x8_t)__s0, __p1, 19); \
  __ret; \
})
#ifdef __LITTLE_ENDIAN__
#define vrshr_n_u16(__p0, __p1) __extension__ ({ \
  uint16x4_t __s0 = __p0; \
  uint16x4_t __ret; \
  __ret = (uint16x4_t) __builtin_neon_vrshr_n_v((int8x8_t)__s0, __p1, 17); \
  __ret; \
})
#else
#define vrshr_n_u16(__p0, __p1) __extension__ ({ \
  uint16x4_t __s0 = __p0; \
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  uint16x4_t __ret; \
  __ret = (uint16x4_t) __builtin_neon_vrshr_n_v((int8x8_t)__rev0, __p1, 17); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vrshr_n_s8(__p0, __p1) __extension__ ({ \
  int8x8_t __s0 = __p0; \
  int8x8_t __ret; \
  __ret = (int8x8_t) __builtin_neon_vrshr_n_v((int8x8_t)__s0, __p1, 0); \
  __ret; \
})
#else
#define vrshr_n_s8(__p0, __p1) __extension__ ({ \
  int8x8_t __s0 = __p0; \
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  int8x8_t __ret; \
  __ret = (int8x8_t) __builtin_neon_vrshr_n_v((int8x8_t)__rev0, __p1, 0); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vrshr_n_s32(__p0, __p1) __extension__ ({ \
  int32x2_t __s0 = __p0; \
  int32x2_t __ret; \
  __ret = (int32x2_t) __builtin_neon_vrshr_n_v((int8x8_t)__s0, __p1, 2); \
  __ret; \
})
#else
#define vrshr_n_s32(__p0, __p1) __extension__ ({ \
  int32x2_t __s0 = __p0; \
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  int32x2_t __ret; \
  __ret = (int32x2_t) __builtin_neon_vrshr_n_v((int8x8_t)__rev0, __p1, 2); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#define vrshr_n_s64(__p0, __p1) __extension__ ({ \
  int64x1_t __s0 = __p0; \
  int64x1_t __ret; \
  __ret = (int64x1_t) __builtin_neon_vrshr_n_v((int8x8_t)__s0, __p1, 3); \
  __ret; \
})
#ifdef __LITTLE_ENDIAN__
#define vrshr_n_s16(__p0, __p1) __extension__ ({ \
  int16x4_t __s0 = __p0; \
  int16x4_t __ret; \
  __ret = (int16x4_t) __builtin_neon_vrshr_n_v((int8x8_t)__s0, __p1, 1); \
  __ret; \
})
#else
#define vrshr_n_s16(__p0, __p1) __extension__ ({ \
  int16x4_t __s0 = __p0; \
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  int16x4_t __ret; \
  __ret = (int16x4_t) __builtin_neon_vrshr_n_v((int8x8_t)__rev0, __p1, 1); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vrshrn_n_u32(__p0, __p1) __extension__ ({ \
  uint32x4_t __s0 = __p0; \
  uint16x4_t __ret; \
  __ret = (uint16x4_t) __builtin_neon_vrshrn_n_v((int8x16_t)__s0, __p1, 17); \
  __ret; \
})
#else
#define vrshrn_n_u32(__p0, __p1) __extension__ ({ \
  uint32x4_t __s0 = __p0; \
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  uint16x4_t __ret; \
  __ret = (uint16x4_t) __builtin_neon_vrshrn_n_v((int8x16_t)__rev0, __p1, 17); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_vrshrn_n_u32(__p0, __p1) __extension__ ({ \
  uint32x4_t __s0 = __p0; \
  uint16x4_t __ret; \
  __ret = (uint16x4_t) __builtin_neon_vrshrn_n_v((int8x16_t)__s0, __p1, 17); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vrshrn_n_u64(__p0, __p1) __extension__ ({ \
  uint64x2_t __s0 = __p0; \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_vrshrn_n_v((int8x16_t)__s0, __p1, 18); \
  __ret; \
})
#else
#define vrshrn_n_u64(__p0, __p1) __extension__ ({ \
  uint64x2_t __s0 = __p0; \
  uint64x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_vrshrn_n_v((int8x16_t)__rev0, __p1, 18); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#define __noswap_vrshrn_n_u64(__p0, __p1) __extension__ ({ \
  uint64x2_t __s0 = __p0; \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_vrshrn_n_v((int8x16_t)__s0, __p1, 18); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vrshrn_n_u16(__p0, __p1) __extension__ ({ \
  uint16x8_t __s0 = __p0; \
  uint8x8_t __ret; \
  __ret = (uint8x8_t) __builtin_neon_vrshrn_n_v((int8x16_t)__s0, __p1, 16); \
  __ret; \
})
#else
#define vrshrn_n_u16(__p0, __p1) __extension__ ({ \
  uint16x8_t __s0 = __p0; \
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint8x8_t __ret; \
  __ret = (uint8x8_t) __builtin_neon_vrshrn_n_v((int8x16_t)__rev0, __p1, 16); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_vrshrn_n_u16(__p0, __p1) __extension__ ({ \
  uint16x8_t __s0 = __p0; \
  uint8x8_t __ret; \
  __ret = (uint8x8_t) __builtin_neon_vrshrn_n_v((int8x16_t)__s0, __p1, 16); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vrshrn_n_s32(__p0, __p1) __extension__ ({ \
  int32x4_t __s0 = __p0; \
  int16x4_t __ret; \
  __ret = (int16x4_t) __builtin_neon_vrshrn_n_v((int8x16_t)__s0, __p1, 1); \
  __ret; \
})
#else
#define vrshrn_n_s32(__p0, __p1) __extension__ ({ \
  int32x4_t __s0 = __p0; \
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  int16x4_t __ret; \
  __ret = (int16x4_t) __builtin_neon_vrshrn_n_v((int8x16_t)__rev0, __p1, 1); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_vrshrn_n_s32(__p0, __p1) __extension__ ({ \
  int32x4_t __s0 = __p0; \
  int16x4_t __ret; \
  __ret = (int16x4_t) __builtin_neon_vrshrn_n_v((int8x16_t)__s0, __p1, 1); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vrshrn_n_s64(__p0, __p1) __extension__ ({ \
  int64x2_t __s0 = __p0; \
  int32x2_t __ret; \
  __ret = (int32x2_t) __builtin_neon_vrshrn_n_v((int8x16_t)__s0, __p1, 2); \
  __ret; \
})
#else
#define vrshrn_n_s64(__p0, __p1) __extension__ ({ \
  int64x2_t __s0 = __p0; \
  int64x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  int32x2_t __ret; \
  __ret = (int32x2_t) __builtin_neon_vrshrn_n_v((int8x16_t)__rev0, __p1, 2); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#define __noswap_vrshrn_n_s64(__p0, __p1) __extension__ ({ \
  int64x2_t __s0 = __p0; \
  int32x2_t __ret; \
  __ret = (int32x2_t) __builtin_neon_vrshrn_n_v((int8x16_t)__s0, __p1, 2); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vrshrn_n_s16(__p0, __p1) __extension__ ({ \
  int16x8_t __s0 = __p0; \
  int8x8_t __ret; \
  __ret = (int8x8_t) __builtin_neon_vrshrn_n_v((int8x16_t)__s0, __p1, 0); \
  __ret; \
})
#else
#define vrshrn_n_s16(__p0, __p1) __extension__ ({ \
  int16x8_t __s0 = __p0; \
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  int8x8_t __ret; \
  __ret = (int8x8_t) __builtin_neon_vrshrn_n_v((int8x16_t)__rev0, __p1, 0); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_vrshrn_n_s16(__p0, __p1) __extension__ ({ \
  int16x8_t __s0 = __p0; \
  int8x8_t __ret; \
  __ret = (int8x8_t) __builtin_neon_vrshrn_n_v((int8x16_t)__s0, __p1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vrsqrteq_u32(uint32x4_t __p0) {
  uint32x4_t __ret;
  __ret = (uint32x4_t) __builtin_neon_vrsqrteq_v((int8x16_t)__p0, 50);
  return __ret;
}
#else
__ai uint32x4_t vrsqrteq_u32(uint32x4_t __p0) {
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = (uint32x4_t) __builtin_neon_vrsqrteq_v((int8x16_t)__rev0, 50);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai float32x4_t vrsqrteq_f32(float32x4_t __p0) {
  float32x4_t __ret;
  __ret = (float32x4_t) __builtin_neon_vrsqrteq_v((int8x16_t)__p0, 41);
  return __ret;
}
#else
__ai float32x4_t vrsqrteq_f32(float32x4_t __p0) {
  float32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  float32x4_t __ret;
  __ret = (float32x4_t) __builtin_neon_vrsqrteq_v((int8x16_t)__rev0, 41);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vrsqrte_u32(uint32x2_t __p0) {
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vrsqrte_v((int8x8_t)__p0, 18);
  return __ret;
}
#else
__ai uint32x2_t vrsqrte_u32(uint32x2_t __p0) {
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vrsqrte_v((int8x8_t)__rev0, 18);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai float32x2_t vrsqrte_f32(float32x2_t __p0) {
  float32x2_t __ret;
  __ret = (float32x2_t) __builtin_neon_vrsqrte_v((int8x8_t)__p0, 9);
  return __ret;
}
#else
__ai float32x2_t vrsqrte_f32(float32x2_t __p0) {
  float32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  float32x2_t __ret;
  __ret = (float32x2_t) __builtin_neon_vrsqrte_v((int8x8_t)__rev0, 9);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai float32x4_t vrsqrtsq_f32(float32x4_t __p0, float32x4_t __p1) {
  float32x4_t __ret;
  __ret = (float32x4_t) __builtin_neon_vrsqrtsq_v((int8x16_t)__p0, (int8x16_t)__p1, 41);
  return __ret;
}
#else
__ai float32x4_t vrsqrtsq_f32(float32x4_t __p0, float32x4_t __p1) {
  float32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  float32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  float32x4_t __ret;
  __ret = (float32x4_t) __builtin_neon_vrsqrtsq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 41);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai float32x2_t vrsqrts_f32(float32x2_t __p0, float32x2_t __p1) {
  float32x2_t __ret;
  __ret = (float32x2_t) __builtin_neon_vrsqrts_v((int8x8_t)__p0, (int8x8_t)__p1, 9);
  return __ret;
}
#else
__ai float32x2_t vrsqrts_f32(float32x2_t __p0, float32x2_t __p1) {
  float32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  float32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  float32x2_t __ret;
  __ret = (float32x2_t) __builtin_neon_vrsqrts_v((int8x8_t)__rev0, (int8x8_t)__rev1, 9);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
#define vrsraq_n_u8(__p0, __p1, __p2) __extension__ ({ \
  uint8x16_t __s0 = __p0; \
  uint8x16_t __s1 = __p1; \
  uint8x16_t __ret; \
  __ret = (uint8x16_t) __builtin_neon_vrsraq_n_v((int8x16_t)__s0, (int8x16_t)__s1, __p2, 48); \
  __ret; \
})
#else
#define vrsraq_n_u8(__p0, __p1, __p2) __extension__ ({ \
  uint8x16_t __s0 = __p0; \
  uint8x16_t __s1 = __p1; \
  uint8x16_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint8x16_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint8x16_t __ret; \
  __ret = (uint8x16_t) __builtin_neon_vrsraq_n_v((int8x16_t)__rev0, (int8x16_t)__rev1, __p2, 48); \
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vrsraq_n_u32(__p0, __p1, __p2) __extension__ ({ \
  uint32x4_t __s0 = __p0; \
  uint32x4_t __s1 = __p1; \
  uint32x4_t __ret; \
  __ret = (uint32x4_t) __builtin_neon_vrsraq_n_v((int8x16_t)__s0, (int8x16_t)__s1, __p2, 50); \
  __ret; \
})
#else
#define vrsraq_n_u32(__p0, __p1, __p2) __extension__ ({ \
  uint32x4_t __s0 = __p0; \
  uint32x4_t __s1 = __p1; \
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  uint32x4_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 3, 2, 1, 0); \
  uint32x4_t __ret; \
  __ret = (uint32x4_t) __builtin_neon_vrsraq_n_v((int8x16_t)__rev0, (int8x16_t)__rev1, __p2, 50); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vrsraq_n_u64(__p0, __p1, __p2) __extension__ ({ \
  uint64x2_t __s0 = __p0; \
  uint64x2_t __s1 = __p1; \
  uint64x2_t __ret; \
  __ret = (uint64x2_t) __builtin_neon_vrsraq_n_v((int8x16_t)__s0, (int8x16_t)__s1, __p2, 51); \
  __ret; \
})
#else
#define vrsraq_n_u64(__p0, __p1, __p2) __extension__ ({ \
  uint64x2_t __s0 = __p0; \
  uint64x2_t __s1 = __p1; \
  uint64x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  uint64x2_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 1, 0); \
  uint64x2_t __ret; \
  __ret = (uint64x2_t) __builtin_neon_vrsraq_n_v((int8x16_t)__rev0, (int8x16_t)__rev1, __p2, 51); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vrsraq_n_u16(__p0, __p1, __p2) __extension__ ({ \
  uint16x8_t __s0 = __p0; \
  uint16x8_t __s1 = __p1; \
  uint16x8_t __ret; \
  __ret = (uint16x8_t) __builtin_neon_vrsraq_n_v((int8x16_t)__s0, (int8x16_t)__s1, __p2, 49); \
  __ret; \
})
#else
#define vrsraq_n_u16(__p0, __p1, __p2) __extension__ ({ \
  uint16x8_t __s0 = __p0; \
  uint16x8_t __s1 = __p1; \
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint16x8_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint16x8_t __ret; \
  __ret = (uint16x8_t) __builtin_neon_vrsraq_n_v((int8x16_t)__rev0, (int8x16_t)__rev1, __p2, 49); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vrsraq_n_s8(__p0, __p1, __p2) __extension__ ({ \
  int8x16_t __s0 = __p0; \
  int8x16_t __s1 = __p1; \
  int8x16_t __ret; \
  __ret = (int8x16_t) __builtin_neon_vrsraq_n_v((int8x16_t)__s0, (int8x16_t)__s1, __p2, 32); \
  __ret; \
})
#else
#define vrsraq_n_s8(__p0, __p1, __p2) __extension__ ({ \
  int8x16_t __s0 = __p0; \
  int8x16_t __s1 = __p1; \
  int8x16_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  int8x16_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  int8x16_t __ret; \
  __ret = (int8x16_t) __builtin_neon_vrsraq_n_v((int8x16_t)__rev0, (int8x16_t)__rev1, __p2, 32); \
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vrsraq_n_s32(__p0, __p1, __p2) __extension__ ({ \
  int32x4_t __s0 = __p0; \
  int32x4_t __s1 = __p1; \
  int32x4_t __ret; \
  __ret = (int32x4_t) __builtin_neon_vrsraq_n_v((int8x16_t)__s0, (int8x16_t)__s1, __p2, 34); \
  __ret; \
})
#else
#define vrsraq_n_s32(__p0, __p1, __p2) __extension__ ({ \
  int32x4_t __s0 = __p0; \
  int32x4_t __s1 = __p1; \
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  int32x4_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 3, 2, 1, 0); \
  int32x4_t __ret; \
  __ret = (int32x4_t) __builtin_neon_vrsraq_n_v((int8x16_t)__rev0, (int8x16_t)__rev1, __p2, 34); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vrsraq_n_s64(__p0, __p1, __p2) __extension__ ({ \
  int64x2_t __s0 = __p0; \
  int64x2_t __s1 = __p1; \
  int64x2_t __ret; \
  __ret = (int64x2_t) __builtin_neon_vrsraq_n_v((int8x16_t)__s0, (int8x16_t)__s1, __p2, 35); \
  __ret; \
})
#else
#define vrsraq_n_s64(__p0, __p1, __p2) __extension__ ({ \
  int64x2_t __s0 = __p0; \
  int64x2_t __s1 = __p1; \
  int64x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  int64x2_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 1, 0); \
  int64x2_t __ret; \
  __ret = (int64x2_t) __builtin_neon_vrsraq_n_v((int8x16_t)__rev0, (int8x16_t)__rev1, __p2, 35); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vrsraq_n_s16(__p0, __p1, __p2) __extension__ ({ \
  int16x8_t __s0 = __p0; \
  int16x8_t __s1 = __p1; \
  int16x8_t __ret; \
  __ret = (int16x8_t) __builtin_neon_vrsraq_n_v((int8x16_t)__s0, (int8x16_t)__s1, __p2, 33); \
  __ret; \
})
#else
#define vrsraq_n_s16(__p0, __p1, __p2) __extension__ ({ \
  int16x8_t __s0 = __p0; \
  int16x8_t __s1 = __p1; \
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  int16x8_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 7, 6, 5, 4, 3, 2, 1, 0); \
  int16x8_t __ret; \
  __ret = (int16x8_t) __builtin_neon_vrsraq_n_v((int8x16_t)__rev0, (int8x16_t)__rev1, __p2, 33); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vrsra_n_u8(__p0, __p1, __p2) __extension__ ({ \
  uint8x8_t __s0 = __p0; \
  uint8x8_t __s1 = __p1; \
  uint8x8_t __ret; \
  __ret = (uint8x8_t) __builtin_neon_vrsra_n_v((int8x8_t)__s0, (int8x8_t)__s1, __p2, 16); \
  __ret; \
})
#else
#define vrsra_n_u8(__p0, __p1, __p2) __extension__ ({ \
  uint8x8_t __s0 = __p0; \
  uint8x8_t __s1 = __p1; \
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint8x8_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint8x8_t __ret; \
  __ret = (uint8x8_t) __builtin_neon_vrsra_n_v((int8x8_t)__rev0, (int8x8_t)__rev1, __p2, 16); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vrsra_n_u32(__p0, __p1, __p2) __extension__ ({ \
  uint32x2_t __s0 = __p0; \
  uint32x2_t __s1 = __p1; \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_vrsra_n_v((int8x8_t)__s0, (int8x8_t)__s1, __p2, 18); \
  __ret; \
})
#else
#define vrsra_n_u32(__p0, __p1, __p2) __extension__ ({ \
  uint32x2_t __s0 = __p0; \
  uint32x2_t __s1 = __p1; \
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  uint32x2_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 1, 0); \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_vrsra_n_v((int8x8_t)__rev0, (int8x8_t)__rev1, __p2, 18); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#define vrsra_n_u64(__p0, __p1, __p2) __extension__ ({ \
  uint64x1_t __s0 = __p0; \
  uint64x1_t __s1 = __p1; \
  uint64x1_t __ret; \
  __ret = (uint64x1_t) __builtin_neon_vrsra_n_v((int8x8_t)__s0, (int8x8_t)__s1, __p2, 19); \
  __ret; \
})
#ifdef __LITTLE_ENDIAN__
#define vrsra_n_u16(__p0, __p1, __p2) __extension__ ({ \
  uint16x4_t __s0 = __p0; \
  uint16x4_t __s1 = __p1; \
  uint16x4_t __ret; \
  __ret = (uint16x4_t) __builtin_neon_vrsra_n_v((int8x8_t)__s0, (int8x8_t)__s1, __p2, 17); \
  __ret; \
})
#else
#define vrsra_n_u16(__p0, __p1, __p2) __extension__ ({ \
  uint16x4_t __s0 = __p0; \
  uint16x4_t __s1 = __p1; \
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  uint16x4_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 3, 2, 1, 0); \
  uint16x4_t __ret; \
  __ret = (uint16x4_t) __builtin_neon_vrsra_n_v((int8x8_t)__rev0, (int8x8_t)__rev1, __p2, 17); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vrsra_n_s8(__p0, __p1, __p2) __extension__ ({ \
  int8x8_t __s0 = __p0; \
  int8x8_t __s1 = __p1; \
  int8x8_t __ret; \
  __ret = (int8x8_t) __builtin_neon_vrsra_n_v((int8x8_t)__s0, (int8x8_t)__s1, __p2, 0); \
  __ret; \
})
#else
#define vrsra_n_s8(__p0, __p1, __p2) __extension__ ({ \
  int8x8_t __s0 = __p0; \
  int8x8_t __s1 = __p1; \
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  int8x8_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 7, 6, 5, 4, 3, 2, 1, 0); \
  int8x8_t __ret; \
  __ret = (int8x8_t) __builtin_neon_vrsra_n_v((int8x8_t)__rev0, (int8x8_t)__rev1, __p2, 0); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vrsra_n_s32(__p0, __p1, __p2) __extension__ ({ \
  int32x2_t __s0 = __p0; \
  int32x2_t __s1 = __p1; \
  int32x2_t __ret; \
  __ret = (int32x2_t) __builtin_neon_vrsra_n_v((int8x8_t)__s0, (int8x8_t)__s1, __p2, 2); \
  __ret; \
})
#else
#define vrsra_n_s32(__p0, __p1, __p2) __extension__ ({ \
  int32x2_t __s0 = __p0; \
  int32x2_t __s1 = __p1; \
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 1, 0); \
  int32x2_t __ret; \
  __ret = (int32x2_t) __builtin_neon_vrsra_n_v((int8x8_t)__rev0, (int8x8_t)__rev1, __p2, 2); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#define vrsra_n_s64(__p0, __p1, __p2) __extension__ ({ \
  int64x1_t __s0 = __p0; \
  int64x1_t __s1 = __p1; \
  int64x1_t __ret; \
  __ret = (int64x1_t) __builtin_neon_vrsra_n_v((int8x8_t)__s0, (int8x8_t)__s1, __p2, 3); \
  __ret; \
})
#ifdef __LITTLE_ENDIAN__
#define vrsra_n_s16(__p0, __p1, __p2) __extension__ ({ \
  int16x4_t __s0 = __p0; \
  int16x4_t __s1 = __p1; \
  int16x4_t __ret; \
  __ret = (int16x4_t) __builtin_neon_vrsra_n_v((int8x8_t)__s0, (int8x8_t)__s1, __p2, 1); \
  __ret; \
})
#else
#define vrsra_n_s16(__p0, __p1, __p2) __extension__ ({ \
  int16x4_t __s0 = __p0; \
  int16x4_t __s1 = __p1; \
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 3, 2, 1, 0); \
  int16x4_t __ret; \
  __ret = (int16x4_t) __builtin_neon_vrsra_n_v((int8x8_t)__rev0, (int8x8_t)__rev1, __p2, 1); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x4_t vrsubhn_u32(uint32x4_t __p0, uint32x4_t __p1) {
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vrsubhn_v((int8x16_t)__p0, (int8x16_t)__p1, 17);
  return __ret;
}
#else
__ai uint16x4_t vrsubhn_u32(uint32x4_t __p0, uint32x4_t __p1) {
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  uint32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vrsubhn_v((int8x16_t)__rev0, (int8x16_t)__rev1, 17);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
__ai uint16x4_t __noswap_vrsubhn_u32(uint32x4_t __p0, uint32x4_t __p1) {
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vrsubhn_v((int8x16_t)__p0, (int8x16_t)__p1, 17);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vrsubhn_u64(uint64x2_t __p0, uint64x2_t __p1) {
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vrsubhn_v((int8x16_t)__p0, (int8x16_t)__p1, 18);
  return __ret;
}
#else
__ai uint32x2_t vrsubhn_u64(uint64x2_t __p0, uint64x2_t __p1) {
  uint64x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  uint64x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vrsubhn_v((int8x16_t)__rev0, (int8x16_t)__rev1, 18);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
__ai uint32x2_t __noswap_vrsubhn_u64(uint64x2_t __p0, uint64x2_t __p1) {
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vrsubhn_v((int8x16_t)__p0, (int8x16_t)__p1, 18);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x8_t vrsubhn_u16(uint16x8_t __p0, uint16x8_t __p1) {
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vrsubhn_v((int8x16_t)__p0, (int8x16_t)__p1, 16);
  return __ret;
}
#else
__ai uint8x8_t vrsubhn_u16(uint16x8_t __p0, uint16x8_t __p1) {
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vrsubhn_v((int8x16_t)__rev0, (int8x16_t)__rev1, 16);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
__ai uint8x8_t __noswap_vrsubhn_u16(uint16x8_t __p0, uint16x8_t __p1) {
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vrsubhn_v((int8x16_t)__p0, (int8x16_t)__p1, 16);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x4_t vrsubhn_s32(int32x4_t __p0, int32x4_t __p1) {
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vrsubhn_v((int8x16_t)__p0, (int8x16_t)__p1, 1);
  return __ret;
}
#else
__ai int16x4_t vrsubhn_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vrsubhn_v((int8x16_t)__rev0, (int8x16_t)__rev1, 1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
__ai int16x4_t __noswap_vrsubhn_s32(int32x4_t __p0, int32x4_t __p1) {
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vrsubhn_v((int8x16_t)__p0, (int8x16_t)__p1, 1);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x2_t vrsubhn_s64(int64x2_t __p0, int64x2_t __p1) {
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vrsubhn_v((int8x16_t)__p0, (int8x16_t)__p1, 2);
  return __ret;
}
#else
__ai int32x2_t vrsubhn_s64(int64x2_t __p0, int64x2_t __p1) {
  int64x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int64x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vrsubhn_v((int8x16_t)__rev0, (int8x16_t)__rev1, 2);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
__ai int32x2_t __noswap_vrsubhn_s64(int64x2_t __p0, int64x2_t __p1) {
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vrsubhn_v((int8x16_t)__p0, (int8x16_t)__p1, 2);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x8_t vrsubhn_s16(int16x8_t __p0, int16x8_t __p1) {
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vrsubhn_v((int8x16_t)__p0, (int8x16_t)__p1, 0);
  return __ret;
}
#else
__ai int8x8_t vrsubhn_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vrsubhn_v((int8x16_t)__rev0, (int8x16_t)__rev1, 0);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
__ai int8x8_t __noswap_vrsubhn_s16(int16x8_t __p0, int16x8_t __p1) {
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vrsubhn_v((int8x16_t)__p0, (int8x16_t)__p1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
#define vset_lane_p8(__p0, __p1, __p2) __extension__ ({ \
  poly8_t __s0 = __p0; \
  poly8x8_t __s1 = __p1; \
  poly8x8_t __ret; \
  __ret = (poly8x8_t) __builtin_neon_vset_lane_i8(__s0, (poly8x8_t)__s1, __p2); \
  __ret; \
})
#else
#define vset_lane_p8(__p0, __p1, __p2) __extension__ ({ \
  poly8_t __s0 = __p0; \
  poly8x8_t __s1 = __p1; \
  poly8x8_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 7, 6, 5, 4, 3, 2, 1, 0); \
  poly8x8_t __ret; \
  __ret = (poly8x8_t) __builtin_neon_vset_lane_i8(__s0, (poly8x8_t)__rev1, __p2); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_vset_lane_p8(__p0, __p1, __p2) __extension__ ({ \
  poly8_t __s0 = __p0; \
  poly8x8_t __s1 = __p1; \
  poly8x8_t __ret; \
  __ret = (poly8x8_t) __builtin_neon_vset_lane_i8(__s0, (poly8x8_t)__s1, __p2); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vset_lane_p16(__p0, __p1, __p2) __extension__ ({ \
  poly16_t __s0 = __p0; \
  poly16x4_t __s1 = __p1; \
  poly16x4_t __ret; \
  __ret = (poly16x4_t) __builtin_neon_vset_lane_i16(__s0, (poly16x4_t)__s1, __p2); \
  __ret; \
})
#else
#define vset_lane_p16(__p0, __p1, __p2) __extension__ ({ \
  poly16_t __s0 = __p0; \
  poly16x4_t __s1 = __p1; \
  poly16x4_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 3, 2, 1, 0); \
  poly16x4_t __ret; \
  __ret = (poly16x4_t) __builtin_neon_vset_lane_i16(__s0, (poly16x4_t)__rev1, __p2); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_vset_lane_p16(__p0, __p1, __p2) __extension__ ({ \
  poly16_t __s0 = __p0; \
  poly16x4_t __s1 = __p1; \
  poly16x4_t __ret; \
  __ret = (poly16x4_t) __builtin_neon_vset_lane_i16(__s0, (poly16x4_t)__s1, __p2); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsetq_lane_p8(__p0, __p1, __p2) __extension__ ({ \
  poly8_t __s0 = __p0; \
  poly8x16_t __s1 = __p1; \
  poly8x16_t __ret; \
  __ret = (poly8x16_t) __builtin_neon_vsetq_lane_i8(__s0, (poly8x16_t)__s1, __p2); \
  __ret; \
})
#else
#define vsetq_lane_p8(__p0, __p1, __p2) __extension__ ({ \
  poly8_t __s0 = __p0; \
  poly8x16_t __s1 = __p1; \
  poly8x16_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  poly8x16_t __ret; \
  __ret = (poly8x16_t) __builtin_neon_vsetq_lane_i8(__s0, (poly8x16_t)__rev1, __p2); \
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_vsetq_lane_p8(__p0, __p1, __p2) __extension__ ({ \
  poly8_t __s0 = __p0; \
  poly8x16_t __s1 = __p1; \
  poly8x16_t __ret; \
  __ret = (poly8x16_t) __builtin_neon_vsetq_lane_i8(__s0, (poly8x16_t)__s1, __p2); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsetq_lane_p16(__p0, __p1, __p2) __extension__ ({ \
  poly16_t __s0 = __p0; \
  poly16x8_t __s1 = __p1; \
  poly16x8_t __ret; \
  __ret = (poly16x8_t) __builtin_neon_vsetq_lane_i16(__s0, (poly16x8_t)__s1, __p2); \
  __ret; \
})
#else
#define vsetq_lane_p16(__p0, __p1, __p2) __extension__ ({ \
  poly16_t __s0 = __p0; \
  poly16x8_t __s1 = __p1; \
  poly16x8_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 7, 6, 5, 4, 3, 2, 1, 0); \
  poly16x8_t __ret; \
  __ret = (poly16x8_t) __builtin_neon_vsetq_lane_i16(__s0, (poly16x8_t)__rev1, __p2); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_vsetq_lane_p16(__p0, __p1, __p2) __extension__ ({ \
  poly16_t __s0 = __p0; \
  poly16x8_t __s1 = __p1; \
  poly16x8_t __ret; \
  __ret = (poly16x8_t) __builtin_neon_vsetq_lane_i16(__s0, (poly16x8_t)__s1, __p2); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsetq_lane_u8(__p0, __p1, __p2) __extension__ ({ \
  uint8_t __s0 = __p0; \
  uint8x16_t __s1 = __p1; \
  uint8x16_t __ret; \
  __ret = (uint8x16_t) __builtin_neon_vsetq_lane_i8(__s0, (int8x16_t)__s1, __p2); \
  __ret; \
})
#else
#define vsetq_lane_u8(__p0, __p1, __p2) __extension__ ({ \
  uint8_t __s0 = __p0; \
  uint8x16_t __s1 = __p1; \
  uint8x16_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint8x16_t __ret; \
  __ret = (uint8x16_t) __builtin_neon_vsetq_lane_i8(__s0, (int8x16_t)__rev1, __p2); \
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_vsetq_lane_u8(__p0, __p1, __p2) __extension__ ({ \
  uint8_t __s0 = __p0; \
  uint8x16_t __s1 = __p1; \
  uint8x16_t __ret; \
  __ret = (uint8x16_t) __builtin_neon_vsetq_lane_i8(__s0, (int8x16_t)__s1, __p2); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsetq_lane_u32(__p0, __p1, __p2) __extension__ ({ \
  uint32_t __s0 = __p0; \
  uint32x4_t __s1 = __p1; \
  uint32x4_t __ret; \
  __ret = (uint32x4_t) __builtin_neon_vsetq_lane_i32(__s0, (int32x4_t)__s1, __p2); \
  __ret; \
})
#else
#define vsetq_lane_u32(__p0, __p1, __p2) __extension__ ({ \
  uint32_t __s0 = __p0; \
  uint32x4_t __s1 = __p1; \
  uint32x4_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 3, 2, 1, 0); \
  uint32x4_t __ret; \
  __ret = (uint32x4_t) __builtin_neon_vsetq_lane_i32(__s0, (int32x4_t)__rev1, __p2); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_vsetq_lane_u32(__p0, __p1, __p2) __extension__ ({ \
  uint32_t __s0 = __p0; \
  uint32x4_t __s1 = __p1; \
  uint32x4_t __ret; \
  __ret = (uint32x4_t) __builtin_neon_vsetq_lane_i32(__s0, (int32x4_t)__s1, __p2); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsetq_lane_u64(__p0, __p1, __p2) __extension__ ({ \
  uint64_t __s0 = __p0; \
  uint64x2_t __s1 = __p1; \
  uint64x2_t __ret; \
  __ret = (uint64x2_t) __builtin_neon_vsetq_lane_i64(__s0, (int64x2_t)__s1, __p2); \
  __ret; \
})
#else
#define vsetq_lane_u64(__p0, __p1, __p2) __extension__ ({ \
  uint64_t __s0 = __p0; \
  uint64x2_t __s1 = __p1; \
  uint64x2_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 1, 0); \
  uint64x2_t __ret; \
  __ret = (uint64x2_t) __builtin_neon_vsetq_lane_i64(__s0, (int64x2_t)__rev1, __p2); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#define __noswap_vsetq_lane_u64(__p0, __p1, __p2) __extension__ ({ \
  uint64_t __s0 = __p0; \
  uint64x2_t __s1 = __p1; \
  uint64x2_t __ret; \
  __ret = (uint64x2_t) __builtin_neon_vsetq_lane_i64(__s0, (int64x2_t)__s1, __p2); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsetq_lane_u16(__p0, __p1, __p2) __extension__ ({ \
  uint16_t __s0 = __p0; \
  uint16x8_t __s1 = __p1; \
  uint16x8_t __ret; \
  __ret = (uint16x8_t) __builtin_neon_vsetq_lane_i16(__s0, (int16x8_t)__s1, __p2); \
  __ret; \
})
#else
#define vsetq_lane_u16(__p0, __p1, __p2) __extension__ ({ \
  uint16_t __s0 = __p0; \
  uint16x8_t __s1 = __p1; \
  uint16x8_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint16x8_t __ret; \
  __ret = (uint16x8_t) __builtin_neon_vsetq_lane_i16(__s0, (int16x8_t)__rev1, __p2); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_vsetq_lane_u16(__p0, __p1, __p2) __extension__ ({ \
  uint16_t __s0 = __p0; \
  uint16x8_t __s1 = __p1; \
  uint16x8_t __ret; \
  __ret = (uint16x8_t) __builtin_neon_vsetq_lane_i16(__s0, (int16x8_t)__s1, __p2); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsetq_lane_s8(__p0, __p1, __p2) __extension__ ({ \
  int8_t __s0 = __p0; \
  int8x16_t __s1 = __p1; \
  int8x16_t __ret; \
  __ret = (int8x16_t) __builtin_neon_vsetq_lane_i8(__s0, (int8x16_t)__s1, __p2); \
  __ret; \
})
#else
#define vsetq_lane_s8(__p0, __p1, __p2) __extension__ ({ \
  int8_t __s0 = __p0; \
  int8x16_t __s1 = __p1; \
  int8x16_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  int8x16_t __ret; \
  __ret = (int8x16_t) __builtin_neon_vsetq_lane_i8(__s0, (int8x16_t)__rev1, __p2); \
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_vsetq_lane_s8(__p0, __p1, __p2) __extension__ ({ \
  int8_t __s0 = __p0; \
  int8x16_t __s1 = __p1; \
  int8x16_t __ret; \
  __ret = (int8x16_t) __builtin_neon_vsetq_lane_i8(__s0, (int8x16_t)__s1, __p2); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsetq_lane_f32(__p0, __p1, __p2) __extension__ ({ \
  float32_t __s0 = __p0; \
  float32x4_t __s1 = __p1; \
  float32x4_t __ret; \
  __ret = (float32x4_t) __builtin_neon_vsetq_lane_f32(__s0, (float32x4_t)__s1, __p2); \
  __ret; \
})
#else
#define vsetq_lane_f32(__p0, __p1, __p2) __extension__ ({ \
  float32_t __s0 = __p0; \
  float32x4_t __s1 = __p1; \
  float32x4_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 3, 2, 1, 0); \
  float32x4_t __ret; \
  __ret = (float32x4_t) __builtin_neon_vsetq_lane_f32(__s0, (float32x4_t)__rev1, __p2); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_vsetq_lane_f32(__p0, __p1, __p2) __extension__ ({ \
  float32_t __s0 = __p0; \
  float32x4_t __s1 = __p1; \
  float32x4_t __ret; \
  __ret = (float32x4_t) __builtin_neon_vsetq_lane_f32(__s0, (float32x4_t)__s1, __p2); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsetq_lane_s32(__p0, __p1, __p2) __extension__ ({ \
  int32_t __s0 = __p0; \
  int32x4_t __s1 = __p1; \
  int32x4_t __ret; \
  __ret = (int32x4_t) __builtin_neon_vsetq_lane_i32(__s0, (int32x4_t)__s1, __p2); \
  __ret; \
})
#else
#define vsetq_lane_s32(__p0, __p1, __p2) __extension__ ({ \
  int32_t __s0 = __p0; \
  int32x4_t __s1 = __p1; \
  int32x4_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 3, 2, 1, 0); \
  int32x4_t __ret; \
  __ret = (int32x4_t) __builtin_neon_vsetq_lane_i32(__s0, (int32x4_t)__rev1, __p2); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_vsetq_lane_s32(__p0, __p1, __p2) __extension__ ({ \
  int32_t __s0 = __p0; \
  int32x4_t __s1 = __p1; \
  int32x4_t __ret; \
  __ret = (int32x4_t) __builtin_neon_vsetq_lane_i32(__s0, (int32x4_t)__s1, __p2); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsetq_lane_s64(__p0, __p1, __p2) __extension__ ({ \
  int64_t __s0 = __p0; \
  int64x2_t __s1 = __p1; \
  int64x2_t __ret; \
  __ret = (int64x2_t) __builtin_neon_vsetq_lane_i64(__s0, (int64x2_t)__s1, __p2); \
  __ret; \
})
#else
#define vsetq_lane_s64(__p0, __p1, __p2) __extension__ ({ \
  int64_t __s0 = __p0; \
  int64x2_t __s1 = __p1; \
  int64x2_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 1, 0); \
  int64x2_t __ret; \
  __ret = (int64x2_t) __builtin_neon_vsetq_lane_i64(__s0, (int64x2_t)__rev1, __p2); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#define __noswap_vsetq_lane_s64(__p0, __p1, __p2) __extension__ ({ \
  int64_t __s0 = __p0; \
  int64x2_t __s1 = __p1; \
  int64x2_t __ret; \
  __ret = (int64x2_t) __builtin_neon_vsetq_lane_i64(__s0, (int64x2_t)__s1, __p2); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsetq_lane_s16(__p0, __p1, __p2) __extension__ ({ \
  int16_t __s0 = __p0; \
  int16x8_t __s1 = __p1; \
  int16x8_t __ret; \
  __ret = (int16x8_t) __builtin_neon_vsetq_lane_i16(__s0, (int16x8_t)__s1, __p2); \
  __ret; \
})
#else
#define vsetq_lane_s16(__p0, __p1, __p2) __extension__ ({ \
  int16_t __s0 = __p0; \
  int16x8_t __s1 = __p1; \
  int16x8_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 7, 6, 5, 4, 3, 2, 1, 0); \
  int16x8_t __ret; \
  __ret = (int16x8_t) __builtin_neon_vsetq_lane_i16(__s0, (int16x8_t)__rev1, __p2); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_vsetq_lane_s16(__p0, __p1, __p2) __extension__ ({ \
  int16_t __s0 = __p0; \
  int16x8_t __s1 = __p1; \
  int16x8_t __ret; \
  __ret = (int16x8_t) __builtin_neon_vsetq_lane_i16(__s0, (int16x8_t)__s1, __p2); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vset_lane_u8(__p0, __p1, __p2) __extension__ ({ \
  uint8_t __s0 = __p0; \
  uint8x8_t __s1 = __p1; \
  uint8x8_t __ret; \
  __ret = (uint8x8_t) __builtin_neon_vset_lane_i8(__s0, (int8x8_t)__s1, __p2); \
  __ret; \
})
#else
#define vset_lane_u8(__p0, __p1, __p2) __extension__ ({ \
  uint8_t __s0 = __p0; \
  uint8x8_t __s1 = __p1; \
  uint8x8_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint8x8_t __ret; \
  __ret = (uint8x8_t) __builtin_neon_vset_lane_i8(__s0, (int8x8_t)__rev1, __p2); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_vset_lane_u8(__p0, __p1, __p2) __extension__ ({ \
  uint8_t __s0 = __p0; \
  uint8x8_t __s1 = __p1; \
  uint8x8_t __ret; \
  __ret = (uint8x8_t) __builtin_neon_vset_lane_i8(__s0, (int8x8_t)__s1, __p2); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vset_lane_u32(__p0, __p1, __p2) __extension__ ({ \
  uint32_t __s0 = __p0; \
  uint32x2_t __s1 = __p1; \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_vset_lane_i32(__s0, (int32x2_t)__s1, __p2); \
  __ret; \
})
#else
#define vset_lane_u32(__p0, __p1, __p2) __extension__ ({ \
  uint32_t __s0 = __p0; \
  uint32x2_t __s1 = __p1; \
  uint32x2_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 1, 0); \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_vset_lane_i32(__s0, (int32x2_t)__rev1, __p2); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#define __noswap_vset_lane_u32(__p0, __p1, __p2) __extension__ ({ \
  uint32_t __s0 = __p0; \
  uint32x2_t __s1 = __p1; \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_vset_lane_i32(__s0, (int32x2_t)__s1, __p2); \
  __ret; \
})
#endif

#define vset_lane_u64(__p0, __p1, __p2) __extension__ ({ \
  uint64_t __s0 = __p0; \
  uint64x1_t __s1 = __p1; \
  uint64x1_t __ret; \
  __ret = (uint64x1_t) __builtin_neon_vset_lane_i64(__s0, (int64x1_t)__s1, __p2); \
  __ret; \
})
#ifdef __LITTLE_ENDIAN__
#define vset_lane_u16(__p0, __p1, __p2) __extension__ ({ \
  uint16_t __s0 = __p0; \
  uint16x4_t __s1 = __p1; \
  uint16x4_t __ret; \
  __ret = (uint16x4_t) __builtin_neon_vset_lane_i16(__s0, (int16x4_t)__s1, __p2); \
  __ret; \
})
#else
#define vset_lane_u16(__p0, __p1, __p2) __extension__ ({ \
  uint16_t __s0 = __p0; \
  uint16x4_t __s1 = __p1; \
  uint16x4_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 3, 2, 1, 0); \
  uint16x4_t __ret; \
  __ret = (uint16x4_t) __builtin_neon_vset_lane_i16(__s0, (int16x4_t)__rev1, __p2); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_vset_lane_u16(__p0, __p1, __p2) __extension__ ({ \
  uint16_t __s0 = __p0; \
  uint16x4_t __s1 = __p1; \
  uint16x4_t __ret; \
  __ret = (uint16x4_t) __builtin_neon_vset_lane_i16(__s0, (int16x4_t)__s1, __p2); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vset_lane_s8(__p0, __p1, __p2) __extension__ ({ \
  int8_t __s0 = __p0; \
  int8x8_t __s1 = __p1; \
  int8x8_t __ret; \
  __ret = (int8x8_t) __builtin_neon_vset_lane_i8(__s0, (int8x8_t)__s1, __p2); \
  __ret; \
})
#else
#define vset_lane_s8(__p0, __p1, __p2) __extension__ ({ \
  int8_t __s0 = __p0; \
  int8x8_t __s1 = __p1; \
  int8x8_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 7, 6, 5, 4, 3, 2, 1, 0); \
  int8x8_t __ret; \
  __ret = (int8x8_t) __builtin_neon_vset_lane_i8(__s0, (int8x8_t)__rev1, __p2); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_vset_lane_s8(__p0, __p1, __p2) __extension__ ({ \
  int8_t __s0 = __p0; \
  int8x8_t __s1 = __p1; \
  int8x8_t __ret; \
  __ret = (int8x8_t) __builtin_neon_vset_lane_i8(__s0, (int8x8_t)__s1, __p2); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vset_lane_f32(__p0, __p1, __p2) __extension__ ({ \
  float32_t __s0 = __p0; \
  float32x2_t __s1 = __p1; \
  float32x2_t __ret; \
  __ret = (float32x2_t) __builtin_neon_vset_lane_f32(__s0, (float32x2_t)__s1, __p2); \
  __ret; \
})
#else
#define vset_lane_f32(__p0, __p1, __p2) __extension__ ({ \
  float32_t __s0 = __p0; \
  float32x2_t __s1 = __p1; \
  float32x2_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 1, 0); \
  float32x2_t __ret; \
  __ret = (float32x2_t) __builtin_neon_vset_lane_f32(__s0, (float32x2_t)__rev1, __p2); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#define __noswap_vset_lane_f32(__p0, __p1, __p2) __extension__ ({ \
  float32_t __s0 = __p0; \
  float32x2_t __s1 = __p1; \
  float32x2_t __ret; \
  __ret = (float32x2_t) __builtin_neon_vset_lane_f32(__s0, (float32x2_t)__s1, __p2); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vset_lane_s32(__p0, __p1, __p2) __extension__ ({ \
  int32_t __s0 = __p0; \
  int32x2_t __s1 = __p1; \
  int32x2_t __ret; \
  __ret = (int32x2_t) __builtin_neon_vset_lane_i32(__s0, (int32x2_t)__s1, __p2); \
  __ret; \
})
#else
#define vset_lane_s32(__p0, __p1, __p2) __extension__ ({ \
  int32_t __s0 = __p0; \
  int32x2_t __s1 = __p1; \
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 1, 0); \
  int32x2_t __ret; \
  __ret = (int32x2_t) __builtin_neon_vset_lane_i32(__s0, (int32x2_t)__rev1, __p2); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#define __noswap_vset_lane_s32(__p0, __p1, __p2) __extension__ ({ \
  int32_t __s0 = __p0; \
  int32x2_t __s1 = __p1; \
  int32x2_t __ret; \
  __ret = (int32x2_t) __builtin_neon_vset_lane_i32(__s0, (int32x2_t)__s1, __p2); \
  __ret; \
})
#endif

#define vset_lane_s64(__p0, __p1, __p2) __extension__ ({ \
  int64_t __s0 = __p0; \
  int64x1_t __s1 = __p1; \
  int64x1_t __ret; \
  __ret = (int64x1_t) __builtin_neon_vset_lane_i64(__s0, (int64x1_t)__s1, __p2); \
  __ret; \
})
#ifdef __LITTLE_ENDIAN__
#define vset_lane_s16(__p0, __p1, __p2) __extension__ ({ \
  int16_t __s0 = __p0; \
  int16x4_t __s1 = __p1; \
  int16x4_t __ret; \
  __ret = (int16x4_t) __builtin_neon_vset_lane_i16(__s0, (int16x4_t)__s1, __p2); \
  __ret; \
})
#else
#define vset_lane_s16(__p0, __p1, __p2) __extension__ ({ \
  int16_t __s0 = __p0; \
  int16x4_t __s1 = __p1; \
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 3, 2, 1, 0); \
  int16x4_t __ret; \
  __ret = (int16x4_t) __builtin_neon_vset_lane_i16(__s0, (int16x4_t)__rev1, __p2); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_vset_lane_s16(__p0, __p1, __p2) __extension__ ({ \
  int16_t __s0 = __p0; \
  int16x4_t __s1 = __p1; \
  int16x4_t __ret; \
  __ret = (int16x4_t) __builtin_neon_vset_lane_i16(__s0, (int16x4_t)__s1, __p2); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x16_t vshlq_u8(uint8x16_t __p0, int8x16_t __p1) {
  uint8x16_t __ret;
  __ret = (uint8x16_t) __builtin_neon_vshlq_v((int8x16_t)__p0, (int8x16_t)__p1, 48);
  return __ret;
}
#else
__ai uint8x16_t vshlq_u8(uint8x16_t __p0, int8x16_t __p1) {
  uint8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x16_t __ret;
  __ret = (uint8x16_t) __builtin_neon_vshlq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 48);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x4_t vshlq_u32(uint32x4_t __p0, int32x4_t __p1) {
  uint32x4_t __ret;
  __ret = (uint32x4_t) __builtin_neon_vshlq_v((int8x16_t)__p0, (int8x16_t)__p1, 50);
  return __ret;
}
#else
__ai uint32x4_t vshlq_u32(uint32x4_t __p0, int32x4_t __p1) {
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint32x4_t __ret;
  __ret = (uint32x4_t) __builtin_neon_vshlq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 50);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint64x2_t vshlq_u64(uint64x2_t __p0, int64x2_t __p1) {
  uint64x2_t __ret;
  __ret = (uint64x2_t) __builtin_neon_vshlq_v((int8x16_t)__p0, (int8x16_t)__p1, 51);
  return __ret;
}
#else
__ai uint64x2_t vshlq_u64(uint64x2_t __p0, int64x2_t __p1) {
  uint64x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int64x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint64x2_t __ret;
  __ret = (uint64x2_t) __builtin_neon_vshlq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 51);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint16x8_t vshlq_u16(uint16x8_t __p0, int16x8_t __p1) {
  uint16x8_t __ret;
  __ret = (uint16x8_t) __builtin_neon_vshlq_v((int8x16_t)__p0, (int8x16_t)__p1, 49);
  return __ret;
}
#else
__ai uint16x8_t vshlq_u16(uint16x8_t __p0, int16x8_t __p1) {
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint16x8_t __ret;
  __ret = (uint16x8_t) __builtin_neon_vshlq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 49);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x16_t vshlq_s8(int8x16_t __p0, int8x16_t __p1) {
  int8x16_t __ret;
  __ret = (int8x16_t) __builtin_neon_vshlq_v((int8x16_t)__p0, (int8x16_t)__p1, 32);
  return __ret;
}
#else
__ai int8x16_t vshlq_s8(int8x16_t __p0, int8x16_t __p1) {
  int8x16_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x16_t __ret;
  __ret = (int8x16_t) __builtin_neon_vshlq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 32);
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x4_t vshlq_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vshlq_v((int8x16_t)__p0, (int8x16_t)__p1, 34);
  return __ret;
}
#else
__ai int32x4_t vshlq_s32(int32x4_t __p0, int32x4_t __p1) {
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int32x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int32x4_t __ret;
  __ret = (int32x4_t) __builtin_neon_vshlq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 34);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int64x2_t vshlq_s64(int64x2_t __p0, int64x2_t __p1) {
  int64x2_t __ret;
  __ret = (int64x2_t) __builtin_neon_vshlq_v((int8x16_t)__p0, (int8x16_t)__p1, 35);
  return __ret;
}
#else
__ai int64x2_t vshlq_s64(int64x2_t __p0, int64x2_t __p1) {
  int64x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int64x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  int64x2_t __ret;
  __ret = (int64x2_t) __builtin_neon_vshlq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 35);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int16x8_t vshlq_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vshlq_v((int8x16_t)__p0, (int8x16_t)__p1, 33);
  return __ret;
}
#else
__ai int16x8_t vshlq_s16(int16x8_t __p0, int16x8_t __p1) {
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int16x8_t __ret;
  __ret = (int16x8_t) __builtin_neon_vshlq_v((int8x16_t)__rev0, (int8x16_t)__rev1, 33);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint8x8_t vshl_u8(uint8x8_t __p0, int8x8_t __p1) {
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vshl_v((int8x8_t)__p0, (int8x8_t)__p1, 16);
  return __ret;
}
#else
__ai uint8x8_t vshl_u8(uint8x8_t __p0, int8x8_t __p1) {
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  uint8x8_t __ret;
  __ret = (uint8x8_t) __builtin_neon_vshl_v((int8x8_t)__rev0, (int8x8_t)__rev1, 16);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai uint32x2_t vshl_u32(uint32x2_t __p0, int32x2_t __p1) {
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vshl_v((int8x8_t)__p0, (int8x8_t)__p1, 18);
  return __ret;
}
#else
__ai uint32x2_t vshl_u32(uint32x2_t __p0, int32x2_t __p1) {
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  uint32x2_t __ret;
  __ret = (uint32x2_t) __builtin_neon_vshl_v((int8x8_t)__rev0, (int8x8_t)__rev1, 18);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

__ai uint64x1_t vshl_u64(uint64x1_t __p0, int64x1_t __p1) {
  uint64x1_t __ret;
  __ret = (uint64x1_t) __builtin_neon_vshl_v((int8x8_t)__p0, (int8x8_t)__p1, 19);
  return __ret;
}
#ifdef __LITTLE_ENDIAN__
__ai uint16x4_t vshl_u16(uint16x4_t __p0, int16x4_t __p1) {
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vshl_v((int8x8_t)__p0, (int8x8_t)__p1, 17);
  return __ret;
}
#else
__ai uint16x4_t vshl_u16(uint16x4_t __p0, int16x4_t __p1) {
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  uint16x4_t __ret;
  __ret = (uint16x4_t) __builtin_neon_vshl_v((int8x8_t)__rev0, (int8x8_t)__rev1, 17);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int8x8_t vshl_s8(int8x8_t __p0, int8x8_t __p1) {
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vshl_v((int8x8_t)__p0, (int8x8_t)__p1, 0);
  return __ret;
}
#else
__ai int8x8_t vshl_s8(int8x8_t __p0, int8x8_t __p1) {
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 7, 6, 5, 4, 3, 2, 1, 0);
  int8x8_t __ret;
  __ret = (int8x8_t) __builtin_neon_vshl_v((int8x8_t)__rev0, (int8x8_t)__rev1, 0);
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
__ai int32x2_t vshl_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vshl_v((int8x8_t)__p0, (int8x8_t)__p1, 2);
  return __ret;
}
#else
__ai int32x2_t vshl_s32(int32x2_t __p0, int32x2_t __p1) {
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 1, 0);
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 1, 0);
  int32x2_t __ret;
  __ret = (int32x2_t) __builtin_neon_vshl_v((int8x8_t)__rev0, (int8x8_t)__rev1, 2);
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0);
  return __ret;
}
#endif

__ai int64x1_t vshl_s64(int64x1_t __p0, int64x1_t __p1) {
  int64x1_t __ret;
  __ret = (int64x1_t) __builtin_neon_vshl_v((int8x8_t)__p0, (int8x8_t)__p1, 3);
  return __ret;
}
#ifdef __LITTLE_ENDIAN__
__ai int16x4_t vshl_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vshl_v((int8x8_t)__p0, (int8x8_t)__p1, 1);
  return __ret;
}
#else
__ai int16x4_t vshl_s16(int16x4_t __p0, int16x4_t __p1) {
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__p0, __p0, 3, 2, 1, 0);
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__p1, __p1, 3, 2, 1, 0);
  int16x4_t __ret;
  __ret = (int16x4_t) __builtin_neon_vshl_v((int8x8_t)__rev0, (int8x8_t)__rev1, 1);
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0);
  return __ret;
}
#endif

#ifdef __LITTLE_ENDIAN__
#define vshlq_n_u8(__p0, __p1) __extension__ ({ \
  uint8x16_t __s0 = __p0; \
  uint8x16_t __ret; \
  __ret = (uint8x16_t) __builtin_neon_vshlq_n_v((int8x16_t)__s0, __p1, 48); \
  __ret; \
})
#else
#define vshlq_n_u8(__p0, __p1) __extension__ ({ \
  uint8x16_t __s0 = __p0; \
  uint8x16_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint8x16_t __ret; \
  __ret = (uint8x16_t) __builtin_neon_vshlq_n_v((int8x16_t)__rev0, __p1, 48); \
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vshlq_n_u32(__p0, __p1) __extension__ ({ \
  uint32x4_t __s0 = __p0; \
  uint32x4_t __ret; \
  __ret = (uint32x4_t) __builtin_neon_vshlq_n_v((int8x16_t)__s0, __p1, 50); \
  __ret; \
})
#else
#define vshlq_n_u32(__p0, __p1) __extension__ ({ \
  uint32x4_t __s0 = __p0; \
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  uint32x4_t __ret; \
  __ret = (uint32x4_t) __builtin_neon_vshlq_n_v((int8x16_t)__rev0, __p1, 50); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vshlq_n_u64(__p0, __p1) __extension__ ({ \
  uint64x2_t __s0 = __p0; \
  uint64x2_t __ret; \
  __ret = (uint64x2_t) __builtin_neon_vshlq_n_v((int8x16_t)__s0, __p1, 51); \
  __ret; \
})
#else
#define vshlq_n_u64(__p0, __p1) __extension__ ({ \
  uint64x2_t __s0 = __p0; \
  uint64x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  uint64x2_t __ret; \
  __ret = (uint64x2_t) __builtin_neon_vshlq_n_v((int8x16_t)__rev0, __p1, 51); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vshlq_n_u16(__p0, __p1) __extension__ ({ \
  uint16x8_t __s0 = __p0; \
  uint16x8_t __ret; \
  __ret = (uint16x8_t) __builtin_neon_vshlq_n_v((int8x16_t)__s0, __p1, 49); \
  __ret; \
})
#else
#define vshlq_n_u16(__p0, __p1) __extension__ ({ \
  uint16x8_t __s0 = __p0; \
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint16x8_t __ret; \
  __ret = (uint16x8_t) __builtin_neon_vshlq_n_v((int8x16_t)__rev0, __p1, 49); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vshlq_n_s8(__p0, __p1) __extension__ ({ \
  int8x16_t __s0 = __p0; \
  int8x16_t __ret; \
  __ret = (int8x16_t) __builtin_neon_vshlq_n_v((int8x16_t)__s0, __p1, 32); \
  __ret; \
})
#else
#define vshlq_n_s8(__p0, __p1) __extension__ ({ \
  int8x16_t __s0 = __p0; \
  int8x16_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  int8x16_t __ret; \
  __ret = (int8x16_t) __builtin_neon_vshlq_n_v((int8x16_t)__rev0, __p1, 32); \
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vshlq_n_s32(__p0, __p1) __extension__ ({ \
  int32x4_t __s0 = __p0; \
  int32x4_t __ret; \
  __ret = (int32x4_t) __builtin_neon_vshlq_n_v((int8x16_t)__s0, __p1, 34); \
  __ret; \
})
#else
#define vshlq_n_s32(__p0, __p1) __extension__ ({ \
  int32x4_t __s0 = __p0; \
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  int32x4_t __ret; \
  __ret = (int32x4_t) __builtin_neon_vshlq_n_v((int8x16_t)__rev0, __p1, 34); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vshlq_n_s64(__p0, __p1) __extension__ ({ \
  int64x2_t __s0 = __p0; \
  int64x2_t __ret; \
  __ret = (int64x2_t) __builtin_neon_vshlq_n_v((int8x16_t)__s0, __p1, 35); \
  __ret; \
})
#else
#define vshlq_n_s64(__p0, __p1) __extension__ ({ \
  int64x2_t __s0 = __p0; \
  int64x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  int64x2_t __ret; \
  __ret = (int64x2_t) __builtin_neon_vshlq_n_v((int8x16_t)__rev0, __p1, 35); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vshlq_n_s16(__p0, __p1) __extension__ ({ \
  int16x8_t __s0 = __p0; \
  int16x8_t __ret; \
  __ret = (int16x8_t) __builtin_neon_vshlq_n_v((int8x16_t)__s0, __p1, 33); \
  __ret; \
})
#else
#define vshlq_n_s16(__p0, __p1) __extension__ ({ \
  int16x8_t __s0 = __p0; \
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  int16x8_t __ret; \
  __ret = (int16x8_t) __builtin_neon_vshlq_n_v((int8x16_t)__rev0, __p1, 33); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vshl_n_u8(__p0, __p1) __extension__ ({ \
  uint8x8_t __s0 = __p0; \
  uint8x8_t __ret; \
  __ret = (uint8x8_t) __builtin_neon_vshl_n_v((int8x8_t)__s0, __p1, 16); \
  __ret; \
})
#else
#define vshl_n_u8(__p0, __p1) __extension__ ({ \
  uint8x8_t __s0 = __p0; \
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint8x8_t __ret; \
  __ret = (uint8x8_t) __builtin_neon_vshl_n_v((int8x8_t)__rev0, __p1, 16); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vshl_n_u32(__p0, __p1) __extension__ ({ \
  uint32x2_t __s0 = __p0; \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_vshl_n_v((int8x8_t)__s0, __p1, 18); \
  __ret; \
})
#else
#define vshl_n_u32(__p0, __p1) __extension__ ({ \
  uint32x2_t __s0 = __p0; \
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_vshl_n_v((int8x8_t)__rev0, __p1, 18); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#define vshl_n_u64(__p0, __p1) __extension__ ({ \
  uint64x1_t __s0 = __p0; \
  uint64x1_t __ret; \
  __ret = (uint64x1_t) __builtin_neon_vshl_n_v((int8x8_t)__s0, __p1, 19); \
  __ret; \
})
#ifdef __LITTLE_ENDIAN__
#define vshl_n_u16(__p0, __p1) __extension__ ({ \
  uint16x4_t __s0 = __p0; \
  uint16x4_t __ret; \
  __ret = (uint16x4_t) __builtin_neon_vshl_n_v((int8x8_t)__s0, __p1, 17); \
  __ret; \
})
#else
#define vshl_n_u16(__p0, __p1) __extension__ ({ \
  uint16x4_t __s0 = __p0; \
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  uint16x4_t __ret; \
  __ret = (uint16x4_t) __builtin_neon_vshl_n_v((int8x8_t)__rev0, __p1, 17); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vshl_n_s8(__p0, __p1) __extension__ ({ \
  int8x8_t __s0 = __p0; \
  int8x8_t __ret; \
  __ret = (int8x8_t) __builtin_neon_vshl_n_v((int8x8_t)__s0, __p1, 0); \
  __ret; \
})
#else
#define vshl_n_s8(__p0, __p1) __extension__ ({ \
  int8x8_t __s0 = __p0; \
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  int8x8_t __ret; \
  __ret = (int8x8_t) __builtin_neon_vshl_n_v((int8x8_t)__rev0, __p1, 0); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vshl_n_s32(__p0, __p1) __extension__ ({ \
  int32x2_t __s0 = __p0; \
  int32x2_t __ret; \
  __ret = (int32x2_t) __builtin_neon_vshl_n_v((int8x8_t)__s0, __p1, 2); \
  __ret; \
})
#else
#define vshl_n_s32(__p0, __p1) __extension__ ({ \
  int32x2_t __s0 = __p0; \
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  int32x2_t __ret; \
  __ret = (int32x2_t) __builtin_neon_vshl_n_v((int8x8_t)__rev0, __p1, 2); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#define vshl_n_s64(__p0, __p1) __extension__ ({ \
  int64x1_t __s0 = __p0; \
  int64x1_t __ret; \
  __ret = (int64x1_t) __builtin_neon_vshl_n_v((int8x8_t)__s0, __p1, 3); \
  __ret; \
})
#ifdef __LITTLE_ENDIAN__
#define vshl_n_s16(__p0, __p1) __extension__ ({ \
  int16x4_t __s0 = __p0; \
  int16x4_t __ret; \
  __ret = (int16x4_t) __builtin_neon_vshl_n_v((int8x8_t)__s0, __p1, 1); \
  __ret; \
})
#else
#define vshl_n_s16(__p0, __p1) __extension__ ({ \
  int16x4_t __s0 = __p0; \
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  int16x4_t __ret; \
  __ret = (int16x4_t) __builtin_neon_vshl_n_v((int8x8_t)__rev0, __p1, 1); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vshll_n_u8(__p0, __p1) __extension__ ({ \
  uint8x8_t __s0 = __p0; \
  uint16x8_t __ret; \
  __ret = (uint16x8_t) __builtin_neon_vshll_n_v((int8x8_t)__s0, __p1, 49); \
  __ret; \
})
#else
#define vshll_n_u8(__p0, __p1) __extension__ ({ \
  uint8x8_t __s0 = __p0; \
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint16x8_t __ret; \
  __ret = (uint16x8_t) __builtin_neon_vshll_n_v((int8x8_t)__rev0, __p1, 49); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_vshll_n_u8(__p0, __p1) __extension__ ({ \
  uint8x8_t __s0 = __p0; \
  uint16x8_t __ret; \
  __ret = (uint16x8_t) __builtin_neon_vshll_n_v((int8x8_t)__s0, __p1, 49); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vshll_n_u32(__p0, __p1) __extension__ ({ \
  uint32x2_t __s0 = __p0; \
  uint64x2_t __ret; \
  __ret = (uint64x2_t) __builtin_neon_vshll_n_v((int8x8_t)__s0, __p1, 51); \
  __ret; \
})
#else
#define vshll_n_u32(__p0, __p1) __extension__ ({ \
  uint32x2_t __s0 = __p0; \
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  uint64x2_t __ret; \
  __ret = (uint64x2_t) __builtin_neon_vshll_n_v((int8x8_t)__rev0, __p1, 51); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#define __noswap_vshll_n_u32(__p0, __p1) __extension__ ({ \
  uint32x2_t __s0 = __p0; \
  uint64x2_t __ret; \
  __ret = (uint64x2_t) __builtin_neon_vshll_n_v((int8x8_t)__s0, __p1, 51); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vshll_n_u16(__p0, __p1) __extension__ ({ \
  uint16x4_t __s0 = __p0; \
  uint32x4_t __ret; \
  __ret = (uint32x4_t) __builtin_neon_vshll_n_v((int8x8_t)__s0, __p1, 50); \
  __ret; \
})
#else
#define vshll_n_u16(__p0, __p1) __extension__ ({ \
  uint16x4_t __s0 = __p0; \
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  uint32x4_t __ret; \
  __ret = (uint32x4_t) __builtin_neon_vshll_n_v((int8x8_t)__rev0, __p1, 50); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_vshll_n_u16(__p0, __p1) __extension__ ({ \
  uint16x4_t __s0 = __p0; \
  uint32x4_t __ret; \
  __ret = (uint32x4_t) __builtin_neon_vshll_n_v((int8x8_t)__s0, __p1, 50); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vshll_n_s8(__p0, __p1) __extension__ ({ \
  int8x8_t __s0 = __p0; \
  int16x8_t __ret; \
  __ret = (int16x8_t) __builtin_neon_vshll_n_v((int8x8_t)__s0, __p1, 33); \
  __ret; \
})
#else
#define vshll_n_s8(__p0, __p1) __extension__ ({ \
  int8x8_t __s0 = __p0; \
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  int16x8_t __ret; \
  __ret = (int16x8_t) __builtin_neon_vshll_n_v((int8x8_t)__rev0, __p1, 33); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_vshll_n_s8(__p0, __p1) __extension__ ({ \
  int8x8_t __s0 = __p0; \
  int16x8_t __ret; \
  __ret = (int16x8_t) __builtin_neon_vshll_n_v((int8x8_t)__s0, __p1, 33); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vshll_n_s32(__p0, __p1) __extension__ ({ \
  int32x2_t __s0 = __p0; \
  int64x2_t __ret; \
  __ret = (int64x2_t) __builtin_neon_vshll_n_v((int8x8_t)__s0, __p1, 35); \
  __ret; \
})
#else
#define vshll_n_s32(__p0, __p1) __extension__ ({ \
  int32x2_t __s0 = __p0; \
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  int64x2_t __ret; \
  __ret = (int64x2_t) __builtin_neon_vshll_n_v((int8x8_t)__rev0, __p1, 35); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#define __noswap_vshll_n_s32(__p0, __p1) __extension__ ({ \
  int32x2_t __s0 = __p0; \
  int64x2_t __ret; \
  __ret = (int64x2_t) __builtin_neon_vshll_n_v((int8x8_t)__s0, __p1, 35); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vshll_n_s16(__p0, __p1) __extension__ ({ \
  int16x4_t __s0 = __p0; \
  int32x4_t __ret; \
  __ret = (int32x4_t) __builtin_neon_vshll_n_v((int8x8_t)__s0, __p1, 34); \
  __ret; \
})
#else
#define vshll_n_s16(__p0, __p1) __extension__ ({ \
  int16x4_t __s0 = __p0; \
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  int32x4_t __ret; \
  __ret = (int32x4_t) __builtin_neon_vshll_n_v((int8x8_t)__rev0, __p1, 34); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_vshll_n_s16(__p0, __p1) __extension__ ({ \
  int16x4_t __s0 = __p0; \
  int32x4_t __ret; \
  __ret = (int32x4_t) __builtin_neon_vshll_n_v((int8x8_t)__s0, __p1, 34); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vshrq_n_u8(__p0, __p1) __extension__ ({ \
  uint8x16_t __s0 = __p0; \
  uint8x16_t __ret; \
  __ret = (uint8x16_t) __builtin_neon_vshrq_n_v((int8x16_t)__s0, __p1, 48); \
  __ret; \
})
#else
#define vshrq_n_u8(__p0, __p1) __extension__ ({ \
  uint8x16_t __s0 = __p0; \
  uint8x16_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint8x16_t __ret; \
  __ret = (uint8x16_t) __builtin_neon_vshrq_n_v((int8x16_t)__rev0, __p1, 48); \
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vshrq_n_u32(__p0, __p1) __extension__ ({ \
  uint32x4_t __s0 = __p0; \
  uint32x4_t __ret; \
  __ret = (uint32x4_t) __builtin_neon_vshrq_n_v((int8x16_t)__s0, __p1, 50); \
  __ret; \
})
#else
#define vshrq_n_u32(__p0, __p1) __extension__ ({ \
  uint32x4_t __s0 = __p0; \
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  uint32x4_t __ret; \
  __ret = (uint32x4_t) __builtin_neon_vshrq_n_v((int8x16_t)__rev0, __p1, 50); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vshrq_n_u64(__p0, __p1) __extension__ ({ \
  uint64x2_t __s0 = __p0; \
  uint64x2_t __ret; \
  __ret = (uint64x2_t) __builtin_neon_vshrq_n_v((int8x16_t)__s0, __p1, 51); \
  __ret; \
})
#else
#define vshrq_n_u64(__p0, __p1) __extension__ ({ \
  uint64x2_t __s0 = __p0; \
  uint64x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  uint64x2_t __ret; \
  __ret = (uint64x2_t) __builtin_neon_vshrq_n_v((int8x16_t)__rev0, __p1, 51); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vshrq_n_u16(__p0, __p1) __extension__ ({ \
  uint16x8_t __s0 = __p0; \
  uint16x8_t __ret; \
  __ret = (uint16x8_t) __builtin_neon_vshrq_n_v((int8x16_t)__s0, __p1, 49); \
  __ret; \
})
#else
#define vshrq_n_u16(__p0, __p1) __extension__ ({ \
  uint16x8_t __s0 = __p0; \
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint16x8_t __ret; \
  __ret = (uint16x8_t) __builtin_neon_vshrq_n_v((int8x16_t)__rev0, __p1, 49); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vshrq_n_s8(__p0, __p1) __extension__ ({ \
  int8x16_t __s0 = __p0; \
  int8x16_t __ret; \
  __ret = (int8x16_t) __builtin_neon_vshrq_n_v((int8x16_t)__s0, __p1, 32); \
  __ret; \
})
#else
#define vshrq_n_s8(__p0, __p1) __extension__ ({ \
  int8x16_t __s0 = __p0; \
  int8x16_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  int8x16_t __ret; \
  __ret = (int8x16_t) __builtin_neon_vshrq_n_v((int8x16_t)__rev0, __p1, 32); \
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vshrq_n_s32(__p0, __p1) __extension__ ({ \
  int32x4_t __s0 = __p0; \
  int32x4_t __ret; \
  __ret = (int32x4_t) __builtin_neon_vshrq_n_v((int8x16_t)__s0, __p1, 34); \
  __ret; \
})
#else
#define vshrq_n_s32(__p0, __p1) __extension__ ({ \
  int32x4_t __s0 = __p0; \
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  int32x4_t __ret; \
  __ret = (int32x4_t) __builtin_neon_vshrq_n_v((int8x16_t)__rev0, __p1, 34); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vshrq_n_s64(__p0, __p1) __extension__ ({ \
  int64x2_t __s0 = __p0; \
  int64x2_t __ret; \
  __ret = (int64x2_t) __builtin_neon_vshrq_n_v((int8x16_t)__s0, __p1, 35); \
  __ret; \
})
#else
#define vshrq_n_s64(__p0, __p1) __extension__ ({ \
  int64x2_t __s0 = __p0; \
  int64x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  int64x2_t __ret; \
  __ret = (int64x2_t) __builtin_neon_vshrq_n_v((int8x16_t)__rev0, __p1, 35); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vshrq_n_s16(__p0, __p1) __extension__ ({ \
  int16x8_t __s0 = __p0; \
  int16x8_t __ret; \
  __ret = (int16x8_t) __builtin_neon_vshrq_n_v((int8x16_t)__s0, __p1, 33); \
  __ret; \
})
#else
#define vshrq_n_s16(__p0, __p1) __extension__ ({ \
  int16x8_t __s0 = __p0; \
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  int16x8_t __ret; \
  __ret = (int16x8_t) __builtin_neon_vshrq_n_v((int8x16_t)__rev0, __p1, 33); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vshr_n_u8(__p0, __p1) __extension__ ({ \
  uint8x8_t __s0 = __p0; \
  uint8x8_t __ret; \
  __ret = (uint8x8_t) __builtin_neon_vshr_n_v((int8x8_t)__s0, __p1, 16); \
  __ret; \
})
#else
#define vshr_n_u8(__p0, __p1) __extension__ ({ \
  uint8x8_t __s0 = __p0; \
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint8x8_t __ret; \
  __ret = (uint8x8_t) __builtin_neon_vshr_n_v((int8x8_t)__rev0, __p1, 16); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vshr_n_u32(__p0, __p1) __extension__ ({ \
  uint32x2_t __s0 = __p0; \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_vshr_n_v((int8x8_t)__s0, __p1, 18); \
  __ret; \
})
#else
#define vshr_n_u32(__p0, __p1) __extension__ ({ \
  uint32x2_t __s0 = __p0; \
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_vshr_n_v((int8x8_t)__rev0, __p1, 18); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#define vshr_n_u64(__p0, __p1) __extension__ ({ \
  uint64x1_t __s0 = __p0; \
  uint64x1_t __ret; \
  __ret = (uint64x1_t) __builtin_neon_vshr_n_v((int8x8_t)__s0, __p1, 19); \
  __ret; \
})
#ifdef __LITTLE_ENDIAN__
#define vshr_n_u16(__p0, __p1) __extension__ ({ \
  uint16x4_t __s0 = __p0; \
  uint16x4_t __ret; \
  __ret = (uint16x4_t) __builtin_neon_vshr_n_v((int8x8_t)__s0, __p1, 17); \
  __ret; \
})
#else
#define vshr_n_u16(__p0, __p1) __extension__ ({ \
  uint16x4_t __s0 = __p0; \
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  uint16x4_t __ret; \
  __ret = (uint16x4_t) __builtin_neon_vshr_n_v((int8x8_t)__rev0, __p1, 17); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vshr_n_s8(__p0, __p1) __extension__ ({ \
  int8x8_t __s0 = __p0; \
  int8x8_t __ret; \
  __ret = (int8x8_t) __builtin_neon_vshr_n_v((int8x8_t)__s0, __p1, 0); \
  __ret; \
})
#else
#define vshr_n_s8(__p0, __p1) __extension__ ({ \
  int8x8_t __s0 = __p0; \
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  int8x8_t __ret; \
  __ret = (int8x8_t) __builtin_neon_vshr_n_v((int8x8_t)__rev0, __p1, 0); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vshr_n_s32(__p0, __p1) __extension__ ({ \
  int32x2_t __s0 = __p0; \
  int32x2_t __ret; \
  __ret = (int32x2_t) __builtin_neon_vshr_n_v((int8x8_t)__s0, __p1, 2); \
  __ret; \
})
#else
#define vshr_n_s32(__p0, __p1) __extension__ ({ \
  int32x2_t __s0 = __p0; \
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  int32x2_t __ret; \
  __ret = (int32x2_t) __builtin_neon_vshr_n_v((int8x8_t)__rev0, __p1, 2); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#define vshr_n_s64(__p0, __p1) __extension__ ({ \
  int64x1_t __s0 = __p0; \
  int64x1_t __ret; \
  __ret = (int64x1_t) __builtin_neon_vshr_n_v((int8x8_t)__s0, __p1, 3); \
  __ret; \
})
#ifdef __LITTLE_ENDIAN__
#define vshr_n_s16(__p0, __p1) __extension__ ({ \
  int16x4_t __s0 = __p0; \
  int16x4_t __ret; \
  __ret = (int16x4_t) __builtin_neon_vshr_n_v((int8x8_t)__s0, __p1, 1); \
  __ret; \
})
#else
#define vshr_n_s16(__p0, __p1) __extension__ ({ \
  int16x4_t __s0 = __p0; \
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  int16x4_t __ret; \
  __ret = (int16x4_t) __builtin_neon_vshr_n_v((int8x8_t)__rev0, __p1, 1); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vshrn_n_u32(__p0, __p1) __extension__ ({ \
  uint32x4_t __s0 = __p0; \
  uint16x4_t __ret; \
  __ret = (uint16x4_t) __builtin_neon_vshrn_n_v((int8x16_t)__s0, __p1, 17); \
  __ret; \
})
#else
#define vshrn_n_u32(__p0, __p1) __extension__ ({ \
  uint32x4_t __s0 = __p0; \
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  uint16x4_t __ret; \
  __ret = (uint16x4_t) __builtin_neon_vshrn_n_v((int8x16_t)__rev0, __p1, 17); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_vshrn_n_u32(__p0, __p1) __extension__ ({ \
  uint32x4_t __s0 = __p0; \
  uint16x4_t __ret; \
  __ret = (uint16x4_t) __builtin_neon_vshrn_n_v((int8x16_t)__s0, __p1, 17); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vshrn_n_u64(__p0, __p1) __extension__ ({ \
  uint64x2_t __s0 = __p0; \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_vshrn_n_v((int8x16_t)__s0, __p1, 18); \
  __ret; \
})
#else
#define vshrn_n_u64(__p0, __p1) __extension__ ({ \
  uint64x2_t __s0 = __p0; \
  uint64x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_vshrn_n_v((int8x16_t)__rev0, __p1, 18); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#define __noswap_vshrn_n_u64(__p0, __p1) __extension__ ({ \
  uint64x2_t __s0 = __p0; \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_vshrn_n_v((int8x16_t)__s0, __p1, 18); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vshrn_n_u16(__p0, __p1) __extension__ ({ \
  uint16x8_t __s0 = __p0; \
  uint8x8_t __ret; \
  __ret = (uint8x8_t) __builtin_neon_vshrn_n_v((int8x16_t)__s0, __p1, 16); \
  __ret; \
})
#else
#define vshrn_n_u16(__p0, __p1) __extension__ ({ \
  uint16x8_t __s0 = __p0; \
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint8x8_t __ret; \
  __ret = (uint8x8_t) __builtin_neon_vshrn_n_v((int8x16_t)__rev0, __p1, 16); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_vshrn_n_u16(__p0, __p1) __extension__ ({ \
  uint16x8_t __s0 = __p0; \
  uint8x8_t __ret; \
  __ret = (uint8x8_t) __builtin_neon_vshrn_n_v((int8x16_t)__s0, __p1, 16); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vshrn_n_s32(__p0, __p1) __extension__ ({ \
  int32x4_t __s0 = __p0; \
  int16x4_t __ret; \
  __ret = (int16x4_t) __builtin_neon_vshrn_n_v((int8x16_t)__s0, __p1, 1); \
  __ret; \
})
#else
#define vshrn_n_s32(__p0, __p1) __extension__ ({ \
  int32x4_t __s0 = __p0; \
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  int16x4_t __ret; \
  __ret = (int16x4_t) __builtin_neon_vshrn_n_v((int8x16_t)__rev0, __p1, 1); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_vshrn_n_s32(__p0, __p1) __extension__ ({ \
  int32x4_t __s0 = __p0; \
  int16x4_t __ret; \
  __ret = (int16x4_t) __builtin_neon_vshrn_n_v((int8x16_t)__s0, __p1, 1); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vshrn_n_s64(__p0, __p1) __extension__ ({ \
  int64x2_t __s0 = __p0; \
  int32x2_t __ret; \
  __ret = (int32x2_t) __builtin_neon_vshrn_n_v((int8x16_t)__s0, __p1, 2); \
  __ret; \
})
#else
#define vshrn_n_s64(__p0, __p1) __extension__ ({ \
  int64x2_t __s0 = __p0; \
  int64x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  int32x2_t __ret; \
  __ret = (int32x2_t) __builtin_neon_vshrn_n_v((int8x16_t)__rev0, __p1, 2); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#define __noswap_vshrn_n_s64(__p0, __p1) __extension__ ({ \
  int64x2_t __s0 = __p0; \
  int32x2_t __ret; \
  __ret = (int32x2_t) __builtin_neon_vshrn_n_v((int8x16_t)__s0, __p1, 2); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vshrn_n_s16(__p0, __p1) __extension__ ({ \
  int16x8_t __s0 = __p0; \
  int8x8_t __ret; \
  __ret = (int8x8_t) __builtin_neon_vshrn_n_v((int8x16_t)__s0, __p1, 0); \
  __ret; \
})
#else
#define vshrn_n_s16(__p0, __p1) __extension__ ({ \
  int16x8_t __s0 = __p0; \
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  int8x8_t __ret; \
  __ret = (int8x8_t) __builtin_neon_vshrn_n_v((int8x16_t)__rev0, __p1, 0); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#define __noswap_vshrn_n_s16(__p0, __p1) __extension__ ({ \
  int16x8_t __s0 = __p0; \
  int8x8_t __ret; \
  __ret = (int8x8_t) __builtin_neon_vshrn_n_v((int8x16_t)__s0, __p1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsli_n_p8(__p0, __p1, __p2) __extension__ ({ \
  poly8x8_t __s0 = __p0; \
  poly8x8_t __s1 = __p1; \
  poly8x8_t __ret; \
  __ret = (poly8x8_t) __builtin_neon_vsli_n_v((int8x8_t)__s0, (int8x8_t)__s1, __p2, 4); \
  __ret; \
})
#else
#define vsli_n_p8(__p0, __p1, __p2) __extension__ ({ \
  poly8x8_t __s0 = __p0; \
  poly8x8_t __s1 = __p1; \
  poly8x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  poly8x8_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 7, 6, 5, 4, 3, 2, 1, 0); \
  poly8x8_t __ret; \
  __ret = (poly8x8_t) __builtin_neon_vsli_n_v((int8x8_t)__rev0, (int8x8_t)__rev1, __p2, 4); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsli_n_p16(__p0, __p1, __p2) __extension__ ({ \
  poly16x4_t __s0 = __p0; \
  poly16x4_t __s1 = __p1; \
  poly16x4_t __ret; \
  __ret = (poly16x4_t) __builtin_neon_vsli_n_v((int8x8_t)__s0, (int8x8_t)__s1, __p2, 5); \
  __ret; \
})
#else
#define vsli_n_p16(__p0, __p1, __p2) __extension__ ({ \
  poly16x4_t __s0 = __p0; \
  poly16x4_t __s1 = __p1; \
  poly16x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  poly16x4_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 3, 2, 1, 0); \
  poly16x4_t __ret; \
  __ret = (poly16x4_t) __builtin_neon_vsli_n_v((int8x8_t)__rev0, (int8x8_t)__rev1, __p2, 5); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsliq_n_p8(__p0, __p1, __p2) __extension__ ({ \
  poly8x16_t __s0 = __p0; \
  poly8x16_t __s1 = __p1; \
  poly8x16_t __ret; \
  __ret = (poly8x16_t) __builtin_neon_vsliq_n_v((int8x16_t)__s0, (int8x16_t)__s1, __p2, 36); \
  __ret; \
})
#else
#define vsliq_n_p8(__p0, __p1, __p2) __extension__ ({ \
  poly8x16_t __s0 = __p0; \
  poly8x16_t __s1 = __p1; \
  poly8x16_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  poly8x16_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  poly8x16_t __ret; \
  __ret = (poly8x16_t) __builtin_neon_vsliq_n_v((int8x16_t)__rev0, (int8x16_t)__rev1, __p2, 36); \
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsliq_n_p16(__p0, __p1, __p2) __extension__ ({ \
  poly16x8_t __s0 = __p0; \
  poly16x8_t __s1 = __p1; \
  poly16x8_t __ret; \
  __ret = (poly16x8_t) __builtin_neon_vsliq_n_v((int8x16_t)__s0, (int8x16_t)__s1, __p2, 37); \
  __ret; \
})
#else
#define vsliq_n_p16(__p0, __p1, __p2) __extension__ ({ \
  poly16x8_t __s0 = __p0; \
  poly16x8_t __s1 = __p1; \
  poly16x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  poly16x8_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 7, 6, 5, 4, 3, 2, 1, 0); \
  poly16x8_t __ret; \
  __ret = (poly16x8_t) __builtin_neon_vsliq_n_v((int8x16_t)__rev0, (int8x16_t)__rev1, __p2, 37); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsliq_n_u8(__p0, __p1, __p2) __extension__ ({ \
  uint8x16_t __s0 = __p0; \
  uint8x16_t __s1 = __p1; \
  uint8x16_t __ret; \
  __ret = (uint8x16_t) __builtin_neon_vsliq_n_v((int8x16_t)__s0, (int8x16_t)__s1, __p2, 48); \
  __ret; \
})
#else
#define vsliq_n_u8(__p0, __p1, __p2) __extension__ ({ \
  uint8x16_t __s0 = __p0; \
  uint8x16_t __s1 = __p1; \
  uint8x16_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint8x16_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint8x16_t __ret; \
  __ret = (uint8x16_t) __builtin_neon_vsliq_n_v((int8x16_t)__rev0, (int8x16_t)__rev1, __p2, 48); \
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsliq_n_u32(__p0, __p1, __p2) __extension__ ({ \
  uint32x4_t __s0 = __p0; \
  uint32x4_t __s1 = __p1; \
  uint32x4_t __ret; \
  __ret = (uint32x4_t) __builtin_neon_vsliq_n_v((int8x16_t)__s0, (int8x16_t)__s1, __p2, 50); \
  __ret; \
})
#else
#define vsliq_n_u32(__p0, __p1, __p2) __extension__ ({ \
  uint32x4_t __s0 = __p0; \
  uint32x4_t __s1 = __p1; \
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  uint32x4_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 3, 2, 1, 0); \
  uint32x4_t __ret; \
  __ret = (uint32x4_t) __builtin_neon_vsliq_n_v((int8x16_t)__rev0, (int8x16_t)__rev1, __p2, 50); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsliq_n_u64(__p0, __p1, __p2) __extension__ ({ \
  uint64x2_t __s0 = __p0; \
  uint64x2_t __s1 = __p1; \
  uint64x2_t __ret; \
  __ret = (uint64x2_t) __builtin_neon_vsliq_n_v((int8x16_t)__s0, (int8x16_t)__s1, __p2, 51); \
  __ret; \
})
#else
#define vsliq_n_u64(__p0, __p1, __p2) __extension__ ({ \
  uint64x2_t __s0 = __p0; \
  uint64x2_t __s1 = __p1; \
  uint64x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  uint64x2_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 1, 0); \
  uint64x2_t __ret; \
  __ret = (uint64x2_t) __builtin_neon_vsliq_n_v((int8x16_t)__rev0, (int8x16_t)__rev1, __p2, 51); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsliq_n_u16(__p0, __p1, __p2) __extension__ ({ \
  uint16x8_t __s0 = __p0; \
  uint16x8_t __s1 = __p1; \
  uint16x8_t __ret; \
  __ret = (uint16x8_t) __builtin_neon_vsliq_n_v((int8x16_t)__s0, (int8x16_t)__s1, __p2, 49); \
  __ret; \
})
#else
#define vsliq_n_u16(__p0, __p1, __p2) __extension__ ({ \
  uint16x8_t __s0 = __p0; \
  uint16x8_t __s1 = __p1; \
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint16x8_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint16x8_t __ret; \
  __ret = (uint16x8_t) __builtin_neon_vsliq_n_v((int8x16_t)__rev0, (int8x16_t)__rev1, __p2, 49); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsliq_n_s8(__p0, __p1, __p2) __extension__ ({ \
  int8x16_t __s0 = __p0; \
  int8x16_t __s1 = __p1; \
  int8x16_t __ret; \
  __ret = (int8x16_t) __builtin_neon_vsliq_n_v((int8x16_t)__s0, (int8x16_t)__s1, __p2, 32); \
  __ret; \
})
#else
#define vsliq_n_s8(__p0, __p1, __p2) __extension__ ({ \
  int8x16_t __s0 = __p0; \
  int8x16_t __s1 = __p1; \
  int8x16_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  int8x16_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  int8x16_t __ret; \
  __ret = (int8x16_t) __builtin_neon_vsliq_n_v((int8x16_t)__rev0, (int8x16_t)__rev1, __p2, 32); \
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsliq_n_s32(__p0, __p1, __p2) __extension__ ({ \
  int32x4_t __s0 = __p0; \
  int32x4_t __s1 = __p1; \
  int32x4_t __ret; \
  __ret = (int32x4_t) __builtin_neon_vsliq_n_v((int8x16_t)__s0, (int8x16_t)__s1, __p2, 34); \
  __ret; \
})
#else
#define vsliq_n_s32(__p0, __p1, __p2) __extension__ ({ \
  int32x4_t __s0 = __p0; \
  int32x4_t __s1 = __p1; \
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  int32x4_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 3, 2, 1, 0); \
  int32x4_t __ret; \
  __ret = (int32x4_t) __builtin_neon_vsliq_n_v((int8x16_t)__rev0, (int8x16_t)__rev1, __p2, 34); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsliq_n_s64(__p0, __p1, __p2) __extension__ ({ \
  int64x2_t __s0 = __p0; \
  int64x2_t __s1 = __p1; \
  int64x2_t __ret; \
  __ret = (int64x2_t) __builtin_neon_vsliq_n_v((int8x16_t)__s0, (int8x16_t)__s1, __p2, 35); \
  __ret; \
})
#else
#define vsliq_n_s64(__p0, __p1, __p2) __extension__ ({ \
  int64x2_t __s0 = __p0; \
  int64x2_t __s1 = __p1; \
  int64x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  int64x2_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 1, 0); \
  int64x2_t __ret; \
  __ret = (int64x2_t) __builtin_neon_vsliq_n_v((int8x16_t)__rev0, (int8x16_t)__rev1, __p2, 35); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsliq_n_s16(__p0, __p1, __p2) __extension__ ({ \
  int16x8_t __s0 = __p0; \
  int16x8_t __s1 = __p1; \
  int16x8_t __ret; \
  __ret = (int16x8_t) __builtin_neon_vsliq_n_v((int8x16_t)__s0, (int8x16_t)__s1, __p2, 33); \
  __ret; \
})
#else
#define vsliq_n_s16(__p0, __p1, __p2) __extension__ ({ \
  int16x8_t __s0 = __p0; \
  int16x8_t __s1 = __p1; \
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  int16x8_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 7, 6, 5, 4, 3, 2, 1, 0); \
  int16x8_t __ret; \
  __ret = (int16x8_t) __builtin_neon_vsliq_n_v((int8x16_t)__rev0, (int8x16_t)__rev1, __p2, 33); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsli_n_u8(__p0, __p1, __p2) __extension__ ({ \
  uint8x8_t __s0 = __p0; \
  uint8x8_t __s1 = __p1; \
  uint8x8_t __ret; \
  __ret = (uint8x8_t) __builtin_neon_vsli_n_v((int8x8_t)__s0, (int8x8_t)__s1, __p2, 16); \
  __ret; \
})
#else
#define vsli_n_u8(__p0, __p1, __p2) __extension__ ({ \
  uint8x8_t __s0 = __p0; \
  uint8x8_t __s1 = __p1; \
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint8x8_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint8x8_t __ret; \
  __ret = (uint8x8_t) __builtin_neon_vsli_n_v((int8x8_t)__rev0, (int8x8_t)__rev1, __p2, 16); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsli_n_u32(__p0, __p1, __p2) __extension__ ({ \
  uint32x2_t __s0 = __p0; \
  uint32x2_t __s1 = __p1; \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_vsli_n_v((int8x8_t)__s0, (int8x8_t)__s1, __p2, 18); \
  __ret; \
})
#else
#define vsli_n_u32(__p0, __p1, __p2) __extension__ ({ \
  uint32x2_t __s0 = __p0; \
  uint32x2_t __s1 = __p1; \
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  uint32x2_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 1, 0); \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_vsli_n_v((int8x8_t)__rev0, (int8x8_t)__rev1, __p2, 18); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#define vsli_n_u64(__p0, __p1, __p2) __extension__ ({ \
  uint64x1_t __s0 = __p0; \
  uint64x1_t __s1 = __p1; \
  uint64x1_t __ret; \
  __ret = (uint64x1_t) __builtin_neon_vsli_n_v((int8x8_t)__s0, (int8x8_t)__s1, __p2, 19); \
  __ret; \
})
#ifdef __LITTLE_ENDIAN__
#define vsli_n_u16(__p0, __p1, __p2) __extension__ ({ \
  uint16x4_t __s0 = __p0; \
  uint16x4_t __s1 = __p1; \
  uint16x4_t __ret; \
  __ret = (uint16x4_t) __builtin_neon_vsli_n_v((int8x8_t)__s0, (int8x8_t)__s1, __p2, 17); \
  __ret; \
})
#else
#define vsli_n_u16(__p0, __p1, __p2) __extension__ ({ \
  uint16x4_t __s0 = __p0; \
  uint16x4_t __s1 = __p1; \
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  uint16x4_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 3, 2, 1, 0); \
  uint16x4_t __ret; \
  __ret = (uint16x4_t) __builtin_neon_vsli_n_v((int8x8_t)__rev0, (int8x8_t)__rev1, __p2, 17); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsli_n_s8(__p0, __p1, __p2) __extension__ ({ \
  int8x8_t __s0 = __p0; \
  int8x8_t __s1 = __p1; \
  int8x8_t __ret; \
  __ret = (int8x8_t) __builtin_neon_vsli_n_v((int8x8_t)__s0, (int8x8_t)__s1, __p2, 0); \
  __ret; \
})
#else
#define vsli_n_s8(__p0, __p1, __p2) __extension__ ({ \
  int8x8_t __s0 = __p0; \
  int8x8_t __s1 = __p1; \
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  int8x8_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 7, 6, 5, 4, 3, 2, 1, 0); \
  int8x8_t __ret; \
  __ret = (int8x8_t) __builtin_neon_vsli_n_v((int8x8_t)__rev0, (int8x8_t)__rev1, __p2, 0); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsli_n_s32(__p0, __p1, __p2) __extension__ ({ \
  int32x2_t __s0 = __p0; \
  int32x2_t __s1 = __p1; \
  int32x2_t __ret; \
  __ret = (int32x2_t) __builtin_neon_vsli_n_v((int8x8_t)__s0, (int8x8_t)__s1, __p2, 2); \
  __ret; \
})
#else
#define vsli_n_s32(__p0, __p1, __p2) __extension__ ({ \
  int32x2_t __s0 = __p0; \
  int32x2_t __s1 = __p1; \
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 1, 0); \
  int32x2_t __ret; \
  __ret = (int32x2_t) __builtin_neon_vsli_n_v((int8x8_t)__rev0, (int8x8_t)__rev1, __p2, 2); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#define vsli_n_s64(__p0, __p1, __p2) __extension__ ({ \
  int64x1_t __s0 = __p0; \
  int64x1_t __s1 = __p1; \
  int64x1_t __ret; \
  __ret = (int64x1_t) __builtin_neon_vsli_n_v((int8x8_t)__s0, (int8x8_t)__s1, __p2, 3); \
  __ret; \
})
#ifdef __LITTLE_ENDIAN__
#define vsli_n_s16(__p0, __p1, __p2) __extension__ ({ \
  int16x4_t __s0 = __p0; \
  int16x4_t __s1 = __p1; \
  int16x4_t __ret; \
  __ret = (int16x4_t) __builtin_neon_vsli_n_v((int8x8_t)__s0, (int8x8_t)__s1, __p2, 1); \
  __ret; \
})
#else
#define vsli_n_s16(__p0, __p1, __p2) __extension__ ({ \
  int16x4_t __s0 = __p0; \
  int16x4_t __s1 = __p1; \
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 3, 2, 1, 0); \
  int16x4_t __ret; \
  __ret = (int16x4_t) __builtin_neon_vsli_n_v((int8x8_t)__rev0, (int8x8_t)__rev1, __p2, 1); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsraq_n_u8(__p0, __p1, __p2) __extension__ ({ \
  uint8x16_t __s0 = __p0; \
  uint8x16_t __s1 = __p1; \
  uint8x16_t __ret; \
  __ret = (uint8x16_t) __builtin_neon_vsraq_n_v((int8x16_t)__s0, (int8x16_t)__s1, __p2, 48); \
  __ret; \
})
#else
#define vsraq_n_u8(__p0, __p1, __p2) __extension__ ({ \
  uint8x16_t __s0 = __p0; \
  uint8x16_t __s1 = __p1; \
  uint8x16_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint8x16_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint8x16_t __ret; \
  __ret = (uint8x16_t) __builtin_neon_vsraq_n_v((int8x16_t)__rev0, (int8x16_t)__rev1, __p2, 48); \
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsraq_n_u32(__p0, __p1, __p2) __extension__ ({ \
  uint32x4_t __s0 = __p0; \
  uint32x4_t __s1 = __p1; \
  uint32x4_t __ret; \
  __ret = (uint32x4_t) __builtin_neon_vsraq_n_v((int8x16_t)__s0, (int8x16_t)__s1, __p2, 50); \
  __ret; \
})
#else
#define vsraq_n_u32(__p0, __p1, __p2) __extension__ ({ \
  uint32x4_t __s0 = __p0; \
  uint32x4_t __s1 = __p1; \
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  uint32x4_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 3, 2, 1, 0); \
  uint32x4_t __ret; \
  __ret = (uint32x4_t) __builtin_neon_vsraq_n_v((int8x16_t)__rev0, (int8x16_t)__rev1, __p2, 50); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsraq_n_u64(__p0, __p1, __p2) __extension__ ({ \
  uint64x2_t __s0 = __p0; \
  uint64x2_t __s1 = __p1; \
  uint64x2_t __ret; \
  __ret = (uint64x2_t) __builtin_neon_vsraq_n_v((int8x16_t)__s0, (int8x16_t)__s1, __p2, 51); \
  __ret; \
})
#else
#define vsraq_n_u64(__p0, __p1, __p2) __extension__ ({ \
  uint64x2_t __s0 = __p0; \
  uint64x2_t __s1 = __p1; \
  uint64x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  uint64x2_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 1, 0); \
  uint64x2_t __ret; \
  __ret = (uint64x2_t) __builtin_neon_vsraq_n_v((int8x16_t)__rev0, (int8x16_t)__rev1, __p2, 51); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsraq_n_u16(__p0, __p1, __p2) __extension__ ({ \
  uint16x8_t __s0 = __p0; \
  uint16x8_t __s1 = __p1; \
  uint16x8_t __ret; \
  __ret = (uint16x8_t) __builtin_neon_vsraq_n_v((int8x16_t)__s0, (int8x16_t)__s1, __p2, 49); \
  __ret; \
})
#else
#define vsraq_n_u16(__p0, __p1, __p2) __extension__ ({ \
  uint16x8_t __s0 = __p0; \
  uint16x8_t __s1 = __p1; \
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint16x8_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint16x8_t __ret; \
  __ret = (uint16x8_t) __builtin_neon_vsraq_n_v((int8x16_t)__rev0, (int8x16_t)__rev1, __p2, 49); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsraq_n_s8(__p0, __p1, __p2) __extension__ ({ \
  int8x16_t __s0 = __p0; \
  int8x16_t __s1 = __p1; \
  int8x16_t __ret; \
  __ret = (int8x16_t) __builtin_neon_vsraq_n_v((int8x16_t)__s0, (int8x16_t)__s1, __p2, 32); \
  __ret; \
})
#else
#define vsraq_n_s8(__p0, __p1, __p2) __extension__ ({ \
  int8x16_t __s0 = __p0; \
  int8x16_t __s1 = __p1; \
  int8x16_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  int8x16_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  int8x16_t __ret; \
  __ret = (int8x16_t) __builtin_neon_vsraq_n_v((int8x16_t)__rev0, (int8x16_t)__rev1, __p2, 32); \
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsraq_n_s32(__p0, __p1, __p2) __extension__ ({ \
  int32x4_t __s0 = __p0; \
  int32x4_t __s1 = __p1; \
  int32x4_t __ret; \
  __ret = (int32x4_t) __builtin_neon_vsraq_n_v((int8x16_t)__s0, (int8x16_t)__s1, __p2, 34); \
  __ret; \
})
#else
#define vsraq_n_s32(__p0, __p1, __p2) __extension__ ({ \
  int32x4_t __s0 = __p0; \
  int32x4_t __s1 = __p1; \
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  int32x4_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 3, 2, 1, 0); \
  int32x4_t __ret; \
  __ret = (int32x4_t) __builtin_neon_vsraq_n_v((int8x16_t)__rev0, (int8x16_t)__rev1, __p2, 34); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsraq_n_s64(__p0, __p1, __p2) __extension__ ({ \
  int64x2_t __s0 = __p0; \
  int64x2_t __s1 = __p1; \
  int64x2_t __ret; \
  __ret = (int64x2_t) __builtin_neon_vsraq_n_v((int8x16_t)__s0, (int8x16_t)__s1, __p2, 35); \
  __ret; \
})
#else
#define vsraq_n_s64(__p0, __p1, __p2) __extension__ ({ \
  int64x2_t __s0 = __p0; \
  int64x2_t __s1 = __p1; \
  int64x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  int64x2_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 1, 0); \
  int64x2_t __ret; \
  __ret = (int64x2_t) __builtin_neon_vsraq_n_v((int8x16_t)__rev0, (int8x16_t)__rev1, __p2, 35); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsraq_n_s16(__p0, __p1, __p2) __extension__ ({ \
  int16x8_t __s0 = __p0; \
  int16x8_t __s1 = __p1; \
  int16x8_t __ret; \
  __ret = (int16x8_t) __builtin_neon_vsraq_n_v((int8x16_t)__s0, (int8x16_t)__s1, __p2, 33); \
  __ret; \
})
#else
#define vsraq_n_s16(__p0, __p1, __p2) __extension__ ({ \
  int16x8_t __s0 = __p0; \
  int16x8_t __s1 = __p1; \
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  int16x8_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 7, 6, 5, 4, 3, 2, 1, 0); \
  int16x8_t __ret; \
  __ret = (int16x8_t) __builtin_neon_vsraq_n_v((int8x16_t)__rev0, (int8x16_t)__rev1, __p2, 33); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsra_n_u8(__p0, __p1, __p2) __extension__ ({ \
  uint8x8_t __s0 = __p0; \
  uint8x8_t __s1 = __p1; \
  uint8x8_t __ret; \
  __ret = (uint8x8_t) __builtin_neon_vsra_n_v((int8x8_t)__s0, (int8x8_t)__s1, __p2, 16); \
  __ret; \
})
#else
#define vsra_n_u8(__p0, __p1, __p2) __extension__ ({ \
  uint8x8_t __s0 = __p0; \
  uint8x8_t __s1 = __p1; \
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint8x8_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint8x8_t __ret; \
  __ret = (uint8x8_t) __builtin_neon_vsra_n_v((int8x8_t)__rev0, (int8x8_t)__rev1, __p2, 16); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsra_n_u32(__p0, __p1, __p2) __extension__ ({ \
  uint32x2_t __s0 = __p0; \
  uint32x2_t __s1 = __p1; \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_vsra_n_v((int8x8_t)__s0, (int8x8_t)__s1, __p2, 18); \
  __ret; \
})
#else
#define vsra_n_u32(__p0, __p1, __p2) __extension__ ({ \
  uint32x2_t __s0 = __p0; \
  uint32x2_t __s1 = __p1; \
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  uint32x2_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 1, 0); \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_vsra_n_v((int8x8_t)__rev0, (int8x8_t)__rev1, __p2, 18); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#define vsra_n_u64(__p0, __p1, __p2) __extension__ ({ \
  uint64x1_t __s0 = __p0; \
  uint64x1_t __s1 = __p1; \
  uint64x1_t __ret; \
  __ret = (uint64x1_t) __builtin_neon_vsra_n_v((int8x8_t)__s0, (int8x8_t)__s1, __p2, 19); \
  __ret; \
})
#ifdef __LITTLE_ENDIAN__
#define vsra_n_u16(__p0, __p1, __p2) __extension__ ({ \
  uint16x4_t __s0 = __p0; \
  uint16x4_t __s1 = __p1; \
  uint16x4_t __ret; \
  __ret = (uint16x4_t) __builtin_neon_vsra_n_v((int8x8_t)__s0, (int8x8_t)__s1, __p2, 17); \
  __ret; \
})
#else
#define vsra_n_u16(__p0, __p1, __p2) __extension__ ({ \
  uint16x4_t __s0 = __p0; \
  uint16x4_t __s1 = __p1; \
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  uint16x4_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 3, 2, 1, 0); \
  uint16x4_t __ret; \
  __ret = (uint16x4_t) __builtin_neon_vsra_n_v((int8x8_t)__rev0, (int8x8_t)__rev1, __p2, 17); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsra_n_s8(__p0, __p1, __p2) __extension__ ({ \
  int8x8_t __s0 = __p0; \
  int8x8_t __s1 = __p1; \
  int8x8_t __ret; \
  __ret = (int8x8_t) __builtin_neon_vsra_n_v((int8x8_t)__s0, (int8x8_t)__s1, __p2, 0); \
  __ret; \
})
#else
#define vsra_n_s8(__p0, __p1, __p2) __extension__ ({ \
  int8x8_t __s0 = __p0; \
  int8x8_t __s1 = __p1; \
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  int8x8_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 7, 6, 5, 4, 3, 2, 1, 0); \
  int8x8_t __ret; \
  __ret = (int8x8_t) __builtin_neon_vsra_n_v((int8x8_t)__rev0, (int8x8_t)__rev1, __p2, 0); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsra_n_s32(__p0, __p1, __p2) __extension__ ({ \
  int32x2_t __s0 = __p0; \
  int32x2_t __s1 = __p1; \
  int32x2_t __ret; \
  __ret = (int32x2_t) __builtin_neon_vsra_n_v((int8x8_t)__s0, (int8x8_t)__s1, __p2, 2); \
  __ret; \
})
#else
#define vsra_n_s32(__p0, __p1, __p2) __extension__ ({ \
  int32x2_t __s0 = __p0; \
  int32x2_t __s1 = __p1; \
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 1, 0); \
  int32x2_t __ret; \
  __ret = (int32x2_t) __builtin_neon_vsra_n_v((int8x8_t)__rev0, (int8x8_t)__rev1, __p2, 2); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#define vsra_n_s64(__p0, __p1, __p2) __extension__ ({ \
  int64x1_t __s0 = __p0; \
  int64x1_t __s1 = __p1; \
  int64x1_t __ret; \
  __ret = (int64x1_t) __builtin_neon_vsra_n_v((int8x8_t)__s0, (int8x8_t)__s1, __p2, 3); \
  __ret; \
})
#ifdef __LITTLE_ENDIAN__
#define vsra_n_s16(__p0, __p1, __p2) __extension__ ({ \
  int16x4_t __s0 = __p0; \
  int16x4_t __s1 = __p1; \
  int16x4_t __ret; \
  __ret = (int16x4_t) __builtin_neon_vsra_n_v((int8x8_t)__s0, (int8x8_t)__s1, __p2, 1); \
  __ret; \
})
#else
#define vsra_n_s16(__p0, __p1, __p2) __extension__ ({ \
  int16x4_t __s0 = __p0; \
  int16x4_t __s1 = __p1; \
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 3, 2, 1, 0); \
  int16x4_t __ret; \
  __ret = (int16x4_t) __builtin_neon_vsra_n_v((int8x8_t)__rev0, (int8x8_t)__rev1, __p2, 1); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsri_n_p8(__p0, __p1, __p2) __extension__ ({ \
  poly8x8_t __s0 = __p0; \
  poly8x8_t __s1 = __p1; \
  poly8x8_t __ret; \
  __ret = (poly8x8_t) __builtin_neon_vsri_n_v((int8x8_t)__s0, (int8x8_t)__s1, __p2, 4); \
  __ret; \
})
#else
#define vsri_n_p8(__p0, __p1, __p2) __extension__ ({ \
  poly8x8_t __s0 = __p0; \
  poly8x8_t __s1 = __p1; \
  poly8x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  poly8x8_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 7, 6, 5, 4, 3, 2, 1, 0); \
  poly8x8_t __ret; \
  __ret = (poly8x8_t) __builtin_neon_vsri_n_v((int8x8_t)__rev0, (int8x8_t)__rev1, __p2, 4); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsri_n_p16(__p0, __p1, __p2) __extension__ ({ \
  poly16x4_t __s0 = __p0; \
  poly16x4_t __s1 = __p1; \
  poly16x4_t __ret; \
  __ret = (poly16x4_t) __builtin_neon_vsri_n_v((int8x8_t)__s0, (int8x8_t)__s1, __p2, 5); \
  __ret; \
})
#else
#define vsri_n_p16(__p0, __p1, __p2) __extension__ ({ \
  poly16x4_t __s0 = __p0; \
  poly16x4_t __s1 = __p1; \
  poly16x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  poly16x4_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 3, 2, 1, 0); \
  poly16x4_t __ret; \
  __ret = (poly16x4_t) __builtin_neon_vsri_n_v((int8x8_t)__rev0, (int8x8_t)__rev1, __p2, 5); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsriq_n_p8(__p0, __p1, __p2) __extension__ ({ \
  poly8x16_t __s0 = __p0; \
  poly8x16_t __s1 = __p1; \
  poly8x16_t __ret; \
  __ret = (poly8x16_t) __builtin_neon_vsriq_n_v((int8x16_t)__s0, (int8x16_t)__s1, __p2, 36); \
  __ret; \
})
#else
#define vsriq_n_p8(__p0, __p1, __p2) __extension__ ({ \
  poly8x16_t __s0 = __p0; \
  poly8x16_t __s1 = __p1; \
  poly8x16_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  poly8x16_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  poly8x16_t __ret; \
  __ret = (poly8x16_t) __builtin_neon_vsriq_n_v((int8x16_t)__rev0, (int8x16_t)__rev1, __p2, 36); \
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsriq_n_p16(__p0, __p1, __p2) __extension__ ({ \
  poly16x8_t __s0 = __p0; \
  poly16x8_t __s1 = __p1; \
  poly16x8_t __ret; \
  __ret = (poly16x8_t) __builtin_neon_vsriq_n_v((int8x16_t)__s0, (int8x16_t)__s1, __p2, 37); \
  __ret; \
})
#else
#define vsriq_n_p16(__p0, __p1, __p2) __extension__ ({ \
  poly16x8_t __s0 = __p0; \
  poly16x8_t __s1 = __p1; \
  poly16x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  poly16x8_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 7, 6, 5, 4, 3, 2, 1, 0); \
  poly16x8_t __ret; \
  __ret = (poly16x8_t) __builtin_neon_vsriq_n_v((int8x16_t)__rev0, (int8x16_t)__rev1, __p2, 37); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsriq_n_u8(__p0, __p1, __p2) __extension__ ({ \
  uint8x16_t __s0 = __p0; \
  uint8x16_t __s1 = __p1; \
  uint8x16_t __ret; \
  __ret = (uint8x16_t) __builtin_neon_vsriq_n_v((int8x16_t)__s0, (int8x16_t)__s1, __p2, 48); \
  __ret; \
})
#else
#define vsriq_n_u8(__p0, __p1, __p2) __extension__ ({ \
  uint8x16_t __s0 = __p0; \
  uint8x16_t __s1 = __p1; \
  uint8x16_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint8x16_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint8x16_t __ret; \
  __ret = (uint8x16_t) __builtin_neon_vsriq_n_v((int8x16_t)__rev0, (int8x16_t)__rev1, __p2, 48); \
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsriq_n_u32(__p0, __p1, __p2) __extension__ ({ \
  uint32x4_t __s0 = __p0; \
  uint32x4_t __s1 = __p1; \
  uint32x4_t __ret; \
  __ret = (uint32x4_t) __builtin_neon_vsriq_n_v((int8x16_t)__s0, (int8x16_t)__s1, __p2, 50); \
  __ret; \
})
#else
#define vsriq_n_u32(__p0, __p1, __p2) __extension__ ({ \
  uint32x4_t __s0 = __p0; \
  uint32x4_t __s1 = __p1; \
  uint32x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  uint32x4_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 3, 2, 1, 0); \
  uint32x4_t __ret; \
  __ret = (uint32x4_t) __builtin_neon_vsriq_n_v((int8x16_t)__rev0, (int8x16_t)__rev1, __p2, 50); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsriq_n_u64(__p0, __p1, __p2) __extension__ ({ \
  uint64x2_t __s0 = __p0; \
  uint64x2_t __s1 = __p1; \
  uint64x2_t __ret; \
  __ret = (uint64x2_t) __builtin_neon_vsriq_n_v((int8x16_t)__s0, (int8x16_t)__s1, __p2, 51); \
  __ret; \
})
#else
#define vsriq_n_u64(__p0, __p1, __p2) __extension__ ({ \
  uint64x2_t __s0 = __p0; \
  uint64x2_t __s1 = __p1; \
  uint64x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  uint64x2_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 1, 0); \
  uint64x2_t __ret; \
  __ret = (uint64x2_t) __builtin_neon_vsriq_n_v((int8x16_t)__rev0, (int8x16_t)__rev1, __p2, 51); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsriq_n_u16(__p0, __p1, __p2) __extension__ ({ \
  uint16x8_t __s0 = __p0; \
  uint16x8_t __s1 = __p1; \
  uint16x8_t __ret; \
  __ret = (uint16x8_t) __builtin_neon_vsriq_n_v((int8x16_t)__s0, (int8x16_t)__s1, __p2, 49); \
  __ret; \
})
#else
#define vsriq_n_u16(__p0, __p1, __p2) __extension__ ({ \
  uint16x8_t __s0 = __p0; \
  uint16x8_t __s1 = __p1; \
  uint16x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint16x8_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint16x8_t __ret; \
  __ret = (uint16x8_t) __builtin_neon_vsriq_n_v((int8x16_t)__rev0, (int8x16_t)__rev1, __p2, 49); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsriq_n_s8(__p0, __p1, __p2) __extension__ ({ \
  int8x16_t __s0 = __p0; \
  int8x16_t __s1 = __p1; \
  int8x16_t __ret; \
  __ret = (int8x16_t) __builtin_neon_vsriq_n_v((int8x16_t)__s0, (int8x16_t)__s1, __p2, 32); \
  __ret; \
})
#else
#define vsriq_n_s8(__p0, __p1, __p2) __extension__ ({ \
  int8x16_t __s0 = __p0; \
  int8x16_t __s1 = __p1; \
  int8x16_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  int8x16_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  int8x16_t __ret; \
  __ret = (int8x16_t) __builtin_neon_vsriq_n_v((int8x16_t)__rev0, (int8x16_t)__rev1, __p2, 32); \
  __ret = __builtin_shufflevector(__ret, __ret, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsriq_n_s32(__p0, __p1, __p2) __extension__ ({ \
  int32x4_t __s0 = __p0; \
  int32x4_t __s1 = __p1; \
  int32x4_t __ret; \
  __ret = (int32x4_t) __builtin_neon_vsriq_n_v((int8x16_t)__s0, (int8x16_t)__s1, __p2, 34); \
  __ret; \
})
#else
#define vsriq_n_s32(__p0, __p1, __p2) __extension__ ({ \
  int32x4_t __s0 = __p0; \
  int32x4_t __s1 = __p1; \
  int32x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  int32x4_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 3, 2, 1, 0); \
  int32x4_t __ret; \
  __ret = (int32x4_t) __builtin_neon_vsriq_n_v((int8x16_t)__rev0, (int8x16_t)__rev1, __p2, 34); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsriq_n_s64(__p0, __p1, __p2) __extension__ ({ \
  int64x2_t __s0 = __p0; \
  int64x2_t __s1 = __p1; \
  int64x2_t __ret; \
  __ret = (int64x2_t) __builtin_neon_vsriq_n_v((int8x16_t)__s0, (int8x16_t)__s1, __p2, 35); \
  __ret; \
})
#else
#define vsriq_n_s64(__p0, __p1, __p2) __extension__ ({ \
  int64x2_t __s0 = __p0; \
  int64x2_t __s1 = __p1; \
  int64x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  int64x2_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 1, 0); \
  int64x2_t __ret; \
  __ret = (int64x2_t) __builtin_neon_vsriq_n_v((int8x16_t)__rev0, (int8x16_t)__rev1, __p2, 35); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsriq_n_s16(__p0, __p1, __p2) __extension__ ({ \
  int16x8_t __s0 = __p0; \
  int16x8_t __s1 = __p1; \
  int16x8_t __ret; \
  __ret = (int16x8_t) __builtin_neon_vsriq_n_v((int8x16_t)__s0, (int8x16_t)__s1, __p2, 33); \
  __ret; \
})
#else
#define vsriq_n_s16(__p0, __p1, __p2) __extension__ ({ \
  int16x8_t __s0 = __p0; \
  int16x8_t __s1 = __p1; \
  int16x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  int16x8_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 7, 6, 5, 4, 3, 2, 1, 0); \
  int16x8_t __ret; \
  __ret = (int16x8_t) __builtin_neon_vsriq_n_v((int8x16_t)__rev0, (int8x16_t)__rev1, __p2, 33); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsri_n_u8(__p0, __p1, __p2) __extension__ ({ \
  uint8x8_t __s0 = __p0; \
  uint8x8_t __s1 = __p1; \
  uint8x8_t __ret; \
  __ret = (uint8x8_t) __builtin_neon_vsri_n_v((int8x8_t)__s0, (int8x8_t)__s1, __p2, 16); \
  __ret; \
})
#else
#define vsri_n_u8(__p0, __p1, __p2) __extension__ ({ \
  uint8x8_t __s0 = __p0; \
  uint8x8_t __s1 = __p1; \
  uint8x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint8x8_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 7, 6, 5, 4, 3, 2, 1, 0); \
  uint8x8_t __ret; \
  __ret = (uint8x8_t) __builtin_neon_vsri_n_v((int8x8_t)__rev0, (int8x8_t)__rev1, __p2, 16); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsri_n_u32(__p0, __p1, __p2) __extension__ ({ \
  uint32x2_t __s0 = __p0; \
  uint32x2_t __s1 = __p1; \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_vsri_n_v((int8x8_t)__s0, (int8x8_t)__s1, __p2, 18); \
  __ret; \
})
#else
#define vsri_n_u32(__p0, __p1, __p2) __extension__ ({ \
  uint32x2_t __s0 = __p0; \
  uint32x2_t __s1 = __p1; \
  uint32x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  uint32x2_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 1, 0); \
  uint32x2_t __ret; \
  __ret = (uint32x2_t) __builtin_neon_vsri_n_v((int8x8_t)__rev0, (int8x8_t)__rev1, __p2, 18); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#define vsri_n_u64(__p0, __p1, __p2) __extension__ ({ \
  uint64x1_t __s0 = __p0; \
  uint64x1_t __s1 = __p1; \
  uint64x1_t __ret; \
  __ret = (uint64x1_t) __builtin_neon_vsri_n_v((int8x8_t)__s0, (int8x8_t)__s1, __p2, 19); \
  __ret; \
})
#ifdef __LITTLE_ENDIAN__
#define vsri_n_u16(__p0, __p1, __p2) __extension__ ({ \
  uint16x4_t __s0 = __p0; \
  uint16x4_t __s1 = __p1; \
  uint16x4_t __ret; \
  __ret = (uint16x4_t) __builtin_neon_vsri_n_v((int8x8_t)__s0, (int8x8_t)__s1, __p2, 17); \
  __ret; \
})
#else
#define vsri_n_u16(__p0, __p1, __p2) __extension__ ({ \
  uint16x4_t __s0 = __p0; \
  uint16x4_t __s1 = __p1; \
  uint16x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  uint16x4_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 3, 2, 1, 0); \
  uint16x4_t __ret; \
  __ret = (uint16x4_t) __builtin_neon_vsri_n_v((int8x8_t)__rev0, (int8x8_t)__rev1, __p2, 17); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsri_n_s8(__p0, __p1, __p2) __extension__ ({ \
  int8x8_t __s0 = __p0; \
  int8x8_t __s1 = __p1; \
  int8x8_t __ret; \
  __ret = (int8x8_t) __builtin_neon_vsri_n_v((int8x8_t)__s0, (int8x8_t)__s1, __p2, 0); \
  __ret; \
})
#else
#define vsri_n_s8(__p0, __p1, __p2) __extension__ ({ \
  int8x8_t __s0 = __p0; \
  int8x8_t __s1 = __p1; \
  int8x8_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 7, 6, 5, 4, 3, 2, 1, 0); \
  int8x8_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 7, 6, 5, 4, 3, 2, 1, 0); \
  int8x8_t __ret; \
  __ret = (int8x8_t) __builtin_neon_vsri_n_v((int8x8_t)__rev0, (int8x8_t)__rev1, __p2, 0); \
  __ret = __builtin_shufflevector(__ret, __ret, 7, 6, 5, 4, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vsri_n_s32(__p0, __p1, __p2) __extension__ ({ \
  int32x2_t __s0 = __p0; \
  int32x2_t __s1 = __p1; \
  int32x2_t __ret; \
  __ret = (int32x2_t) __builtin_neon_vsri_n_v((int8x8_t)__s0, (int8x8_t)__s1, __p2, 2); \
  __ret; \
})
#else
#define vsri_n_s32(__p0, __p1, __p2) __extension__ ({ \
  int32x2_t __s0 = __p0; \
  int32x2_t __s1 = __p1; \
  int32x2_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 1, 0); \
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 1, 0); \
  int32x2_t __ret; \
  __ret = (int32x2_t) __builtin_neon_vsri_n_v((int8x8_t)__rev0, (int8x8_t)__rev1, __p2, 2); \
  __ret = __builtin_shufflevector(__ret, __ret, 1, 0); \
  __ret; \
})
#endif

#define vsri_n_s64(__p0, __p1, __p2) __extension__ ({ \
  int64x1_t __s0 = __p0; \
  int64x1_t __s1 = __p1; \
  int64x1_t __ret; \
  __ret = (int64x1_t) __builtin_neon_vsri_n_v((int8x8_t)__s0, (int8x8_t)__s1, __p2, 3); \
  __ret; \
})
#ifdef __LITTLE_ENDIAN__
#define vsri_n_s16(__p0, __p1, __p2) __extension__ ({ \
  int16x4_t __s0 = __p0; \
  int16x4_t __s1 = __p1; \
  int16x4_t __ret; \
  __ret = (int16x4_t) __builtin_neon_vsri_n_v((int8x8_t)__s0, (int8x8_t)__s1, __p2, 1); \
  __ret; \
})
#else
#define vsri_n_s16(__p0, __p1, __p2) __extension__ ({ \
  int16x4_t __s0 = __p0; \
  int16x4_t __s1 = __p1; \
  int16x4_t __rev0;  __rev0 = __builtin_shufflevector(__s0, __s0, 3, 2, 1, 0); \
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 3, 2, 1, 0); \
  int16x4_t __ret; \
  __ret = (int16x4_t) __builtin_neon_vsri_n_v((int8x8_t)__rev0, (int8x8_t)__rev1, __p2, 1); \
  __ret = __builtin_shufflevector(__ret, __ret, 3, 2, 1, 0); \
  __ret; \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1_p8(__p0, __p1) __extension__ ({ \
  poly8x8_t __s1 = __p1; \
  __builtin_neon_vst1_v(__p0, (int8x8_t)__s1, 4); \
})
#else
#define vst1_p8(__p0, __p1) __extension__ ({ \
  poly8x8_t __s1 = __p1; \
  poly8x8_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst1_v(__p0, (int8x8_t)__rev1, 4); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1_p16(__p0, __p1) __extension__ ({ \
  poly16x4_t __s1 = __p1; \
  __builtin_neon_vst1_v(__p0, (int8x8_t)__s1, 5); \
})
#else
#define vst1_p16(__p0, __p1) __extension__ ({ \
  poly16x4_t __s1 = __p1; \
  poly16x4_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 3, 2, 1, 0); \
  __builtin_neon_vst1_v(__p0, (int8x8_t)__rev1, 5); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_p8(__p0, __p1) __extension__ ({ \
  poly8x16_t __s1 = __p1; \
  __builtin_neon_vst1q_v(__p0, (int8x16_t)__s1, 36); \
})
#else
#define vst1q_p8(__p0, __p1) __extension__ ({ \
  poly8x16_t __s1 = __p1; \
  poly8x16_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst1q_v(__p0, (int8x16_t)__rev1, 36); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_p16(__p0, __p1) __extension__ ({ \
  poly16x8_t __s1 = __p1; \
  __builtin_neon_vst1q_v(__p0, (int8x16_t)__s1, 37); \
})
#else
#define vst1q_p16(__p0, __p1) __extension__ ({ \
  poly16x8_t __s1 = __p1; \
  poly16x8_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst1q_v(__p0, (int8x16_t)__rev1, 37); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_u8(__p0, __p1) __extension__ ({ \
  uint8x16_t __s1 = __p1; \
  __builtin_neon_vst1q_v(__p0, (int8x16_t)__s1, 48); \
})
#else
#define vst1q_u8(__p0, __p1) __extension__ ({ \
  uint8x16_t __s1 = __p1; \
  uint8x16_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst1q_v(__p0, (int8x16_t)__rev1, 48); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_u32(__p0, __p1) __extension__ ({ \
  uint32x4_t __s1 = __p1; \
  __builtin_neon_vst1q_v(__p0, (int8x16_t)__s1, 50); \
})
#else
#define vst1q_u32(__p0, __p1) __extension__ ({ \
  uint32x4_t __s1 = __p1; \
  uint32x4_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 3, 2, 1, 0); \
  __builtin_neon_vst1q_v(__p0, (int8x16_t)__rev1, 50); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_u64(__p0, __p1) __extension__ ({ \
  uint64x2_t __s1 = __p1; \
  __builtin_neon_vst1q_v(__p0, (int8x16_t)__s1, 51); \
})
#else
#define vst1q_u64(__p0, __p1) __extension__ ({ \
  uint64x2_t __s1 = __p1; \
  uint64x2_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 1, 0); \
  __builtin_neon_vst1q_v(__p0, (int8x16_t)__rev1, 51); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_u16(__p0, __p1) __extension__ ({ \
  uint16x8_t __s1 = __p1; \
  __builtin_neon_vst1q_v(__p0, (int8x16_t)__s1, 49); \
})
#else
#define vst1q_u16(__p0, __p1) __extension__ ({ \
  uint16x8_t __s1 = __p1; \
  uint16x8_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst1q_v(__p0, (int8x16_t)__rev1, 49); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_s8(__p0, __p1) __extension__ ({ \
  int8x16_t __s1 = __p1; \
  __builtin_neon_vst1q_v(__p0, (int8x16_t)__s1, 32); \
})
#else
#define vst1q_s8(__p0, __p1) __extension__ ({ \
  int8x16_t __s1 = __p1; \
  int8x16_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst1q_v(__p0, (int8x16_t)__rev1, 32); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_f32(__p0, __p1) __extension__ ({ \
  float32x4_t __s1 = __p1; \
  __builtin_neon_vst1q_v(__p0, (int8x16_t)__s1, 41); \
})
#else
#define vst1q_f32(__p0, __p1) __extension__ ({ \
  float32x4_t __s1 = __p1; \
  float32x4_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 3, 2, 1, 0); \
  __builtin_neon_vst1q_v(__p0, (int8x16_t)__rev1, 41); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_s32(__p0, __p1) __extension__ ({ \
  int32x4_t __s1 = __p1; \
  __builtin_neon_vst1q_v(__p0, (int8x16_t)__s1, 34); \
})
#else
#define vst1q_s32(__p0, __p1) __extension__ ({ \
  int32x4_t __s1 = __p1; \
  int32x4_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 3, 2, 1, 0); \
  __builtin_neon_vst1q_v(__p0, (int8x16_t)__rev1, 34); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_s64(__p0, __p1) __extension__ ({ \
  int64x2_t __s1 = __p1; \
  __builtin_neon_vst1q_v(__p0, (int8x16_t)__s1, 35); \
})
#else
#define vst1q_s64(__p0, __p1) __extension__ ({ \
  int64x2_t __s1 = __p1; \
  int64x2_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 1, 0); \
  __builtin_neon_vst1q_v(__p0, (int8x16_t)__rev1, 35); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_s16(__p0, __p1) __extension__ ({ \
  int16x8_t __s1 = __p1; \
  __builtin_neon_vst1q_v(__p0, (int8x16_t)__s1, 33); \
})
#else
#define vst1q_s16(__p0, __p1) __extension__ ({ \
  int16x8_t __s1 = __p1; \
  int16x8_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst1q_v(__p0, (int8x16_t)__rev1, 33); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1_u8(__p0, __p1) __extension__ ({ \
  uint8x8_t __s1 = __p1; \
  __builtin_neon_vst1_v(__p0, (int8x8_t)__s1, 16); \
})
#else
#define vst1_u8(__p0, __p1) __extension__ ({ \
  uint8x8_t __s1 = __p1; \
  uint8x8_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst1_v(__p0, (int8x8_t)__rev1, 16); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1_u32(__p0, __p1) __extension__ ({ \
  uint32x2_t __s1 = __p1; \
  __builtin_neon_vst1_v(__p0, (int8x8_t)__s1, 18); \
})
#else
#define vst1_u32(__p0, __p1) __extension__ ({ \
  uint32x2_t __s1 = __p1; \
  uint32x2_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 1, 0); \
  __builtin_neon_vst1_v(__p0, (int8x8_t)__rev1, 18); \
})
#endif

#define vst1_u64(__p0, __p1) __extension__ ({ \
  uint64x1_t __s1 = __p1; \
  __builtin_neon_vst1_v(__p0, (int8x8_t)__s1, 19); \
})
#ifdef __LITTLE_ENDIAN__
#define vst1_u16(__p0, __p1) __extension__ ({ \
  uint16x4_t __s1 = __p1; \
  __builtin_neon_vst1_v(__p0, (int8x8_t)__s1, 17); \
})
#else
#define vst1_u16(__p0, __p1) __extension__ ({ \
  uint16x4_t __s1 = __p1; \
  uint16x4_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 3, 2, 1, 0); \
  __builtin_neon_vst1_v(__p0, (int8x8_t)__rev1, 17); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1_s8(__p0, __p1) __extension__ ({ \
  int8x8_t __s1 = __p1; \
  __builtin_neon_vst1_v(__p0, (int8x8_t)__s1, 0); \
})
#else
#define vst1_s8(__p0, __p1) __extension__ ({ \
  int8x8_t __s1 = __p1; \
  int8x8_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst1_v(__p0, (int8x8_t)__rev1, 0); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1_f32(__p0, __p1) __extension__ ({ \
  float32x2_t __s1 = __p1; \
  __builtin_neon_vst1_v(__p0, (int8x8_t)__s1, 9); \
})
#else
#define vst1_f32(__p0, __p1) __extension__ ({ \
  float32x2_t __s1 = __p1; \
  float32x2_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 1, 0); \
  __builtin_neon_vst1_v(__p0, (int8x8_t)__rev1, 9); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1_s32(__p0, __p1) __extension__ ({ \
  int32x2_t __s1 = __p1; \
  __builtin_neon_vst1_v(__p0, (int8x8_t)__s1, 2); \
})
#else
#define vst1_s32(__p0, __p1) __extension__ ({ \
  int32x2_t __s1 = __p1; \
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 1, 0); \
  __builtin_neon_vst1_v(__p0, (int8x8_t)__rev1, 2); \
})
#endif

#define vst1_s64(__p0, __p1) __extension__ ({ \
  int64x1_t __s1 = __p1; \
  __builtin_neon_vst1_v(__p0, (int8x8_t)__s1, 3); \
})
#ifdef __LITTLE_ENDIAN__
#define vst1_s16(__p0, __p1) __extension__ ({ \
  int16x4_t __s1 = __p1; \
  __builtin_neon_vst1_v(__p0, (int8x8_t)__s1, 1); \
})
#else
#define vst1_s16(__p0, __p1) __extension__ ({ \
  int16x4_t __s1 = __p1; \
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 3, 2, 1, 0); \
  __builtin_neon_vst1_v(__p0, (int8x8_t)__rev1, 1); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1_lane_p8(__p0, __p1, __p2) __extension__ ({ \
  poly8x8_t __s1 = __p1; \
  __builtin_neon_vst1_lane_v(__p0, (int8x8_t)__s1, __p2, 4); \
})
#else
#define vst1_lane_p8(__p0, __p1, __p2) __extension__ ({ \
  poly8x8_t __s1 = __p1; \
  poly8x8_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst1_lane_v(__p0, (int8x8_t)__rev1, __p2, 4); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1_lane_p16(__p0, __p1, __p2) __extension__ ({ \
  poly16x4_t __s1 = __p1; \
  __builtin_neon_vst1_lane_v(__p0, (int8x8_t)__s1, __p2, 5); \
})
#else
#define vst1_lane_p16(__p0, __p1, __p2) __extension__ ({ \
  poly16x4_t __s1 = __p1; \
  poly16x4_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 3, 2, 1, 0); \
  __builtin_neon_vst1_lane_v(__p0, (int8x8_t)__rev1, __p2, 5); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_lane_p8(__p0, __p1, __p2) __extension__ ({ \
  poly8x16_t __s1 = __p1; \
  __builtin_neon_vst1q_lane_v(__p0, (int8x16_t)__s1, __p2, 36); \
})
#else
#define vst1q_lane_p8(__p0, __p1, __p2) __extension__ ({ \
  poly8x16_t __s1 = __p1; \
  poly8x16_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst1q_lane_v(__p0, (int8x16_t)__rev1, __p2, 36); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_lane_p16(__p0, __p1, __p2) __extension__ ({ \
  poly16x8_t __s1 = __p1; \
  __builtin_neon_vst1q_lane_v(__p0, (int8x16_t)__s1, __p2, 37); \
})
#else
#define vst1q_lane_p16(__p0, __p1, __p2) __extension__ ({ \
  poly16x8_t __s1 = __p1; \
  poly16x8_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst1q_lane_v(__p0, (int8x16_t)__rev1, __p2, 37); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_lane_u8(__p0, __p1, __p2) __extension__ ({ \
  uint8x16_t __s1 = __p1; \
  __builtin_neon_vst1q_lane_v(__p0, (int8x16_t)__s1, __p2, 48); \
})
#else
#define vst1q_lane_u8(__p0, __p1, __p2) __extension__ ({ \
  uint8x16_t __s1 = __p1; \
  uint8x16_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst1q_lane_v(__p0, (int8x16_t)__rev1, __p2, 48); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_lane_u32(__p0, __p1, __p2) __extension__ ({ \
  uint32x4_t __s1 = __p1; \
  __builtin_neon_vst1q_lane_v(__p0, (int8x16_t)__s1, __p2, 50); \
})
#else
#define vst1q_lane_u32(__p0, __p1, __p2) __extension__ ({ \
  uint32x4_t __s1 = __p1; \
  uint32x4_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 3, 2, 1, 0); \
  __builtin_neon_vst1q_lane_v(__p0, (int8x16_t)__rev1, __p2, 50); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_lane_u64(__p0, __p1, __p2) __extension__ ({ \
  uint64x2_t __s1 = __p1; \
  __builtin_neon_vst1q_lane_v(__p0, (int8x16_t)__s1, __p2, 51); \
})
#else
#define vst1q_lane_u64(__p0, __p1, __p2) __extension__ ({ \
  uint64x2_t __s1 = __p1; \
  uint64x2_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 1, 0); \
  __builtin_neon_vst1q_lane_v(__p0, (int8x16_t)__rev1, __p2, 51); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_lane_u16(__p0, __p1, __p2) __extension__ ({ \
  uint16x8_t __s1 = __p1; \
  __builtin_neon_vst1q_lane_v(__p0, (int8x16_t)__s1, __p2, 49); \
})
#else
#define vst1q_lane_u16(__p0, __p1, __p2) __extension__ ({ \
  uint16x8_t __s1 = __p1; \
  uint16x8_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst1q_lane_v(__p0, (int8x16_t)__rev1, __p2, 49); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_lane_s8(__p0, __p1, __p2) __extension__ ({ \
  int8x16_t __s1 = __p1; \
  __builtin_neon_vst1q_lane_v(__p0, (int8x16_t)__s1, __p2, 32); \
})
#else
#define vst1q_lane_s8(__p0, __p1, __p2) __extension__ ({ \
  int8x16_t __s1 = __p1; \
  int8x16_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst1q_lane_v(__p0, (int8x16_t)__rev1, __p2, 32); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_lane_f32(__p0, __p1, __p2) __extension__ ({ \
  float32x4_t __s1 = __p1; \
  __builtin_neon_vst1q_lane_v(__p0, (int8x16_t)__s1, __p2, 41); \
})
#else
#define vst1q_lane_f32(__p0, __p1, __p2) __extension__ ({ \
  float32x4_t __s1 = __p1; \
  float32x4_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 3, 2, 1, 0); \
  __builtin_neon_vst1q_lane_v(__p0, (int8x16_t)__rev1, __p2, 41); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_lane_s32(__p0, __p1, __p2) __extension__ ({ \
  int32x4_t __s1 = __p1; \
  __builtin_neon_vst1q_lane_v(__p0, (int8x16_t)__s1, __p2, 34); \
})
#else
#define vst1q_lane_s32(__p0, __p1, __p2) __extension__ ({ \
  int32x4_t __s1 = __p1; \
  int32x4_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 3, 2, 1, 0); \
  __builtin_neon_vst1q_lane_v(__p0, (int8x16_t)__rev1, __p2, 34); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_lane_s64(__p0, __p1, __p2) __extension__ ({ \
  int64x2_t __s1 = __p1; \
  __builtin_neon_vst1q_lane_v(__p0, (int8x16_t)__s1, __p2, 35); \
})
#else
#define vst1q_lane_s64(__p0, __p1, __p2) __extension__ ({ \
  int64x2_t __s1 = __p1; \
  int64x2_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 1, 0); \
  __builtin_neon_vst1q_lane_v(__p0, (int8x16_t)__rev1, __p2, 35); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_lane_s16(__p0, __p1, __p2) __extension__ ({ \
  int16x8_t __s1 = __p1; \
  __builtin_neon_vst1q_lane_v(__p0, (int8x16_t)__s1, __p2, 33); \
})
#else
#define vst1q_lane_s16(__p0, __p1, __p2) __extension__ ({ \
  int16x8_t __s1 = __p1; \
  int16x8_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst1q_lane_v(__p0, (int8x16_t)__rev1, __p2, 33); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1_lane_u8(__p0, __p1, __p2) __extension__ ({ \
  uint8x8_t __s1 = __p1; \
  __builtin_neon_vst1_lane_v(__p0, (int8x8_t)__s1, __p2, 16); \
})
#else
#define vst1_lane_u8(__p0, __p1, __p2) __extension__ ({ \
  uint8x8_t __s1 = __p1; \
  uint8x8_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst1_lane_v(__p0, (int8x8_t)__rev1, __p2, 16); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1_lane_u32(__p0, __p1, __p2) __extension__ ({ \
  uint32x2_t __s1 = __p1; \
  __builtin_neon_vst1_lane_v(__p0, (int8x8_t)__s1, __p2, 18); \
})
#else
#define vst1_lane_u32(__p0, __p1, __p2) __extension__ ({ \
  uint32x2_t __s1 = __p1; \
  uint32x2_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 1, 0); \
  __builtin_neon_vst1_lane_v(__p0, (int8x8_t)__rev1, __p2, 18); \
})
#endif

#define vst1_lane_u64(__p0, __p1, __p2) __extension__ ({ \
  uint64x1_t __s1 = __p1; \
  __builtin_neon_vst1_lane_v(__p0, (int8x8_t)__s1, __p2, 19); \
})
#ifdef __LITTLE_ENDIAN__
#define vst1_lane_u16(__p0, __p1, __p2) __extension__ ({ \
  uint16x4_t __s1 = __p1; \
  __builtin_neon_vst1_lane_v(__p0, (int8x8_t)__s1, __p2, 17); \
})
#else
#define vst1_lane_u16(__p0, __p1, __p2) __extension__ ({ \
  uint16x4_t __s1 = __p1; \
  uint16x4_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 3, 2, 1, 0); \
  __builtin_neon_vst1_lane_v(__p0, (int8x8_t)__rev1, __p2, 17); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1_lane_s8(__p0, __p1, __p2) __extension__ ({ \
  int8x8_t __s1 = __p1; \
  __builtin_neon_vst1_lane_v(__p0, (int8x8_t)__s1, __p2, 0); \
})
#else
#define vst1_lane_s8(__p0, __p1, __p2) __extension__ ({ \
  int8x8_t __s1 = __p1; \
  int8x8_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst1_lane_v(__p0, (int8x8_t)__rev1, __p2, 0); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1_lane_f32(__p0, __p1, __p2) __extension__ ({ \
  float32x2_t __s1 = __p1; \
  __builtin_neon_vst1_lane_v(__p0, (int8x8_t)__s1, __p2, 9); \
})
#else
#define vst1_lane_f32(__p0, __p1, __p2) __extension__ ({ \
  float32x2_t __s1 = __p1; \
  float32x2_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 1, 0); \
  __builtin_neon_vst1_lane_v(__p0, (int8x8_t)__rev1, __p2, 9); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1_lane_s32(__p0, __p1, __p2) __extension__ ({ \
  int32x2_t __s1 = __p1; \
  __builtin_neon_vst1_lane_v(__p0, (int8x8_t)__s1, __p2, 2); \
})
#else
#define vst1_lane_s32(__p0, __p1, __p2) __extension__ ({ \
  int32x2_t __s1 = __p1; \
  int32x2_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 1, 0); \
  __builtin_neon_vst1_lane_v(__p0, (int8x8_t)__rev1, __p2, 2); \
})
#endif

#define vst1_lane_s64(__p0, __p1, __p2) __extension__ ({ \
  int64x1_t __s1 = __p1; \
  __builtin_neon_vst1_lane_v(__p0, (int8x8_t)__s1, __p2, 3); \
})
#ifdef __LITTLE_ENDIAN__
#define vst1_lane_s16(__p0, __p1, __p2) __extension__ ({ \
  int16x4_t __s1 = __p1; \
  __builtin_neon_vst1_lane_v(__p0, (int8x8_t)__s1, __p2, 1); \
})
#else
#define vst1_lane_s16(__p0, __p1, __p2) __extension__ ({ \
  int16x4_t __s1 = __p1; \
  int16x4_t __rev1;  __rev1 = __builtin_shufflevector(__s1, __s1, 3, 2, 1, 0); \
  __builtin_neon_vst1_lane_v(__p0, (int8x8_t)__rev1, __p2, 1); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1_p8_x2(__p0, __p1) __extension__ ({ \
  poly8x8x2_t __s1 = __p1; \
  __builtin_neon_vst1_x2_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], 4); \
})
#else
#define vst1_p8_x2(__p0, __p1) __extension__ ({ \
  poly8x8x2_t __s1 = __p1; \
  poly8x8x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst1_x2_v(__p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], 4); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1_p16_x2(__p0, __p1) __extension__ ({ \
  poly16x4x2_t __s1 = __p1; \
  __builtin_neon_vst1_x2_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], 5); \
})
#else
#define vst1_p16_x2(__p0, __p1) __extension__ ({ \
  poly16x4x2_t __s1 = __p1; \
  poly16x4x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 3, 2, 1, 0); \
  __builtin_neon_vst1_x2_v(__p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], 5); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_p8_x2(__p0, __p1) __extension__ ({ \
  poly8x16x2_t __s1 = __p1; \
  __builtin_neon_vst1q_x2_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], 36); \
})
#else
#define vst1q_p8_x2(__p0, __p1) __extension__ ({ \
  poly8x16x2_t __s1 = __p1; \
  poly8x16x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst1q_x2_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], 36); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_p16_x2(__p0, __p1) __extension__ ({ \
  poly16x8x2_t __s1 = __p1; \
  __builtin_neon_vst1q_x2_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], 37); \
})
#else
#define vst1q_p16_x2(__p0, __p1) __extension__ ({ \
  poly16x8x2_t __s1 = __p1; \
  poly16x8x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst1q_x2_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], 37); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_u8_x2(__p0, __p1) __extension__ ({ \
  uint8x16x2_t __s1 = __p1; \
  __builtin_neon_vst1q_x2_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], 48); \
})
#else
#define vst1q_u8_x2(__p0, __p1) __extension__ ({ \
  uint8x16x2_t __s1 = __p1; \
  uint8x16x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst1q_x2_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], 48); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_u32_x2(__p0, __p1) __extension__ ({ \
  uint32x4x2_t __s1 = __p1; \
  __builtin_neon_vst1q_x2_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], 50); \
})
#else
#define vst1q_u32_x2(__p0, __p1) __extension__ ({ \
  uint32x4x2_t __s1 = __p1; \
  uint32x4x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 3, 2, 1, 0); \
  __builtin_neon_vst1q_x2_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], 50); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_u64_x2(__p0, __p1) __extension__ ({ \
  uint64x2x2_t __s1 = __p1; \
  __builtin_neon_vst1q_x2_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], 51); \
})
#else
#define vst1q_u64_x2(__p0, __p1) __extension__ ({ \
  uint64x2x2_t __s1 = __p1; \
  uint64x2x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 1, 0); \
  __builtin_neon_vst1q_x2_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], 51); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_u16_x2(__p0, __p1) __extension__ ({ \
  uint16x8x2_t __s1 = __p1; \
  __builtin_neon_vst1q_x2_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], 49); \
})
#else
#define vst1q_u16_x2(__p0, __p1) __extension__ ({ \
  uint16x8x2_t __s1 = __p1; \
  uint16x8x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst1q_x2_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], 49); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_s8_x2(__p0, __p1) __extension__ ({ \
  int8x16x2_t __s1 = __p1; \
  __builtin_neon_vst1q_x2_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], 32); \
})
#else
#define vst1q_s8_x2(__p0, __p1) __extension__ ({ \
  int8x16x2_t __s1 = __p1; \
  int8x16x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst1q_x2_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], 32); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_f32_x2(__p0, __p1) __extension__ ({ \
  float32x4x2_t __s1 = __p1; \
  __builtin_neon_vst1q_x2_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], 41); \
})
#else
#define vst1q_f32_x2(__p0, __p1) __extension__ ({ \
  float32x4x2_t __s1 = __p1; \
  float32x4x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 3, 2, 1, 0); \
  __builtin_neon_vst1q_x2_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], 41); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_s32_x2(__p0, __p1) __extension__ ({ \
  int32x4x2_t __s1 = __p1; \
  __builtin_neon_vst1q_x2_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], 34); \
})
#else
#define vst1q_s32_x2(__p0, __p1) __extension__ ({ \
  int32x4x2_t __s1 = __p1; \
  int32x4x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 3, 2, 1, 0); \
  __builtin_neon_vst1q_x2_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], 34); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_s64_x2(__p0, __p1) __extension__ ({ \
  int64x2x2_t __s1 = __p1; \
  __builtin_neon_vst1q_x2_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], 35); \
})
#else
#define vst1q_s64_x2(__p0, __p1) __extension__ ({ \
  int64x2x2_t __s1 = __p1; \
  int64x2x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 1, 0); \
  __builtin_neon_vst1q_x2_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], 35); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_s16_x2(__p0, __p1) __extension__ ({ \
  int16x8x2_t __s1 = __p1; \
  __builtin_neon_vst1q_x2_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], 33); \
})
#else
#define vst1q_s16_x2(__p0, __p1) __extension__ ({ \
  int16x8x2_t __s1 = __p1; \
  int16x8x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst1q_x2_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], 33); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1_u8_x2(__p0, __p1) __extension__ ({ \
  uint8x8x2_t __s1 = __p1; \
  __builtin_neon_vst1_x2_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], 16); \
})
#else
#define vst1_u8_x2(__p0, __p1) __extension__ ({ \
  uint8x8x2_t __s1 = __p1; \
  uint8x8x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst1_x2_v(__p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], 16); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1_u32_x2(__p0, __p1) __extension__ ({ \
  uint32x2x2_t __s1 = __p1; \
  __builtin_neon_vst1_x2_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], 18); \
})
#else
#define vst1_u32_x2(__p0, __p1) __extension__ ({ \
  uint32x2x2_t __s1 = __p1; \
  uint32x2x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 1, 0); \
  __builtin_neon_vst1_x2_v(__p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], 18); \
})
#endif

#define vst1_u64_x2(__p0, __p1) __extension__ ({ \
  uint64x1x2_t __s1 = __p1; \
  __builtin_neon_vst1_x2_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], 19); \
})
#ifdef __LITTLE_ENDIAN__
#define vst1_u16_x2(__p0, __p1) __extension__ ({ \
  uint16x4x2_t __s1 = __p1; \
  __builtin_neon_vst1_x2_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], 17); \
})
#else
#define vst1_u16_x2(__p0, __p1) __extension__ ({ \
  uint16x4x2_t __s1 = __p1; \
  uint16x4x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 3, 2, 1, 0); \
  __builtin_neon_vst1_x2_v(__p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], 17); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1_s8_x2(__p0, __p1) __extension__ ({ \
  int8x8x2_t __s1 = __p1; \
  __builtin_neon_vst1_x2_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], 0); \
})
#else
#define vst1_s8_x2(__p0, __p1) __extension__ ({ \
  int8x8x2_t __s1 = __p1; \
  int8x8x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst1_x2_v(__p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], 0); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1_f32_x2(__p0, __p1) __extension__ ({ \
  float32x2x2_t __s1 = __p1; \
  __builtin_neon_vst1_x2_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], 9); \
})
#else
#define vst1_f32_x2(__p0, __p1) __extension__ ({ \
  float32x2x2_t __s1 = __p1; \
  float32x2x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 1, 0); \
  __builtin_neon_vst1_x2_v(__p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], 9); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1_s32_x2(__p0, __p1) __extension__ ({ \
  int32x2x2_t __s1 = __p1; \
  __builtin_neon_vst1_x2_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], 2); \
})
#else
#define vst1_s32_x2(__p0, __p1) __extension__ ({ \
  int32x2x2_t __s1 = __p1; \
  int32x2x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 1, 0); \
  __builtin_neon_vst1_x2_v(__p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], 2); \
})
#endif

#define vst1_s64_x2(__p0, __p1) __extension__ ({ \
  int64x1x2_t __s1 = __p1; \
  __builtin_neon_vst1_x2_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], 3); \
})
#ifdef __LITTLE_ENDIAN__
#define vst1_s16_x2(__p0, __p1) __extension__ ({ \
  int16x4x2_t __s1 = __p1; \
  __builtin_neon_vst1_x2_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], 1); \
})
#else
#define vst1_s16_x2(__p0, __p1) __extension__ ({ \
  int16x4x2_t __s1 = __p1; \
  int16x4x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 3, 2, 1, 0); \
  __builtin_neon_vst1_x2_v(__p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], 1); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1_p8_x3(__p0, __p1) __extension__ ({ \
  poly8x8x3_t __s1 = __p1; \
  __builtin_neon_vst1_x3_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], (int8x8_t)__s1.val[2], 4); \
})
#else
#define vst1_p8_x3(__p0, __p1) __extension__ ({ \
  poly8x8x3_t __s1 = __p1; \
  poly8x8x3_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst1_x3_v(__p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], (int8x8_t)__rev1.val[2], 4); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1_p16_x3(__p0, __p1) __extension__ ({ \
  poly16x4x3_t __s1 = __p1; \
  __builtin_neon_vst1_x3_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], (int8x8_t)__s1.val[2], 5); \
})
#else
#define vst1_p16_x3(__p0, __p1) __extension__ ({ \
  poly16x4x3_t __s1 = __p1; \
  poly16x4x3_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 3, 2, 1, 0); \
  __builtin_neon_vst1_x3_v(__p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], (int8x8_t)__rev1.val[2], 5); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_p8_x3(__p0, __p1) __extension__ ({ \
  poly8x16x3_t __s1 = __p1; \
  __builtin_neon_vst1q_x3_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], (int8x16_t)__s1.val[2], 36); \
})
#else
#define vst1q_p8_x3(__p0, __p1) __extension__ ({ \
  poly8x16x3_t __s1 = __p1; \
  poly8x16x3_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst1q_x3_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], (int8x16_t)__rev1.val[2], 36); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_p16_x3(__p0, __p1) __extension__ ({ \
  poly16x8x3_t __s1 = __p1; \
  __builtin_neon_vst1q_x3_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], (int8x16_t)__s1.val[2], 37); \
})
#else
#define vst1q_p16_x3(__p0, __p1) __extension__ ({ \
  poly16x8x3_t __s1 = __p1; \
  poly16x8x3_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst1q_x3_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], (int8x16_t)__rev1.val[2], 37); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_u8_x3(__p0, __p1) __extension__ ({ \
  uint8x16x3_t __s1 = __p1; \
  __builtin_neon_vst1q_x3_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], (int8x16_t)__s1.val[2], 48); \
})
#else
#define vst1q_u8_x3(__p0, __p1) __extension__ ({ \
  uint8x16x3_t __s1 = __p1; \
  uint8x16x3_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst1q_x3_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], (int8x16_t)__rev1.val[2], 48); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_u32_x3(__p0, __p1) __extension__ ({ \
  uint32x4x3_t __s1 = __p1; \
  __builtin_neon_vst1q_x3_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], (int8x16_t)__s1.val[2], 50); \
})
#else
#define vst1q_u32_x3(__p0, __p1) __extension__ ({ \
  uint32x4x3_t __s1 = __p1; \
  uint32x4x3_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 3, 2, 1, 0); \
  __builtin_neon_vst1q_x3_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], (int8x16_t)__rev1.val[2], 50); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_u64_x3(__p0, __p1) __extension__ ({ \
  uint64x2x3_t __s1 = __p1; \
  __builtin_neon_vst1q_x3_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], (int8x16_t)__s1.val[2], 51); \
})
#else
#define vst1q_u64_x3(__p0, __p1) __extension__ ({ \
  uint64x2x3_t __s1 = __p1; \
  uint64x2x3_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 1, 0); \
  __builtin_neon_vst1q_x3_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], (int8x16_t)__rev1.val[2], 51); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_u16_x3(__p0, __p1) __extension__ ({ \
  uint16x8x3_t __s1 = __p1; \
  __builtin_neon_vst1q_x3_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], (int8x16_t)__s1.val[2], 49); \
})
#else
#define vst1q_u16_x3(__p0, __p1) __extension__ ({ \
  uint16x8x3_t __s1 = __p1; \
  uint16x8x3_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst1q_x3_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], (int8x16_t)__rev1.val[2], 49); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_s8_x3(__p0, __p1) __extension__ ({ \
  int8x16x3_t __s1 = __p1; \
  __builtin_neon_vst1q_x3_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], (int8x16_t)__s1.val[2], 32); \
})
#else
#define vst1q_s8_x3(__p0, __p1) __extension__ ({ \
  int8x16x3_t __s1 = __p1; \
  int8x16x3_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst1q_x3_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], (int8x16_t)__rev1.val[2], 32); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_f32_x3(__p0, __p1) __extension__ ({ \
  float32x4x3_t __s1 = __p1; \
  __builtin_neon_vst1q_x3_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], (int8x16_t)__s1.val[2], 41); \
})
#else
#define vst1q_f32_x3(__p0, __p1) __extension__ ({ \
  float32x4x3_t __s1 = __p1; \
  float32x4x3_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 3, 2, 1, 0); \
  __builtin_neon_vst1q_x3_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], (int8x16_t)__rev1.val[2], 41); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_s32_x3(__p0, __p1) __extension__ ({ \
  int32x4x3_t __s1 = __p1; \
  __builtin_neon_vst1q_x3_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], (int8x16_t)__s1.val[2], 34); \
})
#else
#define vst1q_s32_x3(__p0, __p1) __extension__ ({ \
  int32x4x3_t __s1 = __p1; \
  int32x4x3_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 3, 2, 1, 0); \
  __builtin_neon_vst1q_x3_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], (int8x16_t)__rev1.val[2], 34); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_s64_x3(__p0, __p1) __extension__ ({ \
  int64x2x3_t __s1 = __p1; \
  __builtin_neon_vst1q_x3_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], (int8x16_t)__s1.val[2], 35); \
})
#else
#define vst1q_s64_x3(__p0, __p1) __extension__ ({ \
  int64x2x3_t __s1 = __p1; \
  int64x2x3_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 1, 0); \
  __builtin_neon_vst1q_x3_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], (int8x16_t)__rev1.val[2], 35); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_s16_x3(__p0, __p1) __extension__ ({ \
  int16x8x3_t __s1 = __p1; \
  __builtin_neon_vst1q_x3_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], (int8x16_t)__s1.val[2], 33); \
})
#else
#define vst1q_s16_x3(__p0, __p1) __extension__ ({ \
  int16x8x3_t __s1 = __p1; \
  int16x8x3_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst1q_x3_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], (int8x16_t)__rev1.val[2], 33); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1_u8_x3(__p0, __p1) __extension__ ({ \
  uint8x8x3_t __s1 = __p1; \
  __builtin_neon_vst1_x3_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], (int8x8_t)__s1.val[2], 16); \
})
#else
#define vst1_u8_x3(__p0, __p1) __extension__ ({ \
  uint8x8x3_t __s1 = __p1; \
  uint8x8x3_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst1_x3_v(__p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], (int8x8_t)__rev1.val[2], 16); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1_u32_x3(__p0, __p1) __extension__ ({ \
  uint32x2x3_t __s1 = __p1; \
  __builtin_neon_vst1_x3_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], (int8x8_t)__s1.val[2], 18); \
})
#else
#define vst1_u32_x3(__p0, __p1) __extension__ ({ \
  uint32x2x3_t __s1 = __p1; \
  uint32x2x3_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 1, 0); \
  __builtin_neon_vst1_x3_v(__p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], (int8x8_t)__rev1.val[2], 18); \
})
#endif

#define vst1_u64_x3(__p0, __p1) __extension__ ({ \
  uint64x1x3_t __s1 = __p1; \
  __builtin_neon_vst1_x3_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], (int8x8_t)__s1.val[2], 19); \
})
#ifdef __LITTLE_ENDIAN__
#define vst1_u16_x3(__p0, __p1) __extension__ ({ \
  uint16x4x3_t __s1 = __p1; \
  __builtin_neon_vst1_x3_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], (int8x8_t)__s1.val[2], 17); \
})
#else
#define vst1_u16_x3(__p0, __p1) __extension__ ({ \
  uint16x4x3_t __s1 = __p1; \
  uint16x4x3_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 3, 2, 1, 0); \
  __builtin_neon_vst1_x3_v(__p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], (int8x8_t)__rev1.val[2], 17); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1_s8_x3(__p0, __p1) __extension__ ({ \
  int8x8x3_t __s1 = __p1; \
  __builtin_neon_vst1_x3_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], (int8x8_t)__s1.val[2], 0); \
})
#else
#define vst1_s8_x3(__p0, __p1) __extension__ ({ \
  int8x8x3_t __s1 = __p1; \
  int8x8x3_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst1_x3_v(__p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], (int8x8_t)__rev1.val[2], 0); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1_f32_x3(__p0, __p1) __extension__ ({ \
  float32x2x3_t __s1 = __p1; \
  __builtin_neon_vst1_x3_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], (int8x8_t)__s1.val[2], 9); \
})
#else
#define vst1_f32_x3(__p0, __p1) __extension__ ({ \
  float32x2x3_t __s1 = __p1; \
  float32x2x3_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 1, 0); \
  __builtin_neon_vst1_x3_v(__p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], (int8x8_t)__rev1.val[2], 9); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1_s32_x3(__p0, __p1) __extension__ ({ \
  int32x2x3_t __s1 = __p1; \
  __builtin_neon_vst1_x3_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], (int8x8_t)__s1.val[2], 2); \
})
#else
#define vst1_s32_x3(__p0, __p1) __extension__ ({ \
  int32x2x3_t __s1 = __p1; \
  int32x2x3_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 1, 0); \
  __builtin_neon_vst1_x3_v(__p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], (int8x8_t)__rev1.val[2], 2); \
})
#endif

#define vst1_s64_x3(__p0, __p1) __extension__ ({ \
  int64x1x3_t __s1 = __p1; \
  __builtin_neon_vst1_x3_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], (int8x8_t)__s1.val[2], 3); \
})
#ifdef __LITTLE_ENDIAN__
#define vst1_s16_x3(__p0, __p1) __extension__ ({ \
  int16x4x3_t __s1 = __p1; \
  __builtin_neon_vst1_x3_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], (int8x8_t)__s1.val[2], 1); \
})
#else
#define vst1_s16_x3(__p0, __p1) __extension__ ({ \
  int16x4x3_t __s1 = __p1; \
  int16x4x3_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 3, 2, 1, 0); \
  __builtin_neon_vst1_x3_v(__p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], (int8x8_t)__rev1.val[2], 1); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1_p8_x4(__p0, __p1) __extension__ ({ \
  poly8x8x4_t __s1 = __p1; \
  __builtin_neon_vst1_x4_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], (int8x8_t)__s1.val[2], (int8x8_t)__s1.val[3], 4); \
})
#else
#define vst1_p8_x4(__p0, __p1) __extension__ ({ \
  poly8x8x4_t __s1 = __p1; \
  poly8x8x4_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[3] = __builtin_shufflevector(__s1.val[3], __s1.val[3], 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst1_x4_v(__p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], (int8x8_t)__rev1.val[2], (int8x8_t)__rev1.val[3], 4); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1_p16_x4(__p0, __p1) __extension__ ({ \
  poly16x4x4_t __s1 = __p1; \
  __builtin_neon_vst1_x4_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], (int8x8_t)__s1.val[2], (int8x8_t)__s1.val[3], 5); \
})
#else
#define vst1_p16_x4(__p0, __p1) __extension__ ({ \
  poly16x4x4_t __s1 = __p1; \
  poly16x4x4_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 3, 2, 1, 0); \
  __rev1.val[3] = __builtin_shufflevector(__s1.val[3], __s1.val[3], 3, 2, 1, 0); \
  __builtin_neon_vst1_x4_v(__p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], (int8x8_t)__rev1.val[2], (int8x8_t)__rev1.val[3], 5); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_p8_x4(__p0, __p1) __extension__ ({ \
  poly8x16x4_t __s1 = __p1; \
  __builtin_neon_vst1q_x4_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], (int8x16_t)__s1.val[2], (int8x16_t)__s1.val[3], 36); \
})
#else
#define vst1q_p8_x4(__p0, __p1) __extension__ ({ \
  poly8x16x4_t __s1 = __p1; \
  poly8x16x4_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[3] = __builtin_shufflevector(__s1.val[3], __s1.val[3], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst1q_x4_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], (int8x16_t)__rev1.val[2], (int8x16_t)__rev1.val[3], 36); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_p16_x4(__p0, __p1) __extension__ ({ \
  poly16x8x4_t __s1 = __p1; \
  __builtin_neon_vst1q_x4_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], (int8x16_t)__s1.val[2], (int8x16_t)__s1.val[3], 37); \
})
#else
#define vst1q_p16_x4(__p0, __p1) __extension__ ({ \
  poly16x8x4_t __s1 = __p1; \
  poly16x8x4_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[3] = __builtin_shufflevector(__s1.val[3], __s1.val[3], 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst1q_x4_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], (int8x16_t)__rev1.val[2], (int8x16_t)__rev1.val[3], 37); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_u8_x4(__p0, __p1) __extension__ ({ \
  uint8x16x4_t __s1 = __p1; \
  __builtin_neon_vst1q_x4_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], (int8x16_t)__s1.val[2], (int8x16_t)__s1.val[3], 48); \
})
#else
#define vst1q_u8_x4(__p0, __p1) __extension__ ({ \
  uint8x16x4_t __s1 = __p1; \
  uint8x16x4_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[3] = __builtin_shufflevector(__s1.val[3], __s1.val[3], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst1q_x4_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], (int8x16_t)__rev1.val[2], (int8x16_t)__rev1.val[3], 48); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_u32_x4(__p0, __p1) __extension__ ({ \
  uint32x4x4_t __s1 = __p1; \
  __builtin_neon_vst1q_x4_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], (int8x16_t)__s1.val[2], (int8x16_t)__s1.val[3], 50); \
})
#else
#define vst1q_u32_x4(__p0, __p1) __extension__ ({ \
  uint32x4x4_t __s1 = __p1; \
  uint32x4x4_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 3, 2, 1, 0); \
  __rev1.val[3] = __builtin_shufflevector(__s1.val[3], __s1.val[3], 3, 2, 1, 0); \
  __builtin_neon_vst1q_x4_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], (int8x16_t)__rev1.val[2], (int8x16_t)__rev1.val[3], 50); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_u64_x4(__p0, __p1) __extension__ ({ \
  uint64x2x4_t __s1 = __p1; \
  __builtin_neon_vst1q_x4_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], (int8x16_t)__s1.val[2], (int8x16_t)__s1.val[3], 51); \
})
#else
#define vst1q_u64_x4(__p0, __p1) __extension__ ({ \
  uint64x2x4_t __s1 = __p1; \
  uint64x2x4_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 1, 0); \
  __rev1.val[3] = __builtin_shufflevector(__s1.val[3], __s1.val[3], 1, 0); \
  __builtin_neon_vst1q_x4_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], (int8x16_t)__rev1.val[2], (int8x16_t)__rev1.val[3], 51); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_u16_x4(__p0, __p1) __extension__ ({ \
  uint16x8x4_t __s1 = __p1; \
  __builtin_neon_vst1q_x4_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], (int8x16_t)__s1.val[2], (int8x16_t)__s1.val[3], 49); \
})
#else
#define vst1q_u16_x4(__p0, __p1) __extension__ ({ \
  uint16x8x4_t __s1 = __p1; \
  uint16x8x4_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[3] = __builtin_shufflevector(__s1.val[3], __s1.val[3], 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst1q_x4_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], (int8x16_t)__rev1.val[2], (int8x16_t)__rev1.val[3], 49); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_s8_x4(__p0, __p1) __extension__ ({ \
  int8x16x4_t __s1 = __p1; \
  __builtin_neon_vst1q_x4_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], (int8x16_t)__s1.val[2], (int8x16_t)__s1.val[3], 32); \
})
#else
#define vst1q_s8_x4(__p0, __p1) __extension__ ({ \
  int8x16x4_t __s1 = __p1; \
  int8x16x4_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[3] = __builtin_shufflevector(__s1.val[3], __s1.val[3], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst1q_x4_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], (int8x16_t)__rev1.val[2], (int8x16_t)__rev1.val[3], 32); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_f32_x4(__p0, __p1) __extension__ ({ \
  float32x4x4_t __s1 = __p1; \
  __builtin_neon_vst1q_x4_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], (int8x16_t)__s1.val[2], (int8x16_t)__s1.val[3], 41); \
})
#else
#define vst1q_f32_x4(__p0, __p1) __extension__ ({ \
  float32x4x4_t __s1 = __p1; \
  float32x4x4_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 3, 2, 1, 0); \
  __rev1.val[3] = __builtin_shufflevector(__s1.val[3], __s1.val[3], 3, 2, 1, 0); \
  __builtin_neon_vst1q_x4_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], (int8x16_t)__rev1.val[2], (int8x16_t)__rev1.val[3], 41); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_s32_x4(__p0, __p1) __extension__ ({ \
  int32x4x4_t __s1 = __p1; \
  __builtin_neon_vst1q_x4_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], (int8x16_t)__s1.val[2], (int8x16_t)__s1.val[3], 34); \
})
#else
#define vst1q_s32_x4(__p0, __p1) __extension__ ({ \
  int32x4x4_t __s1 = __p1; \
  int32x4x4_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 3, 2, 1, 0); \
  __rev1.val[3] = __builtin_shufflevector(__s1.val[3], __s1.val[3], 3, 2, 1, 0); \
  __builtin_neon_vst1q_x4_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], (int8x16_t)__rev1.val[2], (int8x16_t)__rev1.val[3], 34); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_s64_x4(__p0, __p1) __extension__ ({ \
  int64x2x4_t __s1 = __p1; \
  __builtin_neon_vst1q_x4_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], (int8x16_t)__s1.val[2], (int8x16_t)__s1.val[3], 35); \
})
#else
#define vst1q_s64_x4(__p0, __p1) __extension__ ({ \
  int64x2x4_t __s1 = __p1; \
  int64x2x4_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 1, 0); \
  __rev1.val[3] = __builtin_shufflevector(__s1.val[3], __s1.val[3], 1, 0); \
  __builtin_neon_vst1q_x4_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], (int8x16_t)__rev1.val[2], (int8x16_t)__rev1.val[3], 35); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1q_s16_x4(__p0, __p1) __extension__ ({ \
  int16x8x4_t __s1 = __p1; \
  __builtin_neon_vst1q_x4_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], (int8x16_t)__s1.val[2], (int8x16_t)__s1.val[3], 33); \
})
#else
#define vst1q_s16_x4(__p0, __p1) __extension__ ({ \
  int16x8x4_t __s1 = __p1; \
  int16x8x4_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[3] = __builtin_shufflevector(__s1.val[3], __s1.val[3], 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst1q_x4_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], (int8x16_t)__rev1.val[2], (int8x16_t)__rev1.val[3], 33); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1_u8_x4(__p0, __p1) __extension__ ({ \
  uint8x8x4_t __s1 = __p1; \
  __builtin_neon_vst1_x4_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], (int8x8_t)__s1.val[2], (int8x8_t)__s1.val[3], 16); \
})
#else
#define vst1_u8_x4(__p0, __p1) __extension__ ({ \
  uint8x8x4_t __s1 = __p1; \
  uint8x8x4_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[3] = __builtin_shufflevector(__s1.val[3], __s1.val[3], 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst1_x4_v(__p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], (int8x8_t)__rev1.val[2], (int8x8_t)__rev1.val[3], 16); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1_u32_x4(__p0, __p1) __extension__ ({ \
  uint32x2x4_t __s1 = __p1; \
  __builtin_neon_vst1_x4_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], (int8x8_t)__s1.val[2], (int8x8_t)__s1.val[3], 18); \
})
#else
#define vst1_u32_x4(__p0, __p1) __extension__ ({ \
  uint32x2x4_t __s1 = __p1; \
  uint32x2x4_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 1, 0); \
  __rev1.val[3] = __builtin_shufflevector(__s1.val[3], __s1.val[3], 1, 0); \
  __builtin_neon_vst1_x4_v(__p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], (int8x8_t)__rev1.val[2], (int8x8_t)__rev1.val[3], 18); \
})
#endif

#define vst1_u64_x4(__p0, __p1) __extension__ ({ \
  uint64x1x4_t __s1 = __p1; \
  __builtin_neon_vst1_x4_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], (int8x8_t)__s1.val[2], (int8x8_t)__s1.val[3], 19); \
})
#ifdef __LITTLE_ENDIAN__
#define vst1_u16_x4(__p0, __p1) __extension__ ({ \
  uint16x4x4_t __s1 = __p1; \
  __builtin_neon_vst1_x4_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], (int8x8_t)__s1.val[2], (int8x8_t)__s1.val[3], 17); \
})
#else
#define vst1_u16_x4(__p0, __p1) __extension__ ({ \
  uint16x4x4_t __s1 = __p1; \
  uint16x4x4_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 3, 2, 1, 0); \
  __rev1.val[3] = __builtin_shufflevector(__s1.val[3], __s1.val[3], 3, 2, 1, 0); \
  __builtin_neon_vst1_x4_v(__p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], (int8x8_t)__rev1.val[2], (int8x8_t)__rev1.val[3], 17); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1_s8_x4(__p0, __p1) __extension__ ({ \
  int8x8x4_t __s1 = __p1; \
  __builtin_neon_vst1_x4_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], (int8x8_t)__s1.val[2], (int8x8_t)__s1.val[3], 0); \
})
#else
#define vst1_s8_x4(__p0, __p1) __extension__ ({ \
  int8x8x4_t __s1 = __p1; \
  int8x8x4_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[3] = __builtin_shufflevector(__s1.val[3], __s1.val[3], 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst1_x4_v(__p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], (int8x8_t)__rev1.val[2], (int8x8_t)__rev1.val[3], 0); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1_f32_x4(__p0, __p1) __extension__ ({ \
  float32x2x4_t __s1 = __p1; \
  __builtin_neon_vst1_x4_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], (int8x8_t)__s1.val[2], (int8x8_t)__s1.val[3], 9); \
})
#else
#define vst1_f32_x4(__p0, __p1) __extension__ ({ \
  float32x2x4_t __s1 = __p1; \
  float32x2x4_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 1, 0); \
  __rev1.val[3] = __builtin_shufflevector(__s1.val[3], __s1.val[3], 1, 0); \
  __builtin_neon_vst1_x4_v(__p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], (int8x8_t)__rev1.val[2], (int8x8_t)__rev1.val[3], 9); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst1_s32_x4(__p0, __p1) __extension__ ({ \
  int32x2x4_t __s1 = __p1; \
  __builtin_neon_vst1_x4_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], (int8x8_t)__s1.val[2], (int8x8_t)__s1.val[3], 2); \
})
#else
#define vst1_s32_x4(__p0, __p1) __extension__ ({ \
  int32x2x4_t __s1 = __p1; \
  int32x2x4_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 1, 0); \
  __rev1.val[3] = __builtin_shufflevector(__s1.val[3], __s1.val[3], 1, 0); \
  __builtin_neon_vst1_x4_v(__p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], (int8x8_t)__rev1.val[2], (int8x8_t)__rev1.val[3], 2); \
})
#endif

#define vst1_s64_x4(__p0, __p1) __extension__ ({ \
  int64x1x4_t __s1 = __p1; \
  __builtin_neon_vst1_x4_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], (int8x8_t)__s1.val[2], (int8x8_t)__s1.val[3], 3); \
})
#ifdef __LITTLE_ENDIAN__
#define vst1_s16_x4(__p0, __p1) __extension__ ({ \
  int16x4x4_t __s1 = __p1; \
  __builtin_neon_vst1_x4_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], (int8x8_t)__s1.val[2], (int8x8_t)__s1.val[3], 1); \
})
#else
#define vst1_s16_x4(__p0, __p1) __extension__ ({ \
  int16x4x4_t __s1 = __p1; \
  int16x4x4_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 3, 2, 1, 0); \
  __rev1.val[3] = __builtin_shufflevector(__s1.val[3], __s1.val[3], 3, 2, 1, 0); \
  __builtin_neon_vst1_x4_v(__p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], (int8x8_t)__rev1.val[2], (int8x8_t)__rev1.val[3], 1); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst2_p8(__p0, __p1) __extension__ ({ \
  poly8x8x2_t __s1 = __p1; \
  __builtin_neon_vst2_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], 4); \
})
#else
#define vst2_p8(__p0, __p1) __extension__ ({ \
  poly8x8x2_t __s1 = __p1; \
  poly8x8x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst2_v(__p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], 4); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst2_p16(__p0, __p1) __extension__ ({ \
  poly16x4x2_t __s1 = __p1; \
  __builtin_neon_vst2_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], 5); \
})
#else
#define vst2_p16(__p0, __p1) __extension__ ({ \
  poly16x4x2_t __s1 = __p1; \
  poly16x4x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 3, 2, 1, 0); \
  __builtin_neon_vst2_v(__p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], 5); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst2q_p8(__p0, __p1) __extension__ ({ \
  poly8x16x2_t __s1 = __p1; \
  __builtin_neon_vst2q_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], 36); \
})
#else
#define vst2q_p8(__p0, __p1) __extension__ ({ \
  poly8x16x2_t __s1 = __p1; \
  poly8x16x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst2q_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], 36); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst2q_p16(__p0, __p1) __extension__ ({ \
  poly16x8x2_t __s1 = __p1; \
  __builtin_neon_vst2q_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], 37); \
})
#else
#define vst2q_p16(__p0, __p1) __extension__ ({ \
  poly16x8x2_t __s1 = __p1; \
  poly16x8x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst2q_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], 37); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst2q_u8(__p0, __p1) __extension__ ({ \
  uint8x16x2_t __s1 = __p1; \
  __builtin_neon_vst2q_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], 48); \
})
#else
#define vst2q_u8(__p0, __p1) __extension__ ({ \
  uint8x16x2_t __s1 = __p1; \
  uint8x16x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst2q_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], 48); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst2q_u32(__p0, __p1) __extension__ ({ \
  uint32x4x2_t __s1 = __p1; \
  __builtin_neon_vst2q_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], 50); \
})
#else
#define vst2q_u32(__p0, __p1) __extension__ ({ \
  uint32x4x2_t __s1 = __p1; \
  uint32x4x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 3, 2, 1, 0); \
  __builtin_neon_vst2q_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], 50); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst2q_u16(__p0, __p1) __extension__ ({ \
  uint16x8x2_t __s1 = __p1; \
  __builtin_neon_vst2q_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], 49); \
})
#else
#define vst2q_u16(__p0, __p1) __extension__ ({ \
  uint16x8x2_t __s1 = __p1; \
  uint16x8x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst2q_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], 49); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst2q_s8(__p0, __p1) __extension__ ({ \
  int8x16x2_t __s1 = __p1; \
  __builtin_neon_vst2q_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], 32); \
})
#else
#define vst2q_s8(__p0, __p1) __extension__ ({ \
  int8x16x2_t __s1 = __p1; \
  int8x16x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst2q_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], 32); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst2q_f32(__p0, __p1) __extension__ ({ \
  float32x4x2_t __s1 = __p1; \
  __builtin_neon_vst2q_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], 41); \
})
#else
#define vst2q_f32(__p0, __p1) __extension__ ({ \
  float32x4x2_t __s1 = __p1; \
  float32x4x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 3, 2, 1, 0); \
  __builtin_neon_vst2q_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], 41); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst2q_s32(__p0, __p1) __extension__ ({ \
  int32x4x2_t __s1 = __p1; \
  __builtin_neon_vst2q_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], 34); \
})
#else
#define vst2q_s32(__p0, __p1) __extension__ ({ \
  int32x4x2_t __s1 = __p1; \
  int32x4x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 3, 2, 1, 0); \
  __builtin_neon_vst2q_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], 34); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst2q_s16(__p0, __p1) __extension__ ({ \
  int16x8x2_t __s1 = __p1; \
  __builtin_neon_vst2q_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], 33); \
})
#else
#define vst2q_s16(__p0, __p1) __extension__ ({ \
  int16x8x2_t __s1 = __p1; \
  int16x8x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst2q_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], 33); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst2_u8(__p0, __p1) __extension__ ({ \
  uint8x8x2_t __s1 = __p1; \
  __builtin_neon_vst2_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], 16); \
})
#else
#define vst2_u8(__p0, __p1) __extension__ ({ \
  uint8x8x2_t __s1 = __p1; \
  uint8x8x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst2_v(__p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], 16); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst2_u32(__p0, __p1) __extension__ ({ \
  uint32x2x2_t __s1 = __p1; \
  __builtin_neon_vst2_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], 18); \
})
#else
#define vst2_u32(__p0, __p1) __extension__ ({ \
  uint32x2x2_t __s1 = __p1; \
  uint32x2x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 1, 0); \
  __builtin_neon_vst2_v(__p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], 18); \
})
#endif

#define vst2_u64(__p0, __p1) __extension__ ({ \
  uint64x1x2_t __s1 = __p1; \
  __builtin_neon_vst2_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], 19); \
})
#ifdef __LITTLE_ENDIAN__
#define vst2_u16(__p0, __p1) __extension__ ({ \
  uint16x4x2_t __s1 = __p1; \
  __builtin_neon_vst2_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], 17); \
})
#else
#define vst2_u16(__p0, __p1) __extension__ ({ \
  uint16x4x2_t __s1 = __p1; \
  uint16x4x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 3, 2, 1, 0); \
  __builtin_neon_vst2_v(__p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], 17); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst2_s8(__p0, __p1) __extension__ ({ \
  int8x8x2_t __s1 = __p1; \
  __builtin_neon_vst2_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], 0); \
})
#else
#define vst2_s8(__p0, __p1) __extension__ ({ \
  int8x8x2_t __s1 = __p1; \
  int8x8x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst2_v(__p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], 0); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst2_f32(__p0, __p1) __extension__ ({ \
  float32x2x2_t __s1 = __p1; \
  __builtin_neon_vst2_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], 9); \
})
#else
#define vst2_f32(__p0, __p1) __extension__ ({ \
  float32x2x2_t __s1 = __p1; \
  float32x2x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 1, 0); \
  __builtin_neon_vst2_v(__p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], 9); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst2_s32(__p0, __p1) __extension__ ({ \
  int32x2x2_t __s1 = __p1; \
  __builtin_neon_vst2_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], 2); \
})
#else
#define vst2_s32(__p0, __p1) __extension__ ({ \
  int32x2x2_t __s1 = __p1; \
  int32x2x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 1, 0); \
  __builtin_neon_vst2_v(__p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], 2); \
})
#endif

#define vst2_s64(__p0, __p1) __extension__ ({ \
  int64x1x2_t __s1 = __p1; \
  __builtin_neon_vst2_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], 3); \
})
#ifdef __LITTLE_ENDIAN__
#define vst2_s16(__p0, __p1) __extension__ ({ \
  int16x4x2_t __s1 = __p1; \
  __builtin_neon_vst2_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], 1); \
})
#else
#define vst2_s16(__p0, __p1) __extension__ ({ \
  int16x4x2_t __s1 = __p1; \
  int16x4x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 3, 2, 1, 0); \
  __builtin_neon_vst2_v(__p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], 1); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst2_lane_p8(__p0, __p1, __p2) __extension__ ({ \
  poly8x8x2_t __s1 = __p1; \
  __builtin_neon_vst2_lane_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], __p2, 4); \
})
#else
#define vst2_lane_p8(__p0, __p1, __p2) __extension__ ({ \
  poly8x8x2_t __s1 = __p1; \
  poly8x8x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst2_lane_v(__p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], __p2, 4); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst2_lane_p16(__p0, __p1, __p2) __extension__ ({ \
  poly16x4x2_t __s1 = __p1; \
  __builtin_neon_vst2_lane_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], __p2, 5); \
})
#else
#define vst2_lane_p16(__p0, __p1, __p2) __extension__ ({ \
  poly16x4x2_t __s1 = __p1; \
  poly16x4x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 3, 2, 1, 0); \
  __builtin_neon_vst2_lane_v(__p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], __p2, 5); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst2q_lane_p16(__p0, __p1, __p2) __extension__ ({ \
  poly16x8x2_t __s1 = __p1; \
  __builtin_neon_vst2q_lane_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], __p2, 37); \
})
#else
#define vst2q_lane_p16(__p0, __p1, __p2) __extension__ ({ \
  poly16x8x2_t __s1 = __p1; \
  poly16x8x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst2q_lane_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], __p2, 37); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst2q_lane_u32(__p0, __p1, __p2) __extension__ ({ \
  uint32x4x2_t __s1 = __p1; \
  __builtin_neon_vst2q_lane_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], __p2, 50); \
})
#else
#define vst2q_lane_u32(__p0, __p1, __p2) __extension__ ({ \
  uint32x4x2_t __s1 = __p1; \
  uint32x4x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 3, 2, 1, 0); \
  __builtin_neon_vst2q_lane_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], __p2, 50); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst2q_lane_u16(__p0, __p1, __p2) __extension__ ({ \
  uint16x8x2_t __s1 = __p1; \
  __builtin_neon_vst2q_lane_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], __p2, 49); \
})
#else
#define vst2q_lane_u16(__p0, __p1, __p2) __extension__ ({ \
  uint16x8x2_t __s1 = __p1; \
  uint16x8x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst2q_lane_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], __p2, 49); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst2q_lane_f32(__p0, __p1, __p2) __extension__ ({ \
  float32x4x2_t __s1 = __p1; \
  __builtin_neon_vst2q_lane_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], __p2, 41); \
})
#else
#define vst2q_lane_f32(__p0, __p1, __p2) __extension__ ({ \
  float32x4x2_t __s1 = __p1; \
  float32x4x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 3, 2, 1, 0); \
  __builtin_neon_vst2q_lane_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], __p2, 41); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst2q_lane_s32(__p0, __p1, __p2) __extension__ ({ \
  int32x4x2_t __s1 = __p1; \
  __builtin_neon_vst2q_lane_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], __p2, 34); \
})
#else
#define vst2q_lane_s32(__p0, __p1, __p2) __extension__ ({ \
  int32x4x2_t __s1 = __p1; \
  int32x4x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 3, 2, 1, 0); \
  __builtin_neon_vst2q_lane_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], __p2, 34); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst2q_lane_s16(__p0, __p1, __p2) __extension__ ({ \
  int16x8x2_t __s1 = __p1; \
  __builtin_neon_vst2q_lane_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], __p2, 33); \
})
#else
#define vst2q_lane_s16(__p0, __p1, __p2) __extension__ ({ \
  int16x8x2_t __s1 = __p1; \
  int16x8x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst2q_lane_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], __p2, 33); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst2_lane_u8(__p0, __p1, __p2) __extension__ ({ \
  uint8x8x2_t __s1 = __p1; \
  __builtin_neon_vst2_lane_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], __p2, 16); \
})
#else
#define vst2_lane_u8(__p0, __p1, __p2) __extension__ ({ \
  uint8x8x2_t __s1 = __p1; \
  uint8x8x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst2_lane_v(__p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], __p2, 16); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst2_lane_u32(__p0, __p1, __p2) __extension__ ({ \
  uint32x2x2_t __s1 = __p1; \
  __builtin_neon_vst2_lane_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], __p2, 18); \
})
#else
#define vst2_lane_u32(__p0, __p1, __p2) __extension__ ({ \
  uint32x2x2_t __s1 = __p1; \
  uint32x2x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 1, 0); \
  __builtin_neon_vst2_lane_v(__p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], __p2, 18); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst2_lane_u16(__p0, __p1, __p2) __extension__ ({ \
  uint16x4x2_t __s1 = __p1; \
  __builtin_neon_vst2_lane_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], __p2, 17); \
})
#else
#define vst2_lane_u16(__p0, __p1, __p2) __extension__ ({ \
  uint16x4x2_t __s1 = __p1; \
  uint16x4x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 3, 2, 1, 0); \
  __builtin_neon_vst2_lane_v(__p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], __p2, 17); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst2_lane_s8(__p0, __p1, __p2) __extension__ ({ \
  int8x8x2_t __s1 = __p1; \
  __builtin_neon_vst2_lane_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], __p2, 0); \
})
#else
#define vst2_lane_s8(__p0, __p1, __p2) __extension__ ({ \
  int8x8x2_t __s1 = __p1; \
  int8x8x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst2_lane_v(__p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], __p2, 0); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst2_lane_f32(__p0, __p1, __p2) __extension__ ({ \
  float32x2x2_t __s1 = __p1; \
  __builtin_neon_vst2_lane_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], __p2, 9); \
})
#else
#define vst2_lane_f32(__p0, __p1, __p2) __extension__ ({ \
  float32x2x2_t __s1 = __p1; \
  float32x2x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 1, 0); \
  __builtin_neon_vst2_lane_v(__p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], __p2, 9); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst2_lane_s32(__p0, __p1, __p2) __extension__ ({ \
  int32x2x2_t __s1 = __p1; \
  __builtin_neon_vst2_lane_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], __p2, 2); \
})
#else
#define vst2_lane_s32(__p0, __p1, __p2) __extension__ ({ \
  int32x2x2_t __s1 = __p1; \
  int32x2x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 1, 0); \
  __builtin_neon_vst2_lane_v(__p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], __p2, 2); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst2_lane_s16(__p0, __p1, __p2) __extension__ ({ \
  int16x4x2_t __s1 = __p1; \
  __builtin_neon_vst2_lane_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], __p2, 1); \
})
#else
#define vst2_lane_s16(__p0, __p1, __p2) __extension__ ({ \
  int16x4x2_t __s1 = __p1; \
  int16x4x2_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 3, 2, 1, 0); \
  __builtin_neon_vst2_lane_v(__p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], __p2, 1); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst3_p8(__p0, __p1) __extension__ ({ \
  poly8x8x3_t __s1 = __p1; \
  __builtin_neon_vst3_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], (int8x8_t)__s1.val[2], 4); \
})
#else
#define vst3_p8(__p0, __p1) __extension__ ({ \
  poly8x8x3_t __s1 = __p1; \
  poly8x8x3_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst3_v(__p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], (int8x8_t)__rev1.val[2], 4); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst3_p16(__p0, __p1) __extension__ ({ \
  poly16x4x3_t __s1 = __p1; \
  __builtin_neon_vst3_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], (int8x8_t)__s1.val[2], 5); \
})
#else
#define vst3_p16(__p0, __p1) __extension__ ({ \
  poly16x4x3_t __s1 = __p1; \
  poly16x4x3_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 3, 2, 1, 0); \
  __builtin_neon_vst3_v(__p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], (int8x8_t)__rev1.val[2], 5); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst3q_p8(__p0, __p1) __extension__ ({ \
  poly8x16x3_t __s1 = __p1; \
  __builtin_neon_vst3q_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], (int8x16_t)__s1.val[2], 36); \
})
#else
#define vst3q_p8(__p0, __p1) __extension__ ({ \
  poly8x16x3_t __s1 = __p1; \
  poly8x16x3_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst3q_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], (int8x16_t)__rev1.val[2], 36); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst3q_p16(__p0, __p1) __extension__ ({ \
  poly16x8x3_t __s1 = __p1; \
  __builtin_neon_vst3q_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], (int8x16_t)__s1.val[2], 37); \
})
#else
#define vst3q_p16(__p0, __p1) __extension__ ({ \
  poly16x8x3_t __s1 = __p1; \
  poly16x8x3_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst3q_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], (int8x16_t)__rev1.val[2], 37); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst3q_u8(__p0, __p1) __extension__ ({ \
  uint8x16x3_t __s1 = __p1; \
  __builtin_neon_vst3q_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], (int8x16_t)__s1.val[2], 48); \
})
#else
#define vst3q_u8(__p0, __p1) __extension__ ({ \
  uint8x16x3_t __s1 = __p1; \
  uint8x16x3_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst3q_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], (int8x16_t)__rev1.val[2], 48); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst3q_u32(__p0, __p1) __extension__ ({ \
  uint32x4x3_t __s1 = __p1; \
  __builtin_neon_vst3q_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], (int8x16_t)__s1.val[2], 50); \
})
#else
#define vst3q_u32(__p0, __p1) __extension__ ({ \
  uint32x4x3_t __s1 = __p1; \
  uint32x4x3_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 3, 2, 1, 0); \
  __builtin_neon_vst3q_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], (int8x16_t)__rev1.val[2], 50); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst3q_u16(__p0, __p1) __extension__ ({ \
  uint16x8x3_t __s1 = __p1; \
  __builtin_neon_vst3q_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], (int8x16_t)__s1.val[2], 49); \
})
#else
#define vst3q_u16(__p0, __p1) __extension__ ({ \
  uint16x8x3_t __s1 = __p1; \
  uint16x8x3_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst3q_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], (int8x16_t)__rev1.val[2], 49); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst3q_s8(__p0, __p1) __extension__ ({ \
  int8x16x3_t __s1 = __p1; \
  __builtin_neon_vst3q_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], (int8x16_t)__s1.val[2], 32); \
})
#else
#define vst3q_s8(__p0, __p1) __extension__ ({ \
  int8x16x3_t __s1 = __p1; \
  int8x16x3_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst3q_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], (int8x16_t)__rev1.val[2], 32); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst3q_f32(__p0, __p1) __extension__ ({ \
  float32x4x3_t __s1 = __p1; \
  __builtin_neon_vst3q_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], (int8x16_t)__s1.val[2], 41); \
})
#else
#define vst3q_f32(__p0, __p1) __extension__ ({ \
  float32x4x3_t __s1 = __p1; \
  float32x4x3_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 3, 2, 1, 0); \
  __builtin_neon_vst3q_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], (int8x16_t)__rev1.val[2], 41); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst3q_s32(__p0, __p1) __extension__ ({ \
  int32x4x3_t __s1 = __p1; \
  __builtin_neon_vst3q_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], (int8x16_t)__s1.val[2], 34); \
})
#else
#define vst3q_s32(__p0, __p1) __extension__ ({ \
  int32x4x3_t __s1 = __p1; \
  int32x4x3_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 3, 2, 1, 0); \
  __builtin_neon_vst3q_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], (int8x16_t)__rev1.val[2], 34); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst3q_s16(__p0, __p1) __extension__ ({ \
  int16x8x3_t __s1 = __p1; \
  __builtin_neon_vst3q_v(__p0, (int8x16_t)__s1.val[0], (int8x16_t)__s1.val[1], (int8x16_t)__s1.val[2], 33); \
})
#else
#define vst3q_s16(__p0, __p1) __extension__ ({ \
  int16x8x3_t __s1 = __p1; \
  int16x8x3_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst3q_v(__p0, (int8x16_t)__rev1.val[0], (int8x16_t)__rev1.val[1], (int8x16_t)__rev1.val[2], 33); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst3_u8(__p0, __p1) __extension__ ({ \
  uint8x8x3_t __s1 = __p1; \
  __builtin_neon_vst3_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], (int8x8_t)__s1.val[2], 16); \
})
#else
#define vst3_u8(__p0, __p1) __extension__ ({ \
  uint8x8x3_t __s1 = __p1; \
  uint8x8x3_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst3_v(__p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], (int8x8_t)__rev1.val[2], 16); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst3_u32(__p0, __p1) __extension__ ({ \
  uint32x2x3_t __s1 = __p1; \
  __builtin_neon_vst3_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], (int8x8_t)__s1.val[2], 18); \
})
#else
#define vst3_u32(__p0, __p1) __extension__ ({ \
  uint32x2x3_t __s1 = __p1; \
  uint32x2x3_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 1, 0); \
  __builtin_neon_vst3_v(__p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], (int8x8_t)__rev1.val[2], 18); \
})
#endif

#define vst3_u64(__p0, __p1) __extension__ ({ \
  uint64x1x3_t __s1 = __p1; \
  __builtin_neon_vst3_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], (int8x8_t)__s1.val[2], 19); \
})
#ifdef __LITTLE_ENDIAN__
#define vst3_u16(__p0, __p1) __extension__ ({ \
  uint16x4x3_t __s1 = __p1; \
  __builtin_neon_vst3_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], (int8x8_t)__s1.val[2], 17); \
})
#else
#define vst3_u16(__p0, __p1) __extension__ ({ \
  uint16x4x3_t __s1 = __p1; \
  uint16x4x3_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 3, 2, 1, 0); \
  __builtin_neon_vst3_v(__p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], (int8x8_t)__rev1.val[2], 17); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst3_s8(__p0, __p1) __extension__ ({ \
  int8x8x3_t __s1 = __p1; \
  __builtin_neon_vst3_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], (int8x8_t)__s1.val[2], 0); \
})
#else
#define vst3_s8(__p0, __p1) __extension__ ({ \
  int8x8x3_t __s1 = __p1; \
  int8x8x3_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 7, 6, 5, 4, 3, 2, 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 7, 6, 5, 4, 3, 2, 1, 0); \
  __builtin_neon_vst3_v(__p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], (int8x8_t)__rev1.val[2], 0); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst3_f32(__p0, __p1) __extension__ ({ \
  float32x2x3_t __s1 = __p1; \
  __builtin_neon_vst3_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], (int8x8_t)__s1.val[2], 9); \
})
#else
#define vst3_f32(__p0, __p1) __extension__ ({ \
  float32x2x3_t __s1 = __p1; \
  float32x2x3_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 1, 0); \
  __builtin_neon_vst3_v(__p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], (int8x8_t)__rev1.val[2], 9); \
})
#endif

#ifdef __LITTLE_ENDIAN__
#define vst3_s32(__p0, __p1) __extension__ ({ \
  int32x2x3_t __s1 = __p1; \
  __builtin_neon_vst3_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], (int8x8_t)__s1.val[2], 2); \
})
#else
#define vst3_s32(__p0, __p1) __extension__ ({ \
  int32x2x3_t __s1 = __p1; \
  int32x2x3_t __rev1; \
  __rev1.val[0] = __builtin_shufflevector(__s1.val[0], __s1.val[0], 1, 0); \
  __rev1.val[1] = __builtin_shufflevector(__s1.val[1], __s1.val[1], 1, 0); \
  __rev1.val[2] = __builtin_shufflevector(__s1.val[2], __s1.val[2], 1, 0); \
  __builtin_neon_vst3_v(__p0, (int8x8_t)__rev1.val[0], (int8x8_t)__rev1.val[1], (int8x8_t)__rev1.val[2], 2); \
})
#endif

#define vst3_s64(__p0, __p1) __extension__ ({ \
  int64x1x3_t __s1 = __p1; \
  __builtin_neon_vst3_v(__p0, (int8x8_t)__s1.val[0], (int8x8_t)__s1.val[1], (int8x8_t)__s1.val[2], 3); \
})
#ifdef __LITTLE_ENDIAN__
#define vst3_s16(__p0, __p1) __extension__ ({ \
  int16x4x3_t __s1 = 