# -*- coding: utf-8 -*-
# Copyright 2011 Jörn Schumacher, Henning Saul
# Copyright 2021 Christian Prasch
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

import sys, os, urllib, urllib.parse
import xbmc, xbmcplugin, xbmcgui, xbmcaddon, xbmcvfs
#import web_pdb

from libs.tagesschau_json_api import VideoContentProvider, addon
from libs.subtitles import download_subtitles
from infotagger.listitem import ListItemInfoTag

# -- Constants ----------------------------------------------
ADDON_ID = 'plugin.video.tagesschau'
FANART_IMG = xbmcvfs.translatePath('special://home/addons/' + ADDON_ID + '/resources/assets/fanart.jpg')
LOGO_IMG   = xbmcvfs.translatePath('special://home/addons/' + ADDON_ID + '/resources/assets/logo.jpg')
ICON_IMG   = xbmcvfs.translatePath('special://home/addons/' + ADDON_ID + '/resources/assets/icon.png')
ACTION_PARAM = 'action'
FEED_PARAM = 'feed'
ID_PARAM = 'tsid'
URL_PARAM = 'url'

# -- Settings -----------------------------------------------
quality_id = addon.getSetting('quality')
quality = ['M', 'L', 'X'][int(quality_id)]
show_fanart = addon.getSettingBool('show_fanart')

# -- I18n ---------------------------------------------------
language = addon.getLocalizedString
strings = { 'latest_videos':       language(30100),
            'latest_broadcasts':   language(30101),
            'livestreams':         language(30102),
            'tagesschau_20':       language(30104),
            'tagesthemen':         language(30105)
}

#-- Subtitles ------------------------------------------------

profile_dir = xbmcvfs.translatePath(addon.getAddonInfo('profile'))
subtitles_dir  = os.path.join(profile_dir, 'Subtitles')

# ------------------------------------------------------------


def addVideoContentDirectory(title, method):
    url_data = { ACTION_PARAM: 'list_feed', FEED_PARAM: method  }
    url = 'plugin://' + ADDON_ID + '/?' + urllib.parse.urlencode(url_data)
    li = xbmcgui.ListItem()
    li.setArt({'thumb':ICON_IMG, 'landscape':LOGO_IMG, 'icon':ICON_IMG})
    li.setProperty('Fanart_Image', FANART_IMG)

    infoLabels={ "title": str(title), "plot": str(title) }
    ListItemInfoTag(li, 'video').set_info(infoLabels)
    
    xbmcplugin.setContent(int(sys.argv[1]), 'files')
    xbmcplugin.addDirectoryItem(handle=int(sys.argv[1]), url=url, listitem=li, isFolder=True)

def getListItem(videocontent):
    title = str(videocontent.title)

    image_url = videocontent.image_url()
    if(not image_url):
        image_url = LOGO_IMG

    fanart_url = videocontent.fanart_url()
    if((not fanart_url) or (not show_fanart)):
        fanart_url = FANART_IMG

    li = xbmcgui.ListItem(title)
    li.setArt({'thumb':image_url, 'landscape':image_url})
    li.setProperty('Fanart_Image', fanart_url)
    li.setProperty('IsPlayable', 'true')

    infoLabels={ "title": str(title),
                 "plot": str(videocontent.description),
                 "duration": (videocontent.duration or 0)/60,
                 "mediatype": "video"
               }
    ListItemInfoTag(li, 'video').set_info(infoLabels)

    if( videocontent.timestamp ):
        infoLabels={ "premiered": str(videocontent.timestamp.strftime('%d.%m.%Y')),
                     "aired": str(videocontent.timestamp.strftime('%d.%m.%Y')),
                     "date": str(videocontent.timestamp.strftime('%d.%m.%Y'))
                   }
        ListItemInfoTag(li, 'video').set_info(infoLabels)

    return li

def getUrl(videocontent, method):
    url_data = { ACTION_PARAM: 'play_video' }
    url_data[URL_PARAM] = urllib.parse.quote(videocontent.video_url(quality))
    return 'plugin://' + ADDON_ID + '?' + urllib.parse.urlencode(url_data)

def addVideoContentItem(videocontent, method):
    li = getListItem(videocontent)
    url = getUrl(videocontent, method)
    xbmcplugin.setContent(int(sys.argv[1]), 'videos')
    return xbmcplugin.addDirectoryItem(int(sys.argv[1]), url, li, False)

def addVideoContentItems(videocontents, method):
    items = []
    videocontents = sorted(videocontents)
    for videocontent in videocontents:
        li = getListItem(videocontent)
        url = getUrl(videocontent, method)
        items.append((url, li, False))
    xbmcplugin.setContent(int(sys.argv[1]), 'videos')
    return xbmcplugin.addDirectoryItems(int(sys.argv[1]), items, len(items))

def get_params():
    paramstring = sys.argv[2]
    params = urllib.parse.parse_qs(urllib.parse.urlparse(paramstring).query)

    for key in params:
        params[key] = params[key][0]
    return params


def tagesschau():
    # TODO: can't figure out how to set fanart for root/back folder of plugin
    # http://trac.xbmc.org/ticket/8228?
    xbmcplugin.setPluginFanart(int(sys.argv[1]), 'special://home/addons/' + ADDON_ID + '/resources/assets/fanart.jpg')
    xbmcplugin.addSortMethod(int(sys.argv[1]), xbmcplugin.SORT_METHOD_NONE)

    params = get_params()
    provider = VideoContentProvider()

    if params.get(ACTION_PARAM) == 'play_video':
        subtitles_file = None
        # expecting either url or feed and id param
        url = params.get(URL_PARAM)
        if url:
            url = urllib.parse.unquote(url)
        else:
            videos_method = getattr(provider, params[FEED_PARAM])
            videos = videos_method()
            tsid = urllib.parse.unquote(params[ID_PARAM])
            # find video with matching tsid
            for video in videos:
                if video.tsid == tsid:
                    url = video.video_url(quality)
                    # try to download and and convert subtitles to local SRT file
                    subtitles_url = 'https://www.tagesschau.de/multimedia/video/' + str(video.video_id()) + '~subtitle.html'
                    subtitles_file = download_subtitles(subtitles_url, subtitles_dir)

        listitem = xbmcgui.ListItem(path=url)
        if(subtitles_file != None):
            # the following only works in Gotham, see
            # http://forum.xbmc.org/showthread.php?tid=154805&page=7&highlight=subtitle
            listitem.setProperty('upnp:subtitle:1', subtitles_file)
            listitem.setSubtitles((subtitles_file,))

        xbmcplugin.setResolvedUrl(handle=int(sys.argv[1]), succeeded=(url != None), listitem=listitem)

    elif params.get(ACTION_PARAM) == 'list_feed':
        # list video for a directory
        videos_method = getattr(provider, params[FEED_PARAM])
        videos = videos_method()
        addVideoContentItems(videos, params[FEED_PARAM])

    else:
        # add directories for other feeds
        add_named_directory = lambda x: addVideoContentDirectory(strings[x], x)
        add_named_directory('livestreams')
        add_named_directory('latest_videos')
        add_named_directory('latest_broadcasts')
        add_named_directory('tagesschau_20')
        add_named_directory('tagesthemen')

    xbmcplugin.endOfDirectory(int(sys.argv[1]))
