// Copyright 2012-present Oliver Eilhard. All rights reserved.
// Use of this source code is governed by a MIT-license.
// See http://olivere.mit-license.org/license.txt for details.

package elastic

// CardinalityAggregation is a single-value metrics aggregation that
// calculates an approximate count of distinct values.
// Values can be extracted either from specific fields in the document
// or generated by a script.
// See: https://www.elastic.co/guide/en/elasticsearch/reference/7.0/search-aggregations-metrics-cardinality-aggregation.html
type CardinalityAggregation struct {
	field              string
	script             *Script
	format             string
	missing            interface{}
	subAggregations    map[string]Aggregation
	meta               map[string]interface{}
	precisionThreshold *int64
	rehash             *bool
}

func NewCardinalityAggregation() *CardinalityAggregation {
	return &CardinalityAggregation{
		subAggregations: make(map[string]Aggregation),
	}
}

func (a *CardinalityAggregation) Field(field string) *CardinalityAggregation {
	a.field = field
	return a
}

func (a *CardinalityAggregation) Script(script *Script) *CardinalityAggregation {
	a.script = script
	return a
}

func (a *CardinalityAggregation) Format(format string) *CardinalityAggregation {
	a.format = format
	return a
}

func (a *CardinalityAggregation) Missing(missing interface{}) *CardinalityAggregation {
	a.missing = missing
	return a
}
func (a *CardinalityAggregation) SubAggregation(name string, subAggregation Aggregation) *CardinalityAggregation {
	a.subAggregations[name] = subAggregation
	return a
}

// Meta sets the meta data to be included in the aggregation response.
func (a *CardinalityAggregation) Meta(metaData map[string]interface{}) *CardinalityAggregation {
	a.meta = metaData
	return a
}

func (a *CardinalityAggregation) PrecisionThreshold(threshold int64) *CardinalityAggregation {
	a.precisionThreshold = &threshold
	return a
}

func (a *CardinalityAggregation) Rehash(rehash bool) *CardinalityAggregation {
	a.rehash = &rehash
	return a
}

func (a *CardinalityAggregation) Source() (interface{}, error) {
	// Example:
	//	{
	//    "aggs" : {
	//      "author_count" : {
	//        "cardinality" : { "field" : "author" }
	//      }
	//    }
	//	}
	// This method returns only the "cardinality" : { "field" : "author" } part.

	source := make(map[string]interface{})
	opts := make(map[string]interface{})
	source["cardinality"] = opts

	// ValuesSourceAggregationBuilder
	if a.field != "" {
		opts["field"] = a.field
	}
	if a.script != nil {
		src, err := a.script.Source()
		if err != nil {
			return nil, err
		}
		opts["script"] = src
	}
	if a.missing != nil {
		opts["missing"] = a.missing
	}
	if a.format != "" {
		opts["format"] = a.format
	}
	if a.precisionThreshold != nil {
		opts["precision_threshold"] = *a.precisionThreshold
	}
	if a.rehash != nil {
		opts["rehash"] = *a.rehash
	}

	// AggregationBuilder (SubAggregations)
	if len(a.subAggregations) > 0 {
		aggsMap := make(map[string]interface{})
		source["aggregations"] = aggsMap
		for name, aggregate := range a.subAggregations {
			src, err := aggregate.Source()
			if err != nil {
				return nil, err
			}
			aggsMap[name] = src
		}
	}

	// Add Meta data if available
	if len(a.meta) > 0 {
		source["meta"] = a.meta
	}

	return source, nil
}
